! ***********************************************************************
!
!   Copyright (C) 2010  Aaron Dotter
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module free_eos_table
      implicit none
      ! FreeEOS uses an abundance array called eps:
      ! EPS(:) = ( H,He,C,N,O,Ne,Na,Mg,Al,Si,P,S,Cl,Ar,Ca,Ti,Cr,Mn,Fe,Ni)
      ! consisting of 20 elements, each entry of eps is the mass fraction 
      ! of that element divided by its atomic weight
      integer, parameter :: Neps = 20, eosPT = 1, eosDT = 2
      integer :: ifopt, ifmod, ifion, kif
      !!!!!eventually get rid of this block
      integer, parameter :: num_eos_basic_results = 17
      integer, parameter :: i_lnPgas = 1, i_lnE = 2, i_lnS = 3, i_grad_ad = 4
      integer, parameter :: i_chiRho = 5, i_chiT = 6, i_Cp = 7, i_Cv = 8
      integer, parameter :: i_dE_dRho = 9, i_dS_dT = 10, i_dS_dRho = 11
      integer, parameter :: i_mu = 12, i_lnfree_e = 13, i_gamma1 = 14
      integer, parameter :: i_gamma3 = 15, i_eta = 16, i_lnRho = 17
      double precision, parameter :: ln10 = 2.302585093d0
      double precision :: free_eos_res(num_eos_basic_results)
      !!!!!

      contains 

         !for the 4 basic EOS options
         subroutine free_eos_set_version(eos_version,eos_type)
         ! eos_type (=kif) = 1 for rho(P,T)
         ! eos_type (=kif) = 2 for P(Rho,T)
         !integer, parameter :: kif = 2 !for P(Rho,T)
         !integer, parameter :: ifopt = 3, ifmod = 1, ifion = -2 !EOS1
         !integer, parameter :: ifopt = 3, ifmod = 1, ifion = -1 !EOS1a
         !integer, parameter :: ifopt = 2, ifmod = 1, ifion = -1 !EOS2
         !integer, parameter :: ifopt = 1, ifmod = 1, ifion =  0 !EOS3
         !integer, parameter :: ifopt = 1, ifmod=101, ifion =  0 !EOS4
         integer, intent(in) :: eos_version, eos_type
         integer :: my_ifopt, my_ifmod, my_ifion
         select case (eos_version)
            case (1) !EOS1 could also try EOS1a with ifion=-1
               my_ifopt = 3; my_ifmod = 1; my_ifion = -2
            case (2) !EOS2
               my_ifopt = 2; my_ifmod = 1; my_ifion = -1
            case (3) !EOS3
               my_ifopt = 1; my_ifmod = 1; my_ifion =  0
            case (4) !EOS4
               my_ifopt = 1; my_ifmod=101; my_ifion =  0
            case default !default to EOS4
               my_ifopt = 1; my_ifmod=101; my_ifion =  0
         end select
         call free_eos_set_options(my_ifopt,my_ifmod,my_ifion,eos_type)
         end subroutine free_eos_set_version

         !for complete control over EOS options
         subroutine free_eos_set_options(my_ifopt,my_ifmod,my_ifion,my_kif)
         integer, intent(in) :: my_ifopt, my_ifmod, my_ifion, my_kif
         ifopt = my_ifopt; ifmod = my_ifmod; ifion = my_ifion; kif = my_kif
         end subroutine free_eos_set_options

         subroutine free_eos_eval(match_var,logT,mass_frac,result)
         implicit none
         !integer, intent(in)  :: ifopt, ifmod, ifion, kif
         double precision, intent(in)  :: match_var,logT,mass_frac(Neps)
         double precision, intent(out) :: result(num_eos_basic_results)
         integer :: iter
         double precision :: logf, T, Rho, logRho, P, logP, Cf, Cp, Qf, Qp, Sf, &
            St, grada, RTP, Rmue, fh2, fhe2, fhe3, xmu1, xmu3, eta, gamma1, Prad, &
            gamma2, gamma3, h2rat, h2plusrat, lambda, gamma_e, chiRho, chiT
         double precision, dimension(Neps) :: atom_wgt, eps
         double precision, dimension(3) :: degeneracy,pressure,density,energy,enthalpy,entropy 

         !eventually get rid fo this
         atom_wgt(:) = (/ 1.007825d0, 4.0026d0,  12.0111d0, 14.0067d0, 15.9994d0, &
                         20.179d0,   22.98977d0, 24.305d0,  26.9815d0, 28.086d0, &
                         30.9738d0,  32.06d0,    35.453d0,  39.948d0,  40.08d0, &
                         47.9d0,     51.996d0,   54.938d0,  55.847d0,  58.71d0 /)

         eps(:) = mass_frac(:) / atom_wgt(:)

         call free_eos( ifopt, ifmod, ifion, kif, eps, Neps, &
            match_var, logT, logf, T, Rho, logRho, P, logP, &
            Cf, Cp, Qf, Qp, Sf, St, grada, RTP, Rmue, fh2, &
            fhe2, fhe3, xmu1, xmu3, eta, gamma1, gamma2, gamma3, &
            h2rat, h2plusrat, lambda, gamma_e, degeneracy, &
            pressure, density, energy, enthalpy, entropy, iter )

         if(kif == 2) then ! P(rho,T)
            chiRho = pressure(2)
            chiT   = pressure(3)
         elseif(kif == 1) then ! rho(P,T)
            chiT = -density(3)/density(2)
            chiRho = 1.d0/density(2)
         endif

         Prad  = (4d0*5.670400d-5*T**4)/(3d0*2.99792458d10)

         result(i_lnPgas) = safe_log10(P - Prad)
         result(i_lnE)    = safe_log10(energy(1))
         result(i_lnS)    = safe_log10(entropy(1))
         result(i_grad_ad)= grada
         result(i_chiRho) = chiRho
         result(i_chiT)   = chiT
         result(i_Cp)     = Cp
         result(i_Cv)     = energy(3)/T !(1/T)*dE/dlnT
         result(i_dE_dRho)= energy(2)/Rho !(1/Rho)*dE/dlnRho
         result(i_dS_dT)  = entropy(3)/T !(1/T)*dS/dlnT
         result(i_dS_dRho)= entropy(2)/Rho !(1/Rho)*dS/dlnRho
         result(i_mu)     = 1d0/xmu1
         result(i_lnfree_e)= safe_log(xmu3)
         result(i_gamma1) = gamma1
         result(i_gamma3) = gamma3
         result(i_eta)    = eta
         result(i_lnRho)  = safe_log10(Rho)

      end subroutine free_eos_eval

      elemental double precision function safe_log10(x)
         double precision, intent(in) :: x
         safe_log10 = log10(max(1d-99, x))
      end function safe_log10

      elemental double precision function safe_log(x)
         double precision, intent(in) :: x
         safe_log = log(max(1d-99, x))
      end function safe_log

      end module free_eos_table





      program make_free_eos_table
      use free_eos_table
      implicit none
      character(len=64) :: eosDT_file, eosPT_file, table_list, mass_list, data_dir, table_file
      integer :: io_unit, num_logQs, num_logTs, num_logWs, table_version, i, eos_type
      integer ::  num_tables, eos_version, ierr
      double precision :: log10T, log10Rho, logT, logRho, match_var, mass_frac(Neps)
      double precision :: dlog10T, dlog10Q, log10Qmin, log10Qmax, log10Tmin, log10Tmax
      double precision :: log10Wmin, log10Wmax, dlog10W, log10Pgas, logPgas, X, Y, Z
      logical :: do_eosDT, do_eosPT
      logical, parameter :: debug = .false.

      namelist / eos_table / table_list, num_tables, mass_list, table_version, eos_version, &
              do_eosPT, do_eosDT, log10Tmin, log10Tmax, dlog10T, log10Qmin, log10Qmax, &
              dlog10Q, log10Wmin, log10Wmax, dlog10W

      io_unit = 40
      data_dir = 'data/'

      call read_namelist
 
      call set_mass_fractions

      open(1,file=trim(table_list))
      do i = 1,num_tables

         read(1,*) Z, X, eosDT_file, eosPT_file
         Z=Z+1d-7
         Y = 1d0 - (X+Z)
         if(debug) write(*,*) 'X,Y,Z=', X, Y, Z
         mass_frac(1) = X*mass_frac(1)
         mass_frac(2) = Y*mass_frac(2)
         mass_frac(3:Neps) = Z*mass_frac(3:Neps)

         if(do_eosDT)then
            eos_type = eosDT
            call free_eos_set_version( eos_version, eos_type )
            write(*,'(a5,a24,3(a3,f5.2))') &
            'file=',trim(eosDT_file),' X=',X,' Y=',Y,' Z=',Z
            table_file = trim(data_dir) // trim(eosDT_file)
            open(io_unit,file=trim(table_file))
            call write_table
            close(io_unit)
         endif

         if(do_eosPT)then
            eos_type = eosPT
            call free_eos_set_version( eos_version, eos_type )
            write(*,'(a5,a24,3(a3,f5.2))') &
            'file=',trim(eosPT_file),' X=',X,' Y=',Y,' Z=',Z
            table_file = trim(data_dir) // trim(eosPT_file)
            open(io_unit,file=trim(table_file))
            call write_table
            close(io_unit)
         endif

      enddo

      close(1)

      contains


        subroutine read_namelist
!       table_list, mass_list, table_version, eos_version, &
!       do_eosPT, do_eosDT, log10Tmin, log10Tmax, dlog10T, log10Qmin, log10Qmax, &
!       dlog10Q, log10Wmin, log10Wmax, dlog10W

        ! set defaults
        table_list = 'table_list.txt'
        mass_list = 'mass_frac.txt'
        table_version = 48
        eos_version = 4
        num_tables = 1

        do_eosDT = .true.
        do_eosPT = .true.

        !set T range
        log10Tmin = 3d0
        log10Tmax = 8.2d0
        dlog10T = 0.02d0        !default 0.02

        !for eosDT
        log10Qmin = -9d0
        log10Qmax =  4.5d0
        dlog10Q = 0.03d0        !default 0.03


        !for eosPT
        log10Wmin = -14.5d0     !-17.2d0
        log10Wmax = -2.9d0
        dlog10W = 0.1d0
         
        !now, read namelist
        open(io_unit,file='inlist',action='read',delim='quote',status='old',iostat=ierr)
        if(ierr/=0) stop 'free_eos_table: problem opening inlist file'
        read(io_unit, nml=eos_table, iostat=ierr) !'
        if(ierr/=0) stop 'free_eos_table: problem reading inlist file'
        close(io_unit)

        num_logTs = 1 + int( (log10Tmax - log10Tmin) / dlog10T )
        num_logQs = 1 + int( (log10Qmax - log10Qmin) / dlog10Q )      
        num_logWs = 1 + int( (log10Wmax - log10Wmin) / dlog10W )

        if(debug)then
           write(*,*) 'dlog10T = ', dlog10T
           write(*,*) 'num_logTs = ', num_logTs
           write(*,*) 'dlog10Q = ', dlog10Q
           write(*,*) 'num_logQs = ', num_logQs
           write(*,*) 'dlog10W = ', dlog10W
           write(*,*) 'num_logWs = ', num_logWs
           !stop
        endif
        end subroutine read_namelist

        subroutine set_mass_fractions
        character(len=2) :: element
        ! EPS(:) = ( H,He,C,N,O,Ne,Na,Mg,Al,Si,P,S,Cl,Ar,Ca,Ti,Cr,Mn,Fe,Ni)
        ! mass_frac(:) = (/  1d0,  1d0,   0.171836d0,0.050335d0,0.467356d0, &
        !           0.104831d0,0.002090d0,0.039924d0,0.003603d0,0.044057d0, &
        !           0.000423d0,0.023513d0,0.000292d0,0.004335d0,0.003896d0, &
        !           0.000195d0,0.001117d0,0.000779d0,0.076433d0,0.004985d0  /)
        mass_frac(1) = 1d0 !H
        mass_frac(2) = 1d0 !He
        open(io_unit,file=trim(mass_list),action='read',status='old',iostat=ierr)
        if(ierr/=0) then 
           write(*,*) 'free_eos_table: problem opening mass fractions list: ', trim(mass_list)
           stop
        endif
        read(io_unit,*,iostat=ierr) !header line
        if(ierr/=0) then
           write(*,*) 'free_eos_table: problem reading mass fractions list: ', trim(mass_list)
           stop
        endif
        do i=3,Neps !read mass fractions of C - Ni
           read(io_unit,*,iostat=ierr) element, mass_frac(i)
           if(ierr/=0) then
              write(*,*) 'free_eos_table: problem reading mass fractions list: ', trim(mass_list)
              stop
           endif
        enddo

        close(io_unit)

        if( abs(sum(mass_frac(3:Neps))-1d0) > 1d-12 ) then
           write(*,*) 'free_eos_table: WARNING! Mass fractions of Z do not sum to 1'
        endif

        if(debug)then
           do i=1,Neps
              write(*,*) i, mass_frac(i)
           enddo
        endif

        end subroutine set_mass_fractions

        subroutine write_table
           double precision :: var, var_min, var_max, dvar

           call write_header

           if(eos_type == eosDT)then
              var_min = log10Qmin
              var_max = log10Qmax
              var = log10Qmin
              dvar = dlog10Q
           endif
           
           if(eos_type == eosPT)then
              var_min = log10Wmin
              var_max = log10Wmax
              var = log10Wmin
              dvar = dlog10W
           endif


           do while (var <= var_max)
              log10T = log10Tmin
              
              call write_sub_header(var)

              do while (log10T <= log10Tmax)

                 if(debug) write(*,*) 'log10var, log10T=', var, log10T

                 if(eos_type == eosDT)then
                    log10Rho = var + 2d0*log10T - 12.0d0
                    logRho = log10Rho*ln10
                    logT   = log10T*ln10
                    match_var = logRho
                 endif

                 if(eos_type == eosPT)then
                    log10Pgas = var + 4d0*log10T
                    logPgas = log10Pgas*ln10
                    logT   = log10T*ln10
                    match_var = logPgas
                 endif
         
                 call free_eos_eval(match_var,logT,mass_frac,free_eos_res)

                 if(eos_type == eosDT )then 
                    write(io_unit,'(f4.2,1p,99(e13.5))') log10T, &
                       free_eos_res(i_lnPgas:i_lnS), free_eos_res(i_chiRho:i_gamma3), &
                       free_eos_res(i_grad_ad), free_eos_res(i_eta)
                 endif
         
                 if(eos_type == eosPT )then 
                    write(io_unit,'(f4.2,1p,99(0p,1e13.5))') log10T, &
                       free_eos_res(i_lnRho), free_eos_res(i_lnE), free_eos_res(i_lnS), &
                       free_eos_res(i_chiRho:i_gamma3), free_eos_res(i_grad_ad), free_eos_res(i_eta)
                 endif

                 log10T = log10T + dlog10T
            
              enddo
           
              var = var + dvar

           enddo
        end subroutine write_table

        subroutine write_header !once at top of file
           if( eos_type == eosDT )then
              write(io_unit,'(99(a14))') 'version', 'X', 'Z', 'num logTs', 'logT min', &
              'logT max', 'del logT', 'num logQs', 'logQ min', 'logQ max', 'del logQ'

              write(io_unit,'(i14,2f14.4,2(i10,4x,3(f14.4)))') table_version, X, Z, &
              num_logTs, log10Tmin, log10Tmax, dlog10T, num_logQs, log10Qmin, &
              log10Qmax, dlog10Q
           endif

           if( eos_type == eosPT )then
              write(io_unit,'(99(a14))') 'version', 'X', 'Z', 'num logTs', 'logT min', &
              'logT max', 'del logT', 'num logWs', 'logW min', 'logW max', 'del logW'

              write(io_unit,'(i14,2f14.4,2(i10,4x,3(f14.4)))') table_version, X, Z, &
              num_logTs, log10Tmin, log10Tmax, dlog10T, num_logWs, log10Wmin, &
              log10Wmax, dlog10W
           endif
        end subroutine write_header

        subroutine write_sub_header(log10var) !every logQ value
           double precision, intent(in) :: log10var
           if(eos_type == eosDT) write(io_unit,'(/,7x,a)') 'logQ = logRho - 2*logT + 12' 
           if(eos_type == eosPT) write(io_unit,'(/,7x,a)') 'logW = logPgas - 4*logT'

           write(io_unit,'(2x,f14.6/)') log10var

           if(eos_type == eosDT)then 
              write(io_unit,'(a4,99(a12,1x))') 'logT', &
                 'logPgas', 'logE', 'logS', 'chiRho', 'chiT', 'Cp', &
                 'Cv', 'dE_dRho', 'dS_dT', 'dS_dRho', 'mu', &
                 'log_free_e', 'gamma1', 'gamma3', 'grad_ad', 'eta'
           endif
           
           if(eos_type == eosPT)then
              write(io_unit,'(a4,99(a12,1x))') 'logT', &
                 'logRho', 'logE', 'logS', 'chiRho', 'chiT', 'Cp', &
                 'Cv', 'dE_dRho', 'dS_dT', 'dS_dRho', 'mu', &
                 'log_free_e', 'gamma1', 'gamma3', 'grad_ad', 'eta'
           endif

        end subroutine write_sub_header

      end program make_free_eos_table
