c ***********************************************************************
!
!   Copyright (C) 2006, 2007, 2008  Bill Paxton, Frank Timmes
!
!   This file is part of MESA.
!
!   MESA is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
c ***********************************************************************

      module helm_opal_scvh_driver
      implicit none
      
      
      character (len=256) :: data_dir
		integer, parameter :: imax = 261, jmax = 101  ! dimensions of our version of helm table
      

		contains


		subroutine setup_eos
		   use helm_alloc
		   use eos_def
		   use const_lib
		
		   integer :: ierr
         character (len=256) :: eos_file_prefix, my_mesa_dir
		      
	      data_dir = 'eos_input_data' ! where to find the input data files
         
         my_mesa_dir = '../..'         
         call const_init(my_mesa_dir,ierr)     
      	if (ierr /= 0) then
      	   write(*,*) 'const_init failed'
      	   stop 1
      	end if        
         call eos_def_init
         
         ierr = 0
			call alloc_helm_table(eos_ht, imax, jmax, ierr)
			if (ierr /= 0) then
			   write(*,*) 'alloc helm table failed'
			   stop 1
			end if
		
			call read_helm_table(eos_ht, data_dir, ierr) ! initialize helm
			if (ierr /= 0) then
			   write(*,*) 'read helm table failed'
			   stop 1
			end if
			
			eos_ht% with_coulomb_corrections = .true.
			
			if (.not. eos_ht% with_coulomb_corrections) write(*,*) 'no coulomb corrections for helm'

		end subroutine setup_eos

 		


      
      subroutine get_azbar(X, Z, abar, zbar)
         double precision, intent(in) :: X, Z
         double precision, intent(out) :: abar, zbar

         integer :: i, ih1, ihe4, ic12, in14, io16, ine20, img24
         integer, parameter :: ionmax = 7
         double precision :: zbarxx, ytot1, Y,aion(ionmax),zion(ionmax),xmass(ionmax),ymass(ionmax)

			double precision, parameter :: Zfrac_C = 0.173312d0
			double precision, parameter :: Zfrac_N = 0.053177d0
			double precision, parameter :: Zfrac_O = 0.482398d0
			double precision, parameter :: Zfrac_Ne = 0.098675d0
			double precision, parameter :: Zfrac_Mg = 1d0 - (Zfrac_C + Zfrac_N + Zfrac_O + Zfrac_Ne)
         
         Y = 1 - (X+Z)
         
	      ih1        = 1
	      zion(ih1)  = 1.0d0
	      aion(ih1)  = 1.0d0 
	      xmass(ih1) = X

	      ihe4        = 2
	      zion(ihe4)  = 2.0d0
	      aion(ihe4)  = 4.0d0 
	      xmass(ihe4) = Y

	      ic12        = 3
	      zion(ic12)  = 6.0d0
	      aion(ic12)  = 12.0d0 
	      xmass(ic12) = Z * Zfrac_C

	      in14        = 4
	      zion(in14)  = 7.0d0
	      aion(in14)  = 14.0d0 
	      xmass(in14) = Z * Zfrac_N

	      io16        = 5
	      zion(io16)  = 8.0d0
	      aion(io16)  = 16.0d0 
	      xmass(io16) = Z * Zfrac_O

	      ine20       = 6
	      zion(ine20)  = 10.0d0
	      aion(ine20)  = 20.0d0 
	      xmass(ine20) = Z * Zfrac_Ne

	      img24       = 7
	      zion(img24)  = 12.0d0
	      aion(img24)  = 24.0d0 
	      xmass(img24) = Z * Zfrac_Mg

	      zbarxx  = 0.0d0
	      ytot1   = 0.0d0
	      do i=1,ionmax
	         ymass(i) = xmass(i)/aion(i)
	         ytot1    = ytot1 + ymass(i)
	         zbarxx   = zbarxx + zion(i) * ymass(i)
	      enddo
	      abar   = 1.0d0/ytot1
	      zbar   = zbarxx * abar
      
      end subroutine get_azbar


      subroutine helm_opal_scvh(
     >					helm_only, opal_scvh_only, opal_only, scvh_only, search_for_SCVH,
     >					include_radiation, logT, logRho, temp, den, abar, zbar, X, Z,
     >               logPgas, logE, logS, chiRho, chiT, 
     >               Cp, Cv, dE_dRho, dS_dT, dS_dRho, 
     >               mu, free_e, gamma1, gamma3, grad_ad, eta,
     >               data_dir, ierr)

!     >     helm_only, opal_scvh_only, opal_only, scvh_only,
!     >		include_radiation,logT,logRho,temp,den,abar,zbar,X,Z,
!     >		pout,dpoutdd,dpoutdt,
!     >		eout,deoutdd,gamma3,
!     >		sout,dsoutdd,dsoutdt,
!     >		xout,dxoutdd,dxoutdt,
!     >		mu_M_out,logNe_out,eta_ele_out,data_dir,ierr)
     
		use eos_def
		use helm
		use opal_scvh_driver
		use utils_lib, only: is_bad_num
		use const_def, only: crad, ln10
		
!..logT and logRho are log10's of temp and den
      implicit none
      save

!..mixes the helmholtz and opal_scvh equation of state

		logical, intent(in) :: helm_only, opal_scvh_only, opal_only, 
     >		scvh_only, search_for_SCVH, include_radiation
      double precision, intent(in) :: logT,logRho,temp,den,abar,zbar,X,Z
      double precision, intent(out) ::  
     >               logPgas, logE, logS, chiRho, chiT, 
     >               Cp, Cv, dE_dRho, dS_dT, dS_dRho, 
     >               mu, free_e, gamma1, gamma3, grad_ad, eta
      character (len=*), intent(in) ::  data_dir
		integer, intent(out) :: ierr


!..define the boundary between helm and opal/scvh

      include "eos_regions_defs.dek"


      double precision :: logNe, 
     >   logPgas_helm, logE_helm, logS_helm, chiRho_helm, chiT_helm, 
     >   Cp_helm, Cv_helm, dE_dRho_helm, dS_dT_helm, dS_dRho_helm, 
     >   mu_helm, gamma1_helm, gamma3_helm, grad_ad_helm, logNe_helm, eta_helm, 
     >   logPgas_opalscvh, logE_opalscvh, logS_opalscvh, chiRho_opalscvh, chiT_opalscvh, 
     >   Cp_opalscvh, Cv_opalscvh, dE_dRho_opalscvh, dS_dT_opalscvh, dS_dRho_opalscvh, 
     >   mu_opalscvh, logNe_opalscvh, gamma1_opalscvh, gamma3_opalscvh, grad_ad_opalscvh,  
     >   eta_opalscvh
     
      integer iregion
      double precision :: pa,pb,ea,eb,sa,sb
		double precision :: a, b
		double precision, parameter :: pi = 3.1415926535897932384d0
		logical :: have_called_helm
		double precision, parameter :: helm_min_temp = 1d3

!..some physical constants
      double precision clight
      parameter        (clight  = 2.99792458d10)
      double precision avo
      parameter        (avo  = 6.0221367d23)


!..loading the opal_scvh tables
      integer          ifirst
      data             ifirst/0/


      double precision :: helm_res(num_helm_results), dlnPgas_dlnY, Pgas, Prad
      
      include 'formats.dek'

		ierr = 0
		have_called_helm = .false.
		
		if (helm_only) then
			alfa = 1
		else if (opal_scvh_only .or. opal_only .or. scvh_only) then
			alfa = 0
		else
		   if (include_radiation) then
		      logT1 = logT1_default
		      logT2 = logT2_default
		   else
		      logT1 = logT1_no_rad_default
		      logT2 = logT2_no_rad_default
		   end if
         include 'eos_regions_code.dek'
		end if
		   
      beta = 1-alfa

      if (beta .ne. 0D0) then
         call interpolate_opal_scvh(
     >      opal_only, scvh_only, include_radiation, search_for_SCVH,
     >      logT, logRho, temp, den, abar, zbar, X, Z,
     >      logPgas_opalscvh, logE_opalscvh, logS_opalscvh, chiRho_opalscvh, chiT_opalscvh, 
     >      Cp_opalscvh, Cv_opalscvh, dE_dRho_opalscvh, dS_dT_opalscvh, dS_dRho_opalscvh, 
     >      mu_opalscvh, logNe_opalscvh, gamma1_opalscvh, gamma3_opalscvh, grad_ad_opalscvh,  
     >      eta_opalscvh, dlnPgas_dlnY,
     >      data_dir, ierr)
			
			Pgas = 10d0**logPgas_opalscvh
			Prad = crad*temp**4/3
			     
         if (is_bad_num(dE_dRho_opalscvh)) then
            ierr = -1
            if (.false.) then
               write(*,1) 'logT', logT
               write(*,1) 'logRho', logRho
               write(*,1) 'logQ', logRho - 2*logT + 12
               write(*,1) 'abar', abar
               write(*,1) 'zbar', zbar
               write(*,1) 'X', X
               write(*,1) 'Z', Z
               write(*,*)
               write(*,1) 'helm_opal_scvh: dE_dRho_opalscvh', dE_dRho_opalscvh
               stop 1
            end if
         end if

			if (ierr /= 0) then
			   !write(*,*) 'failed in interpolate_opal_scvh -- use helm instead'
			   !return
			   ! use helm instead
			   beta = 0
			   alfa = 1
			end if
			
      end if
      
      if (alfa .ne. 0D0) then
			call get_helmeos(include_radiation)
			if (ierr /= 0) then
			   write(*,*) 'failed in get_helmeos'
			   return
			end if
      end if
      
      if (alfa .eq. 0d0) then ! no HELM

         logPgas = logPgas_opalscvh
         logE = logE_opalscvh
         logS = logS_opalscvh
         chiRho = chiRho_opalscvh
         chiT = chiT_opalscvh
         Cp = Cp_opalscvh
         Cv = Cv_opalscvh
         dE_dRho = dE_dRho_opalscvh
         dS_dT = dS_dT_opalscvh
         dS_dRho = dS_dRho_opalscvh
         mu = mu_opalscvh
         logNe = logNe_opalscvh
         gamma1 = gamma1_opalscvh
         gamma3 = gamma3_opalscvh
         grad_ad = grad_ad_opalscvh
         eta = eta_opalscvh
         
      else if (beta .eq. 0d0) then ! pure HELM

         logPgas = logPgas_helm
         logE = logE_helm
         logS = logS_helm
         chiRho = chiRho_helm
         chiT = chiT_helm
         Cp = Cp_helm
         Cv = Cv_helm
         dE_dRho = dE_dRho_helm
         dS_dT = dS_dT_helm
         dS_dRho = dS_dRho_helm
         mu = mu_helm
         logNe = logNe_helm
         gamma1 = gamma1_helm
         gamma3 = gamma3_helm
         grad_ad = grad_ad_helm
         eta = eta_helm
      
      else ! combine alfa * helm + beta * opalscvh
      
         if (.false.) then
            logPgas = alfa*logPgas_helm + beta*logPgas_opalscvh
            logE = alfa*logE_helm + beta*logE_opalscvh
            logS = alfa*logS_helm + beta*logS_opalscvh
            chiRho = alfa*chiRho_helm + beta*chiRho_opalscvh
            chiT = alfa*chiT_helm + beta*chiT_opalscvh
            Cp = alfa*Cp_helm + beta*Cp_opalscvh
            Cv = alfa*Cv_helm + beta*Cv_opalscvh
            dE_dRho = alfa*dE_dRho_helm + beta*dE_dRho_opalscvh
            dS_dT = alfa*dS_dT_helm + beta*dS_dT_opalscvh
            dS_dRho = alfa*dS_dRho_helm + beta*dS_dRho_opalscvh
            mu = alfa*mu_helm + beta*mu_opalscvh
            gamma1 = alfa*gamma1_helm + beta*gamma1_opalscvh
            gamma3 = alfa*gamma3_helm + beta*gamma3_opalscvh
            grad_ad = alfa*grad_ad_helm + beta*grad_ad_opalscvh
            logNe = alfa*logNe_helm + beta*logNe_opalscvh
         
         else
         
            call blend(
     >         alfa, beta, den, temp, Prad, 
     >         logPgas_helm, logPgas_opalscvh, 
     >         logS_helm, logS_opalscvh, dS_dT_helm, dS_dT_opalscvh, dS_dRho_helm, dS_dRho_opalscvh,
     >         chiT_helm, chiT_opalscvh, chiRho_helm, chiRho_opalscvh, mu_helm, mu_opalscvh, logNe_helm, logNe_opalscvh,
     >         logE_helm, logE_opalscvh, Cv_helm, Cv_opalscvh, dE_dRho_helm, dE_dRho_opalscvh,
     >         gamma1_helm, gamma1_opalscvh, gamma3_helm, gamma3_opalscvh, grad_ad_helm, grad_ad_opalscvh,
     >         logPgas, logE, logS, chiRho, chiT, Cp, Cv, dE_dRho, dS_dT, dS_dRho,
     >         mu, gamma1, gamma3, grad_ad, logNe)
            
         end if
         
         eta = alfa*eta_helm + beta*eta_opalscvh

      end if
		
		free_e = 10**logNe / (avo * den) ! convert to mean number of free electrons per nucleon     
		
      if (is_bad_num(logPgas) .or. is_bad_num(logE) .or. is_bad_num(logS) .or.
     >      is_bad_num(chiRho) .or. is_bad_num(chiT) .or. is_bad_num(Cp) .or. 
     >      is_bad_num(gamma1) .or. is_bad_num(gamma3) .or. is_bad_num(grad_ad)) then
         write(*,1) 'logT', logT
         write(*,1) 'logRho', logRho
         write(*,1) 'logQ', logRho - 2*logT + 12
         write(*,1) 'abar', abar
         write(*,1) 'zbar', zbar
         write(*,1) 'X', X
         write(*,1) 'Z', Z
         write(*,*)
         write(*,1) 'logPgas', logPgas
         write(*,1) 'logE', logE
         write(*,1) 'logS', logS
         write(*,1) 'chiRho', chiRho
         write(*,1) 'chiT', chiT
         write(*,1) 'Cp', Cp
         write(*,1) 'gamma1', gamma1
         write(*,1) 'gamma3', gamma3
         write(*,1) 'grad_ad', grad_ad
         write(*,*)
   		write(*,1) 'alfa', alfa
   		write(*,1) 'beta', beta
   		write(*,*) 'helm_opal_scvh'
         stop 1
      end if

		
		contains
		
		subroutine get_helmeos(include_radiation)
		   logical, intent(in) :: include_radiation
		   
		   logical, parameter :: clip_to_table_boundaries = .false.
		   logical, parameter :: always_skip_elec_pos = .false.
		   
		   include 'formats.dek'
		   
			if (have_called_helm) return
			have_called_helm = .true.
     
         call helmeos2(temp, logT, den, logRho, X, abar, zbar, helm_res, 
     >         clip_to_table_boundaries, include_radiation, always_skip_elec_pos, ierr)
			if (ierr /= 0) then
			   write(*,*) 'failed in helmeos2'
			   write(*,1) 'temp', temp
			   write(*,1) 'logT', logT
			   write(*,1) 'den', den
			   write(*,1) 'logRho', logRho
			   write(*,1) 'Z', Z
			   write(*,1) 'X', X
			   write(*,1) 'abar', abar
			   write(*,1) 'zbar', zbar
			   write(*,*) 'clip_to_table_boundaries', clip_to_table_boundaries
			   write(*,*) 'include_radiation', include_radiation
				if (.not. include_radiation) then
				   write(*,*) 'try it with radiation included'
				   ierr = 0
               call helmeos2(temp, logT, den, logRho, X, abar, zbar, helm_res, 
     >            clip_to_table_boundaries, .true., always_skip_elec_pos, ierr)
               write(*,2) 'ierr with radiation', ierr
				end if
				write(*,*) 'stop in get_helmeos'
			   stop 1
			end if

			logPgas_helm = log10(helm_res(h_pgas))
			logE_helm = log10(helm_res(h_etot))
			logS_helm = log10(helm_res(h_stot))
			gamma1_helm = helm_res(h_gam1)
         gamma3_helm = helm_res(h_gam3)
			grad_ad_helm = helm_res(h_nabad)
			dE_dRho_helm = helm_res(h_ded)
			dS_dRho_helm = helm_res(h_dsd)
			dS_dT_helm = helm_res(h_dst)
			chiRho_helm = helm_res(h_chid)
			chiT_helm = helm_res(h_chit)
			Cv_helm = helm_res(h_cv)
			Cp_helm = helm_res(h_cp)

			if (helm_res(h_xne) > 0d0) then
				logNe_he lm = log10(helm_res(h_xne)) ! assuming complete ionization
			else
				logNe_helm = -99d0
			end if
			
			mu_helm = abar / (1 + zbar)
			eta_helm = helm_res(h_etaele)
			
			if (is_bad_num(logE_helm)) then
			   write(*,1) 'helm_res(h_etot)', helm_res(h_etot)
				write(*,*) 'stop in get_helmeos'
			   stop 1
			end if
			
		end subroutine get_helmeos

      end subroutine


      end module helm_opal_scvh_driver
