! ***********************************************************************
!
!   Copyright (C) 2014  Bill Paxton
!
!   This file is part of MESA.
!
!   MESA is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module eval_eosDE
      
		use eos_def
		use eos_lib
		use const_def
		use utils_lib
		use chem_def
		
      implicit none

      integer :: eos_handle

      integer, parameter :: species = 7
      integer, parameter :: h1=1, he4=2, c12=3, n14=4, o16=5, ne20=6, mg24=7
      integer, target :: chem_id_array(species)
      integer, pointer, dimension(:) :: chem_id, net_iso
      double precision :: xa(species)


      
      contains
      

      subroutine Init_Composition(X_in, Zinit_in, abar, zbar)
         use chem_lib
         double precision, intent(IN) :: X_in, Zinit_in
         double precision, intent(out) :: abar, zbar

         double precision, parameter :: Zfrac_C = 0.173312d0
         double precision, parameter :: Zfrac_N = 0.053177d0
         double precision, parameter :: Zfrac_O = 0.482398d0
         double precision, parameter :: Zfrac_Ne = 0.098675d0
         
         double precision :: frac, sumx, mass_correction
         double precision :: X, Y, z2bar, ye, Zinit, XC, XO
         
         chem_id => chem_id_array
         
         allocate(net_iso(num_chem_isos))         
         net_iso(:) = 0
         
         chem_id(h1) = ih1; net_iso(ih1) = h1
         chem_id(he4) = ihe4; net_iso(ihe4) = he4
         chem_id(c12) = ic12; net_iso(ic12) = c12
         chem_id(n14) = in14; net_iso(in14) = n14
         chem_id(o16) = io16; net_iso(io16) = o16
         chem_id(ne20) = ine20; net_iso(ine20) = ne20
         chem_id(mg24) = img24; net_iso(img24) = mg24
         
         X = X_in
         Zinit = Zinit_in
         XC = 0; XO = 0
         Y = 1 - (X + Zinit + XC + XO)
         if (Y < 0) then ! adjust XC and XO
            if (XC + XO <= 0) then
               write(*,*) 'bad args to Init_Composition'
               stop 1
            end if
            frac = (1 - X - Zinit) / (XC + XO)
            if (frac <= 0) stop 'bad args to Init_Composition'
            XC = frac*XC; XO = frac*XO
            Y = 1 - (X+Zinit+XC+XO)
            if (Y < -1d-10) then
               write(*,*) 'screw up in Init_Composition'
               stop 1
            end if
            if (Y < 0) Y = 0
         end if
      
         xa(h1) = X
         xa(he4) = Y
         xa(c12) = Zinit * Zfrac_C + XC
         xa(n14) = Zinit * Zfrac_N
         xa(o16) = Zinit * Zfrac_O + XO
         xa(ne20) = Zinit * Zfrac_Ne
         xa(species) = 1 - sum(xa(1:species-1))
         
         call basic_composition_info( &
               species, chem_id, xa, X, Y, abar, zbar, z2bar, ye, &
               mass_correction, sumx)
         
      end subroutine Init_Composition

      
      subroutine eos_4_DE( &
            logE, energy, logRho, Rho, &
            abar, zbar, X, Z, logT_guess, &
            logT, logPgas, logS, chiRho, chiT, &
            Cp, Cv, dE_dRho, dS_dT, dS_dRho, &
            mu, log_free_e, gamma1, gamma3, grad_ad, eta, &
            call_number, ierr)
         double precision, intent(in) :: &
            logE, energy, logRho, Rho, &
            abar, zbar, X, Z, logT_guess
         double precision, intent(out) :: &
            logT, logPgas, logS, chiRho, chiT, &
            Cp, Cv, dE_dRho, dS_dT, dS_dRho, &
            mu, log_free_e, gamma1, gamma3, grad_ad, eta
         integer, intent(in) :: call_number
         integer, intent(out) :: ierr

         double precision :: &
            other, other_tol, logT_tol, logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, &
            logT_result, res(num_eos_basic_results), d_eos_dlnd(num_eos_basic_results), &
            d_eos_dlnT(num_eos_basic_results), &
            d_dabar_const_TRho(num_eos_basic_results), &
            d_dzbar_const_TRho(num_eos_basic_results)
            
         integer:: which_other, max_iter, eos_calls
         
         include 'formats.dek'

         ierr = 0
         
         which_other = i_lnE
         other = logE*ln10
         other_tol = 1d-8*ln10
      
         max_iter = 100
            
         logT_tol = 1d-8
         logT_bnd1 = arg_not_provided
         logT_bnd2 = arg_not_provided
         other_at_bnd1 = arg_not_provided
         other_at_bnd2 = arg_not_provided

         call eosDT_get_T( &
            eos_handle, Z, X, abar, zbar, &
            species, chem_id, net_iso, xa, &
            logRho, which_other, other, &
            logT_tol, other_tol, max_iter, logT_guess, &
            logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, &
            logT_result, res, d_eos_dlnd, d_eos_dlnT, &
            d_dabar_const_TRho, d_dzbar_const_TRho, &
            eos_calls, ierr)

         if (ierr /= 0) then
            return
            
            write(*,*) 'eos_4_DE failed in eosDT_get_T'
            write(*,1) 'Z = ', Z
            write(*,1) 'X = ', X
            !write(*,1) 'abar = ', abar
            !write(*,1) 'zbar = ', zbar
            write(*,1) 'logRho = ', logRho
            write(*,1) 'logE = ', logE
            write(*,1) 'logT_guess = ', logT_guess
            write(*,*)
            return
            stop 1
         end if
         
         logT = logT_result         
         logPgas = res(i_lnPgas)/ln10
         logS = res(i_lnS)/ln10
         chiRho = res(i_chiRho)
         chiT = res(i_chiT)
         Cp = res(i_Cp)
         Cv = res(i_Cv)
         dE_dRho = res(i_dE_dRho)
         dS_dT = res(i_dS_dT)
         dS_dRho = res(i_dS_dRho)
         mu = res(i_mu)
         log_free_e = res(i_lnfree_e)/ln10
         gamma1 = res(i_gamma1)
         gamma3 = res(i_gamma3)
         grad_ad = res(i_grad_ad)
         eta = res(i_eta)
      
      end subroutine eos_4_DE

      
      end module eval_eosDE

