! ***********************************************************************
!
!   Copyright (C) 2014  Bill Paxton
!
!   This file is part of MESA.
!
!   MESA is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module create_eosDE_files
      
		use eos_def
		use eos_lib
		use const_def
		use chem_def
		
      implicit none



      logical, parameter :: use_shared_data_dir = .true.   ! MUST BE .true. FOR RELEASE
      !logical, parameter :: use_shared_data_dir = .false.

		integer, parameter :: version_number = 51 ! update this to force rebuilding of caches
		! update min_version in eosDE_load_tables to force rebuild of data files

      integer :: ix, ios, io_unit, handle
		double precision :: whichz

      ! control what is done by saving the Z in whichz.txt and, if necessary, setting the Xs below
      
      integer, parameter :: num_Xs = 5, num_Zs = 1 ! currently limited to a single Z data file per run
      double precision :: Xs(num_Xs)
      
      
      contains
      
      subroutine do_create_eosDE_files
      
   	   Xs(1:num_Xs) = (/ 0.00d0, 0.20d0, 0.40d0, 0.60d0, 0.80d0 /)
   		io_unit = 40
         open(UNIT=io_unit, FILE=trim("whichz.txt"), ACTION='READ', STATUS='OLD', IOSTAT=ios)
   		if (ios /= 0) call do_stop('failed to open whichz.txt')
   		read(io_unit, fmt=*, iostat=ios) whichz
   		if (ios /= 0) call do_stop('failed to read Z from whichz.txt')
         do ix = num_Xs, 1, -1
            call make_eosDE_files(whichz, Xs(ix))
         end do
   		if (whichz == 0) call make_eosDE_files(whichz, 1d0)
   		
      end subroutine do_create_eosDE_files
      
      
      subroutine make_eosDE_files(Z, X)
         use eval_eosDE
         double precision, intent(in) :: Z, X

         integer :: ierr, i, j, num_logVs, num_logEs, call_number
         character (len=256) :: dir, fname
         double precision :: &
            abar, zbar, logV_min, logV_max, del_logV, &
            logE_min, logE_max, del_logE, &
            logV, logRho, Rho, logT_guess, logT_guess1, logE, energy, &
            logT, logPgas, logS, chiRho, chiT, &
            Cp, Cv, dE_dRho, dS_dT, dS_dRho, &
            mu, log_free_e, gamma1, gamma3, grad_ad, eta
            
         include 'formats.dek'

   		ierr = 0

         dir = 'data' ! where to put the new data files

         call Init_Composition(X, Z, abar, zbar)
             		
         logV_min = 3.2d0 ! smaller logV_min runs into problems at low logE
         logV_max = 11.0d0 ! larger logV_Max runs into problems at high logE
         del_logV = 0.02d0
      
         logE_min = 11d0
         logE_max = 16d0
         del_logE = 0.02d0
            
         num_logVs = 1 + int((logV_max - logV_min) / del_logV)
         num_logEs = 1 + int((logE_max - logE_min) / del_logE)
      
         if (X < 0.005) then
            write(fname,'(a,a,i1,a)') trim(dir), &
               '/eosDE_data/mesa-eosDE_0', floor(100d0*Z + 0.5), 'z00x.data'
         else if (X < 1) then
            write(fname,'(a,a,i1,a,i2,a)') trim(dir), '/eosDE_data/mesa-eosDE_0',  &
         		floor(100d0*Z + 0.5), 'z', floor(100d0*X + 0.5), 'x.data'
         else
            fname = trim(dir) // '/eosDE_data/mesa-eosDE_00z100x.data'
         end if
      
         write(*,*) trim(fname)

         open(unit=io_unit, file=trim(fname), action='write', status='replace', iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to open ' // trim(fname)
            stop 1
         end if
      
         write(io_unit,'(99(a14))') &
            'version', 'X', 'Z', 'num logEs', 'logE min', 'logE max', 'del logE',  &
         	'num logVs', 'logV min', 'logV max', 'del logV'
      
         write(io_unit,'(i14,2f14.4,2(i10,4x,3(f14.4)))') &
               version_number, X, Z, num_logEs, logE_min, logE_max, del_logE, &
         		num_logVs, logV_min, logV_max, del_logV
         
         call_number = 0
         logT_guess1 = 3
         do i = 1, num_logVs
            logV = logV_min + (i-1) * del_logV
            
            if (mod(i,10) == 0) write(*,*) 'logV = ', i

            write(io_unit,'(/,7x,a)') 'logV = logRho - 0.7*logE + 20' 
            write(io_unit,'(2x,f14.6/)') logV
            write(io_unit,'(a7,10x,99(a11,7x))') 'logE', &
               'logT', 'logPgas', 'logS', 'chiRho', 'chiT', &
               'Cp', 'Cv', 'dE_dRho', 'dS_dT', 'dS_dRho', &
               'mu', 'log_free_e', 'gamma1', 'gamma3', 'grad_ad', 'eta'

            logT_guess = logT_guess1
            do j = 1, num_logEs
               logE = logE_min + (j-1) * del_logE
               energy = 10**logE
               logRho = logV + 0.7*logE - 20d0
               Rho = 10**logRho
               
               ierr = 0
               call_number = call_number + 1
               
               call eos_4_DE( &
                  logE, energy, logRho, Rho, &
                  abar, zbar, X, Z, logT_guess, &
                  logT, logPgas, logS, chiRho, chiT, &
                  Cp, Cv, dE_dRho, dS_dT, dS_dRho, &
                  mu, log_free_e, gamma1, gamma3, grad_ad, eta, &
                  call_number, ierr)
   				if (ierr /= 0) then
   				   logT = 0
   				   logPgas = -99
   				   logS = -99
   				   chiRho = -99
   				   chiT = -99
   				   Cp = -99
   				   Cv = -99
   				   dE_dRho = -99
   				   dS_dT = -99
   				   dS_dRho = -99
   				   mu = -99
   				   log_free_e = -99
   				   gamma1 = -99
   				   gamma3 = -99
   				   grad_ad = -99
   				   eta = -99
   					write(*,*) 'bad logE logV logRho X Z logT_guess', &
   					   logE, logV, logRho, X, Z, logT_guess
   					stop
   					!call do_stop('failed in eos_4_DE')
   				end if
   				
   				if (j == 1) logT_guess1 = logT
   				logT_guess = logT

               write(io_unit,'(f11.6,99(e18.9))') logE, &
                     logT, logPgas, logS, chiRho, chiT, &
                     Cp, Cv, dE_dRho, dS_dT, dS_dRho, &
                     mu, log_free_e, gamma1, gamma3, grad_ad, eta
                         
            end do
         
            write(io_unit,*)
         
         end do
      
         write(io_unit,*)
         write(io_unit,*)

         close(io_unit)
      
         write(*,*)

      end subroutine make_eosDE_files


      subroutine setup_eos
         use const_lib
         use chem_lib
         use crlibm_lib
         use eval_eosDE, only: eos_handle

         character (len=256) :: my_mesa_dir, data_dir, eos_file_prefix
         integer :: info
         logical, parameter :: use_cache = .true.
         
         info = 0
         eos_file_prefix = 'mesa'
         
         my_mesa_dir = '../..'         
         call const_init(my_mesa_dir,info)     
      	if (info /= 0) then
      	   write(*,*) 'const_init failed'
      	   stop 1
      	end if        
         
         call crlibm_init

         if (use_shared_data_dir) then
            data_dir = '../../data' ! test using shared data
         else
            data_dir = '../data' ! test using local data
            
            write(*,*)
            write(*,*) 'TESTING WITH LOCAL DATA'
            write(*,*)
            
         end if
         
      	call chem_init('isotopes.data', info)
      	if (info /= 0) then
      	   write(*,*) 'chem_init failed'
      	   stop 1
      	end if

         call eos_init(eos_file_prefix, '', '', '', use_cache, info)
         if (info /= 0) then
            write(*,*) 'failed in eos_init'
            stop 1
         end if

         eos_handle = alloc_eos_handle(info)
         if (info /= 0) then
            write(*,*) 'failed in alloc_eos_handle'
            stop 1
         end if
      
      end subroutine setup_eos


      subroutine do_stop(str)
         character (len=*) :: str
         write(*,*) trim(str)
         stop 1
      end subroutine do_stop
      
      
      end module create_eosDE_files
