! ***********************************************************************
!
!   Copyright (C) 2013  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************


      module crlibm_lib
      use const_def, only: dp, ln10, pi
      implicit none
      
      contains
      
#ifdef offload
      !dir$ options /offload_attribute_target=mic
#endif
      
      subroutine crlibm_init
         !write(*,*) 'USING STUB FOR CRLIB_M'
      end subroutine crlibm_init
      
      
      real(dp) function ln10_cr() result(res)
         res = ln10
      end function ln10_cr
      
      
      real(dp) function log_cr(x) result(res)
         real(dp), intent(in) :: x
         res = log(x) ! call log_rz(x,res)
      end function log_cr
      
      real(dp) function log10_cr(x) result(res) ! log base 10
         real(dp), intent(in) :: x
         res = log10(x) ! call log10_rz(x,res)
      end function log10_cr
      
      real(dp) function log1p_cr(x) result(res) ! log1p = log(1 + x)
         real(dp), intent(in) :: x
         res = log(1d0 + x) ! call log1p_rz(x,res)
      end function log1p_cr
      
      real(dp) function log2_cr(x) result(res) ! log base 2
         real(dp), intent(in) :: x
         res = log(x)/log(2d0) ! call log2_rz(x,res)
      end function log2_cr
      
      real(dp) function safe_log_cr(x) result(res)
         real(dp), intent(in) :: x
         res = log(max(x,1d-99)) ! call log_rz(max(x,1d-99),res)
      end function safe_log_cr
      
      real(dp) function safe_log10_cr(x) result(res) ! log base 10
         real(dp), intent(in) :: x
         res = log10(max(x,1d-99)) ! call log10_rz(max(x,1d-99),res)
      end function safe_log10_cr
      
      real(dp) function exp_cr(x) result(res) ! E^x
         real(dp), intent(in) :: x
         res = exp(x) ! call exp_rz(x,res)
      end function exp_cr
      
      real(dp) function exp10_cr(x) result(res) ! 10^x
         real(dp), intent(in) :: x
         res = 10d0**x ! call exp10_rz(x,res)
      end function exp10_cr
      
      real(dp) function expm1_cr(x) result(res) ! expm1 = E^x - 1
         real(dp), intent(in) :: x
         res = exp(x) - 1d0 ! call expm1_rz(x,res)
      end function expm1_cr
      
      real(dp) function pow_cr(x,y) result(res) ! x^y
         real(dp), intent(in) :: x,y
         res = x**y ! res = exp_cr(log_cr(x)*y)
      end function pow_cr
      
      real(dp) function powi_cr(x,i) result(res) ! x^i
         real(dp), intent(in) :: x
         integer, intent(in) :: i
         res = x**i ! res = exp_cr(log_cr(x)*dble(i))
      end function powi_cr
      
      real(dp) function pow2(x) result(res) ! x^2
         real(dp), intent(in) :: x
         res = x*x
      end function pow2
      
      real(dp) function pow3(x) result(res) ! x^3
         real(dp), intent(in) :: x
         res = x*x*x
      end function pow3
      
      real(dp) function pow4(x) result(res) ! x^4
         real(dp), intent(in) :: x
         res = x*x*x*x
      end function pow4
      
      real(dp) function pow5(x) result(res) ! x^5
         real(dp), intent(in) :: x
         res = x*x*x*x*x
      end function pow5
      
      real(dp) function pow6(x) result(res) ! x^6
         real(dp), intent(in) :: x
         res = x*x*x*x*x*x
      end function pow6
      
      real(dp) function pow7(x) result(res) ! x^7
         real(dp), intent(in) :: x
         res = x*x*x*x*x*x*x
      end function pow7
      
      real(dp) function pow8(x) result(res) ! x^8
         real(dp), intent(in) :: x
         res = x*x*x*x*x*x*x*x
      end function pow8

      
      real(dp) function cos_cr(x) result(res)
         real(dp), intent(in) :: x
         res = cos(x) ! call cos_rz(x,res)
      end function cos_cr
      
      real(dp) function sin_cr(x) result(res)
         real(dp), intent(in) :: x
         res = sin(x) ! call sin_rz(x,res)
      end function sin_cr
      
      real(dp) function tan_cr(x) result(res)
         real(dp), intent(in) :: x
         res = tan(x) ! call tan_rz(x,res)
      end function tan_cr
      
      
      real(dp) function cospi_cr(x) result(res) ! cosine of pi times x
         real(dp), intent(in) :: x
         res = cos(x*pi) ! call cospi_rz(x,res)
      end function cospi_cr
      
      real(dp) function sinpi_cr(x) result(res) ! sine of pi times x
         real(dp), intent(in) :: x
         res = sin(x*pi) ! call sinpi_rz(x,res)
      end function sinpi_cr
      
      real(dp) function tanpi_cr(x) result(res) ! tangent of pi times x
         real(dp), intent(in) :: x
         res = tan(x*pi) ! call tanpi_rz(x,res)
      end function tanpi_cr
      
      
      real(dp) function acos_cr(x) result(res)
         real(dp), intent(in) :: x
         res = acos(x*pi) ! call acos_rz(x,res)
      end function acos_cr
      
      real(dp) function asin_cr(x) result(res)
         real(dp), intent(in) :: x
         res = asin(x*pi) ! call asin_rz(x,res)
      end function asin_cr
      
      real(dp) function atan_cr(x) result(res)
         real(dp), intent(in) :: x
         res = atan(x*pi) ! call atan_rz(x,res)
      end function atan_cr
      
      
      real(dp) function acospi_cr(x) result(res) ! acos(x)/pi
         real(dp), intent(in) :: x
         res = acos(x)/pi ! call acospi_rz(x,res)
      end function acospi_cr
      
      real(dp) function asinpi_cr(x) result(res) ! asin(x)/pi
         real(dp), intent(in) :: x
         res = asin(x)/pi ! call asinpi_rz(x,res)
      end function asinpi_cr
      
      real(dp) function atanpi_cr(x) result(res) ! atan(x)/pi
         real(dp), intent(in) :: x
         res = atan(x)/pi ! call atanpi_rz(x,res)
      end function atanpi_cr
      
      
      real(dp) function cosh_cr(x) result(res)
         real(dp), intent(in) :: x
         res = cosh(x) ! call cosh_rz(x,res)
      end function cosh_cr
      
      real(dp) function sinh_cr(x) result(res)
         real(dp), intent(in) :: x
         res = sinh(x) ! call sinh_rz(x,res)
      end function sinh_cr
      
      real(dp) function tanh_cr(x) result(res)
         real(dp), intent(in) :: x
         res = tanh(x) ! call tanh_rz(x,res)
      end function tanh_cr
      
      
      ! the following are not part of the crlibm, but are included here for convenience.
                  
            
      
      subroutine str_to_vector(str, vec, n, ierr)
         character (len=*), intent(in) :: str
         real(dp), pointer, intent(out) :: vec(:)
         integer, intent(out) :: n, ierr
         integer :: maxlen, i, j, k, l
         logical, parameter :: dbg = .false.
         include 'formats'
         maxlen = size(vec,dim=1)
         if (dbg) write(*,*) 'maxlen', maxlen
         n = 0
         ierr = 0
         l = len_trim(str)
         if (dbg) write(*,*) 'l', l
         j = 1
         do i=1,maxlen
            do while (j < l .and. str(j:j) == ' ')
               j = j+1
            end do
            k = j
            do while (k < l .and. str(k:k) /= ' ')
               k = k+1
            end do
            call str_to_double(str(j:k),vec(i),ierr)
            if (ierr /= 0) then
               if (dbg) write(*,'(a)') 'str_to_double failed', j, k
               if (dbg) write(*,'(a)') 'str_to_double failed for "' // str(j:k) // '"'
               return
            end if
            n = i
            if (dbg) write(*,2) 'vec(i)', i, vec(i)
            if (k == l) return
            j = k+1
         end do
         if (dbg) write(*,'(a)') 'str_to_vector failed', n, maxlen
      end subroutine str_to_vector
      
      
      subroutine str_to_double(str,x,ierr)
         character (len=*), intent(in) :: str
         real(dp), intent(out) :: x
         integer, intent(out) :: ierr
         
         character (len = 26) :: xstr
         integer :: l, j, i, k, kk
         real(dp) :: f, z, pow_ten
         logical :: negative, neg_exp
         logical, parameter :: dbg = .false.
         
         include 'formats'
         if (dbg) write(*,*) 'str_to_double "' // trim(str) // '"'
         
         ierr = 0
         l = len_trim(str)
         if (l == 0) then
            x = 0; return
         end if
         j = 1
         do while (str(j:j) == ' ')
            j = j+1
         end do
         negative = (str(j:j) == '-')
         if (negative) j = j+1
         x = 0d0
         do while (j <= l) ! before decimal point
            i = ichar(str(j:j)) - ichar('0')
            if (i < 0 .or. i > 9) exit
            x = x*10d0 + i
            j = j + 1
         end do

         !call double_to_str_1pd26pt16(x,xstr)
         !write(*,'(i3,a26)') j, xstr
         if (dbg) write(*,'(i3,1pd26.16)') j, x
!-1.6348128874011620E-23
!1234567890123456789012345         
         if (str(j:j) == '.' .and. j < l) then ! after decimal point
            k = j+1
            do while (.true.)
               if (dbg) write(*,2) 'str(k:k) ' // trim(str(k:k)), k
               i = ichar(str(k:k)) - ichar('0')
               if (i < 0 .or. i > 9) then ! not a digit
                  k = k-1; exit
               end if
               if (k == l) exit
               k = k+1
            end do
            kk = k ! last digit after decimal point
            f = 0d0
            do while (kk > j)
               if (dbg) write(*,2) 'str(kk:kk) ' // trim(str(kk:kk)), kk
               i = ichar(str(kk:kk)) - ichar('0')
               if (i < 0 .or. i > 9) exit
               f = f*0.1d0 + dble(i)
               kk = kk - 1
            end do
            x = x + f*0.1d0
            j = k+1
         end if
         if (dbg) write(*,'(/,i3,1pd26.16)') j, x
         
         i = 0
         if (j < l) then
            neg_exp = .false.
            if (str(j:j) == 'd' .or. str(j:j) == 'D' .or. &
                  str(j:j) == 'e' .or. str(j:j) == 'E') then
               j = j+1
               if (str(j:j) == '+') then
                  j = j+1
               else if (str(j:j) == '-') then
                  j = j+1
                  neg_exp = .true.
               end if 
            else if (str(j:j) == '+') then
               j = j+1
            else if (str(j:j) == '-') then
               j = j+1
               neg_exp = .true.
            else
               ierr = -1; return
            end if
            if (j <= l) then
               if (dbg) write(*,*) 'exponent "' // trim(str(j:l)) // '"'
               read(str(j:l),*,iostat=ierr) i
            else
               ierr = -1
            end if
            if (ierr /= 0) return
            if (neg_exp) i = -i
         end if
         
         !x = x*(10d0**i)
         if (i /= 0) then
            pow_ten = exp_cr(ln10_cr()*dble(i))
            !call double_to_str_1pd26pt16(pow_ten,xstr)
            !write(*,'(i3,a26)') -1, xstr
            if (dbg) write(*,'(i3,1pd26.16)') -1, pow_ten
            x = x*pow_ten
         end if

         !call double_to_str_1pd26pt16(x,xstr)
         !write(*,'(i3,a26)') j, xstr
         if (dbg) write(*,'(i3,1pd26.16)') j, x
         
         if (negative .and. x /= 0d0) x = -x

         !call double_to_str_1pd26pt16(x,xstr)
         !write(*,'(i3,a26)') j, xstr
         if (dbg) write(*,'(i3,1pd26.16)') j, x
         
      end subroutine str_to_double
      
      
!    0.0000000000000000D+00
!12345678901234567890123456     
      subroutine double_to_str_1pd26pt16(x,str)
         real(dp), intent(in) :: x
         integer, parameter :: l=26
         character(len=l), intent(out) :: str
         real(dp) :: y, lgy, pow_ten
         integer :: i, absi, j, iy
         character (len=20) :: buf
         if (x == 0d0) then
            str = '    0.0000000000000000D+00'
            return
         end if
         
         
         call split_double(x, y, i)
         if (y < 0d0) then
            str(1:4) = '   -'
            y = -y
         else
            str(1:4) = '    '
         end if

         
         iy = floor(y) ! y > 0
         write(buf,'(i1)') iy
         str(5:5) = buf(1:1)
         str(6:6) = '.'
         do j=7,22
            y = (y - iy)*10d0
            iy = floor(y)
            write(buf,'(i1)') iy
            str(j:j) = buf(1:1)
         end do
         
         absi = abs(i)
         if (absi < 100) then
            if (i >= 0) then
               str(23:24) = 'D+'
            else
               str(23:24) = 'D-'
            end if
            if (absi < 10) then
               write(buf,'(i1)') absi
               str(25:25) = '0'
               str(26:26) = buf(1:1)
            else
               write(buf,'(i2)') absi
               str(25:26) = buf(1:2)
            end if
         else
            if (i >= 0) then
               str(23:23) = '+'
            else
               str(23:23) = '-'
            end if
            write(buf,'(i3)') absi
            str(24:26) = buf(1:3)
         end if
         
      end subroutine double_to_str_1pd26pt16
      
      
      subroutine split_double(x, y, i)
         use utils_lib, only: is_bad_num
         real(dp), intent(in) :: x
         real(dp), intent(out) :: y
         integer, intent(out) :: i
         ! if x==0 then y = 0 and i = 0
         ! else 1.0 <= abs(y) < 10.0
         ! and x = y * 10d0**i
         real(dp) :: z, lgy, pow_ten
         include 'formats'
         if (is_bad_num(x) .or. x == 0d0) then
            y = x; i = 0; return
         end if
         y = abs(x)
         lgy = log10_cr(y)
         i = floor(lgy)
         pow_ten = exp_cr(ln10_cr()*dble(i))
         y = y/pow_ten
         y = sign(y,x)
      end subroutine split_double
      
#ifdef offload
      !dir$ end options
#endif

      end module crlibm_lib
