! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module colors_def
      implicit none
		
		! color indices are differences in magnitudes in different wavelength bands
		! as a reminder for non-experts like myself, here's how it goes
		!
		! msun := apparent magnitude of sun is -26.81
		! Fsun := solar flux at 1AU is 1.36e6 erg/s/cm^2
		!
		! "apparent magnitude" m of star with flux F is m = msun - 2.5 log10(F/Fsun)
		! "absolute magnitude" M for star of apparent magnitude m at distance d is M = m - 5 log(d/d0)
		! 		where the standard distance d0 is 10pc.
		! 		i.e., absolute magnitude is what the apparent magnitude would be if star were at 10 parsecs.
		!
		! thus absolute magnitude of sun is about 4.75
		! 
		! "bolometric magnitude" = absolute magnitude using flux integrated over all wavelengths
		! 		can be derived from the current stellar luminosity using the equation
		! 		log(Lstar/Lsun) = (Mbol_sun - Mbol_star)/2.5 using Mbol_sun = 4.75 (LCB)
		!
		! "visual magnitude" = absolute magnitude only using flux in visible wavelengths
		!      more precisely, this is magnitude as measured with filter centered at 5500A, 890A width.
		!
		! "bolometric correction" = bolometric magnitude minus visual magnitude
		!      for the sun, the bolometric correction is about -0.11
		!      thus visual magnitude of sun is about 4.86 = Mbol_sun - BC_sun = 4.75 - (-0.11)
		!
		! in order of increasing wavelength, the "color" magnitudes are as follows:
		!
		! "U" is the ultraviolet magnitude, center at 365nm.
		! "B" is the        blue magnitude, center at 440nm.
		! "V" is the      visual magnitude, center at 550nm.
		! "R" is the         red magnitude, center at 600nm.
		! "I" is the   infra-red magnitude, center at 800nm.
		
		! in addition, longer wavelength "colors" have been defined as well
		! by order of increasing wavelength, these are J, H, K, L, and M.
		
		! "color index" is the difference between 2 color magnitudes
		! for example, B-V is colors_B - colors_V
		! smaller B-V means larger brightness in blue band compared to visual band, means bluer star.


  		! color magnitude data from Lejeune, Cuisinier, Buser (1998) A&AS 130, 65-75. [LCB]
		! the coverage is approximately Teff from 50,000K to 2000K, log g 5.5 to -1.02, [Fe/H} 1.0 to -5.0
		!
		! but not all combination of these are actually represented in the tables.  
		! the current implementation limits the given arguments to the actual range in the tables.
		! and it does a simple linear interpolation between tabulated values.

		! BTW: they use [Fe/H] as a parameter;
		! the evolution code uses log10(Z/Zsun) as an approximation for this.
				
#ifdef offload
      !dir$ options /offload_attribute_target=mic
#endif
		
		integer, parameter :: bol  = 1		! Bolometric magnitude (e.g., 4.7846 for sun)
		integer, parameter :: bcv  = 2		! Bolometric correction in V (energy-weighted)
		integer, parameter :: umb  = 3		! (U-B) color (energy-weighted)
		integer, parameter :: bmv  = 4		! (B-V) color
		integer, parameter :: vmr  = 5		! (V-R) color
		integer, parameter :: vmi  = 6		! (V-I) color
		integer, parameter :: vmk  = 7		! (V-K) color
		integer, parameter :: rmi  = 8		! (R-I) color
		integer, parameter :: imk  = 9		! (I-K) color
		integer, parameter :: jmh  = 10		! (J-H) color
		integer, parameter :: hmk  = 11		! (H-K) color
		integer, parameter :: kml  = 12		! (K-L) color
		integer, parameter :: jmk  = 13		! (J-K) color
		integer, parameter :: jml  = 14		! (J-L) color
		integer, parameter :: jmlp = 15		! (J-L') color
		integer, parameter :: kmm  = 16		! (K-M) color
		
		integer, parameter :: n_colors  = 16
		
		character (len=8) :: colors_name(n_colors) ! e.g., colors_name(bmv) is 'B-V'
		
		
		
		! THE FOLLOWING ARE PRIVATE DEFS -- NOT FOR USE BY CLIENTS
		
		type :: lgz_list ! sorted in decreasing order of lgz
			double precision :: lgz ! [Fe_H]
			type (lgz_list), pointer :: nxt
			double precision :: colors(n_colors)
		end type

		type :: lgt_list ! sorted in decreasing order of lgt
			double precision :: lgt ! logTeff
			type (lgt_list), pointer :: nxt
			type (lgz_list), pointer :: zlist
		end type

		type :: lgg_list ! sorted in decreasing order of lgg
			double precision :: lgg ! log g
			type (lgg_list), pointer :: nxt
			type (lgt_list), pointer :: tlist
		end type
		
		type (lgg_list), pointer :: ghead
		
		! these are only for copying from host to target
		type (lgg_list), pointer :: ghead_temp
		type (lgt_list), pointer :: thead_temp
		type (lgz_list), pointer :: zhead_temp
		
		
		
		contains
		
		
		subroutine init_colors_names
			colors_name(bol)  = 'bol'		! Bolometric magnitude
			colors_name(bcv)  = 'bcv'		! Bolometric correction in V (energy-weighted)
			colors_name(umb)  = 'U-B'		! (U-B) color (energy-weighted)
			colors_name(bmv)  = 'B-V'		! (B-V) color
			colors_name(vmr)  = 'V-R'		! (V-R) color
			colors_name(vmi)  = 'V-I'		! (V-I) color
			colors_name(vmk)  = 'V-K'		! (V-K) color
			colors_name(rmi)  = 'R-I'		! (R-I) color
			colors_name(imk)  = 'I-K'		! (I-K) color
			colors_name(jmh)  = 'J-H'		! (J-H) color
			colors_name(hmk)  = 'H-K'		! (H-K) color
			colors_name(kml)  = 'K-L'		! (K-L) color
			colors_name(jmk)  = 'J-K'		! (J-K) color
			colors_name(jml)  = 'J-L'		! (J-L) color
			colors_name(jmlp) = 'J-Lp'	   ! (J-L') color
			colors_name(kmm)  = 'K-M'		! (K-M) color
		end subroutine init_colors_names
		
		
#ifdef offload
		subroutine copy_to_target
         !dir$ offload target(mic)
         call init_for_copy_to_target
         call copy_glist(ghead)
         !dir$ offload target(mic)
         call finish_copy_to_target
		end subroutine copy_to_target
		
		recursive subroutine copy_glist(ghead)
		   type (lgg_list), pointer :: ghead
			double precision :: lgg ! log g
		   if (associated(ghead% nxt)) call copy_glist(ghead% nxt)
		   if (associated(ghead% tlist)) call copy_tlist(ghead% tlist)
		   lgg = ghead% lgg
         !dir$ offload target(mic) in(lgg)
         call new_ghead(lgg)
		end subroutine copy_glist
		
		recursive subroutine copy_tlist(thead)
		   type (lgt_list), pointer :: thead
			double precision :: lgt ! [Fe_H]
		   if (associated(thead% nxt)) call copy_tlist(thead% nxt)
		   if (associated(thead% zlist)) call copy_zlist(thead% zlist)
		   lgt = thead% lgt
         !dir$ offload target(mic) in(lgt)
         call new_thead(lgt)
		end subroutine copy_tlist
		
		recursive subroutine copy_zlist(zhead)
		   type (lgz_list), pointer :: zhead
			double precision :: lgz ! [Fe_H]
			double precision :: colors(n_colors)
		   integer :: i
		   if (associated(zhead% nxt)) call copy_zlist(zhead% nxt)
		   lgz = zhead% lgz
		   do i=1,n_colors
		      colors(i) = zhead% colors(i)
		   end do
         !dir$ offload target(mic) in(lgz, colors)
         call new_zhead(lgz, colors)
		end subroutine copy_zlist
		
		! the following run on the target only
		subroutine init_for_copy_to_target
		   nullify(ghead_temp, thead_temp, zhead_temp)
		end subroutine init_for_copy_to_target
		
		subroutine finish_copy_to_target
		   ghead => ghead_temp
		   nullify(ghead_temp)
		   if (associated(thead_temp)) then
		      write(*,*) 'colors_init, finish_copy_to_target: ERROR thead_temp'
		      flush(6)
		      stop 1
		   end if
		   if (associated(zhead_temp)) then
		      write(*,*) 'colors_init, finish_copy_to_target: ERROR zhead_temp'
		      flush(6)
		      stop 1
		   end if
		end subroutine finish_copy_to_target
		
		subroutine new_zhead(lgz,colors)
			double precision, intent(in) :: lgz ! [Fe_H]
			double precision, intent(in) :: colors(n_colors)
		   type (lgz_list), pointer :: zhead
		   integer :: i
		   allocate(zhead)
		   zhead% lgz = lgz
		   do i=1,n_colors
		      zhead% colors(i) = colors(i)
		   end do
		   zhead% nxt => zhead_temp
		   zhead_temp => zhead
		end subroutine new_zhead
		
		subroutine new_thead(lgt)
			double precision, intent(in) :: lgt ! logTeff
		   type (lgt_list), pointer :: thead
		   allocate(thead)
		   thead% lgt = lgt
		   thead% zlist => zhead_temp
		   nullify(zhead_temp)
		   thead% nxt => thead_temp
		   thead_temp => thead
		end subroutine new_thead
		
		subroutine new_ghead(lgg)
			double precision, intent(in) :: lgg ! log g
		   type (lgg_list), pointer :: ghead
		   allocate(ghead)
		   ghead% lgg = lgg
		   ghead% tlist => thead_temp
		   nullify(thead_temp)
		   ghead% nxt => ghead_temp
		   ghead_temp => ghead
		end subroutine new_ghead
#endif

		
#ifdef offload
      !dir$ end options
#endif

      end module colors_def

