! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton, Ed Brown
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************
 
      module nuclide_set_mod
      use chem_def	
      
      implicit none

      contains

      
      subroutine read_ZA(filename, Z, A, nnuc, ierr)
         use utils_lib
         use utils_def, only: eof_token
         character (len=*), intent(in) :: filename
         integer, dimension(:), pointer :: Z, A ! will be allocated
         integer, intent(out) :: nnuc, ierr
         
         integer :: iounit, n, i, t, id, pass, vals(3), j
      	integer :: ilo, ihi
         character (len=256) :: buffer, string

         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         
         nullify(Z)
         nullify(A)
         
   	   ilo=0; ihi=0

         do pass = 1, 2
         
            if (pass == 2) then ! allocate the arrays
               nnuc = ihi
               allocate(Z(nnuc), A(nnuc), stat=ierr)
               if (failed('allocate')) return
               ilo=0; ihi=0
               !write(*,*) 'nnuc', nnuc
            end if
         
            call openfile(ierr)
            if (ierr /= 0) return

            n = 0
            i = 0
            
   trip_loop: do ! read triples Z, A_lo, A_hi
               
               do j=1,3
                  t = token(iounit, n, i, buffer, string)
                  if (t == eof_token) exit trip_loop
                  read(string,fmt=*,iostat=ierr) vals(j)
                  if (failed('bad data? error in reading')) then
                     write(*,*) 'pass', pass
                     write(*,*) '   n', n
                     write(*,*) '   i', i
                     write(*,*) 'buffer <' // trim(buffer) // '>'
                     write(*,*) 'string <' // trim(string) // '>'
                     return
                  end if
               end do
               
               call setZA(vals(1),vals(2),vals(3))

            end do trip_loop
            
            close(iounit)
            
         end do
         
         call free_iounit(iounit)

         
         contains
         
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            if (ierr == 0) then
               failed = .false.
               return
            end if
            failed = .true.
            close(iounit)
            call free_iounit(iounit)
            write(*,*) trim(str) // ' ' // trim(filename)
         end function failed
         
         subroutine openfile(ierr)
            integer, intent(out) :: ierr
            !write(*,*) 'open ' // trim(filename)
            open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
            if (failed('failed to open file')) return
         end subroutine openfile
      
      
         subroutine setZA(zee,alo,ahi)
            integer, intent(in) :: zee, alo, ahi
            integer :: i
            include 'formats.dek'
         	ilo = ihi+1; ihi=ilo+ahi-alo
         	if (pass == 1) return
         	if (ihi > nnuc) then
         	   write(*,2) 'ihi', ihi
         	   write(*,2) 'nnuc', nnuc
         	   write(*,*) 'bug in read_ZA'
         	   stop 1
         	end if
         	Z(ilo:ihi) = zee
         	A(ilo:ihi) = [(i, i=alo,ahi)]
         end subroutine setZA


      end subroutine read_ZA
      
      
      subroutine do_composition_info(Y, nuclides, abar, zbar, z2bar, ye)
      	double precision, intent(in) :: Y(:)
      	type(nuclide_data), intent(in) :: nuclides
         double precision, intent(out) :: abar, zbar, z2bar, ye
         double precision :: Z, yzsum, yz2sum, ysum
         integer :: i
         yzsum=0; yz2sum=0; ysum=0
         do i = 1, nuclides% nnuclides
            ysum = ysum + Y(i)
            Z = nuclides% Z(i)
            yzsum = yzsum + Y(i)*Z
            yz2sum = yz2sum + Y(i)*Z**2
         end do
         abar = 1d0/ysum
         zbar = abar*yzsum
         z2bar = abar*yz2sum
         ye = zbar/abar
      end subroutine do_composition_info
      

      function rank_in_set(iso, set)
      	character(len=iso_name_length), intent(in) :: iso
      	type(nuclide_set), dimension(:), intent(in) :: set
      	integer :: rank_in_set
      	integer :: low, high, mid, i
      	integer, parameter :: max_cycles = 20
      	low = 1
      	high = size(set)
      	if (adjustl(iso) < adjustl(set(low)% nuclide) .or. adjustl(iso) > adjustl(set(high)% nuclide)) then
      		rank_in_set = nuclide_not_found
      		return
      	end if	
      	do i = 1, max_cycles
      		if (high-low <=1) then
      			if (adjustl(iso) == adjustl(set(high)% nuclide)) then
      				rank_in_set = set(high)% rank
      			else if (adjustl(iso) == adjustl(set(low)% nuclide)) then
      				rank_in_set = set(low)% rank
      			else
      				rank_in_set = nuclide_not_found
      			end if
      			return
      		end if
      		mid = (high+low)/2
      		if (adjustl(iso) <= adjustl(set(mid)% nuclide)) then
      			high = mid
      		else if (adjustl(iso) > adjustl(set(mid)% nuclide)) then
      			low = mid
      		end if
      	end do
      	rank_in_set = nuclide_not_found
      end function rank_in_set
      

      subroutine sort_nuclide_set(set)
      	type(nuclide_set), dimension(:), intent(inout) :: set	
      	integer :: n, i, ir, j, l
      	type(nuclide_set) :: ts
	
      	n = size(set)
      	if (size(set) < 2) return
      	l = n/2+1
      	ir = n
      	do
      		if (l > 1) then
      			l = l-1
      			ts = set(l)
      		else
      			ts = set(ir)
      			set(ir) = set(1)
      			ir = ir-1
      			if (ir == 1) then
      				set(1) = ts
      				return
      			end if
      		end if
      		i = l
      		j = l+l
      		do
      			if (j > ir) exit
      			if (j < ir) then
      				if (compare_lt(set(j), set(j+1))) j = j+1
      			end if
      			if (compare_lt(ts, set(j))) then
      				set(i) = set(j)
      				i = j
      				j = j+j
      			else
      				j = ir + 1
      			end if
      		end do
      		set(i) = ts
      	end do

      	contains
      	
      	logical function compare_lt(a, b)
      		type(nuclide_set), intent(in) :: a, b
      		compare_lt = (adjustl(a% nuclide) < adjustl(b% nuclide))
      	end function compare_lt
      	
      end subroutine sort_nuclide_set
      

      subroutine read_nuclide_set(filename, set, err)
	
      	character(len=*), intent(in) :: filename
      	type(nuclide_set), dimension(:), pointer, intent(out) :: set
      	integer, intent(out) :: err
      	integer, parameter :: iounit=10
      	integer :: ios, i, nset
      	character(len=5) :: nuclide
	
      	err = 0
      	open(unit=iounit, file=filename, iostat=err, status="old", action="read")
      	if ( err /= 0 ) return

      	! first line must be size of set
      	read (iounit, *) nset
      	allocate(set(nset), stat=err)
      	if (err /= 0) stop "set: Allocation request denied"
         
         ios = 0
      	do i = 1, nset
      		read(iounit, *, iostat=ios) set(i)% rank, set(i)% nuclide
      		set(i)% nuclide = adjustr(set(i)% nuclide)
      		if (ios /= 0) then
      			print '("end of file with ", i0, " nuclides")', i
      			err = -1
      			exit
      		end if
      	end do	
      end subroutine read_nuclide_set
      

      end module nuclide_set_mod
