! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton and Pablo Marchant
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module binary_ctrls_io
      
      use const_def
      use binary_def

      implicit none
      
      include "binary_controls.inc"      
      
      logical :: read_extra_binary_controls_inlist1
      character (len=256) :: extra_binary_controls_inlist1_name 
   
      logical :: read_extra_binary_controls_inlist2
      character (len=256) :: extra_binary_controls_inlist2_name 
   
      logical :: read_extra_binary_controls_inlist3
      character (len=256) :: extra_binary_controls_inlist3_name 
   
      logical :: read_extra_binary_controls_inlist4
      character (len=256) :: extra_binary_controls_inlist4_name 
   
      logical :: read_extra_binary_controls_inlist5
      character (len=256) :: extra_binary_controls_inlist5_name 
      
      namelist /binary_controls/ &
         ! specifications for starting model
         m1, &
         m2, &
         initial_period_in_days, &
         initial_separation_in_Rsuns, &
         initial_eccentricity, &

         ! controls for output
         history_name, &
         append_to_star_history, &
         log_directory, &
         history_dbl_format, &
         history_int_format, &
         history_txt_format, &
         photostep, &
         photo_digits, &

         ! timestep controls
         fm, &
         fa, &
         fr, &
         fj, &
         fe, &
         fm_limit, &
         fr_limit, &
         fe_limit, &
         fr_dt_limit, &
         fdm, &
         dt_softening_factor, &
         varcontrol_case_a, &
         varcontrol_case_b, &
         varcontrol_ms, &
         varcontrol_post_ms, &
         
         ! when to stop
         accretor_overflow_terminate, &
         terminate_if_initial_overflow, &
             
         ! mass transfer controls
         mass_transfer_alpha, &
         mass_transfer_beta, &
         mass_transfer_delta, &
         mass_transfer_gamma, &
         limit_retention_by_mdot_edd, &
         use_es_opacity_for_mdot_edd, &
         use_this_for_mdot_edd, &
         mdot_scheme, &
         cur_mdot_frac, &
         max_explicit_abs_mdot, &
         max_tries_to_achieve, &
         implicit_scheme_tolerance, &
         initial_change_factor, &
         change_factor_fraction, &
         implicit_lambda, &
         max_change_factor, &
         min_change_factor, &
         starting_mdot, &
         roche_min_mdot, &
         min_mdot_for_implicit, &
         max_implicit_abs_mdot, &
         do_enhance_wind_1, &
         do_enhance_wind_2, &
         tout_B_wind_1, &
         tout_B_wind_2, &
         do_wind_mass_transfer_1, &
         do_wind_mass_transfer_2, &
         wind_BH_alpha_1, &
         wind_BH_alpha_2, &
         max_wind_transfer_fraction_1, &
         max_wind_transfer_fraction_2, &

         ! orbital jdot controls
         do_jdot_gr, &
         do_jdot_ml, &
         do_jdot_ls, &
         do_jdot_missing_wind, &
         do_jdot_mb, &
         include_accretor_mb, &
         magnetic_braking_gamma, &
         keep_mb_on, &
         jdot_multiplier, &

         ! rotation and sync controls
         do_j_accretion, &
         do_tidal_sync, &
         sync_type_1, &
         sync_type_2, &
         sync_mode_1, &
         sync_mode_2, &
         Ftid_1, &
         Ftid_2, &
         do_initial_orbit_sync_1, &
         do_initial_orbit_sync_2, &
         tidal_reduction, &
         
         ! eccentricity controls
         do_tidal_circ, &
         circ_type_1, &
         circ_type_2, &
         use_eccentricity_enhancement, &
         max_abs_edot_tidal, &
         max_abs_edot_enhance, &
         min_eccentricity, &
         max_eccentricity, &

         ! irradiation controls
         accretion_powered_irradiation, &
         accretor_radius_for_irrad, &
         col_depth_for_eps_extra, &
         irrad_flux_at_std_distance, &
         std_distance_for_irradiation, &
         max_F_irr, &
         
         ! miscellaneous controls
         keep_donor_fixed, &
         mdot_limit_donor_switch, &
         use_other_rlo_mdot, &
         use_other_tsync, &
         use_other_mdot_edd, &
         use_other_accreted_material_j, &
         use_other_jdot_gr, &
         use_other_jdot_ml, &
         use_other_jdot_ls, &
         use_other_jdot_missing_wind, &
         use_other_jdot_mb, &
         use_other_extra_jdot, &
         use_other_binary_wind_transfer, &

         ! extra files
         read_extra_binary_controls_inlist1, extra_binary_controls_inlist1_name, &
         read_extra_binary_controls_inlist2, extra_binary_controls_inlist2_name, &
         read_extra_binary_controls_inlist3, extra_binary_controls_inlist3_name, &
         read_extra_binary_controls_inlist4, extra_binary_controls_inlist4_name, &
         read_extra_binary_controls_inlist5, extra_binary_controls_inlist5_name

      contains
      
      
      subroutine do_one_binary_setup(b, inlist, ierr)
         use utils_lib
         type (binary_info), pointer :: b
         character (len=*), intent(in) :: inlist
         integer, intent(out) :: ierr

         include 'formats'

         call set_default_binary_controls
         call read_binary_controls(b, inlist, ierr)

      end subroutine do_one_binary_setup


      subroutine read_binary_controls(b, filename, ierr)
         use utils_lib
         type (binary_info), pointer :: b
         character(*), intent(in) :: filename
         integer, intent(out) :: ierr
         
         call read_binary_controls_file(b, filename, 1, ierr)
         
      end subroutine read_binary_controls
         
         
      recursive subroutine read_binary_controls_file(b, filename, level, ierr)
         use utils_lib
         character(*), intent(in) :: filename
         type (binary_info), pointer :: b
         integer, intent(in) :: level  
         integer, intent(out) :: ierr
         logical :: read_extra1, read_extra2, read_extra3, read_extra4, read_extra5
         character (len=256) :: message, extra1, extra2, extra3, extra4, extra5
         integer :: unit 
         
         ierr = 0        
         
         if (level >= 10) then
            write(*,*) 'ERROR: too many levels of nested extra binary controls inlist files'
            ierr = -1
            return
         end if

         if (len_trim(filename) > 0) then
            unit=alloc_iounit(ierr); if (ierr /= 0) return
            open(unit=unit, file=trim(filename), action='read', delim='quote', status='old', iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'Failed to open binary control namelist file ', trim(filename)
               return
            end if
            read(unit, nml=binary_controls, iostat=ierr)  
            close(unit)
            if (ierr /= 0) then
               write(*, *) 
               write(*, *) 
               write(*, *) 
               write(*, *) 
               write(*, '(a)') &
                  'Failed while trying to read binary control namelist file: ' // trim(filename)
               write(*, '(a)') &
                  'Perhaps the following runtime error message will help you find the problem.'
               write(*, *) 
               open(unit=unit, file=trim(filename), action='read', delim='quote', status='old', iostat=ierr)
               read(unit, nml=binary_controls)
               close(unit)
               call free_iounit(unit)
               return
            end if
            call free_iounit(unit)
         end if
         
         call store_binary_controls(b, ierr)
         
         ! recursive calls to read other inlists
         
         read_extra1 = read_extra_binary_controls_inlist1
         read_extra_binary_controls_inlist1 = .false.
         extra1 = extra_binary_controls_inlist1_name
         extra_binary_controls_inlist1_name = 'undefined'
         
         read_extra2 = read_extra_binary_controls_inlist2
         read_extra_binary_controls_inlist2 = .false.
         extra2 = extra_binary_controls_inlist2_name
         extra_binary_controls_inlist2_name = 'undefined'
         
         read_extra3 = read_extra_binary_controls_inlist3
         read_extra_binary_controls_inlist3 = .false.
         extra3 = extra_binary_controls_inlist3_name
         extra_binary_controls_inlist3_name = 'undefined'
         
         read_extra4 = read_extra_binary_controls_inlist4
         read_extra_binary_controls_inlist4 = .false.
         extra4 = extra_binary_controls_inlist4_name
         extra_binary_controls_inlist4_name = 'undefined'
         
         read_extra5 = read_extra_binary_controls_inlist5
         read_extra_binary_controls_inlist5 = .false.
         extra5 = extra_binary_controls_inlist5_name
         extra_binary_controls_inlist5_name = 'undefined'
         
         if (read_extra1) then
            write(*,*) 'read ' // trim(extra1)
            call read_binary_controls_file(b, extra1, level+1, ierr)
            if (ierr /= 0) return
         end if
         
         if (read_extra2) then
            write(*,*) 'read ' // trim(extra2)
            call read_binary_controls_file(b, extra2, level+1, ierr)
            if (ierr /= 0) return
         end if
         
         if (read_extra3) then
            write(*,*) 'read ' // trim(extra3)
            call read_binary_controls_file(b, extra3, level+1, ierr)
            if (ierr /= 0) return
         end if
         
         if (read_extra4) then
            write(*,*) 'read ' // trim(extra4)
            call read_binary_controls_file(b, extra4, level+1, ierr)
            if (ierr /= 0) return
         end if
         
         if (read_extra5) then
            write(*,*) 'read ' // trim(extra5)
            call read_binary_controls_file(b, extra5, level+1, ierr)
            if (ierr /= 0) return
         end if
         
      end subroutine read_binary_controls_file


      subroutine set_default_binary_controls
         include 'binary_controls.defaults'
      end subroutine set_default_binary_controls


      subroutine store_binary_controls(b, ierr)
         use utils_lib, only: mkdir
         type (binary_info), pointer :: b
         integer, intent(out) :: ierr
         
         ierr = 0
         
         ! specifications for starting model
         b% m1 = m1
         b% m2 = m2
         b% initial_period_in_days = initial_period_in_days
         b% initial_separation_in_Rsuns = initial_separation_in_Rsuns
         b% initial_eccentricity = initial_eccentricity

         ! controls for output
         b% history_name = history_name
         b% append_to_star_history = append_to_star_history
         b% log_directory = log_directory
         CALL mkdir(b% log_directory)
         b% history_dbl_format = history_dbl_format
         b% history_int_format = history_int_format
         b% history_txt_format = history_txt_format
         b% photostep = photostep
         b% photo_digits = photo_digits

         ! timestep controls
         b% fm = fm
         b% fa = fa
         b% fr = fr
         b% fj = fj
         b% fe = fe
         b% fm_limit = fm_limit
         b% fr_limit = fr_limit
         b% fe_limit = fe_limit
         b% fr_dt_limit = fr_dt_limit
         b% fdm = fdm
         b% dt_softening_factor = dt_softening_factor
         b% varcontrol_case_a = varcontrol_case_a
         b% varcontrol_case_b = varcontrol_case_b
         b% varcontrol_ms = varcontrol_ms
         b% varcontrol_post_ms = varcontrol_post_ms

         ! when to stop
         b% accretor_overflow_terminate = accretor_overflow_terminate
         b% terminate_if_initial_overflow = terminate_if_initial_overflow

         ! mass transfer controls
         b% mass_transfer_alpha = mass_transfer_alpha
         b% mass_transfer_beta = mass_transfer_beta
         b% mass_transfer_delta = mass_transfer_delta
         b% mass_transfer_gamma = mass_transfer_gamma
         b% limit_retention_by_mdot_edd = limit_retention_by_mdot_edd
         b% use_es_opacity_for_mdot_edd = use_es_opacity_for_mdot_edd
         b% use_this_for_mdot_edd = use_this_for_mdot_edd
         b% mdot_scheme = mdot_scheme
         b% cur_mdot_frac = cur_mdot_frac
         b% max_explicit_abs_mdot = max_explicit_abs_mdot
         b% max_tries_to_achieve = max_tries_to_achieve
         b% implicit_scheme_tolerance = implicit_scheme_tolerance
         b% initial_change_factor = initial_change_factor
         b% change_factor_fraction = change_factor_fraction
         b% implicit_lambda = implicit_lambda
         b% max_change_factor = max_change_factor
         b% min_change_factor = min_change_factor
         b% starting_mdot = starting_mdot
         b% roche_min_mdot = roche_min_mdot
         b% min_mdot_for_implicit = min_mdot_for_implicit
         b% max_implicit_abs_mdot = max_implicit_abs_mdot
         b% do_enhance_wind_1 = do_enhance_wind_1
         b% do_enhance_wind_2 = do_enhance_wind_2
         b% tout_B_wind_1 = tout_B_wind_1
         b% tout_B_wind_2 = tout_B_wind_2
         b% do_wind_mass_transfer_1 = do_wind_mass_transfer_1
         b% do_wind_mass_transfer_2 = do_wind_mass_transfer_2
         b% wind_BH_alpha_1 = wind_BH_alpha_1
         b% wind_BH_alpha_2 = wind_BH_alpha_2
         b% max_wind_transfer_fraction_1 = max_wind_transfer_fraction_1
         b% max_wind_transfer_fraction_2 = max_wind_transfer_fraction_2

         ! orbital jdot controls
         b% do_jdot_gr = do_jdot_gr
         b% do_jdot_ml = do_jdot_ml
         b% do_jdot_ls = do_jdot_ls
         b% do_jdot_missing_wind = do_jdot_missing_wind
         b% do_jdot_mb = do_jdot_mb
         b% include_accretor_mb = include_accretor_mb
         b% magnetic_braking_gamma = magnetic_braking_gamma
         b% keep_mb_on = keep_mb_on
         b% jdot_multiplier = jdot_multiplier

         ! rotation and sync controls
         b% do_j_accretion = do_j_accretion
         b% do_tidal_sync = do_tidal_sync
         b% sync_type_1 = sync_type_1
         b% sync_type_2 = sync_type_2
         b% sync_mode_1 = sync_mode_1
         b% sync_mode_2 = sync_mode_2
         b% Ftid_1 = Ftid_1
         b% Ftid_2 = Ftid_2
         b% do_initial_orbit_sync_1 = do_initial_orbit_sync_1
         b% do_initial_orbit_sync_2 = do_initial_orbit_sync_2
         b% tidal_reduction = tidal_reduction
         
         ! eccentricity controls
         b% do_tidal_circ = do_tidal_circ
         b% circ_type_1 = circ_type_1
         b% circ_type_2 = circ_type_2
         b% use_eccentricity_enhancement = use_eccentricity_enhancement
         b% max_abs_edot_tidal = max_abs_edot_tidal
         b% max_abs_edot_enhance = max_abs_edot_enhance
         b% min_eccentricity = min_eccentricity
         b% max_eccentricity = max_eccentricity

         ! irradiation controls
         b% accretion_powered_irradiation = accretion_powered_irradiation
         b% accretor_radius_for_irrad = accretor_radius_for_irrad
         b% col_depth_for_eps_extra = col_depth_for_eps_extra
         b% irrad_flux_at_std_distance = irrad_flux_at_std_distance
         b% std_distance_for_irradiation = std_distance_for_irradiation
         b% max_F_irr = max_F_irr
         
         ! miscellaneous controls
         b% keep_donor_fixed = keep_donor_fixed
         b% mdot_limit_donor_switch = mdot_limit_donor_switch
         b% use_other_rlo_mdot = use_other_rlo_mdot
         b% use_other_tsync = use_other_tsync
         b% use_other_mdot_edd = use_other_mdot_edd
         b% use_other_accreted_material_j = use_other_accreted_material_j
         b% use_other_jdot_gr = use_other_jdot_gr
         b% use_other_jdot_ml = use_other_jdot_ml
         b% use_other_jdot_ls = use_other_jdot_ls
         b% use_other_jdot_missing_wind = use_other_jdot_missing_wind
         b% use_other_jdot_mb = use_other_jdot_mb
         b% use_other_extra_jdot = use_other_extra_jdot
         b% use_other_binary_wind_transfer = use_other_binary_wind_transfer
         
      end subroutine store_binary_controls


      subroutine set_binary_controls_for_writing(b, ierr)
         type (binary_info), pointer :: b
         integer, intent(out) :: ierr
         
         ierr = 0

         ! specifications for starting model
         m1 = b% m1
         m2 = b% m2
         initial_period_in_days = b% initial_period_in_days
         initial_separation_in_Rsuns = b% initial_separation_in_Rsuns
         initial_eccentricity = b% initial_eccentricity

         ! controls for output
         history_name = b% history_name
         append_to_star_history = b% append_to_star_history
         log_directory = b% log_directory
         history_dbl_format = b% history_dbl_format
         history_int_format = b% history_int_format
         history_txt_format = b% history_txt_format
         photostep = b% photostep
         photo_digits = b% photo_digits

         ! timestep controls
         fm = b% fm
         fa = b% fa
         fr = b% fr
         fj = b% fj
         fe = b% fe
         fm_limit = b% fm_limit
         fr_limit = b% fr_limit
         fe_limit = b% fe_limit
         fr_dt_limit = b% fr_dt_limit
         fdm = b% fdm
         dt_softening_factor = b% dt_softening_factor
         varcontrol_case_a = b% varcontrol_case_a
         varcontrol_case_b = b% varcontrol_case_b
         varcontrol_ms = b% varcontrol_ms
         varcontrol_post_ms = b% varcontrol_post_ms

         ! when to stop
         accretor_overflow_terminate = b% accretor_overflow_terminate
         terminate_if_initial_overflow = b% terminate_if_initial_overflow

         ! mass transfer controls
         mass_transfer_alpha = b% mass_transfer_alpha
         mass_transfer_beta = b% mass_transfer_beta
         mass_transfer_delta = b% mass_transfer_delta
         mass_transfer_gamma = b% mass_transfer_gamma
         limit_retention_by_mdot_edd = b% limit_retention_by_mdot_edd
         use_es_opacity_for_mdot_edd = b% use_es_opacity_for_mdot_edd
         use_this_for_mdot_edd = b% use_this_for_mdot_edd
         mdot_scheme = b% mdot_scheme
         cur_mdot_frac = b% cur_mdot_frac
         max_explicit_abs_mdot = b% max_explicit_abs_mdot
         max_tries_to_achieve = b% max_tries_to_achieve
         implicit_scheme_tolerance = b% implicit_scheme_tolerance
         initial_change_factor = b% initial_change_factor
         change_factor_fraction = b% change_factor_fraction
         implicit_lambda = b% implicit_lambda
         max_change_factor = b% max_change_factor
         min_change_factor = b% min_change_factor
         starting_mdot = b% starting_mdot
         roche_min_mdot = b% roche_min_mdot
         min_mdot_for_implicit = b% min_mdot_for_implicit
         max_implicit_abs_mdot = b% max_implicit_abs_mdot
         do_enhance_wind_1 = b% do_enhance_wind_1
         do_enhance_wind_2 = b% do_enhance_wind_2
         tout_B_wind_1 = b% tout_B_wind_1
         tout_B_wind_2 = b% tout_B_wind_2
         do_wind_mass_transfer_1 = b% do_wind_mass_transfer_1
         do_wind_mass_transfer_2 = b% do_wind_mass_transfer_2
         wind_BH_alpha_1 = b% wind_BH_alpha_1
         wind_BH_alpha_2 = b% wind_BH_alpha_2
         max_wind_transfer_fraction_1 = b% max_wind_transfer_fraction_1
         max_wind_transfer_fraction_2 = b% max_wind_transfer_fraction_2

         ! orbital jdot controls
         do_jdot_gr = b% do_jdot_gr
         do_jdot_ml = b% do_jdot_ml
         do_jdot_ls = b% do_jdot_ls
         do_jdot_missing_wind = b% do_jdot_missing_wind
         do_jdot_mb = b% do_jdot_mb
         include_accretor_mb = b% include_accretor_mb
         magnetic_braking_gamma = b% magnetic_braking_gamma
         keep_mb_on = b% keep_mb_on
         jdot_multiplier = b% jdot_multiplier

         ! rotation and sync controls
         do_j_accretion = b% do_j_accretion
         do_tidal_sync = b% do_tidal_sync
         sync_type_1 = b% sync_type_1
         sync_type_2 = b% sync_type_2
         sync_mode_1 = b% sync_mode_1
         sync_mode_2 = b% sync_mode_2
         Ftid_1 = b% Ftid_1
         Ftid_2 = b% Ftid_2
         do_initial_orbit_sync_1 = b% do_initial_orbit_sync_1
         do_initial_orbit_sync_2 = b% do_initial_orbit_sync_2
         tidal_reduction = b% tidal_reduction
         
         ! eccentricity controls
         do_tidal_circ = b% do_tidal_circ
         circ_type_1 = b% circ_type_1
         circ_type_2 = b% circ_type_2
         use_eccentricity_enhancement = b% use_eccentricity_enhancement
         max_abs_edot_tidal = b% max_abs_edot_tidal
         max_abs_edot_enhance = b% max_abs_edot_enhance
         min_eccentricity = b% min_eccentricity
         max_eccentricity = b% max_eccentricity

         ! irradiation controls
         accretion_powered_irradiation = b% accretion_powered_irradiation
         accretor_radius_for_irrad = b% accretor_radius_for_irrad
         col_depth_for_eps_extra = b% col_depth_for_eps_extra
         irrad_flux_at_std_distance = b% irrad_flux_at_std_distance
         std_distance_for_irradiation = b% std_distance_for_irradiation
         max_F_irr = b% max_F_irr
         
         ! miscellaneous controls
         keep_donor_fixed = b% keep_donor_fixed
         mdot_limit_donor_switch = b% mdot_limit_donor_switch
         use_other_rlo_mdot = b% use_other_rlo_mdot
         use_other_tsync = b% use_other_tsync
         use_other_mdot_edd = b% use_other_mdot_edd
         use_other_accreted_material_j = b% use_other_accreted_material_j
         use_other_jdot_gr = b% use_other_jdot_gr
         use_other_jdot_ml = b% use_other_jdot_ml
         use_other_jdot_ls = b% use_other_jdot_ls
         use_other_jdot_missing_wind = b% use_other_jdot_missing_wind
         use_other_jdot_mb = b% use_other_jdot_mb
         use_other_extra_jdot = b% use_other_extra_jdot
         use_other_binary_wind_transfer = b% use_other_binary_wind_transfer
         
      end subroutine set_binary_controls_for_writing
      
      subroutine write_binary_controls(io,ierr)
         integer, intent(in) :: io
         integer, intent(out) :: ierr
         write(io, nml=binary_controls, iostat=ierr)  
      end subroutine write_binary_controls


      end module binary_ctrls_io

