! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module atm_lib
      use const_def, only: dp
      
      implicit none

      contains


      subroutine atm_init(use_cache, ierr)      
         use mod_atm, only : do_init_atm
         logical, intent(in) :: use_cache
         integer, intent(out) :: ierr
         call do_init_atm(use_cache, ierr)
      end subroutine atm_init


      subroutine atm_shutdown
         use mod_atm, only: do_atm_shutdown
         call do_atm_shutdown
      end subroutine atm_shutdown
      
      
      ! because there are a variety of ways to calculate atmospheric boundary conditions,
      ! there are also a variety of routines with different interfaces.
      ! here's a summary:
         ! atm_get_grey
            ! this is the most primitive option, but often is good enough.
            ! for given tau, L, R, M, and kap, get P.  get T from Eddington T(tau)
         ! atm_get_grey_and_kap
            ! for given tau, L, R, M, this iterates to get consistent kap, P, and T
         ! atm_get_table
            ! this uses non-grey atmosphere tables to find P and T for given Teff and logg
            ! for (Teff,logg) pairs not covered by the tables, uses atm_get_grey_and_kap
         ! atm_get_int_T_tau
            ! this integrates a T(tau) relation to find T and P at a given tau
         ! atm_grey_irradiated_get
            ! this uses information about irradiation to return tau and T for a given P
         ! atm_Paczynski_grey
            ! create an atmosphere for given base conditions.
            ! inspired by B. Paczynski, 1969, Acta Astr., vol. 19, 1.
      
      
      ! op_mono not yet available on mic.  is okay on host.
      subroutine atm_get_grey_and_op_mono_kap( &
            Pextra_factor, tau, kap_guess, &
            cgrav, M, R, L, X, Z, abar, zbar,  &
            species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            use_op_mono_alt_get_kap, op_mono_min_X_to_include, &
            op_mono_data_path, op_mono_data_cache_filename, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM,  &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM,  &
            kap, Teff, iters, err, ierr) 
         use mod_atm, only: do_atm_grey_and_op_mono_kap
         
         ! cgs units
         real(dp), intent(in) :: Pextra_factor
         real(dp), intent(in) :: tau   ! optical depth
         real(dp), intent(in) :: kap_guess ! initial guess for opacity
         real(dp), intent(in) :: cgrav ! gravitational constant
         real(dp), intent(in) :: M     ! enclosed mass
         real(dp), intent(in) :: R     ! radius of photosphere
         real(dp), intent(in) :: L     ! luminosity
         real(dp), intent(in) :: X     ! hydrogen mass fraction
         real(dp), intent(in) :: Z     ! metallicity
         real(dp), intent(in) :: abar  ! mean number of nucleons per nucleus
         real(dp), intent(in) :: zbar  ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         integer, intent(in) :: max_tries
         real(dp), intent(in) :: atol, rtol
         
         integer, intent(in) :: eos_handle
         integer, intent(in) :: kap_handle
         
         logical, intent(in) :: use_op_mono_alt_get_kap
         real(dp), intent(in) :: op_mono_min_X_to_include
         character (len=*), intent(in) :: &
            op_mono_data_path, op_mono_data_cache_filename
         
         real(dp), intent(out) :: lnT ! natural log of temperature at base of atmosphere
         real(dp), intent(out) :: lnP ! natural log of pressure at base of atmosphere (Pgas + Prad)
         
         ! partial derivatives of lnT and lnP
         real(dp), intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM
         real(dp), intent(out) :: dlnP_dL, dlnP_dlnR, dlnP_dlnM

         real(dp), intent(out) :: kap ! opacity consistent with lnT and lnP
         real(dp), intent(out) :: Teff ! temperature at photosphere

         integer, intent(out) :: iters
         real(dp), intent(out) :: err
         
         integer, intent(out) :: ierr  ! == 0 means AOK

         call do_atm_grey_and_op_mono_kap( &
            Pextra_factor, tau, kap_guess, cgrav, M, R, L, X, Z, abar, zbar,  &
            species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            use_op_mono_alt_get_kap, op_mono_min_X_to_include, &
            op_mono_data_path, op_mono_data_cache_filename, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM,  &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM,  &
            kap, Teff, iters, err, ierr) 
     
      end subroutine atm_get_grey_and_op_mono_kap
      
#ifdef offload
      !dir$ options /offload_attribute_target=mic
      ! the following can be called on mic
#endif      
      
      subroutine atm_get_grey( &
            Pextra_factor, tau, cgrav, M, R, L, kap, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap,  &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
     
         use mod_atm, only: get_grey
                  
         ! cgs units
         real(dp), intent(in) :: Pextra_factor
         real(dp), intent(in) :: tau ! optical depth
         real(dp), intent(in) :: cgrav ! gravitational constant
         real(dp), intent(in) :: M ! enclosed mass
         real(dp), intent(in) :: R ! radius of photosphere
         real(dp), intent(in) :: L ! luminosity
         real(dp), intent(in) :: kap 
            ! opacity above photosphere (average by mass)
         
         real(dp), intent(out) :: Teff ! based on assuming M,R,L are at photosphere
         real(dp), intent(out) :: lnT ! natural log of temperature at base of atmosphere
         real(dp), intent(out) :: lnP ! natural log of pressure at base of atmosphere (Pgas + Prad)
         
         ! partial derivatives of lnT and lnP
         real(dp), intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap
         real(dp), intent(out) :: dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
         
         integer, intent(out) :: ierr  ! == 0 means AOK

         call get_grey( &
            Pextra_factor, tau, cgrav, M, R, L, kap, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap,  &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
     
      end subroutine atm_get_grey
      
      
      ! like atm_get_grey,
      ! but adjusts kap until it is consistent with lnP and lnT for given composition
      ! this involves repeated calls on the eos and kap modules
      subroutine atm_get_grey_and_kap( &
            Pextra_factor, tau, kap_guess, &
            cgrav, M, R, L, X, Z, abar, zbar,  &
            species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM,  &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM,  &
            kap, Teff, iters, err, ierr) 
         use mod_atm, only: do_atm_grey_and_kap
         
         ! cgs units
         real(dp), intent(in) :: Pextra_factor
         real(dp), intent(in) :: tau   ! optical depth
         real(dp), intent(in) :: kap_guess ! initial guess for opacity
         real(dp), intent(in) :: cgrav ! gravitational constant
         real(dp), intent(in) :: M     ! enclosed mass
         real(dp), intent(in) :: R     ! radius of photosphere
         real(dp), intent(in) :: L     ! luminosity
         real(dp), intent(in) :: X     ! hydrogen mass fraction
         real(dp), intent(in) :: Z     ! metallicity
         real(dp), intent(in) :: abar  ! mean number of nucleons per nucleus
         real(dp), intent(in) :: zbar  ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         integer, intent(in) :: max_tries
         real(dp), intent(in) :: atol, rtol
         
         integer, intent(in) :: eos_handle
         integer, intent(in) :: kap_handle
         
         real(dp), intent(out) :: lnT ! natural log of temperature at base of atmosphere
         real(dp), intent(out) :: lnP ! natural log of pressure at base of atmosphere (Pgas + Prad)
         
         ! partial derivatives of lnT and lnP
         real(dp), intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM
         real(dp), intent(out) :: dlnP_dL, dlnP_dlnR, dlnP_dlnM

         real(dp), intent(out) :: kap ! opacity consistent with lnT and lnP
         real(dp), intent(out) :: Teff ! temperature at photosphere

         integer, intent(out) :: iters
         real(dp), intent(out) :: err
         
         integer, intent(out) :: ierr  ! == 0 means AOK

         call do_atm_grey_and_kap( &
            Pextra_factor, tau, kap_guess, cgrav, M, R, L, X, Z, abar, zbar,  &
            species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM,  &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM,  &
            kap, Teff, iters, err, ierr) 
     
      end subroutine atm_get_grey_and_kap


      ! based on Guillot, T, and Havel, M., A&A 527, A20 (2011). T(tau) equation 6.
      subroutine atm_grey_irradiated_get( &
            T_eq, kap_v, kap_th_guess, use_kap_th_guess, kap_v_div_kap_th, &
            P_boundary, cgrav, M_boundary, R_boundary, L_intrinsic,  &
            X, Z, abar, zbar, species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM,  &
            kap_th, tau_boundary, Teff, iters, err, ierr) 
         use mod_atm, only: do_atm_grey_irradiated
         
         ! cgs units
         real(dp), intent(in) :: T_eq ! equilibrium temperature based on irradiation
            ! e.g., might use T_eq = Teff_star*(R_star/(2*distance_to_star))^(1/2)
         real(dp), intent(in) :: kap_v ! opacity for irradiation
         real(dp), intent(in) :: kap_th_guess ! starting guess for thermal opacity
         logical, intent(in) :: use_kap_th_guess ! if true, use it.  else iterate.
         real(dp), intent(in) :: kap_v_div_kap_th ! fixed value for kap_v/kap_th
            ! if this arg is > 0, then use it and ignore the kap_v arg.
            ! else calculate kap_th and use that to get kap_v/kap_th using the given kap_v
         real(dp), intent(in) :: P_boundary ! pressure at base of atmosphere
         real(dp), intent(in) :: cgrav ! gravitational constant
         real(dp), intent(in) :: M_boundary ! mass at base of atmosphere
         real(dp), intent(in) :: R_boundary ! radius at base of atmosphere
         real(dp), intent(in) :: L_intrinsic ! luminosity at base of atmosphere
         real(dp), intent(in) :: X ! hydrogen mass fraction
         real(dp), intent(in) :: Z ! metallicity
         real(dp), intent(in) :: abar ! mean number of nucleons per nucleus
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         integer, intent(in) :: max_tries
         real(dp), intent(in) :: atol, rtol
         
         integer, intent(in) :: eos_handle
         integer, intent(in) :: kap_handle
         
         real(dp), intent(out) :: lnT ! natural log of temperature at boundary
         ! partial derivatives of lnT
         real(dp), intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM

         real(dp), intent(out) :: kap_th ! thermal opacity at boundary
         real(dp), intent(out) :: tau_boundary ! optical depth at boundary
         real(dp), intent(out) :: Teff ! temperature at tau = 2/3

         integer, intent(out) :: iters
         real(dp), intent(out) :: err
         
         integer, intent(out) :: ierr  ! == 0 means AOK
         
         call do_atm_grey_irradiated( &
            T_eq, kap_v, kap_th_guess, use_kap_th_guess, kap_v_div_kap_th, &
            P_boundary, cgrav, M_boundary, R_boundary, L_intrinsic,  &
            X, Z, abar, zbar, species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM,  &
            kap_th, tau_boundary, Teff, iters, err, ierr) 
      
      end subroutine atm_grey_irradiated_get
      
      
      subroutine atm_get_table( &
            which_atm_option, off_table_option, cgrav, M, R, L, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, &
            ! args for atm_get_grey_and_kap in case (Teff,logg) pair is off-table
            Pextra_factor, kap_guess, X, Z, abar, zbar, &
            species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            Teff, ierr) 
         use mod_atm, only: do_atm_get
         use atm_def, only: atm_grey_and_kap
         
         ! cgs units
         integer, intent(in) :: which_atm_option ! one of the table options
         integer, intent(in) :: off_table_option ! switch to this if args off table
         real(dp), intent(in) :: cgrav ! gravitational constant
         real(dp), intent(in) :: M     ! enclosed mass
         real(dp), intent(in) :: R     ! radius of photosphere
         real(dp), intent(in) :: L     ! luminosity
         
         ! args for atm_get_grey_and_kap in case (Teff,logg) pair is off-table
         real(dp), intent(in) :: Pextra_factor
         real(dp), intent(in) :: kap_guess ! initial guess for opacity
         real(dp), intent(in) :: X     ! hydrogen mass fraction
         real(dp), intent(in) :: Z     ! metallicity
         real(dp), intent(in) :: abar  ! mean number of nucleons per nucleus
         real(dp), intent(in) :: zbar  ! mean charge per nucleus
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         integer, intent(in) :: max_tries
         real(dp), intent(in) :: atol, rtol
         
         integer, intent(in) :: eos_handle
         integer, intent(in) :: kap_handle
         
         ! results
         real(dp), intent(out) :: lnT ! natural log of temperature at base of atmosphere
         real(dp), intent(out) :: lnP ! natural log of pressure at base of atmosphere (Pgas + Prad)
         
         ! partial derivatives of lnT and lnP
         real(dp), intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM
         real(dp), intent(out) :: dlnP_dL, dlnP_dlnR, dlnP_dlnM

         real(dp), intent(out) :: Teff ! temperature at photosphere
         
         integer, intent(out) :: ierr  ! == 0 means AOK
         
         real*8 :: dlnT_dlnkap, dlnP_dlnkap, kap, err, tau
         integer :: iters
         
         ierr = 0
         tau = atm_tau_base(which_atm_option, ierr)
         if (ierr /= 0) return
         call do_atm_get( &
            which_atm_option, off_table_option, &
            Pextra_factor, tau, kap_guess,  &
            cgrav, M, R, L, X, Z, abar, zbar,  &
            species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap,  &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap,  &
            kap, Teff, iters, err, ierr) 
     
      end subroutine atm_get_table
      
      
      ! atm_get_int_T_tau integrates from tau=0 to tau_base
      subroutine atm_get_int_T_tau( &
            errtol, cgrav, M, R, L, X, Z, abar, zbar, &
            species, chem_id, net_iso, xa, &
            which_atm_option, eos_handle, kap_handle, save_atm_structure_info, &
            tau_base, skip_partials, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap,  &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            num_atm_structure_points, atm_structure_results, &
            ierr) 
         use integrate_atm, only: do_atm_int
         
         real(dp), intent(in) :: errtol ! for integrator

         ! cgs units
         
         ! input values base of atmosphere
         real(dp), intent(in) :: cgrav ! gravitational constant
         real(dp), intent(in) :: M     ! enclosed mass
         real(dp), intent(in) :: R     ! radius
         real(dp), intent(in) :: L     ! luminosity
         real(dp), intent(in) :: X     ! hydrogen mass fraction
         real(dp), intent(in) :: Z     ! metallicity
         real(dp), intent(in) :: abar  ! mean number of nucleons per nucleus
         real(dp), intent(in) :: zbar  ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         integer, intent(in) :: which_atm_option ! must be one of the T(tau) options
         integer, intent(in) :: eos_handle
         integer, intent(in) :: kap_handle
         logical, intent(in) :: save_atm_structure_info
         logical, intent(in) :: skip_partials
         real(dp), intent(in) :: tau_base ! stop integration when reach this optical depth
            ! this defines the base of the atmosphere. does not need to be at photosphere.
         
         real(dp), intent(out) :: Teff ! temperature at photosphere
         real(dp), intent(out) :: lnT ! natural log of temperature at base of atmosphere
         real(dp), intent(out) :: lnP ! natural log of pressure at base of atmosphere (Pgas + Prad)
         
         ! partial derivatives of lnT and lnP (not evaluated if skip_partials is true)
         real(dp), intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap
         real(dp), intent(out) :: dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
         integer, intent(out) :: num_atm_structure_points
         real*8, pointer :: atm_structure_results(:,:) ! will be allocated if necessary
            ! (num_results_for_create_atm, num_atm_points)
         
         integer, intent(out) :: ierr

         call do_atm_int( &
            errtol, cgrav, M, R, L, X, Z, abar, zbar,  &
            species, chem_id, net_iso, xa, &
            which_atm_option, eos_handle, kap_handle, save_atm_structure_info, &
            tau_base, skip_partials, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap,  &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            num_atm_structure_points, atm_structure_results, &
            ierr) 
     
      end subroutine atm_get_int_T_tau      

      
      subroutine atm_get_surf_PT_for_Teff( &
            tau, Teff, lnT, lnP, ierr)
         use eos_lib, only: Radiation_Pressure
         use crlibm_lib, only: pow_cr, log_cr
         ! cgs units
         real(dp), intent(in) :: tau ! optical depth
         real(dp), intent(out) :: Teff
         real(dp), intent(out) :: lnT ! natural log of temperature at base of atmosphere
         real(dp), intent(out) :: lnP ! natural log of pressure at base of atmosphere (Pgas + Prad)
         integer, intent(out) :: ierr  ! == 0 means AOK
         real(dp) :: Teff4, T4, T, P
         Teff4 = Teff*Teff*Teff*Teff
         T4 = 0.75d0*Teff4*(tau + 2d0/3d0) ! Eddington T(tau)
         T = pow_cr(T4,0.25d0)
         lnT = log_cr(T)
         P = Radiation_Pressure(T)
         lnP = log_cr(P)
      end subroutine atm_get_surf_PT_for_Teff
      
      
      ! some utility routines
      
      integer function atm_option(which_atm_option, ierr)
         use atm_def
         character (len=*), intent(in) :: which_atm_option
         integer, intent(out) :: ierr
         ierr = 0
         if (which_atm_option == 'simple_photosphere') then
            atm_option = atm_simple_photosphere
         else if (which_atm_option == 'Eddington_grey') then
            atm_option = atm_Eddington_grey
         else if (which_atm_option == 'Krishna_Swamy') then
            atm_option = atm_Krishna_Swamy
         else if (which_atm_option == 'solar_Hopf_grey') then
            atm_option = atm_solar_Hopf_grey
         else if (which_atm_option == 'tau_100_tables') then
            atm_option = atm_tau_100_tables
         else if (which_atm_option == 'tau_10_tables') then
            atm_option = atm_tau_10_tables
         else if (which_atm_option == 'tau_1_tables') then
            atm_option = atm_tau_1_tables
         else if (which_atm_option == 'tau_1m1_tables') then
            atm_option = atm_tau_1m1_tables
         else if (which_atm_option == 'photosphere_tables') then
            atm_option = atm_photosphere_tables
         else if (which_atm_option == 'grey_and_kap') then
            atm_option = atm_grey_and_kap
         else if (which_atm_option == 'grey_irradiated') then
            atm_option = atm_grey_irradiated
         else if (which_atm_option == 'Paczynski_grey') then
            atm_option = atm_Paczynski_grey
         else if (which_atm_option == 'WD_tau_25_tables') then
            atm_option = atm_WD_tau_25_tables
         else if (which_atm_option == 'fixed_Teff') then
            atm_option = atm_fixed_Teff
         else if (which_atm_option == 'fixed_Tsurf') then
            atm_option = atm_fixed_Tsurf
         else if (which_atm_option == 'fixed_Psurf') then
            atm_option = atm_fixed_Psurf
         else
            ierr = -1
            atm_option = -1
         end if 
      end function atm_option
      
      
      subroutine atm_option_str(which_atm_option, atm_option, ierr)
         use atm_def
         integer, intent(in) :: which_atm_option
         character (len=*), intent(out) :: atm_option
         integer, intent(out) :: ierr
         ierr = 0
         if (which_atm_option == atm_simple_photosphere) then
            atm_option = 'simple_photosphere'
         else if (which_atm_option == atm_Eddington_grey) then
            atm_option = 'Eddington_grey'
         else if (which_atm_option == atm_Krishna_Swamy) then
            atm_option = 'Krishna_Swamy'
         else if (which_atm_option == atm_solar_Hopf_grey) then
            atm_option = 'solar_Hopf_grey'
         else if (which_atm_option == atm_tau_100_tables) then
            atm_option = 'tau_100_tables'
         else if (which_atm_option == atm_tau_10_tables) then
            atm_option = 'tau_10_tables'
         else if (which_atm_option == atm_tau_1_tables) then
            atm_option = 'tau_1_tables'
         else if (which_atm_option == atm_tau_1m1_tables) then
            atm_option = 'tau_1m1_tables'
         else if (which_atm_option == atm_photosphere_tables) then
            atm_option = 'photosphere_tables'
         else if (which_atm_option == atm_grey_and_kap) then
            atm_option = 'grey_and_kap'
         else if (which_atm_option == atm_grey_irradiated) then
            atm_option = 'grey_irradiated'
         else if (which_atm_option == atm_Paczynski_grey) then
            atm_option = 'Paczynski_grey'
         else if (which_atm_option == atm_WD_tau_25_tables) then
            atm_option = 'WD_tau_25_tables'
         else if (which_atm_option == atm_fixed_Teff) then
            atm_option = 'fixed_Teff'
         else if (which_atm_option == atm_fixed_Tsurf) then
            atm_option = 'fixed_Tsurf'
         else if (which_atm_option == atm_fixed_Psurf) then
            atm_option = 'fixed_Psurf'
         else
            ierr = -1
            atm_option = ''
         end if 
      end subroutine atm_option_str
      
      
      ! tau_base is the optical depth at the base of the atmosphere
      ! for all cases except the tau tables, the base of the atmosphere is the photosphere
      ! here are the tau_base values for the various atmosphere options
      ! 2/3 for simple_photosphere, Eddington grey integration, and photosphere tables
      ! 0.3121563 for Krishna Swamy T-tau integration
      ! 100 for tau 100 tables
      
      real(dp) function atm_tau_base(which_atm_option, ierr)
         use atm_def
         integer, intent(in) :: which_atm_option
         integer, intent(out) :: ierr
         ierr = 0
         atm_tau_base = 0
         select case (which_atm_option)
            case (atm_simple_photosphere)
               atm_tau_base = 2d0/3d0
            case (atm_Eddington_grey)
               atm_tau_base = 2d0/3d0
            case (atm_Krishna_Swamy)
               atm_tau_base = 0.3121563d0
            case (atm_solar_Hopf_grey)
               atm_tau_base = 0.4116433502d0
            case (atm_tau_100_tables)
               atm_tau_base = 100d0
            case (atm_tau_10_tables)
               atm_tau_base = 10d0
            case (atm_tau_1_tables)
               atm_tau_base = 1d0
            case (atm_tau_1m1_tables)
               atm_tau_base = 1d-1
            case (atm_photosphere_tables)
               atm_tau_base = 2d0/3d0
            case (atm_grey_and_kap)
               atm_tau_base = 2d0/3d0
            case (atm_grey_irradiated)
               atm_tau_base = 2d0/3d0
            case (atm_Paczynski_grey)
               atm_tau_base = 2d0/3d0
            case (atm_WD_tau_25_tables)
               atm_tau_base = 25.1188d0
            case (atm_fixed_Teff)
               atm_tau_base = 2d0/3d0
            case (atm_fixed_Tsurf)
               atm_tau_base = 2d0/3d0
            case (atm_fixed_Psurf)
               atm_tau_base = 2d0/3d0
            case default
               write(*,*) 'atm_tau_base: bad value for which_atm_option', which_atm_option
               ierr = -1
         end select
      end function atm_tau_base
      
      
      real(dp) function eval_T4_of_tau(which_atm_option, Teff4, tau)
         use integrate_atm, only: T4_of_tau
         use crlibm_lib, only: log_cr
         integer, intent(in) :: which_atm_option
         real(dp), intent(in) :: Teff4, tau
         real(dp) :: dT4_dtau
         eval_T4_of_tau = &
            T4_of_tau(which_atm_option, tau, log_cr(max(1d-99,tau)), Teff4, dT4_dtau)
      end function eval_T4_of_tau
      
      
      logical function atm_by_table(which_atm_option)
         use atm_def
         integer, intent(in) :: which_atm_option
         atm_by_table = &
            (which_atm_option == atm_tau_100_tables &
               .or. which_atm_option == atm_tau_10_tables &
               .or. which_atm_option == atm_tau_1_tables &
               .or. which_atm_option == atm_tau_1m1_tables &
               .or. which_atm_option == atm_photosphere_tables &
               .or. which_atm_option == atm_WD_tau_25_tables)
      end function atm_by_table
      
      
      logical function atm_by_integration(which_atm_option)
         use atm_def
         integer, intent(in) :: which_atm_option
         atm_by_integration = &
            (which_atm_option == atm_Eddington_grey &
               .or. which_atm_option == atm_Krishna_Swamy &
               .or. which_atm_option == atm_solar_Hopf_grey)
      end function atm_by_integration


      real(dp) function atm_Teff(L, R) ! assumes L and R at photosphere
         use crlibm_lib, only: pow_cr
         use const_def, only: pi, boltz_sigma
         real(dp), intent(in) :: L, R
         atm_Teff = pow_cr(L / (4d0*pi*R*R*boltz_sigma), 0.25d0)
      end function atm_Teff


      real(dp) function atm_L(Teff, R)
         use const_def, only: pi, boltz_sigma
         real(dp), intent(in) :: Teff, R
         atm_L = 4d0*pi*R*R*boltz_sigma*Teff*Teff*Teff*Teff
      end function atm_L


      real(dp) function atm_R(Teff, L)
         use const_def, only: pi, boltz_sigma
         real(dp), intent(in) :: Teff, L
         atm_R = sqrt(L/(4d0*pi*boltz_sigma*Teff*Teff*Teff*Teff))
      end function atm_R
      
      
      real(dp) function eval_Paczynski_gradr(P,opacity,L,m,cgrav,Pr,tau,T,r,rho)
         use integrate_atm, only: get1_Paczynski_gradr
         real(dp), intent(in) :: P,opacity,L,m,cgrav,Pr,tau,T,r,rho
         eval_Paczynski_gradr = get1_Paczynski_gradr(P,opacity,L,m,cgrav,Pr,tau,T,r,rho)
      end function eval_Paczynski_gradr
      
      
      real(dp) function eval_Lrad(P,opacity,gradr,m,cgrav,Pr)
         use const_def, only: clight, pi
         real(dp), intent(in) :: P,opacity,gradr,m,cgrav,Pr
         eval_Lrad = 16d0*pi*clight*m*cgrav*Pr*gradr/(P*opacity)
      end function eval_Lrad


#ifdef offload
      !dir$ end options
#endif
      
      end module atm_lib

