! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module mod_atm

      use const_def
      use atm_def
      use utils_lib, only: is_bad_num
      use crlibm_lib

      implicit none
      logical, parameter :: dbg = .false.
      

      
      contains
      

      real(dp) function eval_Teff(L, R)
         real(dp), intent(in) :: L, R
         eval_Teff = pow_cr(L / (4d0*pi*R*R*boltz_sigma), 0.25d0)
      end function eval_Teff
      
      
      subroutine do_init_atm(use_cache, ierr)
         use table_atm, only: table_atm_init
         use integrate_atm, only: int_atm_init
         integer, intent(out) :: ierr
         logical, intent(in) :: use_cache                  
         ierr = 0
         E2_f1 => E2_f_ary
         E2_f(1:4,1:npairs) => E2_f1(1:4*npairs)
         call table_atm_init(use_cache, ierr)
         call int_atm_init
         call set_E2_pairs
#ifdef offload
         call create_E2_interpolant(ierr)
         call copy_atm_info_to_coprocessor(ierr) 
#endif         
      end subroutine do_init_atm

     
      subroutine do_atm_shutdown
      end subroutine do_atm_shutdown

      
      
      subroutine do_atm_grey_and_op_mono_kap( &
            Pextra_factor, tau, kap_guess, &
            cgrav, M, R, L, X, Z, abar, zbar, & 
            species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            use_op_mono_alt_get_kap, op_mono_min_X_to_include, &
            op_mono_data_path, op_mono_data_cache_filename, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, & 
            kap, Teff, iters, err, ierr) 
         
         use integrate_atm, only: get_atm_eos_op_mono_kap_info
         
         ! cgs units
         real(dp), intent(in) :: Pextra_factor, tau   ! optical depth
         real(dp), intent(in) :: kap_guess ! initial guess for opacity
         real(dp), intent(in) :: cgrav, M, R, L, X, Z, abar, zbar
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in)  :: xa(:) ! mass fractions
         
         integer, intent(in) :: max_tries
         real(dp), intent(in) :: atol, rtol
         
         integer, intent(in) :: eos_handle
         integer, intent(in) :: kap_handle
         
         logical, intent(in) :: use_op_mono_alt_get_kap
         real(dp), intent(in) :: op_mono_min_X_to_include
         character (len=*), intent(in) :: &
            op_mono_data_path, op_mono_data_cache_filename
         
         real(dp), intent(out) :: lnT ! natural log of temperature at base of atmosphere
         real(dp), intent(out) :: lnP ! natural log of pressure at base of atmosphere (Pgas + Prad)
         
         ! partial derivatives of lnT and lnP (not evaluated if skip_partials is true)
         real(dp), intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM
         real(dp), intent(out) :: dlnP_dL, dlnP_dlnR, dlnP_dlnM

         real(dp), intent(out) :: kap ! opacity consistent with lnT and lnP
         real(dp), intent(out) :: Teff ! temperature at photosphere

         integer, intent(out) :: iters
         real(dp), intent(out) :: err
         
         integer, intent(out) :: ierr  ! == 0 means AOK

         real(dp) :: g, dlnT_dlnkap, dlnP_dlnkap, T, P, Prad, Pgas, lnPgas, &
            Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            kap_prev, dlnkap_dlnRho, dlnkap_dlnT, dlnP_dlnPgas, df_dkap

         include 'formats.dek'

         if (L <= 0) then
            ierr = -1
            return
         end if
         ierr = 0
         Teff = eval_Teff(L, R)
         g = cgrav * M / (R*R)
         err = 1d99
         
         kap = kap_guess
         
         do iters = 1, max_tries
         
            call eval_grey( &
               tau, g, Pextra_factor, kap, L, cgrav, M, Teff, &
               lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, & 
               lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
               ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in eval_grey'
               return
            end if
            
            if (.false.) then
               write(*,1) 'tau', tau
               write(*,1) 'L', L
               write(*,1) 'M', M
               write(*,1) 'R', R
               write(*,1) 'kap', kap
               write(*,1) 'lnT', lnT
               write(*,1) 'lnP', lnP
            end if
            
            T = exp_cr(lnT)
            P = exp_cr(lnP)
            Prad = crad*T*T*T*T/3
            Pgas = max(1d-99, P - Prad)
            lnPgas = log_cr(Pgas)
         
            kap_prev = kap
            call get_atm_eos_op_mono_kap_info( &
               -1, kap_handle, eos_handle, tau, cgrav, L, m, r, &
               Pgas, lnPgas, T, lnT, Z, 1-(X+Z), X, abar, zbar, &
               species, chem_id, net_iso, xa, &
               use_op_mono_alt_get_kap, op_mono_min_X_to_include, &
               op_mono_data_path, op_mono_data_cache_filename, &
               Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
            if (ierr /= 0) then
               return
               
               write(*,*) 'failed in get_atm_eos_kap_info'
               write(*,1) 'Pgas', Pgas
               write(*,1) 'lnPgas', lnPgas
               write(*,1) 'T', T
               write(*,1) 'lnT', lnT
               write(*,1) 'Z', Z
               write(*,1) 'Y', 1-(X+Z)
               write(*,1) 'X', X
               write(*,1) 'abar', abar
               write(*,1) 'zbar', zbar
               write(*,1) 'kap', kap
               stop 'do_atm_grey_and_kap'
               return
            end if
            
            err = abs(kap_prev - kap)/(atol + rtol*kap)
            
            !write(*,2) 'err kap lgP', iters, err, kap, lnP/ln10

            if (err < 1) exit
            
            kap = kap_prev + 0.5*(kap - kap_prev) ! under correct
            
         end do

     
      end subroutine do_atm_grey_and_op_mono_kap

#ifdef offload
      !dir$ options /offload_attribute_target=mic
#endif      
      
      subroutine do_atm_get( &
            which_atm_option, off_table_option, Pextra_factor, tau, kap_in, & 
            cgrav, M, R, L, X, Z, abar, zbar, & 
            species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, & 
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, & 
            kap_out, Teff, iters, err, ierr) 
                     
         integer, intent(in) :: which_atm_option, off_table_option
         real(dp), intent(in) :: &
            Pextra_factor, tau, kap_in, cgrav, M, R, L, X, Z, abar, zbar
         integer, intent(in) :: species
         integer, pointer :: chem_id(:), net_iso(:)
         real(dp), intent(in) :: xa(:)
         integer, intent(in) :: max_tries
         real(dp), intent(in) :: atol, rtol
         integer, intent(in) :: eos_handle, kap_handle
         real(dp), intent(out) :: kap_out, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, err
         integer, intent(out) :: iters, ierr
         
         
         
         
         real(dp) :: &
            P1, lnP1, dlnP_dL1, dlnP_dlnR1, dlnP_dlnM1, dlnP_dlnkap1, &
            P2, lnP2, dlnP_dL2, dlnP_dlnR2, dlnP_dlnM2, dlnP_dlnkap2, &
            T1, lnT1, dlnT_dL1, dlnT_dlnR1, dlnT_dlnM1, dlnT_dlnkap1, &
            T2, lnT2, dlnT_dL2, dlnT_dlnR2, dlnT_dlnM2, dlnT_dlnkap2
         real(dp) :: &
            alfa, beta, g, logg, d_logg_dlnR, d_logg_dlnM, &
            Prad, dPrad_dlnTeff, dPrad_dlnR, dPrad_dL, dTeff_dlnR, dlnTeff_dlnR, dlnTeff_dL
         
         include 'formats.dek'

         ierr = 0
         
         if (L <= 0 .or. M <= 0 .or. R <= 0 .or. &
               which_atm_option == atm_Eddington_grey .or. &
               which_atm_option == atm_Krishna_Swamy) then
            ierr = -1
            return
         end if

         Teff = eval_Teff(L, R)
         dlnTeff_dlnR = -0.5d0
         dlnTeff_dL = 1/(4*L)
         dTeff_dlnR = Teff*dlnTeff_dlnR
         
         g = cgrav * M / (R*R)
		   logg = log10_cr(g)
         d_logg_dlnR = -2/ln10
		   d_logg_dlnM = 1/ln10
         
         if (dbg) write(*,*) 'call get_alfa_beta: which_atm_option', which_atm_option

         if (off_table_option >= min_atm_option) then
            call get_alfa_beta(ierr)
            if (ierr /= 0) then
               if (dbg) write(*,*) 'failed in get_alfa_beta'
               return
            end if
         else
            alfa=0
            beta=1
         endif         
         if (dbg) write(*,1) 'alfa, beta', alfa, beta

         if (beta > 0) then
            call eval_table(ierr)
            if (ierr /= 0) then
               if (dbg) write(*,*) 'error in eval_table'
               if (off_table_option < min_atm_option) return
               ierr = 0
               beta = 0
               alfa = 1
            end if
         end if
          
         if (alfa > 0) then
            if (off_table_option == atm_simple_photosphere) then
               call eval_grey( &
                  Pextra_factor, tau, g, kap_in, L, cgrav, M, Teff, &
                  lnT1, dlnT_dL1, dlnT_dlnR1, dlnT_dlnM1, dlnT_dlnkap1, & 
                  lnP1, dlnP_dL1, dlnP_dlnR1, dlnP_dlnM1, dlnP_dlnkap1, &
                  ierr)
               if (ierr /= 0) then
                  if (dbg) write(*,*) 'failed in eval_grey'
                  return
               end if
            else if (off_table_option == atm_grey_and_kap) then
               call do_atm_grey_and_kap( &
                  Pextra_factor, tau, kap_in, cgrav, M, R, L, X, Z, abar, zbar, & 
                  species, chem_id, net_iso, xa, &
                  max_tries, atol, rtol, eos_handle, kap_handle, &
                  lnT1, dlnT_dL1, dlnT_dlnR1, dlnT_dlnM1, & 
                  lnP1, dlnP_dL1, dlnP_dlnR1, dlnP_dlnM1, &
                  kap_out, Teff, iters, err, ierr) 
               dlnT_dlnkap1 = 0
               dlnP_dlnkap1 = 0
               if (ierr /= 0) then
                  if (dbg) write(*,*) 'failed in eval_grey'
                  return
               end if
            else
               ierr = -1
               if (dbg) write(*,*) 'invalid off_table_option', off_table_option
               return
            end if
         end if
         
         if (alfa == 1) then ! pure grey
            if (dbg) write(*,*) 'pure grey', lnP1
            if (is_bad_num(lnP1)) then
               write(*,1) 'tau', tau
               write(*,1) 'g', g
               write(*,1) 'kap_in', kap_in
               write(*,1) 'L', L
               write(*,1) 'cgrav', cgrav
               write(*,1) 'M', M
               write(*,1) 'Teff', Teff
               
               stop
            end if
            lnP = lnP1
            dlnP_dL = dlnP_dL1
            dlnP_dlnR = dlnP_dlnR1
            dlnP_dlnM = dlnP_dlnM1
            dlnP_dlnkap = dlnP_dlnkap1
            lnT = lnT1
            dlnT_dL = dlnT_dL1
            dlnT_dlnR = dlnT_dlnR1
            dlnT_dlnM = dlnT_dlnM1
            dlnT_dlnkap = dlnT_dlnkap1
            return
         end if
         
         if (beta == 1) then ! pure table
            if (dbg) write(*,*) 'pure table', lnP2
            lnP = lnP2
            dlnP_dL = dlnP_dL2
            dlnP_dlnR = dlnP_dlnR2
            dlnP_dlnM = dlnP_dlnM2
            dlnP_dlnkap = dlnP_dlnkap2
            lnT = lnT2
            dlnT_dL = dlnT_dL2
            dlnT_dlnR = dlnT_dlnR2
            dlnT_dlnM = dlnT_dlnM2
            dlnT_dlnkap = dlnT_dlnkap2
            return
         end if
         
         ! mixture of table and grey
         if (dbg) write(*,1) 'mixture of table and grey', alfa, beta
         
         lnP = alfa*lnP1 + beta*lnP2
         dlnP_dL = alfa*dlnP_dL1 + beta*dlnP_dL2
         dlnP_dlnR = alfa*dlnP_dlnR1 + beta*dlnP_dlnR2
         dlnP_dlnM = alfa*dlnP_dlnM1 + beta*dlnP_dlnM2
         dlnP_dlnkap = alfa*dlnP_dlnkap1 + beta*dlnP_dlnkap2
         
         lnT = alfa*lnT1 + beta*lnT2
         dlnT_dL = alfa*dlnT_dL1 + beta*dlnT_dL2
         dlnT_dlnR = alfa*dlnT_dlnR1 + beta*dlnT_dlnR2
         dlnT_dlnM = alfa*dlnT_dlnM1 + beta*dlnT_dlnM2
         dlnT_dlnkap = alfa*dlnT_dlnkap1 + beta*dlnT_dlnkap2


         contains

#ifdef offload
         !dir$ attributes offload: mic :: eval_table
#endif         
         subroutine eval_table(ierr)
            use table_atm, only: get_table_values
            use utils_lib, only: is_bad_num
            integer, intent(out) :: ierr

            real(dp) :: Prad, dPrad_dlnTeff, dPrad_dlnR, dPrad_dL, &
               P, Pgas, dPgas_dTeff, dPgas_dlogg, dPgas_dlnR, dPgas_dlnM, dPgas_dlnTeff, &
               T, dT_dTeff, dT_dlogg, dT_dlnR, dT_dlnM, dT_dlnT, dT_dlnTeff
            
            
            include 'formats.dek'
            
            ierr = 0
         
            if (dbg) write(*,*) 'call get_table_values', which_atm_option
            call get_table_values( &
               which_atm_option, Z, logg, Teff, &
               Pgas, dPgas_dTeff, dPgas_dlogg, &
               T, dT_dTeff, dT_dlogg, &
               ierr)

            if (ierr /= 0) then
               if (dbg) write(*,*) 'get_table_values(_at_fixed_Z) ierr', ierr
               return
            end if

            dPgas_dlnR = dPgas_dlogg*d_logg_dlnR + dPgas_dTeff*dTeff_dlnR
            dPgas_dlnM = dPgas_dlogg*d_logg_dlnM
            dPgas_dlnTeff = dPgas_dTeff*Teff

            Prad = crad*T*T*T*T / 3
            dPrad_dlnTeff = 4*Prad*dT_dTeff*Teff/T
            dPrad_dlnR = dPrad_dlnTeff*dlnTeff_dlnR
            dPrad_dL = dPrad_dlnTeff*dlnTeff_dL

            P = Pgas + Prad         
            lnP2 = log_cr(P)
            dlnP_dL2 = (dPgas_dlnTeff + dPrad_dlnTeff)*dlnTeff_dL / P
            dlnP_dlnR2 = (dPgas_dlnR + dPrad_dlnTeff*dlnTeff_dlnR) / P
            dlnP_dlnM2 = dPgas_dlnM/P
            dlnP_dlnkap2 = 0

            dT_dlnTeff = dT_dTeff*Teff
            dT_dlnR = dT_dlogg*d_logg_dlnR + dT_dlnTeff*dlnTeff_dlnR
            dT_dlnM = dT_dlogg*d_logg_dlnM
            
            lnT2 = log_cr(T)
            dlnT_dL2 = dT_dlnTeff*dlnTeff_dL / T
            dlnT_dlnR2 = dT_dlnR / T
            dlnT_dlnM2 = dT_dlnM / T
            dlnT_dlnkap2 = 0
            
            if (dbg .or. is_bad_num(lnP2) .or. is_bad_num(lnT2)) then
               write(*,*) 'eval_table'
               write(*,1) 'Teff', Teff
               write(*,1) 'T', T
               write(*,1) 'dT_dTeff', dT_dTeff
               write(*,1) 'dT_dlogg', dT_dlogg
               write(*,*)
               if (is_bad_num(lnP2) .or. is_bad_num(lnT2)) stop 'eval_table'
            end if            

         end subroutine eval_table

#ifdef offload
         !dir$ attributes offload: mic :: get_alfa_beta
#endif         
         subroutine get_alfa_beta(ierr)
            use table_atm, only: &
               Atm_Info, ai_two_thirds, ai_100, ai_10, ai_1, ai_1m1, ai_wd_25
            integer, intent(out) :: ierr
            
            type (Atm_Info), pointer :: ai
            real(dp) :: logg_max, logg_min, logTeff_max, logTeff_min, Teff_max, Teff_min ! range of tables
            real(dp) :: &
               logTeff, logg_min_margin, logg_max_margin, logTeff_min_margin, logTeff_max_margin, &
               logg1, logg2, logg3, logg4, logTeff1, logTeff2, logTeff3, logTeff4, c_dx, c_dy
            
            integer, parameter :: pure_grey = 1
            integer, parameter :: pure_table = 2
            integer, parameter :: blend_in_x = 3
            integer, parameter :: blend_in_y = 4
            integer, parameter :: blend_corner_out = 5
            integer, parameter :: blend_corner_in = 6
            
            integer :: iregion, logg_index, ng, j

            include 'formats.dek'

            ierr = 0
            
            if (which_atm_option == atm_photosphere_tables) then
               ai => ai_two_thirds
               if (dbg) write(*,*) '( use tau=2/3 tables )'
            else if (which_atm_option == atm_tau_100_tables) then
               ai => ai_100
               if (dbg) write(*,*) '( use tau=100 tables )'
            else if (which_atm_option == atm_tau_10_tables) then
               ai => ai_10
               if (dbg) write(*,*) '( use tau=10 tables )'
            else if (which_atm_option == atm_tau_1_tables) then
               ai => ai_1
               if (dbg) write(*,*) '( use tau=1 tables )'
            else if (which_atm_option == atm_tau_1m1_tables) then
               ai => ai_1m1
               if (dbg) write(*,*) '( use tau=1m1 tables )'
            else if (which_atm_option == atm_WD_tau_25_tables) then
               ai => ai_wd_25
               if (dbg) write(*,*) '( use WD_tau_25_tables )'
            else
               if (dbg) write(*,*) '( not using atm tables )'
               alfa = 1
               beta = 0
               return
            end if
            
            ng = ai% ng
            logg_max = ai% logg_array(ng)
            logg_min = ai% logg_array(1)

            !first, locate current point logg array for use with Teff_bound
            if (logg <= logg_min) then
               logg_index = 1
            else if (logg >= logg_max) then
               logg_index = ng
            else
               logg_index = ng
               do j=1,ng-1
                  if (ai% logg_array(j) <= logg) then
                     logg_index = j
                  end if
               end do
            end if
            
            logTeff = log10_cr(Teff)
            Teff_max = ai% Teff_bound(logg_index)
            Teff_min = ai% Teff_array(1)
            logTeff_max = log10_cr(Teff_max)
            logTeff_min = log10_cr(Teff_min)
            
            if (dbg) then
               write(*,2) 'ng', ng
               do j=1,ng
                  write(*,2) 'logg', j, ai% logg_array(j)
               end do
               write(*,1) 'logg_max', logg_max
               write(*,1) 'logg_min', logg_min
               write(*,1) 'logg', logg
               write(*,*)
               write(*,1) 'T_max', ai% Teff_bound(logg_index)
               write(*,1) 'T_min', ai% Teff_array(1)
               write(*,1) 'logTeff_max', logTeff_max
               write(*,1) 'logTeff_min', logTeff_min
               write(*,1) 'logTeff', logTeff
               write(*,*)
               !write(*,1) 'logg array', ai% logg_array(:)
               !stop
            end if

            logg_max_margin = 0.01d0
            logg_min_margin = 0.5d0
            logTeff_max_margin = 0.01d0
            logTeff_min_margin = 0.01d0
            
            if (which_atm_option == atm_WD_tau_25_tables) then
               logg_max_margin = 0.5d0
               logg_min_margin = 0.5d0
               logTeff_max_margin = 0.2d0
               logTeff_min_margin = 1d0
               logg1 = logg_max + logg_max_margin
               logg2 = logg_max
               logg3 = logg_min
               logg4 = logg_min - logg_min_margin         
               logTeff1 = logTeff_max + logTeff_max_margin
               logTeff2 = logTeff_max
               logTeff3 = logTeff_min
               logTeff4 = logTeff_min - logTeff_min_margin
            else        
               logg1 = logg_max
               logg2 = logg_max - logg_max_margin
               logg3 = logg_min + logg_min_margin
               logg4 = logg_min         
               logTeff1 = logTeff_max
               logTeff2 = logTeff_max - logTeff_max_margin
               logTeff3 = logTeff_min
               logTeff4 = logTeff_min
            end if
            
            if (dbg) then
               write(*,1) 'Teff_max', Teff_max
               write(*,1) 'logTeff_max_margin', logTeff_max_margin
               write(*,1) 'Teff_min', Teff_min
               write(*,1) 'logTeff_min_margin', logTeff_min_margin
               write(*,1) 'logTeff1', logTeff1
               write(*,1) 'logTeff2', logTeff2
               write(*,1) 'logTeff3', logTeff3
               write(*,1) 'logTeff4', logTeff4
               write(*,*)
               write(*,1) 'logg1', logg1
               write(*,1) 'logg2', logg2
               write(*,1) 'logg3', logg3
               write(*,1) 'logg4', logg4
               write(*,*)
            end if
         
            if (logg < logg4 .or. logg > logg1 .or. logTeff > logTeff1) then
               iregion = pure_grey
               if (dbg) then
                  write(*,*) 'case 1'
                  write(*,*) 'logg <= logg4', logg <= logg4
                  write(*,*) 'logg >= logg1', logg >= logg1
                  write(*,*) 'logTeff >= logTeff1', logTeff >= logTeff1
                  write(*,*)
               end if
            else if (logTeff > logTeff2) then
               c_dy = (logTeff - logTeff2) / (logTeff1 - logTeff2)
               if (logg > logg2) then
                  c_dx = (logg - logg2) / (logg1 - logg2)
                  iregion = blend_corner_out
                  if (dbg) write(*,*) 'case 2'
               else if (logg > logg3) then
                  iregion = blend_in_y
                  if (dbg) write(*,*) 'case 3'
               else ! logg > logg4
                  c_dx = (logg - logg3) / (logg4 - logg3)
                  iregion = blend_corner_out
                  if (dbg) write(*,*) 'case 4'
               end if
            else if (logTeff >= logTeff3) then
               if (logg > logg2) then
                  c_dx = (logg - logg2) / (logg1 - logg2)
                  iregion = blend_in_x
                  if (dbg) write(*,*) 'case 5'
               else if (logg > logg3) then
                  iregion = pure_table
                  if (dbg) write(*,*) 'case 6'
               else ! logg > logg4
                  c_dx = (logg - logg3) / (logg4 - logg3)
                  iregion = blend_in_x
                  if (dbg) write(*,*) 'case 7'
               end if
            else if (logTeff > logTeff4) then
               c_dy = (logTeff - logTeff3) / (logTeff4 - logTeff3)
               if (logg > logg2) then
                  c_dx = (logg - logg2) / (logg1 - logg2)
                  iregion = blend_corner_out
                  if (dbg) write(*,*) 'case 8'
               else if (logg > logg3) then
                  iregion = blend_in_y
                  if (dbg) write(*,*) 'case 9'
               else ! logg > logg4
                  c_dx = (logg - logg3) / (logg4 - logg3)
                  iregion = blend_corner_out
                  if (dbg) write(*,*) 'case 10'
               end if
            else ! logTeff <= logTeff4
               iregion = pure_grey
               if (dbg) write(*,*) 'case 11'
            end if
         
            if (iregion == pure_grey) then
               alfa = 1
               beta = 0
            else if (iregion == pure_table) then
               alfa = 0
               beta = 1
            else if (iregion == blend_in_y) then
               alfa = c_dy
               beta = 1 - alfa
            else if (iregion == blend_in_x) then
               alfa = c_dx
               beta = 1 - alfa
            else if (iregion == blend_corner_out) then
               alfa = min(1d0, sqrt(c_dx*c_dx + c_dy*c_dy))
               beta = 1 - alfa
            else
               ierr = -1
               return
            end if
            
         end subroutine get_alfa_beta
         
         
      end subroutine do_atm_get


      subroutine get_grey( &
            Pextra_factor, tau, cgrav, M, R, L, kap, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, & 
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
         real(dp), intent(in) :: Pextra_factor, tau, cgrav, M, R, L, kap 
         real(dp), intent(out) :: Teff, lnT, lnP
         real(dp), intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap
         real(dp), intent(out) :: dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
         integer, intent(out) :: ierr  ! == 0 means AOK         
         real(dp) :: g         
         if (L <= 0) then
            ierr = -1
            return
         end if
         ierr = 0
         Teff = eval_Teff(L, R)
         g = cgrav * M / (R*R)
         call eval_grey( &
            tau, g, Pextra_factor, kap, L, cgrav, M, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, & 
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)         
      end subroutine get_grey


            
      subroutine eval_grey( &
            tau, g, Pextra_factor, kap, L, cgrav, M, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, & 
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
         real(dp), intent(in) :: tau, g, Pextra_factor, kap, L, cgrav, M, Teff
         real(dp), intent(out) :: lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap
         real(dp), intent(out) :: lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
         integer, intent(out) :: ierr

         real(dp) :: &
            P0, Pextra, Pextra2, Pfactor, P, d_logg_dlnR, d_logg_dlnM, &
            dP0_dlnR, dP0_dlnkap, dP0_dL, dP0_dlnM, &
            dPfactor_dlnR, dPfactor_dlnkap, dPfactor_dL, dPfactor_dlnM, &
            lnTeff, dlnTeff_dL, dlnTeff_dlnR, Teff4, T4, T, dlnT_dlnTeff
         
         include 'formats'
         
         ierr = 0
         
         ! P = (tau*g/kap)*(1 + 1.6d-4*kap*(L/Lsun)/(M/Msun))
         ! the factor comes from including nonzero Prad at tau=0
         ! see, e.g., Cox & Giuli, Section 20.1
         P0 = tau*g/kap   
         if (Pextra_factor < 0) then ! old form
            Pextra = 1.6d-4*kap*(L/Lsun)/(M/Msun)
         else      
            Pextra = Pextra_factor*(kap/tau)*(L/M)/(6d0*pi*clight*cgrav)
         end if
         ! thanks to Konstantin Pavlovskii for correcting an earlier version of this
         ! for tau=2/3 the new Pextra is smaller by about a factor of 4
         ! since massive stars can have L/M > 10^4,
         ! this can make a big difference in P surface.

         Pfactor = 1 + Pextra
         P = P0*Pfactor
         lnP = log_cr(P)
         
         d_logg_dlnR = -2
		   d_logg_dlnM = 1

         dP0_dlnR = d_logg_dlnR*P0
         dP0_dlnkap = -P0
         dP0_dL = 0
         dP0_dlnM = d_logg_dlnM*P0
         
         dPfactor_dlnR = 0
         dPfactor_dlnkap = Pextra
         dPfactor_dL = Pextra/L
         dPfactor_dlnM = -Pextra
         
         dlnP_dL = (dP0_dL*Pfactor + P0*dPfactor_dL)/P
         dlnP_dlnR  = (dP0_dlnR*Pfactor + P0*dPfactor_dlnR)/P
         dlnP_dlnM = (dP0_dlnM*Pfactor + P0*dPfactor_dlnM)/P
         dlnP_dlnkap = (dP0_dlnkap*Pfactor + P0*dPfactor_dlnkap)/P
         
         lnTeff = log_cr(Teff)
         dlnTeff_dL = 1/(4*L)
         dlnTeff_dlnR = -0.5d0
         
         Teff4 = Teff*Teff*Teff*Teff
         T4 = 0.75d0*Teff4*(tau + 2d0/3d0) ! eddington
         lnT = log_cr(T4)*0.25d0
         T = exp_cr(lnT)
         dlnT_dlnTeff = 1
         
         dlnT_dL = dlnTeff_dL*dlnT_dlnTeff
         dlnT_dlnR  = dlnTeff_dlnR*dlnT_dlnTeff
         dlnT_dlnM = 0
         dlnT_dlnkap = 0
         
      end subroutine eval_grey
           
      
      ! like atm_get_grey,
      ! but adjusts kap until it is consistent with lnP and lnT for given composition
      ! this involves repeated calls on the eos and kap modules
      subroutine do_atm_grey_and_kap( &
            Pextra_factor, tau, kap_guess, &
            cgrav, M, R, L, X, Z, abar, zbar, & 
            species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, & 
            kap, Teff, iters, err, ierr) 
         
         use integrate_atm, only: get_atm_eos_kap_info
         
         ! cgs units
         real(dp), intent(in) :: Pextra_factor, tau   ! optical depth
         real(dp), intent(in) :: kap_guess ! initial guess for opacity
         real(dp), intent(in) :: cgrav, M, R, L, X, Z, abar, zbar
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in)  :: xa(:) ! mass fractions
         
         integer, intent(in) :: max_tries
         real(dp), intent(in) :: atol, rtol
         
         integer, intent(in) :: eos_handle
         integer, intent(in) :: kap_handle
         
         real(dp), intent(out) :: lnT ! natural log of temperature at base of atmosphere
         real(dp), intent(out) :: lnP ! natural log of pressure at base of atmosphere (Pgas + Prad)
         
         ! partial derivatives of lnT and lnP (not evaluated if skip_partials is true)
         real(dp), intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM
         real(dp), intent(out) :: dlnP_dL, dlnP_dlnR, dlnP_dlnM

         real(dp), intent(out) :: kap ! opacity consistent with lnT and lnP
         real(dp), intent(out) :: Teff ! temperature at photosphere

         integer, intent(out) :: iters
         real(dp), intent(out) :: err
         
         integer, intent(out) :: ierr  ! == 0 means AOK

         real(dp) :: g, dlnT_dlnkap, dlnP_dlnkap, T, P, Prad, Pgas, lnPgas, &
            Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            kap_prev, dlnkap_dlnRho, dlnkap_dlnT, dlnP_dlnPgas, df_dkap

         include 'formats.dek'

         if (L <= 0) then
            ierr = -1
            return
         end if
         ierr = 0
         Teff = eval_Teff(L, R)
         g = cgrav * M / (R*R)
         err = 1d99
         
         kap = kap_guess
         
         do iters = 1, max_tries
         
            call eval_grey( &
               tau, g, Pextra_factor, kap, L, cgrav, M, Teff, &
               lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, & 
               lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
               ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in eval_grey'
               return
            end if
            
            if (.false.) then
               write(*,1) 'tau', tau
               write(*,1) 'L', L
               write(*,1) 'M', M
               write(*,1) 'R', R
               write(*,1) 'kap', kap
               write(*,1) 'lnT', lnT
               write(*,1) 'lnP', lnP
            end if
            
            T = exp_cr(lnT)
            P = exp_cr(lnP)
            Prad = crad*T*T*T*T/3
            Pgas = max(1d-99, P - Prad)
            lnPgas = log_cr(Pgas)
         
            kap_prev = kap
            call get_atm_eos_kap_info( &
               -1, kap_handle, eos_handle, tau, cgrav, L, m, r, &
               Pgas, lnPgas, T, lnT, Z, 1-(X+Z), X, abar, zbar, &
               species, chem_id, net_iso, xa, &
               Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
            if (ierr /= 0) then
               return
               
               write(*,*) 'failed in get_atm_eos_kap_info'
               write(*,1) 'Pgas', Pgas
               write(*,1) 'lnPgas', lnPgas
               write(*,1) 'T', T
               write(*,1) 'lnT', lnT
               write(*,1) 'Z', Z
               write(*,1) 'Y', 1-(X+Z)
               write(*,1) 'X', X
               write(*,1) 'abar', abar
               write(*,1) 'zbar', zbar
               write(*,1) 'kap', kap
               stop 'do_atm_grey_and_kap'
               return
            end if
            
            err = abs(kap_prev - kap)/(atol + rtol*kap)
            
            !write(*,2) 'err kap lgP', iters, err, kap, lnP/ln10

            if (err < 1) exit
            
            kap = kap_prev + 0.5*(kap - kap_prev) ! under correct
            
         end do

     
      end subroutine do_atm_grey_and_kap
      
      
      subroutine do_atm_grey_irradiated( &
            T_eq, kap_v, kap_guess, use_kap_th_guess, kap_v_div_kap_th, & 
            P, cgrav, M, R, L, &
            X, Z, abar, zbar, species, chem_id, net_iso, xa, &
            max_tries, atol, rtol, eos_handle, kap_handle, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
            kap, tau, Teff, iters, err, ierr) 
         
         use integrate_atm, only: get_atm_eos_kap_info
         
         ! cgs units
         real(dp), intent(in) :: T_eq ! equilibrium temperature based on irradiation
            ! e.g., might use T_eq = Teff_star*(R_star/(2*Distance_to_star))^(1/2)
         real(dp), intent(in) :: kap_v ! visible opacity
         real(dp), intent(in) :: kap_guess ! starting guess for thermal opacity
         logical, intent(in) :: use_kap_th_guess ! if true, use it.  else iterate.
         real(dp), intent(in) :: kap_v_div_kap_th
         real(dp), intent(in) :: P ! pressure at base of atmosphere
         real(dp), intent(in) :: cgrav, M, R, L, X, Z, abar, zbar
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         integer, intent(in) :: max_tries
         real(dp), intent(in) :: atol, rtol
         
         integer, intent(in) :: eos_handle
         integer, intent(in) :: kap_handle
         
         real(dp), intent(out) :: lnT ! natural log of temperature at boundary
         ! partial derivatives of lnT
         real(dp), intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM

         real(dp), intent(out) :: kap ! thermal opacity at boundary
         real(dp), intent(out) :: tau ! optical depth at boundary
         real(dp), intent(out) :: Teff ! temperature at tau = 2/3

         integer, intent(out) :: iters
         real(dp), intent(out) :: err
         
         integer, intent(out) :: ierr  ! == 0 means AOK
         
         real(dp) :: &
            T_eq4, g, T_int, T_int4, T4, T, dlnT_dlnkap, dlnP_dlnkap, Prad, Pgas, lnPgas, &
            Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            kap_prev, dlnkap_dlnRho, dlnkap_dlnT, dlnP_dlnPgas, df_dkap, gamma, tau_eff, &
            dT_int4_dL, dT_int4_dR, dg_dlnM, dg_dlnR, lnP, dtau_dg, dtau_dR, dtau_dM, &
            tau_eq1, E2, tau_eq2, tau_eq, dT_int4_dlnR, dtau_dlnR, dtau_dlnM, &
            gamma_tau, dgt_dlnR, dgt_dlnM, exp_minus_gamma_tau, d_emgt_dlnR, d_emgt_dlnM, &
            d_tau_eq1_dlnR, d_tau_eq1_dlnM, dE2_dgt, dE2_dlnR, dE2_dlnM, &
            d_tau_eq2_dlnR, d_tau_eq2_dlnM, d_tau_eq_dlnR, d_tau_eq_dlnM, &
            T4_int, dT4_int_dL, dT4_int_dlnR, dT4_int_dlnM, &
            T4_eq, dT4_eq_dL, dT4_eq_dlnR, dT4_eq_dlnM, &
            dT4_dL, dT4_dlnR, dT4_dlnM, dlnT_dT4

         include 'formats.dek'

         ierr = 0
         lnP = log_cr(P)
         err = 1d99
         
         T_eq4 = T_eq*T_eq*T_eq*T_eq

         g = cgrav*M/(R*R)
         dg_dlnM = g
         dg_dlnR = -2*g
         
         T_int = pow_cr(L / (pi*crad*clight*R*R), 0.25d0)
         T_int4 = T_int*T_int*T_int*T_int
         dT_int4_dL = T_int4/L
         dT_int4_dlnR = -2*T_int4
         gamma = 0
         
         kap = kap_guess
         
         ! we ignore partials of kap in calculating partials of T
         
         do iters = 1,max_tries
            
            if (kap_v_div_kap_th > 0) then
               gamma = kap_v_div_kap_th
            else
               gamma = kap_v / kap
            end if
            
            tau = P*kap/g
            dtau_dg = -tau/g
            dtau_dlnR = dtau_dg*dg_dlnR
            dtau_dlnM = dtau_dg*dg_dlnM

            gamma_tau = gamma*tau

            dgt_dlnR = gamma*dtau_dlnR
            dgt_dlnM = gamma*dtau_dlnM
            
            exp_minus_gamma_tau = exp_cr(-gamma_tau)
            d_emgt_dlnR = -gamma*dtau_dlnR*exp_minus_gamma_tau
            d_emgt_dlnM = -gamma*dtau_dlnM*exp_minus_gamma_tau
            
            tau_eq1 = (2/(3*gamma))*(1 + (gamma_tau/2 - 1)*exp_minus_gamma_tau)
            d_tau_eq1_dlnR = (2/(3*gamma))* &
               ((gamma_tau/2 - 1)*d_emgt_dlnR + gamma*dtau_dlnR/2*exp_minus_gamma_tau)
            d_tau_eq1_dlnM = (2/(3*gamma))* &
               ((gamma_tau/2 - 1)*d_emgt_dlnM + gamma*dtau_dlnM/2*exp_minus_gamma_tau)
            
            call get_E2(gamma_tau, E2, dE2_dgt, ierr)
            if (ierr /= 0) return
            
            dE2_dlnR = dE2_dgt*dgt_dlnR
            dE2_dlnM = dE2_dgt*dgt_dlnM

            tau_eq2 = (2*gamma/3)*(1 - tau*tau/2)*E2
            d_tau_eq2_dlnR = (2*gamma/3)* &
               ((1 - tau*tau/2)*dE2_dlnR - (2*gamma/3)*tau*dtau_dlnR*E2)
            d_tau_eq2_dlnM = (2*gamma/3)* &
               ((1 - tau*tau/2)*dE2_dlnM - (2*gamma/3)*tau*dtau_dlnM*E2)               
            
            tau_eq = tau_eq1 + tau_eq2
            d_tau_eq_dlnR = d_tau_eq1_dlnR + d_tau_eq2_dlnR
            d_tau_eq_dlnM = d_tau_eq1_dlnM + d_tau_eq2_dlnM
            
            T4_int = 0.75d0*T_int4*(2d0/3d0 + tau)
            dT4_int_dL = 0.75d0*dT_int4_dL*(2d0/3d0 + tau)
            dT4_int_dlnR = 0.75d0*dT_int4_dlnR*(2d0/3d0 + tau) + 0.75d0*T_int4*dtau_dlnR
            dT4_int_dlnM = 0.75d0*T_int4*dtau_dlnM
         
            T4_eq = 0.75d0*T_eq4*(2d0/3d0 + tau_eq)
            dT4_eq_dL = 0
            dT4_eq_dlnR = 0.75d0*T_eq4*d_tau_eq2_dlnR
            dT4_eq_dlnM = 0.75d0*T_eq4*d_tau_eq2_dlnM
         
            T4 = T4_int + T4_eq
            dT4_dL = dT4_int_dL + dT4_eq_dL
            dT4_dlnR = dT4_int_dlnR + dT4_eq_dlnR
            dT4_dlnM = dT4_int_dlnM + dT4_eq_dlnM
                 
            lnT = 0.25d0*log_cr(T4)
            dlnT_dT4 = 0.25d0/T4
            dlnT_dL = dlnT_dT4*dT4_dL
            dlnT_dlnR = dlnT_dT4*dT4_dlnR
            dlnT_dlnM = dlnT_dT4*dT4_dlnM
            
            T = exp_cr(lnT)
            Prad = crad*T*T*T*T/3
            Pgas = max(1d-99, P - Prad)
            lnPgas = log_cr(Pgas)
            
            if (use_kap_th_guess) exit

            kap_prev = kap
            call get_atm_eos_kap_info( &
               -1, kap_handle, eos_handle, tau, cgrav, L, m, r, &
               Pgas, lnPgas, T, lnT, Z, 1-(X+Z), X, abar, zbar, &
               species, chem_id, net_iso, xa, &
               Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
            if (ierr /= 0) then
               return
               
               write(*,*) 'failed in get_atm_eos_kap_info'
               write(*,1) 'Pgas', Pgas
               write(*,1) 'lnPgas', lnPgas
               write(*,1) 'T', T
               write(*,1) 'lnT', lnT
               write(*,1) 'Z', Z
               write(*,1) 'Y', 1-(X+Z)
               write(*,1) 'X', X
               write(*,1) 'abar', abar
               write(*,1) 'zbar', zbar
               write(*,1) 'kap', kap
               stop 'do_atm_irradiated'
               return
            end if
            
            err = abs(kap_prev - kap)/(atol + rtol*kap)
            
            
            if (.false.) then
               write(*,2) 'iters', iters
               write(*,1) 'logPgas', lnPgas/ln10
               write(*,1) 'logT', lnT/ln10
               write(*,1) 'logRho', logRho
               write(*,1) 'kap', kap
               write(*,1) 'tau', tau
               write(*,1) 'err', err
               write(*,*)
            end if
            
            !write(*,2) 'kap', iters, kap, err
            !if (iters == 2) stop 'do_atm_irradiated'
            
            if (err < 1) exit
            kap = kap_prev + 0.9*(kap - kap_prev) ! under correct
            
         end do
      
         ! calculate Teff
         tau_eff = 2d0/3d0
         tau_eq1 = (1 + (gamma*tau_eff/2 - 1)*exp_cr(-gamma*tau_eff))*2/(3*gamma)
         E2 = 0.1
         tau_eq2 = (1 - tau_eff*tau_eff/2)*E2*2*gamma/3
         tau_eq = tau_eq1 + tau_eq2
      
         T4 = 0.75d0*T_int4*(2d0/3d0 + tau_eff) + &
              0.75d0*T_eq4*(2d0/3d0 + tau_eq)
         Teff = pow_cr(T4,0.25d0)
         
         
         contains
         
#ifdef offload
         !dir$ attributes offload: mic :: get_E2
#endif         
         subroutine get_E2(x, E2, dE2_dx, ierr)
            use interp_1d_lib
            use interp_1d_def
            real(dp), intent(in) :: x
            real(dp), intent(out) :: E2, dE2_dx
            integer, intent(out) :: ierr            
            real(dp) :: val, slope            
            ierr = 0
            E2 = 0
            dE2_dx = 0            
            if (.not. have_E2_interpolant) then
               call create_E2_interpolant(ierr)
               if (ierr /= 0) return
            end if            
            call interp_value_and_slope(E2_x, npairs, E2_f1, x, val, slope, ierr)
            if (ierr /= 0) then
               write(*,*) 'do_atm_grey_irradiated: failed in interp_value_and_slope'
               return
            end if      
            ! val = log10[E2]
            E2 = exp10_cr(val)
            dE2_dx = slope*ln10*E2         
         end subroutine get_E2         
     
      end subroutine do_atm_grey_irradiated
      
      
      subroutine create_E2_interpolant(ierr)
         use interp_1d_lib
         use interp_1d_def
         integer, intent(out) :: ierr            
         integer, parameter :: nwork = pm_work_size
         real(dp), target :: work_ary(npairs*nwork)
         real(dp), pointer :: work(:)
         integer :: i
         work => work_ary
         do i=1,npairs
            E2_x(i) = E2_pairs(2*i-1)
            E2_f(1,i) = E2_pairs(2*i)
         end do
         ierr = 0
         call interp_pm(E2_x, npairs, E2_f1, nwork, work, 'do_atm_grey_irradiated', ierr)
         if (ierr /= 0) then
            write(*,*) 'do_atm_grey_irradiated: failed in interp_pm'
         end if      
         have_E2_interpolant = .true.
      end subroutine create_E2_interpolant

#ifdef offload
      !dir$ end options
#endif

      end module mod_atm
