! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module integrate_atm
      
      use const_def
      use atm_def
      use chem_def, only: num_chem_isos
      use crlibm_lib
      
      implicit none
      
      
      logical, parameter :: dbg = .false.
      
      
      integer, parameter :: r_T0 = 1                
      integer, parameter :: r_lnT0 = 2              
      integer, parameter :: r_Z = 3                 
      integer, parameter :: r_X = 4                 
      integer, parameter :: r_abar = 5              
      integer, parameter :: r_zbar = 6             
      integer, parameter :: r_g = 7               
      integer, parameter :: r_tau0 = 8            
      integer, parameter :: r_Teff4 = 9           
      integer, parameter :: r_Prad_tau_eq_0 = 10    
      integer, parameter :: r_Prad_tau0 = 11        
      integer, parameter :: r_lntau_stop = 12      
      integer, parameter :: r_Rho = 13              
      integer, parameter :: r_kap = 14              
      integer, parameter :: r_lnT = 15
      integer, parameter :: r_cgrav = 16
      integer, parameter :: r_L = 17
      integer, parameter :: r_m = 18
      integer, parameter :: r_r = 19
      
      integer, parameter :: atm_lrpar = 19
      
      integer, parameter :: i_atm_handle = 1
      integer, parameter :: i_eos_handle = 2
      integer, parameter :: i_kap_handle = 3
      integer, parameter :: i_which_T_tau = 4
      integer, parameter :: i_species = 5
      
      integer, parameter :: atm_lipar = 5


      
      contains

      
      subroutine get_atm_eos_op_mono_kap_info( &
            handle, kap_handle, eos_handle, tau, cgrav, L, m, r, &
            Pgas, lnPgas, T, lnT, Z, Y, X, abar, zbar, &
            species, chem_id, net_iso, xa, &
            use_op_mono_alt_get_kap, op_mono_min_X_to_include, &
            op_mono_data_path, op_mono_data_cache_filename, &
            Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         use eos_def
         use eos_lib
         use kap_def
         use kap_lib

         integer, intent(in) :: handle, kap_handle, eos_handle, species
         real(dp), intent(in) :: &
            tau, cgrav, L, m, r, Pgas, lnPgas, T, lnT, Z, Y, X, &
            abar, zbar
         integer, pointer :: chem_id(:), net_iso(:)
         real(dp), intent(in) :: xa(:)
         logical, intent(in) :: use_op_mono_alt_get_kap
         real(dp), intent(in) :: op_mono_min_X_to_include
         character (len=*), intent(in) :: &
            op_mono_data_path, op_mono_data_cache_filename
         real(dp), intent(out) :: &
            Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            kap, dlnkap_dlnRho, dlnkap_dlnT
         integer, intent(out) :: ierr
      
         integer :: sz, k, nptot, ipe, nrad
         real(dp) :: lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, Prad, P, &
            gradr, gradT
         real(dp), dimension(num_eos_basic_results) :: &
            res, d_dlnRho_const_T, d_dlnT_const_Rho, &
            d_dabar_const_TRho, d_dzbar_const_TRho
         type (Int_Atm_Info), pointer :: ai
         
         real, pointer :: &
            umesh(:), ff(:,:,:,:), rs(:,:,:), ss(:,:,:,:)
         integer :: nel, izzp(species)
         real(dp) :: fap(species), gp1(species)
         logical :: screening
         
         include 'formats.dek'

         ierr = 0
        
         call eosPT_get( &
            eos_handle, Z, X, abar, zbar, &
            species, chem_id, net_iso, xa, &
            Pgas, lnPgas/ln10, T, lnT/ln10, &
            Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            res, d_dlnRho_const_T, d_dlnT_const_Rho, &
            d_dabar_const_TRho, d_dzbar_const_TRho, ierr)
         if (ierr /= 0) then
            !write(*,*) 'failed in eosPT_get'
            ierr=1
            return
         end if
         lnfree_e = res(i_lnfree_e)
         d_lnfree_e_dlnRho = d_dlnRho_const_T(i_lnfree_e)
         d_lnfree_e_dlnT = d_dlnT_const_Rho(i_lnfree_e)
            
         call get_op_mono_args( &
		      species, xa, op_mono_min_X_to_include, chem_id, &
		      nel, izzp, fap, ierr)
         if (ierr /= 0) then
            write(*,*) 'error in get_op_mono_args, ierr = ',ierr
            return
         end if

         call load_op_mono_data( &
            op_mono_data_path, op_mono_data_cache_filename, ierr)
         if (ierr /= 0) then
            write(*,*) 'error while loading OP data, ierr = ',ierr
            return
         end if

         call get_op_mono_params(nptot, ipe, nrad)
         allocate( &
            umesh(nptot), ff(nptot,ipe,4,4), &
            rs(nptot,4,4), ss(nptot,nrad,4,4), stat=ierr)
         if (ierr /= 0) return
            
         screening = .true.
         call kap_get_op_mono( &
            kap_handle, zbar, logRho, lnT/ln10, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            use_op_mono_alt_get_kap, &
            nel, izzp, fap, screening, umesh, ff, rs, ss, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)      
         deallocate(umesh, ff, rs, ss)
         if (ierr /= 0) then
            write(*,*) 'failed in kap_get'
            return
         end if
         
         call finish_get_atm_eos_kap( &
            handle, ai, T, lnT, Pgas, lnPgas, rho, logRho, &
            kap, L, m, cgrav, tau, r, res, &
            lnfree_e, dlnkap_dlnT, dlnkap_dlnRho, ierr)
      
      end subroutine get_atm_eos_op_mono_kap_info
      
#ifdef offload
      !dir$ options /offload_attribute_target=mic
#endif      
      
      ! implementation influenced by notes from Don VandenBerg
      
      
      subroutine do_atm_int( &
            errtol, cgrav, M, R, L, X, Z, abar, zbar, &
            species, chem_id, net_iso, xa, &
            which_atm_option, eos_handle, kap_handle, save_atm_structure_info, &
            tau_base, skip_partials, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, & 
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            atm_structure_num_pts, atm_structure, &
            ierr) 
         real(dp), intent(in) :: errtol, cgrav, M, R, L
         real(dp), intent(in) :: X, Z, abar, zbar, tau_base
         integer, intent(in) :: species
         integer, pointer :: chem_id(:), net_iso(:)
         real(dp), intent(in) :: xa(:) ! mass fractions
         integer, intent(in) :: which_atm_option, eos_handle, kap_handle       
         logical, intent(in) :: save_atm_structure_info
         logical, intent(in) :: skip_partials
         real(dp), intent(out) :: Teff, lnT, lnP
         real(dp), intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap
         real(dp), intent(out) :: dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
         integer, intent(out) :: atm_structure_num_pts
         real(dp), pointer :: atm_structure(:,:) ! will be allocated if necessary
            ! (num_results_for_create_atm, num_atm_points)
         integer, intent(out) :: ierr
         
         real(dp) :: g, lntau_base, Teff4, lnTeff, dlnTeff_dlnR, dlnTeff_dL, &
            dlnP_dlnTeff, dlnT_dlnTeff, lnP1, lnP2, lnT1, lnT2, dlnTeff, lnTeff1, lnTeff2
         integer :: j, op_err, not_used
         
         include 'formats.dek'
         
         ierr = 0
         
         if (.not. valid_T_tau(which_atm_option)) then
            ierr = -1
            return
         end if
         
         g = cgrav*M/(R*R)
         lntau_base = log_cr(tau_base)
         Teff4 = L/(4d0*pi*R*R*boltz_sigma)
         Teff = pow_cr(Teff4,0.25d0)
         lnTeff = log_cr(Teff)
         dlnTeff = 1d-4
         lnTeff1 = lnTeff + dlnTeff
         lnTeff2 = lnTeff - dlnTeff
         
         if (dbg) then
            write(*,*) 'which_atm_option', which_atm_option
            write(*,1) 'M', M
            write(*,1) 'R', R
            write(*,1) 'L', L
            write(*,1) 'X', X
            write(*,1) 'Z', Z
            write(*,1) 'abar', abar
            write(*,1) 'zbar', zbar
            write(*,*)
            write(*,1) 'tau_base', tau_base
            write(*,1) 'Teff4', Teff4
            write(*,1) 'Teff', Teff
            write(*,1) 'g', g
            write(*,*)
         end if
         
         if (skip_partials) then
            call do1( &
               which_atm_option, eos_handle, kap_handle, save_atm_structure_info, &
               tau_base, lntau_base, errtol, g, cgrav, M, R, L, X, Z, abar, zbar, &
               species, chem_id, net_iso, xa, &
               Teff4, Teff, lnTeff, lnP, lnT, &
               atm_structure_num_pts, atm_structure, ierr)
         else
!$OMP PARALLEL DO PRIVATE(j,op_err)
            do j=0,2
               op_err = 0
               select case(j)
                  case (0)
                     call do1( &
                        which_atm_option, eos_handle, kap_handle, save_atm_structure_info, &
                        tau_base, lntau_base, errtol, g, cgrav, M, R, L, X, Z, abar, zbar, &
                        species, chem_id, net_iso, xa, &
                        Teff4, Teff, lnTeff, lnP, lnT, &
                        atm_structure_num_pts, atm_structure, op_err)
                  case (1)
                     call do1( &
                        which_atm_option, eos_handle, kap_handle, .false., &
                        tau_base, lntau_base, errtol, g, cgrav, M, R, L, X, Z, abar, zbar, &
                        species, chem_id, net_iso, xa, &
                        exp_cr(4*lnTeff1), exp_cr(lnTeff1), lnTeff1, lnP1, lnT1, &
                        not_used, atm_structure, op_err)
                  case (2)
                     call do1( &
                        which_atm_option, eos_handle, kap_handle, .false., &
                        tau_base, lntau_base, errtol, g, cgrav, M, R, L, X, Z, abar, zbar, &
                        species, chem_id, net_iso, xa, &
                        exp_cr(4*lnTeff2), exp_cr(lnTeff2), lnTeff2, lnP2, lnT2, &
                        not_used, atm_structure, op_err)
               end select
               if (op_err /= 0) ierr = op_err
            end do
!$OMP END PARALLEL DO         
         end if
                  
         if (ierr /= 0) then
            if (dbg) write(*,2) 'do_atm_int: ierr from do1', ierr
            return
         end if

         dlnP_dlnM = 0; dlnP_dlnkap = 0
         dlnT_dlnM = 0; dlnT_dlnkap = 0
         
         if (skip_partials) then
            dlnT_dL=0; dlnT_dlnR=0; dlnP_dL=0; dlnP_dlnR=0
            return
         end if
         
         ! do numerical estimates of d_dlnTeff
         dlnTeff_dlnR = -0.5d0
         dlnTeff_dL = 1/(4*L)
         
         dlnP_dlnTeff = (lnP1 - lnP2) / (lnTeff1 - lnTeff2)
         dlnP_dL = dlnP_dlnTeff*dlnTeff_dL
         dlnP_dlnR = dlnP_dlnTeff*dlnTeff_dlnR

         dlnT_dlnTeff = (lnT1 - lnT2) / (lnTeff1 - lnTeff2)
         dlnT_dL = dlnT_dlnTeff*dlnTeff_dL
         dlnT_dlnR = dlnT_dlnTeff*dlnTeff_dlnR
         

      end subroutine do_atm_int
      
      
      subroutine do1( &
            which_atm_option, eos_handle, kap_handle, save_atm_structure_info, &
            tau_base, lntau_base, errtol_init, g, cgrav, M, R, L, X, Z, abar, zbar, &
            species, chem_id, net_iso, xa, &
            Teff4, Teff, lnTeff, lnP, lnT, &
            atm_structure_num_pts, atm_structure, ierr)
         integer, intent(in) :: which_atm_option, eos_handle, kap_handle
         logical, intent(in) :: save_atm_structure_info
         real(dp), intent(in) :: tau_base, lntau_base, errtol_init, g, cgrav, M, R, L, X, Z, abar, zbar
         integer, intent(in) :: species
         integer, pointer :: chem_id(:), net_iso(:)
         real(dp), intent(in) :: xa(:) ! mass fractions
         real(dp), intent(in) :: Teff4, Teff, lnTeff
         real(dp), intent(out) :: lnP, lnT
         integer, intent(out) :: atm_structure_num_pts
         real(dp), pointer :: atm_structure(:,:) ! will be allocated if necessary
            ! (num_results_for_create_atm, num_atm_points)
         integer, intent(out) :: ierr
         integer :: j
         real(dp) :: tau_factor, log_tau_factor, tau0, lntau0, errtol
         logical :: okay
         include 'formats.dek'
         tau_factor = 1d-5
         log_tau_factor = -5
         errtol = errtol_init
         okay = .false.
         ierr = 0
         atm_structure_num_pts = 0
         do j=1,3 ! if fail, increase tau_factor so start at larger optical depth
            tau0 = tau_factor*tau_base
            lntau0 = log_tau_factor*ln10 + lntau_base
            if (dbg) write(*,2) 'call do_int_atm_PT', j, lntau0/ln10
            call do_int_atm_PT( &
               cgrav, M, R, L, X, Z, abar, zbar, &
               species, chem_id, net_iso, xa, &
               tau0, lntau0, g, errtol, &
               tau_base, lntau_base, Teff4, Teff, lnTeff, &
               which_atm_option, eos_handle, kap_handle, save_atm_structure_info, &
               lnP, lnT, atm_structure_num_pts, atm_structure, ierr) 
            if (ierr == 0) then
               if (dbg) write(*,*) 'do_int_atm_PT okay'
               okay = .true.; exit
            end if
            tau_factor = tau_factor*10
            log_tau_factor = log_tau_factor + 1
            errtol = errtol*10
            if (dbg) write(*,1) 'retry: log_tau_factor', log_tau_factor
         end do
         if (.not. okay) then
            ierr = -1
            if (dbg) write(*,*) 'do1 failed'
         end if
      end subroutine do1
      
      
      subroutine do_int_atm_PT( &
            cgrav, M, R, L, X, Z, abar, zbar, &
            species, chem_id, net_iso, xa, &
            tau0, lntau0, g, errtol, &
            tau_stop, lntau_stop, Teff4, Teff, lnTeff, &
            which_atm_option, eos_handle, kap_handle, save_atm_structure_info, &
            lnP, lnT, atm_structure_num_pts, atm_structure, ierr) 
         use num_lib
         use eos_lib, only: Radiation_Pressure
         real(dp), intent(in) :: &
            cgrav, M, R, L, X, Z, abar, zbar, tau0, lntau0, g, errtol, &
            tau_stop, lntau_stop, Teff4, Teff, lnTeff
         integer, intent(in) :: species
         integer, pointer :: chem_id(:), net_iso(:)
         real(dp), intent(in) :: xa(:) ! mass fractions
         integer, intent(in) :: which_atm_option, eos_handle, kap_handle
         logical, intent(in) :: save_atm_structure_info
         real(dp), intent(out) :: lnP, lnT ! at tau_stop
         integer, intent(out) :: atm_structure_num_pts
         real(dp), pointer :: atm_structure(:,:) ! will be allocated if necessary
            ! (num_results_for_create_atm, num_atm_points)
         integer, intent(out) :: ierr
         
         integer, parameter :: num_vars = 1, nrdens = 0
         integer :: liwork, lwork, newt_imax, imax, max_steps, &
            idid, lipar, lrpar, i, j, handle
         real(dp), pointer :: work(:)
         integer, pointer :: iwork(:)
         real(dp) :: rtol(1) ! relative error tolerance(s)
         real(dp) :: atol(1) ! absolute error tolerance(s)
         integer, parameter :: itol = 0 ! switch for rtol and atol
         integer, parameter :: iout = 1 ! call solout
         integer, parameter :: lout = 0 ! set to 6 for debugging
         real(dp) :: &
            T0, dT4_dtau, T4_tau_eq_0, Prad_tau_eq_0, T4_tau0, Prad_tau0, &
            lnPgas_guess, dlnPgas, epsx, epsy, lnPgas, dT4_dtau0, dT4_dtau_eq_0, &
            lntau_init, max_step_size, init_step_size, rho_tau_eq_0
         integer, pointer :: ipar(:)
         real(dp), pointer :: rpar(:)
         type (Int_Atm_Info), pointer :: ai
         real(dp), target :: y_ary(num_vars)
         real(dp), pointer :: y(:)
         
         include 'formats.dek'
         
         ierr = 0
         y => y_ary
         
         ! temperature for start of integration
         T4_tau0 = T4_of_tau(which_atm_option, tau0, lntau0, Teff4, dT4_dtau)
         T0 = pow_cr(T4_tau0,0.25d0)
         Prad_tau0 = crad*T4_tau0/3

         ! radiation pressure for start of integration
         T4_tau_eq_0 = T4_of_tau( &
            which_atm_option, 0d0, -99d0, Teff4, dT4_dtau_eq_0)
         Prad_tau_eq_0 = crad*T4_tau_eq_0/3        

         lipar = atm_lipar + species + num_chem_isos
         lrpar = atm_lrpar + species
         
         if (.not. save_atm_structure_info) then
            handle = -1
         else
            handle = do_alloc_atm(ierr)
            if (ierr /= 0) then
               write(*,*) 'do_alloc_atm failed in do_int_atm_PT'
               return
            end if
            call get_atm_ptr(handle,ai,ierr)
            if (ierr /= 0) then
               write(*,*) 'get_atm_ptr failed in do_int_atm_PT'
               return
            end if
            ai% save_atm_structure_info = .false. ! set true for solout only
            ai% atm_structure_num_pts = 0
            ai% atm_structure => atm_structure
         end if
         
         call dopri5_work_sizes(num_vars, nrdens, liwork, lwork)
         allocate(ipar(lipar), rpar(lrpar), iwork(liwork), work(lwork), stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'allocate failed in do_int_atm_PT'
            if (handle > 0) call do_free_atm(handle)
            return
         end if
         
         ipar(i_atm_handle) = handle
         ipar(i_eos_handle) = eos_handle
         ipar(i_kap_handle) = kap_handle
         ipar(i_which_T_tau) = which_atm_option
         ipar(i_species) = species
			i = atm_lipar
			do j=1,species 
			   ipar(i+j) = chem_id(j)
			end do
			i = i+species
			do j=1,num_chem_isos
			   ipar(i+j) = net_iso(j)
			end do
			i = i+num_chem_isos
         
         rpar(r_T0) = T0
         rpar(r_lnT0) = log_cr(T0)
         rpar(r_X) = X
         rpar(r_Z) = Z
         rpar(r_abar) = abar
         rpar(r_zbar) = zbar
         rpar(r_g) = g
         rpar(r_tau0) = tau0
         rpar(r_Teff4) = Teff4
         rpar(r_Prad_tau_eq_0) = Prad_tau_eq_0
         rpar(r_Prad_tau0) = Prad_tau0
         rpar(r_lntau_stop) = lntau_stop
         rpar(r_Rho) = 0
         rpar(r_kap) = 0
         rpar(r_lnT) = 0
         rpar(r_cgrav) = cgrav
         rpar(r_L) = L
         rpar(r_m) = m
         rpar(r_r) = r
         i = atm_lrpar
         rpar(i+1:i+species) = xa(1:species); i = i+species

         !try ideal gas with low initial density as the initial Pgas
         rho_tau_eq_0 = 1d-10
         lnPgas = log_cr( cgas * rho_tau_eq_0 * T0 )
         
         max_step_size = 0  !.5d0
         init_step_size = 1d-3 !max_step_size/2
         max_steps = 500

         iwork(:) = 0
         work(:) = 0
         iwork(5) = nrdens
         
         rtol(:) = errtol
         atol(:) = errtol
         
         y(1) = lnPgas
         lntau_init = lntau0

         call dopri5( &
               num_vars, atm_fcn, lntau_init, y, &
               lntau_stop, & 
               init_step_size, max_step_size, max_steps, & 
               rtol, atol, itol, & 
               atm_solout, iout, & 
               work, lwork, iwork, liwork, & 
               lrpar, rpar, lipar, ipar, & 
               lout, idid)
         
         if (idid < 0) then
            if (dbg) write(*,*) 'failed in dopri5: idid', idid
            ierr = -1
            call dealloc
            return
         end if
         
         lnT = rpar(r_lnT)
         lnP = log_cr(exp_cr(y(1)) + Radiation_Pressure(exp_cr(lnT)))
         
         if (save_atm_structure_info) call set_delta_r
         
         call dealloc
         
         
         contains
         
#ifdef offload
         !dir$ attributes offload: mic :: dealloc
#endif         
         subroutine dealloc
            if (handle > 0) then
               atm_structure_num_pts = ai% atm_structure_num_pts
               atm_structure => ai% atm_structure
               call do_free_atm(handle)
            end if
            deallocate(rpar, ipar, iwork, work)
         end subroutine dealloc
         
#ifdef offload
         !dir$ attributes offload: mic :: set_delta_r
#endif         
         subroutine set_delta_r
            real(dp) :: delta_r, delta_tau, rho, opacity
            integer :: k
            delta_r = 0 ! for the compiler
            do k=ai% atm_structure_num_pts,1,-1
               if (k == ai% atm_structure_num_pts) then
                  delta_r = 0
               else
                  rho = exp_cr(ai% atm_structure(atm_lnd,k))
                  opacity = ai% atm_structure(atm_kap,k)
                  delta_tau = ai% atm_structure(atm_tau,k+1) - ai% atm_structure(atm_tau,k)
                  delta_r = delta_r + delta_tau/(rho*opacity)
               end if 
               ai% atm_structure(atm_delta_r,k) = delta_r
            end do
         end subroutine set_delta_r
         

      end subroutine do_int_atm_PT


      subroutine atm_fcn(n, lntau, h, y, f, lr, rpar, li, ipar, ierr)
         integer, intent(in) :: n, lr, li
         real(dp), intent(in) :: lntau, h
         real(dp), intent(inout) :: y(:) ! (n) ! lnPgas
         real(dp), intent(out) :: f(:) ! (n) ! dlnPgas/dlntau
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr ! nonzero means retry with smaller step.

         real(dp) :: Z, X, abar, zbar, Teff4, g, cgrav, L, m, r, &
            tau, lnPgas, Pgas, T4, dT4_dtau, T, lnT, &
            Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, dPrad_dT4, dlnPgas_dlntau
         
         integer :: which_atm_option, handle, kap_handle, eos_handle, species, i
         integer, dimension(:), pointer :: chem_id, net_iso
         real(dp), dimension(:), pointer :: xa
         
         include 'formats.dek'
         
         ierr = 0       
         
         handle = ipar(i_atm_handle)
         eos_handle = ipar(i_eos_handle)
         kap_handle = ipar(i_kap_handle)
         which_atm_option = ipar(i_which_T_tau)
         species = ipar(i_species)
			i = atm_lipar
			chem_id => ipar(i+1:i+species); i = i+species
			net_iso => ipar(i+1:i+num_chem_isos); i = i+num_chem_isos
         
         Z = rpar(r_Z)
         X = rpar(r_X)
         abar = rpar(r_abar)
         zbar = rpar(r_zbar)
         Teff4 = rpar(r_Teff4)
         g = rpar(r_g)
         cgrav = rpar(r_cgrav)
         L = rpar(r_L)
         m = rpar(r_m)
         r = rpar(r_r)
         i = atm_lrpar
         xa => rpar(i+1:i+species); i = i+species

         tau = exp_cr(lntau)
         lnPgas = y(1)
         Pgas = exp_cr(lnPgas)

         T4 = T4_of_tau(which_atm_option, tau, lntau, Teff4, dT4_dtau)
         T = pow_cr(T4,0.25d0)
         lnT = log_cr(T)
         
         call get_atm_eos_kap_info( &
            handle, kap_handle, eos_handle, tau, cgrav, L, m, r, &
            Pgas, lnPgas, T, lnT, Z, 1-(X+Z), X, abar, zbar, &
            species, chem_id, net_iso, xa, &
            Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         if (ierr /= 0) return
         
         dPrad_dT4 = crad/3
         dlnPgas_dlntau = (tau/Pgas)*(g/kap -  dPrad_dT4*dT4_dtau)
         f(1) = dlnPgas_dlntau

         rpar(r_Rho) = Rho
         rpar(r_kap) = kap
         rpar(r_lnT) = lnT

      end subroutine atm_fcn


      subroutine atm_solout( &
            nr, xold, x, n, y, rwork_y, iwork_y, interp_y, lrpar, rpar, lipar, ipar, irtrn)
         ! nr is the step number.
         ! x is the current x value; xold is the previous x value.
         ! y is the current y value.
         ! irtrn negative means terminate integration.
         ! rwork_y and iwork_y hold info for interp_y
         integer, intent(in) :: nr, n, lrpar, lipar
         real(dp), intent(in) :: xold, x
         real(dp), intent(inout) :: y(:) ! (n)
         ! y can be modified if necessary to keep it in valid range of possible solutions.
         real(dp), intent(inout), target :: rwork_y(*)
         integer, intent(inout), target :: iwork_y(*)
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         interface
            ! this subroutine can be called from your solout routine.
            ! it computes interpolated values for y components during the just completed step.
            real(dp) function interp_y(i, s, rwork_y, iwork_y, ierr)
               use const_def, only: dp
               integer, intent(in) :: i ! result is interpolated approximation of y(i) at x=s.
               real(dp), intent(in) :: s ! interpolation x value (between xold and x).
               real(dp), intent(inout), target :: rwork_y(*)
               integer, intent(inout), target :: iwork_y(*)
               integer, intent(out) :: ierr
            end function interp_y
         end interface
         integer, intent(out) :: irtrn ! < 0 causes solver to return to calling program.

         type (Int_Atm_Info), pointer :: ai
         real(dp) :: lntau, f(n), h
         integer :: handle, ierr
         
         include 'formats.dek'
         
         irtrn = 0
         if (.false. .and. dbg) &
            write(*,2) 'solout: step, log tau, log tau_stop', nr, x/ln10, rpar(r_lntau_stop)/ln10
            
         handle = ipar(i_atm_handle)
			if (handle <= 0) return
			ierr = 0
         call get_atm_ptr(handle,ai,ierr)
         if (ierr /= 0) then
            write(*,*) 'get_atm_ptr failed in atm_solout'
            return
         end if
         
         ai% save_atm_structure_info = .true.
         h = x - xold
         call atm_fcn(n, x, h, y, f, lrpar, rpar, lipar, ipar, ierr)
         ai% save_atm_structure_info = .false.         
         
      end subroutine atm_solout
      
      
      real(dp) function T4_of_tau( &
            which_atm_option, tau, lntau, Teff4, dT4_dtau)
         integer, intent(in) :: which_atm_option
         real(dp), intent(in) :: tau, lntau, Teff4
         real(dp), intent(out) :: dT4_dtau
         real(dp) :: e1, e2, de1_dtau, de2_dtau, h1
         real(dp), parameter :: &
            q1 = 1.0361d0, &  
            q2 = -0.3134d0, &  
            q3 = 2.44799995d0, &  
            q4 = -0.29589999d0, &  
            q5 = 30.0d0

         select case (which_atm_option)
         
            case (atm_Eddington_grey)
            
               T4_of_tau = 0.75d0*Teff4*(tau + two_thirds)
               dT4_dtau = 0.75d0*Teff4
               
            case (atm_Krishna_Swamy)
            
               if (tau == 0) then
                  e1 = 1; de1_dtau = -2.54d0
                  e2 = 1; de2_dtau = -30d0
               else
                  e1 = exp_cr(-2.54d0*tau); de1_dtau = -2.54d0*e1
                  e2 = exp_cr(-30d0*tau); de2_dtau = -30d0*e2
               end if
               T4_of_tau = 0.75d0*Teff4*(tau + 1.39d0 - 0.815d0*e1 - 0.025d0*e2)
               dT4_dtau = 0.75d0*Teff4*(1 - 0.815d0*de1_dtau - 0.025d0*de2_dtau)

            case (atm_solar_Hopf_grey)
               
               if (tau == 0) then
                  e1 = 1; de1_dtau = -q3
                  e2 = 1; de2_dtau = -q5
               else
                  e1 = exp_cr(-q3*tau); de1_dtau = -q3*e1
                  e2 = exp_cr(-q5*tau); de2_dtau = -q5*e2
               end if
               T4_of_tau = 0.75d0*Teff4*(tau + q1 + q2*e1 + q4*e2)
               dT4_dtau = 0.75d0*Teff4*(1 + q2*de1_dtau + q4*de2_dtau)

            case default
               T4_of_tau = -1
               dT4_dtau = -1
               
         end select
      end function T4_of_tau


      logical function valid_T_tau(which_atm_option)
         integer, intent(in) :: which_atm_option
         select case (which_atm_option)
            case (atm_Eddington_grey)
               valid_T_tau = .true.
            case (atm_Krishna_Swamy)
               valid_T_tau = .true.
            case (atm_solar_Hopf_grey)
               valid_T_tau = .true.
            case default
               valid_T_tau = .false.
         end select
      end function valid_T_tau

      
      subroutine get_atm_eos_kap_info( &
            handle, kap_handle, eos_handle, tau, cgrav, L, m, r, &
            Pgas, lnPgas, T, lnT, Z, Y, X, abar, zbar, &
            species, chem_id, net_iso, xa, &
            Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         use eos_def
         use eos_lib
         use kap_def
         use kap_lib

         integer, intent(in) :: handle, kap_handle, eos_handle, species
         real(dp), intent(in) :: &
            tau, cgrav, L, m, r, Pgas, lnPgas, T, lnT, Z, Y, X, &
            abar, zbar
         integer, pointer :: chem_id(:), net_iso(:)
         real(dp), intent(in) :: xa(:)
         real(dp), intent(out) :: &
            Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            kap, dlnkap_dlnRho, dlnkap_dlnT
         integer, intent(out) :: ierr
      
         integer :: sz, k
         real(dp) :: lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, Prad, P, &
            gradr, gradT
         real(dp), dimension(num_eos_basic_results) :: &
            res, d_dlnRho_const_T, d_dlnT_const_Rho, &
            d_dabar_const_TRho, d_dzbar_const_TRho
         type (Int_Atm_Info), pointer :: ai
         
         include 'formats.dek'

         ierr = 0
        
         call eosPT_get( &
            eos_handle, Z, X, abar, zbar, &
            species, chem_id, net_iso, xa, &
            Pgas, lnPgas/ln10, T, lnT/ln10, &
            Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            res, d_dlnRho_const_T, d_dlnT_const_Rho, &
            d_dabar_const_TRho, d_dzbar_const_TRho, ierr)
         if (ierr /= 0) then
            !write(*,*) 'failed in eosPT_get'
            ierr=1
            return
         end if
         lnfree_e = res(i_lnfree_e)
         d_lnfree_e_dlnRho = d_dlnRho_const_T(i_lnfree_e)
         d_lnfree_e_dlnT = d_dlnT_const_Rho(i_lnfree_e)
         
         call kap_get_Type1( &
            kap_handle, zbar, X, Z, logRho, lnT/ln10, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in kap_get'
            return
         end if
         
         call finish_get_atm_eos_kap( &
            handle, ai, T, lnT, Pgas, lnPgas, rho, logRho, &
            kap, L, m, cgrav, tau, r, res, &
            lnfree_e, dlnkap_dlnT, dlnkap_dlnRho, ierr)
      
      end subroutine get_atm_eos_kap_info
      
      
      subroutine finish_get_atm_eos_kap( &
            handle, ai, T, lnT, Pgas, lnPgas, rho, logRho, &
            kap, L, m, cgrav, tau, r, res, &
            lnfree_e, dlnkap_dlnT, dlnkap_dlnRho, ierr)
      
         use utils_lib, only: realloc_double2
         use eos_def
         use eos_lib, only: Radiation_Pressure

         integer, intent(in) :: handle
         real(dp), intent(in) :: T, lnT, Pgas, lnPgas, rho, logRho, &
            kap, L, m, cgrav, tau, r, &
            lnfree_e, dlnkap_dlnT, dlnkap_dlnRho
         real(dp), dimension(num_eos_basic_results) :: res
         type (Int_Atm_Info), pointer :: ai
         integer, intent(out) :: ierr

         integer :: k, sz
         real(dp) :: Prad, P, gradr, gradT
         
         ierr = 0
         
			if (handle <= 0) return
         call get_atm_ptr(handle,ai,ierr)
         if (ierr /= 0) then
            write(*,*) 'get_atm_ptr failed in atm_fcn'
            return
         end if
		   if (.not. ai% save_atm_structure_info) return
         
         k = ai% atm_structure_num_pts + 1
         if (.not. associated(ai% atm_structure)) then
            sz = 100
            allocate(ai% atm_structure(num_results_for_create_atm,sz))
         else
            sz = size(ai% atm_structure,dim=2)
            if (k >= sz) then
               sz = 2*sz + 100
               call realloc_double2( &
                  ai% atm_structure,num_results_for_create_atm,sz,ierr)
            end if
         end if
         
         Prad = Radiation_Pressure(T)
         P = Pgas + Prad
         
         gradr = get1_Paczynski_gradr(P,kap,L,m,cgrav,Prad,tau,T,r,rho)
         gradT = gradr ! by assumption, atm is radiative

         ai% atm_structure_num_pts = k

         ai% atm_structure(atm_delta_r,k) = 0
         ai% atm_structure(atm_lnP,k) = log_cr(P)
         ai% atm_structure(atm_lnd,k) = logRho*ln10
         ai% atm_structure(atm_lnT,k) = lnT
         ai% atm_structure(atm_gradT,k) = gradT
         ai% atm_structure(atm_kap,k) = kap
         ai% atm_structure(atm_gamma1,k) = res(i_gamma1)
         ai% atm_structure(atm_grada,k) = res(i_grad_ad)
         ai% atm_structure(atm_chiT,k) = res(i_chiT)
         ai% atm_structure(atm_chiRho,k) = res(i_chiRho)
         ai% atm_structure(atm_cp,k) = res(i_Cp)
         ai% atm_structure(atm_cv,k) = res(i_Cv)
         ai% atm_structure(atm_tau,k) = tau
         ai% atm_structure(atm_lnfree_e,k) = lnfree_e
         ai% atm_structure(atm_dlnkap_dlnT,k) = dlnkap_dlnT
         ai% atm_structure(atm_dlnkap_dlnd,k) = dlnkap_dlnRho
         ai% atm_structure(atm_lnPgas,k) = lnPgas
         ai% atm_structure(atm_gradr,k) = gradr
      
      end subroutine finish_get_atm_eos_kap

      
      subroutine int_atm_init
         integer :: i
         if (int_atm_is_initialized) return
         do i=1,max_atm_handles
            atm_handles(i)% handle = i
            atm_handles(i)% in_use = .false.
            atm_handles(i)% atm_structure_num_pts = 0
            nullify(atm_handles(i)% atm_structure)
         end do
         int_atm_is_initialized = .true.
      end subroutine int_atm_init

      
      integer function do_alloc_atm(ierr)
         integer, intent(out) :: ierr
         integer :: i
         type (Int_Atm_Info), pointer :: ai
         ierr = 0
         do_alloc_atm = -1
!$omp critical (atm_handle)
         do i = 1, max_atm_handles
            if (.not. atm_handles(i)% in_use) then
               atm_handles(i)% in_use = .true.
               do_alloc_atm = i
               exit
            end if
         end do
!$omp end critical (atm_handle)
         if (do_alloc_atm == -1) then
            ierr = -1
            return
         end if
         if (atm_handles(do_alloc_atm)% handle /= do_alloc_atm) then
            ierr = -1
            return
         end if
         ai => atm_handles(do_alloc_atm)
      end function do_alloc_atm
      
      
      subroutine do_free_atm(handle)
         integer, intent(in) :: handle
         if (handle >= 1 .and. handle <= max_atm_handles) then
            atm_handles(handle)% in_use = .false.
         end if
      end subroutine do_free_atm
      

      subroutine get_atm_ptr(handle,ai,ierr)
         integer, intent(in) :: handle
         type (Int_Atm_Info), pointer :: ai
         integer, intent(out):: ierr         
         if (handle < 1 .or. handle > max_atm_handles) then
            ierr = -1
            return
         end if
         ai => atm_handles(handle)
         ierr = 0
      end subroutine get_atm_ptr

      
      real(dp) function get1_Paczynski_gradr( &
            P,opacity,L,m,cgrav,Pr,tau,T,r,rho)
         use crlibm_lib, only: pow_cr
         real(dp), intent(in) :: P,opacity,L,m,cgrav,Pr,tau,T,r,rho
         real(dp) :: dilution_factor, s, f
         get1_Paczynski_gradr = P*opacity*L / (16*pi*clight*m*cgrav*Pr)
         if (tau < 2d0/3d0) then ! B. Paczynski, 1969, Acta Astr., vol. 19, 1., eqn 14.
            s = (2*crad*T*T*T*sqrt(r))/(3*cgrav*m*rho)*pow_cr(L/(8*pi*boltz_sigma), 0.25d0) ! eqn 15
            f = 1 - 1.5d0*tau ! Paczynski, 1969, eqn 8
            dilution_factor = (1 + f*s*(4*pi*cgrav*clight*m)/(opacity*L))/(1 + f*s)
            get1_Paczynski_gradr = get1_Paczynski_gradr*dilution_factor
         end if
      end function get1_Paczynski_gradr



#ifdef offload
      !dir$ end options
#endif

      end module integrate_atm
      
      
      
      
