#if !defined( ESMF_ ) && !defined( MODEL_WRF )
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !MODULE: flexgrid_read_mod.F90
!
! !DESCRIPTION: Module FLEXGRID\_READ\_MOD contains subroutines for reading the 
!  metfield from disk (in netCDF format).
!\\
!\\
! !INTERFACE: 
!
MODULE FlexGrid_Read_Mod
!
! !USES:
!

  ! GEOS-Chem modules
  USE CMN_SIZE_MOD                        ! Size parameters
#if defined( BPCH_DIAG )
  USE CMN_DIAG_MOD                        ! Diagnostic arrays & counters
  USE DIAG_MOD,      ONLY : AD21          ! Array for ND21 diagnostic  
  USE DIAG_MOD,      ONLY : AD66          ! Array for ND66 diagnostic  
  USE DIAG_MOD,      ONLY : AD67          ! Array for ND67 diagnostic
#endif
  USE ERROR_MOD,     ONLY : ERROR_STOP    ! Stop w/ error message
  USE PhysConstants                       ! Physical constants
  USE TIME_MOD                            ! Date & time routines

  IMPLICIT NONE
  PRIVATE

# include "netcdf.inc"                    ! Include file for netCDF library
!
! !PRIVATE MEMBER FUNCTIONS:
!
  PRIVATE :: FlexGrid_Read_A3cld
  PRIVATE :: FlexGrid_Read_A3dyn
  PRIVATE :: FlexGrid_Read_A3mstC
  PRIVATE :: FlexGrid_Read_A3mstE
!
! !PUBLIC MEMBER FUNCTIONS:
! 
  PUBLIC  :: FlexGrid_Read_CN
  PUBLIC  :: FlexGrid_Read_A1
  PUBLIC  :: FlexGrid_Read_A3
  PUBLIC  :: FlexGrid_Read_I3_1
  PUBLIC  :: FlexGrid_Read_I3_2
  PUBLIC  :: Copy_I3_Fields
!
! !REMARKS:
!  Assumes that you have a netCDF library (either v3 or v4) installed on 
!  your system. 
!
! !REVISION HISTORY:
!  30 Jan 2012 - R. Yantosca - Initial version
!  03 Feb 2012 - R. Yantosca - Add Geos57_Read_A3 wrapper function
!  07 Feb 2012 - R. Yantosca - Now echo info after reading fields from disk
!  10 Feb 2012 - R. Yantosca - Add function Get_Resolution_String
!  05 Apr 2012 - R. Yantosca - Convert units for specific humidity properly
!  15 Nov 2012 - R. Yantosca - Now replace dao_mod.F arrays with State_Met
!  11 Apr 2013 - R. Yantosca - Now pass directory fields via Input_Opt
!  26 Sep 2013 - R. Yantosca - Renamed to geosfp_read_mod.F90
!  14 Jan 2014 - R. Yantosca - Remove "define GEOS572_FILES #ifdef blocks
!  14 Aug 2014 - R. Yantosca - Compute CLDTOPS field in GeosFp_Read_A3mstE
!  03 Dec 2015 - R. Yantosca - Add file ID's as module variables
!  03 Dec 2015 - R. Yantosca - Add CLEANUP_GEOSFP_READ to close any open 
!                              netCDF files left at the end of a simulation
!  02 Feb 2016 - E. Lundgren - Block of diagnostics with if defined BPCH
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!

CONTAINS
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: FlexGrid_Read_cn
!
! !DESCRIPTION: Routine to read variables and attributes from a NetCDF
!  met fields file containing constant (CN) data.  
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE FlexGrid_Read_CN( Input_Opt, State_Met )
!
! !USES:
!
    USE Input_Opt_Mod,      ONLY : OptInput
    USE State_Met_Mod,      ONLY : MetState
    USE Get_Met_Mod
!
! !INPUT PARAMETERS:
!
    TYPE(OptInput), INTENT(IN)    :: Input_Opt   ! Input Options object
!
! !INPUT/OUTPUT PARAMETERS:
!
    TYPE(MetState), INTENT(INOUT) :: State_Met   ! Meteorology State object
!
! !REMARKS:
!  This routine was automatically generated by the Perl script ncCodeRead, 
!  and was subsequently hand-edited for compatibility with GEOS-Chem.
!                                                                             .
!  Even though the netCDF file is self-describing, the GEOS-FP data, 
!  dimensions, and units are pre-specified according to the GMAO GEOS-FP
!  file specification.  Therefore we can "cheat" a little bit and not have
!  to read netCDF attributes to find what these values are.
!
! !REVISION HISTORY:
!  30 Jan 2012 - R. Yantosca - Initial version
!  07 Feb 2012 - R. Yantosca - Now echo info after reading fields from disk
!  10 Feb 2012 - R. Yantosca - Now get a string for the model resolution
!  09 Nov 2012 - M. Payer    - Copy all met fields to the State_Met derived type
!                              object
!  15 Nov 2012 - R. Yantosca - Now replace dao_mod.F arrays with State_Met
!  11 Apr 2013 - R. Yantosca - Now pass directory fields with Input_Opt
!  26 Sep 2013 - R. Yantosca - Renamed to GeosFp_Read_CN
!  06 Nov 2014 - R. Yantosca - Replace TRANSFER_2D with direct casts
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    CHARACTER(LEN=16)  :: stamp              ! Time and date stamp
    CHARACTER(LEN=255) :: v_name             ! netCDF variable name
                                
    ! Arrays                                 
    REAL*4             :: Q(IIPAR,JJPAR)     ! Temporary data arrray

    ! Read FRLAKE
    v_name = "FRLAKE"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%FRLAKE = Q

    ! Read FRLAND
    v_name = "FRLAND"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%FRLAND = Q

    ! Read FRLANDIC
    v_name = "FRLANDIC"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%FRLANDIC = Q
    
    ! Read FROCEAN
    v_name = "FROCEAN"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%FROCEAN = Q
    
    ! Read PHIS
    v_name = "PHIS"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%PHIS = Q

    ! Echo info
    stamp = TimeStamp_String( 20110101, 000000 )
    WRITE( 6, 10 ) stamp
 10 FORMAT( '     - Found all CN     met fields for ', a )

    !======================================================================
    ! Cleanup and quit
    !======================================================================

    ! Convert PHIS from [m2/s2] to [m]
    State_Met%PHIS = State_Met%PHIS / g0

#if defined( BPCH_DIAG )
    ! ND67 diagnostic 
    IF ( ND67 > 0 ) THEN
       AD67(:,:,15) = AD67(:,:,15) + State_Met%PHIS  ! Sfc geopotential [m]
    ENDIF
#endif

  END SUBROUTINE FlexGrid_Read_CN
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: FlexGrid_Read_a1
!
! !DESCRIPTION: Routine to read variables and attributes from a NetCDF
!  met fields file containing 1-hr time-averaged (A1) data.  
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE FlexGrid_Read_A1( YYYYMMDD, HHMMSS, Input_Opt, State_Met )
!
! !USES:
!
    USE Input_Opt_Mod,      ONLY : OptInput
    USE State_Met_Mod,      ONLY : MetState
    USE Get_Met_Mod
!
! !INPUT PARAMETERS:
! 
    INTEGER,        INTENT(IN)    :: YYYYMMDD   ! GMT date in YYYY/MM/DD format
    INTEGER,        INTENT(IN)    :: HHMMSS     ! GMT time in hh:mm:ss   format
    TYPE(OptInput), INTENT(IN)    :: Input_Opt  ! Input Options object
!
! !INPUT/OUTPUT PARAMETERS:
!
    TYPE(MetState), INTENT(INOUT) :: State_Met  ! Meteorology State object
!
! !REMARKS:
!  This routine was automatically generated by the Perl script ncCodeRead, 
!  and was subsequently hand-edited for compatibility with GEOS-Chem.
!                                                                             .
!  Even though the netCDF file is self-describing, the GEOS-FP data, 
!  dimensions, and units are pre-specified according to the GMAO GEOS-FP
!  file specification.  Therefore we can "cheat" a little bit and not have
!  to read netCDF attributes to find what these values are.
!                                                                             .
!  Special handling for surface precipitation fields:
!  ---------------------------------------------------------------------------
!  In GEOS-FP (and in MERRA), the PRECTOT etc. surface precipitation
!  met fields fields have units of [kg/m2/s].  In all other GEOS 
!  versions, PREACC and PRECON have units of [mm/day].  
!                                                                             .
!  Therefore, for backwards compatibility with existing code, apply 
!  the following unit conversion to the GEOS-5 PRECTOT and PRECCON 
!  fields:
!                                                                             .
!      kg  |    m3    | 86400 s | 1000 mm
!    ------+----------+---------+--------- = 86400 
!     m2 s |  1000 kg |  day    |   m
!               ^
!               |
!        1 / density of water 
!
! !REVISION HISTORY:
!  30 Jan 2012 - R. Yantosca - Initial version
!  07 Feb 2012 - R. Yantosca - Now echo info after reading fields from disk
!  10 Feb 2012 - R. Yantosca - Now get a string for the model resolution
!  09 Nov 2012 - M. Payer    - Copy all met fields to the State_Met derived type
!                              object
!  15 Nov 2012 - R. Yantosca - Now replace dao_mod.F arrays with State_Met
!  04 Jan 2013 - M. Payer    - Bug fix: Use State_Met%TSKIN for ND67 surface
!                              skin temperature diagnostic, not State_MET%TS
!  11 Apr 2013 - R. Yantosca - Now pass directory fields with Input_Opt
!  02 Dec 2013 - S. Philip   - Correction for GEOS-FP boundary layer height
!  04 Dec 2013 - R. Yantosca - Now comment out GEOS-FP BL height correction
!  06 Nov 2014 - R. Yantosca - Replace TRANSFER_2D with direct casts
!  23 Sep 2015 - E. Lundgren - Now assign SWGDN to State_Met SWGDN not RADSWG
!  03 Dec 2015 - R. Yantosca - Now open file only once per day
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    CHARACTER(LEN=16)  :: stamp              ! Time and date stamp
    CHARACTER(LEN=255) :: v_name             ! netCDF variable name 

    ! Saved scalars
    INTEGER, SAVE      :: lastDate = -1      ! Stores last YYYYMMDD value
    INTEGER, SAVE      :: lastTime = -1      ! Stores last hhmmss value
                
    ! Arrays                                 
    REAL*4             :: Q(IIPAR,JJPAR)     ! Temporary data arrray

    !======================================================================
    ! Skip if we have already read data for this date & time
    !======================================================================
    IF ( YYYYMMDD == lastDate .and. HHMMSS == lastTime ) THEN
       stamp = TimeStamp_String( YYYYMMDD, HHMMSS )
       WRITE( 6, 20 ) stamp
 20    FORMAT( '     - FLEXGRID A1 met fields for ', a,  &
               ' have been read already'                  ) 
       RETURN
    ENDIF

    !======================================================================
    ! Get met fields from HEMCO
    !======================================================================
    
    ! Read ALBEDO
    v_name = "ALBEDO"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%ALBD = Q

    ! Read CLDTOT
    v_name = "CLDTOT"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%CLDFRC = Q

    ! Read EFLUX
    v_name = "EFLUX"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%EFLUX = Q

    !--------------------------------------------------------------------------
    ! For now, skip reading EVAP. It's not used in GEOS-Chem. (mps, 9/14/17)
    !! Read EVAP
    !v_name = "EVAP"
    !CALL Get_Met_2D( Q, TRIM(v_name) )
    !State_Met%EVAP = Q
    !--------------------------------------------------------------------------

    ! Read FRSEAICE
    v_name = "FRSEAICE"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%FRSEAICE = Q

    ! Read FRSNO
    v_name = "FRSNO"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%FRSNO = Q

    !--------------------------------------------------------------------------
    ! For now, skip reading GRN. It's not used in GEOS-Chem. (mps, 9/14/17)
    !! Read GRN
    !v_name = "GRN"
    !CALL Get_Met_2D( Q, TRIM(v_name) )
    !State_Met%GRN = Q
    !--------------------------------------------------------------------------

    ! Read GWETROOT
    v_name = "GWETROOT"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%GWETROOT = Q

    ! Read GWETTOP
    v_name = "GWETTOP"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%GWETTOP = Q

    ! Read HFLUX from file
    v_name = "HFLUX"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%HFLUX = Q

    ! Read LAI
    v_name = "LAI"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%LAI = Q

    ! Read LWI
    v_name = "LWI"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%LWI = Q

    !--------------------------------------------------------------------------
    ! For now, skip reading RADLWG. It's not used in GEOS-Chem. (mps, 9/14/17)
    !! Read LWGNT 
    !v_name = "LWGNT"
    !CALL Get_Met_2D( Q, TRIM(v_name) )
    !State_Met%RADLWG = Q
    !--------------------------------------------------------------------------

    !-----------------------------------------------------------------------
    ! Comment this out for now, this field isn't needed (bmy, 2/2/12)
    !! Read LWTUP
    !v_name = "LWTUP"
    !CALL Get_Met_2D( Q, TRIM(v_name) )
    !State_Met%LWTUP = Q
    !-----------------------------------------------------------------------

    ! Read PARDF
    v_name = "PARDF"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%PARDF = Q

    ! Read PARDR
    v_name = "PARDR"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%PARDR = Q

    ! Read PBLH
    v_name = "PBLH"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%PBLH = Q

    ! Read PRECANV
    v_name = "PRECANV"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%PRECANV = Q

    ! Read PRECCON
    v_name = "PRECCON"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%PRECCON = Q

    ! Read PRECLSC
    v_name = "PRECLSC"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%PRECLSC = Q

    !--------------------------------------------------------------------------
    ! For now, skip reading PRECSNO. It's not used in GEOS-Chem. (mps, 9/14/17)
    !! Read PRECSNO
    !v_name = "PRECSNO"
    !CALL Get_Met_2D( Q, TRIM(v_name) )
    !State_Met%PRECSNO = Q
    !--------------------------------------------------------------------------

    ! Read PRECTOT
    v_name = "PRECTOT"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%PRECTOT = Q

    !-----------------------------------------------------------------------
    ! Comment this out for now, this field isn't needed (bmy, 2/2/12)
    !! Read QV2M
    !v_name = "QV2M"
    !CALL Get_Met_2D( Q, TRIM(v_name) )
    !State_Met%QV2M = Q
    !-----------------------------------------------------------------------

    ! Read SEAICE00
    v_name = "SEAICE00"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SEAICE00 = Q

    ! Read SEAICE10
    v_name = "SEAICE10"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SEAICE10 = Q

    ! Read SEAICE20
    v_name = "SEAICE20"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SEAICE20 = Q

    ! Read SEAICE30
    v_name = "SEAICE30"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SEAICE30 = Q

    ! Read SEAICE40
    v_name = "SEAICE40"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SEAICE40 = Q

    ! Read SEAICE50
    v_name = "SEAICE50"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SEAICE50 = Q

    ! Read SEAICE60 
    v_name = "SEAICE60"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SEAICE60 = Q

    ! Read SEAICE70
    v_name = "SEAICE70"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SEAICE70 = Q

    ! Read SEAICE80
    v_name = "SEAICE80"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SEAICE80 = Q

    ! Read SEAICE90
    v_name = "SEAICE90"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SEAICE90 = Q

    ! Read SLP
    v_name = "SLP"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SLP = Q

    ! Read SNODP
    v_name = "SNODP"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SNODP = Q

    ! Read SNOMAS
    v_name = "SNOMAS"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SNOMAS = Q

    ! Read SWGDN
    v_name = "SWGDN"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%SWGDN  = Q

    ! Read TO3
    v_name = "TO3"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%TO3 = Q

    ! Read TROPPT
    v_name = "TROPPT"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%TROPP = Q

    ! Read TS
    v_name = "TS"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%TSKIN = Q

    ! Read T2M
    v_name = "T2M"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%TS = Q

    ! Read U10M
    v_name = "U10M"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%U10M = Q

    ! Read USTAR
    v_name = "USTAR"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%USTAR = Q

    ! Read V10M
    v_name = "V10M"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met% V10M = Q

    ! Read Z0M
    v_name = "Z0M"
    CALL Get_Met_2D( Q, TRIM(v_name) )
    State_Met%Z0 = Q

    ! Echo info
    stamp = TimeStamp_String( YYYYMMDD, HHMMSS )
    WRITE( 6, 10 ) stamp                    
 10 FORMAT( '     - Found all A1     met fields for ', a )

    !======================================================================
    ! Diagnostics, cleanup, and quit
    !======================================================================

    ! Increment the # of times A1 fields are read from disk
    CALL Set_Ct_A1( INCREMENT=.TRUE. )
      
    ! Convert surface precip fields from [kg/m2/s] --> [mm/day]
    State_Met%PRECANV = State_Met%PRECANV * 86400d0
    State_Met%PRECCON = State_Met%PRECCON * 86400d0
    State_Met%PRECLSC = State_Met%PRECLSC * 86400d0
    State_Met%PRECTOT = State_Met%PRECTOT * 86400d0

#if defined( MERRA2 )
    ! Convert pressure quantities from [Pa] -> [hPa]
    State_Met%SLP     = State_Met%SLP     * 1e-2_fp
    State_Met%TROPP   = State_Met%TROPP   * 1e-2_fp
#endif

#if defined( BPCH_DIAG )
    ! ND67 diagnostic: surface fields
    IF ( ND67 > 0 ) THEN
       AD67(:,:,1 ) = AD67(:,:,1 ) + State_Met%HFLUX    ! Sens heat flux [W/m2]
       AD67(:,:,2 ) = AD67(:,:,2 ) + State_Met%SWGDN    ! SW rad @ sfc [W/m2]
       AD67(:,:,3 ) = AD67(:,:,3 ) + State_Met%PRECTOT  ! Tot prec [kg/m2/s]
       AD67(:,:,4 ) = AD67(:,:,4 ) + State_Met%PRECCON  ! Sfc conv prec[kg/m2/s]
       AD67(:,:,5 ) = AD67(:,:,5 ) + State_Met%TS       ! T @ 2m height [K]
       AD67(:,:,6 ) = AD67(:,:,6 ) + 0e0                !
       AD67(:,:,7 ) = AD67(:,:,7 ) + State_Met%USTAR    ! Friction vel [m/s]
       AD67(:,:,8 ) = AD67(:,:,8 ) + State_Met%Z0       ! Roughness height [m]
       AD67(:,:,9 ) = AD67(:,:,9 ) + State_Met%PBLH     ! PBL height [m]
       AD67(:,:,10) = AD67(:,:,10) + State_Met%CLDFRC   ! Column cld fraction
       AD67(:,:,11) = AD67(:,:,11) + State_Met%U10M     ! U-wind @ 10m [m/s]
       AD67(:,:,12) = AD67(:,:,12) + State_Met%V10M     ! V-wind @ 10m [m/s]
       AD67(:,:,14) = AD67(:,:,14) + State_Met%ALBD     ! Sfc albedo [unitless]
       AD67(:,:,17) = AD67(:,:,17) + State_Met%TROPP    ! T'pause pressure [hPa]
       AD67(:,:,18) = AD67(:,:,18) + State_Met%SLP      ! Sea level prs [hPa]
       AD67(:,:,19) = AD67(:,:,19) + State_Met%TSKIN    ! Sfc skin temp [K]
       AD67(:,:,20) = AD67(:,:,20) + State_Met%PARDF    ! Diffuse PAR [W/m2]
       AD67(:,:,21) = AD67(:,:,21) + State_Met%PARDR    ! Direct PAR [W/m2]
       AD67(:,:,22) = AD67(:,:,22) + State_Met%GWETTOP  ! Topsoil wetness [frac]
       AD67(:,:,23) = AD67(:,:,23) + State_Met%EFLUX    ! Latent heat flux [W/m2]
    ENDIF
#endif

    ! Save date & time for next iteration
    lastDate = YYYYMMDD
    lastTime = HHMMSS

  END SUBROUTINE FlexGrid_Read_A1
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: FlexGrid_Read_a3
!
! !DESCRIPTION: Convenience wrapper for the following routines which read
!  3-hour time averaged data from disk:
! \begin{itemize}
! \item FlexGrid\_Read\_A3cld
! \item FlexGrid\_Read\_A3dyn
! \item FlexGrid\_Read\_A3mstC
! \item FlexGrid\_Read\_A3mstE
! \end{itemize}
!
! !INTERFACE:
!
  SUBROUTINE FlexGrid_Read_A3( YYYYMMDD, HHMMSS, Input_Opt, State_Met )
!
! !USES:
!
    USE Input_Opt_Mod,      ONLY : OptInput
    USE State_Met_Mod,      ONLY : MetState
!
! !INPUT PARAMETERS:
! 
    INTEGER,        INTENT(IN)    :: YYYYMMDD   ! GMT date in YYYY/MM/DD format
    INTEGER,        INTENT(IN)    :: HHMMSS     ! GMT time in hh:mm:ss   format
    TYPE(OptInput), INTENT(IN)    :: Input_Opt  ! Input Options object
!
! !INPUT/OUTPUT PARAMETERS:
!
    TYPE(MetState), INTENT(INOUT) :: State_Met  ! Meteorology State object
!
! !REVISION HISTORY:
!  30 Jan 2012 - R. Yantosca - Initial version
!  11 Apr 2013 - R. Yantosca - Now pass directory fields with Input_Opt
!  26 Sep 2013 - R. Yantosca - Renamed to GeosFp_Read_A3
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    CHARACTER(LEN=16) :: stamp            ! Time and date stamp

    ! Saved scalars
    INTEGER, SAVE     :: lastDate = -1    ! Stores last YYYYMMDD value
    INTEGER, SAVE     :: lastTime = -1    ! Stores last hhmmss value

    !======================================================================
    ! Call individual routines for reading A3 data
    !======================================================================

    ! Test to see if we have already read this data in
    IF ( YYYYMMDD == lastDate .and. HHMMSS == lastTime ) THEN
       stamp = TimeStamp_String( YYYYMMDD, HHMMSS )
       WRITE( 6, 20 ) stamp
 20    FORMAT( '     - FLEXGRID A3 met fields for ', a,  &
               ' have been read already'                  ) 
       RETURN
    ENDIF

    ! Save date & time for next iteration
    lastDate = YYYYMMDD
    lastTime = HHMMSS

    ! Read all the diffeent A3 files
    CALL FlexGrid_Read_A3cld ( YYYYMMDD, HHMMSS, Input_Opt, State_Met )
    CALL FlexGrid_Read_A3dyn ( YYYYMMDD, HHMMSS, Input_Opt, State_Met )
    CALL FlexGrid_Read_A3mstC( YYYYMMDD, HHMMSS, Input_Opt, State_Met )
    CALL FlexGrid_Read_A3mstE( YYYYMMDD, HHMMSS, Input_Opt, State_Met )

    !======================================================================
    ! Cleanup and quit
    !======================================================================

    ! Increment the # of times that A3 fields have been read
    CALL Set_Ct_A3( INCREMENT=.TRUE. )

    ! Save date & time for next iteration
    lastDate = YYYYMMDD
    lastTime = HHMMSS

  END SUBROUTINE FlexGrid_Read_A3
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: FlexGrid_Read_a3cld
!
! !DESCRIPTION: Routine to read variables and attributes from a NetCDF
!  met fields file containing 3-hr time-averaged (A3) data (cloud fields).
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE FlexGrid_Read_A3cld( YYYYMMDD, HHMMSS, Input_Opt, State_Met )
!
! !USES:
!
    USE Input_Opt_Mod,      ONLY : OptInput
    USE State_Met_Mod,      ONLY : MetState
    USE Get_Met_Mod
!
! !INPUT PARAMETERS:
! 
    INTEGER,        INTENT(IN)    :: YYYYMMDD   ! GMT date in YYYY/MM/DD format
    INTEGER,        INTENT(IN)    :: HHMMSS     ! GMT time in hh:mm:ss   format
    TYPE(OptInput), INTENT(IN)    :: Input_Opt  ! Input Options object
!
! !INPUT/OUTPUT PARAMETERS:
!
    TYPE(MetState), INTENT(INOUT) :: State_Met  ! Meteorology State object
!
! !REMARKS:
!  This routine was automatically generated by the Perl script ncCodeRead, 
!  and was subsequently hand-edited for compatibility with GEOS-Chem.
!                                                                             .
!  Even though the netCDF file is self-describing, the GEOS-FP data, 
!  dimensions, and units are pre-specified according to the GMAO GEOS-FP
!  file specification.  Therefore we can "cheat" a little bit and not have
!  to read netCDF attributes to find what these values are.
!
! !REVISION HISTORY:
!  30 Jan 2012 - R. Yantosca - Initial version
!  07 Feb 2012 - R. Yantosca - Now echo info after reading fields from disk
!  10 Feb 2012 - R. Yantosca - Now get a string for the model resolution
!  05 Apr 2012 - R. Yantosca - Fixed bug: TAUCLI was overwritten w/ TAUCLW
!  09 Nov 2012 - M. Payer    - Copy all met fields to the State_Met derived type
!                              object
!  15 Nov 2012 - R. Yantosca - Now replace dao_mod.F arrays with State_Met
!  11 Apr 2013 - R. Yantosca - Now pass directory fields with Input_Opt
!  26 Sep 2013 - R. Yantosca - Renamed to GeosFp_Read_A3Cld
!  06 Nov 2014 - R. Yantosca - Replace TRANSFER_A6 with TRANSFER_3D
!  03 Dec 2015 - R. Yantosca - Now open file only once per day
!  17 Mar 2016 - M. Sulprizio- Read optical depth into State_Met%OPTD instead of
!                              State_Met%OPTDEP (obsolete).
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    CHARACTER(LEN=16)  :: stamp                    ! Time and date stamp
    CHARACTER(LEN=255) :: v_name                   ! netCDF variable name 
                                  
    ! Arrays                                 
    REAL*4             :: Q(IIPAR,JJPAR,LLPAR)     ! Temporary data arrray

    !======================================================================
    ! Get met fields from HEMCO
    !======================================================================
    
    ! Read CLOUD
    v_name = "CLOUD"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%CLDF = Q
    
    ! Read OPTDEPTH
    v_name = "OPTDEPTH"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%OPTD = Q

    ! Read QI
    v_name = "QI"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%QI = Q

    ! Read QL
    v_name = "QL"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%QL = Q

    ! Read TAUCLI
    v_name = "TAUCLI"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%TAUCLI = Q

    ! Read TAUCLW
    v_name = "TAUCLW"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%TAUCLW = Q

    ! Echo info
    stamp = TimeStamp_String( YYYYMMDD, HHMMSS )
    WRITE( 6, 10 ) stamp
 10 FORMAT( '     - Found all A3cld  met fields for ', a )

    !======================================================================
    ! Diagnostics, cleanup, and quit
    !======================================================================

#if defined( BPCH_DIAG )
    ! ND21 diagnostic: OPTD and CLDF
    IF ( ND21 > 0 ) THEN
       AD21(:,:,1:LD21,1) = AD21(:,:,1:LD21,1) + State_Met%OPTD(:,:,1:LD21)
       AD21(:,:,1:LD21,2) = AD21(:,:,1:LD21,2) + State_Met%CLDF(:,:,1:LD21)
    ENDIF
#endif

  END SUBROUTINE FlexGrid_Read_A3cld
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: FlexGrid_Read_a3dyn
!
! !DESCRIPTION: Routine to read variables and attributes from a NetCDF
!  met fields file containing 3-hr time-averaged (A3) data (dynamics fields).
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE FlexGrid_Read_A3dyn( YYYYMMDD, HHMMSS, Input_Opt, State_Met )
!
! !USES:
!
    USE Input_Opt_Mod,      ONLY : OptInput  
    USE State_Met_Mod,      ONLY : MetState
    USE Get_Met_Mod
!
! !INPUT PARAMETERS:
! 
    INTEGER,        INTENT(IN)    :: YYYYMMDD   ! GMT date in YYYY/MM/DD format
    INTEGER,        INTENT(IN)    :: HHMMSS     ! GMT time in hh:mm:ss   format
    TYPE(OptInput), INTENT(IN)    :: Input_Opt  ! Input Options object
!
! !INPUT/OUTPUT PARAMETERS:
!
    TYPE(MetState), INTENT(INOUT) :: State_Met  ! Meteorology State object
!
! !REMARKS:
!  This routine was automatically generated by the Perl script ncCodeRead, 
!  and was subsequently hand-edited for compatibility with GEOS-Chem.
!                                                                             .
!  Even though the netCDF file is self-describing, the GEOS-FP data, 
!  dimensions, and units are pre-specified according to the GMAO GEOS-FP
!  file specification.  Therefore we can "cheat" a little bit and not have
!  to read netCDF attributes to find what these values are.
!
! !REVISION HISTORY:
!  30 Jan 2012 - R. Yantosca - Initial version
!  07 Feb 2012 - R. Yantosca - Now echo info after reading fields from disk
!  10 Feb 2012 - R. Yantosca - Now get a string for the model resolution
!  09 Nov 2012 - M. Payer    - Copy all met fields to the State_Met derived type
!                              object
!  15 Nov 2012 - R. Yantosca - Now replace dao_mod.F arrays with State_Met
!  11 Apr 2013 - R. Yantosca - Now pass directories with Input_Opt
!  26 Sep 2013 - R. Yantosca - Renamed to GeosFp_Read_A3dyn
!  15 Nov 2013 - R. Yantosca - Now convert RH from [1] to [%], in order
!                              to be consistent with GEOS-Chem convention
!  14 Aug 2014 - R. Yantosca - Now compute CLDTOPS in GeosFP_Read_A3mstE
!  03 Dec 2015 - R. Yantosca - Now open file only once per day
!  03 Feb 2017 - M. Sulprizio- Activate reading OMEGA fields from file
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    CHARACTER(LEN=16)  :: stamp                    ! Time and date stamp
    CHARACTER(LEN=255) :: v_name                   ! netCDF variable name 

    ! Arrays                                 
    REAL*4             :: Q (IIPAR,JJPAR,LLPAR  )  ! Temporary data arrray

    !======================================================================
    ! Get met fields from HEMCO
    !======================================================================

    ! Read DTRAIN
    v_name = "DTRAIN"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%DTRAIN = Q

    ! Read OMEGA
    v_name = "OMEGA"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%OMEGA = Q

    ! Read RH
    v_name = "RH"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%RH = Q

    ! Read U
    v_name = "U"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%U = Q

    ! Read V
    v_name = "V"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%V = Q

    ! Echo info
    stamp = TimeStamp_String( YYYYMMDD, HHMMSS )
    WRITE( 6, 10 ) stamp
 10 FORMAT( '     - Found all A3dyn  met fields for ', a )

    !======================================================================
    ! Unit conversions, diagnostics, cleanup, and quit
    !======================================================================
    
    ! Convert RH from [1] to [%]
    State_Met%RH = State_Met%RH * 100d0

#if defined( BPCH_DIAG )
    ! ND66 diagnostic: U, V, DTRAIN, OMEGA met fields
    IF ( ND66 > 0 ) THEN
       AD66(:,:,1:LD66,1) = AD66(:,:,1:LD66,1) + State_Met%U     (:,:,1:LD66)
       AD66(:,:,1:LD66,2) = AD66(:,:,1:LD66,2) + State_Met%V     (:,:,1:LD66)
       AD66(:,:,1:LD66,6) = AD66(:,:,1:LD66,6) + State_Met%DTRAIN(:,:,1:LD66)
       AD66(:,:,1:LD66,7) = AD66(:,:,1:LD66,7) + State_Met%OMEGA (:,:,1:LD66)
    ENDIF

    ! ND67 diagnostic: CLDTOPS
    IF ( ND67 > 0 ) THEN
       AD67(:,:,16) = AD67(:,:,16) + State_Met%CLDTOPS         ! [levels]
    ENDIF
#endif

  END SUBROUTINE FlexGrid_Read_A3dyn
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: FlexGrid_Read_a3mstc
!
! !DESCRIPTION: Routine to read variables and attributes from a NetCDF
!  met fields file containing 3-hr time-averaged (A3) data (moist fields,
!  saved on level centers).
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE FlexGrid_Read_A3mstC( YYYYMMDD, HHMMSS, Input_Opt, State_Met )
!
! !USES:
!
    USE Input_Opt_Mod,      ONLY : OptInput
    USE State_Met_Mod,      ONLY : MetState
    USE Get_Met_Mod
!
! !INPUT PARAMETERS:
! 
    INTEGER,        INTENT(IN)    :: YYYYMMDD   ! GMT date in YYYY/MM/DD format
    INTEGER,        INTENT(IN)    :: HHMMSS     ! GMT time in hh:mm:ss   format
    TYPE(OptInput), INTENT(IN)    :: Input_Opt  ! Input Options object
!
! !INPUT/OUTPUT PARAMETERS:
!
    TYPE(MetState), INTENT(INOUT) :: State_Met  ! Meteorology State object
!
! !REMARKS:
!  This routine was automatically generated by the Perl script ncCodeRead, 
!  and was subsequently hand-edited for compatibility with GEOS-Chem.
!                                                                             .
!  Even though the netCDF file is self-describing, the GEOS-FP data, 
!  dimensions, and units are pre-specified according to the GMAO GEOS-FP
!  file specification.  Therefore we can "cheat" a little bit and not have
!  to read netCDF attributes to find what these values are.
!
! !REVISION HISTORY:
!  30 Jan 2012 - R. Yantosca - Initial version
!  07 Feb 2012 - R. Yantosca - Now echo info after reading fields from disk
!  10 Feb 2012 - R. Yantosca - Now get a string for the model resolution
!  09 Nov 2012 - M. Payer    - Copy all met fields to the State_Met derived type
!                              object
!  15 Nov 2012 - R. Yantosca - Now replace dao_mod.F arrays with State_Met
!  11 Apr 2013 - R. Yantosca - Now pass directory fields with Input_Opt
!  26 Sep 2013 - R. Yantosca - Renamed to GeosFp_Read_A3mstC
!  03 Dec 2015 - R. Yantosca - Now open file only once per day
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    CHARACTER(LEN=16)  :: stamp                    ! Time and date stamp
    CHARACTER(LEN=255) :: v_name                   ! netCDF variable name 
                                    
    ! Arrays                                 
    REAL*4             :: Q (IIPAR,JJPAR,LLPAR)    ! Temporary data arrray

    !======================================================================
    ! Get met fields from HEMCO
    !======================================================================
    
    ! Read DQRCU  from file
    v_name = "DQRCU"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%DQRCU = Q

    ! Read DQRLSAN
    v_name = "DQRLSAN"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%DQRLSAN = Q

    ! Read REEVAPCN
    v_name = "REEVAPCN"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%REEVAPCN = Q

    ! Read  from file
    v_name = "REEVAPLS"
    CALL Get_Met_3D( Q, TRIM(v_name) )
    State_Met%REEVAPLS = Q

    ! Echo info
    stamp = TimeStamp_String( YYYYMMDD, HHMMSS )
    WRITE( 6, 10 ) stamp
 10 FORMAT( '     - Found all A3mstC met fields for ', a )

    !======================================================================
    ! Cleanup and quit
    !======================================================================

  END SUBROUTINE FlexGrid_Read_A3mstC
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: FlexGrid_Read_a3mste
!
! !DESCRIPTION: Routine to read variables and attributes from a NetCDF
!  met fields file containing 3-hr time-averaged (A3) data (moist fields,
!  saved on level edges).
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE FlexGrid_Read_A3mstE( YYYYMMDD, HHMMSS, Input_Opt, State_Met )
!
! !USES:
!
    USE Input_Opt_Mod,      ONLY : OptInput
    USE State_Met_Mod,      ONLY : MetState
    USE Get_Met_Mod
!
! !INPUT PARAMETERS:
! 
    INTEGER,        INTENT(IN)    :: YYYYMMDD   ! GMT date in YYYY/MM/DD format
    INTEGER,        INTENT(IN)    :: HHMMSS     ! GMT time in hh:mm:ss   format
    TYPE(OptInput), INTENT(IN)    :: Input_Opt  ! Input Options object
!
! !INPUT/OUTPUT PARAMETERS:
!
    TYPE(MetState), INTENT(INOUT) :: State_Met  ! Meteorology State object
!
! !REMARKS:
!  This routine was automatically generated by the Perl script ncCodeRead, 
!  and was subsequently hand-edited for compatibility with GEOS-Chem.
!                                                                             .
!  Even though the netCDF file is self-describing, the GEOS-FP data, 
!  dimensions, and units are pre-specified according to the GMAO GEOS-FP
!  file specification.  Therefore we can "cheat" a little bit and not have
!  to read netCDF attributes to find what these values are.
!
! !REVISION HISTORY:
!  30 Jan 2012 - R. Yantosca - Initial version
!  07 Feb 2012 - R. Yantosca - Now echo info after reading fields from disk
!  10 Feb 2012 - R. Yantosca - Now get a string for the model resolution
!  09 Nov 2012 - M. Payer    - Copy all met fields to the State_Met derived type
!                              object
!  15 Nov 2012 - R. Yantosca - Now replace dao_mod.F arrays with State_Met
!  11 Apr 2013 - R. Yantosca - Now pass directory fields with Input_Opt
!  26 Sep 2013 - R. Yantosca - Renamed to GeosFp_Read_A3mstE
!  26 Sep 2013 - R. Yantosca - Now read CMFMC from GEOSFP*.nc files
!  14 Aug 2014 - R. Yantosca - Now compute CLDTOPS here; it depends on CMFMC
!  03 Dec 2015 - R. Yantosca - Now open file only once per day
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    INTEGER            :: I, J, L                  ! Loop indices
    CHARACTER(LEN=16)  :: stamp                    ! Time and date stamp
    CHARACTER(LEN=255) :: v_name                   ! netCDF variable name 
                                             
    ! Arrays                                 
    REAL*4             :: Qe(IIPAR,JJPAR,LLPAR+1)  ! Temporary data arrray

    !======================================================================
    ! Get met fields from HEMCO
    !======================================================================
    
    ! Read CMFMC (only in GEOSFP*.nc files)
    v_name = "CMFMC"
    CALL Get_Met_3De( Qe, TRIM(v_name) )
    State_Met%CMFMC = Qe

    ! Read PFICU
    v_name = "PFICU"
    CALL Get_Met_3De( Qe, TRIM(v_name) )
    State_Met%PFICU = Qe

    ! Read PFILSAN
    v_name = "PFILSAN"
    CALL Get_Met_3De( Qe, TRIM(v_name) )
    State_Met%PFILSAN = Qe

    ! Read PFLCU
    v_name = "PFLCU"
    CALL Get_Met_3De( Qe, TRIM(v_name) )
    State_Met%PFLCU = Qe

    ! Read  from file
    v_name = "PFLLSAN"
    CALL Get_Met_3De( Qe, TRIM(v_name) )
    State_Met%PFLLSAN = Qe

    ! Echo info
    stamp = TimeStamp_String( YYYYMMDD, HHMMSS )
    WRITE( 6, 10 ) stamp
 10 FORMAT( '     - Found all A3mstE met fields for ', a )

    !=================================================================
    ! Diagnostics, cleanup and quit
    !=================================================================

    ! CLDTOPS = highest location of CMFMC in the column (I,J)
    DO J = 1, JJPAR
    DO I = 1, IIPAR
       State_Met%CLDTOPS(I,J) = 1
       DO L = LLPAR, 1, -1
          IF ( State_Met%CMFMC(I,J,L) > 0d0 ) THEN
             State_Met%CLDTOPS(I,J) = L + 1
             EXIT
          ENDIF
       ENDDO
    ENDDO
    ENDDO

#if defined( BPCH_DIAG )
    ! ND66 diagnostic: CMFMC met field
    IF ( ND66 > 0 ) THEN
       AD66(:,:,1:LD66,5) = AD66(:,:,1:LD66,5) + State_Met%CMFMC(:,:,1:LD66)
    ENDIF
#endif

  END SUBROUTINE FlexGrid_Read_A3mstE
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: FlexGrid_Read_I3_1
!
! !DESCRIPTION: Routine to read variables and attributes from a NetCDF
!  met fields file containing 3-hr instantaneous (I3) data.
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE FlexGrid_Read_I3_1( YYYYMMDD, HHMMSS, Input_Opt, State_Met )
!
! !USES:
!
    USE Input_Opt_Mod,      ONLY : OptInput
    USE State_Met_Mod,      ONLY : MetState
    USE Get_Met_Mod
!
! !INPUT PARAMETERS:
! 
    INTEGER,        INTENT(IN)    :: YYYYMMDD   ! GMT date in YYYY/MM/DD format
    INTEGER,        INTENT(IN)    :: HHMMSS     ! GMT time in hh:mm:ss   format
    TYPE(OptInput), INTENT(IN)    :: Input_Opt  ! Input Options object
!
! !INPUT/OUTPUT PARAMETERS:
!
    TYPE(MetState), INTENT(INOUT) :: State_Met  ! Meteorology State object
!
! !REMARKS:
!  This routine was automatically generated by the Perl script ncCodeRead, 
!  and was subsequently hand-edited for compatibility with GEOS-Chem.
!                                                                             .
!  Even though the netCDF file is self-describing, the GEOS-FP data, 
!  dimensions, and units are pre-specified according to the GMAO GEOS-FP
!  file specification.  Therefore we can "cheat" a little bit and not have
!  to read netCDF attributes to find what these values are.
!
! !REVISION HISTORY:
!  30 Jan 2012 - R. Yantosca - Initial version
!  07 Feb 2012 - R. Yantosca - Now echo info after reading fields from disk
!  10 Feb 2012 - R. Yantosca - Now get a string for the model resolution
!  05 Apr 2012 - R. Yantosca - Now convert QV1 from [kg/kg] to [g/kg]
!  09 Nov 2012 - M. Payer    - Copy all met fields to the State_Met derived type
!                              object
!  15 Nov 2012 - R. Yantosca - Now replace dao_mod.F arrays with State_Met
!  11 Apr 2013 - R. Yantosca - Now pass directory fields with Input_Opt
!  06 Sep 2013 - R. Yantosca - Bug fix: we need to initialize State_Met%T
!                              with State_Met%TMPU1 to avoid errors.  The
!                              State_Met%T field will be set again in INTERP.
!  26 Sep 2013 - R. Yantosca - Renamed to GeosFp_Read_I3_1
!  29 Oct 2013 - R. Yantosca - Now read T_FULLGRID_1 for offline simulations
!  06 Nov 2014 - R. Yantosca - Replace TRANSFER_2D with direct casts
!  16 Apr 2015 - R. Yantosca - Remove reference to T_FULLGRID; it's obsolete
!  12 Jun 2015 - E. Lundgren - Initialize State_MET%SPHU with State_Met%SPHU1
!  03 Dec 2015 - R. Yantosca - Now open file only once per day
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    CHARACTER(LEN=16)  :: stamp                    ! Time and date stamp
    CHARACTER(LEN=255) :: v_name                   ! netCDF variable name 
                                    
    ! Arrays                                 
    REAL*4             :: Q2(IIPAR,JJPAR      )    ! 2D temporary data arrray
    REAL*4             :: Q3(IIPAR,JJPAR,LLPAR)    ! 3D temporary data arrray

    !======================================================================
    ! Get met fields from HEMCO
    !======================================================================

    !-------------------------------------------------
    ! Read 2D data
    !-------------------------------------------------

    ! Read PS
    v_name = "PS1"
    CALL Get_Met_2D( Q2, TRIM(v_name) )
    State_Met%PS1_WET = Q2

    !-------------------------------------------------
    ! Read 3D data
    !-------------------------------------------------

    !----------------------------------------------------------------
    ! Prior to 2/3/12:
    ! For now, skip reading Potential Vorticity (bmy, 2/3/12)
    !! Read PV
    !v_name = "PV"
    !CALL Get_Met_3D( Q3, TRIM(v_name) )
    !!Q3 = ABS(1.0e6*Q3) ! PV to PVU
    !State_Met%PV = Q3
    !----------------------------------------------------------------

    ! Read QV
    v_name = "SPHU1"
    CALL Get_Met_3D( Q3, TRIM(v_name) )
    State_Met%SPHU1 = Q3

    ! Read T
    v_name = "TMPU1"
    CALL Get_Met_3D( Q3, TRIM(v_name) )
    State_Met%TMPU1 = Q3

    ! Echo info
    stamp = TimeStamp_String( YYYYMMDD, HHMMSS )
    WRITE( 6, 10 ) stamp
 10 FORMAT( '     - Found all I3     met fields for ', a )

    !-------------------------------------------------
    ! Unit conversions & special handling
    !-------------------------------------------------
    WHERE ( State_Met%SPHU1 < 0d0 ) 

       ! NOTE: Now set negative Q to a small positive # 
       ! instead of zero, so as not to blow up logarithms
       State_Met%SPHU1 = 1d-32

    ELSEWHERE

       ! Convert GEOS-FP specific humidity from [kg/kg] to [g/kg]
       State_Met%SPHU1 = State_Met%SPHU1 * 1000d0

    ENDWHERE

#if defined( MERRA2 )
    ! Convert PS1_WET from [Pa] to [hPa]
    State_Met%PS1_WET = State_Met%PS1_WET * 1e-2_fp
#endif

    ! Initialize State_Met%T to State_Met%TMPU1 and State_Met%SPHU to
    ! State_Met%SPHU1.  After all future MET field reads (geosfp_read_i3_2)
    ! we will interpolate State_Met%T from the values of State_Met vars 
    ! TMPU1 and TMPU2 and State_Met%SPHU from the values of State_Met vars 
    ! SPHU1 and SPHU2. 
    State_Met%T         = State_Met%TMPU1
    State_Met%SPHU      = State_Met%SPHU1
    
    !======================================================================
    ! Diagnostics, cleanup, and quit
    !======================================================================

    ! Increment the # of times I3 fields have been read
    CALL Set_Ct_I3( INCREMENT=.TRUE. )

#if defined( BPCH_DIAG )
    ! ND66 diagnostic: T1, QV1 met fields
    IF ( ND66 > 0 ) THEN
       AD66(:,:,1:LD66,3) = AD66(:,:,1:LD66,3) + State_Met%TMPU1(:,:,1:LD66)
       AD66(:,:,1:LD66,4) = AD66(:,:,1:LD66,4) + State_Met%SPHU1(:,:,1:LD66)
    ENDIF
#endif

  END SUBROUTINE FlexGrid_Read_I3_1
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: FlexGrid_Read_I3_2
!
! !DESCRIPTION: Routine to read variables and attributes from a NetCDF
!  met fields file containing 3-hr instantaneous (I3) data.
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE FlexGrid_Read_I3_2( YYYYMMDD, HHMMSS, Input_Opt, State_Met )
!
! !USES:
!
    USE Input_Opt_Mod,      ONLY : OptInput
    USE State_Met_Mod,      ONLY : MetState
    USE Get_Met_Mod
!
! !INPUT PARAMETERS:
! 
    INTEGER,        INTENT(IN)    :: YYYYMMDD   ! GMT date in YYYY/MM/DD format
    INTEGER,        INTENT(IN)    :: HHMMSS     ! GMT time in hh:mm:ss   format
    TYPE(OptInput), INTENT(IN)    :: Input_Opt  ! Input Options object
!
! !INPUT/OUTPUT PARAMETERS:
!
    TYPE(MetState), INTENT(INOUT) :: State_Met  ! Meteorology State object
!
! !REMARKS:
!  This routine was automatically generated by the Perl script ncCodeRead, 
!  and was subsequently hand-edited for compatibility with GEOS-Chem.
!                                                                             .
!  Even though the netCDF file is self-describing, the GEOS-FP data, 
!  dimensions, and units are pre-specified according to the GMAO GEOS-FP
!  file specification.  Therefore we can "cheat" a little bit and not have
!  to read netCDF attributes to find what these values are.
!
! !REVISION HISTORY:
!  30 Jan 2012 - R. Yantosca - Initial version
!  07 Feb 2012 - R. Yantosca - Now echo info after reading fields from disk
!  10 Feb 2012 - R. Yantosca - Now get a string for the model resolution
!  05 Apr 2012 - R. Yantosca - Now convert QV2 from [kg/kg] to [g/kg]
!  09 Nov 2012 - M. Payer    - Copy all met fields to the State_Met derived type
!                              object
!  15 Nov 2012 - R. Yantosca - Now replace dao_mod.F arrays with State_Met
!  11 Apr 2013 - R. Yantosca - Now pass directory fields with Input_Opt
!  26 Sep 2013 - R. Yantosca - Rename to GeosFp_Read_I3_2
!  29 Oct 2013 - R. Yantosca - Now read T_FULLGRID_2 for offline simulations
!  06 Nov 2014 - R. Yantosca - Replace TRANSFER_2D with direct casts
!  03 Dec 2015 - R. Yantosca - Now open file only once per day
!  20 Sep 2016 - R. Yantosca - Bug fix: FIRST must be declared as LOGICAL
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    INTEGER            :: I, J, L                  ! Loop indices
    INTEGER            :: X, Y, Z, T               ! netCDF file dimensions
    CHARACTER(LEN=16)  :: stamp                    ! Time and date stamp
    CHARACTER(LEN=255) :: v_name                   ! netCDF variable name 
                                    
    ! Arrays                                 
    REAL*4             :: Q2(IIPAR,JJPAR      )    ! 2D temporary data arrray
    REAL*4             :: Q3(IIPAR,JJPAR,LLPAR)    ! 3D temporary data arrray

    !======================================================================
    ! Get met fields from HEMCO
    !======================================================================

    !-------------------------------------------------
    ! Read 2D data
    !-------------------------------------------------

    ! Read PS
    v_name = "PS2"
    CALL Get_Met_2D( Q2, TRIM(v_name) )
    State_Met%PS2_WET = Q2

    !-------------------------------------------------
    ! Read 3D data
    !-------------------------------------------------

    !----------------------------------------------------------------
    ! Prior to 2/3/12:
    ! For now, skip reading Potential Vorticity (bmy, 2/3/12)
    !! Read PV
    !v_name = "PV"
    !CALL Get_Met_3D( Q3, TRIM(v_name) )
    !!Q3 = ABS(1.0e6*Q3) ! PV to PVU
    !State_Met%PV = Q3
    !----------------------------------------------------------------

    ! Read QV
    v_name = "SPHU2"
    CALL Get_Met_3D( Q3, TRIM(v_name) )
    State_Met%SPHU2 = Q3

    ! Read T
    v_name = "TMPU2"
    CALL Get_Met_3D( Q3, TRIM(v_name) )
    State_Met%TMPU2 = Q3

    ! Echo info
    stamp = TimeStamp_String( YYYYMMDD, HHMMSS )
    WRITE( 6, 10 ) stamp
 10 FORMAT( '     - Found all I3     met fields for ', a )

    !-------------------------------------------------
    ! Unit conversions & special handling
    !-------------------------------------------------
    WHERE ( State_Met%SPHU2 < 0d0 ) 

       ! NOTE: Now set negative Q to a small positive # 
       ! instead of zero, so as not to blow up logarithms
       State_Met%SPHU2 = 1d-32

    ELSEWHERE

       ! Convert specific humidity from [kg/kg] to [g/kg]
       State_Met%SPHU2 = State_Met%SPHU2 * 1000d0

    ENDWHERE

#if defined( MERRA2 )
    ! Convert PS2_WET from [Pa] to [hPa]
    State_Met%PS2_WET = State_Met%PS2_WET * 1e-2_fp
#endif
    
    !======================================================================
    ! Diagnostics, cleanup, and quit
    !======================================================================

    ! Increment the # of times I3 fields have been read
    CALL Set_Ct_I3( INCREMENT=.TRUE. )

#if defined( BPCH_DIAG )
    ! ND66 diagnostic: T2, QV2 met fields
    IF ( ND66 > 0 ) THEN
       AD66(:,:,1:LD66,3) = AD66(:,:,1:LD66,3) + State_Met%TMPU2(:,:,1:LD66)
       AD66(:,:,1:LD66,4) = AD66(:,:,1:LD66,4) + State_Met%SPHU2(:,:,1:LD66)
    ENDIF
#endif

  END SUBROUTINE FlexGrid_Read_I3_2
!EOC  
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: copy_i3_fields
!
! !DESCRIPTION: Subroutine COPY\_I3\_FIELDS copies the I-3 fields at the 
!  end of a 3-hr timestep.  The I-3 fields at the end of a given 3-hr timestep 
!  become the fields at the beginning of the next 3-hr timestep. 
!\\
!\\
! !INTERFACE:
!
      SUBROUTINE COPY_I3_FIELDS( State_Met )
!
! !USES:
!
      USE State_Met_Mod,        ONLY : MetState
!
! !INPUT PARAMETERS:
!
      TYPE(MetState), INTENT(INOUT) :: State_Met   ! Meteorology State object
! 
! !REVISION HISTORY: 
!  13 Apr 2004 - R. Yantosca - Initial version
!  (1 ) Added parallel DO-loops (bmy, 4/13/04)
!  (2 ) Remove support for GEOS-1 and GEOS-STRAT met fields (bmy, 8/4/06)
!  (3 ) Added TROPP (phs 11/10/06)
!  (4 ) Don't copy TROPP2 to TROPP1 for GEOS-5 (bmy, 1/17/07) 
!  16 Aug 2010 - R. Yantosca - Added ProTeX headers
!  20 Aug 2010 - R. Yantosca - Rewrite #if block for clarity
!  20 Aug 2010 - R. Yantosca - Added #if block for MERRA met fields
!  06 Feb 2012 - R. Yantosca - Added #if block for GEOS-5.7.x met fields
!  07 Feb 2012 - R. Yantosca - Renamed to COPY_I3_I6_FIELDS
!  28 Feb 2012 - R. Yantosca - Removed support for GEOS-3
!  09 Nov 2012 - M. Payer    - Replaced all met field arrays with State_Met
!                              derived type object
!  26 Sep 2013 - R. Yantosca - Renamed GEOS_57 Cpp switch to GEOS_FP
!  11 Aug 2015 - R. Yantosca - MERRA2 behaves in the same way as GEOS-FP
!  03 May 2016 - E. Lundgren - Add PS1_DRY update
!  26 Oct 2018 - M. Sulprizio- Moved this routine from dao_mod.F to
!                              flexgrid_read_mod.F90
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
      INTEGER :: I, J, L

      !=================================================================
      ! COPY_I3_FIELDS begins here!
      !=================================================================

      State_Met%PS1_WET = State_Met%PS2_WET ! I3 surface pressure    [hPa]
      State_Met%PS1_DRY = State_Met%PS2_DRY ! I3 surface pressure    [hPa] 
      State_Met%SPHU1   = State_Met%SPHU2   ! I3 specific humidity   [g/kg]
      State_Met%TMPU1   = State_Met%TMPU2   ! I3 temperature         [K]

      END SUBROUTINE COPY_I3_FIELDS
!EOC
END MODULE FlexGrid_Read_Mod
#endif
