! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 


      module run_star_extras

      use star_lib
      use star_def
      use const_def
      
      implicit none
      
      integer :: time0, time1, clock_rate
      real(dp), parameter :: expected_runtime = 1 ! minutes
      integer :: numTacc,j                          !track location of accretion times read

      

      contains
      
      subroutine extras_controls(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
	 integer :: i
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
	

         s% extras_startup => extras_startup
         s% extras_check_model => extras_check_model
         s% extras_finish_step => extras_finish_step
         s% extras_after_evolve => extras_after_evolve
         s% how_many_extra_history_columns => how_many_extra_history_columns
         s% data_for_extra_history_columns => data_for_extra_history_columns
         s% how_many_extra_profile_columns => how_many_extra_profile_columns
         s% data_for_extra_profile_columns => data_for_extra_profile_columns  


           s% max_timestep = 10**(s% x_ctrl(7))*secyer
           s% dt_next = 10**(s% x_ctrl(7))*secyer   !set as thermal timescale  s% cgrav(1)*s% mstar/s% r(1)*s% L(1)]


	 s% other_energy => energy_routine
         s% other_adjust_mdot=> mass_loss
        ! s% other_atm => planetary_atm
      end subroutine extras_controls


      subroutine energy_routine(id, ierr)
         type (star_info), pointer :: s
         !use const_def, only: Rsun
         integer, intent(in) :: id
         integer, intent(out) :: ierr
	 
	 !use star_lib, only: star_ptr
	 double precision :: extraheat,junk,diag_heat
         integer :: k,i,n,counter,z,p,numOfAcc,zeta,jafter,jbefore,indexI
	 double precision :: core_epsrad,Rpl,pressureDep,pressureHere,random_dp
	 real(dp) :: tauHere,Vesc, KE,massTot,Cd,phi,Mpl,Rhopl,H,g,mH,Tacc
	 !real(dp), DIMENSION(30000000) :: readR,readM,readT
         !REAL(dp), DIMENSION(:), ALLOCATABLE :: readR,readM,readT
         real(dp), DIMENSION(700) :: arrayKE
         real(dp), DIMENSION(5000) :: arrayI


         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         ierr = 0


	 ! INITALIZE
         s% extra_heat = 0.d0
         s% d_extra_heat_dlnd(:) = 0d0
         s% d_extra_heat_dlnT(:) = 0d0

         ! EXTRA HEATING CONSTANT IN SPACE AND TIME
         ! Heat due to Thermal inertia of the core
         ! x_ctrl(3) = cv in erg/K/g from inlist

         k = s% nz
         extraheat = -s% x_ctrl(3) * s% M_center * s% T(s% nz) * s% dlnT_dt(s% nz) / s% dm(s% nz) ! erg/g/sec
           !assuming dlnT_dt is in s^-1

         ! EXTRA HEATING CONSTANT IN SPACE AND TIME
         ! Heat produced by radioactive decay due to 40K, 235U, 238U, 232Th, respectively

         k = s% nz
         core_epsrad = 36.7d-8 * exp(-5.543d-10 * s% star_age) ! erg/g/sec   40K
         core_epsrad = core_epsrad + 4.6d-8 * exp(-9.8485d-10 * s% star_age) ! erg/g/sec  235U
         core_epsrad = core_epsrad + 2.3d-8 * exp( -1.5513d-10 * s% star_age)! erg/g/sec  238U
         core_epsrad = core_epsrad + 1.3d-8 * exp( -0.4948d-10 * s% star_age)! erg/g/sec  232Th

         s% extra_heat(k) = (extraheat+ s% x_ctrl(4) * s% M_center * core_epsrad / s% dm(k)) ! erg/g/sec, core heat flux density




      end subroutine energy_routine





      !#########module to test extra heating sensitivity######
         SUBROUTINE init_random_seed()
            INTEGER :: i, n, clock
            INTEGER, DIMENSION(:), ALLOCATABLE :: seed
          
            CALL RANDOM_SEED(size = n)
            ALLOCATE(seed(n))
          
            CALL SYSTEM_CLOCK(COUNT=clock)
          
            seed = clock + 37 * (/ (i - 1, i = 1, n) /)
            CALL RANDOM_SEED(PUT = seed)
          
            DEALLOCATE(seed)
          END SUBROUTINE                     
     !!!!!##############################################



      subroutine mass_loss(id, ierr)
         use star_def
         integer, intent(in) :: id
         integer, intent(out) :: ierr

         real(dp) :: rsol
         real(dp) :: msol

         double precision :: eps0
         double precision :: Rhill
         double precision :: Fxuv
         double precision :: Ktide

         double precision :: sigma
         double precision :: Tphoto
         double precision :: mH,muH,Mpl,Rpl,Tacc
         double precision :: Pphoto,g,H,Peuv,Pratio,Reuv,zHeight, Rp, Rs,rhoS,rhoB0,rhoBP,eLim,rrLim,hv,alphaREC,Feuv,Vs, Feuvterm

         integer :: k,i,j,numTass
         real(dp) :: tauHere
         type(star_info), pointer :: s

         call star_ptr(id, s, ierr)

         if ( s% x_ctrl(1) > 0.d0 ) then
           eps0=s% x_ctrl(1) 
	 endif
         do k = 1, s% nz, +1
           tauHere= s% tau(k)
           if (tauHere .ge. 2/3) exit
  	 end do
         ierr = 0
         ! calcuate the radius at which EUV is absorbed
         sigma=6e-18 
         Tphoto= 10**(s% log_surface_temperature)
         mH= 1.67e-24  
	 muH=1.00794 
         Pphoto= 10**(s% log_surface_pressure)  !dyn/cm2
        
        
         hv= 3.2e-11 ! ergs, is 20 eV
         alphaREC=2.7e-13 !cm^3/s

         g=(s% cgrav(1)*s% mstar)/((Rsun*(10**(s% log_surface_radius)))**2)
         Peuv=(muH*g)/(2*sigma)  !convert from dyn/cm2 to bars
         H=(kerg*Tphoto)/(2*muH*g)    !in cm
         Pratio=LOG(Peuv/Pphoto)
         zHeight=-H*Pratio
         Rp=Rsun*(10**(s% log_surface_radius))
         Reuv=Rp+zHeight

         !Energy Limited 
         Feuv=29.7*((s% star_age+1d6)/1d9)**(-1.23)*(s% x_ctrl(2))**(-2)
         Rhill=(s% x_ctrl(2)*1.495978921d13*((s% mstar)/(3*Msun))**(1/3))
         Ktide=1-(3*Reuv)/(2*Rhill)+ 1/(2*(Rhill/Reuv)**3)
         eLim = -(eps0*pi*Feuv*(Reuv)**3)/(s% cgrav(1)*s% mstar*Ktide)

         !Recombination dominated mass loss (full ionization)  
         Vs= ((kerg*10000)/(mH/2))**0.5   !isothermal speed sound Ct of ionized hydrogen gas at 1d4 K  ! mH/2
         !Rs=(((s% cgrav(1)*s% mstar)-3*s% cgrav(1)*Msun*Reuv**3)/(s% x_ctrl(2)*1.496e13))/(2*Vs**2)  !for highly inflated planets 
         Rs=((s% cgrav(1))*(s% mstar))/(2*Vs**2)
         rhoB0=(muH*g/(2*sigma*kerg*10000))
         rhoBP=(((Feuv/hv)*sigma*rhoB0)/alphaREC)**0.5   !the denisty of the base is the balance between neutral and ionized H
         rhoS=rhoBP*muH*exp((s% cgrav(1)*s% mstar)/(Rp*Vs**2)*(Rp/Reuv-1))
         rrLim =  -(4*pi*rhoS*Vs*Rs**2) 


         rrLim= -pi*((s% cgrav(1)*s% mstar)/Vs**2)**2*Vs*mH*((Feuv*s% cgrav(1)*s% mstar)/(hv*alphaREC*Reuv**2*Vs**2))**(1/2)*exp(2-&
	 (s% cgrav(1)*s% mstar)/(Vs**2*Reuv))


         if (ABS(eLim)<ABS(rrLim)) then  !condition for radiative or energy lim
            s% mstar_dot = eLim
         else 
            s% mstar_dot = eLim
         endif


	 !s% accreted_material_j= 
	 !s% star_mdot = +PLmassperyear

         !s% accreted_material_j = &
              !s% x_ctrl(5)*sqrt(s% cgrav(1) * s% mstar * s% photosphere_r*Rsun)

         !write(*,*) "debug", s% mstar_dot/Msun*secyer, 10**(s% x_ctrl(6))
         !s% mstar_dot = s% mstar_dot + 10**(s% x_ctrl(6))*Msun/secyer



            
      end subroutine mass_loss

      subroutine planetary_atm( &
            id, M, R, L, X, Z, kap, Teff, &
            lnT, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            which_atm_option, switch_to_grey_as_backup, ierr)
          !calculates various planetary atmosphere's 
          !implemented by JO August 2013
          use star_lib, only: star_ptr
      
          implicit none

          type (star_info), pointer :: s

          integer, intent(in) :: id ! star id if available; 0 otherwise
          !     cgs units
          double precision, intent(in) :: M ! enclosed mass at atmosphere/interior boundary
          double precision, intent(in) :: R ! radius at atmosphere/interior boundary
          double precision, intent(in) :: L ! luminosity at atmosphere/interior boundary
          double precision, intent(in) :: X ! hydrogen mass fraction
          double precision, intent(in) :: Z ! metallicity
          double precision, intent(in) :: kap
          ! opacity above photosphere (average by mass)
          ! used to estimate gamma for tau boundary

          double precision, intent(out) :: Teff ! temperature at photosphere (essentially meaningless here)
          double precision, intent(out) :: lnT ! natural log of temperature at base of atmosphere
          double precision, intent(out) :: lnP ! natural log of pressure at base of atmosphere (Pgas + Prad)

          ! partial derivatives of lnT and lnP
          double precision, intent(out) :: dlnT_dL, dlnT_dlnR, dlnT_dlnM,dlnT_dlnkap
          double precision, intent(out) :: dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap

          integer, intent(in) :: which_atm_option
          ! atm_simple_photosphere or one of the options for tables
          ! T(tau) integration is not supported in this routine -- call atm_int_T_tau instead
          logical, intent(in) :: switch_to_grey_as_backup
          ! if you select a table option, but the args are out of the range of the tables,
          ! then this flag determines whether you get an error or the code automatically
          ! switches to option = atm_simple_photosphere as a backup.

          integer, intent(out) :: ierr ! == 0 means AOK



          !local variables
          !opacity variables, see Rogers & Seager 2010 
          double precision, parameter   :: C_atm= 4.786300923226380e-08
          double precision, parameter   :: a_atm=0.68
          double precision, parameter   :: b_atm=0.45
          double precision, parameter   :: b_4=0.1125 !b_kap/4

          double precision              :: gamma, Kappa_v, tau_IR, bound_fact
          double precision              :: Tint4, Teq4

          double precision              :: Patm, Tatm ! cgs valuse of P & T at base 

          double precision              :: Tmin4,Tmax4,Tmiddle4 !max and min temperature to fourth power
          double precision              :: gmin, gmax, gmiddle ! gamma at min and max values
          double precision              :: fmin,fmax,fmiddle !function at max or min

          double precision              :: gatm ! g=GM/R^2
          
          double precision              :: En(0:2) !exponetial integral 0,1,2
          double precision              :: big_bracket

          integer                       :: Max_its,its
          double precision              :: conv_tol ! relative convergance tolerance
          logical                       :: converged

          ! access x_ctrl(1) to see which type of atmosphere is required
          !option  1 - fixed P & T boundary (aka disc atmosphere)
          !option  2 - Guillot (2010) T(tau) relation Eqn (49) with RS10 Opacity fit

          ierr = 0
          call get_star_ptr(id, s, ierr)
          if (ierr /= 0) return

          if (s% x_integer_ctrl(51) .eq. 1) then
          !fixed P & T boundary
             !Pressure in Xctrl(2), Temperature in Xctrl(3)
             !read in values from pointers
             Patm= s% x_ctrl(2)
             Tatm= s% x_ctrl(3)

             s% tau_factor = (s% x_ctrl(7)) / (s% tau_base)

             ! assign outputs
             Teff=(L/(4d0*pi*boltz_sigma*R**2))**0.25 ! effective temperature meaninless here
             Teff=maxval((/ Tatm, Teff/))
             lnP=log(Patm)
             lnT=log(Tatm)
             ! all derivatives zero in this implementatihttps://mail.google.com/mail/u/0/?tab=wm#inboxon
             dlnT_dL = 0; dlnT_dlnR = 0; dlnT_dlnM = 0; dlnT_dlnkap = 0
             dlnP_dL = 0; dlnP_dlnR = 0; dlnP_dlnM = 0; dlnP_dlnkap = 0
             


          elseif (s% x_integer_ctrl(51) .eq. 2) then
             !Guillot (2010) T(tau) relation
             !solves for a given Temperature and Pressure at the atmosphere
             !at fixed optical depth to the out going irradiation
             
             !Equation 49 of Guillot (2010) A&A 520,A27 specifies T(tau)

             !solves this using bi-section method where max and min are controlled using x_ctrl inputs

             !set exit condition to false initially
             converged=.false.

             !calculate gravity
             gatm=standard_cgrav*M/R**2d0 


             !import opacity to optical
             kappa_v=(s% x_ctrl(15))
             !find optical depth at boundary (estimate gamma from kap)
             bound_fact=(s% x_ctrl(16))
             tau_IR=bound_fact*(2d0/3d0)
             if (tau_IR .lt. 2d0/3d0) tau_IR=2d0/3d0
             ! calculate new tau factor 
             s% tau_factor= tau_IR / (s% tau_base)

             !import max iterations
             max_its=(s% x_integer_ctrl(52))
             its=0 ! number of iterations initially zero
             !import convergance tolerance
             conv_tol=(s% x_ctrl(20))

             !evaluate Tint4 & Teq4
             Teq4=(s% x_ctrl(19))**4d0
             Tint4=(L/(4*pi*boltz_sigma*R**2))
             

             !set bi-section max and min
             Tmin4=(s% x_ctrl(17))**4d0
             Tmax4=(s% x_ctrl(18))**4d0


             do while (converged .eqv. .false.)
                !calculate gamma for these values
                gmin=kappa_v/(C_atm*(gatm*tau_IR/(C_atm*Tmin4**b_4))**(a_atm/(1+a_atm))*Tmin4**b_4)
                gmax=kappa_v/(C_atm*(gatm*tau_IR/(C_atm*Tmax4**b_4))**(a_atm/(1+a_atm))*Tmax4**b_4)  !Tmin4
                !now evaluate guillot function at max and min
                call guillot_eval(tau_IR,gmin,Teq4,Tint4,Tmin4,fmin)
                call guillot_eval(tau_IR,gmax,Teq4,Tint4,Tmax4,fmax)

                if ((fmin*fmax) > 0.) then
                   !failed in bi-section method
                   ierr=-1
                   write(*,*) "failed in guillot atm at its",its
                   write(*,*) "failed because Tmin and Tmax do not straddle roots"
                   write(*,*) "fmin=", fmin, "fmax=", fmax, "fmin*fmax", fmin*fmax
                   write(*,*) "Tmin=", Tmin4**0.25, "Tmax=", Tmax4**0.25
                   write(*,*) "gamma_min=", gmin, "gamma_max=", gmax
                   write(*,*) "FAILED in other_atm"                   
                   write(*,*) "Star Info Below"
                   write(*,*) "Mass", M
                   write(*,*) "Radius", R
                   write(*,*) "Luminosity", L
                   write(*,*) "tau_IR", tau_IR
                   
                   Teff = 0; lnT = 0; lnP = 0
                   dlnT_dL = 0; dlnT_dlnR = 0; dlnT_dlnM = 0; dlnT_dlnkap = 0
                   dlnP_dL = 0; dlnP_dlnR = 0; dlnP_dlnM = 0; dlnP_dlnkap = 0
                   return
                endif
                !now find middle Tmean in linear T space
                Tmiddle4=(Tmin4**0.25+(Tmax4**0.25-Tmin4**0.25)/2)**4d0
                gmiddle=kappa_v/(C_atm*(gatm*tau_IR/(C_atm*Tmiddle4**b_4))**(a_atm/(1+a_atm))*Tmiddle4**b_4)
                call guillot_eval(tau_IR,gmiddle,Teq4,Tint4,Tmiddle4,fmiddle)

                if (fmax*fmiddle < 0. ) then
                   ! root sits between gmax and gmiddle - relabel middle to min
                   Tmin4=Tmiddle4                   
                elseif (fmin*fmiddle < 0. ) then
                   ! root sits between gmin and gmiddle - relabel middle to max
                   Tmax4=Tmiddle4
                else
                   !failed somewhere
                   ierr=-1
                   write(*,*) "Failed to find Tmiddle in guillot atm at its",its
                   write(*,*) "fmin,fmiddle,fmax below"
                   write(*,*) fmin,fmiddle,fmax
                   write(*,*) "Tmin,Tmiddle,Tmax below"
                   write(*,*) Tmin4**0.25,Tmiddle4**0.25,Tmax4**0.25
                   write(*,*) "gamma_min,gamma_middle,gamma_max below"
                   write(*,*) gmin,gmiddle,gmax
                   write(*,*) "FAILED in other_atm"
                   write(*,*) "Star Info Below"
                   write(*,*) "Mass", M
                   write(*,*) "Radius", R
                   write(*,*) "Luminosity", L
                   write(*,*) "tau_IR", tau_IR
                   return                
                endif

                !convergence testing
                if ((Tmax4**0.25-Tmin4**0.25)/(Tmax4**0.25) .lt. conv_tol) then
                   ! converged
                   converged=.true.
                   exit
                endif

                its=its+1
                !max_its check
                if (its .gt. max_its) then
                   converged=.true.
                   write(*,*) "Atmosphere not converged: failed"
                   write(*,*) "failed to converge in",its,"iterations"
                   write(*,*) fmin,fmiddle,fmax
                   write(*,*) "Tmin,Tmiddle,Tmax below"
                   write(*,*) Tmin4**0.25,Tmiddle4**0.25,Tmax4**0.25
                   write(*,*) "gamma_min,gamma_middle,gamma_max below"
                   write(*,*) gmin,gmiddle,gmax
                   write(*,*) "FAILED in other_at"
                   write(*,*) "Star Info Below"
                   write(*,*) "Mass", M
                   write(*,*) "Radius", R
                   write(*,*) "Luminosity", L
                   write(*,*) "tau_IR", tau_IR
                   ierr=-1
                   return
                endif
             end do

             !now have converged temperature
             !use Tmin as actual value 
             
             Tatm=Tmin4**0.25
             Patm=(gatm*tau_IR/(C_atm*Tatm**b_atm))**(1/(1+a_atm))

             !just output Teff as Tatm
             Teff=Tatm
             lnT=log(Tatm)
             lnP=log(Patm)
          
             !partials of T
             dlnT_dL=((2d0/3d0+tau_IR)*0.75*Tint4/(4d0*Tatm**4d0))/L !dlnT_dlnL / L
             dlnT_dlnR=0. !zero for fixed optical depth at constant pressure
             dlnT_dlnM=0. !zero for fixed optical depth at constant pressure
             !evaluate exonetial integral
             call enxa ( 2, gmin*tau_IR, En )
             big_bracket=-2d0/(3d0*gmin**2)-gmin*tau_IR/3d0*exp(-gmin*tau_IR)+2d0/(3d0*gmin**2d0)*exp(-gmin*tau_IR) &
                  +2d0/3d0*exp(-gmin*tau_IR)+2d0/3d0*(1-(tau_IR**2d0/2d0))*En(2)  & 
                  -2d0*gmin*tau_IR/3d0*(1-(tau_IR**2d0)/2d0)*En(1)

             dlnT_dlnkap=-big_bracket*gmin*(3d0*Teq4/(16d0*Tatm**4d0))

             !partials of P
             ! just come from P=(gatm*tau/(kappa))
             dlnP_dL=0. !change in pressure with luminosity at fixed T is zero
             dlnP_dlnR=-2d0
             dlnP_dlnM=1d0
             dlnP_dlnkap=-1d0

          elseif ((s% x_integer_ctrl(51)) .eq. 3) then
             !Guillot (2010) T(tau) relation
             !solves for a given Temperature and Pressure at the atmosphere
             !at fixed optical depth to the out going radiation
             !unlike option 2 this assumes that the gamma factor is specified in the inlists
             !calculate gravity
             gatm=standard_cgrav*M/R**2d0 

             bound_fact=(s% x_ctrl(16))
             tau_IR=bound_fact*(2d0/3d0)
             tau_IR=2d0
             if (tau_IR .lt. 2d0/3d0) tau_IR=2d0/3d0
             ! calculate new tau factor 
             s% tau_factor= tau_IR / (s% tau_base)

             !evaluate Tint4 & Teq4
             Teq4=(s% x_ctrl(19))**4d0
             Tint4=(L/(4*pi*boltz_sigma*R**2))
             gmiddle=(s% x_ctrl(15)) ! get gamma from x_ctrl
             
             !get the temperature
             call guillot_eval(tau_IR,gmiddle,Teq4,Tint4,0d0,Tmiddle4)
             !assign atmosphere pressure and temperature
             Tatm=(-Tmiddle4)**0.25 ! Tmiddle4 is given minus sign by guillot_eval
             Patm=(gatm*tau_IR/(C_atm*Tatm**b_atm))**(1/(1+a_atm))
             !outputs
             !just output Teff as Tatm
             Teff=Tatm
             lnT=log(Tatm)
             lnP=log(Patm)

             !assing derivatives
             !partials of T
             dlnT_dL=((2d0/3d0+tau_IR)*0.75*Tint4/(4d0*Tatm**4d0))/L !dlnT_dlnL / L
             dlnT_dlnR=0. !zero for fixed optical depth at constant pressure
             dlnT_dlnM=0. !zero for fixed optical depth at constant pressure

             !evaluate exonetial integral
             call enxa ( 2, gmiddle*tau_IR, En )
             big_bracket=-2d0/(3d0*gmiddle**2)-gmiddle*tau_IR/3d0*exp(-gmiddle*tau_IR)+2d0/(3d0*gmiddle**2d0)*exp(-gmiddle*tau_IR) &
                  +2d0/3d0*exp(-gmiddle*tau_IR)+2d0/3d0*(1-(tau_IR**2d0/2d0))*En(2)  & 
                  -2d0*gmiddle*tau_IR/3d0*(1-(tau_IR**2d0)/2d0)*En(1)

             dlnT_dlnkap=-big_bracket*gmiddle*(3d0*Teq4/(16d0*Tatm**4d0))

             !partials of P
             ! just come from P=(gatm*tau/(kappa))
             dlnP_dL=0. !change in pressure with luminosity at fixed T is zero
             dlnP_dlnR=-2d0
             dlnP_dlnM=1d0
             dlnP_dlnkap=-1d0

          else


            

             write(*,*) "no atmosphere option selected"
             write(*,*) "option set by x_integer_ctrl(1), which is currently",s% x_integer_ctrl(1)
             write(*,*) "FAILED IN other_atm"
             ierr=-1
             Teff = 0; lnT = 0; lnP = 0
             dlnT_dL = 0; dlnT_dlnR = 0; dlnT_dlnM = 0; dlnT_dlnkap = 0
             dlnP_dL = 0; dlnP_dlnR = 0; dlnP_dlnM = 0; dlnP_dlnkap = 0
         
          endif

          return
      end subroutine planetary_atm


       subroutine guillot_eval(tau,gamma,Teq4,Tint4,Tmean4,func)

          implicit none

          !evaluates Equation 49 of Guillot et al. (2010) for input values
          
          double precision, intent(in)    ::  tau !optical depth in IR
          double precision, intent(in)    ::  gamma ! ratio of opacities Kappa_v/Kappa_IR
          double precision, intent(in)    ::  Teq4 !equilibrium temperature to the fourth power
          double precision, intent(in)    ::  Tint4 !Internal temperature to the fourth power
          double precision, intent(in)    ::  Tmean4 !mean atmosphere temperature to the fourth power
          double precision, intent(out)   ::  func  ! value of function

          !internals
          double precision                ::  En(0:2) !exponetial integral 0,1,2
          double precision                ::  big_bracket ! curly brackets of EQN 49
          double precision                ::  gt !gamma*tau
          
          !evaluate gamma*tau
          gt=gamma*tau
          !evaluate exonetial integral
          call enxa ( 2, gt, En )

          big_bracket=(2d0/3d0)+(2d0/(3d0*gamma))*(1d0+(gt/2d0-1d0)*exp(-gt))+2d0*gamma/3d0*(1-(tau**2d0)/3d0)*En(2)

          func=Tmean4-0.75*Tint4*(2d0/3d0+tau)-0.75*Teq4*big_bracket
          
        end subroutine guillot_eval
      
      
      integer function extras_startup(id, restart, ierr)
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_startup = 0
         call system_clock(time0,clock_rate)
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup
      
      
      subroutine extras_after_evolve(id, id_extra, ierr)
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         real(dp) :: dt
         character (len=strlen) :: test
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         call GET_ENVIRONMENT_VARIABLE( &
            "MESA_TEST_SUITE_CHECK_RUNTIME", test, status=ierr, trim_name=.true.)
         if (ierr == 0 .and. trim(test) == 'true' .and. dt > 1.5*expected_runtime) then
            write(*,'(/,a70,2f12.1,99i10/)') &
               'failed: EXCESSIVE runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         else
            write(*,'(/,a50,2f12.1,99i10/)') 'runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         end if
         ierr = 0
      end subroutine extras_after_evolve
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_check_model = keep_going         
      end function extras_check_model


      integer function how_many_extra_history_columns(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_history_columns = 4
      end function how_many_extra_history_columns
      
      
      subroutine data_for_extra_history_columns(id, id_extra, n, names, vals, ierr)
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
	 names(1)= "radius(f)"
	 vals(1)= s% r(1)/6.378d8  !10**(s% log_surface_radius)*Rsun

	 !names(2)= "equilibrium temperature"
	 !vals(2)= s% extra_heat(s% nz)


         !names(2)= "tau at the base of envelope"
         !vals(2)= s% tau(s% nz)

         !names(2)= "surface pressure"
         !vals(2)= 10**(s% log_surface_pressure)

         names(2)= "planet mass"
         vals(2)= s% mstar/5.97e27

	 names(3) = "total extra heat"
	 vals(3) =  s% total_extra_heating

	 names(4) = "core mass"
	 vals(4) =  s% M_center


	

      end subroutine data_for_extra_history_columns

      
      integer function how_many_extra_profile_columns(id, id_extra)
         use star_def, only: star_info
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(id, id_extra, n, nz, names, vals, ierr)
         use star_def, only: star_info, maxlen_profile_column_name
         use const_def, only: dp
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         integer :: k
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going or terminate.
      integer function extras_finish_step(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_finish_step = keep_going
         call store_extra_info(s)
      end function extras_finish_step
      
      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            real(dp) :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info

      subroutine enxa ( n, x, en )

!*****************************************************************************80
!
!! ENXA computes the exponential integral En(x).
!
!  Licensing:
!
!    This routine is copyrighted by Shanjie Zhang and Jianming Jin.  However, 
!    they give permission to incorporate this routine into a user program 
!    provided that the copyright is acknowledged.
!
!  Modified:
!
!    07 July 2012
!
!  Author:
!
!    Shanjie Zhang, Jianming Jin
!
!  Reference:
!
!    Shanjie Zhang, Jianming Jin,
!    Computation of Special Functions,
!    Wiley, 1996,
!    ISBN: 0-471-11963-6,
!    LC: QA351.C45.
!
!  Parameters:
!
!    Input, integer ( kind = 4 ) N, the order.
!
!    Input, real ( kind = 8 ) X, the argument.
!
!    Output, real ( kind = 8 ) EN(0:N), the function values.
!
        implicit none
        
        integer ( kind = 4 ) n

        real ( kind = 8 ) e1
        real ( kind = 8 ) ek
        real ( kind = 8 ) en(0:n)
        integer ( kind = 4 ) k
        real ( kind = 8 ) x
        
        en(0) = exp ( - x ) / x 
        call e1xb ( x, e1 )
        
        en(1) = e1
        do k = 2, n
           ek = ( exp ( - x ) - x * e1 ) / ( k - 1.0D+00 )
           en(k) = ek
           e1 = ek
        end do

        return
      end subroutine enxa

      subroutine e1xb ( x, e1 )

!*****************************************************************************80
!
!! E1XB computes the exponential integral E1(x).
!
!  Licensing:
!
!    This routine is copyrighted by Shanjie Zhang and Jianming Jin.  However, 
!    they give permission to incorporate this routine into a user program 
!    provided that the copyright is acknowledged.
!
!  Modified:
!
!    06 July 2012
!
!  Author:
!
!    Shanjie Zhang, Jianming Jin
!
!  Reference:
!
!    Shanjie Zhang, Jianming Jin,
!    Computation of Special Functions,
!    Wiley, 1996,
!    ISBN: 0-471-11963-6,
!    LC: QA351.C45.
!
!  Parameters:
!
!    Input, real ( kind = 8 ) X, the argument.
!
!    Output, real ( kind = 8 ) E1, the function value.
!
        implicit none

        real ( kind = 8 ) e1
        real ( kind = 8 ) ga
        integer ( kind = 4 ) k
        integer ( kind = 4 ) m
        real ( kind = 8 ) r
        real ( kind = 8 ) t
        real ( kind = 8 ) t0
        real ( kind = 8 ) x
        
        if ( x == 0.0D+00 ) then
           
           e1 = 1.0D+300

        else if ( x <= 1.0D+00 ) then

           e1 = 1.0D+00
           r = 1.0D+00

           do k = 1, 25
              r = -r * k * x / ( k + 1.0D+00 )**2
              e1 = e1 + r
              if ( abs ( r ) <= abs ( e1 ) * 1.0D-15 ) then
                 exit
              end if
           end do
    
           ga = 0.5772156649015328D+00
           e1 = - ga - log ( x ) + x * e1

        else

           m = 20 + int ( 80.0D+00 / x )
           t0 = 0.0D+00
           do k = m, 1, -1
              t0 = k / ( 1.0D+00 + k / ( x + t0 ) )
           end do
           t = 1.0D+00 / ( x + t0 )
           e1 = exp ( -x ) * t
    
        end if

        return
      end subroutine e1xb

      end module run_star_extras
      
