<?php

namespace Drupal\foldershare;

use Drupal\foldershare\Entity\FolderShareUsage;

/**
 * Defines functions to get/set the module's configuration.
 *
 * <B>Warning:</B> This class is strictly internal to the FolderShare
 * module. The class's existance, name, and content may change from
 * release to release without any promise of backwards compatability.
 *
 * The module's settings (configuration) schema is defined in
 * config/schema/MODULE.settings.yml, with install-time defaults set
 * in config/install/MODULE.settings.yml. The defaults provided
 * in this module match those in the YML files.
 *
 * @internal
 * Get/set of a module configuration setting requires two strings: (1) the
 * name of the module's configuration, and (2) the name of the setting.
 * When used frequently in module code, these strings invite typos that
 * can cause the wrong setting to be set or retrieved.
 *
 * This class centralizes get/set for settings and turns all settings
 * accesses into class method calls. The PHP parser can then catch typos
 * in method calls and report them as errors.
 *
 * This class also centralizes and makes accessible the default values
 * for all settings.
 * @endinternal
 *
 * @ingroup foldershare
 */
class Settings {

  /*---------------------------------------------------------------------
   *
   * Files - storage scheme.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default public/private file storage scheme.
   *
   * Known values are:
   *
   * - 'public' = store files in the site's public file system.
   * - 'private' = store files in the site's private file system.
   *
   * @return string
   *   Returns the default file storage scheme as either 'public' or 'private'.
   *
   * @see ::getFileScheme()
   * @see ::setFileScheme()
   */
  public static function getFileSchemeDefault() {
    // Try to use the Core File module's default choice, if it is
    // one of 'public' or 'private'. If not recognized, revert to 'public'.
    switch (file_default_scheme()) {
      default:
      case 'public':
        return 'public';

      case 'private':
        return 'private';
    }
  }

  /**
   * Returns the current public/private file storage scheme.
   *
   * Known values are:
   *
   * - 'public' = store files in the site's public file system.
   * - 'private' = store files in the site's private file system.
   *
   * @return string
   *   Returns the file storage scheme as either 'public' or 'private'.
   *
   * @see ::getFileSchemeDefault()
   * @see ::setFileScheme()
   */
  public static function getFileScheme() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('file_scheme') === NULL) {
      return self::getFileSchemeDefault();
    }

    $scheme = $config->get('file_scheme');
    switch ($scheme) {
      case 'public':
      case 'private':
        return $scheme;
    }

    return self::getFileSchemeDefault();
  }

  /**
   * Sets the current public/private file storage scheme.
   *
   * Known values are:
   *
   * - 'public' = store files in the site's public file system.
   * - 'private '= store files in the site's private file system.
   *
   * Unrecognized values are silently ignored.
   *
   * @param string $scheme
   *   The file storage scheme as either 'public' or 'private'.
   *
   * @see ::getFileScheme()
   * @see ::getFileSchemeDefault()
   */
  public static function setFileScheme(string $scheme) {
    switch ($scheme) {
      case 'public':
      case 'private':
        break;

      default:
        return;
    }

    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('file_scheme', $scheme);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Files - extension restriction enable.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default file name extension restrictions flag.
   *
   * @return bool
   *   Returns the default file name extensions flag. The value is TRUE if
   *   file name extensions are restricted, and FALSE otherwise.
   *
   * @see ::getFileRestrictExtensions()
   * @see ::setFileRestrictExtensions()
   */
  public static function getFileRestrictExtensionsDefault() {
    return FALSE;
  }

  /**
   * Returns the current file name extension restrictions flag.
   *
   * Legal values are TRUE or FALSE.
   *
   * @return bool
   *   Returns TRUE if file name extensions are restricted, and false otherwise.
   *
   * @see ::getAllowedNameExtensions()
   * @see ::getFileRestrictExtensionsDefault()
   * @see ::setFileRestrictExtensions()
   */
  public static function getFileRestrictExtensions() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('file_restrict_extensions') === NULL) {
      return self::getFileRestrictExtensionsDefault();
    }

    return boolval($config->get('file_restrict_extensions'));
  }

  /**
   * Sets the current file name extension restrictions flag.
   *
   * @param bool $value
   *   TRUE if file name extensions are restricted, and FALSE otherwise.
   *
   * @see ::getAllowedNameExtensions()
   * @see ::getFileRestrictExtensions()
   * @see ::getFileRestrictExtensionsDefault()
   */
  public static function setFileRestrictExtensions(bool $value) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('file_restrict_extensions', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Files - extension restriction list.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default list of allowed file name extensions.
   *
   * This list is intentionally broad and includes a large set of
   * well-known extensions for text, web, image, video, audio,
   * graphics, data, archive, office, and programming documents.
   *
   * @return string
   *   Returns a string containing a space or comma-separated default
   *   list of file extensions (without the leading dot).
   *
   * @see ::getAllowedNameExtensions()
   * @see ::setAllowedNameExtensions()
   */
  public static function getAllowedNameExtensionsDefault() {
    return implode(' ', FilenameExtensions::getAll());
  }

  /**
   * Returns the current list of allowed file name extensions.
   *
   * The return list is a single string with space-separated dot-free
   * file name extensions allowed for file uploads and renames.
   *
   * The list is only meaningful if file name extension restrictions are
   * enabled.
   *
   * @return string
   *   Returns a string containing a space or comma-separated list of file
   *   extensions (without the leading dot).
   *
   * @see ::getAllowedNameExtensionsDefault()
   * @see ::getFileRestrictExtensions()
   * @see ::setAllowedNameExtensions()
   * @see ::setFileRestrictExtensions()
   */
  public static function getAllowedNameExtensions() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('file_allowed_extensions') === NULL) {
      return self::getAllowedNameExtensionsDefault();
    }

    return (string) $config->get('file_allowed_extensions');
  }

  /**
   * Sets the current list of allowed file name extensions.
   *
   * The given list must be a single string with space-separated dot-free
   * file name extensions allowed for file uploads and renames.
   *
   * The list is only meaningful if file name extension restrictions are
   * enabled.
   *
   * @param string $ext
   *   A string containing a space or comma-separated list of file
   *   extensions (without the leading dot).
   *
   * @see ::getAllowedNameExtensions()
   * @see ::getAllowedNameExtensionsDefault()
   * @see ::getFileRestrictExtensions()
   * @see ::setFileRestrictExtensions()
   */
  public static function setAllowedNameExtensions(string $ext) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('file_allowed_extensions', $ext);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Command menu - restrictions enable.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default command menu restrictions flag.
   *
   * @return bool
   *   Returns TRUE if the command menu is restricted by default, and
   *   FALSE otherwise.
   *
   * @see ::getCommandMenuRestrict()
   * @see ::setCommandMenuRestrict()
   */
  public static function getCommandMenuRestrictDefault() {
    return FALSE;
  }

  /**
   * Returns the current command menu restrictions flag.
   *
   * When FALSE, all available plugin commands are allowed on the user
   * interface's command menu. When TRUE, this list is restricted to only
   * the commands selected by the site administrator.
   *
   * @return bool
   *   Returns TRUE if the command menu is restricted, and FALSE otherwise.
   *
   * @see ::getCommandMenuRestrictDefault()
   * @see ::setCommandMenuRestrict()
   */
  public static function getCommandMenuRestrict() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('command_menu_restrict') === NULL) {
      return self::getCommandMenuRestrictDefault();
    }

    return boolval($config->get('command_menu_restrict'));
  }

  /**
   * Sets the current command menu restrictions flag.
   *
   * When FALSE, all available plugin commands are allowed on the user
   * interface's command menu. When TRUE, this list is restricted to only
   * the commands selected by the site administrator.
   *
   * @param bool $value
   *   TRUE if the command menu is restricted, and FALSE otherwise.
   *
   * @see ::getCommandMenuRestrict()
   * @see ::getCommandMenuRestrictDefault()
   */
  public static function setCommandMenuRestrict(bool $value) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('command_menu_restrict', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Command menu - restriction list.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns a list of all available plugin command definitions.
   *
   * The returned list includes all currently installed plugin commands,
   * regardless of the module source for the commands. Depending upon
   * module settings, some of these commands are allowed on the command
   * menu, while others are disabled from the user interface.
   *
   * @return array
   *   Returns an array of plugin command definitions.  Array keys are
   *   command keys, and array values are command definitions.
   *
   * @see ::getAllowedCommandDefinitions()
   */
  public static function getAllCommandDefinitions() {
    // Get the command plugin manager.
    $container = \Drupal::getContainer();
    $manager = $container->get('foldershare.plugin.manager.foldersharecommand');
    if ($manager === NULL) {
      // No manager? Return nothing.
      return [];
    }

    // Get all the command definitions. The returned array has array keys
    // as plugin IDs, and array values as definitions.
    return $manager->getDefinitions();
  }

  /**
   * Returns a list of allowed plugin command definitions.
   *
   * The returned list includes all currently installed plugin commands,
   * regardless of the module source for the commands. This list is then
   * filtered to only include those commands that are currently allowed,
   * based upon module settings for command menu restrictions and content.
   *
   * @return array
   *   Returns an array of plugin command definitions.  Array keys are
   *   command keys, and array values are command definitions.
   *
   * @see ::getAllCommandDefinitions()
   */
  public static function getAllowedCommandDefinitions() {
    // Get a list of all installed commands.
    $defs = self::getAllCommandDefinitions();

    // If the command menu is not restricted, return the entire list.
    if (self::getCommandMenuRestrict() === FALSE) {
      return $defs;
    }

    // Otherwise the menu is restricted. Get the current list of allowed
    // command IDs.
    $ids = self::getCommandMenuAllowed();

    // Cull the list of definitions to only those allowed on the command
    // menu.
    $allowed = [];
    foreach ($defs as $id => $def) {
      if (in_array($id, $ids) === TRUE) {
        $allowed[$id] = $def;
      }
    }

    return $allowed;
  }

  /**
   * Returns the default list of allowed plugin commands for menus.
   *
   * The returned list is an array of command plugin IDs. By default,
   * this list includes all command plugins currently installed.
   *
   * @return array
   *   Returns an array of plugin command IDs.
   *
   * @see ::getAllCommandDefinitions()
   * @see ::getCommandMenuAllowed()
   * @see ::setCommandMenuAllowed()
   */
  public static function getCommandMenuAllowedDefault() {
    return array_keys(self::getAllCommandDefinitions());
  }

  /**
   * Returns the current list of allowed plugin commands for menus.
   *
   * The returned list is an array of command plugin IDs.
   *
   * Plugin IDs are not validated and could list plugins that are no
   * longer installed at the site.
   *
   * @return array
   *   Returns an array of plugin command IDs.
   *
   * @see ::getCommandMenuAllowedDefault()
   * @see ::setCommandMenuAllowed()
   */
  public static function getCommandMenuAllowed() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('command_menu_allowed') === NULL) {
      // Nothing set yet. Revert to default.
      return self::getCommandMenuAllowedDefault();
    }

    $ids = $config->get('command_menu_allowed');
    if (is_array($ids) === TRUE) {
      return $ids;
    }

    // The stored value is bogus. Reset it to the default.
    $ids = self::getCommandMenuAllowedDefault();
    self::setCommandMenuAllowed($ids);
    return $ids;
  }

  /**
   * Sets the current list of allowed plugin commands for menus.
   *
   * The given list is an array of command plugin IDs.
   *
   * Plugin IDs are not validated and could list plugins that are no
   * longer installed at the site.
   *
   * @param array $ids
   *   An array of plugin command IDs.
   *
   * @see ::getCommandMenuAllowed()
   * @see ::getCommandMenuAllowedDefault()
   */
  public static function setCommandMenuAllowed(array $ids) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);

    if (empty($ids) === TRUE || is_array($ids) === FALSE) {
      // Set the menu to be an empty list.
      $config->set('command_menu_allowed', []);
      $config->save(TRUE);
    }
    else {
      $config->set('command_menu_allowed', $ids);
      $config->save(TRUE);
    }
  }

  /*---------------------------------------------------------------------
   *
   * Command menu - submenu threshold.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default command submenu threshold.
   *
   * @return int
   *   Returns the submenu threshold.
   *
   * @see ::getCommandMenuSubmenuThreshold()
   * @see ::setCommandMenuSubmenuThreshold()
   */
  public static function getCommandMenuSubmenuThresholdDefault() {
    return 3;
  }

  /**
   * Returns the current command submenu threshold.
   *
   * @return int
   *   Returns the submenu threshold.
   *
   * @see ::getCommandMenuSubmenuThresholdDefault()
   * @see ::setCommandMenuSubmenuThreshold()
   */
  public static function getCommandMenuSubmenuThreshold() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('command_menu_submenu_threshold') === NULL) {
      return self::getCommandMenuSubmenuThresholdDefault();
    }

    return $config->get('command_menu_submenu_threshold');
  }

  /**
   * Sets the current command submenu threshold.
   *
   * The submenu threshold indicates the maximum number of adjacent
   * commands in a command category on the main menu before the commands
   * are moved to a submenu. Having a threshold helps insure the command
   * menu does not get long and unwieldy when there are large number of
   * command plugins installed.
   *
   * @param int $threshold
   *   The submenu threshold. Values <= 2 are ignored.
   *
   * @see ::getCommandMenuSubmenuThreshold()
   * @see ::getCommandMenuSubmenuThresholdDefault()
   */
  public static function setCommandMenuSubmenuThreshold(int $threshold) {
    if ($threshold <= 2) {
      return;
    }

    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('command_menu_submenu_threshold', $threshold);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Commands - report normal completion.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default command normal completion report flag.
   *
   * @return bool
   *   Returns the default command normal completion report flag.
   *
   * @see ::getCommandNormalCompletionReportEnable()
   * @see ::setCommandNormalCompletionReportEnable()
   */
  public static function getCommandNormalCompletionReportEnableDefault() {
    return FALSE;
  }

  /**
   * Returns the current command normal completion report flag.
   *
   * @return bool
   *   Returns the command normal completion report flag.
   *
   * @see ::getCommandNormalCompletionReportEnableDefault()
   * @see ::setCommandNormalCompletionReportEnable()
   */
  public static function getCommandNormalCompletionReportEnable() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('command_normal_completion_report') === NULL) {
      return self::getCommandNormalCompletionReportEnableDefault();
    }

    return boolval($config->get('command_normal_completion_report'));
  }

  /**
   * Sets the current command normal completion report flag.
   *
   * Commands always report errors to Drupal's messenger, which are then
   * often picked up and shown in an error dialog. Optionally, commands
   * also can report normal completion such as "Item deleted" or
   * "Item copied".
   *
   * While normal completion messages are common in Drupal for nodes and
   * other entities, they are not conventional in file systems. Windows
   * Explorer, macOS Finder, and the various Linux file browsers do not
   * put up a dialog or show a message every time a file or folder is
   * created, deleted, moved, copied, or renamed.
   *
   * When this flag is TRUE, normal completion messages are reported to
   * the user. When FALSE (which is recommended), these messages are
   * suppressed.
   *
   * @param bool $enable
   *   The command normal completion report flag.
   *
   * @see ::getCommandNormalCompletionReportEnable()
   * @see ::getCommandNormalCompletionReportEnableDefault()
   */
  public static function setCommandNormalCompletionReportEnable(bool $enable) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('command_normal_completion_report', $enable);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Usage report - rebuild interval.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default rebuild interval for the usage table.
   *
   * @return string
   *   The default rebuild interval.
   *
   * @see ::getUsageReportRebuildInterval()
   * @see ::setUsageReportRebuildInterval()
   */
  public static function getUsageReportRebuildIntervalDefault() {
    return 'manual';
  }

  /**
   * Returns the current rebuild interval for the usage table update.
   *
   * Known values are:
   *
   * - 'manual' = the table is only rebuilt manually.
   * - 'hourly' = the table is rebuild hourly.
   * - 'daily' = the table is rebuild daily.
   * - 'weekly' = the table is rebuilt weekly.
   *
   * @return string
   *   The rebuild interval.
   *
   * @see ::getUsageReportRebuildIntervalDefault()
   * @see ::setUsageReportRebuildInterval()
   */
  public static function getUsageReportRebuildInterval() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('usage_report_rebuild_interval') === NULL) {
      return self::getUsageReportRebuildIntervalDefault();
    }

    $interval = (string) $config->get('usage_report_rebuild_interval');
    switch ($interval) {
      case 'manual':
      case 'hourly':
      case 'daily':
      case 'weekly':
        return $interval;

      default:
        return self::getUsageReportRebuildIntervalDefault();
    }
  }

  /**
   * Sets the rebuild interval for the usage table update.
   *
   * Known values are:
   *
   * - 'manual' = the table is only rebuilt manually.
   * - 'hourly' = the table is rebuild hourly.
   * - 'daily' = the table is rebuild daily.
   * - 'weekly' = the table is rebuilt weekly.
   *
   * Rebuilding the usage table is fairly expensive. For production
   * environments, some administrators will check the table frequently
   * to monitor usage, so they will need a rapid update. Other administrators
   * may rarely check the table, or never check it, so using a less rapid
   * update makes sense.
   *
   * Regardless of this setting, the usage table may be manually updated
   * from the administrator's user interface.
   *
   * @param string $interval
   *   The rebuild interval.
   *
   * @see ::getUsageReportRebuildInterval()
   * @see ::getUsageReportRebuildIntervalDefault()
   */
  public static function setUsageReportRebuildInterval(string $interval) {
    $interval = mb_convert_case($interval, MB_CASE_LOWER);

    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    if ($config->get('usage_report_rebuild_interval') === $interval) {
      // No change.
      return;
    }

    switch ($interval) {
      case 'manual':
      case 'hourly':
      case 'daily':
      case 'weekly':
        break;

      default:
        $interval = self::getUsageReportRebuildIntervalDefault();
        break;
    }

    $config->set('usage_report_rebuild_interval', $interval);
    $config->save(TRUE);

    FolderShareUsage::updateRebuildInterval($interval);
  }

  /*---------------------------------------------------------------------
   *
   * Usage report - latest report time.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default date for the last usage table update.
   *
   * This is only needed if there is no previous date, so this function
   * always returns 'never'.
   *
   * @return string
   *   Returns the string 'never'.
   *
   * @see ::getUsageReportTime()
   * @see ::setUsageReportTime()
   */
  public static function getUsageReportTimeDefault() {
    return 'never';
  }

  /**
   * Returns the date of the most recent usage table update.
   *
   * Known values are:
   *
   * - 'never' = the table has never been updated.
   * - a date string for the most recent update date and time.
   *
   * @return string
   *   The most recent update date.
   *
   * @see ::getUsageReportTimeDefault()
   * @see ::setUsageReportTime()
   *
   * @todo Currently the time is stored in the configuration. It should be
   * moved to module state instead since it is not a configuration parameter.
   */
  public static function getUsageReportTime() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('usage_report_time') === NULL) {
      return self::getUsageReportTimeDefault();
    }

    return (string) $config->get('usage_report_time');
  }

  /**
   * Sets the date of the most recent usage table update.
   *
   * Known values are:
   *
   * - 'never' = the table has never been updated.
   * - a date string for the most recent update date and time.
   *
   * @param string $date
   *   The most recent update date.
   *
   * @see ::getUsageReportTime()
   * @see ::getUsageReportTimeDefault()
   *
   * @todo Currently the time is stored in the configuration. It should be
   * moved to module state instead since it is not a configuration parameter.
   */
  public static function setUsageReportTime(string $date) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('usage_report_time', $date);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Logging - enable.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default activity logging enable flag.
   *
   * @return bool
   *   Returns TRUE if activity logging is enabled by default, and
   *   FALSE otherwise.
   *
   * @see ::getActiveLogEnable()
   * @see ::setActiveLogEnable()
   */
  public static function getActivityLogEnableDefault() {
    return FALSE;
  }

  /**
   * Returns the current activity logging flag.
   *
   * @return bool
   *   Returns if TRUE if activity logging is enabled, and FALSE otherwise.
   *
   * @see ::getActiveLogEnableDefault()
   * @see ::setActiveLogEnable()
   */
  public static function getActivityLogEnable() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('activity_log') === NULL) {
      return self::getActivityLogEnableDefault();
    }

    return boolval($config->get('activity_log'));
  }

  /**
   * Sets the current activity logging flag.
   *
   * After each operation that copies, deletes, moves, otherwise changes the
   * file and folder tree, a log message can be posted. During production
   * use, this is usually disabled. But during development, or if close tracking
   * of activity is needed, this can be enabled.
   *
   * @param bool $value
   *   TRUE to enable activity logging, and FALSE to disable.
   *
   * @see ::getActiveLogEnable()
   * @see ::getActiveLogEnableDefault()
   */
  public static function setActivityLogEnable(bool $value) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('activity_log', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * ZIP archive - name.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default name for new ZIP archives.
   *
   * @return string
   *   Returns 'Archive.zip'.
   *
   * @see ::getNewZipArchiveName()
   * @see ::setNewZipArchiveName()
   */
  public static function getNewZipArchiveNameDefault() {
    return 'Archive.zip';
  }

  /**
   * Returns the current name for new ZIP archives.
   *
   * @return string
   *   Returns the name.
   *
   * @see ::getNewZipArchiveNameDefault()
   * @see ::setNewZipArchiveName()
   */
  public static function getNewZipArchiveName() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('new_zip_archive_name') === NULL) {
      return self::getNewZipArchiveNameDefault();
    }

    return (string) $config->get('new_zip_archive_name');
  }

  /**
   * Sets the current name for new ZIP archives.
   *
   * When a new ZIP archive is created, it nees a name. The default name
   * is something like 'Archive.zip', but it could be anything. Other
   * possible names are 'Compressed.zip', 'Data.zip', 'FilesAndFolders.zip',
   * etc.
   *
   * @param string $value
   *   The name for new ZIP archives.
   *
   * @see ::getNewZipArchiveName()
   * @see ::getNewZipArchiveNameDefault()
   */
  public static function setNewZipArchiveName(string $value) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('new_zip_archive_name', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * ZIP archive - comment.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default comment for new ZIP archives.
   *
   * @return string
   *   Returns 'Created by FolderShare.'
   *
   * @see ::getNewZipArchiveComment()
   * @see ::setNewZipArchiveComment()
   */
  public static function getNewZipArchiveCommentDefault() {
    return 'Created by FolderShare.';
  }

  /**
   * Returns the current comment for new ZIP archives.
   *
   * @return string
   *   Returns the name.
   *
   * @see ::getNewZipArchiveCommentDefault()
   * @see ::setNewZipArchiveComment()
   */
  public static function getNewZipArchiveComment() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('new_zip_archive_comment') === NULL) {
      return self::getNewZipArchiveCommentDefault();
    }

    return (string) $config->get('new_zip_archive_comment');
  }

  /**
   * Sets the current comment for new ZIP archives.
   *
   * When a new ZIP archive is created, a short comment is added to the
   * archive that indicates where the archive came from. This defaults
   * to saying the archive came from this module, but a site could change
   * it to include the website's name.
   *
   * @param string $value
   *   The name for new ZIP archives.
   *
   * @see ::getNewZipArchiveComment()
   * @see ::getNewZipArchiveCommentDefault()
   */
  public static function setNewZipArchiveComment(string $value) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('new_zip_archive_comment', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * ZIP archive - unarchive behavior.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default ZIP unarchive multiple to subfolder flag.
   *
   * @return bool
   *   Returns TRUE.
   *
   * @see ::getZipUnarchiveMultipleToSubfolder()
   * @see ::setZipUnarchiveMultipleToSubfolder()
   */
  public static function getZipUnarchiveMultipleToSubfolderDefault() {
    return TRUE;
  }

  /**
   * Returns the current ZIP unarchive multiple to subfolder flag.
   *
   * @return bool
   *   Returns TRUE if unarchiving multiple items should place them in
   *   a subfolder, and FALSE if they should be placed in the current folder.
   *
   * @see ::getZipUnarchiveMultipleToSubfolderDefault()
   * @see ::setZipUnarchiveMultipleToSubfolder()
   */
  public static function getZipUnarchiveMultipleToSubfolder() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('zip_unarchive_multiple_to_subfolder') === NULL) {
      return self::getZipUnarchiveMultipleToSubfolderDefault();
    }

    return boolval($config->get('zip_unarchive_multiple_to_subfolder'));
  }

  /**
   * Sets the current ZIP unarchive multiple to subfolder flag.
   *
   * When a ZIP archive holds multiple files and folders, there are two
   * possible behaviors when the archive is un-ZIPed:
   *
   * - Create a subfolder named after the archive to contain the items (TRUE).
   * - Unarchive the items directly into the current location (FALSE).
   *
   * @param bool $value
   *   Returns TRUE if unarchiving multiple items should place them in
   *   a subfolder, and FALSE if they should be placed in the current folder.
   *
   * @see ::getZipUnarchiveMultipleToSubfolder()
   * @see ::getZipUnarchiveMultipleToSubfolderDefault()
   */
  public static function setZipUnarchiveMultipleToSubfolder(bool $value) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('zip_unarchive_multiple_to_subfolder', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Locks - content lock duration.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default content lock duration in seconds.
   *
   * @return float
   *   Returns the default content lock duration in seconds.
   *
   * @see ::getContentLockDuration()
   * @see ::setContentLockDuration()
   * @see ::getOperationLockDuration()
   * @see ::setOperationLockDuration()
   */
  public static function getContentLockDurationDefault() {
    return 60.0;
  }

  /**
   * Returns the current content lock duration in seconds.
   *
   * @return float
   *   Returns the content lock duration in seconds.
   *
   * @see ::setContentLockDuration()
   * @see ::getContentLockDurationDefault()
   * @see ::setOperationLockDuration()
   * @see ::getOperationLockDurationDefault()
   */
  public static function getContentLockDuration() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('lock_content_duration') === NULL) {
      return self::getContentLockDurationDefault();
    }

    return floatval($config->get('lock_content_duration'));
  }

  /**
   * Sets the current content lock duration in seconds.
   *
   * For single item locks, this duration sets the length of time before
   * the lock automatically expires. This time is chosen to be long enough
   * for typical single-item operations to complete, and short enough that
   * if something crashes, users and developers don't have too long to wait
   * until the lock expires and work can continue.
   *
   * @param float $value
   *   The content lock duration in seconds.
   *
   * @see ::getContentLockDuration()
   * @see ::getContentLockDurationDefault()
   * @see ::getOperationLockDuration()
   * @see ::getOperationLockDurationDefault()
   */
  public static function setContentLockDuration(float $value) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('lock_content_duration', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Locks - operation lock duration.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default operation lock duration in seconds.
   *
   * @return float
   *   Returns the default operation lock duration in seconds.
   *
   * @see ::getContentLockDuration()
   * @see ::setContentLockDuration()
   * @see ::getOperationLockDuration()
   * @see ::setOperationLockDuration()
   */
  public static function getOperationLockDurationDefault() {
    return 3600.0;
  }

  /**
   * Returns the current operation lock duration in seconds.
   *
   * @return float
   *   Returns the operation lock duration in seconds.
   *
   * @see ::setContentLockDuration()
   * @see ::getContentLockDurationDefault()
   * @see ::setOperationLockDuration()
   * @see ::getOperationLockDurationDefault()
   */
  public static function getOperationLockDuration() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('lock_operation_duration') === NULL) {
      return self::getOperationLockDurationDefault();
    }

    return floatval($config->get('lock_operation_duration'));
  }

  /**
   * Sets the current operation lock duration in seconds.
   *
   * For large operations that operate on an entire folder tree, locks
   * are acquired on the root of the folder tree and they need to last
   * long enough that the entire operation can complete before the folder
   * tree is unlocked. Since such operations could span multiple scheduled
   * task executions, and the rate of those executions depends upon system
   * activity and CRON intervals, it is possible that an operation could
   * span minutes or even hours.
   *
   * @param float $value
   *   The operation lock duration in seconds.
   *
   * @see ::getContentLockDuration()
   * @see ::getContentLockDurationDefault()
   * @see ::getOperationLockDuration()
   * @see ::getOperationLockDurationDefault()
   */
  public static function setOperationLockDuration(float $value) {
    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('lock_operation_duration', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Scheduling - initial task delay.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default initial scheduled task delay in seconds.
   *
   * @return float
   *   Returns the default initial scheduled task delay.
   *
   * @see ::getScheduledTaskInitialDelay()
   * @see ::setScheduledTaskInitialDelay()
   */
  public static function getScheduledTaskInitialDelayDefault() {
    return 3.0;
  }

  /**
   * Returns the current initial scheduled task delay in seconds.
   *
   * @return float
   *   Returns the initial scheduled task delay in seconds.
   *
   * @see ::getScheduledTaskInitialDelayDefault()
   * @see ::setScheduledTaskInitialDelay()
   */
  public static function getScheduledTaskInitialDelay() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('scheduled_task_initial_delay') === NULL) {
      return self::getScheduledTaskInitialDelayDefault();
    }

    return floatval($config->get('scheduled_task_initial_delay'));
  }

  /**
   * Sets the current initial scheduled task delay in seconds.
   *
   * The initial scheduled task delay is the time offset from the current
   * time to the time at which a newly scheduled task should execute.
   * This initial delay is only used for new operations, such as the first
   * task in a copy, delete, or move. Such operations are typically started
   * by the user interface and a delay is needed to give the interface a
   * chance to refresh its page, including any AJAX requests involved.
   *
   * The delay for an initial task should be pretty short, such as a
   * few seconds.
   *
   * Values less than 3 seconds are ignored.
   *
   * @param float $value
   *   The initial scheduled task delay in seconds.
   *
   * @see ::getScheduledTaskInitialDelay()
   * @see ::getScheduledTaskInitialDelayDefault()
   */
  public static function setScheduledTaskInitialDelay(float $value) {
    if ($value < 3) {
      return;
    }

    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('scheduled_task_initial_delay', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Scheduling - continuation task delay.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default continuation scheduled task delay in seconds.
   *
   * @return float
   *   Returns the default continuation scheduled task delay.
   *
   * @see ::getScheduledTaskContinuationDelay()
   * @see ::setScheduledTaskContinuationDelay()
   */
  public static function getScheduledTaskContinuationDelayDefault() {
    return 3.0;
  }

  /**
   * Returns the current continuation scheduled task delay in seconds.
   *
   * @return float
   *   Returns the continuation scheduled task delay in seconds.
   *
   * @see ::getScheduledTaskContinuationDelayDefault()
   * @see ::setScheduledTaskContinuationDelay()
   */
  public static function getScheduledTaskContinuationDelay() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('scheduled_task_continuation_delay') === NULL) {
      return self::getScheduledTaskContinuationDelayDefault();
    }

    return floatval($config->get('scheduled_task_continuation_delay'));
  }

  /**
   * Sets the current continuation scheduled task delay in seconds.
   *
   * The continuation scheduled task delay is the time offset from the current
   * time to the time at which a continuing scheduled task should execute.
   * Continuation tasks are intermediate tasks that perform phase after
   * phase of a multi-step operation. This could be multiple steps in doing
   * a large copy, delete, or move. A delay is needed to let page updates
   * happen, including any AJAX requests involved.
   *
   * The delay for a continuation task should be pretty short, such as a
   * few seconds.
   *
   * Values less than 3 seconds are ignored.
   *
   * @param float $value
   *   The continuation scheduled task delay in seconds.
   *
   * @see ::getScheduledTaskContinuationDelay()
   * @see ::getScheduledTaskContinuationDelayDefault()
   */
  public static function setScheduledTaskContinuationDelay(float $value) {
    if ($value < 3) {
      return;
    }

    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('scheduled_task_continuation_delay', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Scheduling - safety net task delay.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default safety net scheduled task delay in seconds.
   *
   * @return float
   *   Returns the default safety net scheduled task delay.
   *
   * @see ::getScheduledTaskSafetyNetDelay()
   * @see ::setScheduledTaskSafetyNetDelay()
   */
  public static function getScheduledTaskSafetyNetDelayDefault() {
    return 120.0;
  }

  /**
   * Returns the current safety net scheduled task delay in seconds.
   *
   * @return float
   *   Returns the safety net scheduled task delay in seconds.
   *
   * @see ::getScheduledTaskSafetyNetDelayDefault()
   * @see ::setScheduledTaskSafetyNetDelay()
   */
  public static function getScheduledTaskSafetyNetDelay() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('scheduled_task_safety_net_delay') === NULL) {
      return self::getScheduledTaskSafetyNetDelayDefault();
    }

    return floatval($config->get('scheduled_task_safety_net_delay'));
  }

  /**
   * Sets the current safety net scheduled task delay in seconds.
   *
   * The safety net scheduled task delay is the time offset from the current
   * time to the time at which a failsafe "safety net" scheduled task should
   * execute. Safety net tasks are automatically scheduled at the *start* of
   * any operation phase, such as the start of a copy, delete, or move.
   * The task's job is to continue the operation if the current process is
   * killed due to a crash, timeout, or other interrupt.
   *
   * The delay for the safety net task should be large enough to give the
   * task a chance to get some work done, yet short enough that the user
   * doesn't have to wait a long time after a crash before the task resumes.
   *
   * Values less than 3 seconds are ignored.
   *
   * @param float $value
   *   The safety net scheduled task delay in seconds.
   *
   * @see ::getScheduledTaskSafetyNetDelay()
   * @see ::getScheduledTaskSafetyNetDelayDefault()
   */
  public static function setScheduledTaskSafetyNetDelay(float $value) {
    if ($value < 3) {
      return;
    }

    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('scheduled_task_safety_net_delay', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Limits - memory use limit percentage.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default memory use limit percentage.
   *
   * @return float
   *   Returns the default memory use limit percentage as a faction
   *   (e.g. 50% = 0.5).
   *
   * @see ::getMemoryUseLimitPercentage()
   */
  public static function getMemoryUseLimitPercentageDefault() {
    return 0.80;
  }

  /**
   * Returns the current memory use limit percentage.
   *
   * @return float
   *   Returns the memory use limit percentage as a faction
   *   (e.g. 50% = 0.5).
   *
   * @see ::getMemoryUseLimitPercentageDefault()
   * @see ::setMemoryUseLimitPercentage()
   */
  public static function getMemoryUseLimitPercentage() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('memory_use_limit_percentage') === NULL) {
      return self::getMemoryUseLimitPercentageDefault();
    }

    return floatval($config->get('memory_use_limit_percentage'));
  }

  /**
   * Sets the current memory use limit percentage.
   *
   * Large operations, such as copying a big folder tree, may require long
   * run times and loading and storing many entities. Drupal core, and
   * third-party modules, have memory leaks that cause memory use to
   * gradually increase as an active process runs. If memory use reaches
   * PHP's configured "memory_limit", the process will be aborted. This
   * abort can leave operations incomplete and in an indetermine state.
   * It can also corrupt the file system.
   *
   * To avoid memory limit caused aborts, this module's code may monitor
   * its memory use and intentially stop and schedule a background task
   * to continue the work. When that task runs, it may again get near
   * the memory limit and intentionally stop and reschedule, and so on until
   * the work is done.
   *
   * PHP's memory use limit is measured in bytes. Site administrators may
   * set this in the PHP configuration file or in website-specific files,
   * such as Apache's .htaccess. The default value is 128 Mbytes.
   *
   * The memory limit is computed as a percentage of PHP's memory limit.
   * This allows the limit to float up or down based on whatever the
   * website administrator has chosen. The percentage should never be 100%,
   * or code will be aborted. A maximum practical value is probably 90%,
   * which leaves a small margin for the task to clean up and schedule a
   * new task.
   *
   * While this memory limit is primarily used by background tasks, it may
   * be used anywhere in module code.
   *
   * @param float $value
   *   The memory use limit percentage as a faction (e.g. 50% = 0.5).
   *   Values less than 0.3 or greater than 0.9 are ignored.
   *
   * @see ::getMemoryUseLimitPercentage()
   * @see ::getMemoryUseLimitPercentageDefault()
   * @see \Drupal\foldershare\Utilities::getPhpMemoryUseLimit()
   * @see \Drupal\foldershare\Utilities::getMemoryUseLimit()
   * @see \Drupal\foldershare\Utilities::aboveMemoryUseLimit()
   */
  public static function setMemoryUseLimitPercentage(float $value) {
    if ($value > 0.9 || $value < 0.3) {
      return;
    }

    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('memory_use_limit_percentage', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Limits - execution time limit percentage.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default execution time limit percentage.
   *
   * @return float
   *   Returns the default execution time limit percentage
   *
   * @see ::getExecutionTimeLimitPercentage()
   * @see ::setExecutionTimeLimitPercentage()
   * @see ::getResponseExecutionTimeLimitDefault()
   * @see ::getResponseExecutionTimeLimit()
   * @see ::setResponseExecutionTimeLimit()
   */
  public static function getExecutionTimeLimitPercentageDefault() {
    return 0.80;
  }

  /**
   * Returns the current execution time limit percentage.
   *
   * @return float
   *   Returns the execution time limit percentage.
   *
   * @see ::getExecutionTimeLimitPercentageDefault()
   * @see ::setExecutionTimeLimitPercentage()
   * @see ::getResponseExecutionTimeLimitDefault()
   * @see ::getResponseExecutionTimeLimit()
   * @see ::setResponseExecutionTimeLimit()
   * @see \Drupal\foldershare\Utilities::getPhpExecutionTimeLimit()
   * @see \Drupal\foldershare\Utilities::getExecutionTimeLimit()
   * @see \Drupal\foldershare\Utilities::aboveExecutionTimeLimit()
   */
  public static function getExecutionTimeLimitPercentage() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('execution_time_limit_percentage') === NULL) {
      return self::getExecutionTimeLimitPercentageDefault();
    }

    return floatval($config->get('execution_time_limit_percentage'));
  }

  /**
   * Sets the current execution time limit percentage.
   *
   * Large operations, such as a deep folder tree copy, could run for minutes
   * or even an hour or more, but trying to do so can easily exceed two
   * execution time limits:
   *
   * - PHP's 'max_execution_time', set by site administrators in PHP's
   *   configuration file or in website-specific files, such as Apache's
   *   .htaccess. This defaults to 30 seconds.
   *
   * - A web server's maximum time, such as Apache's 'TimeOut' from the
   *   server's configuration file. Other servers have equivalent features.
   *   This often defaults to 300 seconds.
   *
   * If execution time excedes these limits, the process will be aborted.
   * This will cause the current operation to stop immediately and leave
   * things in an indetermine state that can corrupt the file system.
   *
   * To avoid this abort, this module's code may monitor execution time
   * and intentionally stop and schedule a task to continue the operation
   * later. That task will start later in a new process and again do work
   * until it gets close to the execution time limit.
   *
   * The execution time limit is computed as a percentage of PHP's execution
   * time limit. This allows the limit to float up or down based on whatever the
   * website administrator has chosen. The percentage should never be 100%,
   * or code will be aborted. A maximum practical value is probably 90%,
   * which leaves a small margin for the task to clean up and schedule a
   * new task.
   *
   * While this exeuction time limit is primarily used by background tasks,
   * it may be used anywhere in module code.
   *
   * @param float $value
   *   The execution time limit percentage.
   *
   * @see ::getExecutionTimeLimitPercentage()
   * @see ::getExecutionTimeLimitPercentageDefault()
   * @see ::getResponseExecutionTimeLimitDefault()
   * @see ::getResponseExecutionTimeLimit()
   * @see ::setResponseExecutionTimeLimit()
   * @see \Drupal\foldershare\Utilities::getPhpExecutionTimeLimit()
   * @see \Drupal\foldershare\Utilities::getExecutionTimeLimit()
   * @see \Drupal\foldershare\Utilities::aboveExecutionTimeLimit()
   */
  public static function setExecutionTimeLimitPercentage(float $value) {
    if ($value > 0.9 || $value < 0.3) {
      return;
    }

    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('execution_time_limit_percentage', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Limits - immediate response execution time limit.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default immediate response execution time limit, in seconds.
   *
   * @return float
   *   Returns the default immediate response execution time limit, in seconds.
   *
   * @see ::getExecutionTimeLimitPercentage()
   * @see ::getExecutionTimeLimitPercentageDefault()
   * @see ::setExecutionTimeLimitPercentage()
   * @see ::getResponseExecutionTimeLimit()
   * @see ::setResponseExecutionTimeLimit()
   */
  public static function getResponseExecutionTimeLimitDefault() {
    return 5.0;
  }

  /**
   * Returns the current immediate response execution time limit, in seconds.
   *
   * @return float
   *   Returns the immediate response execution time limit, in seconds.
   *
   * @see ::getExecutionTimeLimitPercentage()
   * @see ::getExecutionTimeLimitPercentageDefault()
   * @see ::setExecutionTimeLimitPercentage()
   * @see ::getResponseExecutionTimeLimitDefault()
   * @see ::setResponseExecutionTimeLimit()
   * @see \Drupal\foldershare\Utilities::getPhpExecutionTimeLimit()
   * @see \Drupal\foldershare\Utilities::getExecutionTimeLimit()
   * @see \Drupal\foldershare\Utilities::aboveExecutionTimeLimit()
   */
  public static function getResponseExecutionTimeLimit() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('response_execution_time_limit') === NULL) {
      return self::getResponseExecutionTimeLimitDefault();
    }

    return floatval($config->get('response_execution_time_limit'));
  }

  /**
   * Sets the current immediate response execution time limit, in seconds.
   *
   * The immediate response time limit is the maximum time an operation may
   * take while responding immediately to a request. Any work that takes
   * longer than this should be deferred into a scheduled task.
   *
   * This time limit, in seconds, is typically quite short so that a response
   * can be returned to the user quickly enough for the user to feel that
   * the website is responding properly. A good value is about 5 seconds.
   * Shorter values leave very little time for the operation to do any real
   * work before scheduling a task. Longer values over about 10 seconds
   * reduce the interactivity of the site and annoy users.
   *
   * This time limit differs from the PHP execution time limit, which is
   * usually 30 seconds or more and far too long to maintain an interactive
   * feel for the website. This time limit also differs from this module's
   * execution time limit percentage, which computes a maximum time based
   * on PHP's time limit. Such a maximum time is useful for background tasks,
   * but far too long for responding quickly to a user.
   *
   * @param float $value
   *   The immediate response execution time limit, in seconds. Values
   *   less than 1.0 second are ignored.
   *
   * @see ::getExecutionTimeLimitPercentage()
   * @see ::getExecutionTimeLimitPercentageDefault()
   * @see ::setExecutionTimeLimitPercentage()
   * @see ::getResponseExecutionTimeLimit()
   * @see ::getResponseExecutionTimeLimitDefault()
   * @see \Drupal\foldershare\Utilities::getPhpExecutionTimeLimit()
   * @see \Drupal\foldershare\Utilities::getExecutionTimeLimit()
   * @see \Drupal\foldershare\Utilities::aboveExecutionTimeLimit()
   */
  public static function setResponseExecutionTimeLimit(float $value) {
    if ($value < 1.0) {
      return;
    }

    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('response_execution_time_limit', $value);
    $config->save(TRUE);
  }

  /*---------------------------------------------------------------------
   *
   * Javascript polling interval.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the default status polling interval in seconds.
   *
   * @return float
   *   Returns the default status polling interval in seconds.
   *
   * @see ::getStatusPollingInterval()
   * @see ::setStatusPollingInterval()
   */
  public static function getStatusPollingIntervalDefault() {
    return 5.0;
  }

  /**
   * Returns the current status polling interval in seconds.
   *
   * @return float
   *   Returns the status polling interval in seconds.
   *
   * @see ::getStatusPollingIntervalDefault()
   * @see ::setStatusPollingInterval()
   */
  public static function getStatusPollingInterval() {
    $config = \Drupal::config(Constants::SETTINGS);
    if ($config->get('status_polling_interval') === NULL) {
      return self::getStatusPollingIntervalDefault();
    }

    return floatval($config->get('status_polling_interval'));
  }

  /**
   * Sets the current status polling interval in seconds.
   *
   * @param float $value
   *   The status polling interval in seconds.
   *
   * @see ::getStatusPollingInterval()
   * @see ::getStatusPollingIntervalDefault()
   */
  public static function setStatusPollingInterval(float $value) {
    if ($value < 3) {
      return;
    }

    $config = \Drupal::configFactory()->getEditable(Constants::SETTINGS);
    $config->set('status_polling_interval', $value);
    $config->save(TRUE);
  }

}
