<?php

namespace Drupal\foldershare\Form\AdminSettingsTraits;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

use Drupal\foldershare\Constants;
use Drupal\foldershare\Settings;
use Drupal\foldershare\Utilities;

/**
 * Systems the "System" tab for the module's settings form.
 *
 * <B>Warning:</B> This is an internal trait that is strictly used by
 * the AdminSettings form class. It is a mechanism to group functionality
 * to improve code management.
 *
 * @ingroup foldershare
 */
trait AdminSettingsSystemTab {

  /*---------------------------------------------------------------------
   *
   * Build.
   *
   *---------------------------------------------------------------------*/

  /**
   * Builds the form tab.
   *
   * Settings:
   * - Checkbox to enable logging.
   *
   * @param array $form
   *   An associative array containing the structure of the form. The form
   *   is modified to include additional render elements for the tab.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The current state of the form.
   * @param string $tabGroup
   *   The name of the tab group.
   * @param string $tabMachineName
   *   The untranslated name for the tab, used in CSS class names.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $tabTitle
   *   The translated name for the tab.
   */
  private function buildSystemTab(
    array &$form,
    FormStateInterface $formState,
    string $tabGroup,
    string $tabMachineName,
    TranslatableMarkup $tabTitle) {

    //
    // Setup
    // -----
    // Set up some variables.
    $mh = \Drupal::service('module_handler');

    $dblogInstalled = $mh->moduleExists('dblog');

    $moduleName = Constants::MODULE;

    $tabName = self::makeCssSafe($moduleName . '_' . $tabMachineName . '_tab');
    $cssBase = self::makeCssSafe($moduleName);

    $tabSubtitleClass        = $moduleName . '-settings-subtitle';
    $sectionTitleClass       = $moduleName . '-settings-section-title';
    $sectionClass            = $moduleName . '-settings-section';
    $sectionDescriptionClass = $moduleName . '-settings-section-description';
    $warningClass            = $moduleName . '-warning';

    //
    // Create the tab
    // --------------
    // Start the tab with a title, subtitle, and description.
    $form[$tabName] = [
      '#type'           => 'details',
      '#open'           => FALSE,
      '#group'          => $tabGroup,
      '#title'          => $tabTitle,
      '#description'    => [
        'subtitle'      => [
          '#type'       => 'html_tag',
          '#tag'        => 'h2',
          '#value'      => $this->t(
            'Manage system services.'),
          '#attributes' => [
            'class'     => [$tabSubtitleClass],
          ],
        ],
        'description'   => [
          '#type'       => 'html_tag',
          '#tag'        => 'p',
          '#value'      => $this->t(
            'File and folder activity may be logged, and the per-user usage report updated on a regular schedule.'),
          '#attributes' => [
            'class'     => [
              $cssBase . '-settings-description',
            ],
          ],
        ],
      ],
      '#attributes'     => [
        'class'         => [
          $cssBase . '-settings-tab ',
          $cssBase . '-fields-tab',
        ],
      ],
    ];

    //
    // Logging
    // -------
    // Add a checkbox for logging.
    $form[$tabName]['manage-logging'] = [
      '#type'            => 'item',
      '#markup'          => '<h3>' . $this->t('Logging') . '</h3>',
      '#attributes'      => [
        'class'          => [$sectionTitleClass],
      ],
      'section'          => [
        '#type'          => 'container',
        '#attributes'    => [
          'class'        => [$sectionClass],
        ],
        'description'      => [
          '#type'          => 'html_tag',
          '#tag'           => 'p',
          '#value'         => $this->t(
            "File and folder activity may be reported to the site's log, including each add, delete, move, copy, edit, and share."),
          '#attributes'    => [
            'class'        => [$sectionDescriptionClass],
          ],
        ],
        'enable-logging'   => [
          '#type'          => 'checkbox',
          '#title'         => $this->t('Enable activity logging'),
          '#default_value' => Settings::getActivityLogEnable(),
          '#return_value'  => 'enabled',
          '#required'      => FALSE,
          '#name'          => 'enable-logging',
          '#description'   => $this->t('For performance reasons, it is recommended that sites running on production environments do not enable logging.'),
        ],
      ],
    ];

    if ($dblogInstalled === FALSE) {
      $disabledWarning = [
        '#type'       => 'html_tag',
        '#tag'        => 'p',
        '#value'      => $this->t(
          'The @dblog is currently disabled. To see logged messages by this module, and others, you must enable the module.',
          [
            '@dblog' => Utilities::createRouteLink(
              'system.modules_list',
              'module-dblog',
              'Database logging module'),
          ]),
        '#attributes' => [
          'class'     => [$warningClass],
        ],
      ];

      $form[$tabName]['manage-logging']['section']['warning'] = $disabledWarning;
    }

    //
    // Usage table rebuilds
    // --------------------
    // Add a menu for the usage table rebuild interval.
    $options = [
      'manualy' => $this->t('Manually'),
      'hourly'  => $this->t('Hourly'),
      'daily'   => $this->t('Daily'),
      'weekly'  => $this->t('Weekly'),
    ];

    $usageLink = Utilities::createRouteLink(
      Constants::ROUTE_USAGE,
      '',
      $this->t('Usage report'));

    $form[$tabName]['manage-usage-table-rebuild-interval'] = [
      '#type'            => 'item',
      '#markup'          => '<h3>' . $this->t('Usage report') . '</h3>',
      '#attributes'      => [
        'class'          => [$sectionTitleClass],
      ],
      'section'          => [
        '#type'          => 'container',
        '#attributes'    => [
          'class'        => [$sectionClass],
        ],
        'description'      => [
          '#type'          => 'html_tag',
          '#tag'           => 'p',
          '#value'         => $this->t(
            "The @usageLink lists each user and shows the number of files and folders they own and the storage space they've used. This report may be updated manually or on a regular schedule.",
            [
              '@usageLink' => $usageLink,
            ]),
          '#attributes'    => [
            'class'        => [$sectionDescriptionClass],
          ],
        ],
        'usage-report-rebuild-interval' => [
          '#type'          => 'select',
          '#options'       => $options,
          '#title'         => $this->t('Rebuild interval'),
          '#default_value' => Settings::getUsageReportRebuildInterval(),
          '#required'      => TRUE,
          '#name'          => 'usage-report-rebuild-interval',
          '#description'   => $this->t('For busy sites, a shorter interval is recommended so that the usage report is always fairly uptodate.'),
        ],
      ],
    ];

    //
    // Related settings
    // ----------------
    // Add links to settings.
    $this->buildSystemRelatedSettings($form, $formState, $tabGroup, $tabName);
  }

  /**
   * Builds the related settings section of the tab.
   *
   * @param array $form
   *   An associative array containing the structure of the form. The form
   *   is modified to include additional render elements for the tab.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The current state of the form.
   * @param string $tabGroup
   *   The name of the tab group.
   * @param string $tabName
   *   The CSS-read tab name.
   */
  private function buildSystemRelatedSettings(
    array &$form,
    FormStateInterface $formState,
    string $tabGroup,
    string $tabName) {

    $moduleName        = Constants::MODULE;
    $sectionTitleClass = $moduleName . '-settings-section-title';
    $seeAlsoClass      = $moduleName . '-settings-seealso-section';
    $relatedClass      = $moduleName . '-related-links';
    $markup            = '';

    //
    // Create links.
    // -------------
    // For each of several modules, create a link to the module's configuration
    // page if the module is installed, and no link if it is not.
    //
    // Logging.
    $markup .= $this->buildRelatedSettings(
      '',
      'Logging',
      'system.logging_settings',
      [],
      "Configure the site's logging and error reporting",
      TRUE,
      []);

    // Database logging.
    $markup .= $this->buildRelatedSettings(
      'dblog',
      'Recent log messages',
      'dblog.overview',
      [],
      'Show recent log messages',
      TRUE,
      []);

    //
    // Add to form.
    // ------------
    // Add the links to the end of the form.
    //
    $form[$tabName]['related-settings'] = [
      '#type'            => 'details',
      '#title'           => $this->t('See also'),
      '#open'            => FALSE,
      '#summary_attributes' => [
        'class'          => [$sectionTitleClass],
      ],
      '#attributes'      => [
        'class'          => [$seeAlsoClass],
      ],
      'seealso'          => [
        '#type'          => 'item',
        '#markup'        => '<dl>' . $markup . '</dl>',
        '#attributes'    => [
          'class'        => [$relatedClass],
        ],
      ],
    ];
  }

  /*---------------------------------------------------------------------
   *
   * Validate.
   *
   *---------------------------------------------------------------------*/

  /**
   * Validates form values.
   *
   * @param array $form
   *   The form configuration.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The entered values for the form.
   */
  private function validateSystemTab(
    array &$form,
    FormStateInterface $formState) {

    // Nothing to do.
  }

  /*---------------------------------------------------------------------
   *
   * Submit.
   *
   *---------------------------------------------------------------------*/

  /**
   * Stores submitted form values.
   *
   * @param array $form
   *   The form configuration.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The entered values for the form.
   */
  private function submitSystemTab(
    array &$form,
    FormStateInterface $formState) {

    $value = $formState->getValue('enable-logging');
    $enabled = ($value === 'enabled') ? TRUE : FALSE;
    Settings::setActivityLogEnable($enabled);

    $value = $formState->getValue('usage-report-rebuild-interval');
    Settings::setUsageReportRebuildInterval($value);
  }

}
