<?php

namespace Drupal\foldershare\Entity;

use Drupal\Core\Database\Database;

use Drupal\foldershare\Settings;

/**
 * Manages per-user usage information for the module.
 *
 * This class manages the FolderShare usage table, which has one record for
 * each user that has used the module's features. Each record indicates:
 * - nFolders: the number of folders owned by the user.
 * - nFiles: the number of files owned by the user.
 * - nBytes: the total storage of all files owned by the user.
 *
 * Methods on this class build this table and return its values.
 *
 * The database table is created in MODULE.install when the module is
 * installed.
 *
 * @section access Access control
 * This class's methods do not do access control. The caller should restrict
 * access. Typically access is restricted to administrators.
 *
 * @ingroup foldershare
 *
 * @see \Drupal\foldershare\Entity\FolderShare
 */
final class FolderShareUsage {

  /*--------------------------------------------------------------------
   *
   * Database tables.
   *
   *-------------------------------------------------------------------*/

  /**
   * The name of the per-user usage tracking database table.
   *
   * This name must match the table defined in MODULE.install.
   */
  const USAGE_TABLE = 'foldershare_usage';

  /*--------------------------------------------------------------------
   *
   * Get totals.
   *
   *-------------------------------------------------------------------*/

  /**
   * Returns the total number of bytes.
   *
   * The returned value only includes storage space used for files.
   * Any storage space required in the database for folder or file
   * metadata is not included.
   *
   * @return int
   *   The total number of bytes.
   *
   * @see FolderShare::countNumberOfBytes()
   */
  public static function getNumberOfBytes() {
    return FolderShare::countNumberOfBytes();
  }

  /**
   * Returns the total number of folders.
   *
   * @return int
   *   The total number of folders.
   *
   * @see FolderShare::countNumberOfFolders()
   */
  public static function getNumberOfFolders() {
    return FolderShare::countNumberOfFolders();
  }

  /**
   * Returns the total number of files.
   *
   * @return int
   *   The total number of folders.
   *
   * @see FolderShare::countNumberOfFiles()
   */
  public static function getNumberOfFiles() {
    return FolderShare::countNumberOfFiles();
  }

  /*--------------------------------------------------------------------
   *
   * Get/Set usage.
   *
   *-------------------------------------------------------------------*/

  /**
   * Clears usage statistics for all users.
   */
  public static function clearAllUsage() {
    $connection = Database::getConnection();
    $connection->delete(self::USAGE_TABLE)
      ->execute();
  }

  /**
   * Clears usage statistics for a user.
   *
   * @param int $uid
   *   The user ID of the user whose usage is cleared.
   */
  public static function clearUsage(int $uid) {
    if ($uid < 0) {
      // Invalid UID.
      return;
    }

    $connection = Database::getConnection();
    $connection->delete(self::USAGE_TABLE)
      ->condition('uid', $uid)
      ->execute();
  }

  /**
   * Returns usage statistics of all users.
   *
   * The returned array has one entry for each user in the
   * database. Array keys are user IDs, and array values are associative
   * arrays with keys for specific metrics and values for those
   * metrics. Supported array keys are:
   *
   * - nFolders: the number of folders.
   * - nFiles: the number of files.
   * - nBytes: the total storage of all files.
   *
   * All metrics are for the total number of items or bytes owned by
   * the user.
   *
   * The returned values for bytes used is the current storage space use
   * for each user. This value does not include any database storage space
   * required for file and folder metadata.
   *
   * The returned array only contains records for those users that
   * have current usage . Users who have no recorded metrics
   * will not be listed in the returned array.
   *
   * @return array
   *   An array with user ID array keys. Each array value is an
   *   associative array with keys for each of the above usage.
   */
  public static function getAllUsage() {
    // Query the usage table for all entries.
    $connection = Database::getConnection();
    $select = $connection->select(self::USAGE_TABLE, 'u');
    $select->addField('u', 'uid', 'uid');
    $select->addField('u', 'nFolders', 'nFolders');
    $select->addField('u', 'nFiles', 'nFiles');
    $select->addField('u', 'nBytes', 'nBytes');
    $records = $select->execute()->fetchAll();

    // Build and return an array from the records.  Array keys
    // are user IDs, while values are usage info.
    $usage = [];
    foreach ($records as $record) {
      $usage[$record->uid] = [
        'nFolders' => $record->nFolders,
        'nFiles'   => $record->nFiles,
        'nBytes'   => $record->nBytes,
      ];
    }

    return $usage;
  }

  /**
   * Returns usage statistics for a user.
   *
   * The returned associative array has keys for specific metrics,
   * and values for those metrics. Supported array keys are:
   *
   * - nFolders: the number of folders.
   * - nFiles: the number of files.
   * - nBytes: the total storage of all files.
   *
   * All metrics are for the total number of items or bytes owned by
   * the user.
   *
   * The returned value for bytes used is the current storage space use
   * for the user. This value does not include any database storage space
   * required for file and folder metadata.
   *
   * If there is no recorded usage information for the user, an
   * array is returned with all metric values zero.
   *
   * @param int $uid
   *   The user ID of the user whose usage is to be returned.
   *
   * @return array
   *   An associative array is returned that includes keys for each
   *   of the above usage.
   */
  public static function getUsage(int $uid) {
    if ($uid < 0) {
      // Invalid UID.
      return [
        'nFolders'     => 0,
        'nFiles'       => 0,
        'nBytes'       => 0,
      ];
    }

    // Query the usage table for an entry for this user.
    // There could be none, or one, but not multiple entries.
    $connection = Database::getConnection();
    $select = $connection->select(self::USAGE_TABLE, 'u');
    $select->addField('u', 'uid', 'uid');
    $select->addField('u', 'nFolders', 'nFolders');
    $select->addField('u', 'nFiles', 'nFiles');
    $select->addField('u', 'nBytes', 'nBytes');
    $select->condition('u.uid', $uid, '=');
    $records = $select->execute()->fetchAll();

    // If none, return an empty usage array.
    if (count($records) === 0) {
      return [
        'nFolders'     => 0,
        'nFiles'       => 0,
        'nBytes'       => 0,
      ];
    }

    // Otherwise return the usage.
    $record = array_shift($records);
    return [
      'nFolders'     => $record->nFolders,
      'nFiles'       => $record->nFiles,
      'nBytes'       => $record->nBytes,
    ];
  }

  /**
   * Returns the storage used for a user.
   *
   * The returned value is the current storage space use for the user.
   * This value does not include any database storage space required
   * for file and folder metadata.
   *
   * This is a convenience function that just returns the 'nBytes'
   * value from the user's usage.
   *
   * @param int $uid
   *   The user ID of the user whose usage is to be returned.
   *
   * @return int
   *   The total storage space used for files owned by the user.
   *
   * @see ::getUsage()
   */
  public static function getUsageBytes(int $uid) {
    $usage = self::getUsage($uid);
    return $usage['nBytes'];
  }

  /*---------------------------------------------------------------------
   *
   * Rebuild locks.
   *
   *---------------------------------------------------------------------*/

  /**
   * Acquires a lock for rebuilding the usage table.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * @return bool
   *   Returns TRUE if a lock on this item was acquired, and FALSE otherwise.
   *
   * @see ::releaseUsageLock()
   */
  private static function acquireUsageLock() {
    return \Drupal::lock()->acquire(
      'foldershareusage_lock',
      30);
  }

  /**
   * Releases a lock for rebuilding the usage table.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * @see ::acquireUsageLock()
   */
  private static function releaseUsageLock() {
    \Drupal::lock()->release(
      'foldershareusage_lock');
  }

  /*---------------------------------------------------------------------
   *
   * Rebuild.
   *
   *---------------------------------------------------------------------*/

  /**
   * Immediately rebuilds usage information for all users.
   *
   * All current usage information is deleted and a new set assembled
   * and saved for all users at the site. Users that have no files or
   * folders are not included.
   *
   * @return bool
   *   Returns TRUE on success and FALSE on failure. FALSE is only returned
   *   if another process has the table locked because it is rebuilding
   *   the table.
   *
   * @section locking Process locks
   * This method uses a process lock to insure that the usage table is
   * rebuilt by only one process at a time. If the table is found to be
   * locked, this method returns immediately without updating the table.
   */
  public static function rebuildAllUsage() {
    // LOCK DURING REBUILD.
    if (self::acquireUsageLock() === FALSE) {
      // Another rebuild is already in progress. Abort.
      return FALSE;
    }

    //
    // Clear usage.
    // ------------
    // Empty the usage table to start with.
    $connection = Database::getConnection();
    $connection->delete(self::USAGE_TABLE)->execute();

    //
    // Rebuild usage.
    // --------------
    // For each user, count the number of files, folders, and bytes, then
    // update the usage table.
    $userIds = \Drupal::entityQuery('user')->execute();
    foreach ($userIds as $uid) {
      // Get counts for the user.
      $nFolders = FolderShare::countNumberOfFolders($uid);
      $nFiles   = FolderShare::countNumberOfFiles($uid);
      $nBytes   = FolderShare::countNumberOfBytes($uid);

      // Add a new entry.
      $query = $connection->insert(self::USAGE_TABLE);
      $query->fields(
        [
          'uid'      => $uid,
          'nFolders' => $nFolders,
          'nFiles'   => $nFiles,
          'nBytes'   => $nBytes,
        ]);
      $query->execute();
    }

    //
    // Update date.
    // ------------
    // Update the stored date.
    Settings::setUsageReportTime('@' . (string) time());

    // UNLOCK.
    self::releaseUsageLock();
    return TRUE;
  }

  /**
   * Updates how rebuilds are scheduled when the interval changes.
   *
   * The given rebuild interval is used to guide queuing of automatic
   * usage table rebuilds. Legal values are:
   *
   * - 'manual' = the table is only rebuilt manually.
   * - 'hourly' = the table is rebuild hourly.
   * - 'daily' = the table is rebuild daily.
   * - 'weekly' = the table is rebuilt weekly.
   *
   * @param string $rebuildInterval
   *   The new rebuild interval.
   *
   * @see Settings::getUsageReportRebuildInterval()
   * @see Settings::setUsageReportRebuildInterval()
   */
  public static function updateRebuildInterval(string $rebuildInterval) {
    if ($rebuildInterval === 'manual') {
      // When the interval is manual, there is no scheduled rebuild task.
      // Delete any rebuild tasks that might exist.
      FolderShareScheduledTask::deleteTasks('rebuildusage');
      return;
    }

    // When the interval is not 'manual', there should be a scheduled rebuild
    // task. If there is already a task, delete it so that we can reset
    // the scheduled run time.
    FolderShareScheduledTask::deleteTasks('rebuildusage');

    $timestamp = time();
    switch ($rebuildInterval) {
      default:
      case 'hourly':
        // 60 minutes/hour * 60 seconds/minute.
        $timestamp += (60 * 60);
        break;

      case 'daily':
        // 24 hours/day * 60 minutes/hour * 60 seconds/minute.
        $timestamp += (24 * 60 * 60);
        break;

      case 'weekly':
        // 7 days/week * 24 hours/day * 60 minutes/hour * 60 seconds/minute.
        $timestamp += (7 * 24 * 60 * 60);
        break;
    }

    FolderShareScheduledTask::createTask(
      $timestamp,
      'rebuildusage',
      (int) \Drupal::currentUser()->id(),
      NULL,
      time(),
      "$rebuildInterval rebuild",
      0);
  }

  /*---------------------------------------------------------------------
   *
   * Queue task.
   *
   *---------------------------------------------------------------------*/

  /**
   * Processes a usage rebuild task from the scheduled task queue.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B> This method is public so that it can be called
   * from the module's scheduled task handler.
   *
   * A rebuild task for the usage table causes the table to be cleared
   * and entries added one at a time for each user. Each entry gives the
   * number of files and folders owned by the user, and the storage space
   * required.
   *
   * There is one condition under which usage rebuilding may not complete fully:
   * - The database is too large to query before hitting a timeout.
   *
   * Database queries issued here count the number of matching entries or
   * sum entry sizes. Most of the work, then, is done in the database and
   * this work does not count against the PHP runtime. It is very unlikely
   * that a PHP or web server timeout will occur and interrupt this task.
   *
   * @param int $requester
   *   The user ID of the user that requested the rebuild. This is ignored.
   * @param array $parameters
   *   The queued task's parameters. This is ignored.
   * @param int $started
   *   The timestamp of the start date & time for an operation that causes
   *   a chain of tasks.
   * @param string $comments
   *   A comment on the current task.
   * @param int $executionTime
   *   The accumulated total execution time of the task chain, in seconds.
   */
  public static function processTaskRebuildUsage(
    int $requester,
    array $parameters,
    int $started,
    string $comments,
    int $executionTime) {

    //
    // Validate.
    // ---------
    // If the rebuild interval is now "manual", then no automatic rebuild
    // is requested any more. Exit immediately without rebuilding and without
    // requeueing a new task.
    $rebuildInterval = Settings::getUsageReportRebuildInterval();
    if ($rebuildInterval === 'manual') {
      return;
    }

    //
    // Requeue.
    // --------
    // Requeue first, before rebuilding the usage table. If there is a
    // PHP timeout during the rebuild, this insures that there is already
    // a next rebuild scheduled.
    $timestamp = time();
    switch ($rebuildInterval) {
      default:
      case 'hourly':
        // 60 minutes/hour * 60 seconds/minute.
        $timestamp += (60 * 60);
        break;

      case 'daily':
        // 24 hours/day * 60 minutes/hour * 60 seconds/minute.
        $timestamp += (24 * 60 * 60);
        break;

      case 'weekly':
        // 7 days/week * 24 hours/day * 60 minutes/hour * 60 seconds/minute.
        $timestamp += (7 * 24 * 60 * 60);
        break;
    }

    FolderShareScheduledTask::createTask(
      $timestamp,
      'rebuildusage',
      $requester,
      NULL,
      time(),
      "$rebuildInterval rebuild",
      0);

    //
    // Rebuild.
    // --------
    // Rebuild the usage table.
    //
    // It is possible for this task to have been called at nearly the same
    // time by multiple processes. It is also possible that the site admin
    // triggered a manual rebuild. To handle these collisions, rebuilds
    // use process locks. If another process has the rebuild locked, then
    // this rebuild silently aborts.
    self::rebuildAllUsage();
  }

}
