<?php

namespace Drupal\foldershare\Entity\FolderShareTraits;

use Drupal\Core\Database\Database;

use Drupal\foldershare\Utilities;
use Drupal\foldershare\Entity\FolderShareScheduledTask;

/**
 * File system check all Foldershare entities.
 *
 * This trait includes methods to check the file system and repair it.
 *
 * @section internal Internal trait
 * This trait is internal to the FolderShare module and used to define
 * features of the FolderShare entity class. It is a mechanism to group
 * functionality to improve code management.
 *
 * @ingroup foldershare
 */
trait OperationFsckTrait {

  /*---------------------------------------------------------------------
   *
   * File system check.
   *
   *---------------------------------------------------------------------*/

  /**
   * Runs a file system check to find and fix problems.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * <B>It is strongly recommended that the website be in maintenance
   * mode when this function is called. This prevents file system activity
   * that can collide with this function and create unexpected and
   * possibly destructive results.</B>
   *
   * This method scans the file system looking for problems. These primarily
   * include parent and root linkage problems, such as parent or root IDs
   * that do not refer to existing items, or where the parent and root IDs
   * are not consistent with each other. This method also checks for hidden
   * and disabled items that do not appear to have pending tasks that justify
   * them.
   *
   * This method clears all memory caches before it starts. If $fix is TRUE,
   * it also clears the FolderShare entity caches before any checks, and
   * clears ALL Drupal caches upon completion.
   *
   * @param bool $fix
   *   (optional, default = FALSE) When TRUE, fix file system problems
   *   automatically. When FALSE, just report them to the log.
   *
   * @return bool
   *   Returns TRUE if anything in the file system was fixed or would
   *   have been fixed, and FALSE if no changes are needed.
   *
   * @section logging Operation log
   * A warning message is posted to the log that indicates the number of
   * items in need of repair (when $fix is FALSE) or that have been repaired
   * (when $fix is TRUE). The individual items repaired are not logged.
   *
   * @see ::clearAllSystemHidden()
   * @see ::clearAllSystemDisabled()
   *
   * @todo Recurse through every folder tree and check that all children
   * share the same root ID.
   */
  public static function fsck(bool $fix = FALSE) {
    //
    // Prepare.
    // --------
    // Try to push the PHP timeout to "unlimited" so that a long operation
    // has a better chance to complete. This may not work if the site has
    // PHP timeout changes blocked and it does nothing to stop web server
    // timeouts.
    //
    // However, fsck() is primarily intended for execution from the
    // command line, which has no such timeouts.
    drupal_set_time_limit(0);

    // Flush everything from static (memory) caches to remove the vestiges
    // of whatever operations occurred before this task began.
    Utilities::flushAllMemoryCaches();

    if ($fix === TRUE) {
      // Flush all FolderShare entities from caches everywhere. Below we
      // modify the database directly. Once changed, all cached copies are
      // out of date.
      Utilities::flushAllEntityCaches(self::ENTITY_TYPE_ID);
    }

    $fixed = FALSE;

    //
    // Fix parent/root linkage.
    // ------------------------
    // The parent ID and root ID have the following possibilities:
    //
    // |#|parentid|rootid  |Validity|Fix by...                               |
    // |-|--------|--------|--------|----------------------------------------|
    // |1|NULL    |NULL    |good    |                                        |
    // |2|NULL    |valid   |bad     |Set parentid = rootid                   |
    // |3|NULL    |invalid |bad     |Set parentid = rootid = NULL            |
    // |4|valid   |NULL    |bad     |Set parentid = rootid = NULL            |
    // |5|valid   |valid   |good    |                                        |
    // |6|valid   |invalid |bad     |Set parentid = rootid = NULL            |
    // |7|invalid |NULL    |bad     |Set parentid = rootid = NULL            |
    // |8|invalid |valid   |bad     |Set parentid = rootid                   |
    // |9|invalid |invalid |bad     |Set parentid = rootid = NULL            |
    //
    // In choosing fixes, we have to watch for circular fixes that leave
    // content still invalid. For instance, when the root ID is NULL but
    // the parent ID is valid, we could set the root ID to the parent's
    // root ID. Except we don't know if that root ID is valid. If not,
    // we're just propagating a bad root ID into another item. And if we
    // then update descendants with that bad root ID, we're making a mess.
    //
    // So, we choose to prioritize the root ID over the parent ID. If the
    // root ID is NULL, the parent ID is forced to NULL without checking
    // if it is or is not valid.
    //
    // Cases (1) and (5) do not require any fixes.
    //
    // Cases (4) & (7). NULL rootid and non-NULL parentid.
    // - Set parentid to NULL.
    // - Cases (4) & (7) become case (1).
    $status = self::fsckFixNullRootNonNullParent($fix);
    $fixed = $fixed || $status;

    // Cases (3), (6) & (9). Non-NULL invalid rootid.
    // - Set parentid and rootid to NULL and update descendants.
    // - Cases (3), (6) & (9) become case (1).
    $status = self::fsckFixInvalidRoot($fix);
    $fixed = $fixed || $status;

    // Case (2). Non-NULL valid rootid and NULL parentid.
    // - Set parentid to rootid.
    // - Case (2) becomes case (5).
    $status = self::fsckFixValidRootNullParent($fix);
    $fixed = $fixed || $status;

    // Case (8). Non-NULL valid rootid and non-NULL invalid parentid.
    // - Set parentid to rootid.
    // - Case (8) becomes case (5).
    $status = self::fsckFixValidRootInvalidParent($fix);
    $fixed = $fixed || $status;

    //
    // Fix root consistency.
    // ---------------------
    // The root ID for every descendant of a root item must be that root
    // item's ID.
    $status = self::fsckFixConsistentRoot($fix);
    $fixed = $fixed || $status;

    //
    // Fix hidden & disabled.
    // ----------------------
    // Items should not be marked hidden or disabled if there are no
    // associated tasks pending.
    $status = self::fsckFixHidden($fix);
    $fixed = $fixed || $status;

    $status = self::fsckFixDisabled($fix);
    $fixed = $fixed || $status;

    //
    // Finish.
    // -------
    // Flush ALL caches everywhere.
    //
    // The activities above may have changed any number of FolderShare
    // entities. Because they do direct database operations, it is not
    // easy to track exactly which items have changed. If we did, it could
    // be a huge array of entity IDs that would be awkward to create and
    // use within PHP's memory limits.
    //
    // Without tracking every entity changed, it is not possibly to
    // selectively invalidate cache entries relating just to those entities.
    // Further, Drupal does not provide any way to flush all caches of all
    // entries for entities of this entity type. For instance, there is no
    // way to flush out of the render cache everything for all FolderShare
    // entities. The only way to flush out of any cache is to provide an
    // explicit list of entity IDs to flush.
    //
    // The only safe way to be sure we have flushed all out of date cache
    // entries from every cache is to flush everything.
    if ($fix === TRUE && $fixed === TRUE) {
      drupal_flush_all_caches();
    }

    return $fixed;
  }

  /**
   * Fixes items with a NULL root ID and non-NULL parent ID.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * <B>Problem</B>
   * If a root ID is NULL, the item has no parent and is supposed to be
   * a root item. In that case its parent ID should be NULL too. If this
   * is not the case, the item has become corrupted.
   *
   * <B>Fix</B>
   * Leave the root ID as NULL and set the parent ID to NULL too. There is
   * no need to update descendants.
   *
   * @param bool $fix
   *   (optional, default = TRUE) When TRUE, fix file system problems
   *   automatically. When FALSE, just report them to the log.
   *
   * @return bool
   *   Returns TRUE if anything in the file system was fixed or would
   *   have been fixed, and FALSE if no changes are needed.
   *
   * @section logging Operation log
   * A warning message is posted to the log that indicates the number of
   * items in need of repair (when $fix is FALSE) or that have been repaired
   * (when $fix is TRUE). The individual items repaired are not logged.
   */
  private static function fsckFixNullRootNonNullParent(bool $fix = FALSE) {
    $connection = Database::getConnection();

    //
    // Count problems.
    // ---------------
    // Issue a query to find items where there is a NULL root ID, and
    // a non-NULL parent ID.
    $select = $connection->select(self::BASE_TABLE, 'fs');
    $select->addField('fs', 'id', 'id');
    $select->isNull('rootid');
    $select->isNotNull('parentid');
    $ids = $select->execute()->fetchCol(0);
    $count = count($ids);

    if ($count === 0) {
      return FALSE;
    }

    if ($fix === FALSE) {
      \Drupal::logger('FolderShare:  File system check')->error(
        "Found @count invalid items with NULL root but non-NULL parent. \nThese items are incorrectly connected into the folder tree. This will cause them to be listed within a folder when they should be listed in a root list. Repair is needed. \nEntity IDs: @ids\n",
        [
          '@count' => $count,
          '@ids'   => implode(', ', $ids),
        ]);
      return TRUE;
    }

    //
    // Fix problems.
    // -------------
    // Issue an update to find all items where the root ID is NULL, and
    // the parent ID is not NULL. Set the parent ID to NULL.
    $update = $connection->update(self::BASE_TABLE);
    $update->isNull('rootid');
    $update->isNotNull('parentid');
    $update->fields([
      'parentid' => NULL,
    ]);
    $count = $update->execute();

    if ($count === 0) {
      return FALSE;
    }

    \Drupal::logger('FolderShare: File system fix')->warning(
      "Corrected @count items with NULL root but non-NULL parent. \nEach invalid item's parent has been set to NULL to be consistent with its NULL root. This corrects their folder tree connections and insures that they are listed properly in a root list. \nEntity IDs: @ids\n",
      [
        '@count' => $count,
        '@ids'   => implode(', ', $ids),
      ]);

    return TRUE;
  }

  /**
   * Fixes items with an invalid root ID.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * <B>Problem</B>
   * If the root ID refers to a non-existent entity, it is invalid and
   * the item has become corrupted.
   *
   * <B>Fix</B>
   * Set the root ID and parent ID to NULL. Update descendants.
   *
   * @param bool $fix
   *   (optional, default = TRUE) When TRUE, fix file system problems
   *   automatically. When FALSE, just report them to the log.
   *
   * @return bool
   *   Returns TRUE if anything in the file system was fixed or would
   *   have been fixed, and FALSE if no changes are needed.
   *
   * @section logging Operation log
   * A warning message is posted to the log that indicates the number of
   * items in need of repair (when $fix is FALSE) or that have been repaired
   * (when $fix is TRUE). The individual items repaired are not logged.
   */
  private static function fsckFixInvalidRoot(bool $fix = FALSE) {
    $connection = Database::getConnection();

    //
    // Count problems.
    // ---------------
    // Issue a query to find cases with a non-NULL rootid that does not
    // match any other entity.
    $subquery = $connection->select(self::BASE_TABLE, 'subfs');
    $subquery->addField('subfs', 'id', 'subfsid');
    $subquery->where('fs.rootid = subfs.id');

    $select = $connection->select(self::BASE_TABLE, 'fs');
    $select->addField('fs', 'id', 'id');
    $select->isNotNull('rootid');
    $select->notExists($subquery);

    $ids = $select->execute()->fetchCol(0);
    $count = count($ids);

    if ($count === 0) {
      return FALSE;
    }

    if ($fix === FALSE) {
      \Drupal::logger('FolderShare:  File system check')->error(
        "Found @count invalid items with non-existent roots. \nThese items are incorrectly connected into the folder tree. They are listed as children of a parent folder, but they indicate roots that do not exist. This will cause attempts to view or change these items to fail with error messages and page crashes because the root is bad and access grants are not available. Repair is needed. \nEntity IDs: @ids\n",
        [
          '@count' => $count,
          '@ids'   => implode(', ', $ids),
        ]);
      return TRUE;
    }

    //
    // Fix problems.
    // -------------
    // Issue an update to set all of the found items to a NULL parent ID
    // and root ID, which makes them roots.
    //
    // Then use the first query's ID list to loop to update all
    // descendants to set their root ID to the item.
    $subquery = $connection->select(self::BASE_TABLE, 'subfs');
    $subquery->addField('subfs', 'id', 'subfsid');
    $subquery->where('{' . self::BASE_TABLE . '}.rootid = subfs.id');

    $update = $connection->update(self::BASE_TABLE);
    $update->isNotNull('rootid');
    $update->notExists($subquery);
    $update->fields([
      'rootid'   => NULL,
      'parentid' => NULL,
    ]);
    $update->execute();

    foreach ($ids as $id) {
      // Set the root of all descendants to be the item itself.
      self::setDescendantsRootItemId($id, $id);
    }

    \Drupal::logger('FolderShare: File system fix')->warning(
      "Corrected @count items with non-existent root. \nEach item's parent and root have been set to NULL. This reconnects them into the folder tree so that they are listed in a root list. All of their descendants also have been updated to use the corrected item as their root. \nEntity IDs (root-level only): @ids\n",
      [
        '@count' => $count,
        '@ids'   => implode(', ', $ids),
      ]);

    return TRUE;
  }

  /**
   * Fixes items with valid root ID and NULL parent ID.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * <B>Problem</B>
   * If a root ID is not NULL and valid, the item is a root and it is
   * supposed to have a NULL parent. If the parent is not NULL, the item
   * has become corrupted.
   *
   * <B>Fix</B>
   * Set the parent ID to the root ID. There is no need to update descendants.
   *
   * @param bool $fix
   *   (optional, default = TRUE) When TRUE, fix and log file system problems
   *   automatically. When FALSE, just report them to the log.
   *
   * @return bool
   *   Returns TRUE if anything in the file system was fixed or would
   *   have been fixed, and FALSE if no changes are needed.
   *
   * @section logging Operation log
   * A warning message is posted to the log that indicates the number of
   * items in need of repair (when $fix is FALSE) or that have been repaired
   * (when $fix is TRUE). The individual items repaired are not logged.
   */
  private static function fsckFixValidRootNullParent(bool $fix = FALSE) {
    $connection = Database::getConnection();

    //
    // Count problems.
    // ---------------
    // Issue a query to find items with a NULL parent ID, and a
    // non-NULL root ID.
    $select = $connection->select(self::BASE_TABLE, 'fs');
    $select->addField('fs', 'id', 'id');
    $select->isNotNull('rootid');
    $select->isNull('parentid');
    $ids = $select->execute()->fetchCol(0);
    $count = count($ids);

    if ($count === 0) {
      return FALSE;
    }

    if ($fix === FALSE) {
      \Drupal::logger('FolderShare:  File system check')->error(
        "Found @count invalid items with valid root but NULL parent. \nThese items are incorrectly connected into the folder tree. This will cause them to be listed in a root list, but use access grants as if they are children of a folder in another root list. Repair is needed. \nEntity IDs: @ids\n",
        [
          '@count' => $count,
          '@ids'   => implode(', ', $ids),
        ]);
      return TRUE;
    }

    //
    // Fix problems.
    // -------------
    // Issue an update to find all items where the root ID is not NULL,
    // but the parent ID is NULL. Set the parent ID to the root ID.
    $update = $connection->update(self::BASE_TABLE);
    $update->isNotNull('rootid');
    $update->isNull('parentid');
    $update->expression('parentid', 'rootid');
    $count = $update->execute();

    if ($count === 0) {
      return FALSE;
    }

    \Drupal::logger('FolderShare: File system fix')->warning(
      "Corrected @count items with valid root but NULL parent. \nEach item's parent has been set to it's root. This will retain their access grants and cause them to be correctly listed as children of the parent instead of incorrectly showing them in a root list. \nEntity IDs: @ids\n",
      [
        '@count' => $count,
        '@ids'   => implode(', ', $ids),
      ]);

    return TRUE;
  }

  /**
   * Fixes items with valid root IDs and invalid parent IDs.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * <B>Problem</B>
   * If the root ID is valid, but the parent ID refers to a non-existent
   * entity, it is invalid and the item has become corrupted.
   *
   * <B>Fix</B>
   * Set the parent ID to the root ID. There is no need to update descendants.
   *
   * @param bool $fix
   *   (optional, default = TRUE) When TRUE, fix and log file system problems
   *   automatically. When FALSE, just report them to the log.
   *
   * @return bool
   *   Returns TRUE if anything in the file system was fixed or would
   *   have been fixed, and FALSE if no changes are needed.
   *
   * @section logging Operation log
   * A warning message is posted to the log that indicates the number of
   * items in need of repair (when $fix is FALSE) or that have been repaired
   * (when $fix is TRUE). The individual items repaired are not logged.
   */
  private static function fsckFixValidRootInvalidParent(bool $fix = FALSE) {
    $connection = Database::getConnection();

    //
    // Count problems.
    // ---------------
    // Issue a query to find items with an invalid parent ID.
    $subquery = $connection->select(self::BASE_TABLE, 'subfs');
    $subquery->addField('subfs', 'id', 'subfsid');
    $subquery->where('fs.parentid = subfs.id');

    $select = $connection->select(self::BASE_TABLE, 'fs');
    $select->addField('fs', 'id', 'id');
    $select->isNotNull('parentid');
    $select->isNotNull('rootid');
    $select->notExists($subquery);

    $ids = $select->execute()->fetchCol(0);
    $count = count($ids);

    if ($count === 0) {
      return FALSE;
    }

    if ($fix === FALSE) {
      \Drupal::logger('FolderShare:  File system check')->error(
        "Found @count invalid items with non-existent parents. \nThese items are incorrectly connected into the folder tree. They are inaccessible because they are marked as children of a non-existent parent folder. Repair is needed. \nEntity IDs: @ids\n",
        [
          '@count' => $count,
          '@ids'   => implode(', ', $ids),
        ]);
      return TRUE;
    }

    //
    // Fix problems.
    // -------------
    // Issue an update to find all invalid parent IDs and set them to
    // the root ID.
    $subquery = $connection->select(self::BASE_TABLE, 'subfs');
    $subquery->addField('subfs', 'id', 'subfsid');
    $subquery->where('{' . self::BASE_TABLE . '}.parentid = subfs.id');

    $update = $connection->update(self::BASE_TABLE);
    $update->isNotNull('parentid');
    $update->isNotNull('rootid');
    $update->notExists($subquery);
    $update->expression('parentid', 'rootid');
    $count = $update->execute();

    if ($count === 0) {
      return FALSE;
    }

    \Drupal::logger('FolderShare: File system fix')->warning(
      "Corrected @count items with invalid parents. \nEach item's parent has been set to it's root. This reconnects them into the folder tree so that they are listed in owner root lists. \nEntity IDs: @ids\n",
      [
        '@count' => $count,
        '@ids'   => implode(', ', $ids),
      ]);

    return TRUE;
  }

  /**
   * Fixes items with incorrect root IDs.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * <B>Problem</B>
   * If an item is a descendant of a root item, but its root ID is not set
   * to that root item's ID, then it is incorrect and the item has become
   * corrupted.
   *
   * <B>Fix</B>
   * Set the root ID to the highest ancestor's ID.
   *
   * @param bool $fix
   *   (optional, default = TRUE) When TRUE, fix and log file system problems
   *   automatically. When FALSE, just report them to the log.
   *
   * @return bool
   *   Returns TRUE if anything in the file system was fixed or would
   *   have been fixed, and FALSE if no changes are needed.
   *
   * @section logging Operation log
   * A warning message is posted to the log that indicates the number of
   * items in need of repair (when $fix is FALSE) or that have been repaired
   * (when $fix is TRUE). The individual items repaired are not logged.
   */
  private static function fsckFixConsistentRoot(bool $fix = FALSE) {
    $connection = Database::getConnection();

    //
    // Get all root IDs.
    // -----------------
    // Get a list of all root IDs owned by anybody. We'll be looping
    // over these. Get IDs for any user and any name, and include all
    // disabled AND hidden items.
    $rootIds = self::findAllRootItemIds(
      self::ANY_USER_ID,
      '',
      TRUE,
      TRUE);

    //
    // Count problems.
    // ---------------
    // Iteratively recurse down through each root folder tree and count
    // the number of descendants that have the wrong root ID.
    if ($fix === FALSE) {
      $count = 0;
      foreach ($rootIds as $rootId) {
        $rootCount = 0;
        $pending = [$rootId];

        while (empty($pending) === FALSE) {
          // Get the next pending ID.
          $id = (int) array_shift($pending);

          // Count all children that have the wrong root ID.
          $select = $connection->select(self::BASE_TABLE, 'fs');
          $select->condition('parentid', $id, '=');
          $select->condition('rootid', $rootId, '<>');
          $rootCount += $select->countQuery()->execute()->fetchField();

          // Get folder children and add them to the pending list.
          $select = $connection->select(self::BASE_TABLE, 'fs');
          $select->addField('fs', 'id', 'id');
          $select->condition('parentid', $id, '=');
          $select->condition('kind', self::FOLDER_KIND, '=');

          $pending = array_merge($pending, $select->execute()->fetchCol(0));
        }

        $count += $rootCount;
      }

      if ($count === 0) {
        return FALSE;
      }

      \Drupal::logger('FolderShare:  File system check')->error(
        "Found @count invalid descendant items with incorrect root. \nThese items are incorrectly connected into the folder tree. They are children of a parent folder, but their root is not the same as the parent's. This will cause incorrect access grants. Repair is needed.\n",
        [
          '@count' => $count,
        ]);
      return TRUE;
    }

    //
    // Fix problems.
    // -------------
    // Issue an update to find all invalid parent IDs and set them to
    // the root ID.
    foreach ($rootIds as $rootId) {
      $pending = [$rootId];

      while (empty($pending) === FALSE) {
        // Get the next pending ID.
        $id = (int) array_shift($pending);

        // Update all children that have the wrong root ID.
        $update = $connection->update(self::BASE_TABLE);
        $update->condition('parentid', $id, '=');
        $update->condition('rootid', $rootId, '<>');
        $update->fields([
          'rootid' => $rootId,
        ]);
        $count += $update->execute();

        // Get folder children and add them to the pending list.
        $select = $connection->select(self::BASE_TABLE, 'fs');
        $select->addField('fs', 'id', 'id');
        $select->condition('parentid', $id, '=');
        $select->condition('kind', self::FOLDER_KIND, '=');

        $pending = array_merge($pending, $select->execute()->fetchCol(0));
      }
    }

    if ($count === 0) {
      return FALSE;
    }

    \Drupal::logger('FolderShare: File system fix')->warning(
      "Corrected @count descendant items with incorrect root. \nEach item's root has been set to the highest ancestor's ID. This does not move items in the folder tree. It just corrects their access grants to be based on the correct root.\n",
      [
        '@count' => $count,
      ]);

    return TRUE;
  }

  /**
   * Fixes hidden items.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * <B>Problem</B>
   * If an item is marked as hidden, it is in the process of being deleted.
   * But if there are no delete tasks pending, there is no reason for an
   * item to be hidden. A prior delete may have crashed.
   *
   * <B>Fix</B>
   * Set the systemhidden flag to FALSE to show it again.
   *
   * @param bool $fix
   *   (optional, default = TRUE) When TRUE, fix and log file system problems
   *   automatically. When FALSE, just report them to the log.
   *
   * @return bool
   *   Returns TRUE if anything in the file system was fixed or would
   *   have been fixed, and FALSE if no changes are needed.
   *
   * @section logging Operation log
   * A warning message is posted to the log that indicates the number of
   * items in need of repair (when $fix is FALSE) or that have been repaired
   * (when $fix is TRUE). The individual items repaired are not logged.
   */
  private static function fsckFixHidden(bool $fix = FALSE) {

    if ($fix === FALSE) {
      //
      // Find hidden & tasks.
      // --------------------
      // Find all hidden items and then look for pending delete tasks.
      // If there are hidden items, but no delete tasks, there may be
      // a problem.
      $ids = self::findAllHiddenIds();
      $count = count($ids);

      if ($count === 0) {
        return FALSE;
      }

      // Get the number of phase 1 and phase 2 delete tasks.
      $nTasks =
        FolderShareScheduledTask::findNumberOfTasks('delete-hide') +
        FolderShareScheduledTask::findNumberOfTasks('delete-delete');

      if ($nTasks === 0) {
        \Drupal::logger('FolderShare:  File system check')->error(
          "Found @count hidden items but no pending delete tasks. \nThese items are hidden from view and inaccessible. This normally means that they are about to be deleted, but there are no delete tasks pending. Repair is needed. \nEntity IDs: @ids\n",
          [
            '@count' => $count,
            '@ids'   => implode(', ', $ids),
          ]);
        return TRUE;
      }

      \Drupal::logger('FolderShare:  File system check')->warning(
        "Found @count hidden items and @taskCount pending delete tasks. \nHidden items are normal while a delete task prepares to delete them. The pending tasks should be allowed to complete. Repair is probably not needed. \nEntity IDs: @ids\n",
        [
          '@count'     => $count,
          '@taskCount' => $nTasks,
          '@ids'       => implode(', ', $ids),
        ]);
      return FALSE;
    }

    //
    // Clear hidden.
    // -------------
    // Reset the hidden flag so that items can be seen. This will reveal
    // content that has not been fully deleted.
    $count = self::clearAllSystemHidden();

    if ($count === 0) {
      return FALSE;
    }

    \Drupal::logger('FolderShare: File system fix')->warning(
      "Corrected @count hidden items by forcing them to be visible. \nThis will cause these items to be listed in root lists and folders. Since they were originally marked hidden prior to deletion, users may be confused by their sudden reappearance.\n",
      [
        '@count' => $count,
      ]);
    return TRUE;
  }

  /**
   * Fixes disabled items.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * <B>Problem</B>
   * If an item is marked as disabled, it is in use by a copy or move.
   * But if there are no copy or move tasks pending, there is no reason for an
   * item to be disabled. A prior copy or move may have crashed.
   *
   * <B>Fix</B>
   * Set the systemdisabled flag to FALSE to show it again.
   *
   * @param bool $fix
   *   (optional, default = TRUE) When TRUE, fix and log file system problems
   *   automatically. When FALSE, just report them to the log.
   *
   * @return bool
   *   Returns TRUE if anything in the file system was fixed or would
   *   have been fixed, and FALSE if no changes are needed.
   *
   * @section logging Operation log
   * A warning message is posted to the log that indicates the number of
   * items in need of repair (when $fix is FALSE) or that have been repaired
   * (when $fix is TRUE). The individual items repaired are not logged.
   */
  private static function fsckFixDisabled(bool $fix = FALSE) {

    if ($fix === FALSE) {
      //
      // Find disabled & tasks.
      // ----------------------
      // Find all disabled items and then look for pending copy and move
      // tasks. If there are disabled items, but no tasks, there may be
      // a problem.
      $ids = self::findAllDisabledIds();
      $count = count($ids);

      if ($count === 0) {
        return FALSE;
      }

      // Get the number of copy and delete tasks.
      $nTasks =
        FolderShareScheduledTask::findNumberOfTasks('copy-to-root') +
        FolderShareScheduledTask::findNumberOfTasks('copy-to-folder') +
        FolderShareScheduledTask::findNumberOfTasks('move-to-root') +
        FolderShareScheduledTask::findNumberOfTasks('move-to-folder');

      if ($nTasks === 0) {
        \Drupal::logger('FolderShare:  File system check')->error(
          "Found @count disabled items but no pending copy or move tasks. \nThese items are disabled, visible in lists, but they cannot be changed or deleted. Disabled items are normal during copy and move operations, but there are no copy or move tasks pending. Repair is needed. \nEntity IDs: @ids\n",
          [
            '@count' => $count,
            '@ids'   => implode(', ', $ids),
          ]);
        return TRUE;
      }

      \Drupal::logger('FolderShare:  File system check')->warning(
        "Found @count disabled items and @taskCount pending copy and/or move tasks. \nDisabled items are normal during copy and move operations in order to block user activity until the operation completes. The pending tasks should be allowed to complete. Repair is probably not needed. \nEntity IDs: @ids\n",
        [
          '@count'     => $count,
          '@taskCount' => $nTasks,
          '@ids'       => implode(', ', $ids),
        ]);
      return FALSE;
    }

    //
    // Clear disabled.
    // ---------------
    // Reset the disabled flag so that items can be shown normally.
    $count = self::clearAllSystemDisabled();

    if ($count === 0) {
      return FALSE;
    }

    \Drupal::logger('FolderShare: File system fix')->warning(
      "Corrected @count disabled items by forcing them to be enabled. \nThis will cause these items to be listed normally and changeable. However, since these items were original disabled during a copy or move that apparently did not complete, users may be confused by items that don't seem complete.\n",
      [
        '@count' => $count,
      ]);
    return TRUE;
  }

}
