<?php

namespace Drupal\foldershare\Entity\FolderShareTraits;

use Drupal\Core\Database\Database;

use Drupal\foldershare\Utilities;

/**
 * Get/set FolderShare entity system disabled field.
 *
 * This trait includes get and set methods for FolderShare entity
 * system disabled field.
 *
 * @section internal Internal trait
 * This trait is internal to the FolderShare module and used to define
 * features of the FolderShare entity class. It is a mechanism to group
 * functionality to improve code management.
 *
 * @ingroup foldershare
 */
trait GetSetSystemDisabledTrait {

  /*---------------------------------------------------------------------
   *
   * SystemDisabled field.
   *
   *---------------------------------------------------------------------*/

  /**
   * {@inheritdoc}
   */
  public function isSystemDisabled() {
    $value = $this->get('systemdisabled')->value;

    // An empty field is also FALSE.
    if (empty($value) === TRUE || $value === FALSE) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Sets the system disabled flag.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * The caller must call save() for the change to take effect.
   *
   * @param bool $state
   *   The new flag state.
   *
   * @section locking Process locks
   * This method does not lock access. The caller should lock around changes
   * to the entity.
   *
   * @see ::isSystemDisabled()
   */
  public function setSystemDisabled(bool $state) {
    $this->systemdisabled->setValue($state);
  }

  /**
   * Clears the system disabled flag on all hidden items.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B>
   *
   * This method is intended for use in system debugging and file system
   * fixes in order to quickly reset all disabled items. Since disabled
   * items are usually in the process of being changed (copied, moved,
   * ownership changed, etc.), reset all disabled items can cause confusion.
   *
   * The caller should clear the entity cache and the render cache to insure
   * that changed items are visible to the user.
   *
   * @return int
   *   Returns the number of items changed.
   *
   * @see ::setSystemDisabled()
   */
  public static function clearAllSystemDisabled() {
    $connection = Database::getConnection();

    // Clear the disabled flag on all disabled items.
    $query = $connection->update(self::BASE_TABLE);
    $query->condition('systemdisabled', TRUE, '=');
    $query->fields([
      'systemdisabled' => (int) FALSE,
    ]);
    return $query->execute();
  }

  /*---------------------------------------------------------------------
   *
   * Standard error messages.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns a standard disabled message.
   *
   * This method provides a generic message that may be used by
   * operations that need to report that an item is disabled.
   *
   * @param mixed $operationName
   *   The past tense name of the operation being performed, such as
   *   'deleted', 'moved', or 'copied'. The value may be a string or
   *   translated markup (perferred).
   * @param string $itemName
   *   (optional, default = NULL = multiple items) The name of a single item
   *   involved in an operation that cannot be done. If NULL, a multi-item
   *   message is returned instead.
   *
   * @return \Drupal\Core\Render\Markup
   *   Returns a markup object containing a formatted standard
   *   exception message.
   */
  public static function getStandardDisabledMessage(
    $operationName,
    string $itemName = NULL) {

    if (empty($operationName) === TRUE) {
      $operationName = t('updated');
    }

    if (empty($itemName) === TRUE) {
      return Utilities::createFormattedMessage(
        t('This item is in use by another user or by the system and cannot be @operation at this time.',
          [
            '@operation' => $operationName,
          ]),
          t('Please try again in a moment.'));
    }

    return Utilities::createFormattedMessage(
      t(
        'The item "@name" is in use by another user or by the system and cannot be @operation at this time.',
        [
          '@name'      => $itemName,
          '@operation' => $operationName,
        ]),
      t('Please try again in a moment.'));
  }

}
