<?php

namespace Drupal\foldershare\Entity;

use Drupal\Core\Database\Database;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;

use Drupal\foldershare\Entity\Exception\ValidationException;

/**
 * Describes an internal task to be performed at a time in the future.
 *
 * The module's internal scheduled tasks are used to perform background
 * activity to finish long operations, such as copying, moving, or deleting
 * a large folder tree. Each task has two descriptive fields:
 *
 * - The operation name used to dispatch the task to one of several
 *   well-known task handlers.
 *
 * - The operation's parameters, such as lists of entity IDs to copy, move,
 *   or delete.
 *
 * Each task has three dates/times:
 * - The date/time the task was first started.
 * - The date/time the current task object was created.
 * - The date/time the task is scheduled to run.
 *
 * When the task is first started, the creation date matches the start date.
 * The scheduled run date is a short time in the future. Each time the task
 * is run, it is removed from the database and its parameters passed to
 * the appropriate task handler. If a handler cannot finish the task within
 * execution and memory use limits, the handler creates a new task to continue
 * the work. The new task has the same original start date, a new creation
 * date, and a future scheduled date.
 *
 * For debugging and monitoring, each task also has:
 * - A brief text comment describing the task.
 * - A total execution time so far, measured in seconds.
 *
 * Like all entities, a task also has:
 * - An ID.
 * - A unique ID.
 * - A user ID for the user that started the task.
 *
 * The user ID is the current user when the task was first started. The task
 * later may be run by CRON or at the end of pages delivered to any user.
 * in the latter cases, the current user is whomever is running CRON or
 * whomever got the most recent page, but the user ID in the task remains the
 * ID of the original user that started the task. It is this original user
 * that owns any new content created by the task.
 *
 * <B>Task processing</B>
 * The list of scheduled tasks is processed in one of three ways:
 * - At the end of a request.
 * - When CRON runs.
 * - From a drush command.
 *
 * The module provides an event subscriber that listens for the "terminate"
 * event sent after a request has been finished. This is normally at the end
 * of every page delivered to a user, or after any REST or AJAX request.
 * At this time, the event subscriber calls this class's executeTasks() method.
 *
 * The module includes a CRON hook that is called each time CRON is invoked,
 * whether via an external CRON trigger or via a "terminate" event listened
 * to by the Automated Cron module. At this time, the hook calls this class's
 * executeTasks() method.
 *
 * The module includes drush commands to list tasks, delete tasks, and run
 * tasks. Drush can call this class's executeTasks() method.
 *
 * In any case, executeTasks() quickly checks if there are any tasks ready
 * to run, then runs them. A task is ready to run only if its scheduled time
 * is equal to the current time or in the past. If there are multiple tasks
 * ready to run, they are executed in order of their scheduled times.
 *
 * <B>Task visibility</B>
 * Tasks are strictly an internal implementation detail of this module.
 * They are not intended to be seen by users or administrators. For this
 * reason, the entity definition intentionally omits features:
 * - The entity type is marked as internal.
 * - None of the entity type's fields are viewable.
 * - None of the entity type's fields are editable.
 * - The entity type is not fieldable.
 * - The entity type has no "views_builder" to present view pages.
 * - The entity type has no "views_data" for creating views.
 * - The entity type has no "list_builder" to show lists.
 * - The entity type has no edit forms.
 * - The entity type has no access controller.
 * - The entity type has no admin permission.
 * - The entity type has no routes.
 * - The entity type has no caches.
 *
 * <B>Warning:</B> This class is strictly internal to the FolderShare
 * module. The class's existance, name, and content may change from
 * release to release without any promise of backwards compatability.
 *
 * @ingroup foldershare
 *
 * @ContentEntityType(
 *   id               = "foldershare_scheduledtask",
 *   label            = @Translation("FolderShare internal scheduled task"),
 *   base_table       = "foldershare_scheduledtask",
 *   internal         = TRUE,
 *   persistent_cache = FALSE,
 *   render_cache     = FALSE,
 *   static_cache     = FALSE,
 *   fieldable        = FALSE,
 *   entity_keys      = {
 *     "id"           = "id",
 *     "uuid"         = "uuid",
 *     "label"        = "operation",
 *   },
 * )
 */
final class FolderShareScheduledTask extends ContentEntityBase {

  /*---------------------------------------------------------------------
   *
   * Fields.
   *
   *---------------------------------------------------------------------*/

  /**
   * Indicates if task execution is enabled.
   *
   * In normal use, this value is TRUE. However, if task execution needs
   * to be disabled FOR THE CURRENT PROCESS ONLY, this flag may be set
   * to FALSE. Future calls to executeTasks() will return doing nothing.
   *
   * This is primarily used by drush commands to disable task execution
   * during a command so that tasks don't interfer with whatever the command
   * is trying to do.
   *
   * @var bool
   *
   * @see ::executeTasks()
   * @see ::isTaskExecutionEnabled()
   * @see ::setTaskExecutionEnabled()
   */
  private static $enabled = TRUE;

  /*---------------------------------------------------------------------
   *
   * Constants - Entity type id.
   *
   *---------------------------------------------------------------------*/

  /**
   * The entity type id for the FolderShare Scheduled Task entity.
   *
   * This is 'foldershare_scheduledtask' and it must match the entity type
   * declaration in this class's comment block.
   *
   * @var string
   */
  const ENTITY_TYPE_ID = 'foldershare_scheduledtask';

  /*---------------------------------------------------------------------
   *
   * Constants - Database tables.
   *
   *---------------------------------------------------------------------*/

  /**
   * The base table for 'foldershare_scheduledtask' entities.
   *
   * This is 'foldershare_scheduledtask' and it must match the base table
   * declaration in this class's comment block.
   *
   * @var string
   */
  const BASE_TABLE = 'foldershare_scheduledtask';

  /*---------------------------------------------------------------------
   *
   * Entity definition.
   *
   *---------------------------------------------------------------------*/

  /**
   * Defines the fields used by instances of this class.
   *
   * The following fields are defined, along with their intended
   * public or private access:
   *
   * | Field            | Allow for view | Allow for edit |
   * | ---------------- | -------------- | -------------- |
   * | id               | no             | no             |
   * | uuid             | no             | no             |
   * | uid              | no             | no             |
   * | created          | no             | no             |
   * | operation        | no             | no             |
   * | parameters       | no             | no             |
   * | scheduled        | no             | no             |
   * | started          | no             | no             |
   * | comments         | no             | no             |
   * | executiontime    | no             | no             |
   *
   * Some fields are supported by parent class methods:
   *
   * | Field            | Get method                           |
   * | ---------------- | ------------------------------------ |
   * | id               | ContentEntityBase::id()              |
   * | uuid             | ContentEntityBase::uuid()            |
   * | operation        | ContentEntityBase::getName()         |
   *
   * Some fields are supported by methods in this class:
   *
   * | Field            | Get method                           |
   * | ---------------- | ------------------------------------ |
   * | uid              | getRequester()                       |
   * | created          | getCreatedTime()                     |
   * | operation        | getOperation()                       |
   * | parameters       | getParameters()                      |
   * | scheduled        | getScheduledTime()                   |
   * | started          | getStartedTime()                     |
   * | comments         | getComments()                        |
   * | executiontime    | getAccumulatedExecutionTime()        |
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entityType
   *   The entity type for which we are returning base field definitions.
   *
   * @return array
   *   An array of field definitions where keys are field names and
   *   values are BaseFieldDefinition objects.
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entityType) {
    //
    // Base class fields
    // -----------------
    // The parent ContentEntityBase class supports several standard
    // entity fields:
    //
    // - id: the entity ID
    // - uuid: the entity unique ID
    // - langcode: the content language
    // - revision: the revision ID
    // - bundle: the entity bundle
    //
    // The parent class ONLY defines these fields if they exist in
    // THIS class's comment block declaring class fields.  Of the
    // above fields, we only define these for this class:
    //
    // - id
    // - uuid
    //
    // By invoking the parent class, we don't have to define these
    // ourselves below.
    $fields = parent::baseFieldDefinitions($entityType);

    // Entity id.
    // This field was already defined by the parent class.
    $fields[$entityType->getKey('id')]
      ->setDescription(t('The ID of the task.'))
      ->setDisplayConfigurable('view', FALSE);

    // Unique id (UUID).
    // This field was already defined by the parent class.
    $fields[$entityType->getKey('uuid')]
      ->setDescription(t('The UUID of the task.'))
      ->setDisplayConfigurable('view', FALSE)
      ->setDisplayConfigurable('form', FALSE);

    //
    // Common fields
    // -------------
    // Tasks have several fields describing the task, when it was started,
    // and when it should next run.
    //
    // Operation.
    $fields['operation'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Operation'))
      ->setDescription(t('The task operation code.'))
      ->setRequired(TRUE)
      ->setSettings([
        'default_value'   => '',
        'max_length'      => 256,
        'text_processing' => FALSE,
      ])
      ->setDisplayConfigurable('view', FALSE)
      ->setDisplayConfigurable('form', FALSE);

    // Requester (original) user id.
    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Requester'))
      ->setDescription(t("The user ID that requested the task."))
      ->setRequired(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setDefaultValueCallback(
        'Drupal\foldershare\Entity\FolderShareScheduledTask::getCurrentUserId')
      ->setDisplayConfigurable('view', FALSE)
      ->setDisplayConfigurable('form', FALSE);

    // Creation date.
    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created date'))
      ->setDescription(t('The date and time when this task entry was created.'))
      ->setRequired(TRUE)
      ->setDisplayConfigurable('view', FALSE)
      ->setDisplayConfigurable('form', FALSE);

    // Scheduled time to run.
    $fields['scheduled'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Scheduled time'))
      ->setDescription(t('The date and time when the task is scheduled to run.'))
      ->setRequired(TRUE)
      ->setDisplayConfigurable('view', FALSE)
      ->setDisplayConfigurable('form', FALSE);

    // Task parameters.
    $fields['parameters'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Task parameters'))
      ->setDescription(t('The JSON parameters for the task.'))
      ->setRequired(FALSE)
      ->setDisplayConfigurable('view', FALSE)
      ->setDisplayConfigurable('form', FALSE);

    // Original operation date.
    $fields['started'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Original start date'))
      ->setDescription(t('The date and time when the operation started.'))
      ->setRequired(FALSE)
      ->setDisplayConfigurable('view', FALSE)
      ->setDisplayConfigurable('form', FALSE);

    // Comments.
    $fields['comments'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Comments'))
      ->setDescription(t("The task's comments."))
      ->setRequired(FALSE)
      ->setDisplayConfigurable('view', FALSE)
      ->setDisplayConfigurable('form', FALSE);

    // Accumulated run time.
    $fields['executiontime'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Accumulated execution time'))
      ->setDescription(t("The task's total execution time to date."))
      ->setRequired(FALSE)
      ->setDisplayConfigurable('view', FALSE)
      ->setDisplayConfigurable('form', FALSE);

    return $fields;
  }

  /*---------------------------------------------------------------------
   *
   * General utilities.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the current user ID.
   *
   * This function provides the deault value callback for the 'uid'
   * base field definition.
   *
   * @return array
   *   An array of default values. In this case, the array only
   *   contains the current user ID.
   *
   * @see ::baseFieldDefinitions()
   */
  public static function getCurrentUserId() {
    return [\Drupal::currentUser()->id()];
  }

  /*---------------------------------------------------------------------
   *
   * Create.
   *
   *---------------------------------------------------------------------*/

  /**
   * Creates a new task.
   *
   * Every task has:
   * - An operation name that indicates the type of task to perform.
   * - A set of parameters for that task (this may be empty).
   * - A user ID for the user that requested the task.
   * - A timestamp for the future time at which to run the task.
   * - A timestamp for when the operation was first started.
   * - Comments to describe the task during debugging.
   * - An accumulated run time, in seconds.
   *
   * Operation names must be one of the well-known names supported by this
   * module. Parameters must be appropriate for the operation.
   *
   * This method should be used in preference to create() in order to
   * insure that the operation is valid and to use JSON encoding to process
   * task parameters to be saved with the task.
   *
   * @param int $timestamp
   *   The future time at which the task will be executed.
   * @param string $operation
   *   The name of the operation. This must be one of this module's
   *   well-known internal operations.
   * @param int $requester
   *   (optional, default = (-1) = current user) The user ID of the
   *   individual causing the task to be created.
   * @param array $parameters
   *   (optional, default = NULL) An array of parameters to save with the
   *   task and pass to the task when it is executed.
   * @param int $started
   *   (optional, default = 0) The timestamp of the start date & time for
   *   an operation that causes a chain of tasks.
   * @param string $comments
   *   (optional, default = '') A comment on the current task.
   * @param int $executionTime
   *   (optional, default = 0) The accumulated total execution time of the
   *   task chain, in seconds.
   *
   * @return \Drupal\foldershare\Entity\FolderShareScheduledTask
   *   Returns the newly created task. The task will already have been saved
   *   to the task table.
   *
   * @throws \Drupal\foldershare\Entity\Exception\ValidationException
   *   Throws an exception if the parameters array cannot be encoded as JSON
   *   (which is very unlikely).
   */
  public static function createTask(
    int $timestamp,
    string $operation,
    int $requester = (-1),
    array $parameters = NULL,
    int $started = 0,
    string $comments = '',
    int $executionTime = 0) {

    // Validate the operation.
    switch ($operation) {
      case 'changeowner':
      case 'copy-to-folder':
      case 'copy-to-root':
      case 'delete-hide':
      case 'delete-delete':
      case 'move-to-folder':
      case 'move-to-root':
      case 'rebuildusage':
        break;

      default:
        throw new ValidationException(t(
          'Unrecognized "@taskName" operation for internal task.',
          [
            '@taskName' => $operation,
          ]));
    }

    // Insure we have a requester.
    if ($requester < 0) {
      $requester = (int) \Drupal::currentUser()->id();
    }

    // Convert parameters to a JSON encoding.
    if ($parameters === NULL) {
      $json = '';
    }
    else {
      $json = json_encode($parameters);
      if ($json === FALSE) {
        throw new ValidationException(t(
          'Task parameters cannot be JSON encoded for internal "@taskName" operation.',
          [
            '@taskName' => $operation,
          ]));
      }
    }

    // Create the task. Let the ID, UUID, and creation date be automatically
    // assigned.
    $task = self::create([
      // Required fields.
      'operation'     => $operation,
      'uid'           => $requester,
      'scheduled'     => $timestamp,

      // Optional fields.
      'parameters'    => $json,
      'started'       => $started,
      'comments'      => $comments,
      'executiontime' => $executionTime,
    ]);

    $task->save();

    return $task;
  }

  /*---------------------------------------------------------------------
   *
   * Delete.
   *
   *---------------------------------------------------------------------*/

  /**
   * Deletes all tasks.
   *
   * Any task already executing will continue to execute until it finishes.
   * That execution may add new tasks, which will not be deleted.
   *
   * <B>Warning:</B> Deleting all tasks ends any pending operations, such as
   * those to delete, copy, or move content. This can leave these operations
   * in an indetermine state, with parts of the operation incomplete, locks
   * still locked, and entities marked hidden or disabled. This should only
   * be done as a last resort.
   *
   * @see ::deleteTask()
   * @see ::deleteTasks()
   * @see ::findNumberOfTasks()
   */
  public static function deleteAllTasks() {
    // Truncate the task table to delete everything.
    //
    // Since this entity type does not support a persistent cache, a static
    // cache, or a render cache, we do not have to worry about caches getting
    // out of sync with the database.
    $connection = Database::getConnection();
    $truncate = $connection->truncate(self::BASE_TABLE);
    $truncate->execute();
  }

  /**
   * Deletes a task.
   *
   * The task is deleted. If it is already executing, it will continue to
   * execute until it finishes. That execution may add new tasks, which
   * will not be deleted.
   *
   * <B>Warning:</B> Deleting a task ends any pending operation, such as one
   * to delete, copy, or move content. This can leave an operation in an
   * indetermine state, with parts of the operation incomplete, locks
   * still locked, and entities marked hidden or disabled. This should only
   * be done as a last resort.
   *
   * @param \Drupal\foldershare\Entity\FolderShareScheduledTask $task
   *   The task to delete.
   *
   * @see ::deleteAllTasks()
   * @see ::deleteTasks()
   * @see ::findNumberOfTasks()
   */
  public static function deleteTask(FolderShareScheduledTask $task) {
    if ($task === NULL) {
      return;
    }

    // It is possible that the task object has been loaded by more than one
    // process, then deleted by more than one process. The first delete
    // actually removes it from the entity table. The second delete does
    // nothing.
    try {
      $task->delete();
    }
    catch (\Exception $e) {
      // Do nothing.
    }
  }

  /**
   * Deletes tasks for a specific operation.
   *
   * All tasks with the indicated operation name are deleted. If a task is
   * already executing, it will continue to execute until it finishes. That
   * execution may add new tasks, which will not be deleted.
   *
   * <B>Warning:</B> Deleting a task ends any pending operation, such as one
   * to delete, copy, or move content. This can leave an operation in an
   * indetermine state, with parts of the operation incomplete, locks
   * still locked, and entities marked hidden or disabled. This should only
   * be done as a last resort.
   *
   * @param string $operation
   *   The name of the operation for which to delete all tasks.
   *
   * @see ::deleteAllTasks()
   * @see ::deleteTask()
   * @see ::findNumberOfTasks()
   */
  public static function deleteTasks(string $operation) {
    if (empty($operation) === TRUE) {
      self::deleteAllTasks();
    }

    $connection = Database::getConnection();
    $query = $connection->delete(self::BASE_TABLE);
    $query->condition('operation', $operation, '=');
    $query->execute();
  }

  /*---------------------------------------------------------------------
   *
   * Fields access.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the task's approximate accumulated execution time in seconds.
   *
   * A task may keep track of its accumulated execution time through a
   * chain of tasks, starting with the initial run of the task, followed
   * by a series of continuation runs.
   *
   * The execution time is approximate. If an operation schedules a
   * safety net task, runs for awhile, and is interrupted before it can
   * swap the safety net task with a continuation task, then the accumulated
   * execution time of the interrupted task will not have had a chance to
   * be saved into a continuation task.
   *
   * @return int
   *   Returns the approximate accumulated execution time in seconds.
   */
  public function getAccumulatedExecutionTime() {
    return $this->get('executiontime')->value;
  }

  /**
   * Returns the task's creation timestamp.
   *
   * @return int
   *   Returns the creation timestamp for this task.
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * Returns the task's optional comments.
   *
   * Comments are optional and may be used by a task to annotate why the
   * task exists or how it is progressing.
   *
   * @return string
   *   Returns the comments for the task.
   */
  public function getComments() {
    return $this->get('comments')->value;
  }

  /**
   * Returns the task's operation name.
   *
   * @return string
   *   Returns the name of the task operation.
   */
  public function getOperation() {
    return $this->get('operation')->value;
  }

  /**
   * Returns the task's parameters.
   *
   * @return array
   *   Returns an array of task parameters.
   */
  public function getParameters() {
    return $this->get('parameters')->value;
  }

  /**
   * Returns the user ID of the user that initiated the task.
   *
   * @return int
   *   Returns the requester's user ID.
   */
  public function getRequester() {
    return (int) $this->get('uid')->target_id;
  }

  /**
   * Returns the task's scheduled run timestamp.
   *
   * @return int
   *   Returns the scheduled run timestamp for this task.
   */
  public function getScheduledTime() {
    return $this->get('scheduled')->value;
  }

  /**
   * Returns the task's operation start timestamp.
   *
   * The start time is the time when an operation began, such as
   * the request time for a copy, move, or delete. This operation led to
   * the creation of the task object, which has created and scheduled times.
   * If that task reschedules itself into a continuing series of tasks,
   * all of them should share the same operation started timestamp.
   *
   * @return int
   *   Returns the original start timestamp for this task.
   */
  public function getStartedTime() {
    return $this->get('started')->value;
  }

  /*---------------------------------------------------------------------
   *
   * Find tasks.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns the number of scheduled tasks ready to be executed.
   *
   * Ready tasks are those with a task timestamp that is equal to
   * the current time or in the past.
   *
   * @param int $timestamp
   *   The timestamp used to select which tasks are ready.
   *
   * @see ::findNumberOfTasks()
   * @see ::findReadyTaskIds()
   * @see ::executeTasks()
   */
  public static function findNumberOfReadyTasks(int $timestamp) {
    $connection = Database::getConnection();
    $select = $connection->select(self::BASE_TABLE, "st");
    $select->condition('scheduled', $timestamp, '<=');

    return (int) $select->countQuery()->execute()->fetchField();
  }

  /**
   * Returns the number of scheduled tasks.
   *
   * If an operation name is provided, the returned number only counts
   * tasks with that name. If no name is given, all tasks are counted.
   *
   * @param string $operation
   *   (optional, default = '' = any) When set, returns the number of
   *   scheduled tasks with the given operation. Otherwise returns the
   *   number of all scheduled tasks.
   *
   * @return int
   *   Returns the number of tasks.
   *
   * @see ::findNumberOfReadyTasks()
   * @see ::findReadyTaskIds()
   */
  public static function findNumberOfTasks(string $operation = '') {
    $connection = Database::getConnection();
    $select = $connection->select(self::BASE_TABLE, "st");
    if (empty($operation) === FALSE) {
      $select->condition('operation', $operation, '=');
    }

    return (int) $select->countQuery()->execute()->fetchField();
  }

  /**
   * Returns an ordered array of scheduled and ready task IDs.
   *
   * Ready tasks are those with a task timestamp that is equal to
   * the current time or in the past. The returned array is ordered
   * from oldest to newest.
   *
   * @param int $timestamp
   *   The timestamp used to select which tasks are ready.
   *
   * @return int[]
   *   Returns an array of task IDs for ready tasks, ordered from
   *   oldest to newest.
   *
   * @see ::findNumberOfReadyTasks()
   * @see ::findNumberOfTasks()
   * @see ::executeTasks()
   */
  public static function findReadyTaskIds(int $timestamp) {
    $connection = Database::getConnection();
    $select = $connection->select(self::BASE_TABLE, "st");
    $select->addField('st', 'id', 'id');
    $select->condition('scheduled', $timestamp, '<=');
    $select->orderBy('scheduled');

    return $select->execute()->fetchCol(0);
  }

  /**
   * Returns an ordered array of all scheduled task IDs.
   *
   * If an operation name is provided, the returned list only includes
   * tasks with that name. If no name is given, all tasks are included.
   *
   * The returned array is ordered from oldest to newest.
   *
   * @param string $operation
   *   (optional, default = '' = any) When set, returns the scheduled tasks
   *   with the given operation. Otherwise returns all scheduled tasks.
   *
   * @return int[]
   *   Returns an array of task IDs, ordered from oldest to newest.
   *
   * @see ::findReadyTaskIds()
   * @see ::findNumberOfTasks()
   */
  public static function findTaskIds(string $operation = '') {
    $connection = Database::getConnection();
    $select = $connection->select(self::BASE_TABLE, "st");
    $select->addField('st', 'id', 'id');
    if (empty($operation) === FALSE) {
      $select->condition('operation', $operation, '=');
    }

    $select->orderBy('scheduled');

    return $select->execute()->fetchCol(0);
  }

  /*---------------------------------------------------------------------
   *
   * Execute tasks.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns TRUE if task execution is enabled, and FALSE otherwise.
   *
   * At the start of every process, this is TRUE and ready tasks will execute
   * each time executeTasks() is called.
   *
   * Task execution can be disabled FOR THE CURRENT PROCESS ONLY by calling
   * setTaskExecutionEnabled() with a FALSE argument.
   *
   * @return bool
   *   Returns TRUE if enabled.
   *
   * @see ::setTaskExecutionEnabled()
   * @see ::executeTasks()
   */
  public static function isTaskExecutionEnabled() {
    return self::$enabled;
  }

  /**
   * Enables or disables task execution.
   *
   * At the start of every process, this is TRUE and ready tasks will execute
   * each time executeTasks() is called.
   *
   * When set to FALSE, executeTasks() will return immediately, doing nothing.
   * This may be used to temporarily disable task execution FOR THE CURRENT
   * PROCESS ONLY. This has no effect on other processes or future processes.
   *
   * A common use of execution disabling is by drush, which needs to execute
   * commands without necessarily running pending tasks.
   *
   * @param bool $enable
   *   TRUE to enable, FALSE to disable.
   *
   * @see ::isTaskExecutionEnabled()
   * @see ::executeTasks()
   */
  public static function setTaskExecutionEnabled(bool $enable) {
    self::$enabled = $enable;
  }

  /**
   * Executes ready tasks.
   *
   * All tasks with scheduled times equal to or earlier than the given
   * time stamp will be considered ready to run and executed in oldest to
   * newest order. Tasks are deleted just before they are executed. Tasks
   * may add more tasks.
   *
   * Task execution will abort if execution has been disabled FOR THE
   * CURRENT PROCESS ONLY using setTaskExecutionEnabled().
   *
   * @param int $timestamp
   *   The time used to find all ready tasks. Any task scheduled to run at
   *   this time, or earlier, is considered ready and will be executed.
   *   This is typically set to the current time, or the time at which an
   *   HTTP request was made.
   *
   * @see ::findNumberOfReadyTasks()
   * @see ::findReadyTaskIds()
   * @see ::isTaskExecutionEnabled()
   * @see ::setTaskExecutionEnabled()
   */
  public static function executeTasks(int $timestamp) {
    // If execution is disabled, return immediately.
    if (self::$enabled === FALSE) {
      return;
    }

    //
    // Quick reject.
    // -------------
    // This function is called after every page is sent to a user. It is
    // essential that it quickly decide if there is anything to do.
    try {
      if (self::findNumberOfReadyTasks($timestamp) === 0) {
        // Nothing to do.
        return;
      }
    }
    catch (\Exception $e) {
      // Query failed?
      return;
    }

    //
    // Get ready tasks.
    // ----------------
    // A task is ready if the current time is greater than or equal to
    // its scheduled time.
    try {
      $taskIds = self::findReadyTaskIds($timestamp);
    }
    catch (\Exception $e) {
      // Query failed?
      return;
    }

    //
    // Execute tasks.
    // --------------
    // Loop through the tasks and execute them.
    //
    // If a task fails to load, it has already been deleted. It may have
    // been serviced by another execution of this same method running in
    // another process after delivering a page for another user.
    foreach ($taskIds as $taskId) {
      $task = self::load($taskId);
      if ($task === NULL) {
        continue;
      }

      // Delete the task to reduce the chance that another process will
      // try to service the same task at the same time. This can still
      // happen and tasks must be written to consider this.
      $task->delete();

      // Copy out the task's values.
      $operation     = $task->getOperation();
      $json          = $task->getParameters();
      $requester     = $task->getRequester();
      $comments      = $task->getComments();
      $started       = $task->getStartedTime();
      $executionTime = $task->getAccumulatedExecutionTime();

      unset($task);

      // Decode JSON-encoded task parameters.
      if (empty($json) === TRUE) {
        $parameters = [];
      }
      else {
        $parameters = json_decode($json, TRUE, 512, JSON_OBJECT_AS_ARRAY);
        if ($parameters === NULL) {
          // The parameters could not be decoded! This should not happen
          // since they were encoded using json_encode() during task creation.
          // There is nothing we can do with the task.
          \Drupal::logger('FolderShare: Programmer error')->error(
            'Missing or malformed parameters for internal "@taskName" task.',
            [
              '@taskName' => $operation,
            ]);
          continue;
        }
      }

      // Dispatch to module task handlers.
      try {
        switch ($operation) {
          case 'changeowner':
            FolderShare::processTaskChangeOwner(
              $requester,
              $parameters,
              $started,
              $comments,
              $executionTime);
            break;

          case 'copy-to-folder':
            FolderShare::processTaskCopyToFolder(
              $requester,
              $parameters,
              $started,
              $comments,
              $executionTime);
            break;

          case 'copy-to-root':
            FolderShare::processTaskCopyToRoot(
              $requester,
              $parameters,
              $started,
              $comments,
              $executionTime);
            break;

          case 'delete-hide':
            FolderShare::processTaskDelete1(
              $requester,
              $parameters,
              $started,
              $comments,
              $executionTime);
            break;

          case 'delete-delete':
            FolderShare::processTaskDelete2(
              $requester,
              $parameters,
              $started,
              $comments,
              $executionTime);
            break;

          case 'move-to-folder':
            FolderShare::processTaskMoveToFolder(
              $requester,
              $parameters,
              $started,
              $comments,
              $executionTime);
            break;

          case 'move-to-root':
            FolderShare::processTaskMoveToRoot(
              $requester,
              $parameters,
              $started,
              $comments,
              $executionTime);
            break;

          case 'rebuildusage':
            FolderShareUsage::processTaskRebuildUsage(
              $requester,
              $parameters,
              $started,
              $comments,
              $executionTime);
            break;

          default:
            // Unknown operation. This should not happen since operation names
            // were validated during task creation.
            \Drupal::logger('FolderShare: Programmer error')->error(
              'Unrecognized "@taskName" operation for internal task execution.',
              [
                '@taskName' => $operation,
              ]);
            break;
        }
      }
      catch (\Exception $e) {
        // Unexpected exception.
      }
    }
  }

}
