<?php

/**
 * @file
 * Implements help hooks for the module.
 */

use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\user\Entity\User;

use Drupal\foldershare\Branding;
use Drupal\foldershare\Constants;
use Drupal\foldershare\Utilities;

/**
 * Implements hook_help().
 *
 * This function responds to the help route for the FolderShare module
 * and provides a single page with multiple sections to document the
 * purpose, uses, permissions, and access control mechanisms of the
 * module.
 */
function foldershare_help(string $routeName, RouteMatchInterface $routeMatch) {
  //
  // The function returns a render array containing multiple help sections,
  // partly mandated by Drupal best practices.
  //
  // Check the route.
  $helpRoute = Constants::ROUTE_HELP . '.' . Constants::MODULE;
  if ($routeName !== $helpRoute) {
    // The route is not for our help page. Do nothing.
    return;
  }

  //
  // Setup
  // -----
  // Create class names and get the module's human-readable title.
  $pageClass = Constants::MODULE . '-admin-help';
  $moduleTitle = \Drupal::service('module_handler')->getName(Constants::MODULE);

  // Get routed links to the module's configuration and permissions.
  $moduleConfiguration = Utilities::createRouteLink(
    Constants::ROUTE_SETTINGS,
    '',
    'module configuration');
  $modulePermissions = Utilities::createRouteLink(
    'user.admin_permissions',
    'module-' . Constants::MODULE,
    'module permissions');

  $anonymousUser = User::load(0);
  $anonymousName = $anonymousUser->getDisplayName();

  //
  // About
  // -----
  // This section is recommended by Drupal best practices.
  $help = '';
  $help .= '<h4>' . t('About') . '</h4>';
  $help .= '<p>' . t(
    '<strong>@moduletitle</strong> manages shared files and folders like those for Windows, Mac, or Linux systems. Files are stored on the web server and tracked in the website database. Access is controlled using website user accounts, permissions, and shared access selections made by users.',
    [
      '@moduletitle' => $moduleTitle,
    ]) . '</p>';

  //
  // Uses
  // ----
  // This section is recommended by Drupal best practices.
  $help .= '<h4>' . t('Uses') . '</h4>';
  $help .= '<p>' . t(
    'Users of <strong>@moduletitle</strong> may create and edit a folder tree of shared files and folders managed by the website. Users may upload files into folders, and move, copy, rename, search, compress, delete, and download files and folders.',
    [
      '@moduletitle' => $moduleTitle,
      '@anonymous'   => $anonymousName,
    ]) . '</p>';
  $help .= '<p>' . t(
    'By default, users have exclusive access to their files and folders, but users may elect to share folder trees with specific users and allow them to view and edit folder content. Users also may elect to publish their content for world access by sharing their content with the site\'s generic "@anonymous" user.',
    [
      '@anonymous'   => $anonymousName,
    ]) . '</p>';
  $help .= '<p>' . t(
    "These abilities may be used to organize files and folders that contain personal or shared documents, image libraries, videos, or current work by a work group. Access controls enable users to limit access to specific friends and collaborators and treat the website's folders as a shared work space for a project.") . '</p>';

  //
  // Permissions
  // -----------
  // Explain role-based permissions.
  $help .= '<h4>' . t('Permissions') . '</h4>';
  $help .= '<p>' . t(
    "<strong>@moduletitle's</strong> @perm are assigned to user roles to give broad categories of users the ability to view, author, share, and administer file and folder content. These permissions work together with access controls that can grant access to specific users for specific folder trees. The module's permissions include:",
    [
      '@moduletitle' => $moduleTitle,
      '@perm'        => $modulePermissions,
    ]) . '</p>';

  $help .= '<dl>';
  $help .= '<dt><em>' . t('View files and folders') . '</em></dt>';
  $help .= '<dd>' . t(
    "Users with this permission may view and download their own content. When a user has this permission and they are also granted view access to specific shared content, the user may view and download that shared content too.") . '</dd>';
  $help .= '<dd>' . t(
    "This permission is usually granted to all users, including the anonymous user.") . '</dd>';
  $help .= '<dd>' . t(
    "If the anonymous user does not have view permission, then site visitors that do not have accounts cannot view any files and folders.") . '</dd>';

  $help .= '<dt><em>' . t('Author files and folders') . '</em></dt>';
  $help .= '<dd>' . t(
    "Users with this permission may create, delete, and modify their own content. When a user has this permission and they are also granted author access to specific shared content, the user may create, delete, and modify that shared content too.") . '</dd>';
  $help .= '<dd>' . t(
    "This permission is usually only enabled for authenticated users.") . '</dd>';

  $help .= '<dt><em>' . t('Share files and folders with other users') . '</em></dt>';
  $help .= '<dd>' . t(
    "Users with this permission may share specific content with other users, granting selected users view and (optionally) author access. Users granted view access to shared content may view and download the content. Users granted author access may created, delete, and modify the shared content.") . '</dd>';
  $help .= '<dd>' . t(
    "This permission is usually enabled for all authenticated users.") . '</dd>';
  $help .= '<dd>' . t(
    "Sites that do not need content sharing can disable this permission for all user roles. Users will still be able to access their own content.") . '</dd>';

  $help .= '<dt><em>' . t('Share files and folders with the public') . '</em></dt>';
  $help .= '<dd>' . t(
    "Users with this permission may share specific content with anonymous 'public' users that do not have accounts on the site. Anonymous users can only be granted view access, not author access.") . '</dd>';
  $help .= '<dd>' . t(
    "This permission is usually only granted to trusted content authors who have responsability for general site content, such as documentation, released software, press kits, and so forth.") . '</dd>';
  $help .= '<dd>' . t(
    "Sites that do not want users to share content for anonymous access can disable this permission for all user roles. Anonymous users will still be able to access content owned by the anonymous user, if any.") . '</dd>';

  $help .= '<dt><em>' . t('Administer files and folders') . '</em></dt>';
  $help .= '<dd>' . t(
    "Users with this permission may create, delete, and modify their own content and content owned by any user. They may change ownership and sharing grants for specific content, but they cannot change the site configuration.") . '</dd>';
  $help .= '<dd>' . t(
    "This permission is usually only granted to site administrators, content moderators, and those handling customer support. It enables a user to moderate content and fix problems. It should not be granted to all users or anonymous users.") . '</dd>';

  $help .= '</dl>';

  $help .= '<p>' . t(
    'These permissions are designed to be additive. Users that are granted <em>author</em> access should also be granted <em>view</em> access so that they can view the content they create, and similarly for the other permissions.') . '</p>';

  //
  // Shared access
  // -------------
  // Note that shared access goes beyond role-based permissions.
  $help .= '<h4>' . t('Shared access') . '</h4>';
  $help .= '<p>' . t(
    'In addition to role-based permissions, users with <em>share with others</em> access may grant specific users with view and author access to selected top-level files and folder, and all of the content within.') . '</p>';
  $help .= '<p>' . t(
    'When granted <em>view</em> access, users may view all content within the folder tree. They may browser through folders, view files, copy items to their own folder trees, and download any of the content.') . '</p>';
  $help .= '<p>' . t(
    'When granted <em>author</em> access, users may modify content within the folder tree. They may create new subfolders, upload files, edit file and folder descriptions, rename items, move items about, and delete them.') . '</p>';
  $help .= '<p>' . t(
    "Users that are granted view or author access to a folder tree cannot change the folder tree's access controls to grant other users with access. Only the owner of the folder tree can adjust its access controls.") . '</p>';

  //
  // Administration
  // --------------
  // Link to module's configuration page.
  $help .= '<h4>' . t('Administration') . '</h4>';
  $help .= '<p>' . t(
    "<strong>@moduletitle's</strong> @config may be adjusted to control where and how files are uploaded, where they are stored on the web server, whether file name extensions are restricted, how files and folders are displayed and searched, and more. All configuration choices are explained on the module configuration form.",
    [
      '@moduletitle' => $moduleTitle,
      '@config'      => $moduleConfiguration,
    ]) . '</p>';

  //
  // PHP directives
  // --------------
  // Link to the PHP manual.
  $help .= '<h4>' . t('PHP directives') . '</h4>';
  $help .= '<p>' . t(
    "Like all modules that deal with files, <strong>@moduletitle</strong> is governed by several PHP directives. These are typically set in a host's <code>php.ini</code> file or in a site's <code>.htaccess</code> file:",
    [
      '@moduletitle' => $moduleTitle,
    ]);
  $help .= '<ul>';
  $help .= '<li><code>' . t('upload_max_filesize') . '</code></li>';
  $help .= '<li><code>' . t('post_max_size') . '</code></li>';
  $help .= '<li><code>' . t('max_file_uploads') . '</code></li>';
  $help .= '<li><code>' . t('memory_limit') . '</code></li>';
  $help .= '</ul></p>';

  $help .= '<p>' . t(
    "For example, the <code>upload_max_filesize</code> limits the size of any single uploaded file, while <code>max_file_uploads</code> limits the number of files that can be uploaded at once. Since a single upload operation with multiple files includes them all in a single form post, <code>post_max_size</code> needs to be at least as large as <code>(max_file_uploads * upload_max_filesize)</code>. And since the form, and all of its file data, need to be loaded into memory for processing, <code>memory_limit</code> needs to be at least as large as <code>post_max_size</code>.") . '</p>';
  $help .= '<p>' . t(
    "See the PHP manual's page on @phpmanual for further information on these values and where they can be set.",
    [
      '@phpmanual' => Utilities::createUrlLink('http://php.net/manual/en/ini.core.php', 'Core php.ini directives'),
    ]) . '</p>';

  //
  // Recommendations
  // ---------------
  // List recommended modules and configuration.
  $help .= '<h4>' . t('Recommendations') . '</h4>';
  $help .= '<p>' . t(
    'Recommended configuration for this module:') . '</p>';
  $help .= '<ul class="' . Constants::MODULE . '-help-recommend-list">';
  $help .= '<li>' . t(
    "The module's file name extension restrictions may be disabled so that users can upload files of any type. Internally, this module stores all files on the server in a directory marked to prevent a web server from executing anything. This prevents the web server from being tricked into executing malicious uploads. Disabling file name extension restrictions for uploaded files retains these protections and is therefore safe to do.") . '</li>';
  $help .= '<li>' . t(
    "Uploaded files may be stored in a <em>Public</em> or <em>Private</em> file system, but in both cases the files are normally served via this module, not the web server, so that file access controls can be checked. For this module's files, then, the performance difference between <em>Public</em> and <em>Private</em> file systems is negligible. However, a <em>Private</em> file system also prevents malicious users from guessing file paths and trying to trick the system into downloading files they should not access. A <em>Private</em> file system is therefore recommended.") . '</li>';
  $help .= '<li>' . t(
    'The module provides permissions that may be enabled for specific roles. A typical configuration grants <em>view</em> access to anonymous users; <em>view</em>, <em>author</em>, and one or both <em>share</em> accesses to authenticated users, and everything to administrators.') . '</li>';
  $help .= '</ul>';

  $help .= '<p>' . t(
    'Recommended configurations for related modules:') . '</p>';
  $help .= '<ul class="' . Constants::MODULE . '-help-recommend-list">';
  $help .= '<li>' . t(
    'If the "Field UI" module is installed, fields can be added to the file and folder definitions, page layouts can be adjusted, and field formatting set.') . '</li>';
  $help .= '<li>' . t(
    'If the "View UI" module is installed, the columns of file and folder lists can be adjusted and field formatting set.') . '</li>';
  $help .= '<li>' . t(
    'If the "Comment" module is installed, a default comment type is available that enables users to add comments to files and folders.') . '</li>';
  $help .= '<li>' . t(
    'If the "Search" module is installed, a search plugin is available that enables users to search for files and folders, and optional search through some file content.') . '</li>';
  $help .= '<li>' . t(
    'If the "REST" module is installed, a REST resource is included that enables users to get and post content to the site via command-line tools and scripts. If the "REST UI" module is installed, the REST configuration can be adjusted.') . '</li>';
  $help .= '<li>' . t(
    "Using the site's block configuration, enable the breadcrumbs block and place it on the page. This module automatically builds breadcrumbs to show the folder hierarchy.") . '</li>';
  $help .= '<li>' . t(
    'Configure Cron (or equivalent) to run regularly. Hourly runs are recommended as a minimum. This module uses Cron to schedule background tasks that keep folder sizes uptodate and create search indexes of file and folder content. The shorter the period between Cron runs, the more uptodate these will be.') . '</li>';
  $help .= '<li>' . t(
    'Add a text filter that supports some or all of HTML, then assign it to the module\'s folder "description" field by using the folder "Manage display" tab. This will enable users to enter formatted text describing their folders. Also consider enabling the WYSIWYG editor for the text filter.') . '</li>';
  $help .= '</ul>';

  //
  // Render element
  // --------------
  // Return a render element instead of the HTML string itself. This is
  // needed in order to include module libraries to style the help page.
  $page = [
    '#attached'  => [
      'library'  => [
        Constants::LIBRARY_MODULE,
        Constants::LIBRARY_ADMIN,
      ],
    ],
    'page'       => [
      '#type'    => 'container',
      '#tree'    => TRUE,
      '#attributes' => [
        'class'     => [$pageClass],
      ],
      'branding' => Branding::getBannerBranding(),
      'help'     => [
        '#type'  => 'html_tag',
        '#tag'   => 'div',
        '#value' => $help,
      ],
    ],
  ];

  return $page;
}
