<?php

namespace Drupal\foldershare\Plugin\FolderShareCommand;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\Entity\User;

use Drupal\foldershare\Constants;
use Drupal\foldershare\Settings;
use Drupal\foldershare\Utilities;
use Drupal\foldershare\FolderShareInterface;
use Drupal\foldershare\Entity\FolderShare;
use Drupal\foldershare\Entity\FolderShareAccessControlHandler;

/**
 * Defines a command plugin to change share grants on a root item.
 *
 * The command sets the access grants for the root item of the selected
 * entity. Access grants enable/disable view and author access for
 * individual users.
 *
 * Configuration parameters:
 * - 'parentId': the parent folder, if any.
 * - 'selectionIds': selected entity who's root item is shared.
 * - 'grants': the new access grants.
 *
 * @ingroup foldershare
 *
 * @FolderShareCommand(
 *  id              = "foldersharecommand_share",
 *  label           = @Translation("Share"),
 *  menuNameDefault = @Translation("Share..."),
 *  menuName        = @Translation("Share..."),
 *  description     = @Translation("Share selected top-level files and folders, and all of their contents."),
 *  category        = "settings",
 *  weight          = 10,
 *  userConstraints = {
 *    "authenticated",
 *  },
 *  parentConstraints = {
 *    "kinds"   = {
 *      "rootlist",
 *    },
 *    "access"  = "view",
 *  },
 *  selectionConstraints = {
 *    "types"   = {
 *      "one",
 *    },
 *    "kinds"   = {
 *      "any",
 *    },
 *    "access"  = "share",
 *  },
 * )
 */
class Share extends FolderShareCommandBase {

  /*--------------------------------------------------------------------
   *
   * Configuration.
   *
   *--------------------------------------------------------------------*/

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    // Include room for the new grants in the configuration.
    $config = parent::defaultConfiguration();
    $config['grants'] = [];
    return $config;
  }

  /*--------------------------------------------------------------------
   *
   * Configuration form.
   *
   *--------------------------------------------------------------------*/

  /**
   * {@inheritdoc}
   */
  public function hasConfigurationForm() {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(bool $forPage) {
    // The description varies for page vs. dialog:
    //
    // - Dialog: The description is longer and has the form "Adjust shared
    //   access for NAME, and all of its contents."
    //
    // - Page: The description is as for a dialog, except the name is not
    //   included because it is already in the title.
    $selectionIds = $this->getSelectionIds();
    $item = FolderShare::load(reset($selectionIds));

    $description = [];

    if ($forPage === TRUE) {
      // Page description. The page title already gives the name of the
      // item. Don't include the item's name again here.
      if ($item->isFolder() === TRUE) {
        $description[] = t(
          'Adjust shared access for this folder, and all of its contents.',
          [
            '@name' => $item->getName(),
          ]);
      }
      else {
        $description[] = t(
          'Adjust share shared access this @operand.',
          [
            '@operand' => Utilities::translateKind($item->getKind()),
          ]);
      }
    }
    else {
      // Dialog description. Include the name of the item to be changed.
      if ($item->isFolder() === TRUE) {
        $description[] = t(
          'Adjust share shared access "@name", and all of its contents.',
          [
            '@name' => $item->getName(),
          ]);
      }
      else {
        $description[] = t(
          'Adjust share shared access "@name".',
          [
            '@name' => $item->getName(),
          ]);
      }
    }

    $description[] = t(
      '<ul><li>Users with "View" access may view, copy, and download files and folders.</li><li>Users with "Author" access may view, copy, download, edit, rename, move, delete, and upload files and folders.</li></ul>');

    return $description;
  }

  /**
   * {@inheritdoc}
   */
  public function getTitle(bool $forPage) {
    // The title varies for page vs. dialog:
    //
    // - Dialog: The title is short and has the form "Share OPERAND",
    //   where OPERAND is the kind of item (e.g. "file"). By not putting
    //   the item's name in the title, we keep the dialog title short and
    //   avoid cropping or wrapping.
    //
    // - Page: The title is longer and has the form "Share "NAME"?"
    //   This follows Drupal convention.
    $selectionIds = $this->getSelectionIds();
    $item = FolderShare::load(reset($selectionIds));

    if ($forPage === TRUE) {
      // Page title. Include the name of the item.
      return t(
        'Share "@name"',
        [
          '@name' => $item->getName(),
        ]);
    }

    // Dialog title. Include the operand kind.
    return t(
      'Share @operand',
      [
        '@operand' => Utilities::translateKind($item->getKind()),
      ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getSubmitButtonName() {
    return t('Save');
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(
    array $form,
    FormStateInterface $formState) {

    // Always a selection that is a root item.
    $selectionIds = $this->getSelectionIds();
    $item = FolderShare::load(reset($selectionIds));

    //
    // Get current user permissions
    // ----------------------------
    // Does the current user have permission to share with other users
    // and/or share with the public?
    $currentUser = \Drupal::currentUser();
    $hasShareWithUsersPermission = AccessResult::allowedIfHasPermission(
      $currentUser,
      Constants::SHARE_PERMISSION)->isAllowed();
    $hasShareWithPublicPermission = AccessResult::allowedIfHasPermission(
      $currentUser,
      Constants::SHARE_PUBLIC_PERMISSION)->isAllowed();

    //
    // Get user info
    // -------------
    // The returned list has UID keys and values that are arrays with one
    // or more of:
    //
    // '' = no grants.
    // 'view' = granted view access.
    // 'author' = granted author access.
    //
    // The returned array cannot be empty. It always contains at least
    // the owner of the folder, who always has 'view' and 'author' access.
    $grants = $this->getAccessGrants($item);

    // Load all referenced users. We need their display names for the
    // form's list of users. Add the users into an array keyed by
    // the display name, then sort on those keys so that we can create
    // a sorted list in the form.
    $loadedUsers = User::loadMultiple(array_keys($grants));
    $users = [];
    foreach ($loadedUsers as $user) {
      $users[$user->getDisplayName()] = $user;
    }

    ksort($users, SORT_NATURAL);

    // Get the anonymous user. This user always exists and is always UID = 0.
    $anonymousUser = User::load(0);

    // Save the root folder for use when the form is submitted later.
    $this->entity = $item;

    $ownerId = $item->getOwnerId();
    $owner   = User::load($ownerId);

    //
    // Start table of users and grants
    // -------------------------------
    // Create a table that has 'User' and 'Access' columns.
    $form[Constants::MODULE . '_share_table'] = [
      '#type'       => 'table',
      '#attributes' => [
        'class'     => [Constants::MODULE . '-share-table'],
      ],
      '#header'     => [
        t('User'),
        t('Access'),
      ],
    ];

    //
    // Define 2nd header
    // -----------------
    // The secondary header breaks the 'Access' column into
    // three child columns (visually) for "None", "View", and "Author".
    // CSS makes these child columns fixed width, and the same width
    // as the radio buttons in the rows below so that they line up.
    $rows    = [];
    $tnone   = t('None');
    $tview   = t('View');
    $tauthor = t('Author');
    $tmarkup = '<span>' . $tnone . '</span><span>' .
      $tview . '</span><span>' . $tauthor . '</span>';
    $rows[]  = [
      'User' => [
        '#type'       => 'item',
        // Do not include markup for user column since we need it to be
        // blank for this row.
        '#markup'     => '',
        '#value'      => (-1),
        '#attributes' => [
          'class'     => [Constants::MODULE . '-share-subheader-user'],
        ],
      ],
      'Access'        => [
        '#type'       => 'item',
        '#markup'     => $tmarkup,
        '#attributes' => [
          'class'     => [Constants::MODULE . '-share-subheader-grants'],
        ],
      ],
      '#attributes'   => [
        'class'       => [Constants::MODULE . '-share-subheader'],
      ],
    ];

    //
    // Add anonymous user row
    // ----------------------
    // If Sharing with anonymous is allowed for this user, include a row
    // to do so at the top of the table.
    if ($hasShareWithPublicPermission === TRUE) {
      $rows[] = $this->buildRow(
        $anonymousUser,
        $owner,
        $grants[0]);
    }

    //
    // Add user rows
    // -------------
    // If sharing with other users is allowed for this user, include a list
    // of users with whome to share.
    if ($hasShareWithUsersPermission === TRUE) {
      foreach ($users as $user) {
        // Skip NULL users. This may occur when a folder included a grant to
        // a user account that has since been deleted.
        if ($user === NULL) {
          continue;
        }

        // Skip accounts:
        // - Anonymous (already handled above).
        // - Blocked accounts.
        // - The root owner.
        // - The special admin (UID = 1).
        $uid = (int) $user->id();
        if ($user->isAnonymous() === TRUE ||
            $user->isBlocked() === TRUE ||
            $uid === 1 ||
            $uid === $ownerId) {
          continue;
        }

        // Add the row.
        $rows[] = $this->buildRow(
          $user,
          $owner,
          $grants[$uid]);
      }
    }

    $form[Constants::MODULE . '_share_table'] = array_merge(
      $form[Constants::MODULE . '_share_table'],
      $rows);

    return $form;
  }

  /**
   * Builds and returns a form row for a user.
   *
   * The form row includes the user column with the user's ID and a link
   * to their profile page.  Beside the user column, the row includes
   * one, two, or three radio buttons for 'none', 'view', and 'author'
   * access grants based upon the grants provided and the user's current
   * module permissions.
   *
   * @param \Drupal\user\Entity\User $user
   *   The user for whom to build the row.
   * @param \Drupal\user\Entity\User $owner
   *   The owner of the current root item.
   * @param array $grant
   *   The access grants for the user.
   *
   * @return array
   *   The form table row description for the user.
   */
  private function buildRow(
    User $user,
    User $owner,
    array $grant) {

    //
    // Get account attributes
    // ----------------------
    // Watch for special users.
    $rowIsForAnonymous   = $user->isAnonymous();
    $rowIsForBlockedUser = $user->isBlocked();
    $rowIsForRootOwner   = ((int) $user->id() === (int) $owner->id());

    //
    // Check row user permissions
    // --------------------------
    // Get what the row user is permitted to do based on permissions alone.
    $permittedToView =
      FolderShareAccessControlHandler::mayAccess('view', $user);
    $permittedToAuthor =
      FolderShareAccessControlHandler::mayAccess('update', $user);

    //
    // Get grants for this root
    // ------------------------
    // Check the access grants on the root item and see what the row user
    // has been granted to do, if anything.
    $currentlyGrantedView = in_array('view', $grant);
    $currentlyGrantedAuthor = in_array('author', $grant);

    // If the user is explicitly granted author access, then automatically
    // include view access.
    if ($currentlyGrantedAuthor === TRUE) {
      $currentlyGrantedView = TRUE;
    }

    // Reduce access grants if they don't have permissions.
    if ($permittedToView === FALSE) {
      // The row user doesn't have view permission, so any grants they
      // have on the row's root are irrelevant.
      $currentlyGrantedView   = FALSE;
      $currentlyGrantedAuthor = FALSE;
    }

    if ($permittedToAuthor === FALSE) {
      // The row user doesn't have author permission, so any author grant
      // they have on the row's root is irrelevant.
      $currentlyGrantedAuthor = FALSE;
    }

    //
    // Override for special cases
    // --------------------------
    // While this function is not supposed to be called for the root's
    // owner or blocked accounts, if it is then disable those rows.
    $rowDisabled = FALSE;

    if (($rowIsForBlockedUser === TRUE && $rowIsForAnonymous === FALSE) ||
        $rowIsForRootOwner === TRUE) {
      // Blocked users cannot have their sharing set. They never have access.
      // Owners cannot have their sharing set. They always have access.
      $rowDisabled = TRUE;
    }

    //
    // Build row
    // ---------
    // Start by creating the radio button options array based upon the grants.
    $radios = ['none' => ''];
    $default = 'none';

    if ($permittedToView === TRUE) {
      // User has view permissions, so allow a 'view' choice.
      $radios['view'] = '';
    }

    if ($permittedToAuthor === TRUE) {
      // User has author permissions, so allow a 'author' choice.
      $radios['author'] = '';
    }

    if ($currentlyGrantedView === TRUE) {
      // User has been granted view access.
      $default = 'view';
    }

    if ($currentlyGrantedAuthor === TRUE) {
      // User has been granted author access (which for our purposes
      // includes view access).
      $default = 'author';
    }

    if ($permittedToView === FALSE) {
      // Disable the entire row if the user has no view permission.
      $rowDisabled = TRUE;
    }

    // Create an annotated user name.
    $name = $user->getDisplayName();
    if ($rowIsForAnonymous === TRUE) {
      $nameMarkup = t(
        'Everyone that can access this website',
        [
          '@name' => $name,
        ]);
    }
    else {
      $nameMarkup = t(
        '@name',
        [
          '@name' => $name,
        ]);
    }

    // Create the row. Provide the user's UID as the row value, which we'll
    // use later during validation and submit handling. Show a link to the
    // user's profile.
    return [
      'User' => [
        '#type'          => 'item',
        '#value'         => $user->id(),
        '#markup'        => $nameMarkup,
        '#attributes'    => [
          'class'        => [Constants::MODULE . '-share-user'],
        ],
      ],

      // Disable the row if the user has no permissions.
      'Access' => [
        '#type'          => 'radios',
        '#options'       => $radios,
        '#default_value' => $default,
        '#disabled'      => $rowDisabled,
        '#attributes'    => [
          'class'        => [Constants::MODULE . '-share-grants'],
        ],
      ],
      '#attributes'      => [
        'class'          => [Constants::MODULE . '-share-row'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(
    array &$form,
    FormStateInterface $formState) {

    // The selection always exists and is always a root item.
    $selectionIds = $this->getSelectionIds();
    $item = FolderShare::load(reset($selectionIds));

    // Get the original grants. The returned array is indexed by UID.
    $ownerId = $item->getOwnerId();
    $grants = [];
    $originalGrants = $this->getAccessGrants($item);

    // Copy the owner's grants forward, regardless of what may have been
    // set in the form.
    if (isset($originalGrants[$ownerId]) === TRUE) {
      // The owner always has full access, but copy this anyway.
      $grants[$ownerId] = $originalGrants[$ownerId];
    }

    //
    // Validate
    // --------
    // Validate that the new grants make sense.
    //
    // Loop through the form's table of users and access grants. For each
    // user, see if 'none', 'view', or 'author' radio buttons are set
    // and create the associated grant in a user-grant array.
    $values = $formState->getValue(Constants::MODULE . '_share_table');
    foreach ($values as $item) {
      // Get the user for this row.
      $uid = $item['User'];

      // Ignore negative user IDs, which is only used for the 2nd header
      // row that labels the radio buttons in the access column.
      if ($uid < 0) {
        continue;
      }

      // Ignore any row for the root item owner.
      if ($uid === $ownerId) {
        continue;
      }

      // Ignore blocked accounts that aren't anonymous.
      $user = User::load($uid);
      if ($user->isBlocked() === TRUE && $user->isAnonymous() === FALSE) {
        continue;
      }

      // Get the access being granted.
      switch ($item['Access']) {
        case 'view':
          // Make sure the user has view permission.
          if (FolderShareAccessControlHandler::mayAccess('view', $user) === TRUE) {
            $grants[$uid] = ['view'];
          }
          break;

        case 'author':
          // Make sure the user has author permission.
          if (FolderShareAccessControlHandler::mayAccess('update', $user) === TRUE) {
            // Author grants ALWAYS include view grants.
            $grants[$uid] = [
              'view',
              'author',
            ];
          }
          break;

        default:
          break;
      }
    }

    $this->configuration['grants'] = $grants;

    try {
      $this->validateParameters();
    }
    catch (\Exception $e) {
      $formState->setErrorByName(
        Constants::MODULE . '_share_table',
        $e->getMessage());
      return;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(
    array &$form,
    FormStateInterface $formState) {

    $this->execute();
  }

  /*---------------------------------------------------------------------
   *
   * Utilities.
   *
   * These utility functions help in creation of the configuration form.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns an array of users and their access grants for a root item.
   *
   * The returned array has one entry per user. The array key is the
   * user's ID, and the array value is an array with one or more of:
   * - 'view'
   * - 'author'
   *
   * In normal use, the array for a user contains only one of these
   * possibilities:
   * - ['view'] = the user only has view access to the folder.
   * - ['view', 'author'] = the user has view and author access to the folder.
   *
   * While it is technically possible for a user to have 'author', but no
   * 'view' access, this is a strange and largely unusable configuration
   * and one this form does not support.
   *
   * @param \Drupal\foldershare\FolderShareInterface $root
   *   The root item object queried to get a list of users and their
   *   access grants.
   *
   * @return string[]
   *   The array of access grants for the folder.  Array keys are
   *   user IDs, while array values are arrays that contain values
   *   'view', and/or 'author'.
   *
   * @todo Reduce this function so that it only returns a list of users
   * with explicit view or author access to the folder. Do
   * not include all users at the site. However, this can only be done
   * when the form is updated to support add/delete users.
   */
  private function getAccessGrants(FolderShareInterface $root) {
    // Get the folder's access grants.
    $grants = $root->getAccessGrants();

    // For now, add all other site users and default them to nothing.
    foreach (\Drupal::entityQuery('user')->execute() as $uid) {
      if (isset($grants[$uid]) === FALSE) {
        $grants[$uid] = [];
      }
    }

    return $grants;
  }

  /*--------------------------------------------------------------------
   *
   * Execute.
   *
   *--------------------------------------------------------------------*/

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Always a selection that is a root item.
    $selectionIds = $this->getSelectionIds();
    $item = FolderShare::load(reset($selectionIds));

    try {
      $item->share($this->configuration['grants']);
    }
    catch (\Exception $e) {
      \Drupal::messenger()->addMessage($e->getMessage(), 'error');
    }

    // Flush the render cache because sharing may have changed what
    // content is viewable throughout the folder tree under this root.
    Cache::invalidateTags(['rendered']);

    if (Settings::getCommandNormalCompletionReportEnable() === TRUE) {
      \Drupal::messenger()->addMessage(
        t("The shared access configuration has been updated."),
        'status');
    }
  }

}
