<?php

namespace Drupal\foldershare\Plugin\FolderShareCommand;

use Drupal\Core\Url;

use Drupal\foldershare\Constants;

/**
 * Defines a command plugin to download files or folders.
 *
 * The command downloads a single selected entity.
 *
 * Configuration parameters:
 * - 'parentId': the parent folder, if any.
 * - 'selectionIds': selected entities to download.
 *
 * @ingroup foldershare
 *
 * @FolderShareCommand(
 *  id              = "foldersharecommand_download",
 *  label           = @Translation("Download"),
 *  menuNameDefault = @Translation("Download"),
 *  menuName        = @Translation("Download"),
 *  description     = @Translation("Download selected files and folders. Folders and lists of files are automatically downloaded as a newly-created ZIP archive."),
 *  category        = "import & export",
 *  weight          = 20,
 *  parentConstraints = {
 *    "kinds"   = {
 *      "rootlist",
 *      "any",
 *    },
 *    "access"  = "view",
 *  },
 *  selectionConstraints = {
 *    "types"   = {
 *      "parent",
 *      "one",
 *      "many",
 *    },
 *    "kinds"   = {
 *      "file",
 *      "image",
 *      "folder",
 *    },
 *    "access"  = "view",
 *  },
 * )
 */
class Download extends FolderShareCommandBase {

  /*---------------------------------------------------------------------
   *
   * Execute behavior.
   *
   *---------------------------------------------------------------------*/

  /**
   * {@inheritdoc}
   */
  public function getExecuteBehavior() {
    // Prior to (and thus instead of) executing, redirect to
    // another page.
    return self::PRE_EXECUTE_PAGE_REDIRECT;
  }

  /**
   * {@inheritdoc}
   */
  public function getExecuteRedirectUrl() {
    $ids = $this->getSelectionIds();
    if (empty($ids) === TRUE) {
      $ids[] = $this->getParentId();
    }

    // Redirect to the download controller, passing it the list of
    // entity IDs to download.
    return Url::fromRoute(
      Constants::ROUTE_DOWNLOAD,
      [
        'encoded' => implode(',', $ids),
      ]);
  }

  /*--------------------------------------------------------------------
   *
   * Execute.
   *
   *--------------------------------------------------------------------*/

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // Do nothing. This is never called because of the pre-execute redirect.
  }

}
