<?php

namespace Drupal\foldershare\Form\AdminSettingsTraits;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\StreamWrapper\PrivateStream;
use Drupal\Core\StringTranslation\TranslatableMarkup;

use Drupal\foldershare\Constants;
use Drupal\foldershare\Settings;
use Drupal\foldershare\Utilities;
use Drupal\foldershare\Entity\FolderShare;

/**
 * Manages the "Files" tab for the module's settings form.
 *
 * <B>Warning:</B> This is an internal trait that is strictly used by
 * the AdminSettings form class. It is a mechanism to group functionality
 * to improve code management.
 *
 * @ingroup foldershare
 */
trait AdminSettingsFilesTab {

  /*---------------------------------------------------------------------
   *
   * Build.
   *
   *---------------------------------------------------------------------*/

  /**
   * Builds the form tab.
   *
   * Settings:
   * - File system (public or private).
   * - Enable/disable file extension restrictions.
   * - Set a list of allowed file extensions.
   * - Reset the allowed file extensions list to the default.
   * - Set maximum upload file size.
   * - Set maxumum number of files per upload.
   * - Links to related module settings.
   *
   * @param array $form
   *   An associative array containing the structure of the form. The form
   *   is modified to include additional render elements for the tab.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The current state of the form.
   * @param string $tabGroup
   *   The name of the tab group.
   * @param string $tabMachineName
   *   The untranslated name for the tab, used in CSS class names.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $tabTitle
   *   The translated name for the tab.
   */
  private function buildFilesTab(
    array &$form,
    FormStateInterface $formState,
    string $tabGroup,
    string $tabMachineName,
    TranslatableMarkup $tabTitle) {

    //
    // Setup
    // -----
    // Set up some variables.
    $moduleName = Constants::MODULE;
    $tabName = self::makeCssSafe($moduleName . '_' . $tabMachineName . '_tab');
    $cssBase = self::makeCssSafe($moduleName);

    $sectionTitleClass       = $moduleName . '-settings-section-title';
    $sectionClass            = $moduleName . '-settings-section';
    $sectionDescriptionClass = $moduleName . '-settings-section-description';
    $sectionDefaultClass     = $moduleName . '-settings-section-default';
    $warningClass            = $moduleName . '-warning';

    $storageSchemeInput      = $moduleName . '_files_storage_scheme';
    $restrictExtensionsInput = $moduleName . '_files_restrict_extensions';
    $allowedExtensionsList   = $moduleName . '_files_allowed_extensions';
    $restoreExtensionsButton = $moduleName . '_files_restore_extensions';

    //
    // Create the tab
    // --------------
    // Start the tab with a title, subtitle, and description.
    $form[$tabName] = [
      '#type'           => 'details',
      '#open'           => FALSE,
      '#group'          => $tabGroup,
      '#title'          => $tabTitle,
      '#description'    => [
        'subtitle'      => [
          '#type'       => 'html_tag',
          '#tag'        => 'h2',
          '#value'      => $this->t(
            'Select how files are managed'),
          '#attributes' => [
            'class'     => [
              $cssBase . '-settings-subtitle',
            ],
          ],
        ],
        'description'   => [
          '#type'       => 'html_tag',
          '#tag'        => 'p',
          '#value'      => $this->t(
            "Files are stored on the site's server and tracked in the site's database. Administrators may restrict uploaded files based upon their name extension."),
          '#attributes' => [
            'class'     => [
              $cssBase . '-settings-description',
            ],
          ],
        ],
      ],
      '#attributes'     => [
        'class'         => [
          $cssBase . '-settings-tab ',
          $cssBase . '-fields-tab',
        ],
      ],
    ];

    //
    // Create warnings
    // ---------------
    // If the site currently has stored files, then the storage scheme
    // and subdirectory choice cannot be changed.
    //
    // If the site doesn't have stored files, but the private file system
    // has not been configured, then the storage scheme cannot be changed.
    $inUseWarning          = '';
    $noPrivateWarning      = '';
    $storageSchemeDisabled = FALSE;

    if (FolderShare::hasFiles() === TRUE) {
      // Create a link to the Drupal core system page to delete all content
      // for the entity type.
      $lnk = Link::createFromRoute(
        'Delete all',
        Constants::ROUTE_DELETEALL,
        [
          'entity_type_id' => FolderShare::ENTITY_TYPE_ID,
        ],
        []);

      $inUseWarning = [
        '#type'       => 'html_tag',
        '#tag'        => 'p',
        '#value'      => $this->t(
          'This setting is disabled because there are already files under management by this module. To change this setting, you must @deleteall files first.',
          [
            '@deleteall' => $lnk->toString(),
          ]),
        '#attributes' => [
          'class'     => [$warningClass],
        ],
      ];
      $storageSchemeDisabled = TRUE;
    }

    if (empty(PrivateStream::basePath()) === TRUE) {
      $noPrivateWarning = [
        '#type'       => 'html_tag',
        '#tag'        => 'p',
        '#value'      => $this->t(
          'This setting is disabled and restricted to the <em>Public</em> file system. To change this setting, you must configure the site to support a <em>Private</em> file system. See the @filedoc module\'s documentation and the "file_private_path" setting in the site\'s "settings.php" file.',
          [
            '@filedoc' => Utilities::createDocLink('file', 'File'),
          ]),
        '#attributes' => [
          'class'     => [$warningClass],
        ],
      ];
      $storageSchemeDisabled = TRUE;
    }

    //
    // Storage scheme
    // --------------
    // Select whether files are stored in the public or private file system.
    $options = [
      'public'  => $this->t('Public'),
      'private' => $this->t('Private'),
    ];

    // Add section title and description.
    $form[$tabName]['file-system'] = [
      '#type'            => 'item',
      '#markup'          => '<h3>' . $this->t('File system') . '</h3>',
      '#attributes'      => [
        'class'          => [$sectionTitleClass],
      ],
      'section'          => [
        '#type'          => 'container',
        '#attributes'    => [
          'class'        => [$sectionClass],
        ],
        'description'    => [
          '#type'        => 'html_tag',
          '#tag'         => 'p',
          '#value'       => $this->t(
            "Select whether the module should use the site's <em>Public</em> or <em>Private</em> file system to store files. A <em>Private</em> file system provides better security."),
          '#attributes'  => [
            'class'      => [$sectionDescriptionClass],
          ],
        ],
      ],
    ];

    // Add warning if there are things disabled.
    if (empty($noPrivateWarning) === FALSE) {
      $form[$tabName]['file-system']['section']['warning'] = $noPrivateWarning;
    }
    elseif (empty($inUseWarning) === FALSE) {
      $form[$tabName]['file-system']['section']['warning'] = $inUseWarning;
    }

    // Add label and widget.
    $form[$tabName]['file-system']['section'][$storageSchemeInput] = [
      '#type'          => 'select',
      '#options'       => $options,
      '#default_value' => Settings::getFileScheme(),
      '#required'      => TRUE,
      '#disabled'      => $storageSchemeDisabled,
      '#title'         => $this->t('File system:'),
      '#description'   => [
        '#type'        => 'html_tag',
        '#tag'         => 'p',
        '#value'       => $this->t(
          'Default: @default',
          [
            '@default'   => $options[Settings::getFileSchemeDefault()],
          ]),
        '#attributes'    => [
          'class'        => [$sectionDefaultClass],
        ],
      ],
    ];

    //
    // File name extensions
    // --------------------
    // Select whether file uploads should be restricted based on their
    // file name extensions, and what extensions are allowed.
    $form[$tabName]['fileextensions'] = [
      '#type'            => 'item',
      '#markup'          => '<h3>' . $this->t('File name extensions') . '</h3>',
      '#attributes'      => [
        'class'          => [$sectionTitleClass],
      ],
      'section'          => [
        '#type'          => 'container',
        '#attributes'    => [
          'class'        => [$sectionClass],
        ],
        'description'      => [
          '#type'          => 'html_tag',
          '#tag'           => 'p',
          '#value'         => $this->t(
            'Select whether uploaded files should be restricted to a specific set of allowed file name extensions.'),
          '#attributes'    => [
            'class'        => [$sectionDescriptionClass],
          ],
        ],
        $restrictExtensionsInput => [
          '#type'          => 'checkbox',
          '#title'         => $this->t('Enable restrictions'),
          '#default_value' => Settings::getFileRestrictExtensions(),
          '#return_value'  => 'enabled',
          '#required'      => FALSE,
          '#name'          => $restrictExtensionsInput,
        ],
        $allowedExtensionsList => [
          '#type'          => 'textarea',
          '#title'         => $this->t('Allowed file name extensions:'),
          '#default_value' => Settings::getAllowedNameExtensions(),
          '#required'      => FALSE,
          '#rows'          => 7,
          '#name'          => $allowedExtensionsList,
          '#description'   => $this->t(
            'Separate extensions with spaces and do not include dots. Changing this list does not affect files that are already on the site.'),
          '#states'        => [
            'invisible'    => [
              'input[name="' . $restrictExtensionsInput . '"]' => [
                'checked' => FALSE,
              ],
            ],
          ],
        ],
        $restoreExtensionsButton => [
          '#type'  => 'submit',
          '#value' => $this->t('Restore original configuration'),
          '#name'  => $restoreExtensionsButton,
          '#states'        => [
            'invisible'    => [
              'input[name="' . $restrictExtensionsInput . '"]' => [
                'checked' => FALSE,
              ],
            ],
          ],
        ],
      ],
    ];

    // Get the current extensions, if any, on the form.
    $value = $formState->getValue($allowedExtensionsList);
    if ($value !== NULL) {
      $form[$tabName]['fileextensions']['section'][$allowedExtensionsList]['#value'] = $value;
    }

    //
    // Related settings
    // ----------------
    // Add links for related settings.
    $this->buildFilesRelatedSettings($form, $formState, $tabGroup, $tabName);
  }

  /**
   * Builds the related settings section of the tab.
   *
   * @param array $form
   *   An associative array containing the structure of the form. The form
   *   is modified to include additional render elements for the tab.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The current state of the form.
   * @param string $tabGroup
   *   The name of the tab group.
   * @param string $tabName
   *   The CSS-read tab name.
   */
  private function buildFilesRelatedSettings(
    array &$form,
    FormStateInterface $formState,
    string $tabGroup,
    string $tabName) {

    $moduleName        = Constants::MODULE;
    $sectionTitleClass = $moduleName . '-settings-section-title';
    $seeAlsoClass      = $moduleName . '-settings-seealso-section';
    $relatedClass      = $moduleName . '-related-links';
    $markup            = '';

    //
    // Create links.
    // -------------
    // For each of several modules, create a link to the module's settings
    // page if the module is installed, and no link if it is not.
    //
    // Cron.
    $markup .= $this->buildRelatedSettings(
      // No specific module.
      '',
      'Cron',
      'system.cron_settings',
      [],
      "Configure the site's automatic administrative tasks.",
      TRUE,
      [
        'system'                => 'System',
        'automated-cron-module' => 'Automated Cron',
      ]);

    // File.
    $markup .= $this->buildRelatedSettings(
      // No specific module.
      '',
      'File',
      'system.file_system_settings',
      [],
      "Configure the site's <em>Public</em> and <em>Private</em> file systems.",
      TRUE,
      ['file' => 'File']);

    //
    // Add to form.
    // ------------
    // Add the links to the end of the form.
    //
    $form[$tabName]['related-settings'] = [
      '#type'            => 'details',
      '#title'           => $this->t('See also'),
      '#open'            => FALSE,
      '#summary_attributes' => [
        'class'          => [$sectionTitleClass],
      ],
      '#attributes'      => [
        'class'          => [$seeAlsoClass],
      ],
      'seealso'          => [
        '#type'          => 'item',
        '#markup'        => '<dl>' . $markup . '</dl>',
        '#attributes'    => [
          'class'        => [$relatedClass],
        ],
      ],
    ];
  }

  /*---------------------------------------------------------------------
   *
   * Validate.
   *
   *---------------------------------------------------------------------*/

  /**
   * Validates form values.
   *
   * @param array $form
   *   The form configuration.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The entered values for the form.
   */
  private function validateFilesTab(
    array &$form,
    FormStateInterface $formState) {

    // Setup
    // -----
    // Get user input.
    $moduleName = Constants::MODULE;
    $trigger = $formState->getTriggeringElement();

    //
    // Restore filename extensions.
    // ----------------------------
    // Check if the reset button was pressed.
    $allowedExtensionsList = $moduleName . '_files_allowed_extensions';
    $restoreExtensionsButton = $moduleName . '_files_restore_extensions';

    if ($trigger['#name'] === $restoreExtensionsButton) {
      // Revert extensions configuration.
      $value = Settings::getAllowedNameExtensionsDefault();

      // Immediately change the configuration and file field.
      Settings::setAllowedNameExtensions($value);
      FolderShare::setAllowedNameExtensions($value);

      // Reset the form's extensions list.
      $formState->setValue($allowedExtensionsList, $value);

      // Rebuild the page.
      $formState->setRebuild(TRUE);
      $formState->cleanValues();
      $formState->setTriggeringElement(NULL);

      \Drupal::messenger()->addMessage(
        t('The file extensions list has been restored to its original values.'),
        'status');
      return;
    }

    //
    // File scheme
    // -----------
    // It must be "public" or "private". If "private", the site's private
    // file system must have been configured.
    $storageSchemeInput = $moduleName . '_files_storage_scheme';

    $hasPrivate = empty(PrivateStream::basePath()) === FALSE;

    $scheme = $formState->getValue($storageSchemeInput);
    if ($scheme !== 'public' && $scheme !== 'private') {
      $formState->setErrorByName(
        $storageSchemeInput,
        $this->t('The file storage scheme must be either "Public" or "Private".'));
    }

    if ($scheme === 'private' && $hasPrivate === FALSE) {
      $formState->setErrorByName(
        $storageSchemeInput,
        $this->t('The "Private" file storage scheme is not available at this time because the website has not been configured to support it.'));
    }
  }

  /*---------------------------------------------------------------------
   *
   * Submit.
   *
   *---------------------------------------------------------------------*/

  /**
   * Stores submitted form values.
   *
   * @param array $form
   *   The form configuration.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The entered values for the form.
   */
  private function submitFilesTab(
    array &$form,
    FormStateInterface $formState) {

    $moduleName = Constants::MODULE;

    //
    // Set file scheme.
    // ----------------
    // Update file storage scheme if there are no files under management.
    if (FolderShare::hasFiles() === FALSE) {
      $storageSchemeInput = $moduleName . '_files_storage_scheme';
      Settings::setFileScheme($formState->getValue($storageSchemeInput));
    }

    //
    // File extensions
    // ---------------
    // Updated whether extensions are checked, and which ones
    // are allowed.
    $allowedExtensionsList = $moduleName . '_files_allowed_extensions';
    $restrictExtensionsInput = $moduleName . '_files_restrict_extensions';

    $value      = $formState->getValue($restrictExtensionsInput);
    $enabled    = ($value === 'enabled') ? TRUE : FALSE;
    $extensions = $formState->getValue($allowedExtensionsList);

    Settings::setFileRestrictExtensions($enabled);
    Settings::setAllowedNameExtensions($extensions);

    // If extensions are enabled, forward them to the file field.
    // Otherwise, clear the extension list on the file field.
    if ($enabled === TRUE) {
      FolderShare::setAllowedNameExtensions($extensions);
    }
    else {
      FolderShare::setAllowedNameExtensions('');
    }
  }

}
