<?php

namespace Drupal\foldershare\EventSubscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\PostResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

use Drupal\foldershare\Entity\FolderShareScheduledTask;

/**
 * Handles executing scheduled tasks after user requests are serviced.
 *
 * HTTP requests to the server are processed by Drupal and a response
 * posted back to the user. Upon completion, the user gets their next
 * page and a "terminate" event is sent. This class responds to such
 * events and uses them to service the scheduled task queue. Because
 * these events occur very often (once per page), the task queue is
 * serviced very frequently, which allows for fairly fine-grained task
 * scheduling.
 *
 * Task servicing is disabled automatically when the site is in maintenance
 * mode. This prevents activity that could interfer with maintenance
 * activity, such as installing modules or performing module updates.
 *
 * To register this event subscriber, an entry in "foldershare.services.yml"
 * is required:
 *
 * @code
 * services:
 *   foldershare.scheduledtask.subscriber:
 *     class: Drupal\foldershare\EventSubscriber\FolderShareScheduledTaskHandler
 *     arguments: []
 *     tags:
 *       - { name: event_subscriber }
 * @endcode
 *
 * @ingroup foldershare
 *
 * @see foldershare.services.yml
 * @see \Drupal\foldershare\Entity\FolderShareScheduledTask
 */
class FolderShareScheduledTaskHandler implements EventSubscriberInterface {

  /**
   * Constructs the handler.
   */
  public function __construct() {
  }

  /*---------------------------------------------------------------------
   *
   * Event handling.
   *
   *---------------------------------------------------------------------*/

  /**
   * Responds to a terminate event after a posted response.
   *
   * If the site is not in maintenance mode, the scheduled task list of
   * the FolderShare module is checked and ready tasks run.
   *
   * @param \Symfony\Component\HttpKernel\Event\PostResponseEvent $event
   *   The termination event.
   */
  public function onTerminate(PostResponseEvent $event) {
    $mode = \Drupal::state()->get('system.maintenance_mode');
    if ((int) $mode === 0) {
      // Invoke the task handler. The handler searches for any pending tasks
      // and executes them.
      $requestTime = (int) $event->getRequest()->server->get('REQUEST_TIME');
      FolderShareScheduledTask::executeTasks($requestTime);
    }
  }

  /**
   * Subscribes to the terminate event.
   *
   * @return array
   *   Returns an array of event listener definitions, including one that
   *   invokes this class's terminate event handler.
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::TERMINATE => [
        [
          'onTerminate',
          100,
        ],
      ],
    ];
  }

}
