<?php

namespace Drupal\foldershare\Entity\FolderShareTraits;

use Drupal\foldershare\Entity\Exception\LockException;

/**
 * Share Foldershare entities.
 *
 * This trait includes methods to share FolderShare entities.
 *
 * @section internal Internal trait
 * This trait is internal to the FolderShare module and used to define
 * features of the FolderShare entity class. It is a mechanism to group
 * functionality to improve code management.
 *
 * @ingroup foldershare
 */
trait OperationShareTrait {

  /*---------------------------------------------------------------------
   *
   * Share FolderShare entity.
   *
   *---------------------------------------------------------------------*/

  /**
   * {@inheritdoc}
   */
  public function share(array $grants) {
    if ($this->isRootItem() === FALSE) {
      // Not a root. Only roots have access grants.
      return;
    }

    //
    // Lock root folder tree.
    // ----------------------
    // Lock the folder tree rooted on this item. This prevents other operations
    // from modifying this item or its folder tree until grants are updated.
    //
    // LOCK ROOT FOLDER TREE.
    if (self::acquireRootOperationLock($this->id()) === FALSE) {
      throw new LockException(
        self::getStandardLockExceptionMessage(t('updated'), $this->getName()));
    }

    //
    // Update access grants.
    // ---------------------
    // Set the grants and save.
    $oldGrants = $this->getAccessGrants();

    $this->setAccessGrants($grants);
    $this->save();

    //
    // Unlock root folder tree.
    // ------------------------
    // After access grants are set, we're done.
    //
    // UNLOCK ROOT FOLDER TREE.
    self::releaseRootOperationLock($this->id());

    //
    // Hook & log.
    // -----------
    // Announce the change.
    self::postOperationHook(
      'share',
      [
        $this,
        $oldGrants,
        $grants,
      ]);
    self::log(
      'notice',
      'Changed sharing for entity @id ("%name"). <br>Old grants: %oldGrants. <br>New grants: %newGrants.',
      [
        '@id'        => $this->id(),
        '%name'      => $this->getName(),
        '%oldGrants' => self::accessGrantsToString($oldGrants),
        '%newGrants' => self::accessGrantsToString($grants),
        'link'       => $this->toLink(t('View'))->toString(),
      ]);
  }

  /*---------------------------------------------------------------------
   *
   * Unshare FolderShare entity.
   *
   *---------------------------------------------------------------------*/

  /**
   * {@inheritdoc}
   */
  public function unshare(int $uid, string $access = '') {
    if ($this->isRootItem() === FALSE) {
      // Not a root. Only roots have access grants.
      return;
    }

    //
    // Lock root folder tree.
    // ----------------------
    // Lock the folder tree rooted on this item. This prevents other operations
    // from modifying this item or its folder tree until grants are updated.
    //
    // LOCK ROOT FOLDER TREE.
    if (self::acquireRootOperationLock($this->id()) === FALSE) {
      throw new LockException(
        self::getStandardLockExceptionMessage(t('updated'), $this->getName()));
    }

    //
    // Update access grants.
    // ---------------------
    // Remove the indicated user from the access grants.
    $oldGrants = $this->getAccessGrants();

    if (empty($access) === TRUE) {
      $this->deleteAccessGrant($uid, 'view');
      $this->deleteAccessGrant($uid, 'author');
      $access = 'view & author';
    }
    else {
      $this->deleteAccessGrant($uid, $access);
    }

    $newGrants = $this->getAccessGrants();
    $this->save();

    //
    // Unlock root folder tree.
    // ------------------------
    // After access grants are set, we're done.
    //
    // UNLOCK ROOT FOLDER TREE.
    self::releaseRootOperationLock($this->id());

    //
    // Hook & log.
    // -----------
    // Announce the change.
    self::postOperationHook(
      'share',
      [
        $this,
        $oldGrants,
        $newGrants,
      ]);
    self::log(
      'notice',
      'Release sharing for entity @id ("%name") for user @uid to @access. <br>Old grants: %oldGrants. <br>New grants: %newGrants.',
      [
        '@id'        => $this->id(),
        '%name'      => $this->getName(),
        '@uid'       => $uid,
        '@access'    => $access,
        '%oldGrants' => self::accessGrantsToString($oldGrants),
        '%newGrants' => self::accessGrantsToString($newGrants),
        'link'       => $this->toLink(t('View'))->toString(),
      ]);
  }

  /**
   * Unshares multiple items.
   *
   * Each of the indicated items has its access grants adjusted to remove
   * the indicated user for shared access. The $access argument may be
   * 'view' or 'author', or left empty to unshare for both.
   *
   * @param int[] $ids
   *   An array of integer FolderShare entity IDs to unshare. Invalid IDs
   *   are silently skipped.
   * @param int $uid
   *   The user ID of the user to unshare for.
   * @param string $access
   *   The access grant to unshare. One of 'view' or 'author'. An empty
   *   string unshares for 'view' AND 'author'.
   *
   * @throws \Drupal\foldershare\Entity\Exception\LockException
   *   Throws an exception if this item cannot be locked for exclusive use.
   *
   * @section locking Process locks
   * This method locks each item's root folder tree for the duration of the
   * update. This will prevent any other edit operation from being performed
   * on the same folder tree until the update is done.
   *
   * @section hooks Post-operation hooks
   * This method calls the "hook_foldershare_post_operation_share" hook
   * after each item is updated.
   *
   * @section logging Operation log
   * This method posts a log message after each item has been updated.
   *
   * @see ::unshare()
   */
  public static function unshareMultiple(
    array $ids,
    int $uid,
    string $access = '') {

    $nLockExceptions = 0;
    foreach ($ids as $id) {
      $item = self::load($id);
      if ($item !== NULL) {
        try {
          $item->unshare($uid, $access);
        }
        catch (LockException $e) {
          ++$nLockExceptions;
        }
      }
    }

    if ($nLockExceptions !== 0) {
      throw new LockException(
        self::getStandardLockExceptionMessage(t('updated'), NULL));
    }
  }

}
