<?php

namespace Drupal\foldershare\Entity\FolderShareTraits;

use Drupal\foldershare\Constants;

/**
 * Manages hooks.
 *
 * This trait includes internal methods to invoke module hooks in a
 * standard way.
 *
 * @section internal Internal trait
 * This trait is internal to the FolderShare module and used to define
 * features of the FolderShare entity class. It is a mechanism to group
 * functionality to improve code management.
 *
 * @ingroup foldershare
 */
trait ManageHookTrait {

  /*---------------------------------------------------------------------
   *
   * Operation hooks.
   *
   *---------------------------------------------------------------------*/

  /**
   * Invokes a post-operation hook.
   *
   * <B>This method is internal and strictly for use by the FolderShare
   * module itself.</B> This method is public so that it can be called
   * from classes throughout the module.
   *
   * The $operationName argument gives the base name of an operation, such
   * as "delete" or "new_folder". This method converts the name to lower case,
   * and prefixes the name with "foldershare_post_operation_".
   *
   * The $item argument gives the FolderShare entity to which the hook applies.
   * This is provided to the hook as its only argument.
   *
   * @param string $operationName
   *   The name of the operation.
   * @param array|\Drupal\foldershare\FolderShareInterface $args
   *   (optional, default = NULL) Appropriate arguments for the hook, such
   *   as a FolderShare entity or an array of entity IDs.
   */
  public static function postOperationHook(
    string $operationName,
    $args = NULL) {

    try {
      // Convert the operation to lower case and create the hook name.
      // We assume the operation does not have any illegal method name
      // characters since the name is coming from authored code.
      $op = mb_convert_case($operationName, MB_CASE_LOWER);
      $hookName = Constants::MODULE . '_post_operation_' . $op;

      if ($args === NULL) {
        $args = [];
      }
      elseif (is_array($args) === FALSE) {
        // Arguments passed to invokeAll() must be an array. When the hook
        // is called, these are expanded into adjacent function arguments.
        $args = [$args];
      }

      // Invoke hooks.
      return \Drupal::moduleHandler()->invokeAll($hookName, $args);
    }
    catch (\Exception $e) {
      self::logStandardUnexpectedExceptionMessage(
        'FolderShare::postOperationHook',
        $e);
      return [];
    }
  }

}
