<?php

/**
 * @file
 * Implements drush commands specific to the module.
 */

use Drupal\Core\Cache\Cache;
use Drupal\user\Entity\User;

use Drupal\foldershare\Settings;
use Drupal\foldershare\Entity\FolderShare;
use Drupal\foldershare\Entity\FolderShareUsage;
use Drupal\foldershare\Entity\FolderShareScheduledTask;

/**
 * Implements hook_drush_command().
 */
function foldershare_drush_command() {
  $commands['foldershare'] = [
    'description' => 'List FolderShare module commands.',
    'aliases'     => [
      'fs',
    ],
    'options'     => [
      'all'      => [
        'description' => 'List all commands, including those for debugging.',
        'hidden'  => TRUE,
      ],
    ],
    'bootstrap'   => DRUSH_BOOTSTRAP_NONE,
    'category'    => 'San Diego Supercomputer Center (SDSC)',
    'topic'       => TRUE,
    'examples'    => [
      'drush foldershare' => 'List FolderShare commands',
    ],
  ];

  $commands['foldershare-disabled'] = [
    'description' => 'List, enable, and disable items marked as "disabled".',
    'hidden'      => TRUE,
    'aliases'     => [
      'fsdisabled',
    ],
    'arguments'   => [
      'id'        => [
        'description' => 'The entity ID to disable or enable',
      ],
    ],
    'options'     => [
      'disable'      => [
        'description' => 'Disable the item',
      ],
      'enable'    => [
        'description' => 'Enable an item',
      ],
      'enableall' => [
        'description' => 'Enable all disabled items',
      ],
      'list'      => [
        'description' => 'List all disabled items',
      ],
    ],
    'examples'    => [
      'drush foldershare-disabled' => 'List disabled items',
      'drush foldershare-disabled --list' => 'List disabled items',
      'drush foldershare-disabled --disable ID' => 'Disable item with entity ID',
      'drush foldershare-disabled --enable ID' => 'Enable item with entity ID',
      'drush foldershare-disabled --enableall' => 'Enable all items',
    ],
    'drupal dependencies' => [
      'foldershare',
    ],
    'bootstrap'   => DRUSH_BOOTSTRAP_DRUPAL_FULL,
    'category'    => 'San Diego Supercomputer Center (SDSC)',
  ];

  $commands['foldershare-fsck'] = [
    'description' => 'Find and fix file system problems.',
    'aliases'     => [
      'fsfsck',
    ],
    'options'     => [
      'fix'       => [
        'description' => 'Automatically fix problems',
      ],
    ],
    'examples'    => [
      'drush foldershare-fsck' => 'Find file system problems',
      'drush foldershare-fsck --fix' => 'Find and fix file system problems',
    ],
    'drupal dependencies' => [
      'foldershare',
    ],
    'bootstrap'   => DRUSH_BOOTSTRAP_DRUPAL_FULL,
    'category'    => 'San Diego Supercomputer Center (SDSC)',
  ];

  $commands['foldershare-hidden'] = [
    'description' => 'List, hide, or show items marked as "hidden".',
    'hidden'      => TRUE,
    'aliases'     => [
      'fshidden',
    ],
    'arguments'   => [
      'id'        => [
        'description' => 'The entity ID to hide or show',
      ],
    ],
    'options'     => [
      'hide'      => [
        'description' => 'Hide an item',
      ],
      'list'      => [
        'description' => 'List all hidden items',
      ],
      'show'      => [
        'description' => 'show an item',
      ],
      'showall'   => [
        'description' => 'show all hidden items',
      ],
    ],
    'examples'    => [
      'drush foldershare-hidden' => 'List hidden items',
      'drush foldershare-hidden --list' => 'List hidden items',
      'drush foldershare-hidden --hide ID' => 'Hide item with entity ID',
      'drush foldershare-hidden --show ID' => 'Show item with entity ID',
      'drush foldershare-hidden --showall' => 'Show all items',
    ],
    'drupal dependencies' => [
      'foldershare',
    ],
    'bootstrap'   => DRUSH_BOOTSTRAP_DRUPAL_FULL,
    'category'    => 'San Diego Supercomputer Center (SDSC)',
  ];

  $commands['foldershare-lock'] = [
    'description' => 'List, acquire, or release locks.',
    'hidden'      => TRUE,
    'aliases'     => [
      'fslock',
    ],
    'arguments'   => [
      'id'        => [
        'description' => "The root entity ID to lock or unlock",
      ],
    ],
    'options'     => [
      'list'       => [
        'description' => 'List root folder locks',
      ],
      'lock'       => [
        'description' => 'Lock the root folder of an item',
      ],
      'unlock'       => [
        'description' => 'Unlock the root folder of an item',
      ],
      'unlockall'       => [
        'description' => 'Unlock all root folders',
      ],
    ],
    'examples'    => [
      'drush foldershare-lock' => 'List locked root items',
      'drush foldershare-lock --list' => 'List locked root items',
      'drush foldershare-lock --lock ID' => 'Lock root item with entity ID',
      'drush foldershare-lock --unlock ID' => 'Unlock root item with entity ID',
      'drush foldershare-lock --unlockall' => 'Unlock all root items',
    ],
    'drupal dependencies' => [
      'foldershare',
    ],
    'bootstrap'   => DRUSH_BOOTSTRAP_DRUPAL_FULL,
    'category'    => 'San Diego Supercomputer Center (SDSC)',
  ];

  $commands['foldershare-config'] = [
    'description' => 'List module config.',
    'aliases'     => [
      'fsconfig',
    ],
    'examples'    => [
      'drush foldershare-config' => 'List configuration',
    ],
    'drupal dependencies' => [
      'foldershare',
    ],
    'bootstrap'   => DRUSH_BOOTSTRAP_DRUPAL_FULL,
    'category'    => 'San Diego Supercomputer Center (SDSC)',
  ];

  $commands['foldershare-tasks'] = [
    'description' => 'List, run, and delete module tasks.',
    'hidden'      => TRUE,
    'aliases'     => [
      'fstasks',
    ],
    'options'     => [
      'list'      => [
        'description' => 'List scheduled tasks',
      ],
      'delete'    => [
        'description' => 'Delete a task by its ID',
      ],
      'deleteall' => [
        'description' => 'Delete all tasks',
      ],
      'run'       => [
        'description' => 'Run tasks that are ready',
      ],
    ],
    'arguments'   => [
      'id'        => [
        'description' => "The ID of a task",
      ],
    ],
    'examples'    => [
      'drush foldershare-tasks' => 'List scheduled tasks',
      'drush foldershare-tasks --list' => 'List scheduled tasks',
      'drush foldershare-tasks --run' => 'Run ready scheduled tasks',
    ],
    'drupal dependencies' => [
      'foldershare',
    ],
    'bootstrap'   => DRUSH_BOOTSTRAP_DRUPAL_FULL,
    'category'    => 'San Diego Supercomputer Center (SDSC)',
  ];

  $commands['foldershare-usage'] = [
    'description' => 'List module usage by user.',
    'aliases'     => [
      'fsusage',
    ],
    'options'     => [
      'list'      => [
        'description' => 'List usage per user',
      ],
      'update'    => [
        'description' => 'Update the usage table',
      ],
    ],
    'examples'    => [
      'drush foldershare-usage' => 'List usage per user',
      'drush foldershare-usage --list' => 'List usage per user',
      'drush foldershare-usage --update' => 'Update the usage table',
    ],
    'drupal dependencies' => [
      'foldershare',
    ],
    'bootstrap'   => DRUSH_BOOTSTRAP_DRUPAL_FULL,
    'category'    => 'San Diego Supercomputer Center (SDSC)',
  ];

  $commands['foldershare-version'] = [
    'description' => 'Show module version number.',
    'aliases'     => [
      'fsversion',
    ],
    'examples'    => [
      'drush foldershare-version' => 'Show version number',
    ],
    'drupal dependencies' => [
      'foldershare',
    ],
    'bootstrap'   => DRUSH_BOOTSTRAP_DRUPAL_FULL,
    'category'    => 'San Diego Supercomputer Center (SDSC)',
  ];

  $commands['foldershare-benchmark'] = [
    'description' => 'Benchmark selected module features.',
    'hidden'      => TRUE,
    'aliases'     => [
      'fsbenchmark',
    ],
    'options'     => [
      'folder'    => [
        'description' => 'Benchmark folder create and delete',
      ],
      'hook'    => [
        'description' => 'Benchmark a hook search',
      ],
      'lock'    => [
        'description' => 'Benchmark process lock and release',
      ],
      'log'    => [
        'description' => 'Benchmark adding to the log',
      ],
      'config'    => [
        'description' => 'Benchmark get and set of configuration',
      ],
      'system'    => [
        'description' => 'Benchmark misc system calls',
      ],
    ],
    'drupal dependencies' => [
      'foldershare',
    ],
    'bootstrap'   => DRUSH_BOOTSTRAP_DRUPAL_FULL,
    'category'    => 'San Diego Supercomputer Center (SDSC)',
  ];

  return $commands;
}

/*---------------------------------------------------------------------
 *
 * Help.
 *
 *---------------------------------------------------------------------*/

/**
 * Lists available commands.
 */
function drush_foldershare() {
  FolderShareScheduledTask::setTaskExecutionEnabled(FALSE);

  $showAll = (empty(drush_get_option('all')) === FALSE);

  drush_print(
    dt('Commands for the FolderShare module:'));
  drush_print(sprintf(
    "  %-22s %s",
    "foldershare",
    dt("List these commands.")));
  if ($showAll === TRUE) {
    drush_print(sprintf(
      "  %-22s %s",
      "foldershare-benchmark",
      dt("Benchmark selected module features.")));
  }
  if ($showAll === TRUE) {
    drush_print(sprintf(
      "  %-22s %s",
      "foldershare-disabled",
      dt('List, enable, and disable items marked as "disabled".')));
  }
  drush_print(sprintf(
    "  %-22s %s",
    "foldershare-fsck",
    dt("Find and fix filesystem problems.")));
  if ($showAll === TRUE) {
    drush_print(sprintf(
      "  %-22s %s",
      "foldershare-hidden",
      dt('List, enable, and disable items marked as "hidden".')));
  }
  if ($showAll === TRUE) {
    drush_print(sprintf(
      "  %-22s %s",
      "foldershare-locks",
      dt('List, aquire, and release locks.')));
  }
  drush_print(sprintf(
    "  %-22s %s",
    "foldershare-config",
    dt("List module configuration.")));
  if ($showAll === TRUE) {
    drush_print(sprintf(
      "  %-22s %s",
      "foldershare-tasks",
      dt('List, run, and delete module tasks.')));
  }
  drush_print(sprintf(
    "  %-22s %s",
    "foldershare-usage",
    dt("List module usage by user.")));
  drush_print(sprintf(
    "  %-22s %s",
    "foldershare-version",
    dt("Show module version number.")));
}

/*---------------------------------------------------------------------
 *
 * Fix - disabled flags.
 *
 *---------------------------------------------------------------------*/

/**
 * Operates on disabled flags.
 *
 * --list       (default) List all hidden items.
 * --disable    Disable an item.
 * --enable     Enable an item.
 * --enableall  Enable all items.
 * --undisable  Enable an item.
 *
 * @param int $id
 *   (optional, default = (-1)) The FolderShare entity ID to enable or
 *   disable. The $id is ignored if the 'list' option is used, and
 *   required for the 'enable' and 'disable' options.
 */
function drush_foldershare_disabled($id = (-1)) {
  FolderShareScheduledTask::setTaskExecutionEnabled(FALSE);

  $doList      = TRUE;
  $doDisable   = FALSE;
  $doEnableAll = FALSE;

  if (empty(drush_get_option('list')) === TRUE) {
    if (empty(drush_get_option('enable')) === FALSE ||
        empty(drush_get_option('undisable')) === FALSE) {
      $doList = FALSE;
    }
    elseif (empty(drush_get_option('disable')) === FALSE) {
      $doList = FALSE;
      $doDisable = TRUE;
    }
    elseif (empty(drush_get_option('enableall')) === FALSE) {
      $doList = FALSE;
      $doEnableAll = TRUE;
    }
  }

  //
  // List disabled items.
  // --------------------
  // Get a list of all disabled items and list them in a table.
  if ($doList === TRUE) {
    drush_print(dt('Disabled files and folders:'));
    drush_print(sprintf(
      "%-20s %10s %s",
      'Owner',
      'Entity ID',
      'Path'));
    drush_print(sprintf(
      "%-20s %10s %s",
      '--------------------',
      '----------',
      '---------------------------------------------'));

    $disabledIds = FolderShare::findAllDisabledIds();
    foreach ($disabledIds as $id) {
      $item = FolderShare::load($id);
      if ($item === NULL) {
        // The item does not exist.
        continue;
      }

      $ownerId = $item->getOwnerId();
      $path = $item->getPath();
      if ($item->isFolder() === TRUE) {
        $path .= '/';
      }

      $u = User::load($ownerId);
      if ($u === NULL) {
        $ownerName = sprintf('%s (%d)', dt('unknown'), $ownerId);
      }
      else {
        $ownerName = sprintf('%s (%d)', $u->getDisplayName(), $ownerId);
      }

      drush_print(sprintf(
        "%-20s %10d %s",
        $ownerName,
        $id,
        $path));
    }

    return;
  }

  //
  // Enable all items.
  // -----------------
  // Find and enable all disabled items, then clear caches.
  if ($doEnableAll === TRUE) {
    drush_print(dt("Enable all disabled items"));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt("Items marked disabled are being used by operations such as a copy or"));
    drush_print(dt("move. The mark also is used by copy operations to keep track of their"));
    drush_print(dt("progress. Manually enabling all disabled items will cause them to"));
    drush_print(dt("become available for new operations and corrupt marks used for copy"));
    drush_print(dt("operations already in progress. This can cause operations to collide"));
    drush_print(dt("and make copy operations end prematurely."));
    drush_print("");
    drush_print(dt("Enabling all items manually should only be done during debugging."));
    drush_print("");

    $answer = readline(dt("Are you sure [y|n]? "));
    if ($answer[0] === 'n') {
      drush_print(dt("Aborted."));
      return;
    }

    drush_print(dt("Enabling all disabled items. This may take a few minutes..."));
    $nChanged = FolderShare::clearAllSystemDisabled();

    // Flush the entity cache so that it is not out of date compared to
    // the now-marked entities in the database.
    \Drupal::entityTypeManager()->getStorage(FolderShare::ENTITY_TYPE_ID)
      ->resetCache(NULL);

    \Drupal::service('cache.render')->invalidateAll();
    if ($nChanged === 0) {
      drush_print(dt("Done. No items were changed."));
    }
    else {
      drush_print(dt(
        "Done. @count items were changed.",
        [
          '@count' => $nChanged,
        ]));
    }
    return;
  }

  //
  // Enable or disable.
  // ------------------
  // Use the ID argument and load the entity to change.
  $id = (int) $id;
  if ($id < 0) {
    drush_print(dt('Missing entity ID to enable or disable.'));
    return;
  }

  if ($doDisable === TRUE) {
    drush_print(dt("Disable an item"));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt("Disabling an item will prevent users, including administrators, from"));
    drush_print(dt("starting new operations using it. This includes viewing, downloading,,"));
    drush_print(dt("moving, copying, and deleting. The item will remain visible in lists"));
    drush_print(dt("and show a spinner. Normally the spinner goes away when an operation"));
    drush_print(dt("enables the item after it completes. But with a manually disabled item,"));
    drush_print(dt("there is no associated operation to enable item it again. The item will"));
    drush_print(dt("remain disabled and the spinner will run indefinitely. This is likely to"));
    drush_print(dt("cause user confusion."));
    drush_print("");
    drush_print(dt("Disabling an item manually should only be done during debugging."));
    drush_print("");
  }
  else {
    drush_print(dt("Enable an item"));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt("Items marked disabled are being used by operations such as a copy or"));
    drush_print(dt("move. The mark also is used by copy operations to keep track of their"));
    drush_print(dt("progress. Manually enabling a disabled item will cause it to become"));
    drush_print(dt("available for new operations that can collide if a prior operation"));
    drush_print(dt("is still in progres. This also can cause copy operations to end"));
    drush_print(dt("prematurely and leave some content uncopied. The premature end will"));
    drush_print(dt("not alert the user that the copied item is invalid."));
    drush_print("");
    drush_print(dt("Enabling an item manually should only be done during debugging."));
    drush_print("");
  }

  $answer = readline(dt("Are you sure [y|n]? "));
  if ($answer[0] === 'n') {
    drush_print(dt("Aborted."));
    return;
  }

  $item = FolderShare::load($id);
  if ($item === NULL) {
    drush_print(dt(
      'Invalid entity ID: @id',
      [
        '@id' => $id,
      ]));
    return;
  }

  $path = $item->getPath();
  if ($item->isFolder() === TRUE) {
    $path .= '/';
  }

  if ($doDisable === TRUE) {
    // Disable the item.
    drush_print(dt(
      'Disabling ID @id at @path...',
      [
        '@id'   => $id,
        '@path' => $path,
      ]));
    $item->setSystemDisabled(TRUE);
    $item->save();
  }
  else {
    // Enable the item.
    drush_print(dt(
      'Enabling ID @id at @path...',
      [
        '@id'   => $id,
        '@path' => $path,
      ]));
    $item->setSystemDisabled(FALSE);
    $item->save();
  }

  drush_print(dt("Done. One item was changed."));
}

/*---------------------------------------------------------------------
 *
 * Fix - hidden flags.
 *
 *---------------------------------------------------------------------*/

/**
 * Operates on hidden flags.
 *
 * --list       (default) List all hidden items.
 * --hide       Hide a non-hidden item.
 * --show       Show a hidden item.
 * --showall    Show all hidden items.
 * --unhide     Show a hidden item.
 *
 * @param int $id
 *   (optional, default = (-1)) The FolderShare entity ID to show or
 *   hide. The $id is ignored if the 'list' option is used, and
 *   required for the 'show' and 'hide' options.
 */
function drush_foldershare_hidden($id = (-1)) {
  FolderShareScheduledTask::setTaskExecutionEnabled(FALSE);

  $doList    = TRUE;
  $doHide    = FALSE;
  $doShowAll = FALSE;

  if (empty(drush_get_option('list')) === TRUE) {
    if (empty(drush_get_option('show')) === FALSE ||
        empty(drush_get_option('unhide')) === FALSE) {
      $doList = FALSE;
    }
    elseif (empty(drush_get_option('hide')) === FALSE) {
      $doList = FALSE;
      $doHide = TRUE;
    }
    elseif (empty(drush_get_option('showall')) === FALSE) {
      $doList = FALSE;
      $doShowAll = TRUE;
    }
  }

  //
  // List hidden items.
  // ------------------
  // Get a list of all hidden items and list them in a table.
  if ($doList === TRUE) {
    drush_print(dt('Hidden files and folders:'));
    drush_print(sprintf(
      "%-20s %10s %s",
      dt('Owner'),
      dt('Entity ID'),
      dt('Path')));
    drush_print(sprintf(
      "%-20s %10s %s",
      '--------------------',
      '----------',
      '---------------------------------------------'));

    $hiddenIds = FolderShare::findAllHiddenIds();
    foreach ($hiddenIds as $id) {
      $item = FolderShare::load($id);
      if ($item === NULL) {
        // The item does not exist.
        continue;
      }

      $ownerId = $item->getOwnerId();
      $path = $item->getPath();
      if ($item->isFolder() === TRUE) {
        $path .= '/';
      }

      $u = User::load($ownerId);
      if ($u === NULL) {
        $ownerName = sprintf('%s (%d)', dt('unknown'), $ownerId);
      }
      else {
        $ownerName = sprintf('%s (%d)', $u->getDisplayName(), $ownerId);
      }

      drush_print(sprintf(
        "%-20s %10d %s",
        $ownerName,
        $id,
        $path));
    }

    return;
  }

  //
  // Show all items.
  // ---------------
  // Find and marks all hidden items as shown.
  if ($doShowAll === TRUE) {
    drush_print(dt("Show all items"));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt("Items marked hidden are in the process of being deleted. Manually"));
    drush_print(dt("marking all hidden items as visible will let users see the items as"));
    drush_print(dt("they are being deleted, but it will not abort the delete. This can be"));
    drush_print(dt("confusing to users and it can cause problems if a user tries to use"));
    drush_print(dt("an item just as it is being deleted."));
    drush_print("");
    drush_print(dt("Showing hidden items should only be done during debugging."));
    drush_print("");

    $answer = readline(dt("Are you sure [y|n]? "));
    if ($answer[0] === 'n') {
      drush_print(dt("Aborted."));
      return;
    }

    drush_print(dt("Showing all hidden items..."));
    $nChanged = FolderShare::clearAllSystemHidden();

    // Flush the entity cache so that it is not out of date compared to
    // the now-marked entities in the database.
    \Drupal::entityTypeManager()->getStorage(FolderShare::ENTITY_TYPE_ID)
      ->resetCache(NULL);

    \Drupal::service('cache.render')->invalidateAll();

    if ($nChanged === 0) {
      drush_print(dt("Done. No items were changed."));
    }
    else {
      drush_print(dt(
        "Done. @count items were changed.",
        [
          '@count' => $nChanged,
        ]));
    }
    return;
  }

  //
  // Show or hide.
  // -------------
  // Use the ID argument and load the entity to change.
  if ($id < 0) {
    drush_print(dt('Missing entity ID to show or hide.'));
    return;
  }

  if ($doHide === TRUE) {
    drush_print(dt("Hide an item"));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt("Items marked hidden are in the process of being deleted. Manually"));
    drush_print(dt("marking an item hidden will make it invisible to users, but will not"));
    drush_print(dt("delete it."));
    drush_print("");
    drush_print(dt("Hiding items should only be done during debugging."));
    drush_print("");
  }
  else {
    drush_print(dt("Show an item"));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt("Items marked hidden are in the process of being deleted. Manually"));
    drush_print(dt("marking a hidden item as visible will let users see the item before"));
    drush_print(dt("it is deleted, but it will not abort the delete. This can be confusing"));
    drush_print(dt("to users and it can cause problems if a user tries to use an item just"));
    drush_print(dt("as it is being deleted."));
    drush_print("");
    drush_print(dt("Showing hidden items should only be done during debugging."));
    drush_print("");
  }

  $answer = readline(dt("Are you sure [y|n]? "));
  if ($answer[0] === 'n') {
    drush_print(dt("Aborted."));
    return;
  }

  $item = FolderShare::load($id);
  if ($item === NULL) {
    drush_print(dt(
      'Invalid entity ID: @id',
      [
        '@id' => $id,
      ]));
    return;
  }

  $path = $item->getPath();
  if ($item->isFolder() === TRUE) {
    $path .= '/';
  }

  if ($doHide === TRUE) {
    // Hide the item.
    drush_print(dt(
      'Hiding ID @id at @path...',
      [
        '@id'   => $id,
        '@path' => $path,
      ]));
    $item->setSystemHidden(TRUE);
    $item->save();
  }
  else {
    // Show the item.
    drush_print(dt(
      'Showing ID @id at @path...',
      [
        '@id'   => $id,
        '@path' => $path,
      ]));
    $item->setSystemHidden(FALSE);
    $item->save();
  }

  drush_print(dt("Done. One item was changed."));
}

/*---------------------------------------------------------------------
 *
 * Fix - root locks.
 *
 *---------------------------------------------------------------------*/

/**
 * Operates on root locks.
 *
 * --list       (default) List all root locks.
 * --lock       Lock a root.
 * --unlock     Unlock a root.
 */
function drush_foldershare_lock($id = (-1)) {
  FolderShareScheduledTask::setTaskExecutionEnabled(FALSE);

  $doList = TRUE;
  $doLock = FALSE;

  if (empty(drush_get_option('lock')) === FALSE) {
    $doList = FALSE;
    $doLock = TRUE;
  }
  elseif (empty(drush_get_option('unlock')) === FALSE) {
    $doList = FALSE;
  }

  //
  // Show all locked items.
  // ----------------------
  // Only root items can be locked, so get all roots and count those
  // that are locked.
  if ($doList === TRUE) {
    drush_print(dt('Locked root items:'));
    drush_print(sprintf(
      "%-20s %10s %s",
      dt('Owner'),
      dt('Root ID'),
      dt('Path')));
    drush_print(sprintf(
      "%-20s %10s %s",
      '--------------------',
      '----------',
      '---------------------------------------------'));

    $rootIds = FolderShare::findAllRootItemIds();
    $lockedRootIds = [];
    foreach ($rootIds as $rootId) {
      if (FolderShare::isRootOperationLockAvailable($rootId) === FALSE) {
        $lockedRootIds[] = $rootId;
      }
    }

    if (count($lockedRootIds) === 0) {
      // Nothing to list.
      return;
    }

    foreach ($lockedRootIds as $rootId) {
      $item = FolderShare::load($rootId);
      if ($item === NULL) {
        // The item does not exist.
        continue;
      }

      $ownerId = $item->getOwnerId();
      $path = $item->getPath();
      if ($item->isFolder() === TRUE) {
        $path .= '/';
      }

      $u = User::load($ownerId);
      if ($u === NULL) {
        $ownerName = sprintf('%s (%d)', dt('unknown'), $ownerId);
      }
      else {
        $ownerName = sprintf('%s (%d)', $u->getDisplayName(), $ownerId);
      }

      drush_print(sprintf(
        "%-20s %10d %s",
        $ownerName,
        $id,
        $path));
    }

    return;
  }

  //
  // Lock or unlock.
  // ---------------
  // Use the ID argument and load the entity to change.
  if ($id < 0) {
    drush_print(dt('Missing root entity ID to lock or unlock.'));
    return;
  }

  if ($doLock === TRUE) {
    drush_print(dt("Lock item"));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt("Locking a root item will block all further operations on the entire folder"));
    drush_print(dt("tree under the root. Create, copy, delete, move, and other operations"));
    drush_print(dt("will fail for all users, including administrators."));
    drush_print("");
    drush_print(dt("Locking manually should only be done during debugging."));
    drush_print("");
  }
  else {
    drush_print(dt("Unlock item"));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt("Unlocking a root item will allow new operations to be started anywhere"));
    drush_print(dt("within the folder tree under the root. If there is a pending operation"));
    drush_print(dt("on anything in the folder tree, a new operation can collide by attempting"));
    drush_print(dt("attempting to change the same files and folders. This can produce"));
    drush_print(dt("unexpected results for the user or even corrupt the file system."));
    drush_print("");
    drush_print(dt("Unlocking manually should only be done during debugging."));
    drush_print("");
  }

  $answer = readline(dt("Are you sure [y|n]? "));
  if ($answer[0] === 'n') {
    drush_print(dt("Aborted."));
    return;
  }

  $item = FolderShare::load($id);
  if ($item === NULL) {
    drush_print(dt(
      'Invalid entity ID: @id',
      [
        '@id' => $id,
      ]));
    return;
  }

  if ($item->isRootItem() === FALSE) {
    drush_print(dt(
      'Entity ID is not a root item: @id',
      [
        '@id' => $id,
      ]));
    return;
  }

  $path = $item->getPath();
  if ($item->isFolder() === TRUE) {
    $path .= '/';
  }

  if ($doLock === TRUE) {
    // Lock the item.
    drush_print(dt(
      'Lock root ID @id @path',
      [
        '@id'   => $id,
        '@path' => $path,
      ]));
    FolderShare::acquireRootOperationLock($id);
  }
  else {
    // Unlock the item.
    drush_print(dt(
      'Unlock root ID @id @path',
      [
        '@id'   => $id,
        '@path' => $path,
      ]));
    FolderShare::releaseRootOperationLock($id);
  }

  drush_print("");
  drush_print(dt("Done. One lock was changed."));
}

/*---------------------------------------------------------------------
 *
 * Fix - file system check.
 *
 *---------------------------------------------------------------------*/

/**
 * Checks the file system.
 *
 * --fix        Run file system checks.
 */
function drush_foldershare_fsck() {
  FolderShareScheduledTask::setTaskExecutionEnabled(FALSE);

  $fix = (empty(drush_get_option('fix')) === FALSE);

  // Scan and possibly fix the file system.
  if ($fix === TRUE) {
    drush_print(dt('File system check and repair'));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt('This may take a few minutes.'));
    drush_print("");

    $originalMode = \Drupal::state()->get('system.maintenance_mode');
    if ($originalMode !== 1) {
      drush_print(dt('Entering maintenance mode.'));
      \Drupal::state()->set('system.maintenance_mode', 1);
      drush_print("");
    }

    $wereFixed = FolderShare::fsck(TRUE);

    if ($originalMode !== 1) {
      drush_print(dt('Exiting maintenance mode.'));
      \Drupal::state()->set('system.maintenance_mode', $originalMode);
      drush_print("");
    }

    if ($wereFixed === TRUE) {
      drush_print(dt('Repairs were made.'));
    }
    else {
      drush_print(dt('No repairs were needed or made.'));
    }
  }
  else {
    drush_print(dt('File system check only'));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt('This may take a few minutes.'));
    drush_print("");

    $needToBeFixed = FolderShare::fsck(FALSE);

    if ($needToBeFixed === TRUE) {
      drush_print(dt('Repairs need to be made.'));
      drush_print(dt('To make repairs, run "drush foldershare-fsck --fix".'));
    }
    else {
      drush_print(dt('No repairs are needed.'));
    }
  }
}

/*---------------------------------------------------------------------
 *
 * Module configuration - configuration.
 *
 *---------------------------------------------------------------------*/

/**
 * Lists module configuration.
 *
 * The module's configuration is listed.
 *
 * The configuration for the module's core search and REST web services
 * plugins are also listed.
 */
function drush_foldershare_config() {
  FolderShareScheduledTask::setTaskExecutionEnabled(FALSE);

  $moduleHandler = \Drupal::moduleHandler();
  $typeManager = \Drupal::entityTypeManager();

  drush_print(dt('Module configuration:'));

  //
  // Files.
  drush_print(dt('Files:'));
  drush_print('------------------------------------------------------------------------');
  drush_print(sprintf("%-45s %s",
    dt('File system:'),
    Settings::getFileScheme()));
  drush_print(sprintf("%-45s %s",
    dt('New ZIP archive name:'),
    Settings::getNewZipArchiveName()));
  drush_print(sprintf("%-45s %s",
    dt('New ZIP archive comment:'),
    Settings::getNewZipArchiveComment()));
  drush_print(sprintf("%-45s %s",
    dt('ZIP unarchive to subfolder:'),
    (Settings::getZipUnarchiveMultipleToSubfolder() === TRUE) ? 'true' : 'false'));
  drush_print(sprintf("%-45s %s",
    dt('File name extensions restricted:'),
    (Settings::getFileRestrictExtensions() === TRUE) ? 'true' : 'false'));
  drush_print(sprintf("%-45s\n%s",
    dt('Allowed file name extensions (if restricted):'),
    Settings::getAllowedNameExtensions()));
  drush_print("");

  //
  // Interface.
  drush_print(dt('User interface:'));
  drush_print('------------------------------------------------------------------------');
  drush_print(sprintf("%-45s %s",
    dt('Command menu restricted:'),
    (Settings::getCommandMenuRestrict() === TRUE) ? 'true' : 'false'));
  drush_print(sprintf("%-45s\n%s",
    dt('Allowed commands (if restricted):'),
    implode(', ', Settings::getCommandMenuAllowed())));
  drush_print(sprintf("%-45s %s",
    dt('Commands report normal completion:'),
    (Settings::getCommandNormalCompletionReportEnable() === TRUE) ? 'true' : 'false'));
  drush_print(sprintf("%-45s %d sec",
    dt('Interface refresh polling interval:'),
    Settings::getStatusPollingInterval()));
  drush_print("");

  //
  // Search.
  drush_print(dt('Search:'));
  drush_print('------------------------------------------------------------------------');

  if ($moduleHandler->moduleExists('search') === FALSE) {
    drush_print(dt('Drupal core "search" module is not installed.'));
  }
  else {
    try {
      $searchPage = $typeManager->getStorage('search_page')
        ->load('foldershare_search');
      $searchPlugin = $searchPage->getPlugin();

      drush_print(sprintf("%-45s %s",
        dt('Include file text in search index:'),
        ($searchPlugin->getSearchFileContent() === TRUE) ? 'true' : 'false'));
      drush_print(sprintf("%-45s %d bytes",
        dt('Maximum file content to index for search:'),
        $searchPlugin->getSearchFileSize()));
      drush_print(sprintf("%-45s %s",
        dt('File types to include in search index:'),
        $searchPlugin->getSearchFileExtensions()));
    }
    catch (\Exception $e) {
      // Ignore.
    }
  }
  drush_print("");

  //
  // System.
  drush_print(dt('System:'));
  drush_print('------------------------------------------------------------------------');
  drush_print(sprintf("%-45s %s",
    dt('Activity logging:'),
    (Settings::getActivityLogEnable() === TRUE) ? 'true' : 'false'));
  drush_print(sprintf("%-45s %s",
    dt('Usage table rebuild interval:'),
    Settings::getUsageReportRebuildInterval()));

  $t = Settings::getUsageReportTime();
  if ($t !== 'never') {
    if ($t[0] === '@') {
      $t = date('Y-m-d H:i:s', intval(substr($t, 1)));
    }
    else {
      $t = date('Y-m-d H:i:s', intval($t));
    }
  }
  drush_print(sprintf("%-45s %s",
    dt('Usage table last rebuilt:'),
    $t));
  drush_print(sprintf("%-45s %d sec",
    dt('Content lock duration:'),
    Settings::getContentLockDuration()));
  drush_print(sprintf("%-45s %d sec",
    dt('Operation lock duration:'),
    Settings::getOperationLockDuration()));
  drush_print(sprintf("%-45s %d sec",
    dt('Task initial scheduling delay:'),
    Settings::getScheduledTaskInitialDelay()));
  drush_print(sprintf("%-45s %d sec",
    dt('Task continuation scheduling delay:'),
    Settings::getScheduledTaskContinuationDelay()));
  drush_print(sprintf("%-45s %d sec",
    dt('Task safety-net scheduling delay:'),
    Settings::getScheduledTaskSafetyNetDelay()));
  drush_print(sprintf("%-45s %4.1f %%",
    dt('Memory use limit as % of PHP limit:'),
    100.0 * Settings::getMemoryUseLimitPercentage()));
  drush_print(sprintf("%-45s %4.1f %%",
    dt('Execution time limit as % of PHP limit:'),
    100.0 * Settings::getExecutionTimeLimitPercentage()));
  drush_print("");

  //
  // Web services.
  drush_print(dt('Web services:'));
  drush_print('------------------------------------------------------------------------');
  if (\Drupal::moduleHandler()->moduleExists('rest') === FALSE) {
    drush_print(dt('Drupal core "rest" module not installed.'));
  }
  else {
    try {
      $restPlugin = $typeManager->getStorage('rest_resource_config')
        ->load('entity.foldershare');

      foreach ($restPlugin->getMethods() as $method) {
        $formats = $restPlugin->getFormats($method);
        $auths = $restPlugin->getAuthenticationPRoviders($method);

        drush_print($method);
        drush_print(sprintf("  %-43s %s",
          dt("Serialization formats:"),
          implode(', ', $formats)));
        drush_print(sprintf("  %-43s %s",
          dt("Authentication providers:"),
          implode(', ', $auths)));
      }
    }
    catch (\Exception $e) {
      // Ignore.
    }
  }

  drush_print("");

  //
  // PHP directives.
  drush_print(dt('Relevant PHP directives:'));
  drush_print('------------------------------------------------------------------------');
  drush_print(sprintf("%-45s %s",
    dt('Max upload file size (upload_max_filesize):'),
    ini_get('upload_max_filesize')));
  drush_print(sprintf("%-45s %d",
    dt('Max # of uploaded files (max_file_uploads):'),
    ini_get('max_file_uploads')));

  $t = ini_get('post_max_size');
  if ($t === "0") {
    drush_print(sprintf("%-45s %s",
      dt('Max post message size (post_max_size):'),
      dt('(unlimited)')));
  }
  else {
    drush_print(sprintf("%-45s %s",
      dt('Max post message size (post_max_size):'),
      $t));
  }

  $t = (int) ini_get('max_input_time');
  if ($t === (-1)) {
    drush_print(sprintf("%-45s %s",
      dt('Max post processing time (max_input_time):'),
      dt('(uses max_execution_time)')));
  }
  else {
    drush_print(sprintf("%-45s %d sec",
      dt('Max post processing time (max_input_time):'),
      $t));
  }

  $t = ini_get('memory_limit');
  if ($t === "-1") {
    drush_print(sprintf("%-45s %s",
      dt('Max memory use (memory_limit):'),
      dt('(unlimited)')));
  }
  else {
    drush_print(sprintf("%-45s %s",
      dt('Max memory use (memory_limit):'),
      $t));
  }

  // Drush runs from the command line interpreter, and that always sets the
  // execution time to 0. There is no way to get the value a web-invoked
  // PHP would see.
  drush_print(sprintf("%-45s %s",
    dt('Max execution time (max_execution_time):'),
    dt("(unavailable via drush)")));
}

/*---------------------------------------------------------------------
 *
 * Module configuration - tasks.
 *
 *---------------------------------------------------------------------*/

/**
 * Operates on scheduled tasks.
 *
 * --list       (default) List all scheduled task entities.
 * --run        Run the scheduled task executor.
 * --delete     Delete the indicated task.
 * --deleteall  Delete all scheduled tasks.
 *
 * @param int $id
 *   (optional, default = (-1)) The task entity ID to delete.
 */
function drush_foldershare_tasks($id = -1) {
  //
  // Run tasks.
  // ----------
  // Invoke the task executor.
  if (empty(drush_get_option('run')) === FALSE) {
    drush_print(dt('Executing ready tasks...'));
    FolderShareScheduledTask::setTaskExecutionEnabled(TRUE);
    FolderShareScheduledTask::executeTasks(time());
    return;
  }

  FolderShareScheduledTask::setTaskExecutionEnabled(FALSE);

  //
  // Delete all tasks.
  // -----------------
  // Delete every task entity. This can be dangerous since pending tasks
  // may be in the midst of updating content, such as changing root IDs
  // after a move, or finishing with copy, delete, or change ownership
  // operations.
  if (empty(drush_get_option('deleteall')) === FALSE) {
    $n = FolderShareScheduledTask::findNumberOfTasks();
    if ($n === 0) {
      drush_print(dt("There are currently no tasks to delete."));
      return;
    }

    drush_print(dt("Delete all tasks"));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt("Deleting all tasks will leave pending operations incomplete. This can"));
    drush_print(dt("corrupt the file system - sometimes catastrophically. Deleting all tasks"));
    drush_print(dt("should be done only when there is something deeply wrong and you are"));
    drush_print(dt("very sure that deleting all tasks is the best thing to do. This is"));
    drush_print(dt("likely to cause unexpected and possibly destructive side-effects."));
    drush_print("");
    $answer = readline(dt("Are you sure [y|n]? "));
    if ($answer[0] === 'n') {
      drush_print(dt("Aborted."));
      return;
    }

    drush_print(dt("Deleting all tasks..."));
    FolderShareScheduledTask::deleteAllTasks();

    drush_print("");
    drush_print(dt("Done. You should run 'drush foldershare-fsck' immediately to check for file"));
    drush_print(dt("system corruption."));
    return;
  }

  //
  // Delete task.
  // ------------
  // Delete one task, if an ID was given.
  if (empty(drush_get_option('delete')) === FALSE) {
    $id = (int) $id;
    if ($id < 0) {
      drush_print(dt('Missing task ID to delete.'));
      return;
    }

    drush_print(dt("Delete task"));
    drush_print('------------------------------------------------------------------------');
    drush_print(dt("Deleting a task will leave a pending operation incomplete. This can"));
    drush_print(dt("corrupt the file system - sometimes catastrophically. Deleting a task"));
    drush_print(dt("should be done only when there is something deeply wrong and you are"));
    drush_print(dt("very sure that deleting the task is the best thing to do. This is"));
    drush_print(dt("likely to cause unexpected and possibly destructive side-effects."));
    drush_print("");
    $answer = readline(dt("Are you sure [y|n]? "));
    if ($answer[0] === 'n') {
      drush_print(dt("Aborted."));
      return;
    }

    drush_print(dt("Deleting task @id...", ['@id' => $id]));
    $task = FolderShareScheduledTask::load($id);
    if ($task === NULL) {
      drush_print(dt("Invalid task ID."));
      return;
    }

    $task->delete();

    drush_print("");
    drush_print(dt("Done. You should run 'drush foldershare-fsck' immediately to check for file"));
    drush_print(dt("system corruption."));
    return;
  }

  //
  // List tasks.
  // -----------
  // List all task entities.
  drush_print(dt('Scheduled tasks:'));
  drush_print(sprintf(
    '%10s %-15s %-15s %-19s %-19s %-19s %s',
    dt('Task ID'),
    dt('Requester'),
    dt('Operation'),
    dt('First started'),
    dt('Created'),
    dt('Scheduled'),
    dt('Run time (sec)')));
  drush_print(sprintf(
    '%10s %-15s %-15s %-19s %-19s %-19s %s',
    '----------',
    '---------------',
    '---------------',
    '-------------------',
    '-------------------',
    '-------------------',
    '--------------'));

  $taskIds = FolderShareScheduledTask::findTaskIds();
  foreach ($taskIds as $id) {
    $task = FolderShareScheduledTask::load($id);
    if ($task === NULL) {
      // Task does not exist.
      continue;
    }

    $operation     = $task->getOperation();
    $requester     = $task->getRequester();
    $parameters    = $task->getParameters();
    $comments      = $task->getComments();
    $executionTime = $task->getAccumulatedExecutionTime();
    $createdTime   = date('Y-m-d H:i:s', intval($task->getcreatedTime()));
    $scheduledTime = date('Y-m-d H:i:s', intval($task->getScheduledTime()));
    $startTime     = date('Y-m-d H:i:s', intval($task->getStartedTime()));

    $u = User::load($requester);
    if ($u === NULL) {
      $requesterName = sprintf('%s (%d)', dt('unknown'), $requester);
    }
    else {
      $requesterName = sprintf('%s (%d)', $u->getDisplayName(), $requester);
    }

    drush_print(sprintf(
      '%10d %-15s %-15s %-19s %-19s %-19s %d',
      $id,
      $requesterName,
      $operation,
      $startTime,
      $createdTime,
      $scheduledTime,
      $executionTime));
    if (empty($comments) === FALSE) {
      drush_print(sprintf("%10s %s", "", $comments));
    }
    if (empty($parameters) === FALSE) {
      drush_print(sprintf("%10s %s = %s", "", dt('parameters'), $parameters));
    }
    drush_print("");
  }
}

/*---------------------------------------------------------------------
 *
 * Module configuration - usage.
 *
 *---------------------------------------------------------------------*/

/**
 * Operates on the usage table.
 *
 * --list    (default) List the usage table.
 * --update  Update the usage table now.
 */
function drush_foldershare_usage() {
  FolderShareScheduledTask::setTaskExecutionEnabled(FALSE);

  //
  // Update usage table.
  // -------------------
  // Immediately update the usage table.
  if (empty(drush_get_option('update')) === FALSE) {
    drush_print(dt('Updating FolderShare usage table...'));
    FolderShareUsage::rebuildAllUsage();
    return;
  }

  //
  // List usage table.
  // -----------------
  // Get and present the usage table.
  $t = Settings::getUsageReportTime();
  if ($t !== 'never') {
    if ($t[0] === '@') {
      $t = date('Y-m-d H:i:s', intval(substr($t, 1)));
    }
    else {
      $t = date('Y-m-d H:i:s', intval($t));
    }
  }

  if ($t === 'never') {
    drush_print(dt('Per-user usage: (never updated)'));
  }
  else {
    drush_print(dt(
      'Per-user usage: (last updated @updated)',
      [
        '@updated' => $t,
      ]));
  }

  drush_print(sprintf(
    '%-20s %10s %10s %10s',
    dt('User'),
    dt('Folders'),
    dt('Files'),
    dt('Bytes')));
  drush_print(sprintf(
    '%-20s %10s %10s %10s',
    '--------------------',
    '----------',
    '----------',
    '----------'));
  $allUsage = FolderShareUsage::getAllUsage();
  foreach ($allUsage as $uid => $usage) {
    // Load the user entity so we can get the display name.
    $u = User::load($uid);
    if ($u === NULL) {
      $userName = sprintf('%s (%d)', dt('unknown'), $uid);
    }
    else {
      $userName = sprintf('%s (%d)', $u->getDisplayName(), $uid);
    }

    drush_print(sprintf(
      '%-20s %10d %10d %10d',
      $userName,
      $usage['nFolders'],
      $usage['nFiles'],
      $usage['nBytes']));
  }
}

/*---------------------------------------------------------------------
 *
 * Module configuration - version.
 *
 *---------------------------------------------------------------------*/

/**
 * Shows the version number.
 */
function drush_foldershare_version() {
  FolderShareScheduledTask::setTaskExecutionEnabled(FALSE);

  $modules = system_get_info('module');
  if (isset($modules['foldershare']) === TRUE) {
    $version = $modules['foldershare']['version'];
    drush_print("FolderShare $version");
  }
  else {
    drush_print(sprintf("FolderShare (%s)", dt('unknown')));
  }
}

/*---------------------------------------------------------------------
 *
 * Benchmark.
 *
 *---------------------------------------------------------------------*/

/**
 * Benchmark FolderShare.
 */
function drush_foldershare_benchmark() {
  FolderShareScheduledTask::setTaskExecutionEnabled(FALSE);

  if (empty(drush_get_option('config')) === TRUE &&
      empty(drush_get_option('hook')) === TRUE &&
      empty(drush_get_option('log')) === TRUE &&
      empty(drush_get_option('lock')) === TRUE &&
      empty(drush_get_option('system')) === TRUE &&
      empty(drush_get_option('folder')) === TRUE) {
    drush_print('No benchmark specified. Type "drush help foldershare-benchmark" for a list.');
    return;
  }

  drush_print(dt("Benchmarks"));
  drush_print('------------------------------------------------------------------------');
  drush_print(dt("These benchmarks will automatically enter and exit maintenance mode."));
  drush_print(dt("If there are non-administrator users connected to the site, they will"));
  drush_print(dt("be blocked from activity during benchmarking."));
  drush_print("");
  $answer = readline(dt("Are you ready [y|n]? "));
  if ($answer[0] === 'n') {
    drush_print(dt("Aborted."));
    return;
  }
  drush_print("");

  $warmUpCount = 2;
  $benchmarkCount = 10;

  // Put the site into maintenance mode.
  $originalMode = \Drupal::state()->get('system.maintenance_mode');
  \Drupal::state()->set('system.maintenance_mode', 1);
  Cache::invalidateTags(['rendered']);

  try {
    if (empty(drush_get_option('config')) === FALSE) {
      drush_foldershare_benchmark_config($warmUpCount, $benchmarkCount);
    }
    elseif (empty(drush_get_option('hook')) === FALSE) {
      drush_foldershare_benchmark_hook($warmUpCount, $benchmarkCount);
    }
    elseif (empty(drush_get_option('log')) === FALSE) {
      drush_foldershare_benchmark_log($warmUpCount, $benchmarkCount);
    }
    elseif (empty(drush_get_option('lock')) === FALSE) {
      drush_foldershare_benchmark_lock($warmUpCount, $benchmarkCount);
    }
    elseif (empty(drush_get_option('folder')) === FALSE) {
      drush_foldershare_benchmark_folder($warmUpCount, $benchmarkCount);
    }
    elseif (empty(drush_get_option('system')) === FALSE) {
      drush_foldershare_benchmark_system($warmUpCount, $benchmarkCount);
    }
  }
  catch (\Exception $e) {
    drush_print($e);
  }

  // Restore the original mode.
  \Drupal::state()->set('system.maintenance_mode', $originalMode);
}

/**
 * Benchmarks get/set of FolderShare configuration.
 *
 * All module configuration are managed by the Settings class, which
 * gets and sets values in a configuration saved to the database by Drupal
 * core. Since all of them are treated the same, this method picks one and
 * benchmarks get and set operations using it.
 *
 * @param int $warmUpCount
 *   The number of passes during a warm-up phase.
 * @param int $benchmarkCount
 *   The number of passes during a benchmark phase.
 */
function drush_foldershare_benchmark_config(
  int $warmUpCount,
  int $benchmarkCount) {

  drush_print("Benchmark configuration get/set:");
  drush_print("----------------------------------------------------------------");
  drush_print("This benchmark measures the time to get and set configuration");
  drush_print("values managed by FolderShare's Settings class.");
  drush_print("");

  // Save the original value.
  $originalValue = Settings::getStatusPollingInterval();

  //
  // Get setting.
  // -----------.
  $bogusValue = 0;
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $bogusValue += Settings::getStatusPollingInterval();
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $bogusValue += Settings::getStatusPollingInterval();
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Get configuration", $tdelta));

  //
  // Set setting.
  // -----------.
  $newValue = $originalValue + 0.1;

  for ($i = 0; $i < $warmUpCount; ++$i) {
    Settings::setStatusPollingInterval($newValue);
    $newValue += 0.1;
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    Settings::setStatusPollingInterval($newValue);
    $newValue += 0.1;
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Set configuration", $tdelta));

  // Restore the original value.
  Settings::setStatusPollingInterval($originalValue);
}

/**
 * Benchmarks a hook search.
 *
 * All major FolderShare operations (e.g. copy, move, delete) call post-
 * operation hooks to notify third-party modules. Since our interest is in
 * the cost of searching for hook implementations, and not the cost of any
 * particular hook, this method searches for an intentionally bogus hook
 * and benchmarks the time for the search.
 *
 * @param int $warmUpCount
 *   The number of passes during a warm-up phase.
 * @param int $benchmarkCount
 *   The number of passes during a benchmark phase.
 */
function drush_foldershare_benchmark_hook(
  int $warmUpCount,
  int $benchmarkCount) {

  drush_print("Benchmark hook search:");
  drush_print("----------------------------------------------------------------");
  drush_print("This benchmark measures the time to find all module hooks with");
  drush_print("a chosen name using FolderShare's hook() helper function. The");
  drush_print("hook name used is intentionally bogus so that no hooks are");
  drush_print("invoked. This gives a benchmark of the hook search time alone.");
  drush_print("");

  //
  // Hook search.
  // -----------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    FolderShare::postOperationHook('bogus', [$i]);
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    FolderShare::postOperationHook('bogus', [$i]);
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Search for hook", $tdelta));
}

/**
 * Benchmarks lock and release.
 *
 * All major FolderShare operations (e.g. copy, move, delete) lock a root
 * folder and/or a root list, then release the lock.
 *
 * @param int $warmUpCount
 *   The number of passes during a warm-up phase.
 * @param int $benchmarkCount
 *   The number of passes during a benchmark phase.
 */
function drush_foldershare_benchmark_lock(
  int $warmUpCount,
  int $benchmarkCount) {

  drush_print("Benchmark lock and release:");
  drush_print("----------------------------------------------------------------");
  drush_print("This benchmark measures the time to lock a root folder or a");
  drush_print("root list, then release the lock. Since locks do not involve");
  drush_print("entities, just entity IDs, these benchmarks use a bogus ID.");
  drush_print("");

  //
  // Lock root.
  // ---------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    FolderShare::acquireRootOperationLock(0);
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    FolderShare::acquireRootOperationLock(0);
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Root operation lock", $tdelta));

  //
  // Unlock root.
  // -----------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    FolderShare::releaseRootOperationLock(0);
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    FolderShare::releaseRootOperationLock(0);
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Root operation release", $tdelta));

  //
  // Lock root list.
  // --------------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    FolderShare::acquireUserRootListLock(0);
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    FolderShare::acquireUserRootListLock(0);
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Root list lock", $tdelta));

  //
  // Unlock root list.
  // ----------------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    FolderShare::releaseUserRootListLock(0);
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    FolderShare::releaseUserRootListLock(0);
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Root list release", $tdelta));
}

/**
 * Benchmarks a log post.
 *
 * All major FolderShare operations (e.g. copy, move, delete) call the
 * log() utility function to post a log message. That function checks the
 * module's configuration and either posts it or doesn't it, depending upon
 * whether posting is enabled. Both cases are benchmarked.
 *
 * @param int $warmUpCount
 *   The number of passes during a warm-up phase.
 * @param int $benchmarkCount
 *   The number of passes during a benchmark phase.
 */
function drush_foldershare_benchmark_log(
  int $warmUpCount,
  int $benchmarkCount) {

  drush_print("Benchmark log posts:");
  drush_print("----------------------------------------------------------------");
  drush_print("This benchmark measures the time to create a log message and");
  drush_print("post it, or have the post blocked, depending upon whether the");
  drush_print("module has activity logging enabled.");
  drush_print("");

  // Save the original value.
  $originalValue = Settings::getActivityLogEnable();

  //
  // Logging disabled.
  // ----------------.
  Settings::setActivityLogEnable(FALSE);

  for ($i = 0; $i < $warmUpCount; ++$i) {
    FolderShare::log(
      'notice',
      'Benchmark message using entity @id ("%name") from top level to @newId ("%newName").',
      [
        '@id'      => $i + 123,
        '%name'    => "bogus",
        '@newId'   => 456,
        '%newName' => "fake",
        'link'     => "none",
      ]);
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    FolderShare::log(
      'notice',
      'Benchmark message using entity @id ("%name") from top level to @newId ("%newName").',
      [
        '@id'      => $i + 123,
        '%name'    => "bogus",
        '@newId'   => 456,
        '%newName' => "fake",
        'link'     => "none",
      ]);
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Post to log (disabled)", $tdelta));

  //
  // Logging enabled.
  // ---------------.
  Settings::setActivityLogEnable(TRUE);

  for ($i = 0; $i < $warmUpCount; ++$i) {
    FolderShare::log(
      'notice',
      'Benchmark message using entity @id ("%name") from top level to @newId ("%newName").',
      [
        '@id'      => $i + 123,
        '%name'    => "bogus",
        '@newId'   => 456,
        '%newName' => "fake",
        'link'     => "none",
      ]);
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    FolderShare::log(
      'notice',
      'Benchmark message using entity @id ("%name") from top level to @newId ("%newName").',
      [
        '@id'      => $i + 123,
        '%name'    => "bogus",
        '@newId'   => 456,
        '%newName' => "fake",
        'link'     => "none",
      ]);
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Post to log (enabled)", $tdelta));

  // Restore the original value.
  Settings::setActivityLogEnable($originalValue);
}

/**
 * Benchmarks folder operations.
 *
 * Folders exist solely within the database. This method benchmarks creating,
 * deleting, loading, etc on folders. No file operations take place.
 *
 * @param int $warmUpCount
 *   The number of passes during a warm-up phase.
 * @param int $benchmarkCount
 *   The number of passes during a benchmark phase.
 */
function drush_foldershare_benchmark_folder(
  int $warmUpCount,
  int $benchmarkCount) {

  drush_print("Benchmark folder operations:");
  drush_print("----------------------------------------------------------------");
  drush_print("This benchmark measures the time to create, delete, load,");
  drush_print("copy, and move individual empty folders.");
  drush_print("");

  //
  // Create root folders.
  // -------------------.
  $created = [];
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $created[] = FolderShare::createRootFolder();
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $created[] = FolderShare::createRootFolder();
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Create root folder", $tdelta));

  //
  // Load root folders from memory cache.
  // -----------------------------------.
  // Evidentally, created items are NOT automatically added to the storage
  // manager's memory and database caches. An entry is only added to those
  // caches if an item is loaded. So, start with a redundant load of the
  // same entities created above.
  foreach ($created as $item) {
    FolderShare::load($item->id());
  }

  $n = 0;
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Load root folder (memory cached)", $tdelta));

  //
  // Load root folders from database cache.
  // -------------------------------------.
  // Clear the memory cache. This should leave the storage manager's database
  // cache in place and subsequent loads will load from it.
  \Drupal::service('entity.memory_cache')->deleteAll();

  $n = 0;
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Load root folder (DB cached)", $tdelta));

  //
  // Load root folders from database (uncached).
  // ------------------------------------------.
  // Reseting the storage manager's cache clears everything from memory and
  // database caches. Further loads must assemble the entity from assorted
  // database tables.
  \Drupal::entityTypeManager()->getStorage('foldershare')->resetCache();

  $n = 0;
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Load root folder (uncached)", $tdelta));

  //
  // Duplicate root folders.
  // ----------------------.
  $n = 0;
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $created[] = $created[$n++]->duplicate();
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $created[] = $created[$n++]->duplicate();
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Duplicate root folder", $tdelta));

  //
  // Change owner on root folders.
  // ----------------------------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $item = $created[$i];
    $item->changeOwnerId(1, FALSE);
  }

  $nCreated = count($created);
  $t1 = microtime(TRUE);
  foreach ($created as $item) {
    $item->changeOwnerId(1, FALSE);
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $nCreated;
  drush_print(sprintf("  %-35s %10.6f sec", "Change owner on root folder", $tdelta));

  //
  // Delete root folders.
  // -------------------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $item = array_shift($created);
    $item->delete();
  }

  $nCreated = count($created);
  $t1 = microtime(TRUE);
  foreach ($created as $item) {
    $item->delete();
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $nCreated;
  drush_print(sprintf("  %-35s %10.6f sec", "Delete root folder", $tdelta));

  drush_print("");

  //
  // Create subfolders.
  // -----------------.
  $parent = FolderShare::createRootFolder();
  $created = [];
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $created[] = $parent->createFolder();
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $created[] = $parent->createFolder();
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Create subfolder", $tdelta));

  //
  // Load subfolders from memory cache.
  // ----------------------------------.
  // See comments above on the memory cache.
  foreach ($created as $item) {
    FolderShare::load($item->id());
  }

  $n = 0;
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Load subfolder (memory cached)", $tdelta));

  //
  // Load sub folders from database cache.
  // ------------------------------------.
  // See comments above on the database cache.
  \Drupal::service('entity.memory_cache')->deleteAll();

  $n = 0;
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Load subfolder (DB cached)", $tdelta));

  //
  // Load subfolders from database (uncached).
  // ----------------------------------------.
  // See comments above on reseting the cache.
  \Drupal::entityTypeManager()->getStorage('foldershare')->resetCache();

  $n = 0;
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $item = FolderShare::load($created[$n++]->id());
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Load subfolder (uncached)", $tdelta));

  //
  // Duplicate subfolders.
  // --------------------.
  $n = 0;
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $created[] = $created[$n++]->duplicate();
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $created[] = $created[$n++]->duplicate();
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Duplicate subfolder", $tdelta));

  //
  // Change owner on subfolders.
  // --------------------------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $item = $created[$i];
    $item->changeOwnerId(1, FALSE);
  }

  $nCreated = count($created);
  $t1 = microtime(TRUE);
  foreach ($created as $item) {
    $item->changeOwnerId(1, FALSE);
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $nCreated;
  drush_print(sprintf("  %-35s %10.6f sec", "Change owner on subfolder", $tdelta));

  //
  // Delete subfolders.
  // -----------------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $item = array_shift($created);
    $item->delete();
  }

  $nCreated = count($created);
  $t1 = microtime(TRUE);
  foreach ($created as $item) {
    $item->delete();
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $nCreated;
  drush_print(sprintf("  %-35s %10.6f sec", "Delete subfolder", $tdelta));

  // Clean up.
  $parent->delete();
  $created = [];
}

/**
 * Benchmarks getting system values.
 *
 * Getting PHP ini directive values, the current time, the current memory
 * usage are all queries of system values.
 *
 * @param int $warmUpCount
 *   The number of passes during a warm-up phase.
 * @param int $benchmarkCount
 *   The number of passes during a benchmark phase.
 */
function drush_foldershare_benchmark_system(
  int $warmUpCount,
  int $benchmarkCount) {

  drush_print("Benchmark getting system values:");
  drush_print("----------------------------------------------------------------");
  drush_print("This benchmark measures the time to get system values, such as");
  drush_print("PHP ini directive values, the current time, the current memory");
  drush_print("usage, and so on.");
  drush_print("");

  //
  // Get ini value.
  // -------------.
  $bogusValue = 0;
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $bogusValue += ini_get('max_execution_time');
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $bogusValue += ini_get('max_execution_time');
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Get ini value", $tdelta));

  //
  // Get time.
  // --------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $bogusValue += time();
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $bogusValue += time();
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Get time", $tdelta));

  //
  // Get memory usage.
  // ----------------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $bogusValue += memory_get_usage();
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $bogusValue += memory_get_usage();
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Get memory usage", $tdelta));

  //
  // Garbage collect (no garbage).
  // ----------------------------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    gc_collect_cycles();
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    gc_collect_cycles();
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Garbage collect (no garbage)", $tdelta));

  //
  // Garbage collect (some garbage).
  // ----------------------------.
  for ($i = 0; $i < $warmUpCount; ++$i) {
    $junk = [];
    for ($j = 0; $j < 100; ++$j) {
      $junk[] = new stdClass();
    }
    for ($j = 0; $j < 100; ++$j) {
      unset($junk[$j]);
    }
    gc_collect_cycles();
  }

  $t1 = microtime(TRUE);
  for ($i = 0; $i < $benchmarkCount; ++$i) {
    $junk = [];
    for ($j = 0; $j < 100; ++$j) {
      $junk[] = new stdClass();
    }
    for ($j = 0; $j < 100; ++$j) {
      unset($junk[$j]);
    }
    gc_collect_cycles();
  }
  $t2 = microtime(TRUE);
  $tdelta = ($t2 - $t1) / $benchmarkCount;
  drush_print(sprintf("  %-35s %10.6f sec", "Garbage collect (with garbage)", $tdelta));
}
