! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_load_weak
      
      use weak_def
      use const_def, only: dp
      
      implicit none
      
      integer, parameter :: cache_version = 6
      
      private :: private_load_weak_tables


      contains
      
      
      subroutine load_weak_data_if_necessary(ierr)
         integer, intent(out) :: ierr         
         ierr = 0         
         if (.not. weak_data_loaded) then ! avoid doing critical section if possible
!$omp critical (load_weak_tables)
            if (.not. weak_data_loaded) then
               call private_load_weak_tables(ierr)
               if (ierr == 0) then
                  call load_weak_info_list(ierr)
                  if (ierr == 0) weak_data_loaded = .true.
                  end if
            end if
!$omp end critical (load_weak_tables)
         end if
      end subroutine load_weak_data_if_necessary
      
      
      subroutine load_weak_info_list(ierr)
         use utils_lib
         integer, intent(out) :: ierr
         
         integer :: iounit, i
         character (len=256) :: filename, string
	      character(len=iso_name_length) :: lhs, rhs
	      character(len=2*iso_name_length+1) :: key
	      integer, parameter :: max_num_weak_info = 1000

         logical, parameter :: dbg = .false.

         include 'formats.dek'
         
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return

         filename = trim(weak_data_dir) // '/weak_info.list'
         ierr = 0
         open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
         if (ierr /= 0) then
            call free_iounit(iounit)
            write(*,*) 'failed to open ' // trim(filename)
            call free_iounit(iounit)
            return
         end if
         
         if (dbg) then
            write(*,*)
            write(*,*) 'weak info filename <' // trim(filename) // '>'
            write(*,*)
         end if

         do ! skip to line starting with 'from '
            read(iounit,'(a)',iostat=ierr) string
            if (failed('read weak info comments')) return
            if (len_trim(string) > 4) then
               if (string(1:5) == 'from ') exit
            end if
         end do

         nullify(weak_info_list_dict)
         allocate(weak_info_life_halflife(max_num_weak_info))
         allocate(weak_info_list_Qneu(max_num_weak_info))
         num_weak_info_list_reactions = 0
         do i = 1, max_num_weak_info ! keep reading until end of file
            read(iounit,fmt=*,iostat=ierr) &
               lhs, rhs, weak_info_life_halflife(i), weak_info_list_Qneu(i)
            if (ierr /= 0) then
               ierr = 0; exit
            end if
            call create_weak_dict_key(lhs, rhs, key)
            
            !write(*,'(a)') 'weak info list key ' // trim(key)
            
            call integer_dict_define(weak_info_list_dict, key, i, ierr)
            if (failed('integer_dict_define')) return
            num_weak_info_list_reactions = i
         end do
         
         close(iounit)
         
         call free_iounit(iounit)
         
         if (num_weak_info_list_reactions == 0) then
            ierr = -1
            write(*,*) 'failed trying to read weak_info.list -- no reactions?'
            return
         end if
         
         if (num_weak_info_list_reactions == max_num_weak_info) then
            ierr = -1
            write(*,*) 'failed trying to read weak_info.list -- too many reactions?'
            return
         end if
         
         call integer_dict_create_hash(weak_info_list_dict, ierr)
         if (ierr /= 0) return
         
         call realloc_double(weak_info_life_halflife, num_weak_info_list_reactions, ierr)
         if (ierr /= 0) return
         
         call realloc_double(weak_info_list_Qneu, num_weak_info_list_reactions, ierr)
         if (ierr /= 0) return
         
         
         contains
         
         logical function failed(str)
            character (len=*) :: str
            failed = (ierr /= 0)
            if (failed) then
               call free_iounit(iounit)
               write(*,*) 'failed: ' // trim(str)
            end if
         end function failed
         
         
      end subroutine load_weak_info_list
      
      
      subroutine private_load_weak_tables(ierr)
         use utils_lib
         use chem_lib, only: get_nuclide_index
         use chem_def, only: iso_name_length
         integer, intent(out) :: ierr
         
         integer :: iounit, i, ios, id
         character (len=256) :: filename, cache_filename, string
	      character(len=iso_name_length) :: lhs1, rhs1, lhs2, rhs2, weak_lhs, weak_rhs
	      character(len=2*iso_name_length+1) :: key

         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         
         if (read_cache()) then
            call integer_dict_create_hash(weak_reactions_dict, ierr)
            if (failed('integer_dict_create_hash')) return
            call free_iounit(iounit)
            return
         end if
         
         filename = trim(weak_data_dir) // '/weakreactions.tables'
         ierr = 0
         open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
         if (ierr /= 0) then
            call free_iounit(iounit)
            write(*,*) 'failed to open ' // trim(filename)
            call free_iounit(iounit)
            return
         end if
         
         if (dbg) then
            write(*,*)
            write(*,*) 'weaklib filename <' // trim(filename) // '>'
            write(*,*)
         end if

         do ! skip to after line starting with '='
            read(iounit,'(a)',iostat=ierr) string
            if (failed('read header')) return
            if (len_trim(string) > 0) then
               if (string(1:1) == '=') exit
            end if
         end do

         if (.not. skip_line()) return
         
         read(iounit,*,iostat=ierr) num_weak_reactions
         if (failed('read num_weak_reactions')) return
         
         if (dbg) write(*,2) 'num_weak_reactions', num_weak_reactions
         
         call alloc
         if (failed('allocate')) return
         
         nullify(weak_reactions_dict)
         do i = 1, num_weak_reactions
            if (.not. skip_line()) return
            if (mod(i,2)==1) then ! first of pair
               if (.not. skip_line()) return
               if (.not. skip_line()) return
               read(iounit,fmt='(2a5)',iostat=ierr) lhs1, rhs1
               if (failed('read lhs1, rhs1')) return
               if (lhs1 == 'al-6') lhs1 = 'al26-1'
               if (rhs1 == 'al-6') rhs1 = 'al26-1'
               if (lhs1 == 'al*6') lhs1 = 'al26-2'
               if (rhs1 == 'al*6') rhs1 = 'al26-2'
               read(iounit,fmt='(2a5)',iostat=ierr) lhs2, rhs2
               if (failed('read lhs2, rhs2')) return
               if (lhs2 == 'al-6') lhs2 = 'al26-1'
               if (rhs2 == 'al-6') rhs2 = 'al26-1'
               if (lhs2 == 'al*6') lhs2 = 'al26-2'
               if (rhs2 == 'al*6') rhs2 = 'al26-2'
               if (.not. skip_line()) return
               if (.not. skip_line()) return
               weak_lhs = lhs1
               weak_rhs = rhs1
            else
               weak_lhs = lhs2
               weak_rhs = rhs2
            end if
            call adjust_name(weak_lhs)
            call adjust_name(weak_rhs)
            id = get_nuclide_index(weak_lhs)
            if (id <= 0) then
               write(*,*) 'weaklib FATAL ERROR: unknown nuclide ' // weak_lhs
               stop 1
            end if
            weak_lhs_nuclide_id(i) = id
            id = get_nuclide_index(weak_rhs)
            if (id <= 0) then
               write(*,*) 'weaklib FATAL ERROR: unknown nuclide ' // weak_rhs
               stop 1
            end if
            weak_rhs_nuclide_id(i) = id
            weak_lhs_nuclide_name(i) = weak_lhs
            weak_rhs_nuclide_name(i) = weak_rhs
            call create_weak_dict_key(weak_lhs, weak_rhs, key)
            call integer_dict_define(weak_reactions_dict, key, i, ierr)
            if (failed('integer_dict_define')) return
            if (.not. skip_line()) return
            call read_table(i,i_ldecay)
            if (failed('read ldecay')) return
            if (.not. skip_line()) return
            call read_table(i,i_lcapture)
            if (failed('read lcapture')) return
            if (.not. skip_line()) return
            call read_table(i,i_lneutrino)
            if (failed('read lneutrino')) return
         end do
         
         close(iounit)
         
         call integer_dict_create_hash(weak_reactions_dict, ierr)
         if (failed('integer_dict_create_hash')) return
         
         if (weak_bicubic) then
            do i = 1, num_weak_reactions
               if (ierr == 0) call create_bicubic_interpolant(i,i_ldecay,ierr)
               if (ierr == 0) call create_bicubic_interpolant(i,i_lcapture,ierr)
               if (ierr == 0) call create_bicubic_interpolant(i,i_lneutrino,ierr)
            end do
            if (failed('create_bicubic_interpolant')) return
         else
            do i = 1, num_weak_reactions
               if (ierr == 0) call create_pm_T9_interpolant(i,i_ldecay,ierr)
               if (ierr == 0) call create_pm_T9_interpolant(i,i_lcapture,ierr)
               if (ierr == 0) call create_pm_T9_interpolant(i,i_lneutrino,ierr)
            end do
            if (failed('create_pm_T9_interpolant')) return
         end if

         call write_cache
         
         call free_iounit(iounit)
         
         if (dbg) write(*,*) 'finished load_weak_tables'
         
         
         contains
         
         
         subroutine alloc
         
            allocate( &
               weak_lhs_nuclide_name(num_weak_reactions), weak_rhs_nuclide_name(num_weak_reactions), &
               weak_lhs_nuclide_id(num_weak_reactions), weak_rhs_nuclide_id(num_weak_reactions), &
               weak_reactions_data1(4*weak_num_T9*weak_num_lYeRho*3*num_weak_reactions), stat=ierr)
               
            weak_reactions_data(1:4, 1:weak_num_T9, 1:weak_num_lYeRho, 1:3, 1:num_weak_reactions) => &
               weak_reactions_data1(1:4*weak_num_T9*weak_num_lYeRho*3*num_weak_reactions)
               
         end subroutine alloc
         
         
         subroutine adjust_name(nm)
            character(len=iso_name_length) :: nm
            nm = adjustl(nm)
            if (nm == 'p') then
               nm = 'h1'
            else if (nm == 'n') then
               nm = 'neut'
            end if
         end subroutine adjust_name
         
         
         logical function read_cache()
            use chem_def, only: chem_isos
            integer :: nT9, nlYeRho, version
            read_cache = .false.
            cache_filename = trim(weaklib_cache_dir) // '/weakreactions.bin'
            
            
            return
            
            
            
            
            
            ios = 0
            open(unit=iounit,file=trim(cache_filename),action='read', &
                  status='old',iostat=ios,form='unformatted')
            if (ios /= 0) then
               !write(*,*) 'cannot open ' // trim(cache_filename)
               return
            end if
            read(iounit,iostat=ios) version, num_weak_reactions, nT9, nlYeRho
            if (ios /= 0 .or. version /= cache_version .or. &
                nT9 /= weak_num_T9 .or. nlYeRho /= weak_num_lYeRho) then
               close(iounit)
               return
            end if
            call alloc
            if (ierr /= 0) then
               close(iounit)
               return
            end if
            do i = 1, num_weak_reactions
               read(iounit,iostat=ios) weak_lhs, weak_rhs
               if (ios /= 0) then
                  close(iounit)
                  return
               end if
               weak_lhs_nuclide_name(i) = weak_lhs
               weak_rhs_nuclide_name(i) = weak_rhs
            end do
            read(iounit,iostat=ios) &
               weak_reactions_data(1:4, 1:weak_num_T9, 1:weak_num_lYeRho, 1:3, 1:num_weak_reactions)
            if (ios /= 0) then
               close(iounit)
               return
            end if
            close(iounit)
            do i = 1, num_weak_reactions
		         weak_lhs = weak_lhs_nuclide_name(i)
		         weak_rhs = weak_rhs_nuclide_name(i)
               weak_lhs_nuclide_id(i) = get_nuclide_index(weak_lhs)
               weak_rhs_nuclide_id(i) = get_nuclide_index(weak_rhs)
               if (weak_lhs_nuclide_id(i) <= 0 .or. weak_rhs_nuclide_id(i) <= 0) then
                  ierr = -1
                  return
               end if
               call create_weak_dict_key(weak_lhs, weak_rhs, key)
               call integer_dict_define(weak_reactions_dict, key, i, ierr)
               if (failed('integer_dict_define')) return
            end do
            read_cache = .true.
            if (dbg) write(*,*) 'read ' // trim(cache_filename)
         end function read_cache
         
         
         subroutine write_cache
            open(unit=iounit,file=trim(cache_filename),action='write', &
                  iostat=ios,form='unformatted')
            if (ios /= 0) return
            write(iounit) cache_version, num_weak_reactions, weak_num_T9, weak_num_lYeRho
            do i = 1, num_weak_reactions
               write(iounit,iostat=ios) weak_lhs_nuclide_name(i), weak_rhs_nuclide_name(i)
               if (ios /= 0) then
                  close(iounit)
                  return
               end if
            end do
            write(iounit) weak_reactions_data
            close(iounit)
            if (dbg) write(*,*) 'write ' // trim(cache_filename)
         end subroutine write_cache
         
         
         subroutine create_pm_T9_interpolant(i,ii,ierr)
            ! piecewise monotonic interpolation in T9 for each lYeRho in table
            use interp_1d_def
            use interp_1d_lib
            integer, intent(in) :: i, ii
            integer, intent(out) :: ierr
            integer :: j, m, n

            integer :: nx       ! length of x vector (>= 2)
            real(dp), pointer :: x(:) ! (nx)    ! junction points, strictly monotonic
            real(dp), pointer :: f1(:), f(:,:) ! (4,nx)  ! data & interpolation coefficients
            integer, parameter :: nwork = pm_work_size
            real(dp), pointer :: work(:) ! =(nx,nwork)
            
            ierr = 0
            
            nx = weak_num_T9
            allocate(x(nx), f1(4*nx), work(nx*nwork), stat=ierr)
            if (ierr /= 0) return
            
            f(1:4,1:nx) => f1(1:4*nx)
            
            x = dble(weak_reaction_T9s)
            
            do j=1,weak_num_lYeRho
               do m=1,nx
                  f(1,m) = dble(weak_reactions_data(1,m,j,ii,i))
               end do
               call interp_pm(x, nx, f1, nwork, work, ierr)
               if (ierr /= 0) return
               do n=1,nx
                  do m=1,4
                     weak_reactions_data(m,n,j,ii,i) = real(f(m,n))
                  end do
               end do
            end do

            deallocate(x, f1, work)

         end subroutine create_pm_T9_interpolant
         
         
         subroutine create_bicubic_interpolant(i,ii,ierr)
            use interp_2d_lib_sg
            integer, intent(in) :: i, ii
            integer, intent(out) :: ierr
            integer :: ibcxmin                   ! bc flag for x=xmin
            real :: bcxmin(weak_num_lYeRho)    ! bc data vs. y at x=xmin
            integer :: ibcxmax                   ! bc flag for x=xmax
            real :: bcxmax(weak_num_lYeRho)     ! bc data vs. y at x=xmax
            integer :: ibcymin                   ! bc flag for y=ymin
            real :: bcymin(weak_num_T9)   ! bc data vs. x at y=ymin
            integer :: ibcymax                   ! bc flag for y=ymax
            real :: bcymax(weak_num_T9)   ! bc data vs. x at y=ymax         
            integer :: il1, il2, j, k, m
            
            real, target :: f1_ary(4*weak_num_T9*weak_num_lYeRho)
            real, pointer :: f1(:), f3(:,:,:)
            
            ! just use "not a knot" bc's at edges of tables
            ibcxmin = 0; bcxmin(:) = 0
            ibcxmax = 0; bcxmax(:) = 0
            ibcymin = 0; bcymin(:) = 0
            ibcymax = 0; bcymax(:) = 0
            
            f1 => f1_ary
            f3(1:4,1:weak_num_T9,1:weak_num_lYeRho) => f1(1:4*weak_num_T9*weak_num_lYeRho)
            do k = 1,weak_num_T9
               do j = 1,4
                  do m = 1,weak_num_lYeRho
                     f3(j,k,m) = weak_reactions_data(j,k,m,ii,i)
                  end do
               end do
            end do
            call interp_mkbicub_sg( &
               weak_reaction_T9s, weak_num_T9, &
               weak_reaction_lYeRhos, weak_num_lYeRho, &
               f1, weak_num_T9, &
               ibcxmin, bcxmin, ibcxmax, bcxmax, &
               ibcymin, bcymin, ibcymax, bcymax, &
               il1, il2, ierr)
            do k = 1,weak_num_T9
               do j = 1,4
                  do m = 1,weak_num_lYeRho
                     weak_reactions_data(j,k,m,ii,i) = f3(j,k,m)
                  end do
               end do
            end do
               
         end subroutine create_bicubic_interpolant
         
         
         subroutine read_table(i,ii)
            integer, intent(in) :: i, ii
            integer :: k, j, skip
            real :: buffer(weak_num_T9)
            logical, parameter :: dbg = .false.
            do j = 1, weak_num_lYeRho
               read(iounit,fmt=*,iostat=ierr) skip, buffer
               if (ierr /= 0 .or. j /= skip) then
                  if (dbg) then
                     write(*,*) 'error in reading table', j, skip
                     write(*,*) 'these are the NEXT lines after the error'
                     do k=1,20
                        read(iounit,fmt='(a)') string
                        write(*,'(a)') trim(string)
                     end do
                     write(*,*)
                     stop 'read_table'
                  end if
                  return
               end if
               weak_reactions_data(1,1:weak_num_T9,j,ii,i) = buffer
               if (dbg) write(*,'(a,2i6,99f9.3)') 'read_table', j, skip, buffer
            end do
         end subroutine read_table
         
         
         logical function failed(str)
            character (len=*) :: str
            failed = (ierr /= 0)
            if (failed) then
               call free_iounit(iounit)
               write(*,*) 'failed: ' // trim(str)
            end if
         end function failed
         
         
         logical function skip_line()
            logical, parameter :: dbg = .false.
            if (dbg) then
               read(iounit,fmt='(a)') string
               write(*,'(a)') 'skip line ' // trim(string)
            else
               read(iounit,'(a)',iostat=ierr)
            end if
            skip_line = .not. (failed('skip line'))
         end function skip_line


      end subroutine private_load_weak_tables
      

      end module mod_load_weak

