! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      
      implicit none
      
      
      
      
      integer :: time0, time1, clock_rate
      real(dp), parameter :: expected_runtime = 1 ! minutes
      
      real(dp) :: &
         save_max_years_for_timestep, &
         starting_time_for_injection, &
         ergs_second_this_step, &
         total_ergs_to_inject, &
         seconds_for_inject, &
         total_injected_so_far, &
         shell_mass_for_injection
      integer :: min_nsteps_for_inject
      
      ! these routines are called by the standard run_star check_model
      contains
      
      
      subroutine extras_controls(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         include 'formats'
         ierr = 0
         s% use_other_energy = .true.
         s% other_energy => energy_routine
         s% use_other_mixing = .true.
         s% other_mixing => mixing_routine
         starting_time_for_injection = -1
         ergs_second_this_step = 0d0
         total_ergs_to_inject = 1d51 
         seconds_for_inject = 1
         total_injected_so_far = 0d0
         shell_mass_for_injection = 1d-3*Msun
         min_nsteps_for_inject = 100
         save_max_years_for_timestep = s% max_years_for_timestep
      end subroutine extras_controls
      
      
      subroutine mixing_routine(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         !s% adjust_mlt_gradT_fraction(1:s% nz) = 0.5
         !s% adjust_mlt_gradT_fraction(1:s% nz) = 0 ! force all to gradr
      end subroutine mixing_routine


      subroutine energy_routine(id, ierr)
         use const_def, only: Rsun
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         integer :: k, nz, k_bot, k_top
         real(dp) :: dm_sum, mid_time, max_ergs_g_sec, rate, del_time
         logical, parameter :: dbg = .false.
         include 'formats'
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return

         nz = s% nz
         
         if (s% model_number < 5) then ! give it some time
            s% extra_heat(1:nz) = 0
            return
         end if

         if (s% model_number == 5) then
            s% max_timestep = seconds_for_inject/min_nsteps_for_inject
            s% max_years_for_timestep = s% max_timestep/secyer
            if (dbg) then
               write(*,1) 's% max_timestep', s% max_timestep
               write(*,1) 's% max_years_for_timestep', s% max_years_for_timestep
               write(*,1) 'lg max years', log10(s% max_years_for_timestep)
               write(*,*)
            end if
            starting_time_for_injection = s% time
            return
         end if

         k_bot = nz
         
         if (.true.) then ! inject at center
            k_bot = nz
         else if (.true.) then ! find base
            do k=1,nz
               if (s% X(k) + s% Y(k) < 0.01d0) then
                  k_bot = k
                  exit
               end if
            end do
         else ! find desired mass depth
            dm_sum = 0
            do k=1,nz
               dm_sum = dm_sum + s% dm(k)
               if (dm_sum > 1d-3*Msun) then
                  k_bot = k
                  exit
               end if
            end do
         end if
         
         dm_sum = 0
         k_top = 1
         do k=k_bot-1,1,-1
            dm_sum = dm_sum + s% dm(k)
            if (dm_sum >= shell_mass_for_injection) then
               k_top = k
               exit
            end if
         end do
         
         mid_time = seconds_for_inject/2
         max_ergs_g_sec = 2*total_ergs_to_inject/(shell_mass_for_injection*seconds_for_inject)
         del_time = s% time - starting_time_for_injection

         if (dbg) then
            write(*,1) 'mid step time', s% time + s% dt/2
            write(*,1) 'starting_time_for_injection', starting_time_for_injection
            write(*,1) 'del_time', del_time
            write(*,1) 'mid_time', mid_time
            write(*,1) 'seconds_for_inject', seconds_for_inject
         end if
         
         if (del_time >= 0 .and. del_time < seconds_for_inject) then
            rate = (total_ergs_to_inject - total_injected_so_far)/((seconds_for_inject - del_time)*sum(s% dm(k_top:k_bot)))
         else
            if (dbg) write(*,*) 'not time for inject'
            rate = 0
         end if

         ergs_second_this_step = rate*sum(s% dm(k_top:k_bot))
         if (total_injected_so_far + ergs_second_this_step*s% dt > total_ergs_to_inject) then
            if (dbg) write(*,1) 'reduce rate since almost done', &
               total_ergs_to_inject/(total_injected_so_far + ergs_second_this_step*s% dt)
            rate = (total_ergs_to_inject - total_injected_so_far)/(s% dt*sum(s% dm(k_top:k_bot)))
            ergs_second_this_step = rate*sum(s% dm(k_top:k_bot))
            if (dbg) then
               write(*,1) 'total_injected_so_far/total_ergs_to_inject', &
                  total_injected_so_far/total_ergs_to_inject
               write(*,1) 'expected at end of step', &
                  total_injected_so_far + ergs_second_this_step*s% dt
               write(*,1) 'expected fraction of goal', &
                  (total_injected_so_far + ergs_second_this_step*s% dt)/total_ergs_to_inject
               write(*,*)
            end if
         else if (total_injected_so_far < total_ergs_to_inject) then
            write(*,1) 'fraction injected', &
               total_injected_so_far/total_ergs_to_inject!, del_time/seconds_for_inject, &
               !s% time/secyer, seconds_for_inject/secyer
         end if
         
         s% extra_heat(1:k_top-1) = 0
         s% extra_heat(k_top:k_bot) = rate
         s% extra_heat(k_bot+1:nz) = 0
         
         if (dbg) then
         
         write(*,1) 'total_ergs_to_inject', total_ergs_to_inject
            write(*,1) 'shell_mass_for_injection', shell_mass_for_injection
            write(*,1) 'seconds_for_inject', seconds_for_inject
            write(*,1) 's% dt', s% dt
            write(*,1) 'max_ergs_g_sec', max_ergs_g_sec

            write(*,2) 'nz', nz
            write(*,2) 's% m(k_bot)/Msun', k_bot, s% m(k_bot)/Msun
            write(*,2) 's% m(k_top)/Msun', k_top, s% m(k_top)/Msun
            write(*,1) 'rate', rate
            write(*,1) 'rate/max_ergs_g_sec', rate/max_ergs_g_sec
            write(*,1) 'dm', sum(s% dm(k_top:k_bot))
            write(*,1) 'dm/Msun', sum(s% dm(k_top:k_bot))/Msun
            write(*,1) 'ergs_second_this_step', ergs_second_this_step
            write(*,1) 'ergs_this_step', ergs_second_this_step*s% dt
            !write(*,1) '', 
            !write(*,1) '', 
            !write(*,1) '', 
         
            !write(*,1) 'starting_time_for_injection', starting_time_for_injection
            write(*,*)
            
         end if
         

      end subroutine energy_routine


      
      
      integer function extras_startup(s, id, restart, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         ierr = 0
         extras_startup = 0
         call system_clock(time0,clock_rate)
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup
      
      
      subroutine extras_after_evolve(s, id, id_extra, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         real(dp) :: dt
         ierr = 0
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         if (dt > 10*expected_runtime) then
            write(*,'(/,a30,2f18.6,a,/)') '>>>>>>> EXCESSIVE runtime', &
               dt, expected_runtime, '   <<<<<<<<<  ERROR'
         else
            write(*,'(/,a50,2f18.6,99i10/)') 'runtime, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         end if
      end subroutine extras_after_evolve
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         extras_check_model = keep_going         
      end function extras_check_model


      integer function how_many_extra_history_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_history_columns = 0
      end function how_many_extra_history_columns
      
      
      subroutine data_for_extra_history_columns(s, id, id_extra, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine data_for_extra_history_columns

      
      integer function how_many_extra_profile_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(s, id, id_extra, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         double precision :: vals(nz,n)
         integer, intent(out) :: ierr
         integer :: k
         ierr = 0
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going or terminate.
      integer function extras_finish_step(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer :: ierr
         include 'formats'
         extras_finish_step = keep_going
         total_injected_so_far = total_injected_so_far + ergs_second_this_step*s% dt
         if (total_injected_so_far > 0.999d0*total_ergs_to_inject) then
            s% max_years_for_timestep = save_max_years_for_timestep
            s% max_timestep = secyer*s% max_years_for_timestep
         end if
         
         if (.false.) then
            write(*,1) 'total_ergs_to_inject', total_ergs_to_inject
            write(*,1) 'total_injected_so_far', total_injected_so_far
            write(*,1) 'this step', ergs_second_this_step*s% dt
            write(*,1) 'fraction done', total_injected_so_far/total_ergs_to_inject
            write(*,*)
         end if

         call store_extra_info(s)
      end function extras_finish_step
      
      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         return
         
         i = 0
         call move_int(min_nsteps_for_inject)  
         num_ints = i
         
         i = 0
         call move_dbl(starting_time_for_injection)  
         call move_dbl(ergs_second_this_step)  
         call move_dbl(total_ergs_to_inject)  
         call move_dbl(seconds_for_inject)  
         call move_dbl(total_injected_so_far)  
         call move_dbl(shell_mass_for_injection)  
         call move_dbl(save_max_years_for_timestep)
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            real(dp) :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info

      end module run_star_extras
      
