! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************

!     notes: gravitational mass is s% m_grav(1)/Msun should be 1.0


 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      use utils_lib
      use num_lib, only: safe_log10, qsort
      use run_star_support
      
      implicit none

      real(dp) :: target_log_luminosity, target_log_radius, target_surface_Z_div_X, &
         target_Rcz, target_surf_He, target_csound_rms
      real(dp) :: sigma_log_luminosity, sigma_log_radius, sigma_surface_Z_div_X, &
         sigma_Rcz, sigma_surf_He, sigma_csound_rms
      logical :: use_Rcz_in_search, use_sigma_surf_He_in_search, use_csound_rms_in_search
      logical :: vary_Z_div_X, vary_Y, vary_alpha, vary_f_ov, vary_f0_ov_fraction
      
      real(dp) :: first_Z_div_X, first_Y, first_alpha, first_f_ov, first_f0_ov_fraction
      real(dp) :: min_Z_div_X, min_Y, min_alpha, min_f_ov, min_f0_ov_fraction
      real(dp) :: max_Z_div_X, max_Y, max_alpha, max_f_ov, max_f0_ov_fraction
      
      real(dp) :: rhoend = 1d-4

      real(dp) :: Y_frac_he3 = 1d-4
            
      character (len=100) :: search_type
      integer :: max_tries = 100 ! for BOBYQA



      namelist /calibration_controls/ &
         search_type, max_tries, &
         target_log_luminosity, target_log_radius, target_surface_Z_div_X, &
         target_Rcz, target_surf_He, target_csound_rms, &
         sigma_log_luminosity, sigma_log_radius, sigma_surface_Z_div_X, &
         sigma_Rcz, sigma_surf_He, sigma_csound_rms, &
         use_Rcz_in_search, use_sigma_surf_He_in_search, use_csound_rms_in_search, &
         first_Z_div_X, first_Y, first_alpha, first_f_ov, first_f0_ov_fraction, &
         min_Z_div_X, min_Y, min_alpha, min_f_ov, min_f0_ov_fraction, &
         max_Z_div_X, max_Y, max_alpha, max_f_ov, max_f0_ov_fraction, &
         vary_Z_div_X, vary_Y, vary_alpha, vary_f_ov, vary_f0_ov_fraction, &
         Y_frac_he3, rhoend


      logical :: okay_to_restart

      real(dp) :: next_initial_h1_to_try, next_initial_he3_to_try, &
         next_initial_he4_to_try, next_alpha_to_try, &
         next_Z_div_X_to_try, next_Y_to_try, next_f_ov_to_try, next_f0_ov_fraction_to_try

      ! "best_model" results are for particular parameters
      real(dp) :: &
         best_model_chi_square, &
         best_model_age, &
         best_model_logR, &
         best_model_logL, &
         best_model_Teff, &
         best_model_Rcz, &
         best_model_surf_He, &
         best_model_surf_Z_div_X, &
         best_model_csound_rms
      integer :: best_model_model_number

      ! samples are for different combinations of parameters Y, Z/X, and alpha
      integer, parameter :: max_num_samples = 1000
      integer :: num_samples
      real(dp), dimension(max_num_samples) :: &
         sample_chi_square, &
         sample_age, &
         sample_logR, &
         sample_logL, &
         sample_Teff, &
         sample_init_Y, &
         sample_init_Z_div_X, &
         sample_alpha, &
         sample_f_ov, &
         sample_f0_ov_fraction, &
         sample_init_h1, &
         sample_init_h2, &
         sample_init_he3, &
         sample_init_he4, &
         sample_init_Z, &
         sample_Rcz, &
         sample_surf_He, &
         sample_surf_Z_div_X, &
         sample_csound_rms
      integer, dimension(max_num_samples) :: sample_model_number

      ! info about current model
      integer :: star_id, minimize_i_test
      real(dp) :: chi_square
      real(dp) :: initial_Y, initial_Z_div_X

      integer, parameter :: num_extra_history_columns = 0

      integer :: i_alpha, i_Y, i_Z_div_X, i_f_ov, i_f0_ov_fraction, nvar, &
         equ_L, equ_R, equ_ZX_ratio
      real(dp) :: final_alpha, final_Y, final_Z_div_X, final_f_ov, final_f0_ov_fraction


      ! solar sound speed data
      logical :: have_sound_speed_data = .false.
      integer, parameter :: npts = 79
      real(dp), dimension(npts) :: data_r, data_csound, data_width
      
      
      
      ! for newton
      ! dimensions
      integer, parameter :: nz = 1 ! number of zones
      integer, parameter :: nv = 3 ! max number of variables per zone
      integer, parameter :: neq = nz*nv

      integer :: id_for_calibration, log_unit, i_Z
      logical :: doing_setmatrix
      
      integer, parameter :: max_allowed_max_tries = 500
      integer :: iter
      real(dp), dimension(max_allowed_max_tries) :: &
         alphas, Ys, Zs, log_Ls, log_Rs, surface_ZXs, surface_Ys, Rczs, rms
         
      ! controls
      character (len=256) :: starting_model_name, ending_values_file, &
         calibration_log_file, save_model_tag
      real(dp) :: starting_alpha, starting_Y, starting_Z
      real(dp) :: tol_correction_norm, tol_max_correction, &
         tol_residual_norm1, tol_max_residual1, min_corr_coeff
      real(dp) :: stopping_age, epsder
      real(dp) :: target_surface_ZX_ratio
      logical :: use_calibration_log, save_profile_at_end, load_starting_model, calc_rms
      logical :: calibrate_alpha, calibrate_Y, calibrate_Z
      logical :: use_equ_L, use_equ_R, use_equ_ZX_ratio
      integer :: max_iter_for_resid_tol1
      
      real(dp) :: expected_rms, expected_Rcz, expected_surf_He4
      real(dp) :: expected_Rcz_plus_minus, expected_surf_He4_plus_minus
      ! Bahcall et al 2005, Rcz = 0.713 +- 0.001 Rsun, surf_Y = 0.2485 +- 0.0034
      
      
      namelist /newton_calibration_controls/ &
         starting_model_name, ending_values_file, &
         calibrate_alpha, calibrate_Y, calibrate_Z, &
         use_equ_L, use_equ_R, use_equ_ZX_ratio, &
         starting_alpha, starting_Y, starting_Z, & 
         epsder, target_log_luminosity, target_log_radius, target_surface_ZX_ratio, stopping_age, &
         use_calibration_log, calibration_log_file, save_model_tag, &
         save_profile_at_end, load_starting_model, calc_rms, max_tries, &
         tol_correction_norm, tol_max_correction, min_corr_coeff, &
         max_iter_for_resid_tol1, tol_residual_norm1, tol_max_residual1, &
         expected_rms, expected_Rcz, expected_surf_He4, &
         expected_Rcz_plus_minus, expected_surf_He4_plus_minus

      
      real(dp) :: final_Z
      


      contains
      
      
      subroutine do_run
         type (star_info), pointer :: s
         integer :: id, ierr, i
         real(dp) :: chi2, mass
         
         include 'formats'
                  
         ierr = 0
         okay_to_restart = .true.

         call read_calibration_controls(ierr)
         if (failed('read_calibration_controls')) return

         call init_and_alloc(id, s, ierr)
         if (failed('init_and_alloc')) return
         
         star_id = id
                  
         call star_setup(id, 'inlist', ierr)
         if (failed('star_setup')) return
         
         write(*,*) 'search_type == ' // trim(search_type)
         if (search_type == 'just_do_first_values') then
            call do_bobyqa(.true., ierr)
         else if (search_type == 'bobyqa') then
            call do_bobyqa(.false., ierr)
         else if (search_type == 'newton') then
            call do_newton(ierr)
         else
            write(*,*) 'bad value for search_type ' // trim(search_type)
            ierr = -1
         end if

         
         contains

         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
            end if
         end function failed
      
      
         subroutine read_calibration_controls(ierr)
            integer, intent(out) :: ierr
            character (len=256) :: filename, message
            integer :: unit
            unit=alloc_iounit(ierr)
            if (ierr /= 0) return
            filename = 'inlist_calibration_controls'
            open(unit=unit, file=trim(filename), action='read', delim='quote', iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'Failed to open control namelist file ', trim(filename)
            else
               read(unit, nml=calibration_controls, iostat=ierr)  
               close(unit)
               if (ierr /= 0) then
                  write(*, *) 'Failed while trying to read control namelist file ', trim(filename)
                  write(*, '(a)') &
                     'The following runtime error message might help you find the problem'
                  write(*, *) 
                  open(unit=unit, file=trim(filename), action='read', delim='quote', status='old', iostat=ierr)
                  read(unit, nml=calibration_controls)
                  close(unit)
               end if  
            end if
            call free_iounit(unit)
         end subroutine read_calibration_controls


      end subroutine do_run
      
      
      real(dp) function eval1(id, ierr)
         integer :: id, ierr
         logical :: restart
         logical, parameter :: do_init_and_alloc = .false., do_free_star = .false.
         ierr = 0
         call run1_star( &
            do_init_and_alloc, do_free_star, okay_to_restart, &
            id, restart, &
            extras_controls, &
            extras_startup, &
            extras_check_model, &
            how_many_extra_history_columns, &
            data_for_extra_history_columns, &
            how_many_extra_profile_columns, &
            data_for_extra_profile_columns, &
            extras_finish_step, &
            extras_after_evolve, &
            ierr)
         eval1 = best_model_chi_square
      end function eval1
         

      subroutine bobyqa_fun(n,x,f)
         integer, intent(in) :: n
         real(dp), intent(in) :: x(*)
         real(dp), intent(out) :: f
         
         integer :: ierr, id
         type (star_info), pointer :: s

         include 'formats'
         
         ierr = 0
         id = star_id
         call star_ptr(id,s,ierr)
         if (ierr /= 0) return
         
         if (vary_alpha) then
            next_alpha_to_try = bobyqa_param(x(i_alpha), first_alpha, min_alpha, max_alpha)
            write(*,1) 'next_alpha_to_try', next_alpha_to_try, x(i_alpha)
         end if
         
         if (vary_Y) then
            next_Y_to_try = bobyqa_param(x(i_Y), first_Y, min_Y, max_Y)
            write(*,1) 'next_Y_to_try', next_Y_to_try, x(i_Y)
         end if

         if (vary_Z_div_X) then
            next_Z_div_X_to_try = bobyqa_param(x(i_Z_div_X), first_Z_div_X, min_Z_div_X, max_Z_div_X)
            write(*,1) 'next_Z_div_X_to_try', next_Z_div_X_to_try, x(i_Z_div_X)
         end if
         
         if (vary_f_ov) then
            next_f_ov_to_try = bobyqa_param(x(i_f_ov), first_f_ov, min_f_ov, max_f_ov)
            write(*,1) 'next_f_ov_to_try', next_f_ov_to_try, x(i_f_ov)
         end if
         
         if (vary_f0_ov_fraction) then
            next_f0_ov_fraction_to_try = &
               bobyqa_param(x(i_f0_ov_fraction), first_f0_ov_fraction, min_f0_ov_fraction, max_f0_ov_fraction)
            if (next_f0_ov_fraction_to_try < 0) next_f0_ov_fraction_to_try = 0
            write(*,1) 'next_f0_ov_fraction_to_try', next_f0_ov_fraction_to_try, x(i_f0_ov_fraction)
         end if
         
         f = eval1(star_id, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in eval1'
            stop 'bobyqa_fun'
         end if
         
         !minimize_i_test = minimize_i_test + 1
         call save_best_for_this_sample( &
            minimize_i_test, next_alpha_to_try, next_Y_to_try, next_Z_div_X_to_try, &
            s% overshoot_f_below_nonburn, s% overshoot_f0_below_nonburn)

         write(*,*)
         write(*,*) 'current set of sample results'
         call show_all_sample_results(6,minimize_i_test,ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in show_all_sample_results'
            stop 'bobyqa_fun'
         end if

         write(*,*)
         call save_sample_results_to_file( &
            minimize_i_test,'bobyqa_results.txt',ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in save_sample_results_to_file'
            stop 'bobyqa_fun'
         end if
            
         okay_to_restart = .false. ! only allow restart on 1st call

         minimize_i_test = minimize_i_test + 1

      end subroutine bobyqa_fun

      
      real(dp) function bobyqa_param(x, first, min, max)
         real(dp), intent(in) :: x, first, min, max
         if (x > 0) then
            bobyqa_param = first + x*(max-first)
         else
            bobyqa_param = first + x*(first-min)
         end if
      end function bobyqa_param
            
      
      subroutine do_bobyqa(just_do_first_values, ierr)
         use num_lib
         logical, intent(in) :: just_do_first_values
         integer, intent(out) :: ierr
         
         integer, parameter :: iprint = 0
         real(dp), pointer, dimension(:) :: xl, xu, x, w
         real(dp) :: min_chi2, rhobeg, max_bobyqa_value
         integer :: i, npt
         include 'formats'
         ierr = 0
         minimize_i_test = 1!0
         nvar = 0
         
         if (vary_alpha) then
            nvar = nvar+1; i_alpha = nvar
            if (min_alpha >= max_alpha) then
               write(*,1) 'min_alpha >= max_alpha', min_alpha, max_alpha
               ierr = -1
            end if
         end if
         
         if (vary_Y) then
            nvar = nvar+1; i_Y = nvar
            if (min_Y >= max_Y) then
               write(*,1) 'min_Y >= max_Y', min_Y, max_Y
               ierr = -1
            end if
         end if
         
         if (vary_Z_div_X) then
            nvar = nvar+1; i_Z_div_X = nvar
            if (min_Z_div_X >= max_Z_div_X) then
               write(*,1) 'min_Z_div_X >= max_Z_div_X', min_Z_div_X, max_Z_div_X
               ierr = -1
            end if
         end if
         
         if (vary_f_ov) then
            nvar = nvar+1; i_f_ov = nvar
            if (min_f_ov >= max_f_ov) then
               write(*,1) 'min_f_ov >= max_f_ov', min_f_ov, max_f_ov
               ierr = -1
            end if
         end if
         
         if (vary_f0_ov_fraction) then
            nvar = nvar+1; i_f0_ov_fraction = nvar
            if (min_f0_ov_fraction >= max_f0_ov_fraction) then
               write(*,1) 'min_f0_ov_fraction >= max_f0_ov_fraction', &
                  min_f0_ov_fraction, max_f0_ov_fraction
               ierr = -1
            end if
         end if
         if (ierr /= 0) return
         
         npt = 2*nvar + 1
         
         allocate(xl(nvar), xu(nvar), x(nvar), w((npt+5)*(npt+nvar)+3*nvar*(nvar+5)/2))
         
         xl(:) = -1
         x(:) = 0
         xu(:) = 1
         
         if (just_do_first_values .or. nvar == 0) then         
            call bobyqa_fun(nvar,x,min_chi2)         
         else         
            !       RHOBEG and RHOEND must be set to the initial and final values of a trust
            !       region radius, so both must be positive with RHOEND no greater than
            !       RHOBEG. Typically, RHOBEG should be about one tenth of the greatest
            !       expected change to a variable, while RHOEND should indicate the
            !       accuracy that is required in the final values of the variables. An
            !       error return occurs if any of the differences XU(I)-XL(I), I=1,...,N,
            !       is less than 2*RHOBEG.
            rhobeg = 0.45d0
            max_bobyqa_value = 1d99
            call bobyqa(nvar,npt,x,xl,xu,rhobeg,rhoend,iprint,max_tries,w,bobyqa_fun,max_bobyqa_value)         
         end if
      
         if (vary_alpha) &
            final_alpha = bobyqa_param(x(i_alpha), first_alpha, min_alpha, max_alpha)
         
         if (vary_Y) &
            final_Y = bobyqa_param(x(i_Y), first_Y, min_Y, max_Y)

         if (vary_Z_div_X) &
            final_Z_div_X = bobyqa_param(x(i_Z_div_X), first_Z_div_X, min_Z_div_X, max_Z_div_X)
         
         if (vary_f_ov) &
            final_f_ov = bobyqa_param(x(i_f_ov), first_f_ov, min_f_ov, max_f_ov)
         
         if (vary_f0_ov_fraction) &
            final_f0_ov_fraction = &
               bobyqa_param(x(i_f0_ov_fraction), first_f0_ov_fraction, min_f0_ov_fraction, max_f0_ov_fraction)
         
         deallocate(xl, xu, x, w)
      
      end subroutine do_bobyqa

      
      subroutine save_sample_results_to_file(i_test, results_fname, ierr)
         use utils_lib
         integer, intent(in) :: i_test
         character (len=*), intent(in) :: results_fname
         integer, intent(out) :: ierr
         character (len=256) :: save_model_name
         character (len=4) :: i_test_str
         integer :: iounit
         write(*,*) 'save_sample_results_to_file ' // trim(results_fname)
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) stop 'alloc_iounit failed'
         open(unit=iounit, file=trim(results_fname), action='write', iostat=ierr)
         if (ierr /= 0) return
         call show_all_sample_results(iounit,i_test,ierr)
         close(iounit)
         call free_iounit(iounit)
         if (i_test < 10) then
            write(i_test_str,'(a,i1)') '000', i_test
         else if (i_test < 100) then
            write(i_test_str,'(a,i2)') '00', i_test
         else if (i_test < 1000) then
            write(i_test_str,'(a,i3)') '0', i_test
         else
            write(i_test_str,'(a,i4)') '', i_test
         end if 
         save_model_name = 'trial_out/trial_' // trim(i_test_str) // '.mod'
         !'
         write(*,*) 'save final model to ' // trim(save_model_name)
         call star_write_model(star_id, save_model_name, .true., ierr)
         write(*,*)
         
      end subroutine save_sample_results_to_file
      
      
      subroutine show_all_sample_results(iounit, i_test, ierr)
         integer, intent(in) :: iounit, i_test
         integer, intent(out) :: ierr
         integer :: i, j
         integer, pointer :: index(:)
         ! sort results by increasing sample_chi_square
         allocate(index(i_test), stat=ierr)
         if (ierr /= 0) then
            stop 'failed in allocate before calling qsort from show_all_sample_results'
         end if
         call qsort(index, i_test, sample_chi_square)
         write(iounit,'(8x,99a22)') &
               'chi_square', &
               'init_Y', &
               'init_Z_div_X', &
               'alpha', &
               'f_ov', &
               'f0_ov_fraction', &
               
               'initial_h1', &
               'initial_h2', &
               'initial_he3', &
               'initial_he4', &
               'initial Z', &               

               'logR', &
               'logL', &
               'csound_rms', &
               
               'surf_Z_div_X', &
               'target_surf_Z_div_X', &
               
               'radius_cz', &
               'target_radius_cz', &
               
               'surf_he', &
               'target_surf_he', &
               
               'sigma_surf_Z_div_X', &
               'err_surf_Z_div_X', &
               'sigma_radius_cz', &
               'err_radius_cz', &
               'sigma_surf_he', &
               'err_surf_he', &
               'sigma_csound_rms', &
               'err_csound_rms'
               
         do j = 1, i_test
            i = index(j)
            write(iounit,'(3x,i5,99f22.14)') i, &
               sample_chi_square(i), &
               sample_init_Y(i), &
               sample_init_Z_div_X(i), &
               sample_alpha(i), &
               sample_f_ov(i), &
               sample_f0_ov_fraction(i), &
               
               sample_init_h1(i), &
               sample_init_h2(i), &
               sample_init_he3(i), &
               sample_init_he4(i), &
               sample_init_Z(i), &               

               sample_logR(i), &
               sample_logL(i), &   
               sample_csound_rms(i), &

               sample_surf_Z_div_X(i), &
               target_surface_Z_div_X, &
                             
               sample_Rcz(i), &
               target_Rcz, &
                      
               sample_surf_He(i), &
               target_surf_He, &
                         
               sigma_surface_Z_div_X, &
               (sample_surf_Z_div_X(i) - target_surface_Z_div_X)/sigma_surface_Z_div_X, & 
               
               sigma_Rcz, &
               (sample_Rcz(i) - target_Rcz)/sigma_Rcz, &        
               
               sigma_surf_He, &
               (sample_surf_He(i) - target_surf_He)/sigma_surf_He, &     
               
               sigma_csound_rms, &
               (sample_csound_rms(i) - target_csound_rms)/sigma_csound_rms
               
         end do
         deallocate(index)
         
      end subroutine show_all_sample_results


      subroutine save_best_for_this_sample( &
            itry, alpha, Y, Z_div_X, f_ov, f0_ov)
         integer, intent(in) :: itry
         real(dp), intent(in) :: alpha, Y, Z_div_X, f_ov, f0_ov
      
         sample_alpha(itry) = alpha
         sample_init_Y(itry) = Y 
         sample_init_Z_div_X(itry) = Z_div_X
         sample_f_ov(itry) = f_ov
         sample_f0_ov_fraction(itry) = f0_ov/max(1d-99,f_ov)

         sample_init_h1(itry) = initial_h1
         sample_init_h2(itry) = initial_h2
         sample_init_he3(itry) = initial_he3
         sample_init_he4(itry) = initial_he4
         sample_init_Z(itry) = 1d0 - &
            (initial_h1 + initial_h2 + initial_he3 + initial_he4)         

         sample_chi_square(itry) = best_model_chi_square
         sample_logR(itry) = best_model_logR
         sample_logL(itry) = best_model_logL
         sample_Teff(itry) = best_model_Teff
         sample_Rcz(itry) = best_model_Rcz
         sample_surf_He(itry) = best_model_surf_He
         sample_surf_Z_div_X(itry) = best_model_surf_Z_div_X
         sample_csound_rms(itry) = best_model_csound_rms
         sample_model_number(itry) = best_model_model_number

         num_samples = itry

      end subroutine save_best_for_this_sample


      subroutine setup_solar_data_for_calc_rms(ierr)
         use const_def, only: mesa_data_dir
         integer, intent(out) :: ierr
         
         integer, parameter :: lines_to_skip = 11
         integer :: iounit, i, k
         real(dp) :: jnk
         
         character (len=256) :: fname
         
         have_sound_speed_data = .true.
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) then
            return
         end if
         fname = trim(mesa_data_dir) // '/star_data/solar_csound.data'
         open(iounit, file=trim(fname), action='read', status='old', iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to open ' // trim(fname)
            call free_iounit(iounit)
            return
         end if                  
         
         do i=1,lines_to_skip
            read(iounit,fmt=*,iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed when skipping line', i
               close(iounit)
               call free_iounit(iounit)
               return
            end if
         end do
         
         do i=1,npts
            read(iounit,fmt=*,iostat=ierr) &
               data_r(i), jnk, data_csound(i), jnk, jnk, jnk, data_width(i)
            if (ierr /= 0) then
               write(*,*) 'failed when reading data point', i
               close(iounit)
               call free_iounit(iounit)
               return
            end if
         end do

         close(iounit)
         call free_iounit(iounit)
      
      
      end subroutine setup_solar_data_for_calc_rms


      real(dp) function calc_current_rms(s, nz) ! dR weighted
         use interp_1d_lib
         use interp_1d_def
         type (star_info), pointer :: s
         integer, intent(in) :: nz
         
         logical, parameter :: dbg = .false.
         real(dp), target :: calc_rms_f1_ary(4*nz)
         real(dp), pointer :: calc_rms_f1(:), calc_rms_f(:,:)
         real(dp) :: sumy2, sumdr, dr, y2, cs
         real(dp), parameter :: min_R = 0.094, max_R = 0.94
         real(dp), target :: pm_work_ary(nz*pm_work_size)
         real(dp), pointer :: pm_work(:)
         integer :: k, i, ierr
         
         include 'formats'
         
         calc_current_rms = -1  
         pm_work => pm_work_ary
         calc_rms_f1 => calc_rms_f1_ary
         calc_rms_f(1:4,1:nz) => calc_rms_f1(1:4*nz)

         ierr = 0
         if (.not. have_sound_speed_data) then
            call setup_solar_data_for_calc_rms(ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in setup_solar_data_for_calc_rms'
               return
            end if
         end if
         
         do k=1,nz
            if (k == 1) then
               calc_rms_f(1,k) = s% csound(k)
            else
               calc_rms_f(1,k) = &
                  (s% csound(k)*s% dq(k-1) + s% csound(k-1)*s% dq(k))/(s% dq(k-1) + s% dq(k))
            end if
         end do

         call interp_pm(s% r, nz, calc_rms_f1, pm_work_size, pm_work, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in interp_pm'
            return
         end if

         sumy2 = 0
         sumdr = 0
         do i=1,npts
            if (data_r(i) < min_R .or. data_r(i) > max_R) cycle
            call interp_value(s% r, nz, calc_rms_f1, Rsun*data_r(i), cs, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in interp_value', i
               return
            end if
            if (i == 1) then
               dr = data_r(2) - data_r(1)
            else
               dr = data_r(i) - data_r(i-1)
            end if
            if (dr < 0) dr = -dr
            ! change to weigh by point rather than by dr
            dr = 1
            
            sumdr = sumdr + dr
            y2 = dr*((cs - data_csound(i))/data_csound(i))**2
            sumy2 = sumy2 + y2
            if (dbg) write(*,2) 'rms cs, data_cs, reldiff, y2, dr', i, cs, data_csound(i), &
               (cs - data_csound(i))/data_csound(i), y2, dr
         end do
         
         calc_current_rms = sqrt(sumy2/sumdr)
         if (.true. .or. dbg) write(*,1) 'calc_current_rms', calc_current_rms

      end function calc_current_rms




      
      
      subroutine do_newton(ierr)
         use num_lib
         integer, intent(out) :: ierr

         type (star_info), pointer :: s

         include 'formats'
         
         ierr = 0
         call star_ptr(star_id,s,ierr)
         if (ierr /= 0) return

         call do_calibrate(s, 'inlist_newton_calibration_controls', &
            history_columns_file, profile_columns_file, ierr)
         

      end subroutine do_newton



      
      subroutine do_calibrate(s, calibration_inlist, history_columns_file_in, profile_columns_file_in, ierr)
         use utils_lib
         use mtx_lib, only: lapack_decsol, null_decsolblk_quad, null_decsolblk
         use mtx_def, only: lapack
         use num_def, only: square_matrix_type
         use num_lib
         type (star_info), pointer :: s
         character (len=*) :: calibration_inlist, history_columns_file_in, profile_columns_file_in
         integer, intent(out) :: ierr
         
         integer :: nvar, m1, m2

         integer, parameter :: stencil_zones_subdiagonal = 0
         integer, parameter :: stencil_zones_superdiagonal = 0

         integer, parameter :: nsec = 0 ! number of secondaries per zone
         integer, parameter :: ldy = nz ! leading dimension of y, >= nz

         real(dp), pointer, dimension(:) :: equ1, x1, xold1, xscale1, y1
         real(dp), pointer, dimension(:,:) :: equ, x, xold, xscale, y
         logical :: first_step, nonconv, doing_jacobian
         integer :: matrix_type, matrix_type_current
         integer, parameter :: max_lid = 50000, max_lrd = 100000
         integer, target :: ipar_decsol_ary(max_lid)
         integer, pointer :: ipar_decsol(:)
         real(dp), target :: rpar_decsol_ary(max_lrd)
         real(dp), pointer :: rpar_decsol(:)
         integer :: liwork, lwork, lqwork, lid, lrd
         integer, dimension(:), pointer :: iwork 
         real(dp), dimension(:), pointer :: work
         real(qp), pointer :: qwork(:)
         integer, parameter :: lrpar = 0, lipar = 0
         integer, target :: ipar(lipar)
         real(dp), target :: rpar(lrpar)
         real(dp), pointer, dimension(:) :: AF1 ! for factored jacobian
         integer :: i

         include 'formats.dek'
                  
         write(*,*) 'calibrate'
         
         history_columns_file = history_columns_file_in
         profile_columns_file = profile_columns_file_in
         
         ierr = 0
         iter = 0
         id_for_calibration = s% id
         
         ipar_decsol => ipar_decsol_ary
         rpar_decsol => rpar_decsol_ary
         
         call read_calibration_controls(s, calibration_inlist, ierr)
         if (failed('read_calibration_controls')) return

         write(*,*) 'calibration controls: starting_model_name ' // trim(starting_model_name)
         write(*,1) 'starting_alpha', starting_alpha
         write(*,1) 'starting_Y', starting_Y
         write(*,1) 'starting_Z', starting_Z
         write(*,1) 'target_log_luminosity', target_log_luminosity
         write(*,1) 'target_log_radius', target_log_radius
         write(*,1) 'target_surface_ZX_ratio', target_surface_ZX_ratio
         write(*,1) 'stopping_age', stopping_age
         write(*,1) 'tol_correction_norm', tol_correction_norm
         write(*,1) 'tol_max_correction', tol_max_correction
         write(*,2) 'max_iter_for_resid_tol1', max_iter_for_resid_tol1
         write(*,1) 'tol_residual_norm1', tol_residual_norm1
         write(*,1) 'tol_max_residual1', tol_max_residual1
         write(*,1) 'min_corr_coeff', min_corr_coeff
         write(*,1) 'epsder', epsder
         write(*,'(a40,4x,l10)') 'use_calibration_log', use_calibration_log
         write(*,'(a40,4x,a40)') 'calibration_log_file', calibration_log_file
         write(*,'(a40,4x,l10)') 'calibrate_alpha', calibrate_alpha
         write(*,'(a40,4x,l10)') 'calibrate_Y', calibrate_Y
         write(*,'(a40,4x,l10)') 'calibrate_Z', calibrate_Z
         write(*,'(a40,4x,l10)') 'calc_rms', calc_rms
         write(*,*)
         
         nvar = 0
         if (calibrate_alpha) then
            nvar = nvar + 1
            i_alpha = nvar
         end if
         if (calibrate_Y) then
            nvar = nvar + 1
            i_Y = nvar
         end if
         if (calibrate_Z) then
            nvar = nvar + 1
            i_Z = nvar
         end if
         if (nvar == 0) then
            write(*,*) 'failed to set 1 or more calibration vars'
            ierr = -1
            return
         end if
         
         i = 0
         if (use_equ_L) then
            i = i+1
            equ_L = i
         end if
         if (use_equ_R) then
            i = i+1
            equ_R = i
         end if
         if (use_equ_ZX_ratio) then
            i = i+1
            equ_ZX_ratio = i
         end if
         if (i /= nvar) then
            write(*,*) 'number of "use_equ" must equal number of "calibrate"'
            ierr = -1
            return
         end if
         
         allocate(equ1(nvar*nz), x1(nvar*nz), xold1(nvar*nz), xscale1(nvar*nz), y1(ldy*nsec))
         equ(1:nvar,1:nz) => equ1(1:nvar*nz)
         x(1:nvar,1:nz) => x1(1:nvar*nz)
         xold(1:nvar,1:nz) => xold1(1:nvar*nz)
         xscale(1:nvar,1:nz) => xscale1(1:nvar*nz)
         y(1:ldy,1:nsec) => y1(1:ldy*nsec)

         m1 = (stencil_zones_subdiagonal+1)*nvar-1
         m2 = (stencil_zones_superdiagonal+1)*nvar-1

         if (use_calibration_log) then
            log_unit = alloc_iounit(ierr)
            if (ierr /= 0) return
            open(log_unit, file=trim(calibration_log_file), iostat=ierr)
            if (ierr /= 0) return
            write(log_unit,*) 'calibration controls: starting_model_name ' // trim(starting_model_name)
            write(log_unit,1) 'starting_alpha', starting_alpha
            write(log_unit,1) 'starting_Y', starting_Y
            write(log_unit,1) 'starting_Z', starting_Z
            write(log_unit,1) 'target_log_luminosity', target_log_luminosity
            write(log_unit,1) 'target_log_radius', target_log_radius
            write(log_unit,1) 'target_surface_ZX_ratio', target_surface_ZX_ratio
            write(log_unit,1) 'stopping_age', stopping_age
            write(log_unit,1) 'tol_correction_norm', tol_correction_norm
            write(log_unit,1) 'tol_max_correction', tol_max_correction
            write(log_unit,2) 'max_iter_for_resid_tol1', max_iter_for_resid_tol1
            write(log_unit,1) 'tol_residual_norm1', tol_residual_norm1
            write(log_unit,1) 'tol_max_residual1', tol_max_residual1
            write(log_unit,1) 'min_corr_coeff', min_corr_coeff
            write(log_unit,1) 'epsder', epsder
            write(log_unit,'(a40,4x,l10)') 'use_calibration_log', use_calibration_log
            write(log_unit,'(a40,4x,a40)') 'calibration_log_file', calibration_log_file
            write(log_unit,'(a40,4x,l10)') 'calibrate_Z', calibrate_Z
            write(log_unit,'(a40,4x,l10)') 'calc_rms', calc_rms
            write(log_unit,*)
            close(log_unit)
         end if
      
         if (calibrate_alpha) xold(i_alpha,1) = starting_alpha
         if (calibrate_Y) xold(i_Y,1) = starting_Y
         if (calibrate_Z) xold(i_Z,1) = starting_Z

         x1 = xold1
         
         lid = max_lid
         lrd = max_lrd
                  
         first_step = .true.
         
         doing_jacobian = .false.
         
         matrix_type = square_matrix_type
         
         call newton_work_sizes( &
            m1, m2, nvar, nz, nsec, matrix_type, lwork, liwork, lqwork, ierr)
         if (ierr /= 0) stop 1
         
         allocate(work(lwork), iwork(liwork), qwork(lqwork), stat=ierr)
         if (ierr /= 0) stop 1
         
         work = 0
         iwork = 0
         
         iwork(i_try_really_hard) = 1 ! try really hard for first model
         iwork(i_debug) = 1 ! output info for each iteration

         work(r_tol_max_correction) = tol_max_correction
         iwork(i_max_iter_for_enforce_resid_tol) = max_iter_for_resid_tol1
         work(r_tol_residual_norm) = tol_residual_norm1
         work(r_tol_max_residual) = tol_max_residual1
         work(r_min_corr_coeff) = min_corr_coeff
         
         nullify(AF1)
         call newton( &
            nz, nvar, x1, xold1, &
            matrix_type, m1, m2, &
            lapack_decsol, null_decsolblk, null_decsolblk_quad, &
            lrd, rpar_decsol, lid, ipar_decsol, lapack, &
            tol_correction_norm, default_set_primaries, &
            default_set_secondaries, set_xscale, &
            default_Bdomain, default_xdomain, eval_equations, &
            size_equ, sizeB, default_inspectB, &
            enter_setmatrix, exit_setmatrix, default_failed_in_setmatrix, &
            default_force_another_iter, &
            xscale1, equ1, ldy, nsec, y1, &
            work, lwork, iwork, liwork, qwork, lqwork, AF1, &
            lrpar, rpar, lipar, ipar, &
            nonconv, ierr)
         if (ierr /= 0) stop 1
         deallocate(AF1)
         
         if (nonconv) then
            write(*, *) 'failed to converge'
            stop 2
         end if
         
         if (calibrate_alpha) then
            final_alpha = x(i_alpha,1)
         else
            final_alpha = starting_alpha
         end if
         if (calibrate_Y) then
            final_Y = x(i_Y,1)
         else
            final_Y = starting_Y
         end if
         if (calibrate_Z) then
            final_Z = x(i_Z,1)
         else
            final_Z = starting_Z
         end if

         if (use_calibration_log) then
            open(log_unit, file=trim(calibration_log_file), position='append', iostat=ierr)
            if (ierr /= 0) stop 'failed to open calibration log'
            write(log_unit,*)
            if (calibrate_alpha) write(log_unit,12) 'final alpha', final_alpha
            if (calibrate_Y) write(log_unit,12) 'final Y', final_Y
            if (calibrate_Z) write(log_unit,12) 'final Z', final_Z
            write(log_unit,*)
            close(log_unit)
         else
            write(*,*)
            if (calibrate_alpha) write(*,12) 'final alpha', final_alpha
            if (calibrate_Y) write(*,12) 'final Y', final_Y
            if (calibrate_Z) write(*,12) 'final Z', final_Z
            write(*,*)
         endif
         
         call write_results
         
         call write_ending_values(ierr)
         if (ierr /= 0) stop 3
         
         call star_write_model(s% id, 'final.mod', .true., ierr)
         if (ierr /= 0) stop 4

         deallocate(iwork, work, qwork)
         deallocate(equ, x1, xold1, xscale1, y)
         
         close(log_unit)

         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
            end if
         end function failed
         
         subroutine write_results
            use const_def, only: Rsun
            use mlt_def, only: no_mixing, convective_mixing, overshoot_mixing
            real(dp) :: Rcz=0d0, Rov = 0d0
            integer :: i, io

            do i=1,s%nz-1 !locate overshoot boundary
               if (s% mixing_type(i+1) == no_mixing &
                     .and. s% mixing_type(i) == overshoot_mixing ) then
                  if (s%r(i+1)>0.25*Rsun .and. s%r(i)<0.9*Rsun) then
                     Rov = s%r(i) / Rsun
                     exit
                  endif
               endif
            enddo

            do i=1,s%nz-1 !locate convective boundary
               if (s% mixing_type(i+1) /= convective_mixing &
                     .and. s% mixing_type(i) == convective_mixing) then
                  if (s%r(i+1)>0.25*Rsun .and. s%r(i)<0.9*Rsun) then
                     Rcz = s%r(i) / Rsun
                     exit
                  endif
               endif
            enddo

            if (use_calibration_log) then
               open(log_unit, file=trim(calibration_log_file), position='append', iostat=ierr)
               if (ierr /= 0) stop 'failed to open calibration log'
               io = log_unit
            else
               io = 6
            end if

            write(io,'(a36)') '   Final Results   '
            write(io,'(a36)') '<=================>' 
            write(io,'(a36,1p,e12.5)') 'Log10(R/Rsun) = ', s% log_surface_radius
            write(io,'(a36,1p,e12.5)') 'Log10(L/Lsun) = ', s% log_surface_luminosity
            write(io,'(a36,1p,e12.5)') '[Z/X] = ', log10(surface_ZXs(iter)/target_surface_ZX_ratio)
            write(io,*)
            write(io,'(a36,1p,e12.5)') 'log center Rho = ', s% log_center_density
            write(io,'(a36,1p,e12.5)') 'log center P  = ', s% log_center_pressure
            write(io,'(a36,1p,e12.5)') 'log center T = ', s% log_center_temperature
            write(io,'(a36,1p,e12.5)') 'center H = ', s% center_h1
            write(io,'(a36,1p,e12.5)') 'center He4 = ', s% center_he4
            write(io,'(a36,1p,e12.5)') 'center Z = ', 1d0-(s% center_h1+s% center_he4)
            write(io,'(a36,1p,e12.5)') 'surface H = ', s% surface_h1
            write(io,'(a36,1p,e12.5)') 'surface He4 = ', s% surface_he4
            write(io,'(a36,1p,e12.5)') 'surface Z = ', 1d0-(s% surface_h1+s% surface_he4)
            write(io,'(a36,1p,e12.5)') 'Rcz / Rsun = ', Rcz
            write(io,'(a36,1p,e12.5)') 'Rov / Rsun = ', Rov
            write(io,'(a36,1p,e12.5)') 'rms (m-s)/s = ', rms(iter)
            write(io,*)
            write(io,*)
            write(io,'(a36,1p,e12.5)') 'rms actual-expected = ', rms(iter) - expected_rms
            write(io,'(a36,1p,e12.5)') 'Rcz (actual-expected)/+- = ', &
               (Rcz - expected_Rcz)/expected_Rcz_plus_minus
            write(io,'(a36,1p,e12.5)') 'surf_Y (actual-expected)/+- = ', &
               (s% surface_he4 - expected_surf_He4)/expected_surf_He4_plus_minus
            write(io,*)
            write(io,*)
            if (use_calibration_log) close(log_unit)

         end subroutine write_results

      end subroutine do_calibrate


      subroutine set_xscale(nvar, nz, xold, xscale, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: nvar, nz
         real(dp), pointer :: xold(:,:) ! (nvar, nz)
         real(dp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr
         real(dp), parameter :: xscale_min = 1d-3
         xscale = max(xscale_min, abs(xold))
         ierr = 0
      end subroutine set_xscale


      subroutine sizeB( &
               iter, nvar, nz, x, B, xscale, max_correction, correction_norm, &
               lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: iter, nvar, nz
         real(dp), pointer, dimension(:,:) :: x, B, xscale ! (nvar, nz)
         real(dp), intent(out) :: correction_norm ! a measure of the average correction
         real(dp), intent(out) :: max_correction ! magnitude of the max correction
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr
         include 'formats.dek'
         ierr = 0
         correction_norm = sum(abs(B))/dble(nvar*nz)
         max_correction = maxval(abs(B))
         !write(*,1) 'correction_norm', correction_norm
         !write(*,1) 'max_correction', max_correction
      end subroutine sizeB


      subroutine size_equ( &
            iter, nvar, nz, equ, residual_norm, residual_max, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: iter, nvar, nz
         real(dp), pointer :: equ(:,:) ! (nvar, nz)
         real(dp), intent(out) :: residual_norm, residual_max
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr
         include 'formats.dek'
         ierr = 0
         residual_norm = sum(abs(equ))/(nvar*nz)
         residual_max = maxval(abs(equ))
         !write(*,1) 'residual_norm', residual_norm
         !write(*,1) 'residual_max', residual_max
      end subroutine size_equ


      subroutine enter_setmatrix( &
               iter, nvar, nz, neqs, x, xold, xscale, xder, need_solver_to_eval_jacobian, &
               ldA, A1, idiag, lrpar, rpar, lipar, ipar, ierr)
         use const_def, only: dp
         integer, intent(in) :: iter, nvar, nz, neqs
         real(dp), pointer, dimension(:,:) :: x, xold, xscale, xder ! (nvar, nz)
         logical, intent(out) :: need_solver_to_eval_jacobian
         integer, intent(in) :: ldA ! leading dimension of A
         real(dp), pointer, dimension(:) :: A1 ! =(ldA,neqs)
         integer, intent(inout) :: idiag 
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr
         need_solver_to_eval_jacobian = .true.
         xder=epsder*(xscale+abs(xold))
         ierr = 0
         doing_setmatrix = .true.
      end subroutine enter_setmatrix
      
      
      subroutine exit_setmatrix( &
            iter, nvar, nz, neqs, dx, ldA, A1, idiag, xscale, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: iter, nvar, nz, neqs
         real*8, pointer :: dx(:,:) ! (nvar, nz)
         integer, intent(in) :: ldA ! leading dimension of A
         real(dp), pointer, dimension(:) :: A1 ! =(ldA,neqs)
         integer, intent(inout) :: idiag ! row of A with the matrix diagonal entries
         real(dp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr
         ierr = 0
         doing_setmatrix = .false.
      end subroutine exit_setmatrix


      subroutine eval_equations( &
            iter, nvar, nz, x, xscale, equ, lrpar, rpar, lipar, ipar, ierr)
         use const_def, only:pi
         integer, intent(in) :: iter, nvar, nz
         real(dp), pointer, dimension(:,:) :: x, xscale, equ ! (nvar, nz)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr
      	real(dp), dimension(nvar*nz, nvar*nz) :: A ! square matrix for jacobian
			logical, parameter :: skip_partials = .true.			
			call eval_equ(nvar, nz, equ, x, skip_partials, A, lrpar, rpar, lipar, ipar, ierr)         
      end subroutine eval_equations
      

      subroutine eval_equ( &
            nvar, nz, equ, x, skip_partials, A, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: nvar, nz
         real(dp), intent(out) :: equ(nvar, nz)
         real(dp), intent(in) :: x(nvar, nz)
			logical, intent(in) :: skip_partials
      	real(dp), dimension(nvar*nz, nvar*nz) :: A 
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr
         
         real(dp) :: alpha, surface_ZX_ratio, current_rms
         integer :: i, id_extra, io, id, pre_ms_relax_num_steps
         character (len=256) :: fname
         type (star_info), pointer :: s
         logical :: restore_at_end
         
         include 'formats.dek'
         
         ierr = 0
         id = id_for_calibration
         call star_ptr(id,s,ierr)
         if (failed('star_ptr')) return
         
         if (calibrate_alpha) then
            alpha = x(i_alpha,1)
         else
            alpha = starting_alpha
         end if
         if (calibrate_Y) then
            new_Y = x(i_Y,1)
         else
            new_Y = starting_Y
         end if
         
         if (calibrate_Z) then
            new_Z = x(i_Z,1)
         else
            new_Z = starting_Z
         end if
         
         set_uniform_initial_composition = .true.
         initial_h1 = max(0d0, min(1d0, 1d0 - (new_Y + new_Z)))
         initial_h2 = 0
         initial_he3 = 0
         initial_he4 = new_Y

         call do_star_job_controls_before(id, s, .false., ierr)
         if (failed('do_star_job_controls_before')) return

         s% mixing_length_alpha = alpha         
         s% max_age = stopping_age
         
         if (load_starting_model) then
            call star_read_model(id, starting_model_name, ierr)
            if (failed('star_read_model')) return
         else
            pre_ms_relax_num_steps = 30
            call star_create_pre_ms_model( id, &
               pre_ms_T_c, pre_ms_guess_rho_c, pre_ms_d_log10_P, &
               pre_ms_logT_surf_limit, pre_ms_logP_surf_limit, &
               initial_zfracs, pre_ms_relax_num_steps, ierr )
            if (failed('star_create_pre_ms_model')) return
         endif

         if (len_trim(history_columns_file) > 0) &
            write(*,*) 'read ' // trim(history_columns_file)
         call star_set_history_columns(id, history_columns_file, ierr)
         if (failed('star_set_history_columns')) return
      
         if (len_trim(profile_columns_file) > 0) &
            write(*,*) 'read ' // trim(profile_columns_file)
         call star_set_profile_columns(id, profile_columns_file, ierr)
         if (failed('star_set_profile_columns')) return

         call do_star_job_controls_after(id, s, .false., ierr)
         if (failed('do_star_job_controls_after')) return
         
         if (len_trim(save_model_tag) > 0) then
            if (iter < 10) then
               write(fname,'(a,i1,a)') 'iter_00', iter, trim(save_model_tag)
            else if (iter < 100) then
               write(fname,'(a,i2,a)') 'iter_0', iter, trim(save_model_tag)
            else
               write(fname,'(a,i3,a)') 'iter_', iter, trim(save_model_tag)
            end if
            write(*,*) 'save_model_tag: ', fname
            write(*,*)
            call star_write_model(id, fname, .true., ierr)
            if (failed('star_write_model')) return
         end if

         write(*,*)
         write(*,1) 'next alpha', alpha
         write(*,1) 'next Y', new_Y
         write(*,1) 'next Z', new_Z
         write(*,*)
         
         restore_at_end = .true.
         call star_evolve_to_limit(id, restore_at_end, ierr)
         if (failed('star_evolve_to_limit')) return
         s% star_age = stopping_age
         
         if (len_trim(save_model_tag) > 0) then
            if (iter < 10) then
               write(fname,'(a,i1,a)') 'iter_end_00', iter, trim(save_model_tag)
            else if (iter < 100) then
               write(fname,'(a,i2,a)') 'iter_end_0', iter, trim(save_model_tag)
            else
               write(fname,'(a,i3,a)') 'iter_end_', iter, trim(save_model_tag)
            end if
            write(*,*) 'save_model_tag: ', fname
            write(*,*)
            call star_write_model(id, fname, .true., ierr)
            if (failed('star_write_model')) return
         end if
                  
         if (save_profile_at_end) then
            id_extra = 0
            call save_profile(id, id_extra, &
               no_extra_profile_columns, none_for_extra_profile_columns, 10, ierr)
            if (failed('star_save_profile')) return
         end if
         
         call show_terminal_header(id, ierr)
         if (failed('show_terminal_header')) return
         
         call write_terminal_summary(id, ierr)
         if (failed('write_terminal_summary')) return
         
         surface_ZX_ratio = (1d0 - (s% surface_h1 + s% surface_he3 + s% surface_he4) ) / s% surface_h1

         if (use_equ_R) equ(equ_R,1) = s% log_surface_radius - target_log_radius
         if (use_equ_L) equ(equ_L,1) = s% log_surface_luminosity - target_log_luminosity
         if (use_equ_ZX_ratio) equ(equ_ZX_ratio,1) = surface_ZX_ratio - target_surface_ZX_ratio

         do i=1,10
            write(*,*)
         end do
         
         if (doing_setmatrix) return

         if (use_calibration_log) then
            open(log_unit, file=trim(calibration_log_file), position='append', iostat=ierr)
            if (ierr /= 0) stop 'failed to open calibration log'
            io = log_unit
         else
            io = 6
         end if

         if (iter == max_tries) then
            ierr = -1
            write(io,*) 'reach max_tries'
            return
         end if
         
         iter = iter+1
         current_rms = calc_current_rms(s% nz)

         write(*,*)
         write(*,2) 'finished alpha', iter, alpha
         write(*,2) 'finished Y', iter, new_Y
         write(*,2) 'finished Z', iter, new_Z
         write(*,*)
                  
         Ys(iter) = new_Y
         if (calibrate_alpha) then
            alphas(iter) = alpha
         else
            alphas(iter) = starting_Z
         end if
         if (calibrate_Z) then
            Zs(iter) = new_Z
         else
            Zs(iter) = starting_Z
         end if
         if (calibrate_Z) then
            Zs(iter) = new_Z
         else
            Zs(iter) = starting_Z
         end if
         log_Ls(iter) = s% log_surface_luminosity
         log_Rs(iter) = s% log_surface_radius
         surface_ZXs(iter) = surface_ZX_ratio
         surface_Ys(iter) = s% surface_he3 + s% surface_he4
         rczs(iter) = s% conv_mx1_bot_r
         rms(iter) = current_rms
         

         write(io,'(a8,99(2x,a26))') 'eval#', &
            'alpha', 'initial_Y', 'initial_Z', 'log_L', 'log_R', &
            '[Z/X]', 'surface Z/X', 'surface Y', 'Rcz', 'rms'
         do i=1,iter
            write(io,'(i8,99(2x,1pe26.16))') &
               i, alphas(i), Ys(i), Zs(i), &
               log_Ls(i), log_Rs(i), log10(surface_ZXs(i)/target_surface_ZX_ratio), &
               surface_ZXs(i), surface_Ys(i), rczs(i), rms(i)
         end do
         do i=1,10
            write(io,*)
         end do

         if (use_calibration_log) close(io)
         
         

         contains
         
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
            end if
         end function failed
         
         
         real(dp) function calc_current_rms(nz) ! dR weighted
            use interp_1d_lib
            use interp_1d_def
            
            integer, intent(in) :: nz
            
            integer, parameter :: npts = 79, lines_to_skip = 11
            integer :: iounit, ierr, i, k
            
            character (len=256) :: fname
            real(dp), dimension(npts) :: data_r, data_csound, data_width
            real(dp), target :: pm_work_ary(nz*pm_work_size), f1_ary(4*nz)
            real(dp), pointer :: f1(:), f(:,:)
            real(dp), pointer :: pm_work1(:), pm_work(:,:)
            real(dp) :: jnk, sumy2, sumdr, dr, y2, cs
            real(dp), parameter :: min_R = 0.094, max_R = 0.94
            
            logical, parameter :: dbg = .false.
            
            include 'formats.dek'
            
            f1 => f1_ary
            f(1:4,1:nz) => f1(1:4*nz)
            
            pm_work1 => pm_work_ary
            pm_work(1:nz,1:pm_work_size) => pm_work1(1:nz*pm_work_size)
            
            calc_current_rms = -1
            if (.not. calc_rms) then
               if (dbg) stop 'calc_rms is false?'
               return
            end if
            
            ierr = 0
            iounit = alloc_iounit(ierr)
            if (ierr /= 0) then
               if (dbg) stop 'failed in alloc_iounit'
               return
            end if
            fname = trim(mesa_data_dir) // '/star_data/solar_csound.data'
            open(iounit, file=trim(fname), action='read', status='old', iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed to open ' // trim(fname)
               call free_iounit(iounit)
               return
            end if                  
            
            do i=1,lines_to_skip
               read(iounit,fmt=*,iostat=ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed when skipping line', i
                  close(iounit)
                  call free_iounit(iounit)
                  return
               end if
            end do
            
            do i=1,npts
               read(iounit,fmt=*,iostat=ierr) &
                  data_r(i), jnk, data_csound(i), jnk, jnk, jnk, data_width(i)
               if (ierr /= 0) then
                  write(*,*) 'failed when reading data point', i
                  close(iounit)
                  call free_iounit(iounit)
                  return
               end if
            end do

            close(iounit)
            call free_iounit(iounit)
            
            do k=1,nz
               if (k == 1) then
                  f(1,k) = s% csound(k)
               else
                  f(1,k) = (s% csound(k)*s% dq(k-1) + s% csound(k-1)*s% dq(k))/(s% dq(k-1) + s% dq(k))
               end if
            end do

            call interp_pm(s% r, nz, f1, pm_work_size, pm_work1, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in interp_pm'
               return
            end if
            
            sumy2 = 0
            sumdr = 0
            do i=1,npts
               if (data_r(i) < min_R .or. data_r(i) > max_R) cycle
               call interp_value(s% r, nz, f1, Rsun*data_r(i), cs, ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed in interp_value', i
                  return
               end if
               if (i == 1) then
                  dr = data_r(2) - data_r(1)
               else
                  dr = data_r(i) - data_r(i-1)
               end if
               if (dr < 0) dr = -dr
               ! change to weigh by point rather than by dr
               dr = 1
               
               sumdr = sumdr + dr
               y2 = dr*((cs - data_csound(i))/data_csound(i))**2
               sumy2 = sumy2 + y2
               if (dbg) write(*,2) 'rms cs, data_cs, reldiff, y2, dr', i, cs, data_csound(i), &
                  (cs - data_csound(i))/data_csound(i), y2, dr
            end do
            
            calc_current_rms = sqrt(sumy2/sumdr)
            if (dbg) write(*,1) 'calc_current_rms', calc_current_rms

         end function calc_current_rms
         

      end subroutine eval_equ


      integer function no_extra_profile_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         no_extra_profile_columns = 0
      end function no_extra_profile_columns
      
      
      subroutine none_for_extra_profile_columns(s, id, id_extra, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine none_for_extra_profile_columns


      subroutine read_calibration_controls(s, calibration_inlist, ierr)
         use utils_lib
         type (star_info), pointer :: s
         character (len=*), intent(in) :: calibration_inlist
         integer, intent(out) :: ierr

         character (len=256) :: filename, message
         integer :: unit
         
         11 format(a30, f16.6)

         ierr = 0
         
         ! set defaults
         load_starting_model = .false.
         starting_model_name = 'solar_pre_ms.mod'
         ending_values_file = 'ending_values.data'
         save_model_tag = ''
         stopping_age = 4.57d9 ! years
         target_log_luminosity = 0 ! log10(L/Lsun)
         target_log_radius = 0 ! log10(R/Rsun)
         target_surface_ZX_ratio = 2.293d-2 !GS98 value

         expected_rms = 0.00093d0 ! from instrument paper table
         expected_Rcz = 0.713d0 ! Bahcall et al 2005, Rcz = 0.713 +- 0.001 Rsun
         expected_Rcz_plus_minus = 0.001d0
         expected_surf_He4 = 0.2485d0 ! Bahcall et al 2005, surf_Y = 0.2485 +- 0.0034
         expected_surf_He4_plus_minus = 0.0034d0


         tol_correction_norm = 1d-5
         tol_max_correction = 1d-2
         max_iter_for_resid_tol1 = 7
         tol_residual_norm1 = 1d-5
         tol_max_residual1 = 1d-2
         min_corr_coeff = 1d-3

         epsder = 1d-6 ! for numerical Jacobian derivatives of L and R wrt alpha, Y, and Z
         use_calibration_log = .false.
         calibration_log_file = 'calibration.log'
         calibrate_Z = .true.
         save_profile_at_end = .true.
         calc_rms = .true.
         max_tries = 500

         unit=alloc_iounit(ierr)
         if (ierr /= 0) return

         filename = calibration_inlist
         open(unit=unit, file=trim(filename), action='read', delim='quote', iostat=ierr)
         if (ierr /= 0) then
            write(*, *) 'Failed to open control namelist file ', trim(filename)
         else
            read(unit, nml=newton_calibration_controls, iostat=ierr)  
            close(unit)
            if (ierr /= 0) then
               write(*, *) 'Failed while trying to read control namelist file ', trim(filename)
               write(*, '(a)') &
                  'The following runtime error message might help you find the problem'
               write(*, *) 
               open(unit=unit, file=trim(filename), action='read',  &
                  delim='quote', status='old', iostat=ierr)
               read(unit, nml=newton_calibration_controls)
               close(unit)
            end if  
         end if
         
         call free_iounit(unit)

      end subroutine read_calibration_controls
      
      
      subroutine write_ending_values(ierr)
         integer, intent(out) :: ierr

         character (len=256) :: filename, message
         integer :: unit
         ierr = 0
         unit=alloc_iounit(ierr)
         if (ierr /= 0) return
         filename = ending_values_file
         open(unit=unit, file=trim(filename), iostat=ierr)
         if (ierr /= 0) then
            write(*, *) 'Failed to open ending values file ', trim(filename)
         else
            write(unit, fmt='(3a26)') 'alpha', 'Y', 'Z'
            write(unit, fmt='(3d26.16)') final_alpha, final_Y, final_Z
            close(unit)
         end if
         
         call free_iounit(unit)
      
      end subroutine write_ending_values

      








      integer function extras_check_model(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         include 'formats'
         extras_check_model = keep_going
         if (create_pre_main_sequence_model) return
         if (mod(s% model_number,100) == 0) then
            if (vary_alpha) &
               write(*,'(a20,99f16.8)') 'alpha', s% mixing_length_alpha
            if (vary_Z_div_X) &
               write(*,'(a20,99f16.8)') 'initial Z/X', initial_Z_div_X
            if (vary_Y) &
               write(*,'(a20,99f16.8)') 'initial Y', initial_Y
            if (vary_f_ov) &
               write(*,'(a20,99f16.8)') 'f_ov', s% overshoot_f_below_nonburn
            if (vary_f0_ov_fraction) &
               write(*,'(a20,99f16.8)') 'f0_ov_fraction', &
                  s% overshoot_f0_below_nonburn/s% overshoot_f_below_nonburn
            write(*,*)
         end if
      end function extras_check_model

      
      integer function extras_finish_step(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         extras_finish_step = keep_going
         if (create_pre_main_sequence_model) return
         call store_extra_info(s)
      end function extras_finish_step
      

      subroutine extras_controls(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         real(dp) :: X, Y, Z, Z_div_X
         include 'formats'
         ierr = 0
         if (create_pre_main_sequence_model) return
         ! overwrite various inlist controls
         
         if (vary_alpha) then
            s% mixing_length_alpha = next_alpha_to_try
         else
            s% mixing_length_alpha = first_alpha
         end if
         
         if (vary_Y) then
            Y = next_Y_to_try
         else
            Y = first_Y
         end if
         
         if (vary_Z_div_X) then
            Z_div_X = next_Z_div_X_to_try
         else
            Z_div_X = first_Z_div_X
         end if
         
         if (vary_f_ov) then
            s% overshoot_f_below_nonburn = next_f_ov_to_try
         else
            s% overshoot_f_below_nonburn = first_f_ov
         end if
         
         if (vary_f0_ov_fraction) then
            s% overshoot_f0_below_nonburn = &
               next_f0_ov_fraction_to_try*s% overshoot_f_below_nonburn
         else if (first_f0_ov_fraction >= 0) then
            s% overshoot_f0_below_nonburn = &
               first_f0_ov_fraction*s% overshoot_f_below_nonburn
         end if
         
         initial_Y = Y
         initial_Z_div_X = Z_div_X
         X = (1 - Y)/(1 + Z_div_X)
         Z = X*Z_div_X
         initial_h1 = X
         initial_h2 = 0
         initial_he3 = Y_frac_he3*Y
         initial_he4 = Y - initial_he3
         set_uniform_initial_composition = .true.
      end subroutine extras_controls
      
      
      integer function extras_startup(s, id, restart, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         ierr = 0
         extras_startup = 0
         if (create_pre_main_sequence_model) return
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup


      integer function how_many_extra_history_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         if (create_pre_main_sequence_model) then
            how_many_extra_history_columns = 0
            return
         end if
         how_many_extra_history_columns = num_extra_history_columns
      end function how_many_extra_history_columns
      
      
      subroutine data_for_extra_history_columns(s, id, id_extra, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         integer :: i
         
         include 'formats'
         
         ierr = 0
         if (create_pre_main_sequence_model) return


         return
         
         i = 0
         
         i = i+1
         names(i) = 'chi_square'
         vals(i) = chi_square
         
         if (i /= num_extra_history_columns) then
            write(*,2) 'i', i
            write(*,2) 'num_extra_history_columns', num_extra_history_columns
            stop 'bad num_extra_history_columns'
         end if
         
      end subroutine data_for_extra_history_columns

      
      integer function how_many_extra_profile_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(s, id, id_extra, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         integer :: k
         ierr = 0
         ! here is an example for adding a profile column
         !if (n /= 1) stop 'data_for_extra_profile_columns'
         !names(1) = 'beta'
         !do k = 1, nz
         !   vals(k,1) = s% Pgas(k)/s% P(k)
         !end do
      end subroutine data_for_extra_profile_columns

      
      subroutine extras_after_evolve(s, id, id_extra, ierr)
         use const_def, only: Rsun
         use mlt_def, only: no_mixing, convective_mixing, overshoot_mixing
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         character (len=4) :: i_test_str
         character(len=256) :: final_profile_name
         
         real(dp) :: surface_He, Rcz, surface_Z_div_X, csound_rms, &
            chi2term, chi2sum, chi_square
         integer :: chi2N, i, nz
         
         include 'formats'

         ierr = 0         
         if (create_pre_main_sequence_model) return
         nz = s% nz

         surface_He = s% surface_he3 + s% surface_he4
         surface_Z_div_X = &
            (1d0 - (s% surface_h1 + surface_He))/s% surface_h1
         csound_rms = calc_current_rms(s, nz)
         
         Rcz = 0
         do i = 1, nz-1 ! locate bottom of solar convective zone
            if (s% mixing_type(i+1) /= convective_mixing &
                  .and. s% mixing_type(i) == convective_mixing) then
               if (s% r(i+1) > 0.25*Rsun .and. s% r(i) < 0.9*Rsun) then
                  Rcz = s% r(i)/Rsun
                  exit
               end if
            end if
         end do
         
         chi2sum = 0
         chi2N = 0
         
         chi2term = ((s% log_surface_luminosity - target_log_luminosity)/sigma_log_luminosity)**2
         write(*,1) 'log L chi2term', chi2term
         chi2sum = chi2sum + chi2term
         chi2N = chi2N + 1
         
         chi2term = ((s% log_surface_radius - target_log_radius)/sigma_log_radius)**2
         write(*,1) 'log R chi2term', chi2term
         chi2sum = chi2sum + chi2term
         chi2N = chi2N + 1

         chi2term = ((surface_Z_div_X-target_surface_Z_div_X)/sigma_surface_Z_div_X)**2
         write(*,1) 'Z_div_X chi2term', chi2term
         chi2sum = chi2sum + chi2term
         chi2N = chi2N + 1

         if (use_Rcz_in_search) then
            chi2term = ((Rcz-target_Rcz)/sigma_Rcz)**2
            write(*,1) 'Rcz chi2term', chi2term
            chi2sum = chi2sum + chi2term
            chi2N = chi2N + 1
         end if
         
         if (use_sigma_surf_He_in_search) then
            chi2term = ((surface_He-target_surf_He)/sigma_surf_He)**2
            write(*,1) 'surface_He chi2term', chi2term
            chi2sum = chi2sum + chi2term
            chi2N = chi2N + 1
         end if
         
         if (use_csound_rms_in_search) then
            chi2term = ((csound_rms-target_csound_rms)/sigma_csound_rms)**2
            write(*,1) 'csound_rms chi2term', chi2term
            chi2sum = chi2sum + chi2term
            chi2N = chi2N + 1
         end if
         
         ! add more chi^2 terms as desired
         
         chi_square = chi2sum/chi2N
         write(*,1) 'chi_square', chi_square
         
         best_model_chi_square = chi_square
         best_model_age = s% star_age
         best_model_logR = log10(s% photosphere_r)
         best_model_logL = s% log_surface_luminosity
         best_model_Teff = s% Teff
         best_model_Rcz = Rcz
         best_model_surf_He = surface_He
         best_model_surf_Z_div_X = surface_Z_div_X
         best_model_csound_rms = csound_rms
         
         best_model_model_number = s% model_number

         !save final profile for each iteration
         if (minimize_i_test < 10) then
            write(i_test_str,'(a,i1)') '000', minimize_i_test
         else if (minimize_i_test < 100) then
            write(i_test_str,'(a,i2)') '00', minimize_i_test
         else if (minimize_i_test < 1000) then
            write(i_test_str,'(a,i3)') '0', minimize_i_test
         else
            write(i_test_str,'(a,i4)') '', minimize_i_test
         end if
         final_profile_name = 'trial_out/trial_' // trim(i_test_str) // '.profile'
         write(*,*) 'save final profile to ' // trim(final_profile_name)
         call star_write_profile_info(star_id, final_profile_name, id_extra, &
         how_many_extra_profile_columns, data_for_extra_profile_columns, ierr)
         
      end subroutine extras_after_evolve
      
      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            real(dp) :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info
      

      end module run_star_extras
