! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      use run_star_support
      
      implicit none


         
      ! controls for sample zams
      double precision :: sample_zams_mlo, sample_zams_mhi, sample_zams_dmass
      namelist /sample_zams_job/ &
         sample_zams_mlo, sample_zams_mhi, sample_zams_dmass
      
      
      integer :: time0, time1, clock_rate
      double precision, parameter :: expected_runtime = 0.1 ! minutes

      
      contains


      subroutine do_run
         integer :: id, ierr, id_extra
         double precision :: dt
         logical :: restart
         type (star_info), pointer :: s

         write(*,*) 'do sample zams'

         ierr = 0
         id_extra = 0

         call system_clock(time0,clock_rate)
         
         call init_and_alloc(id, s, ierr)
         
         call star_setup(id, 'inlist', ierr)
         if (failed('star_setup')) return

         call extras_controls(s, ierr)
         if (failed('extras_controls')) return

         call do_star_job_controls_before(id, s, .false., ierr)
         if (failed('do_star_job_controls_before')) return
         
         call do_sample_zams(s, sample_zams_mlo, sample_zams_mhi, sample_zams_dmass, ierr)
         
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         if (dt > 10*expected_runtime) then
            write(*,'(/,a30,2f18.6,a,/)') '>>>>>>> EXCESSIVE runtime', &
               dt, expected_runtime, '   <<<<<<<<<  ERROR'
         else
            write(*,'(/,a50,2f18.6,99i10/)') 'runtime, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         end if
         
         contains
         

         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
            end if
         end function failed


      end subroutine do_run    


      subroutine do_sample_zams(s, mlo, mhi, dmass, ierr)
         type (star_info), pointer :: s
         double precision, intent(in) :: mlo, mhi, dmass
         integer, intent(out) :: ierr

         integer :: i, n
         
         include 'formats'
      
         ierr = 0

         call read_zams_controls(s, 'inlist_zams_specification', ierr)
         if (failed('read_zams_controls')) return
         
         write (*, '(/, a)') &
               '            M/Msun    log(L/Lsun)    log(R/Rsun)      log(tnuc)' // &
               '       Teff           rho_c           T7cntr      log(Pcntr)'
                  
         n = (mhi-mlo)/dmass + 1

         do i=1,n
            s% initial_mass = 10**(mlo+(i-1)*dmass)
            call star_load_zams(s% id, ierr)
            if (failed('star_load_zams')) return
            write(*, '(3x, 99(f15.6))') &
               s% star_mass, &
               s% log_surface_luminosity, &
               s% log_surface_radius, &
               log10(s% nuc_timescale+1d-99), &
               s% Teff, &
               10d0**s% log_center_density, &
               10d0**s% log_center_temperature*1e-7, &
               s% log_center_pressure
         end do
         
         write(*,*)
         write(*, '(a)') 'finished sample_zams'
         write(*,*)
         
         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
               stop 1
            end if
         end function failed
         
      end subroutine do_sample_zams



      subroutine read_zams_controls(s, zams_inlist, ierr)
         use utils_lib
         type (star_info), pointer :: s
         character (len=*), intent(in) :: zams_inlist
         integer, intent(out) :: ierr

         character (len=256) :: filename
         integer :: unit
         
         11 format(a30, f16.6)
         
         ierr = 0
         
         ! set defaults
         sample_zams_mlo = 0
         sample_zams_mhi = 1
         sample_zams_dmass = 0.5

         unit=alloc_iounit(ierr)
         if (ierr /= 0) return

         filename = zams_inlist
         open(unit=unit, file=trim(filename), action='read', delim='quote', iostat=ierr)
         if (ierr /= 0) then
            write(*, *) 'Failed to open control namelist file ', trim(filename)
         else
            read(unit, nml=sample_zams_job, iostat=ierr)  
            close(unit)
            if (ierr /= 0) then
               write(*, *) 'Failed while trying to read control namelist file ', trim(filename)
               write(*, '(a)') &
                  'The following runtime error message might help you find the problem'
               write(*, *) 
               open(unit=unit, file=trim(filename), action='read',  &
                  delim='quote', status='old', iostat=ierr)
               read(unit, nml=sample_zams_job)
               close(unit)
            end if  
         end if
         call free_iounit(unit)

      end subroutine read_zams_controls

      
      include 'standard_run_star_extras.inc'

      end module run_star_extras
      
