! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      
      implicit none
      
      integer :: time0, time1, clock_rate
      double precision, parameter :: expected_runtime = 1 ! minutes

      
      ! these routines are called by the standard run_star check_model
      contains
      
      
      subroutine extras_controls(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         ierr = 0
         
         s% use_other_eos = .true.
         s% use_other_kap = .true.
         s% use_other_mlt = .true.
         s% use_other_mesh_functions = .true.
         
         s% other_eosDT_get => my_eosDT_get
         s% other_eosDT_get_T => my_eosDT_get_T
         s% other_eosDT_get_Rho => my_eosDT_get_Rho
         s% other_eosPT_get => my_eosPT_get
         s% other_eosPT_get_T => my_eosPT_get_T
         s% other_eosPT_get_Pgas => my_eosPT_get_Pgas
         s% other_eosPT_get_Pgas_for_Rho => my_eosPT_get_Pgas_for_Rho
         
         s% other_kap_get_Type1 => my_kap_get_Type1
         s% other_kap_get_Type2 => my_kap_get_Type2

         s% other_mlt => my_mlt

         s% how_many_other_mesh_fcns => how_many_other_mesh_fcns
         s% other_mesh_fcn_data => other_mesh_fcn_data
         
      end subroutine extras_controls
      
      
      integer function extras_startup(s, id, restart, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         ierr = 0
         extras_startup = 0
         call system_clock(time0,clock_rate)
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup
      
      
      subroutine extras_after_evolve(s, id, id_extra, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         double precision :: dt
         ierr = 0
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         if (dt > 10*expected_runtime) then
            write(*,'(/,a30,2f18.6,a,/)') '>>>>>>> EXCESSIVE runtime', &
               dt, expected_runtime, '   <<<<<<<<<  ERROR'
         else
            write(*,'(/,a50,2f18.6,99i10/)') 'runtime, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         end if
      end subroutine extras_after_evolve
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         extras_check_model = keep_going         
      end function extras_check_model


      integer function how_many_extra_history_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_history_columns = 0
      end function how_many_extra_history_columns
      
      
      subroutine data_for_extra_history_columns(s, id, id_extra, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine data_for_extra_history_columns

      
      integer function how_many_extra_profile_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(s, id, id_extra, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         double precision :: vals(nz,n)
         integer, intent(out) :: ierr
         integer :: k
         ierr = 0
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going or terminate.
      integer function extras_finish_step(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer :: ierr
         extras_finish_step = keep_going
         call store_extra_info(s)
      end function extras_finish_step
      
      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            double precision :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info





      
      subroutine my_eosDT_get( &
              id, k, handle, Z, X, abar, zbar, & 
              species, chem_id, net_iso, xa, &
              Rho, log10Rho, T, log10T, & 
              res, d_dlnRho_c_T, d_dlnT_c_Rho, &
              d_dabar_c_TRho, d_dzbar_c_TRho, ierr)

         use chem_def, only: num_chem_isos
         use eos_def
         use eos_lib
         
         ! INPUT
         
         integer, intent(in) :: id ! star id if available; 0 otherwise
         integer, intent(in) :: k ! cell number or 0 if not for a particular cell         
         integer, intent(in) :: handle ! eos handle
         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
         real(dp), intent(in) :: abar
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
         integer, pointer :: net_iso(:) ! maps chem id to species number
         real(dp), intent(in) :: xa(:) ! mass fractions
         real(dp), intent(in) :: Rho, log10Rho ! the density
         real(dp), intent(in) :: T, log10T ! the temperature
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_c_T(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dlnT_c_Rho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dabar_c_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_c_TRho(:) ! (num_eos_basic_results) 
         
         integer, intent(out) :: ierr ! 0 means AOK.
         
         call eosDT_get( &
              handle, Z, X, abar, zbar, & 
              species, chem_id, net_iso, xa, &
              Rho, log10Rho, T, log10T, & 
              res, d_dlnRho_c_T, d_dlnT_c_Rho, &
              d_dabar_c_TRho, d_dzbar_c_TRho, ierr)
         
      end subroutine my_eosDT_get
      
      
      ! the following routine uses gas pressure and temperature as input variables
      subroutine my_eosPT_get(&
               id, k, handle, Z, X, abar, zbar, &
               species, chem_id, net_iso, xa,&
               Pgas, log10Pgas, T, log10T, &
               Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               res, d_dlnRho_const_T, d_dlnT_const_Rho, &
               d_dabar_c_TRho, d_dzbar_c_TRho, ierr)

         use eos_def
         use eos_lib
         use chem_def, only: num_chem_isos

         ! INPUT
         
         integer, intent(in) :: id ! star id if available; 0 otherwise
         integer, intent(in) :: k ! cell number or 0 if not for a particular cell         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: Pgas, log10Pgas ! the gas pressure
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
            ! "arg_not_provided" is defined in mesa const_def
            
         real(dp), intent(in) :: T, log10T ! the temperature
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
                     
         ! OUTPUT
         
         real(dp), intent(out) :: Rho, log10Rho ! density
         real(dp), intent(out) :: dlnRho_dlnPgas_const_T
         real(dp), intent(out) :: dlnRho_dlnT_const_Pgas
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         ! partial derivatives of the basic results wrt lnd and lnT
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results) 
         ! d_dlnRho_const_T(i) = d(res(i))/dlnd|T
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results) 
         ! d_dlnT_const_Rho(i) = d(res(i))/dlnT|Rho
         real(dp), intent(out) :: d_dabar_c_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_c_TRho(:) ! (num_eos_basic_results) 
         
         integer, intent(out) :: ierr ! 0 means AOK.
         
         call eosPT_get( &
            handle, Z, X, abar, zbar, &
            species, chem_id, net_iso, xa,&
            Pgas, log10Pgas, T, log10T, &
            Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            res, d_dlnRho_const_T, d_dlnT_const_Rho, &
            d_dabar_c_TRho, d_dzbar_c_TRho, ierr)
         
      end subroutine my_eosPT_get
      
      

      ! eosDT search routines.
      
      subroutine my_eosDT_get_T( &
               id, k, handle, Z, X, abar, zbar, &
               species, chem_id, net_iso, xa, &
               logRho, which_other, other_value, &
               logT_tol, other_tol, max_iter, logT_guess, & 
               logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, &
               logT_result, res, d_dlnRho_c_T, d_dlnT_c_Rho, &
               d_dabar_c_TRho, d_dzbar_c_TRho, &
               eos_calls, ierr)
     
         ! finds log10 T given values for density and 'other', and initial guess for temperature.
         ! does up to max_iter attempts until logT changes by less than tol.
         
         ! 'other' can be any of the basic result variables for the eos
         ! specify 'which_other' by means of the definitions in eos_def (e.g., i_lnE)
         
         use eos_def
         use eos_lib

         integer, intent(in) :: id ! star id if available; 0 otherwise
         integer, intent(in) :: k ! cell number or 0 if not for a particular cell         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: logRho ! log10 of density
         integer, intent(in) :: which_other ! from eos_def.  e.g., i_lnE
         real(dp), intent(in) :: other_value ! desired value for the other variable
         real(dp), intent(in) :: other_tol
         
         real(dp), intent(in) :: logT_tol
         integer, intent(in) :: max_iter ! max number of iterations        

         real(dp), intent(in) :: logT_guess ! log10 of temperature
         real(dp), intent(in) :: logT_bnd1, logT_bnd2 ! bounds for logT
            ! if don't know bounds, just set to arg_not_provided (defined in const_def)
         real(dp), intent(in) :: other_at_bnd1, other_at_bnd2 ! values at bounds
            ! if don't know these values, just set to arg_not_provided (defined in const_def)

         real(dp), intent(out) :: logT_result ! log10 of temperature
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_c_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_c_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_c_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_c_TRho(:) ! (num_eos_basic_results) 
         
         integer, intent(out) :: eos_calls
         integer, intent(out) :: ierr ! 0 means AOK.
         
         real(dp) :: helm_res(num_helm_results)
         
         call eosDT_get_T( &
            handle, Z, X, abar, zbar, &
            species, chem_id, net_iso, xa, &
            logRho, which_other, other_value, &
            logT_tol, other_tol, max_iter, logT_guess, & 
            logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, &
            logT_result, res, d_dlnRho_c_T, d_dlnT_c_Rho, &
            d_dabar_c_TRho, d_dzbar_c_TRho, &
            helm_res, eos_calls, ierr)
         

      end subroutine my_eosDT_get_T
      

      subroutine my_eosDT_get_Rho( &
               id, k, handle, Z, X, abar, zbar, &
               species, chem_id, net_iso, xa, &
               logT, which_other, other_value, &
               logRho_tol, other_tol, max_iter, logRho_guess,  &
               logRho_bnd1, logRho_bnd2, other_at_bnd1, other_at_bnd2, &
               logRho_result, res, d_dlnRho_c_T, d_dlnT_c_Rho, &
               d_dabar_c_TRho, d_dzbar_c_TRho, eos_calls, ierr)
     
         ! finds log10 Rho given values for temperature and 'other', and initial guess for density.
         ! does up to max_iter attempts until logRho changes by less than tol.
         
         ! 'other' can be any of the basic result variables for the eos
         ! specify 'which_other' by means of the definitions in eos_def (e.g., i_lnE)
         
         use chem_def, only: num_chem_isos
         use eos_def
         use eos_lib
         
         integer, intent(in) :: id ! star id if available; 0 otherwise
         integer, intent(in) :: k ! cell number or 0 if not for a particular cell         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: logT ! log10 of temperature

         integer, intent(in) :: which_other ! from eos_def.  e.g., i_lnE
         real(dp), intent(in) :: other_value ! desired value for the other variable
         real(dp), intent(in) :: other_tol
         
         real(dp), intent(in) :: logRho_tol

         integer, intent(in) :: max_iter ! max number of Newton iterations        

         real(dp), intent(in) :: logRho_guess ! log10 of density
         real(dp), intent(in) :: logRho_bnd1, logRho_bnd2 ! bounds for logRho
            ! if don't know bounds, just set to arg_not_provided (defined in const_def)
         real(dp), intent(in) :: other_at_bnd1, other_at_bnd2 ! values at bounds
            ! if don't know these values, just set to arg_not_provided (defined in const_def)

         real(dp), intent(out) :: logRho_result ! log10 of density

         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_c_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_c_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_c_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_c_TRho(:) ! (num_eos_basic_results) 

         integer, intent(out) :: eos_calls
         integer, intent(out) :: ierr ! 0 means AOK.
         
         real(dp) :: helm_res(num_helm_results)
         
         call eosDT_get_Rho( &
            handle, Z, X, abar, zbar, &
            species, chem_id, net_iso, xa, &
            logT, which_other, other_value, &
            logRho_tol, other_tol, max_iter, logRho_guess,  &
            logRho_bnd1, logRho_bnd2, other_at_bnd1, other_at_bnd2, &
            logRho_result, res, d_dlnRho_c_T, d_dlnT_c_Rho, &
            d_dabar_c_TRho, d_dzbar_c_TRho, helm_res, eos_calls, ierr)
         

      end subroutine my_eosDT_get_Rho
      
      
      
      ! eosPT search routines
      
      subroutine my_eosPT_get_T( &
               id, k, handle, Z, X, abar, zbar, &
               species, chem_id, net_iso, xa,&
               logPgas, which_other, other_value,&
               logT_tol, other_tol, max_iter, logT_guess, &
               logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2,&
               logT_result, Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               res, d_dlnRho_const_T, d_dlnT_const_Rho, &
               d_dabar_c_TRho, d_dzbar_c_TRho, eos_calls, ierr)
     
         ! finds log10 T given values for gas pressure and 'other',
         ! and initial guess for temperature.
         ! does up to max_iter attempts until logT changes by less than tol.
         
         ! 'other' can be any of the basic result variables for the eos
         ! specify 'which_other' by means of the definitions in eos_def (e.g., i_lnE)
         
         use chem_def, only: num_chem_isos
         use eos_def
         use eos_lib
         
         integer, intent(in) :: id ! star id if available; 0 otherwise
         integer, intent(in) :: k ! cell number or 0 if not for a particular cell         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: logPgas ! log10 of gas pressure
         integer, intent(in) :: which_other ! from eos_def.  e.g., i_lnE
         real(dp), intent(in) :: other_value ! desired value for the other variable
         real(dp), intent(in) :: other_tol
         
         real(dp), intent(in) :: logT_tol
         integer, intent(in) :: max_iter ! max number of iterations        

         real(dp), intent(in) :: logT_guess ! log10 of temperature
         real(dp), intent(in) :: logT_bnd1, logT_bnd2 ! bounds for logT
            ! if don't know bounds, just set to arg_not_provided (defined in const_def)
         real(dp), intent(in) :: other_at_bnd1, other_at_bnd2 ! values at bounds
            ! if don't know these values, just set to arg_not_provided (defined in const_def)

         real(dp), intent(out) :: logT_result ! log10 of temperature
         real(dp), intent(out) :: Rho, log10Rho ! density
         real(dp), intent(out) :: dlnRho_dlnPgas_const_T
         real(dp), intent(out) :: dlnRho_dlnT_const_Pgas

         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_c_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_c_TRho(:) ! (num_eos_basic_results) 
         
         integer, intent(out) :: eos_calls
         integer, intent(out) :: ierr ! 0 means AOK.
         
         real(dp) :: helm_res(num_helm_results)
         
         call eosPT_get_T( &
            handle, Z, X, abar, zbar, &
            species, chem_id, net_iso, xa,&
            logPgas, which_other, other_value,&
            logT_tol, other_tol, max_iter, logT_guess, &
            logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2,&
            logT_result, Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            res, d_dlnRho_const_T, d_dlnT_const_Rho, &
            d_dabar_c_TRho, d_dzbar_c_TRho, &
            helm_res, eos_calls, ierr)
         
               
      end subroutine my_eosPT_get_T
      

      subroutine my_eosPT_get_Pgas(&
               id, k, handle, Z, X, abar, zbar, &
               species, chem_id, net_iso, xa,&
               logT, which_other, other_value,&
               logPgas_tol, other_tol, max_iter, logPgas_guess, &
               logPgas_bnd1, logPgas_bnd2, other_at_bnd1, other_at_bnd2,&
               logPgas_result, Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               res, d_dlnRho_const_T, d_dlnT_const_Rho, &
               d_dabar_c_TRho, d_dzbar_c_TRho, eos_calls, ierr)
     
         ! finds log10 Pgas given values for temperature and 'other', and initial guess for gas pressure.
         ! does up to max_iter attempts until logPgas changes by less than tol.
         
         ! 'other' can be any of the basic result variables for the eos
         ! specify 'which_other' by means of the definitions in eos_def (e.g., i_lnE)
         
         use chem_def, only: num_chem_isos
         use eos_def
         use eos_lib
         
         integer, intent(in) :: id ! star id if available; 0 otherwise
         integer, intent(in) :: k ! cell number or 0 if not for a particular cell         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: logT ! log10 of temperature

         integer, intent(in) :: which_other ! from eos_def.  e.g., i_lnE
         real(dp), intent(in) :: other_value ! desired value for the other variable
         real(dp), intent(in) :: other_tol
         
         real(dp), intent(in) :: logPgas_tol

         integer, intent(in) :: max_iter ! max number of Newton iterations        

         real(dp), intent(in) :: logPgas_guess ! log10 of gas pressure
         real(dp), intent(in) :: logPgas_bnd1, logPgas_bnd2 ! bounds for logPgas
            ! if don't know bounds, just set to arg_not_provided (defined in const_def)
         real(dp), intent(in) :: other_at_bnd1, other_at_bnd2 ! values at bounds
            ! if don't know these values, just set to arg_not_provided (defined in const_def)

         real(dp), intent(out) :: logPgas_result ! log10 of gas pressure
         real(dp), intent(out) :: Rho, log10Rho ! density
         real(dp), intent(out) :: dlnRho_dlnPgas_const_T
         real(dp), intent(out) :: dlnRho_dlnT_const_Pgas

         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_c_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_c_TRho(:) ! (num_eos_basic_results) 

         integer, intent(out) :: eos_calls
         integer, intent(out) :: ierr ! 0 means AOK.
         
         real(dp) :: helm_res(num_helm_results)
         
         call eosPT_get_Pgas(&
            handle, Z, X, abar, zbar, &
            species, chem_id, net_iso, xa,&
            logT, which_other, other_value,&
            logPgas_tol, other_tol, max_iter, logPgas_guess, &
            logPgas_bnd1, logPgas_bnd2, other_at_bnd1, other_at_bnd2,&
            logPgas_result, Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            res, d_dlnRho_const_T, d_dlnT_const_Rho, &
            d_dabar_c_TRho, d_dzbar_c_TRho, helm_res, eos_calls, ierr)
         
         
      end subroutine my_eosPT_get_Pgas
      

      subroutine my_eosPT_get_Pgas_for_Rho(&
               id, k, handle, Z, X, abar, zbar, &
               species, chem_id, net_iso, xa,&
               logT, logRho_want,&
               logPgas_tol, logRho_tol, max_iter, logPgas_guess, &
               logPgas_bnd1, logPgas_bnd2, logRho_at_bnd1, logRho_at_bnd2,&
               logPgas_result, Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               res, d_dlnRho_const_T, d_dlnT_const_Rho, &
               d_dabar_c_TRho, d_dzbar_c_TRho, eos_calls, ierr)
     
         ! finds log10 Pgas given values for temperature and density, and initial guess for gas pressure.
         ! does up to max_iter attempts until logPgas changes by less than tol.
         
         use chem_def, only: num_chem_isos         
         use eos_def
         use eos_lib
         
         integer, intent(in) :: id ! star id if available; 0 otherwise
         integer, intent(in) :: k ! cell number or 0 if not for a particular cell         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: logT ! log10 of temperature

         real(dp), intent(in) :: logRho_want ! log10 of desired density
         real(dp), intent(in) :: logRho_tol
         
         real(dp), intent(in) :: logPgas_tol

         integer, intent(in) :: max_iter ! max number of Newton iterations        

         real(dp), intent(in) :: logPgas_guess ! log10 of gas pressure
         real(dp), intent(in) :: logPgas_bnd1, logPgas_bnd2 ! bounds for logPgas
            ! if don't know bounds, just set to arg_not_provided (defined in const_def)
         real(dp), intent(in) :: logRho_at_bnd1, logRho_at_bnd2 ! values at bounds
            ! if don't know these values, just set to arg_not_provided (defined in const_def)

         real(dp), intent(out) :: logPgas_result ! log10 of gas pressure
         real(dp), intent(out) :: Rho, logRho ! density corresponding to logPgas_result
         real(dp), intent(out) :: dlnRho_dlnPgas_const_T
         real(dp), intent(out) :: dlnRho_dlnT_const_Pgas

         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_c_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_c_TRho(:) ! (num_eos_basic_results) 

         integer, intent(out) :: eos_calls
         integer, intent(out) :: ierr ! 0 means AOK.
         
         real(dp) :: helm_res(num_helm_results)

         call eosPT_get_Pgas_for_Rho( &
            handle, Z, X, abar, zbar, &
            species, chem_id, net_iso, xa,&
            logT, logRho_want,&
            logPgas_tol, logRho_tol, max_iter, logPgas_guess, &
            logPgas_bnd1, logPgas_bnd2, logRho_at_bnd1, logRho_at_bnd2,&
            logPgas_result, Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            res, d_dlnRho_const_T, d_dlnT_const_Rho, &
            d_dabar_c_TRho, d_dzbar_c_TRho, helm_res, eos_calls, ierr)
         
         
      end subroutine my_eosPT_get_Pgas_for_Rho


      

      subroutine my_kap_get_Type1( &
            id, k, handle, zbar, X, Zbase, log10_rho, log10_T,  &
            species, chem_id, net_iso, xa, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dln_kap_dlnRho, dln_kap_dlnT, ierr)
         
         use chem_def, only: num_chem_isos
         use kap_lib
 
         ! INPUT
         integer, intent(in) :: id ! star id if available; 0 otherwise
         integer, intent(in) :: k ! cell number or 0 if not for a particular cell         
         integer, intent(in) :: handle ! from alloc_kap_handle
         real(dp), intent(in) :: zbar ! average ion charge
         real(dp), intent(in) :: X ! the hydrogen mass fraction
         real(dp), intent(in) :: Zbase ! the metallicity
         real(dp), intent(in) :: log10_rho ! the density
         real(dp), intent(in) :: log10_T ! the temperature
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         double precision, intent(in) :: lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
            ! free_e := total combined number per nucleon of free electrons and positrons
         
         ! OUTPUT
         real(dp), intent(out) :: kap ! opacity
         real(dp), intent(out) :: dln_kap_dlnRho ! partial derivative at constant T
         real(dp), intent(out) :: dln_kap_dlnT   ! partial derivative at constant Rho
         integer, intent(out) :: ierr ! 0 means AOK.
         
         call kap_get_Type1( &
            handle, zbar, X, Zbase, log10_rho, log10_T,  &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dln_kap_dlnRho, dln_kap_dlnT, ierr)

      end subroutine my_kap_get_Type1
           

      subroutine my_kap_get_Type2( &
            id, k, handle, zbar, X, Z, Zbase, XC, XN, XO, XNe, &
            log10_rho, log10_T, species, chem_id, net_iso, xa, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            frac_Type2, kap, dln_kap_dlnRho, dln_kap_dlnT, ierr)
         
         use kap_lib
 
         ! INPUT
         integer, intent(in) :: id ! star id if available; 0 otherwise
         integer, intent(in) :: k ! cell number or 0 if not for a particular cell         
         integer, intent(in) :: handle ! from alloc_kap_handle
         real(dp), intent(in) :: zbar ! average ion charge
         real(dp), intent(in) :: X, Z, Zbase, XC, XN, XO, XNe ! composition    
         real(dp), intent(in) :: log10_rho ! density
         real(dp), intent(in) :: log10_T ! temperature
         double precision, intent(in) :: lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
            ! free_e := total combined number per nucleon of free electrons and positrons

         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         ! OUTPUT
         real(dp), intent(out) :: frac_Type2
         real(dp), intent(out) :: kap ! opacity
         real(dp), intent(out) :: dln_kap_dlnRho ! partial derivative at constant T
         real(dp), intent(out) :: dln_kap_dlnT   ! partial derivative at constant Rho
         integer, intent(out) :: ierr ! 0 means AOK.
                  
         call kap_get_Type2( &
            handle, zbar, X, Z, Zbase, XC, XN, XO, XNe, log10_rho, log10_T, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            frac_Type2, kap, dln_kap_dlnRho, dln_kap_dlnT, ierr)

      end subroutine my_kap_get_Type2
      
      
      subroutine my_mlt(  &
            id, k, cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, X, opacity, grada, &
            gradr_factor, use_Ledoux_criterion, gradL_composition_term, &
            alpha_semiconvection, semiconvection_option, &
            thermo_haline_coeff, thermohaline_option, &
            dominant_iso_for_thermohaline, &
            mixing_length_alpha, alt_scale_height, &
            MLT_option, use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, prev_Del_T, dt, tau, dbg, &
            mixing_type, Del_T, mlt_basics, mlt_partials1, ierr)
         use mlt_lib, only: mlt_eval
         use mlt_def
         integer, intent(in) :: id ! id for star         
         integer, intent(in) :: k ! cell number or 0 if not for a particular cell         
         real(dp), intent(in) :: cgrav, m, r, T, Rho, L, P
         real(dp), intent(in) :: chiRho, chiT, Cp, Cv, X, opacity, grada, gradr_factor
         logical, intent(in) :: use_Ledoux_criterion
         real(dp), intent(in) :: gradL_composition_term
         real(dp), intent(in) :: alpha_semiconvection, thermo_haline_coeff
         real(dp), intent(in) :: mixing_length_alpha
         logical, intent(in) :: alt_scale_height, use_Henyey_MLT
         character (len=*), intent(in) :: &
            semiconvection_option, thermohaline_option, MLT_option
         integer, intent(in) :: dominant_iso_for_thermohaline
         real(dp), intent(in) :: Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, prev_Del_T, dt, tau
         logical, intent(in) :: dbg
         integer, intent(out) :: mixing_type
         real(dp), intent(out) :: Del_T
         real(dp), intent(out) :: mlt_basics(:) 
         real(dp), intent(out), pointer :: mlt_partials1(:)
         integer, intent(out) :: ierr
         call mlt_eval(  &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, X, opacity, grada,  &
            gradr_factor, use_Ledoux_criterion, gradL_composition_term, &
            alpha_semiconvection, semiconvection_option, &
            thermo_haline_coeff, thermohaline_option, &
            dominant_iso_for_thermohaline, &
            mixing_length_alpha, alt_scale_height, &
            MLT_option, use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, prev_Del_T, dt, tau, dbg, &
            mixing_type, Del_T, mlt_basics, mlt_partials1, ierr)
      end subroutine my_mlt

      
      ! here is an example that adds a mesh function for log(opacity)
      subroutine how_many_other_mesh_fcns(id, n)
         integer, intent(in) :: id
         integer, intent(out) :: n
         n = 1
      end subroutine how_many_other_mesh_fcns
      
      
      subroutine other_mesh_fcn_data(id, nfcns, names, vals1, ierr)
         integer, intent(in) :: id
         integer, intent(in) :: nfcns
         character (len=*) :: names(:)
         real(dp), pointer :: vals1(:) ! =(nz, nfcns)
         integer, intent(out) :: ierr
         integer :: nz, k
         real(dp), pointer :: vals(:,:)
         real(dp), parameter :: weight = 20
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         names(1) = 'kap_function'
         nz = s% nz
         vals(1:nz,1:1) => vals1(1:nz)
         do k=1,nz
            vals(k,1) = weight*log10(s% opacity(k))
         end do
      end subroutine other_mesh_fcn_data


      end module run_star_extras
      
