! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      
      implicit none
      
      integer :: time0, time1, clock_rate
      double precision, parameter :: expected_runtime = 4 ! minutes


      real(dp) :: target_log_luminosity, target_log_radius, target_surface_Z_div_X, &
         target_Rcz, target_surf_He, target_csound_rms
      real(dp) :: sigma_log_luminosity, sigma_log_radius, sigma_surface_Z_div_X, &
         sigma_Rcz, sigma_surf_He, sigma_csound_rms


      ! solar sound speed data
      logical :: have_sound_speed_data = .false.
      integer, parameter :: npts = 79
      real(dp), dimension(npts) :: data_r, data_csound, data_width

      
      ! these routines are called by the standard run_star check_model
      contains





      subroutine setup_solar_data_for_calc_rms(ierr)
         use const_def, only: mesa_data_dir
         use utils_lib
         integer, intent(out) :: ierr
         
         integer, parameter :: lines_to_skip = 11
         integer :: iounit, i, k
         real(dp) :: jnk
         
         character (len=256) :: fname
         
         have_sound_speed_data = .true.
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) then
            return
         end if
         fname = trim(mesa_data_dir) // '/star_data/solar_csound.data'
         open(iounit, file=trim(fname), action='read', status='old', iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to open ' // trim(fname)
            call free_iounit(iounit)
            return
         end if                  
         
         do i=1,lines_to_skip
            read(iounit,fmt=*,iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed when skipping line', i
               close(iounit)
               call free_iounit(iounit)
               return
            end if
         end do
         
         do i=1,npts
            read(iounit,fmt=*,iostat=ierr) &
               data_r(i), jnk, data_csound(i), jnk, jnk, jnk, data_width(i)
            if (ierr /= 0) then
               write(*,*) 'failed when reading data point', i
               close(iounit)
               call free_iounit(iounit)
               return
            end if
         end do

         close(iounit)
         call free_iounit(iounit)
      
      
      end subroutine setup_solar_data_for_calc_rms


      real(dp) function calc_current_rms(s, nz) ! dR weighted
         use interp_1d_lib
         use interp_1d_def
         type (star_info), pointer :: s
         integer, intent(in) :: nz
         
         logical, parameter :: dbg = .false.
         real(dp), target :: calc_rms_f1_ary(4*nz)
         real(dp), pointer :: calc_rms_f1(:), calc_rms_f(:,:)
         real(dp) :: sumy2, sumdr, dr, y2, cs
         real(dp), parameter :: min_R = 0.094, max_R = 0.94
         real(dp), target :: pm_work_ary(nz*pm_work_size)
         real(dp), pointer :: pm_work(:)
         integer :: k, i, ierr
         
         include 'formats'
         
         calc_current_rms = -1  
         pm_work => pm_work_ary
         calc_rms_f1 => calc_rms_f1_ary
         calc_rms_f(1:4,1:nz) => calc_rms_f1(1:4*nz)

         ierr = 0
         if (.not. have_sound_speed_data) then
            call setup_solar_data_for_calc_rms(ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in setup_solar_data_for_calc_rms'
               return
            end if
         end if
         
         do k=1,nz
            if (k == 1) then
               calc_rms_f(1,k) = s% csound(k)
            else
               calc_rms_f(1,k) = &
                  (s% csound(k)*s% dq(k-1) + s% csound(k-1)*s% dq(k))/(s% dq(k-1) + s% dq(k))
            end if
         end do

         call interp_pm(s% r, nz, calc_rms_f1, pm_work_size, pm_work, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in interp_pm'
            return
         end if

         sumy2 = 0
         sumdr = 0
         do i=1,npts
            if (data_r(i) < min_R .or. data_r(i) > max_R) cycle
            call interp_value(s% r, nz, calc_rms_f1, Rsun*data_r(i), cs, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in interp_value', i
               return
            end if
            if (i == 1) then
               dr = data_r(2) - data_r(1)
            else
               dr = data_r(i) - data_r(i-1)
            end if
            if (dr < 0) dr = -dr
            ! change to weigh by point rather than by dr
            dr = 1
            
            sumdr = sumdr + dr
            y2 = dr*((cs - data_csound(i))/data_csound(i))**2
            sumy2 = sumy2 + y2
            if (dbg) write(*,2) 'rms cs, data_cs, reldiff, y2, dr', i, cs, data_csound(i), &
               (cs - data_csound(i))/data_csound(i), y2, dr
         end do
         
         calc_current_rms = sqrt(sumy2/sumdr)
         if (.true. .or. dbg) write(*,1) 'calc_current_rms', calc_current_rms

      end function calc_current_rms

      
      subroutine extras_controls(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         ierr = 0

         target_log_luminosity = 0
         sigma_log_luminosity = 2d-3

         target_log_radius = 0
         sigma_log_radius = 1d-3

         target_surface_Z_div_X = 2.292d-2 ! GS98 value     
         !target_surface_Z_div_X = 1.81d-2 ! Asplund 09 value
         sigma_surface_Z_div_X = 1d-3   
   
         target_Rcz = 0.713d0 ! Bahcall, Serenelli, Basu, 2005
         sigma_Rcz = 0.003d0
   
         target_surf_He = 0.2485d0 ! Bahcall, Serenelli, Basu, 2005
         sigma_surf_He = 0.0034
   
         target_csound_rms = 0
         sigma_csound_rms = 1d-3 ! 7d-4

      end subroutine extras_controls
      
      
      integer function extras_startup(s, id, restart, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         ierr = 0
         extras_startup = 0
         call system_clock(time0,clock_rate)
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup
      
      
      subroutine extras_after_evolve(s, id, id_extra, ierr)
         use run_star_support, only: initial_h1, initial_he3, initial_he4
         use const_def, only: Rsun
         use mlt_def, only: no_mixing, convective_mixing, overshoot_mixing
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         
         real(dp) :: surface_He, Rcz, surface_Z_div_X, csound_rms, dt
         integer :: i, nz
         logical :: okay
         
         include 'formats'

         ierr = 0         
         nz = s% nz

         surface_He = s% surface_he3 + s% surface_he4
         surface_Z_div_X = &
            (1d0 - (s% surface_h1 + surface_He))/s% surface_h1
         csound_rms = calc_current_rms(s, nz)

         do i = 1, nz-1 ! locate bottom of solar convective zone
            if (s% mixing_type(i+1) /= convective_mixing &
                  .and. s% mixing_type(i) == convective_mixing) then
               if (s% r(i+1) > 0.25*Rsun .and. s% r(i) < 0.9*Rsun) then
                  Rcz = s% r(i)/Rsun
                  exit
               end if
            end if
         end do
         
         okay = .true.
         write(*,*)
         write(*,'(30x,99a14)') 'val', 'low', 'hi', 'target', 'sigma'
         call check('log_surface_luminosity', &
            s% log_surface_luminosity, target_log_luminosity, sigma_log_luminosity, 3d0)
         call check('log_surface_radius', &
            s% log_surface_radius, target_log_radius, sigma_log_radius, 3d0)
         call check('surface_Z_div_X', &
            surface_Z_div_X, target_surface_Z_div_X, sigma_surface_Z_div_X, 3d0)
         call check('Rcz', Rcz, target_Rcz, sigma_Rcz, 3d0)
         call check('surface_He', surface_He, target_surf_He, sigma_surf_He, 3d0)
         call check('csound_rms', csound_rms, target_csound_rms, sigma_csound_rms, 5d0)
         write(*,*)
         if (okay) write(*,'(a)') 'all values are within tolerances'
         write(*,*)


         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         if (dt > 10*expected_runtime) then
            write(*,'(/,a30,2f18.6,a,/)') '>>>>>>> EXCESSIVE runtime', &
               dt, expected_runtime, '   <<<<<<<<<  ERROR'
         else
            write(*,'(/,a50,2f18.6,99i10/)') 'runtime, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         end if
         
         
         contains
         
         subroutine check(str, val, target_val, sigma, nsig)
            double precision, intent(in) :: val, target_val, sigma, nsig
            character (len=*) :: str
            real(dp) :: low, hi
            include 'formats'
            low = target_val - nsig*sigma
            hi = target_val + nsig*sigma
            if (low <= val .and. val <= hi) then
               write(*,'(a30,99(1pe14.6))') trim(str), val, low, hi, target_val, sigma
            else
               write(*,'(a30,99(1pe14.6))') '*** BAD *** ' // trim(str), &
                  val, low, hi, target_val, sigma
               okay = .false.
            end if
         end subroutine check
         
         
      end subroutine extras_after_evolve
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         extras_check_model = keep_going         
      end function extras_check_model


      integer function how_many_extra_history_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_history_columns = 0
      end function how_many_extra_history_columns
      
      
      subroutine data_for_extra_history_columns(s, id, id_extra, n, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine data_for_extra_history_columns

      
      integer function how_many_extra_profile_columns(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(s, id, id_extra, n, nz, names, vals, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         double precision :: vals(nz,n)
         integer, intent(out) :: ierr
         integer :: k
         ierr = 0
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going or terminate.
      integer function extras_finish_step(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         integer :: ierr
         extras_finish_step = keep_going
         call store_extra_info(s)
      end function extras_finish_step
      
      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            double precision :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info

      end module run_star_extras
      
