! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      use run_star_support
      
      implicit none
      
      integer, parameter :: max_num_files = 100
      integer :: num_files
      character (len=256) :: model_directory
      character (len=256) :: filenames(max_num_files)
      
      namelist /job/ &
         model_directory, num_files, filenames
      
      integer :: which_file
      logical :: finished_okay

      integer :: time0, time1, clock_rate
      double precision, parameter :: expected_runtime = 1 ! minutes


      contains
      

      subroutine do_run
         use run_star_support
         
         type (star_info), pointer :: s
         integer :: id, ierr, i, j, k, result
         logical :: first_try, restart
         double precision :: dt

         logical, parameter :: dbg = .true., do_init_and_alloc = .false., &
            do_free_star = .false., okay_to_restart = .true.
         
         include 'formats'
         
         ierr = 0
         call system_clock(time0,clock_rate)
         
         call read_controls('inlist_files',ierr)
         if (ierr /= 0) return

         call init_and_alloc(id, s, ierr)
         if (failed('init_and_alloc')) return
         
         which_file = 1
         
         do k = 1, 1 !50
            do i=1, min(num_files, max_num_files)         
               if (len_trim(filenames(i)) == 0) cycle            
               which_file = i
               finished_okay = .false.            
               call run1_star( &
                  do_init_and_alloc, do_free_star, okay_to_restart, &
                  id, restart, &
                  check_models_controls, &
                  extras_startup, &
                  extras_check_model, &
                  how_many_extra_history_columns, &
                  data_for_extra_history_columns, &
                  how_many_extra_profile_columns, &
                  data_for_extra_profile_columns, &
                  finish_step, extras_after_evolve, &
                  ierr)            
               if (failed('run1_star')) return
               if (.not. finished_okay) exit            
               do j = 1, 10
                  write(*,*)
               end do                        
            end do
         end do
         
         write(*,*)

         write (*, *)
         write (*, *)
         write(*, *)
         if (finished_okay) then
            write(*, '(a)') 'finished all vlm checks okay'
         else
            write(*, '(a)') 'failed check ' // trim(filenames(which_file))
         end if
         write(*, *)
         

         call star_shutdown
                  
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         if (dt > 10*expected_runtime) then
            write(*,'(/,a30,2f18.6,a,/)') '>>>>>>> EXCESSIVE runtime', &
               dt, expected_runtime, '   <<<<<<<<<  ERROR'
         else
            write(*,'(/,a30,99f18.6,/)') 'runtime', dt, expected_runtime
         end if


         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
               stop 1
            end if
         end function failed

      end subroutine do_run
      
      
      ! returns either keep_going or terminate.
      integer function finish_step(s, id, id_extra)
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         if (s% model_number == 10) then
            finish_step = terminate
            finished_okay = .true.
            return
         end if
         finish_step = keep_going
         call store_extra_info(s)
      end function finish_step
      
      
      subroutine check_models_controls(s, ierr)
         use star_def, only: star_info
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         ierr = 0
         load_saved_model = .true.
         saved_model_name = trim(model_directory) // '/' // trim(filenames(which_file))
         write(*,'(a)') 'load ' // trim(saved_model_name)
      end subroutine check_models_controls      

      
      subroutine read_controls(filename,ierr)
         use utils_lib
         character (len=*) :: filename
         integer, intent(out) :: ierr

         
         character (len=256) :: message
         integer :: unit
         
         write(*,*) 'read_controls ' // trim(filename)
         
         ! set defaults
         model_directory = '.'
         num_files = 0
         filenames(:) = ''
         
         unit=alloc_iounit(ierr)
         if (ierr /= 0) return

         open(unit=unit, file=trim(filename), action='read', delim='quote', iostat=ierr)
         if (ierr /= 0) then
            write(*, *) 'Failed to open control namelist file ', trim(filename)
         else
            read(unit, nml=job, iostat=ierr)  
            close(unit)
            if (ierr /= 0) then
               write(message, *) 'Failed while trying to read control namelist file ', trim(filename)
               write(*, '(a)') trim(message)
               write(*, '(a)') &
                  'The following runtime error message might help you find the problem'
               write(*, *) 
               open(unit=unit, file=trim(filename), action='read', delim='quote', status='old', iostat=ierr)
               read(unit, nml=job)
               close(unit)
            end if  
         end if
         call free_iounit(unit)

      end subroutine read_controls
      
      
      include 'standard_run_star_extras.inc'
      

      end module run_star_extras
      
