# profiles.rb

$MESA_DIR = '../../..' if $MESA_DIR == nil

class ProfilePlots

    include Math
    include Tioga
    include FigureConstants
    
    def t
        @figure_maker
    end

    def d
        @profile_data[@profile_index]
    end

    
    def initialize(dict)
        
        which = dict['which']
        @filenames = dict['filenames']
        if which == nil
            if @filenames == nil
                which = 'all'
            else
                which = 'none'
            end
        end
        @profile_index = dict['profile_index']
        @profile_index = 0 if @profile_index == nil
        
        @xaxis_by = dict['xaxis_by']
        @reverse_xaxis = dict['reverse_xaxis']
        
        @model = dict['model']
        @model = -1 if @model == nil
        
        @log_dir = dict['log_dir']
        @log_dir = '../LOGS' if @log_dir == nil
        
        @logfile = dict['logfile']
        @logfile = GetModelName.getlogname(@model,@log_dir) if @logfile == nil
        
        @grid_min = dict['grid_min']
        @grid_min = 0 if @grid_min == nil
        @grid_max = dict['grid_max']
        @grid_max = -1 if @grid_max == nil
        
        @show_points = dict['show_points']
        @show_points = false if @show_points == nil
        
        @use_mmid = dict['use_mmid']
        @use_mmid = true if @use_mmid == nil
        
        @num_abundance_line_labels = dict['num_abundance_line_labels']
        @num_abundance_line_labels = 7 if @num_abundance_line_labels == nil

        @abundances_to_skip = dict['abundances_to_skip']
        @abundances_to_skip = [] if @abundances_to_skip == nil

        @no_frills = dict['no_frills']
        @no_frills = false if @no_frills == nil

        @no_legend = dict['no_legend']
        @no_legend = false if @no_legend == nil
        
        @log_mass_frac_ymax = dict['log_mass_frac_ymax']
        @log_mass_frac_ymin = dict['log_mass_frac_ymin']
        
        @xaxis_log_values = dict['xaxis_log_values']
        @xaxis_log_values = false if @xaxis_log_values == nil
        
        puts "log_mass_frac_ymin #{@log_mass_frac_ymin}"
        
        @xmin = dict['xmin']
        @xmax = dict['xmax']    
        
        @force_ymin = dict['force_ymin']
        @force_ymax = dict['force_ymax']    
        
        @logxq_cutoff = dict['logxq_cutoff']
        @logxq_cutoff = -20 if @logxq_cutoff == nil

        @trho_left = dict['trho_left']
        @trho_left = -10.5 if @trho_left == nil

        @trho_right = dict['trho_right']
        @trho_right = 11 if @trho_right == nil

        @trho_top = dict['trho_top']
        @trho_top = 10.4 if @trho_top == nil

        @trho_bottom = dict['trho_bottom']
        @trho_bottom = 1.8 if @trho_bottom == nil

        @mode_prop_ymax = dict['mode_prop_ymax']

        @show_mesh_flag = dict['show_mesh_flag']

        @legend_abundance_lines = dict['legend_abundance_lines']

        fyi = $MESA_DIR + '/data/star_data/plot_info/'
        @opal_clip = Clipdata.new(fyi +'opal_clip.data')
        @scvh_clip = Clipdata.new(fyi +'scvh_clip.data')
        Dvector.read(fyi +'elect.data', @elect_data = [Dvector.new, Dvector.new])
        Dvector.read(fyi +'gamma_4_thirds.data', @gamma_4_thirds = [Dvector.new, Dvector.new])
        Dvector.read(fyi +'kap_rad_cond_eq.data', @kap_rad_cond_eq = [Dvector.new, Dvector.new])
        Dvector.read(fyi + '/psi4.data', @psi4 = [Dvector.new, Dvector.new], 2)        
        @hydrogen_burn = Ignition_data.new(fyi +'/hydrogen_burn.data')
        @helium_burn = Ignition_data.new(fyi +'/helium_burn.data')
        @carbon_burn = Ignition_data.new(fyi +'/carbon_burn.data')
        @oxygen_burn = Ignition_data.new(fyi +'/oxygen_burn.data')
        
        savedir = dict['savedir']
        savedir = 'profiles_out' if savedir == nil
        slave_mode = dict['slave_mode']
        slave_mode = false if slave_mode == nil

        @figure_maker = FigureMaker.default
        t.save_dir = savedir unless savedir == nil
        t.def_eval_function { |str| eval(str) }
        #t.tex_preview_preamble = t.tex_preview_preamble + "\n\t\\include{color_names}\n"
        
        @log_dir = $star_data_path
        @log_dir = "../LOGS" if @log_dir == nil
            
        @have_hr_data = false
        @hr_name = @log_dir + "/star.log"
        @hr_trho_data = HR_History_Data.new

        @mass_50_color = LightSteelBlue
        @mass_95_color = MediumSlateBlue
        @mass_999_color = MediumBlue

        @eta_color = nil # Black   -- if nil, don't show
        
        @show_tau10 = false
        @show_tau100 = false

        @skip_model_number = false

        @mass_xlabel = 'm ($\mathrm{M_\odot}$)'
        @logxm_xlabel = 'log exterior mass ($\mathrm{M_\odot}$)'
        @logxq_xlabel = 'log(1-q)'
        @logP_xlabel = 'log P'
        @log_col_depth_xlabel = 'log column depth (g cm^{-2})'
        @q_xlabel = 'q'
        @tau_xlabel = 'tau'
        @logT_xlabel = 'log T'
        @logR_xlabel = 'log ($\mathrm{R_\odot}$)'
        @radius_xlabel = 'r ($\mathrm{R_\odot}$)'
        @r_div_R_xlabel = 'r/R'
        @acoustic_r_div_R_phot_xlabel = 'acoustic t / T'
        @grid_xlabel = 'grid point'
        
        @subplot_scale = 0.8
        @subplot_margin = 0.01
        @half_plot = 0.38  
        @legend_scale = 0.95
        @legend_on_right = true
        @plot_right = 0.15
        @legend_left = 0.7
        @legend_top = 0.2
        @trio_margin = 0.02
        @side_by_side_margin = 0.51
        @plot_with_legend_right_margin = 0.12
        @legend_left_margin = 0.92
        @legend_scale = 0.9
        
        @secyer = 3.1558149984e7
        @msun = 1.9892e33
        @rsun = 6.9598e10
        @lsun = 3.8418e33
    
        @plot_with_legend_dict = {
            'plot_right' => @plot_right, 
            'legend_left' => @legend_left, 
            'legend_top' => @legend_top }

        @temp_vec1 = Dvector.new
        @temp_vec2 = Dvector.new
        
        @multi_plot = false


        @atol = 1e-30
        @rtol = 1e-8
        
        read_profiles
        
        if (which == 'all') && (not slave_mode)
            
            t.def_figure("Abundances") { show_by('abundances') }
            t.def_figure("Power") { show_by('power') }
            
            t.def_figure("plot_T_RHO") { plot_T_RHO }
            t.def_figure("plot_T_RHO_eos") { plot_T_RHO_eos }
            t.def_figure("plot_T_RHO_kap") { plot_T_RHO_kap }

            t.def_figure("plot_T8_col_depth") { plot_T8_col_depth }

            if @xaxis_by == 'r_div_R'
                t.def_figure("brunt_N_comparison") { show_brunt_N_comparison } unless d.brunt_N_dimensionless == nil
                t.def_figure("brunt_frequency_comparison") { show_brunt_frequency_comparison } unless d.brunt_frequency == nil
                t.def_figure("brunt_duo_for_paper") { show_brunt_duo_for_paper } unless 
                  d.brunt_N_dimensionless == nil || d.brunt_frequency == nil
            end

            if @xaxis_by == 'q'
                t.def_figure("variables") { show_variables }
            end

            if @xaxis_by == 'logxm'
                t.def_figure("log_brunt_N2_comparison") { show_log_brunt_N2_comparison } unless d.log_brunt_N2 == nil
            end
            
            t.def_figure("mode_propagation_nu") { show_by('mode_propagation_nu') } unless d.log_brunt_nu == nil
            t.def_figure("mode_propagation_lognu") { show_by('mode_propagation_lognu') } unless d.log_brunt_nu == nil
            t.def_figure("duo_abundance_propagation_nu") { duo_abundance_propagation_nu } unless d.log_brunt_nu == nil
            t.def_figure("brunt_dlnR_integral") { show_by('brunt_dlnR_integral') } unless d.log_brunt_nu == nil
            t.def_figure("astero_stuff") { show_by('astero_stuff') } unless d.log_brunt_nu == nil
            t.def_figure("duo_astero_stuff_prop_nu") { duo_astero_stuff_prop_nu } unless d.log_brunt_nu == nil
            t.def_figure("duo_astero_stuff_prop_lognu") { duo_astero_stuff_prop_lognu } unless d.log_brunt_nu == nil

            if @xaxis_by == 'radius'
                t.def_figure("bp98_solar_sound_speed_comparison") {
                    show_by('bp98_solar_sound_speed_comparison') }
                t.def_figure("solar_sound_speed_comparison") {
                    show_by('solar_sound_speed_comparison') }
                t.def_figure("solar_sound_speed2_comparison") {
                    show_by('solar_sound_speed2_comparison') }
                t.def_figure("sound_speed_data") { show_by('sound_speed_data') }
                t.def_figure("solar_comparison") { show_solar_comparison }
                t.def_figure("solar_new_comparison") { show_solar_new_comparison }
                t.def_figure("edvs") { show_by('edvs') }
                t.def_figure("edvs_fe") { show_by('edvs_fe') }
            end

            t.def_figure("C_O_Ne_Mg") { show_by('abundance_C_O_Ne_Mg') }
            t.def_figure("Fe_core") { show_by('abundance_Fe_core') }
            t.def_figure("duo_T_Rho") { duo_T_Rho }
            t.def_figure("duo_entropy_Ye") { duo_entropy_Ye }
            t.def_figure("duo_logRho_Ye") { duo_logRho_Ye }
            t.def_figure("duo_velocity_v_div_r") { duo_velocity_v_div_r }
            t.def_figure("duo_velocity_gamma1") { duo_velocity_gamma1 }
            
            t.def_figure("duo_abundance_logdq") { duo_abundance_logdq }
            
            t.def_figure("three_by_two") { three_by_two }
            
            
            t.def_figure("duo_Power_Entropy") { duo_Power_Entropy }
            t.def_figure("duo_energy_mixing") { duo_energy_mixing }
            t.def_figure("trio_T_Rho_entropy") { trio_T_Rho_entropy }
            t.def_figure("trio_T_Rho_logtau") { trio_T_Rho_logtau }
            t.def_figure("trio_T_Rho_velocity") { trio_T_Rho_velocity }
            t.def_figure("Convection") { show_by('convection') }

            
            t.def_figure("Power_plus") { show_by('power_plus') }
            t.def_figure("Neutrinos") { show_by('neutrinos') }
            t.def_figure("heat_capacity") { show_by('heat_capacity') }
            
            unless d.env_eps_grav == nil
               t.def_figure("eps_grav_comparison") { show_by('eps_grav_comparison') }
               t.def_figure("eps_grav_comparison_relative") { show_by('eps_grav_comparison_relative') }
            end 

            t.def_figure("mixing_Ds") { show_by('mixing_Ds') } unless d.log_D_conv == nil
            t.def_figure("dynamo_info") { dynamo_info }

            
        end
        
        if which == 'all'
            @num_columns.times do |i|
                eval(sprintf("t.def_figure(\"%s\") { plot_column %i }", @plot_info[i][0], i))
            end
        end
        
        write_mini if dict['write_mini'] == true

        t.def_enter_page_function { enter_page }
        
        t.tex_preamble = t.tex_preamble + "\n\t\\usepackage{txfonts}\n"
                    
    end
    
    
    def write_mini
      f = File.open('mini_profile.data',"w")
      n = d.logRho.length
      #puts "n #{n}"
      n.times { |i|
          logRho = d.logRho[i]
          logT = d.logT[i]
          #puts "#{i} #{logRho} #{logT}"
          f.printf("%25.15e", d.radius[i])
          f.printf("%25.15e", d.mass[i])
          f.printf("%25.15e", 10**logRho)
          f.printf("%25.15e", 10**logT)
          f.puts
      }
      f.close
    end  
    
    
    def enter_page
        t.page_setup(11*72/2,8.5*72/2)
        t.set_frame_sides(0.15,0.85,0.85,0.15) # left, right, top, bottom in page coords        
    end

    
    def read_profiles
        if @filenames == nil
            @profile_data = [ProfileData.new]
            d.read(@logfile, @logxq_cutoff)
        else
            @profile_data = []
            @filenames.each do |logfile|
                p = ProfileData.new
                @profile_data << p
                p.read(logfile, @logxq_cutoff)
            end
        end
        t.need_to_reload_data = false
        t.model_number = d.model_number
        set_grid_min_max
        @num_columns = d.columns.length
        @plot_info = Array.new(@num_columns)
        @column_dict = Hash.new 
        @num_columns.times {|i|
            @column_dict[d.column_names[i]] = d.columns[i]
            get_min_max_for_plot(i)
            @plot_info[i] = [d.column_names[i], @plot_ymin, @plot_ymax]
        }        
    end
    
    
    def load(model,log = nil)
      puts "load model #{model}"
      log = @log_dir if log == nil
      @logfile = GetModelName.getlogname(model,log)
      read_profiles
      t.need_to_reload_data = true
      return "####04OK " # request refresh
    end
    
    
    def get_min_max_for_plot(n)
        ys = get_ys_for_plot(n)
        if ys == nil
           puts "failed to find ys for column #{n}  #{d.column_names[n]}"
           return
        end
        if ys.length == 0
           puts "bad specification for bounds of plot -- no y values for " +
            "column #{n}  #{d.column_names[n]}"
           return
        end
        ymin = ys.min
        ymin = ys.min_gt(-99) if ymin == -99
        ymin = -99 if ymin == nil
        ymax = ys.max
        dy = ymax - ymin
        ytol = [ymin.abs, ymax.abs].max
        dy = ytol * @rtol + @atol if dy < ytol * @rtol + @atol
        ymargin = 0.02
        @plot_ymax = ymax + ymargin * dy
        @plot_ymin = ymin - ymargin * dy
        @plot_ymax = @force_ymax unless @force_ymax == nil
        @plot_ymin = @force_ymin unless @force_ymin == nil
    end
    
    
    def get_by_name_for_plot(name)
      n = d.column_names.index(name)
      return nil if n == nil
      return d.columns[n][@grid_min..@grid_max]
    end
    
    
    def get_by_name(name)
      n = d.column_names.index(name)
      return d.columns[n]
    end
    
    
    def get_ys_for_plot(col_num)
      #puts "get ys col_num #{col_num}"
      #puts "get ys col_name #{d.column_names[col_num]}"
      col = d.columns[col_num]
      return col[@grid_min..@grid_max]
    end


    def plot_column(n,ylabel=nil)
       ys = get_ys_for_plot(n)
       xs = xs_for_plot()
       xreversed = xaxis_reversed?
       puts "plot_column xreversed #{xreversed}"
       if d.column_names[n] == 'mixing_type'
          if ylabel == nil
              if @multi_plot
                ylabel = '\small none \ conv \ over \ semi \ salt \ rot'
              else
                ylabel = 
            '\small none \quad\quad\  conv \quad\quad\ ' + 
            ' over \quad\quad\  semi \quad\quad\  salt \quad\quad\  rot'
              end
          end
          plot_results(n, xs, ys, ylabel, xreversed, 0, 5.1)
       else
          ylabel = d.column_names[n].tr("_", " ") if ylabel == nil
          plot_results(n, xs, ys, ylabel, xreversed)
       end
       
    end
    
    
    def get_plot_xleft(xs)
        if (xs[0] > xs[-1])
            xleft = @xmin
        else
            xleft = @xmax
        end
        xleft = xs[@grid_max] if xleft == nil
        return xleft
    end
    
    
    def get_plot_xright(xs)
        if (xs[0] > xs[-1])
            xright = @xmax
        else
            xright = @xmin
        end
        xright = xs[@grid_min] if xright == nil
        return xright
    end
    
    
    def plot_results(n, xs, ys, ylabel, xreversed, ymin=nil, ymax=nil)
        plt = @plot_info[n]
        title = plt[0].tr("_", " ")
        ymin = plt[1] if ymin==nil
        ymax  = plt[2] if ymax==nil
        xmargin = 0.07
        ymargin = 0.1
        ythresh = @atol
        height = (ymax < ymin+ythresh)? ythresh : ymax - ymin
        top_boundary = ymax + ymargin * height
        bottom_boundary = ymin - ymargin * height
        
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        
        xs = xs[@grid_min..@grid_max]
        boundaries = [ xleft, xright, top_boundary, bottom_boundary ]
        
        #puts "xs[0] #{xs[0]}"
        #puts "xs[-1] #{xs[-1]}"

          show_model_number
          t.top_edge_type = AXIS_LINE_ONLY
          t.rescale(@legend_scale)
          t.legend_text_dy = 1.1
          t.subplot('right_margin' => @plot_with_legend_right_margin) do
                background
                t.show_title(title)
                xlabel = xlabel_for_plot()
                t.show_xlabel(xlabel)
                t.show_ylabel(ylabel)
                if false
                    puts "xlabel #{xlabel}"
                    puts "xs length #{xs.length}"
                    puts "ys length #{ys.length}"
                    puts "boundaries length #{boundaries.length}"
                    puts "xright #{xright}"
                    puts "top_boundary #{top_boundary}"
                    puts "bottom_boundary #{bottom_boundary}"
                end
                t.show_plot(boundaries) {
                    t.show_polyline(xs,ys,Black)
                    t.show_marker('xs' => xs, 'ys' => ys, 'marker' => Bullet,
                        'color' => Black, 'scale' => 0.4) if @show_points
                }        
          end
    end
    
    def trio_T_Rho_P
        t.set_aspect_ratio(0.5)
        num_plots = 3
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logT')) 
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logRho'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('logP'))
        end
    end
    
        
    def duo_Power_Entropy
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_by('power')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('entropy'))
        end
    end
    
    
    def compare_abundances
        #t.set_aspect_ratio(2)
        #t.rescale(0.8)
        @no_frills = true
        num_plots = 2
        row_margin = 0.01
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.title_visible = false
            show_by('abundances')
        end
        load(1)
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do 
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_by('abundances')
        end
    end
    
    def show_brunt_duo_for_paper
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_brunt_frequency_comparison # 9.8M case
        end
        load(1)
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_brunt_N_comparison # 4M case
        end
    end
    
    
    def duo_T_Rho
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logT')) 
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('logRho'))
        end
    end
        
        
    def duo_abundance_propagation_nu
        #t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.legend_scale = 0.8
        t.legend_text_ystart = 0.8
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_by('abundances')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_by('mode_propagation_nu')
        end
    end
        
        
    def duo_astero_stuff_prop_nu
        t.set_aspect_ratio(0.9)
        t.rescale(1.1)
        num_plots = 2
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.legend_scale = 0.8
        t.legend_text_ystart = 0.8
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_by('astero_stuff')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_by('mode_propagation_nu')
        end
    end
        
        
    def duo_astero_stuff_prop_lognu
        t.set_aspect_ratio(0.9)
        t.rescale(1.1)
        num_plots = 2
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.legend_scale = 0.8
        t.legend_text_ystart = 0.8
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_by('astero_stuff')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_by('mode_propagation_lognu')
        end
    end
    
    
    def duo_velocity_gamma1
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_by('velocity_km_per_sec')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('gamma1'), '$\Gamma_1$')
        end
    end
    
    def duo_velocity_v_div_r
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_by('velocity_km_per_sec')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('v_div_r'))
        end
    end
    
    
    def trio_T_Rho_entropy
        t.set_aspect_ratio(0.5)
        num_plots = 3
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logT')) 
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logRho')) 
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('entropy'))
        end
    end
    
    
    def trio_T_Rho_logtau
        t.set_aspect_ratio(0.5)
        num_plots = 3
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logT'), 'log T (K)') 
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logRho'), 'log $\rho$ (gm cm^-3)') 
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('logtau'), 'log tau')
        end
    end
    
    
    def trio_T_Rho_velocity
        t.set_aspect_ratio(0.5)
        num_plots = 3
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logT')) 
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logRho')) 
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_by('velocity_km_per_sec')
            #plot_column(names.index('velocity'))
        end
    end
    
    
    
    
    
    
    def duo_logRho_Ye
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logRho'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('ye'))
        end
    end
    
    
    def duo_entropy_Ye
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('entropy'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('ye'))
        end
    end
    
    
    
    
    def duo_abundance_logdq
        t.set_aspect_ratio(1)
        num_plots = 4
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins(
                'num_rows' => num_plots, 
                'first_row' => 1, 'last_row' => 3, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_by('abundances')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' =>4, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('logdq'), 'log dm/m_{star}') 
        end
    end
    
    
    
    def duo_Ye_velocity
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('ye'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('velocity'))
        end
    end
    
    
    def duo_energy_mixing
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        @multi_plot = true
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('net_nuclear_energy'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('mixing_type'))
        end
        @multi_plot = false
    end
    
    
    def write_mini_profile(fname)
      puts "write " + fname
      f = File.new(fname, "w")
      out = [
         "zone", d.zone, "",
         "radius", d.radius, "R/Rsun",
         "csound", d.csound.mul(1e-5), "sound speed in km/sec"
         ]
      num = out.length     
      n_per_item = 3
      items = num/n_per_item
      #items.times { |i| puts sprintf("%20s %s", out[i*n_per_item], out[i*n_per_item+2]) }
      items.times { |i| f.printf("%2i  %20s      %s\n", i+1, out[i*n_per_item], out[i*n_per_item+2]) }
      f.printf("\n")
      items.times { |i| f.printf("%20s", out[i*n_per_item]) }
      f.printf("\n")
      d.zone.length.times { |k|
         items.times { |i| 
               if out[i*n_per_item] == "zone"
                  f.printf("%20i", out[i*n_per_item+1][k])
               else
                  f.printf("%20.10e", out[i*n_per_item+1][k])
               end
               }
         f.printf("\n")
      }
      f.close
    end
    
    
    def set_grid_min_max
      if @xaxis_by == 'grid'
          @grid_min = 0 if @grid_min == nil || @grid_min > d.mass.length - 10
          @grid_max = -1 if @grid_max == nil || @grid_max <= @grid_min || @grid_max > d.mass.length
          return
      elsif @xaxis_by == 'mass'
        if @use_mmid && d.mmid != nil
            xs = d.mmid
        else
            xs = d.mass
        end
      elsif @xaxis_by == 'q'
        xs = d.q
      elsif @xaxis_by == 'tau'
        xs = d.tau
      elsif @xaxis_by == 'radius'
        xs = d.radius
      elsif @xaxis_by == 'r_div_R'
        xs = d.r_div_R
      elsif @xaxis_by == 'acoustic_r_div_R_phot'
        xs = d.acoustic_r_div_R_phot
      elsif @xaxis_by == 'logR'
        xs = d.logR
      elsif @xaxis_by == 'logP'
        xs = d.logP
      elsif @xaxis_by == 'logT'
        xs = d.logT
      elsif @xaxis_by == 'log_column_depth'
        xs = d.log_column_depth
      elsif @xaxis_by == 'logxm'
        xs = d.logxm
        @xmin = -9 if @xmin == nil
        @xmax = d.star_mass.log10 if @xmax == nil
      elsif @xaxis_by == 'logxq'
        xs = d.logxq
        @xmin = -9 if @xmin == nil
        @xmax = 0 if @xmax == nil
      elsif
        puts "problem in set_grid_min_max @xaxis_by = #{@xaxis_by}"
        return
      end
      return if xs == nil
      if (xs[0] < xs[-1])
          @grid_min = xs.where_last_le(@xmin) unless @xmin == nil
          @grid_max = xs.where_first_ge(@xmax) unless @xmax == nil
      else
          @grid_max = xs.where_last_ge(@xmin) unless @xmin == nil
          @grid_min = xs.where_first_le(@xmax) unless @xmax == nil
      end
      @grid_min = 0 if @grid_min == nil
      @grid_max = xs.length-1 if @grid_max == nil
      return
      puts "@xmin #{@xmin}"
      puts "@xmax #{@xmax}"
      puts "xs.length #{xs.length}"
      puts "xs[0] #{xs[0]}"
      puts "xs[-1] #{xs[-1]}"
      puts "@grid_min #{@grid_min}"
      puts "@grid_max #{@grid_max}"
      puts "xs[@grid_min] #{xs[@grid_min]}"
      puts "xs[@grid_max] #{xs[@grid_max]}"
    end
    
    
    def two_by_two
        t.rescale(0.7)
        t.line_width = 0.5
        t.title_shift = 0.5
        small = 0.01; large = 0.51
        left1 = small; right1 = large
        left2 = large; right2 = small
        small = 0.06; large = 0.54
        top1 = small; bottom1 = large
        top2 = large; bottom2 = small
        t.yaxis_loc = LEFT
        t.ylabel_side = LEFT
        t.subplot(
            'left_margin' => left1, 'right_margin' => right1, 
            'top_margin' => top1, 'bottom_margin' => bottom1) { plot_H_R }
        t.subplot(
            'left_margin' => left1, 'right_margin' => right1, 
            'top_margin' => top2, 'bottom_margin' => bottom2) { plot_T_RHO }
        t.yaxis_loc = RIGHT
        t.ylabel_side = RIGHT
        t.subplot(
            'left_margin' => left2, 'right_margin' => right2, 
            'top_margin' => top1, 'bottom_margin' => bottom1) { plot_luminosity_Tcentral }
        t.subplot(
            'left_margin' => left2, 'right_margin' => right2,
            'top_margin' => top2, 'bottom_margin' => bottom2) { plot_luminosity_age }
    end 
       
       
    def three_by_two
        t.rescale(0.9)
        t.line_width = 0.5
        t.title_visible = false
        @multi_plot = true
        names = d.column_names
        
        small = 0.00; large = 0.50
        left1 = small; right1 = large
        left2 = large; right2 = small
        
        top1 = 0.005; bottom1 = 0.665
        top2 = 0.335; bottom2 = 0.335
        top3 = 0.665; bottom3 = 0.005
        
        t.yaxis_loc = LEFT
        t.ylabel_side = LEFT
        t.subplot(
            'left_margin' => left1, 'right_margin' => right1, 
            'top_margin' => top1, 'bottom_margin' => bottom1) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logT'), 'log T \rm (K)') 
        end
        t.subplot(
            'left_margin' => left1, 'right_margin' => right1, 
            'top_margin' => top2, 'bottom_margin' => bottom2) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('logRho'), 'log $\rho$ (g cm^{-3})') 
        end
        t.subplot(
            'left_margin' => left1, 'right_margin' => right1, 
            'top_margin' => top3, 'bottom_margin' => bottom3) do 
            t.title_visible = false
            #plot_column(names.index('entropy'), 'entropy (k_B/m_P)')
            plot_column(names.index('entropy'), 's $(k_{\rm B} \ m_{\rm P}^{-1})}$')
        end
        t.yaxis_loc = RIGHT
        t.ylabel_side = RIGHT
        t.subplot(
            'left_margin' => left2, 'right_margin' => right2, 
            'top_margin' => top1, 'bottom_margin' => bottom1) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_by('velocity_km_per_sec')
        end
        t.subplot(
            'left_margin' => left2, 'right_margin' => right2, 
            'top_margin' => top2, 'bottom_margin' => bottom2) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_by('r_div_v')
        end
        t.subplot(
            'left_margin' => left2, 'right_margin' => right2, 
            'top_margin' => top3, 'bottom_margin' => bottom3) do 
            t.title_visible = false
            show_by('ye')
        end
    end 
    
    
    def plot_T8_col_depth
        xs = d.log_column_depth
        ys = d.logT
        left = 6.5; right = 9.2
        top = 3.1; bottom = 0.9
        left = 6; right = xs.max
        top = 8.6; bottom = 8
        t.show_plot('left_boundary' => left, 'right_boundary' => right,
               'top_boundary' => top, 'bottom_boundary' => bottom) do
            t.show_xlabel('log Column Depth (g cm^{-2})')
            t.show_ylabel('log T (K)')
            t.show_polyline(xs, ys)
            k = d.h1.where_first_lt(1e-6)
            unless k == nil
                t.show_marker('x' => xs[k], 'y' => ys[k],
                    'marker' => Bullet, 'scale' => 0.8, 'color' => BrightBlue)
            end
            k = d.eps_nuc.where_max
            unless k == nil
                t.show_marker('x' => xs[k], 'y' => ys[k],
                    'marker' => Bullet, 'scale' => 0.8, 'color' => Crimson)
            end
        end
    end
    
    
    def show_variables
        t.fill_color = White
        t.fill_frame
        t.rescale(1.0)
        
        x1 = 0.7; x2 = 0.9; x3 = 0.2
        x1_just = CENTERED
        x2_just = CENTERED
        x3_just = CENTERED
        
        y0 = 0.02; y1 = 0.2; y2 = 0.35; y3 = 0.5; y4 = 0.65; y5 = 0.8; y6 = 0.98
        
        t.line_width = 1.5
        t.stroke_color = Black
        dy = -0.02
        xleft = 0.1
        xright = 1.05
        xmid = (xleft + xright)/2
        t.stroke_line(xleft, y5+dy, xright, y5+dy)
        t.stroke_line(xleft, y3+dy, xright, y3+dy)
        t.stroke_line(xleft, y1+dy, xright, y1+dy)
        
        t.show_text(
            'text' => 'toward surface', 'x' => xmid, 'y' => y6, 'justification' => x1_just)
        t.show_marker('x' => xmid, 'y' => y6+0.05, 'marker' => ArrowOpen, 'angle' => 90,
            'color' => Black, 'scale' => 6, 'stroke_width' => 0.4)
        t.show_text(
            'text' => 'toward center', 'x' => xmid, 'y' => y0, 'justification' => x1_just)
        t.show_marker('x' => xmid, 'y' => y0-0.02, 'marker' => ArrowOpen, 'angle' => -90,
            'color' => Black, 'scale' => 6, 'stroke_width' => 0.4)
        
        
        t.show_text(
            'text' => '$m_{k-1},\ r_{k-1},\ L_{k-1},\ v_{k-1}$, ...', 
            'x' => x1, 'y' => y5, 'justification' => x1_just)
        t.show_text(
            'text' => 'face k-1', 'x' => x3, 'y' => y5, 'justification' => x3_just, 'scale' => 1.2)
        
        t.show_text(
            'text' => 
    '$m_{k},\ r_{k},\ L_{k},\ v_{k},\ D_{k},\ F_{i,k},\ \overline{dm}_{k},\ \overline{P}_{k},\ \overline{T}_{k},\ \nabla_{T,k}$', 
            'x' => x1, 'y' => y3, 'justification' => x1_just)
        t.show_text(
            'text' => 'face k', 'x' => x3, 'y' => y3, 'justification' => x3_just, 'scale' => 1.2)
        
        t.show_text(
            'text' => '$m_{k+1},\ r_{k+1},\ L_{k+1},\ v_{k+1},\ D_{k+1},\ F_{i,k+1}$, ...', 
            'x' => x1, 'y' => y1, 'justification' => x1_just)
        t.show_text(
            'text' => 'face k+1', 'x' => x3, 'y' => y1, 'justification' => x3_just, 'scale' => 1.2)
        
        
        t.show_text(
            'text' => '$dm_{k-1},\ \rho_{k-1},\ T_{k-1},\ X_{i,k-1},\ P_{k-1}$, ...', 
            'x' => x1, 'y' => y4, 'justification' => x1_just)
        t.show_text(
            'text' => 'cell k-1', 'x' => x3, 'y' => y4, 'justification' => x3_just, 'scale' => 1.2)

        
        t.show_text(
            'text' => '$dm_{k},\ \rho_{k},\ T_{k},\ X_{i,k},\ P_{k},\ \nabla_{\!ad,k},\ \epsilon_{\rm nuc,k},\ \epsilon_{\rm grav,k}$', 
            'x' => x1, 'y' => y2, 'justification' => x1_just)
        t.show_text(
            'text' => 'cell k', 'x' => x3, 'y' => y2, 'justification' => x3_just, 'scale' => 1.2)
        
    end
       

end
