# profile_plots.rb

class ProfilePlots

    include Math
    include Tioga
    include FigureConstants
    
    def background
        return
        
        t.fill_color = FloralWhite
        t.fill_opacity = 0.5
        t.fill_frame
    end
    
    def show_box_labels(title, xlabel=nil, ylabel=nil)
        if title != nil
            t.show_title(title); t.no_title
        end
        if xlabel != nil
            t.show_xlabel(xlabel); t.no_xlabel
        end
        if ylabel != nil
            t.show_ylabel(ylabel); t.no_ylabel
        end
    end
    
    def show_model_number(pos = 1.15, shift = 4)
        return if t.in_subplot or t.model_number < 0 or @skip_model_number == true
        txt = sprintf('%i', t.model_number)
        t.show_text('text' => txt, 'side' => TOP, 'pos' => pos, 'shift' => shift, 'scale' => 0.8,
            'justification' => RIGHT_JUSTIFIED)
        t.show_text('text' => d.initial_mass_str, 'side' => TOP, 'pos' => pos, 'shift' => shift-0.5, 'scale' => 0.5,
            'justification' => RIGHT_JUSTIFIED)
        t.show_text('text' => d.initial_z_str, 'side' => TOP, 'pos' => pos, 'shift' => shift-1.9, 'scale' => 0.5,
            'justification' => RIGHT_JUSTIFIED)
        if d.star_mass < 0.1
            mstr = '$M\!$=%0.4f'
        elsif d.star_mass < 1
            mstr = '$M\!$=%0.3f'
        else
            mstr = '$M\!$=%0.2f'
        end
        t.show_text('text' => sprintf(mstr, d.star_mass), 
            'side' => TOP, 'pos' => pos, 'shift' => shift-3.4, 'scale' => 0.5,
            'justification' => RIGHT_JUSTIFIED)
    end
    
    def get_star_age_title(fmt)
        return sprintf(fmt, d.star_age_str)
    end
    
    def show_mass_locs(xs, ymin)
        y1 = ymin + t.default_text_height_dy * 1.5
        t.save_legend_separator(1.5)
        t.line_type = Line_Type_Dot
        t.line_width = 2

        if @show_tau10
          x = xs[d.mass.where_closest(d.tau10_mass)]
          t.stroke_color = Crimson
          t.stroke_line(x, ymin, x, y1)
          t.save_legend_info('$\tau = 10$')
        end
        
        t.line_type = Line_Type_Solid

        x = xs[d.mass.where_closest(d.star_mass * 0.9999)]
        t.stroke_color = RoyalPurple
        t.stroke_line(x, ymin, x, y1)
        t.save_legend_info('99.99\%')
        
        x = xs[d.mass.where_closest(d.star_mass * 0.999)]
        t.stroke_color = BrightBlue
        t.stroke_line(x, ymin, x, y1)
        t.save_legend_info('99.9\%')
        
        x = xs[d.mass.where_closest(d.star_mass * 0.99)]
        t.stroke_color = Goldenrod
        t.stroke_line(x, ymin, x, y1)
        t.save_legend_info('99\%')
        
        x = xs[d.mass.where_closest(d.star_mass * 0.9)]
        t.stroke_color = Lilac
        t.stroke_line(x, ymin, x, y1)
        t.save_legend_info('90\%')
        
        x = xs[d.mass.where_closest(d.star_mass * 0.5)]
        t.stroke_color = Coral
        t.stroke_line(x, ymin, x, y1)
        t.save_legend_info('50\% $ M_{tot} $')
        
    end
    
    
    def plot_polyline(xs, ys, color, label, linetype)
      t.show_polyline(xs[@grid_min..@grid_max], ys[@grid_min..@grid_max], color, label, linetype)
    end 
    
    def plot_polydots(xs, ys, color, marker, scale)
      t.show_marker('xs' => xs[@grid_min..@grid_max], 'ys' => ys[@grid_min..@grid_max], 
        'color' => color, 'marker' => marker, 'scale' => scale)
    end 

    
    def mode_propagation_nu(xs, xlabel)
      mode_propagation(xs, xlabel, false)
    end

    
    def mode_propagation_lognu(xs, xlabel)
      mode_propagation(xs, xlabel, true)
    end

    
    def mode_propagation(xs, xlabel, lognu_flag)
        title = get_star_age_title('Mode Propagation --- Age %s') if title == nil
        title = nil if @no_frills
        #t.set_aspect_ratio(0.8) if @no_frills
         r = d.photosphere_r
         luminosity = d.photosphere_L
         m = d.star_mass
         teff = d.Teff
         teff_sun = 5777.0
         nu_max_sun = 3100.0
         nu_max1 = nu_max_sun*m/(r**2*sqrt(teff/teff_sun))/1000
         nu_max2 = m/luminosity/((teff/teff_sun)**3.5)*nu_max_sun/1000
         #puts "nu_max1 #{nu_max1}"
         #puts "nu_max2 #{nu_max2}"
         nu_max = nu_max2
         xs_nu = [xs.min, xs.max]
        t.xaxis_log_values = @xaxis_log_values
        ymax = @mode_prop_ymax
        if lognu_flag
           ylabel = 'log frequency (mHz)'
           ymax = d.log_brunt_nu[@grid_min..@grid_max].max - 3 if ymax == nil
           ymin = log10(nu_max) - 1
        else
           ylabel = 'frequency (mHz)'
           ymin = 0
           ymax = d.brunt_nu[@grid_min..@grid_max].max/1000 if ymax == nil
        end
        dy = ymax - ymin
        ymax = ymax + 0.1*dy
        ymin = ymin - 0.1*dy
        ymin = 0
        show_box_labels(title, xlabel, ylabel)
        t.legend_text_dy = 1        
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do

           mixing_type = d.mixing_type
           mixing_type = d.conv_mixing_type if mixing_type == nil
           mixing_type = mixing_type[@grid_min..@grid_max]
           t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
              'top_boundary' => ymax, 'bottom_boundary' => ymin) do
              show_convection_regions(mixing_type, xs, xleft, xright, ymin, ymax)
              end

            cnt = 0
            
            if lognu_flag
            
               cnt = plot_mode_propagation_line(cnt, xs, d.log_brunt_nu - 3, 'log $\mathrm{N_{BV}}$')

               cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl1 - 3, 'log L1')
               cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl2 - 3, 'log L2')
               cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl3 - 3, 'log L3')
               #cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl10, 'log Lamb S l=10') unless d.log_lamb_Sl10 == nil

               lg_nu_max = log10(nu_max)
               puts "nu_max #{nu_max}"
               t.show_polyline(xs_nu, [lg_nu_max, lg_nu_max], DimGray, '$\nu_\mathrm{max}$', Line_Type_Dash)

               #puts "nu_max #{nu_max}  lg_nu_max #{lg_nu_max}"
               #cnt = plot_mode_propagation_line(cnt, xs_nu, 
               #         [lg_nu_max, lg_nu_max], 'log $\nu_\mathrm{max}$', true)
               #lg_0pt5_nu_max = log10(0.5*nu_max)
               #cnt = plot_mode_propagation_line(cnt, xs_nu, 
               #         [lg_0pt5_nu_max, lg_0pt5_nu_max], 'log 0.5 $\nu_\mathrm{max}$', true)

            else
            
               cnt = plot_mode_propagation_line(cnt, xs, d.brunt_nu.div(1e3), '$\mathrm{N_{BV}}$')
               
               cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl1.exp10.div(1e3), 'L1')
               cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl2.exp10.div(1e3), 'L2')
               cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl3.exp10.div(1e3), 'L3')
               #cnt = plot_mode_propagation_line(cnt, xs, d.log_lamb_Sl10.exp10.div(1e3), 'Sl10') unless d.log_lamb_Sl10 == nil
               
               puts "nu_max #{nu_max}"
               t.show_polyline(xs_nu, [nu_max, nu_max], DimGray, '$\nu_\mathrm{max}$', Line_Type_Dash)
               #cnt = plot_mode_propagation_line(cnt, xs_nu, 
               #         [nu_max, nu_max], '$\nu_\mathrm{max}$', true)
               #cnt = plot_mode_propagation_line(cnt, xs_nu, 
               #         [0.5*nu_max, 0.5*nu_max], '0.5 $\nu_\mathrm{max}$', true)

            end

        end
    end

    
    def plot_mode_propagation_line(cnt, xs, ys_in, txt, use_given_xs_ys = false)
        show_lim = 1e-50 #10**ymin
        return cnt if ys_in == nil
        return cnt unless ys_in.max > show_lim

        ys = ys_in
        if use_given_xs_ys
           xs_plot = xs
           ys_plot = ys
        else
           xs_plot = xs[@grid_min..@grid_max]
           ys_plot = ys[@grid_min..@grid_max]
        end
        
        colors = [ BrightBlue, Goldenrod, FireBrick, RoyalPurple, Teal, Coral, Lilac ]
        num_colors = colors.length
        patterns = [ Line_Type_Solid, Line_Type_Dash, Line_Type_Dot, Line_Type_Dot_Dash, Line_Type_Dot_Long_Dash ]
        num_patterns = patterns.length
        
        color = colors[cnt - num_colors*(cnt/num_colors)]
        pattern = patterns[cnt/num_colors]

        t.show_polyline(xs_plot, ys_plot, color, txt, pattern)

        return cnt+1
    end 

    
    def plot_brunt_terms_line(cnt, xs, ys_in, txt, use_given_xs_ys = false)
        show_lim = 1e-50 #10**ymin
        return cnt if ys_in == nil
        #return cnt unless ys_in.max > show_lim

        ys = ys_in
        if use_given_xs_ys
           xs_plot = xs
           ys_plot = ys
        else
           xs_plot = xs[@grid_min..@grid_max]
           ys_plot = ys[@grid_min..@grid_max]
        end
        
        colors = [ BrightBlue, Goldenrod, Coral, FireBrick, RoyalPurple, Lilac ]
        num_colors = colors.length
        patterns = [ Line_Type_Solid, Line_Type_Dash, Line_Type_Dot, Line_Type_Dot_Dash, Line_Type_Dot_Long_Dash ]
        num_patterns = patterns.length
        
        color = colors[cnt - num_colors*(cnt/num_colors)]
        pattern = patterns[cnt/num_colors]

        t.show_polyline(xs_plot, ys_plot, color, txt, pattern)

        return cnt+1
    end 


    
    def abundances(xs, xlabel)
        title = get_star_age_title('Abundances --- Age %s') if title == nil
        title = nil if @no_frills

        t.set_aspect_ratio(0.8) if @no_frills
        show_box_labels(title, xlabel, 'log mass fraction')
        t.legend_text_dy = 0.9    
        ymin = @log_mass_frac_ymin
        if ymin == nil
            ymin = -15.1
            lgz = log10(d.initial_z+1e-9)-1
            ymin = lgz if lgz-1 < ymin
        end 
        ymax = @log_mass_frac_ymax
        if ymax == nil
            if ymin < -8
                ymax = 0.5
            else
                ymax = 0.2
            end
        end
        show_plot_abundances(xs, ymin, ymax)
    end

    
    
    def plot_abundance_line(cnt, xs, ys_in, chem_name)
        show_lim = 1e-50 #10**ymin
        return cnt if ys_in == nil
        return cnt unless ys_in.max > show_lim
        return cnt+1 if @abundances_to_skip.include?(chem_name)
        ys = ys_in.safe_log10
        xs_plot = xs[@grid_min..@grid_max]
        ys_plot = ys[@grid_min..@grid_max]
        txt = nil
        txt = '\it ' + chem_name if @legend_abundance_lines == true || @doing_multiple != true
        
        colors = [ BrightBlue, Goldenrod, Coral, FireBrick, RoyalPurple, Lilac ]
        num_colors = colors.length
        patterns = [ Line_Type_Solid, Line_Type_Dash, Line_Type_Dot, Line_Type_Dot_Dash, Line_Type_Dot_Long_Dash ]
        num_patterns = patterns.length
        
        color = colors[cnt - num_colors*(cnt/num_colors)]
        pattern = patterns[cnt/num_colors]

        t.show_polyline(xs_plot, ys_plot, color, txt, pattern)
        t.show_marker('xs' => xs_plot, 'ys' => ys_plot, 'marker' => Bullet,
            'color' => color, 'scale' => 0.4) if @show_points
        
        return cnt+1 unless @num_abundance_line_labels > 0
        return cnt+1 if txt == nil
        
        xwidth = xs_plot.max - xs_plot.min
        n = @num_abundance_line_labels
        dx = xwidth/n
        xs_reversed = xs_plot.reverse # need to reverse for use with linear_interpolate
        ys_reversed = ys_plot.reverse
        scale = 0.6
        dy = t.default_text_height_dy*scale/3
        n.times do |j|
            x = xs_plot.min + dx*(j+0.5)
            y = Dvector.linear_interpolate(x, xs_reversed, ys_reversed)
            if y < 0*dy
                y = y + dy
            else
                y = y - dy*0
            end
            t.show_label(
                'x' => x, 'y' => y, 
                'text' => txt, 'scale' => scale, 'alignment' => ALIGNED_AT_BASELINE)
        end
        return cnt+1
    end 


   def get_abundance_info(abundances, abund, str)
      return abundances if abund == nil
      ymax = abund[@grid_min..@grid_max].max
      #puts "#{str} #{ymax}"
      abundances << [ ymax, abund, str ]
      return abundances
   end


    def show_plot_abundances(xs, ymin, ymax)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        
        abundances = []
        
         abundances = get_abundance_info(abundances, d.h1, '$^{1}$\rm H')
         abundances = get_abundance_info(abundances, d.h2, '$^{2}$\rm H')
         abundances = get_abundance_info(abundances, d.he3, '$^{3}$\rm He')
         abundances = get_abundance_info(abundances, d.he4, '$^{4}$\rm He')
         abundances = get_abundance_info(abundances, d.li7, '$^{7}$\rm Li')
         abundances = get_abundance_info(abundances, d.be7, '$^{7}$\rm Be')
         abundances = get_abundance_info(abundances, d.b8, '$^{8}$\rm B')
         abundances = get_abundance_info(abundances, d.c12, '$^{12}$\rm C')
         abundances = get_abundance_info(abundances, d.c13, '$^{13}$\rm C')
         abundances = get_abundance_info(abundances, d.c14, '$^{14}$\rm C')
         abundances = get_abundance_info(abundances, d.n13, '$^{13}$\rm N')
         abundances = get_abundance_info(abundances, d.n14, '$^{14}$\rm N')
         abundances = get_abundance_info(abundances, d.n15, '$^{15}$\rm N')
         abundances = get_abundance_info(abundances, d.o14, '$^{14}$\rm O')
         abundances = get_abundance_info(abundances, d.o15, '$^{15}$\rm O')
         abundances = get_abundance_info(abundances, d.o16, '$^{16}$\rm O')
         abundances = get_abundance_info(abundances, d.o17, '$^{17}$\rm O')
         abundances = get_abundance_info(abundances, d.o18, '$^{18}$\rm O')
         abundances = get_abundance_info(abundances, d.f17, '$^{17}$\rm F')
         abundances = get_abundance_info(abundances, d.f18, '$^{18}$\rm F')
         abundances = get_abundance_info(abundances, d.f19, '$^{19}$\rm F')
         abundances = get_abundance_info(abundances, d.ne18, '$^{18}$\rm Ne')
         abundances = get_abundance_info(abundances, d.ne19, '$^{19}$\rm Ne')
         abundances = get_abundance_info(abundances, d.ne20, '$^{20}$\rm Ne')
         abundances = get_abundance_info(abundances, d.ne21, '$^{21}$\rm Ne')
         abundances = get_abundance_info(abundances, d.ne22, '$^{22}$\rm Ne')
         abundances = get_abundance_info(abundances, d.na23, '$^{23}$\rm Na')
         abundances = get_abundance_info(abundances, d.mg22, '$^{22}$\rm Mg')
         abundances = get_abundance_info(abundances, d.mg24, '$^{24}$\rm Mg')
         abundances = get_abundance_info(abundances, d.mg25, '$^{25}$\rm Mg')
         abundances = get_abundance_info(abundances, d.mg26, '$^{26}$\rm Mg')
         
         abundances = get_abundance_info(abundances, d.al26, '$^{26}$\rm Al')
         abundances = get_abundance_info(abundances, d.al27, '$^{27}$\rm Al')
         
         abundances = get_abundance_info(abundances, d.si28, '$^{28}$\rm Si')
         abundances = get_abundance_info(abundances, d.si29, '$^{29}$\rm Si')
         abundances = get_abundance_info(abundances, d.si30, '$^{30}$\rm Si')

         abundances = get_abundance_info(abundances, d.p31, '$^{31}$\rm P')
         
         abundances = get_abundance_info(abundances, d.s30, '$^{30}$\rm S')         
         abundances = get_abundance_info(abundances, d.s32, '$^{32}$\rm S')
         abundances = get_abundance_info(abundances, d.s33, '$^{33}$\rm S')
         abundances = get_abundance_info(abundances, d.s34, '$^{34}$\rm S')
         abundances = get_abundance_info(abundances, d.s36, '$^{36}$\rm S')
         
         abundances = get_abundance_info(abundances, d.cl35, '$^{35}$\rm Cl')
         abundances = get_abundance_info(abundances, d.cl36, '$^{36}$\rm Cl')
         abundances = get_abundance_info(abundances, d.cl37, '$^{37}$\rm Cl')
         
         abundances = get_abundance_info(abundances, d.ar36, '$^{36}$\rm Ar')
         abundances = get_abundance_info(abundances, d.ar37, '$^{37}$\rm Ar')
         abundances = get_abundance_info(abundances, d.ar38, '$^{38}$\rm Ar')
         abundances = get_abundance_info(abundances, d.ar40, '$^{40}$\rm Ar')
         
         abundances = get_abundance_info(abundances, d.k39, '$^{39}$\rm K')
         
         abundances = get_abundance_info(abundances, d.ca40, '$^{40}$\rm Ca')
         abundances = get_abundance_info(abundances, d.ca41, '$^{41}$\rm Ca')
         abundances = get_abundance_info(abundances, d.ca42, '$^{42}$\rm Ca')
         abundances = get_abundance_info(abundances, d.ca44, '$^{44}$\rm Ca')
         abundances = get_abundance_info(abundances, d.ca46, '$^{46}$\rm Ca')
         abundances = get_abundance_info(abundances, d.ca48, '$^{48}$\rm Ca')
         
         abundances = get_abundance_info(abundances, d.sc43, '$^{43}$\rm Sc')
         abundances = get_abundance_info(abundances, d.sc44, '$^{44}$\rm Sc')
         abundances = get_abundance_info(abundances, d.sc45, '$^{45}$\rm Sc')
         
         abundances = get_abundance_info(abundances, d.ti44, '$^{44}$\rm Ti')
         abundances = get_abundance_info(abundances, d.ti45, '$^{45}$\rm Ti')
         abundances = get_abundance_info(abundances, d.ti46, '$^{46}$\rm Ti')
         abundances = get_abundance_info(abundances, d.ti48, '$^{48}$\rm Ti')
         abundances = get_abundance_info(abundances, d.ti50, '$^{50}$\rm Ti')
         abundances = get_abundance_info(abundances, d.ti52, '$^{52}$\rm Ti')
         
         abundances = get_abundance_info(abundances, d.v47, '$^{47}$\rm V')
         abundances = get_abundance_info(abundances, d.v51, '$^{51}$\rm V')
         
         abundances = get_abundance_info(abundances, d.cr48, '$^{48}$\rm Cr')
         abundances = get_abundance_info(abundances, d.cr49, '$^{49}$\rm Cr')
         abundances = get_abundance_info(abundances, d.cr50, '$^{50}$\rm Cr')
         abundances = get_abundance_info(abundances, d.cr51, '$^{51}$\rm Cr')
         abundances = get_abundance_info(abundances, d.cr52, '$^{52}$\rm Cr')
         abundances = get_abundance_info(abundances, d.cr53, '$^{53}$\rm Cr')
         abundances = get_abundance_info(abundances, d.cr54, '$^{54}$\rm Cr')
         abundances = get_abundance_info(abundances, d.cr55, '$^{55}$\rm Cr')
         abundances = get_abundance_info(abundances, d.cr56, '$^{56}$\rm Cr')
         
         abundances = get_abundance_info(abundances, d.mn51, '$^{51}$\rm Mn')
         abundances = get_abundance_info(abundances, d.mn52, '$^{52}$\rm Mn')
         abundances = get_abundance_info(abundances, d.mn53, '$^{53}$\rm Mn')
         abundances = get_abundance_info(abundances, d.mn54, '$^{54}$\rm Mn')
         abundances = get_abundance_info(abundances, d.mn55, '$^{55}$\rm Mn')
         abundances = get_abundance_info(abundances, d.mn56, '$^{56}$\rm Mn')
         abundances = get_abundance_info(abundances, d.mn57, '$^{57}$\rm Mn')
         
         abundances = get_abundance_info(abundances, d.fe52, '$^{52}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe53, '$^{53}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe54, '$^{54}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe55, '$^{55}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe56, '$^{56}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe57, '$^{57}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe58, '$^{58}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe59, '$^{59}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe60, '$^{60}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe61, '$^{61}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe62, '$^{62}$\rm Fe')

         abundances = get_abundance_info(abundances, d.fe64, '$^{64}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe66, '$^{66}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe68, '$^{68}$\rm Fe')
         abundances = get_abundance_info(abundances, d.fe70, '$^{70}$\rm Fe')
         
         abundances = get_abundance_info(abundances, d.co55, '$^{55}$\rm Co')
         abundances = get_abundance_info(abundances, d.co56, '$^{56}$\rm Co')
         abundances = get_abundance_info(abundances, d.co57, '$^{57}$\rm Co')
         abundances = get_abundance_info(abundances, d.co58, '$^{58}$\rm Co')
         abundances = get_abundance_info(abundances, d.co59, '$^{59}$\rm Co')
         abundances = get_abundance_info(abundances, d.co60, '$^{60}$\rm Co')
         abundances = get_abundance_info(abundances, d.co61, '$^{61}$\rm Co')
         abundances = get_abundance_info(abundances, d.co62, '$^{62}$\rm Co')
         
         abundances = get_abundance_info(abundances, d.ni53, '$^{53}$\rm Ni')
         abundances = get_abundance_info(abundances, d.ni55, '$^{55}$\rm Ni')
         abundances = get_abundance_info(abundances, d.ni56, '$^{56}$\rm Ni')
         abundances = get_abundance_info(abundances, d.ni57, '$^{57}$\rm Ni')
         abundances = get_abundance_info(abundances, d.ni58, '$^{58}$\rm Ni')
         abundances = get_abundance_info(abundances, d.ni59, '$^{59}$\rm Ni')
         abundances = get_abundance_info(abundances, d.ni60, '$^{60}$\rm Ni')
         abundances = get_abundance_info(abundances, d.ni61, '$^{61}$\rm Ni')
         abundances = get_abundance_info(abundances, d.ni62, '$^{62}$\rm Ni')
         abundances = get_abundance_info(abundances, d.ni64, '$^{64}$\rm Ni')
         abundances = get_abundance_info(abundances, d.ni66, '$^{66}$\rm Ni')
         
         abundances = get_abundance_info(abundances, d.zn60, '$^{60}$\rm Zn')
         abundances = get_abundance_info(abundances, d.zn62, '$^{62}$\rm Zn')
         abundances = get_abundance_info(abundances, d.zn64, '$^{64}$\rm Zn')
         abundances = get_abundance_info(abundances, d.zn66, '$^{66}$\rm Zn')
         abundances = get_abundance_info(abundances, d.zn68, '$^{68}$\rm Zn')
         abundances = get_abundance_info(abundances, d.zn70, '$^{70}$\rm Zn')
         
         abundances = get_abundance_info(abundances, d.ge64, '$^{64}$\rm Ge')
         abundances = get_abundance_info(abundances, d.ge72, '$^{72}$\rm Ge')
         
         abundances = get_abundance_info(abundances, d.se68, '$^{68}$\rm Se')
         abundances = get_abundance_info(abundances, d.kr72, '$^{72}$\rm Kr')
         abundances = get_abundance_info(abundances, d.sr76, '$^{76}$\rm Sr')
         abundances = get_abundance_info(abundances, d.sn104, '$^{104}$\rm Sn')
         abundances = get_abundance_info(abundances, d.neut, 'neut')
         abundances = get_abundance_info(abundances, d.prot, 'prot')
         
         return if abundances.length == 0
         
         abundances.sort! { |a,b| a[0] <=> b[0] }
         abundances.reverse!
         #puts "after sort"
         #abundances.each { |a| puts "#{a[2]} #{a[0]}" }
         
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            cnt = 0
            abundances.each { |a| 
               cnt = plot_abundance_line(cnt, xs, a[1], a[2]) unless cnt > 24 }
        end
        
    end 


    
    def mixing_Ds(xs, xlabel)
        title = get_star_age_title('mixing Ds --- Age %s') if title == nil
        title = nil if @no_frills

        t.set_aspect_ratio(0.8) if @no_frills
        show_box_labels(title, xlabel, 'log D')
        t.legend_text_dy = 1        
        ymin = -2
        ymax_factor = 1.1
        ymax = d.log_D_mix[@grid_min..@grid_max].max
        if (ymax > 0)
            ymax = ymax*ymax_factor
        else
            ymax = 0
        end
        if d.am_log_D_SSI != nil
            ymax2 = d.am_log_D_SSI[@grid_min..@grid_max].max
            if (ymax2 > 0)
                ymax2 = ymax2*ymax_factor
            else
                ymax2 = 0
            end if
            ymax = ymax2 if ymax2 > ymax
        end
        if d.am_log_D_ES != nil
            ymax2 = d.am_log_D_ES[@grid_min..@grid_max].max
            if (ymax2 > 0)
                ymax2 = ymax2*ymax_factor
            else
                ymax2 = 0
            end if
            ymax = ymax2 if ymax2 > ymax
        end
        if d.am_log_D_GSF != nil
            ymax2 = d.am_log_D_GSF[@grid_min..@grid_max].max
            if (ymax2 > 0)
                ymax2 = ymax2*ymax_factor
            else
                ymax2 = 0
            end if
            ymax = ymax2 if ymax2 > ymax
        end
        if d.am_log_D_ST != nil
            ymax2 = d.am_log_D_ST[@grid_min..@grid_max].max
            if (ymax2 > 0)
                ymax2 = ymax2*ymax_factor
            else
                ymax2 = 0
            end if
            ymax = ymax2 if ymax2 > ymax
        end
        show_plot_mixing_Ds(xs, ymin, ymax)
    end
    
    
    def plot_D_line(cnt, xs, ys, mix_D_name)
        show_lim = -3
        return cnt+1 if ys == nil
        return cnt+1 unless ys.max > show_lim
        xs_plot = xs[@grid_min..@grid_max]
        ys_plot = ys[@grid_min..@grid_max]
        txt = nil
        txt = mix_D_name if @legend_mixing_D_lines == true || @doing_multiple != true
        
        colors = [ Teal, Goldenrod, Lilac, Coral, FireBrick, MediumSlateBlue,
            LightSkyBlue, LightOliveGreen, SeaGreen]
        num_colors = colors.length
        patterns = [ Line_Type_Solid, Line_Type_Dash, Line_Type_Dot, 
                     Line_Type_Dot_Dash, Line_Type_Dot_Long_Dash ]
        num_patterns = patterns.length
        
        color = colors[cnt - num_colors*(cnt/num_colors)]
        pattern = patterns[cnt/num_colors]

        t.show_polyline(xs_plot, ys_plot, color, txt, pattern)
        t.show_marker('xs' => xs_plot, 'ys' => ys_plot, 'marker' => Bullet,
            'color' => color, 'scale' => 0.4) if @show_points
        
        return cnt+1 unless @num_mixing_D_line_labels > 0
        return cnt+1 if txt == nil
        
        xwidth = xs_plot.max - xs_plot.min
        n = @num_mixing_D_line_labels
        dx = xwidth/n
        xs_reversed = xs_plot.reverse # need to reverse for use with linear_interpolate
        ys_reversed = ys_plot.reverse
        scale = 0.6
        dy = t.default_text_height_dy*scale/3
        n.times do |j|
            x = xs_plot.min + dx*(j+0.5)
            y = Dvector.linear_interpolate(x, xs_reversed, ys_reversed)
            if y < 0*dy
                y = y + dy
            else
                y = y - dy*0
            end
            t.show_label(
                'x' => x, 'y' => y, 
                'text' => txt, 'scale' => scale, 'alignment' => ALIGNED_AT_BASELINE)
        end
        return cnt+1
    end 

    
    def show_plot_mixing_Ds(xs, ymin, ymax)
        @num_mixing_D_line_labels = 0
        @legend_mixing_D_lines = true
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            cnt = 0
            cnt = plot_D_line(cnt, xs, d.am_log_D_ST, 'ST')
            cnt = plot_D_line(cnt, xs, d.am_log_D_DSI, 'DSI')
            cnt = plot_D_line(cnt, xs, d.am_log_D_SH, 'SH')
            cnt = plot_D_line(cnt, xs, d.am_log_D_SSI, 'SSI')
            cnt = plot_D_line(cnt, xs, d.am_log_D_ES, 'ES')
            cnt = plot_D_line(cnt, xs, d.am_log_D_GSF, 'GSF')
            cnt = plot_D_line(cnt, xs, d.log_D_conv, 'conv')
            cnt = plot_D_line(cnt, xs, d.log_D_semi, 'semi')
            cnt = plot_D_line(cnt, xs, d.log_D_th, 'th')
            cnt = plot_D_line(cnt, xs, d.log_D_ovr, 'ovr')
            cnt = plot_D_line(cnt, xs, d.log_D_anon, 'anon')
            cnt = plot_D_line(cnt, xs, d.log_D_minimum, 'min')
            cnt = plot_D_line(cnt, xs, d.am_log_D_visc, 'visc')
        end
    end 

    
    def dynamo_info
        @no_frills = true
        title = get_star_age_title('dynamo information --- Age %s') if title == nil
        title = nil if @no_frills

        t.set_aspect_ratio(0.8) if @no_frills
        
        t.rescale(1.3)

        xs = xs_for_plot()
        xlabel = xlabel_for_plot()

        show_box_labels(title, xlabel, nil)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        xs = xs[@grid_min..@grid_max]
        
      t.legend_text_dy = 1.0
      #show_model_number
      t.show_plot_with_legend('legend_left_margin' => 0.05, 'legend_top_margin' => 0.01,
                              'plot_scale' => 1,
                              'legend_scale' => 1.3,
                              'plot_right_margin' => 0) { 
        # NOTE: use subfigure rather than subplot to get combined legend info
        # subplot would be correct if each had its own legend
            t.subfigure {
               t.yaxis_loc = t.ylabel_side = LEFT
               t.right_edge_type = AXIS_HIDDEN
               t.show_ylabel('log B')
               ymin = -2.1
               ymax = 10
               t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
                   'top_boundary' => ymax, 'bottom_boundary' => ymin) do
                    log_B_r = get_by_name_for_plot('dynamo_log_B_r')
                    log_B_phi = get_by_name_for_plot('dynamo_log_B_phi')
                    mixing_type = d.mixing_type[@grid_min..@grid_max]
                    t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
                       'top_boundary' => ymax, 'bottom_boundary' => ymin) do
                       show_convection_regions(mixing_type, xs, xleft, xright, ymin, ymax)
                    end
                    t.show_polyline(xs, log_B_r, Coral, 'B_r')
                    t.show_polyline(xs, log_B_phi, Teal, 'B_{\phi}')
               end
             }
            t.subfigure {
               t.yaxis_loc = t.ylabel_side = RIGHT
               t.left_edge_type = AXIS_HIDDEN
               t.show_ylabel('log $(\omega)$ and log $(j \times 10^{-20}$)')
               ymin = -11
               ymax = 1
               t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
                   'top_boundary' => ymax, 'bottom_boundary' => ymin) do
                    log_j = get_by_name_for_plot('log_j_rot')
                    log_j.sub!(20)
                    log_omega = get_by_name_for_plot('log_omega')
                    t.show_polyline(xs, log_omega, BrightBlue, '\omega')
                    t.show_polyline(xs, log_j, FireBrick, 'j')
               end
             }
         }
         
         log_J_inside = d.log_J_inside
         return if log_J_inside == nil
         
         ms = [ 1.5, 2.5, 3.5 ]
         ms.each do |m|
             k = d.mass.where_closest(m)
             log_J = log_J_inside[k]
             puts "m #{m} log_J #{log_J} J #{10**(log_J)}"
         end
        
    end 
    
    
    def show_convection_regions(mixing_type, xs, xleft, xright, ymin, ymax)
        conv_type = 1
        k = 0; k0 = 0
        in_conv_region = (mixing_type[k].to_i == conv_type)
        t.fill_color = DarkSmoke
        mixing_type.length.times do |k|
            if in_conv_region
                if mixing_type[k].to_i != conv_type
                    dx = xs[k]-xs[k0]
                    if dx > 0
                        t.fill_rect(xs[k0], ymin, dx, ymax-ymin)
                    else
                        t.fill_rect(xs[k], ymin, -dx, ymax-ymin)
                    end
                    in_conv_region = false
                end
            else
                if mixing_type[k].to_i == conv_type
                    in_conv_region = true
                    k0 = k
                end
            end
        end
        if in_conv_region
            k = -1
            dx = xs[k]-xs[k0]
            if dx > 0
                t.fill_rect(xs[k0], ymin, dx, ymax-ymin)
            else
                t.fill_rect(xs[k], ymin, -dx, ymax-ymin)
            end
        end
    end



    
    def astero_stuff(xs, xlabel)
        title = nil
        #t.set_aspect_ratio(0.8)
        ylabel = 'L, T, m, $\epsilon_{nuc}$ scaled by max'
        ymin = -0.02
        ymax = 1.05
        show_box_labels(title, xlabel, ylabel)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do

           mixing_type = d.mixing_type
           mixing_type = d.conv_mixing_type if mixing_type == nil
           mixing_type = mixing_type[@grid_min..@grid_max]
           t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
              'top_boundary' => ymax, 'bottom_boundary' => ymin) do
              show_convection_regions(mixing_type, xs, xleft, xright, ymin, ymax)
           end

           ys = get_by_name('h1')
           t.show_polyline(xs, ys, BrightBlue, '$^{1}$\rm H')

           ys = get_by_name('he4')
           t.show_polyline(xs, ys, Goldenrod, '$^{4}$\rm He')

           ys = get_by_name('q')
           t.show_polyline(xs, ys, Coral, 'm')

           ys = get_by_name('luminosity')
           ys.div!(ys.max)
           t.show_polyline(xs, ys, Lilac, 'L')

           ys = get_by_name('eps_nuc')
           ys.div!(ys.max)
           t.show_polyline(xs, ys, FireBrick, '$\epsilon_{nuc}$')

           ys = get_by_name('temperature')
           ys.div!(ys.max)
           t.show_polyline(xs, ys, Teal, 'T')


        end
    end



    
    def signed_scaled_log10(v,scale)
      v.map {|x| (x > 0)? log10(1+x*scale) : -log10(1-x*scale) }
    end
    
    
    def show_y0(xleft, xright)
      t.context do
        t.line_type = Line_Type_Dash
        t.stroke_color = Black
        t.line_width = 0.5
        t.stroke_line(xleft,0,xright,0)
      end
    end

    
    def rho_T_2Ys(xs, title, xlabel, xleft, xright)
        t.subplot {
            t.yaxis_loc = t.ylabel_side = LEFT;
            t.right_edge_type = AXIS_HIDDEN; density(xs, title, xlabel, xleft, xright) }
        t.subplot {
            t.yaxis_loc = t.ylabel_side = RIGHT;
            t.left_edge_type = AXIS_HIDDEN; temperature(xs, title, xlabel, xleft, xright) }
    end
    
    def convection(xs, xlabel, ymin = -0.4, ymax = 1.3)
    
      #ymin = 0.05
      #ymax = 0.45
       
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        title = get_star_age_title('Convection --- Age %s') if title == nil
        show_box_labels(title, xlabel, '$ dlnT/dlnP $')
        #ymin = 0.22; ymax = 0.27
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            plot_polyline(xs, d.grada, BrightBlue, '$ \nabla_{ad} $', Line_Type_Solid)
            plot_polyline(xs, d.gradr, Goldenrod, '$ \nabla_{rad} $', Line_Type_Solid)
            plot_polyline(xs, d.gradT, Crimson, '$ \nabla_{actual} $', Line_Type_Dot)
            show_mass_locs(xs, ymin)
        end
    end
    
    def power_line(cnt, xs, vals, text, ymin)
        return cnt if vals == nil
        ys = @temp_vec2
        ys.replace(vals).safe_log10!
        return cnt if ys.max < ymin
        
        colors = [ BrightBlue, Goldenrod, Coral, Lilac, FireBrick, RoyalPurple ]
        num_colors = colors.length
        patterns = [ Line_Type_Solid, Line_Type_Dash, Line_Type_Dot, Line_Type_Dot_Dash, Line_Type_Dot_Long_Dash ]
        num_patterns = patterns.length
        
        color = colors[cnt - num_colors*(cnt/num_colors)]
        pattern = patterns[cnt/num_colors]

        plot_polyline(xs, ys, color, text, pattern)
        t.show_marker('xs' => xs, 'ys' => ys, 'marker' => Bullet,
            'color' => color, 'scale' => 0.4) if @show_points
        cnt = cnt+1
    end
    
    
    def power(xs, xlabel, ymin = -4.1, ymax = 15)
        power_plot(xs, xlabel, ymin, ymax, false)
    end
    
    
    def power_plus(xs, xlabel, ymin = -4.1, ymax = 15)
        power_plot(xs, xlabel, ymin, ymax, true)
    end

    
    def power_plot(xs, xlabel, ymin, ymax, do_plus)
        title = get_star_age_title('Power --- Age %s') if title == nil
        show_box_labels(title, xlabel, 'log $ergs/g/s$')
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        
        neg_photo = d.photo.neg

        ary = [ d.non_nuc_neu, 
                d.pp,
                d.cno,
                d.tri_alfa,
                d.burn_c,
                d.burn_n,
                d.burn_o,
                d.burn_ne,
                d.burn_na,
                d.burn_mg,
                d.burn_si,
                d.burn_s,
                d.burn_ar,
                d.burn_ca,
                d.burn_ti,
                d.burn_cr,
                d.burn_fe,
                d.c12_c12,
                d.c12_o16,
                d.o16_o16,
                d.pnhe4,
                d.other,
                neg_photo
                ]
        
        ary << d.dL_dm if do_plus
                
        ymax = Dvector.max_of_many(ary)
        ymax = 1e-99 if ymax < 1e-99
        ymax = ymax.log10
        if ymax <= 14
            ymax = 15.1
        else
            ymax = ymax*1.1
            ymin = -0.1
        end

        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            
            cnt = 0
            cnt = power_line(cnt, xs, d.non_nuc_neu, '\it neutrino losses', ymin)
            cnt = power_line(cnt, xs, neg_photo, '\it photodisintegration', ymin)
            
            cnt = power_line(cnt, xs, d.pp, '\it PP', ymin)
            cnt = power_line(cnt, xs, d.cno, '\it CNO', ymin)
            cnt = power_line(cnt, xs, d.tri_alfa, '$ 3 alpha $', ymin)

            cnt = power_line(cnt, xs, d.c12_c12, '$ C12+C12 $', ymin)
            cnt = power_line(cnt, xs, d.c12_o16, '$ C12+O16 $', ymin)            
            cnt = power_line(cnt, xs, d.o16_o16, '$ O16+O16 $', ymin)
           
            cnt = power_line(cnt, xs, d.burn_c, '\it C', ymin)
            cnt = power_line(cnt, xs, d.burn_n, '\it N', ymin)
            cnt = power_line(cnt, xs, d.burn_o, '\it O', ymin)
            cnt = power_line(cnt, xs, d.burn_ne, '\it Ne', ymin)
            cnt = power_line(cnt, xs, d.burn_na, '\it Na', ymin)
            cnt = power_line(cnt, xs, d.burn_mg, '\it Mg', ymin)
            cnt = power_line(cnt, xs, d.burn_si, '\it Si', ymin)
            cnt = power_line(cnt, xs, d.burn_s, '\it S', ymin)
            cnt = power_line(cnt, xs, d.burn_ar, '\it Ar', ymin)
            cnt = power_line(cnt, xs, d.burn_ca, '\it Ca', ymin)
            cnt = power_line(cnt, xs, d.burn_ti, '\it Ti', ymin)
            cnt = power_line(cnt, xs, d.burn_cr, '\it Cr', ymin)
            cnt = power_line(cnt, xs, d.burn_fe, '\it Fe', ymin)
            
            cnt = power_line(cnt, xs, d.pnhe4, '$ 2 n + 2 p \rightarrow He4 $', ymin)
            cnt = power_line(cnt, xs, d.other, '\it Other', ymin)
            cnt = power_line(cnt, xs, d.dL_dm, '$ dL/dm $', ymin) if do_plus

        end
    end
    
    
    def neutrinos(xs, xlabel, ymin = -11, ymax = 16)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        title = get_star_age_title('Neutrinos --- Age %s') if title == nil
        show_box_labels(title, xlabel, 'log $ergs/g/s$')
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            show_mesh(xs)
            cnt = 0
            cnt = power_line(cnt, xs, d.nonnucneu_phot, 'photo neutrinos', ymin)
            cnt = power_line(cnt, xs, d.nonnucneu_plas, 'plasma neutrinos', ymin)
            cnt = power_line(cnt, xs, d.nonnucneu_brem, 'brem neutrinos', ymin)
            cnt = power_line(cnt, xs, d.nonnucneu_pair, 'pair neutrinos', ymin)
            cnt = power_line(cnt, xs, d.nonnucneu_reco, 'recombination neutrinos', ymin)
            show_mass_locs(xs, ymin) unless xs == d.mass || xs == d.mmid
        end
    end
    
    
    def abundance_C_O_Ne_Mg(xs, xlabel, ymin = -6, ymax = 14)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        title = get_star_age_title('Combined C, O, Ne, Mg --- Age %s') if title == nil
        show_box_labels(title, xlabel, 'mass fraction')
        ys = d.c12.dup
        ys = ys.add!(d.o16)
        ys = ys.add!(d.ne20)
        ys = ys.add!(d.mg24)
        ys_to_plot = ys[@grid_min..@grid_max]
        ymax = ys_to_plot.max
        ymin = ys_to_plot.min
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            t.show_polyline(xs,ys,BrightBlue)
            t.show_marker('xs' => xs, 'ys' => ys, 'marker' => Bullet,
                'color' => BrightBlue, 'scale' => 0.4) if @show_points
        end
    end
    
    
    def abundance_Fe_core(xs, xlabel, ymin = -6, ymax = 14)
        xleft = get_plot_xleft(xs)
        xright = get_plot_xright(xs)
        title = get_star_age_title('Combined Fe and Ni --- Age %s') if title == nil
        show_box_labels(title, xlabel, 'mass fraction')
        ys = Dvector.new(xs.length,0)
        ys = ys.add!(d.fe52) unless d.fe52 == nil
        ys = ys.add!(d.fe53) unless d.fe53 == nil
        ys = ys.add!(d.fe54) unless d.fe54 == nil
        ys = ys.add!(d.co55) unless d.co55 == nil
        ys = ys.add!(d.fe56) unless d.fe56 == nil
        ys = ys.add!(d.ni56) unless d.ni56 == nil
        ys = ys.add!(d.cr56) unless d.cr56 == nil
        ys_to_plot = ys[@grid_min..@grid_max]
        ymax = ys_to_plot.max
        ymin = ys_to_plot.min
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            t.show_polyline(xs,ys,BrightBlue)
            t.show_marker('xs' => xs, 'ys' => ys, 'marker' => Bullet,
                'color' => BrightBlue, 'scale' => 0.4) if @show_points
        end
    end
        
        
    def show_mesh(xs)
        return unless @show_mesh_flag == true
        return if @xaxis_by == 'grid'
        ymin = t.bounds_ymin; ymax = t.bounds_ymax
        t.line_color = LightGray; lw = t.line_width; t.line_width = 0.6
        xs.each_with_index { |x, i| t.stroke_line(x, ymin, x, ymax) if i.mod(10) == 1 }
        t.line_width = lw
    end
    
    def show_properites(name, xs, xlabel)
        t.landscape unless t.in_subplot
        #t.portrait
        show_model_number unless @no_frills
        t.top_edge_type = AXIS_LINE_ONLY
        with_legend = ! @no_legend
        if (with_legend)
            t.rescale(@legend_scale)
            t.legend_text_dy = 1.1
            t.subplot('right_margin' => @plot_with_legend_right_margin) do
                background
                eval_str = sprintf("%s(xs, xlabel)", name)
                eval(eval_str)
            end
            t.set_subframe('left_margin' => @legend_left_margin, 'top_margin' => 0.05)
            t.show_legend
        else
            eval(sprintf("%s(xs, xlabel)", name))
        end
    end

    
    def xs_for_plot()
      if @xaxis_by == 'mass'
         if @use_mmid && d.mmid != nil
            xs = d.mmid
         else
            xs = d.mass
         end
      elsif @xaxis_by == 'q'
          xs = d.q
      elsif @xaxis_by == 'tau'
          xs = d.tau
      elsif @xaxis_by == 'logT'
          xs = d.logT
      elsif @xaxis_by == 'logP'
          xs = d.logP
      elsif @xaxis_by == 'log_column_depth'
          xs = d.log_column_depth
      elsif @xaxis_by == 'radius'
          xs = d.radius
      elsif @xaxis_by == 'r_div_R'
          xs = d.r_div_R
      elsif @xaxis_by == 'acoustic_r_div_R_phot'
          xs = d.acoustic_r_div_R_phot
      elsif @xaxis_by == 'logR'
          xs = d.logR
      elsif @xaxis_by == 'logxm'
          xs = d.logxm
      elsif @xaxis_by == 'logxq'
          xs = d.logxq
      else
          xs = d.zone
          puts("problem in xs_for_plot @xaxis_by = #{@xaxis_by}") unless @xaxis_by == 'grid'
      end
      return xs
    end

    
    def xaxis_reversed?()
      puts "@reverse_xaxis #{@reverse_xaxis}"
      return @reverse_xaxis unless @reverse_xaxis == nil
      return true if @xaxis_by == 'grid'
      return true if @xaxis_by == 'logP'
      return true if @xaxis_by == 'logxm'
      return true if @xaxis_by == 'logxq'
      return true if @xaxis_by == 'log_column_depth'
      return false
    end


    
    def xlabel_for_plot()
      if @xaxis_by == 'mass'
          xlabel = @mass_xlabel
      elsif @xaxis_by == 'q'
          xlabel = @q_xlabel
      elsif @xaxis_by == 'tau'
          xlabel = @tau_xlabel
      elsif @xaxis_by == 'logT'
            xlabel = @logT_xlabel
      elsif @xaxis_by == 'logP'
          xlabel = @logP_xlabel
      elsif @xaxis_by == 'log_column_depth'
          xlabel = @log_col_depth_xlabel
      elsif @xaxis_by == 'radius'
          xlabel = @radius_xlabel
      elsif @xaxis_by == 'r_div_R'
          xlabel = @r_div_R_xlabel
      elsif @xaxis_by == 'acoustic_r_div_R_phot'
          xlabel = @acoustic_r_div_R_phot_xlabel
      elsif @xaxis_by == 'logR'
         if @xaxis_log_values
            xlabel = @radius_xlabel
         else
            xlabel = @logR_xlabel
         end
      elsif @xaxis_by == 'logxm'
          xlabel = @logxm_xlabel
      elsif @xaxis_by == 'logxq'
          xlabel = @logxq_xlabel
      else
          xlabel = @grid_xlabel
      end
      return xlabel
    end

    
    def show_by_q(name)
        xs = d.q
        show_properites(name, xs, @q_xlabel)
    end

    
    def show_by_tau(name)
        xs = d.tau
        show_properites(name, xs, @tau_xlabel)
    end

    
    def show_by_grid(name)
        xs = d.zone
        show_properites(name, xs, @grid_xlabel)
    end
    
    def show_by_mass(name)
        if @use_mmid && d.mmid != nil
            xs = d.mmid
        else
            xs = d.mass
        end
        show_properites(name, xs, @mass_xlabel)
    end
    
    def show_by_logP(name)
        xs = d.logP
        show_properites(name, xs, @logP_xlabel)
    end
    
    def show_by_log_column_depth(name)
        xs = d.log_column_depth
        show_properites(name, xs, @log_col_depth_xlabel)
    end
    
    def show_by_logT(name)
        xs = d.logT
        show_properites(name, xs, @logT_xlabel)
    end
    
    def show_by_logR(name)
        xs = d.logR
        show_properites(name, xs, xlabel_for_plot())
    end
    
    def show_by_radius(name)
        xs = d.radius
        show_properites(name, xs, @radius_xlabel)
    end
    
    def show_by_r_div_R(name)
        xs = d.r_div_R
        puts "xs.max for r_div_R #{xs.max}"
        show_properites(name, xs, @r_div_R_xlabel)
    end
    
    def show_by_acoustic_r_div_R_phot(name)
        xs = d.acoustic_r_div_R_phot
        show_properites(name, xs, @acoustic_r_div_R_phot_xlabel)
    end
    
    def show_by_logxm(name)
        xs = d.logxm
        show_properites(name, xs, @logxm_xlabel)
    end
    
    def show_by_logxq(name)
        xs = d.logxq
        show_properites(name, xs, @logxq_xlabel)
    end
    
    
    def show_by(name)
      if @xaxis_by == 'grid'
        show_by_grid(name)
      elsif @xaxis_by == 'q'
        show_by_q(name)
      elsif @xaxis_by == 'tau'
        show_by_tau(name)
      elsif @xaxis_by == 'mass'
        show_by_mass(name)
      elsif @xaxis_by == 'radius'
        show_by_radius(name)
      elsif @xaxis_by == 'r_div_R'
        show_by_r_div_R(name)
      elsif @xaxis_by == 'acoustic_r_div_R_phot'
        show_by_acoustic_r_div_R_phot(name)
      elsif @xaxis_by == 'logR'
        show_by_logR(name)
      elsif @xaxis_by == 'logP'
        show_by_logP(name)
      elsif @xaxis_by == 'log_column_depth'
        show_by_log_column_depth(name)
      elsif @xaxis_by == 'logT'
        show_by_logT(name)
      elsif @xaxis_by == 'logxm'
        show_by_logxm(name)
      elsif @xaxis_by == 'logxq'
        show_by_logxq(name)
      elsif
        puts "problem in show_by @xaxis_by = #{@xaxis_by}"
      end
    end

    def trio_by
        trio('Profiles --- Age %s', 'power', 'abundances', 'convection')
    end

    def power_and_abundances
        duo('Power and Abundances --- Age %s', 'power', 'abundances')
    end

    def trio_by_both
        t.landscape unless t.in_subplot
        show_model_number
        column_margin = 0.28
        t.rescale(0.75)
        show_box_labels(get_star_age_title('Summary -- Age %s'))
        save_xaxis_by = @xaxis_by
        @xaxis_by = 'mass'
        t.subplot(t.column_margins('num_columns' => 2, 'column' => 1, 'column_margin' => column_margin)) { trio_by }
        @xaxis_by = 'logP'
        t.subplot(t.column_margins('num_columns' => 2, 'column' => 2, 'column_margin' => column_margin)) { trio_by }
        @xaxis_by = save_xaxis_by
    end


    def trio2_by_mass
        t.landscape
        trio('mass', 'Profiles', 'power', 'abundances', 'rho_T')
    end

    def trio2_and_history
        plot_and_history { trio2_by_mass }
    end
    
    def draw_pair(text, val, x_num, y)
        x_text = x_num - 0.48
        t.show_text('text' => text,
                'x' => x_text, 'y' => y, 'justification' => RIGHT_JUSTIFIED)
        t.show_text('text' => '{\sffamily ' + sprintf('%0.6f',val) + '}',
                'x' => x_num, 'y' => y, 'justification' => RIGHT_JUSTIFIED)
    end
    
    def draw_pair_log(text, val, x_num, y)
        draw_pair(text, val.safe_log10, x_num, y)
    end
    
    def info
        t.rescale(0.7)
        xloc = 1.86
        t.show_text('text' => sprintf('Age %s', d.star_age_str),
                'side' => TOP, 'position' => xloc, 'scale' => 0.9, 'shift' => 2,
                'justification' => RIGHT_JUSTIFIED, 'color' => Black)
        t.show_text('text' => sprintf('Mass %0.2f ($ \mathrm{M_{\odot}} $)', d.star_mass),
                'side' => TOP, 'position' => xloc, 'scale' => 0.9, 'shift' => 0.75,
                'justification' => RIGHT_JUSTIFIED, 'color' => Black)
        t.show_text('text' => sprintf('Model %i', d.model_number),
                'side' => BOTTOM, 'position' => xloc, 'scale' => 0.9, 'shift' => 1.75,
                'justification' => RIGHT_JUSTIFIED, 'color' => Black)
        t.rescale(0.58)
        y = 0.97; dy = -t.default_text_height_dy * 1.2
        draw_pair('$ \log L $', d.log_luminosity, xloc, y); y += dy;
        draw_pair('$ \log T_{eff} $', d.log_surface_temp, xloc, y); y += dy;
        draw_pair('$ \log T_{c} $', d.log_center_temp, xloc, y); y += dy;
        draw_pair('$ \log \rho_{c} $', d.log_center_density, xloc, y); y += dy;
        draw_pair('$ \Psi_{c} $', d.center_eta, xloc, y); y += dy;
        draw_pair('Mass', d.star_mass, xloc, y); y += dy;
        draw_pair('He Core', d.h1_boundary_mass, xloc, y); y += dy;
        draw_pair('C/O Core', d.he4_boundary_mass, xloc, y); y += dy;
        y += dy/2;
        draw_pair('$ T_{max} $ Mass', d.mass[d.shell_with_max_temp], xloc, y); y += dy;
        draw_pair_log('$ T_{max} $ $\log \rho $', d.density[d.shell_with_max_temp], xloc, y); y += dy;
        draw_pair_log('$ \log T_{max} $', d.temp_max, xloc, y); y += dy;
        draw_pair_log('$ \log \kappa_{max} $', d.opacity_max, xloc, y); y += dy;
        draw_pair_log('$ \log L_{max} $', d.luminosity_max, xloc, y); y += dy;
        draw_pair('$ \log \rho_{max} $', d.log_center_density, xloc, y); y += dy;
        draw_pair_log('$ \log R_{max} $', d.radius[d.surface_shell], xloc, y); y += dy;
        draw_pair_log('$ \log P_{max}  $', d.pressure[d.center_shell], xloc, y); y += dy;
        y += dy/2;
        draw_pair_log('$ \log L_{\nu}$', d.power_neu, xloc, y); y += dy;
        draw_pair_log('$ \log L_{PP}$', d.power_pp, xloc, y); y += dy;
        draw_pair_log('$ \log L_{CNO}$', d.power_cno, xloc, y); y += dy;
        draw_pair_log('$ \log L_{3 \alpha}$', d.power_3_alpha, xloc, y); y += dy;
        draw_pair_log('$ \log L_{C+\alpha}$', d.power_c_alpha, xloc, y); y += dy;
        draw_pair_log('$ \log L_{N+\alpha}$', d.power_n_alpha, xloc, y); y += dy;
        draw_pair_log('$ \log L_{O+\alpha}$', d.power_o_alpha, xloc, y); y += dy;
        y += dy/2;
        draw_pair('Center XH', d.center_h1, xloc, y); y += dy;
        draw_pair('XHe', d.center_he4, xloc, y); y += dy;
        draw_pair('XC', d.center_c12, xloc, y); y += dy;
        draw_pair('XN', d.center_n14, xloc, y); y += dy;
        draw_pair('XO', d.center_o16, xloc, y); y += dy;
        draw_pair('XNe', d.center_ne20, xloc, y); y += dy;
        y += dy/2;
        draw_pair_log('$ \log t_{nuclear}$', d.nuc_timescale, xloc, y); y += dy;
        draw_pair_log('$ \log t_{thermal}$', d.kh_timescale, xloc, y); y += dy;
        seconds_per_year = 3.155692597e7
        draw_pair_log('$ \log t_{dynamic}$', d.dynamic_time/seconds_per_year, xloc, y); y += dy;
        draw_pair_log('$ \log t_{step}$', d.time_step, xloc, y); y += dy;
        y += dy/2;
    end

    def full_profile
        t.title_shift += 0.5
        t.subplot('right_margin' => 0.17) { trio_by_both }
        t.set_subframe('left_margin' => 0.8)
        info
    end

    
    def plot5(title, plot1, plot2, plot3, plot4, plot5)
        show_model_number
        t.rescale(@subplot_scale)
        if @xaxis_by == 'mass'
          xlabel = @mass_xlabel
        elsif @xaxis_by == 'logP'
          xlabel = @logP_xlabel
        else
          xlabel = @grid_xlabel
        end
        num_plots = 5
        t.legend_scale = 0.6
        t.legend_text_ystart = 0.95
        t.legend_text_dy = 1.2
        row_margin = 0.02
        title = get_star_age_title(title)
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot1, title)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot2, nil)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot3, nil)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 4, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot4, nil)
        end
        t.reset_legend_info
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 5, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_plot_by(plot5, nil)
        end
    end

    
    def quad(title, plot1, plot2, plot3, plot4)
        show_model_number
        t.rescale(@subplot_scale)
        if @xaxis_by == 'mass'
          xlabel = @mass_xlabel
        elsif @xaxis_by == 'logP'
          xlabel = @logP_xlabel
        else
          xlabel = @grid_xlabel
        end
        num_plots = 4
        t.legend_scale = 0.6
        t.legend_text_ystart = 0.95
        t.legend_text_dy = 1.2
        row_margin = 0.02
        title = get_star_age_title(title)
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot1, title)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot2, nil)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot3, nil)
        end
        t.reset_legend_info
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 4, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_plot_by(plot4, nil)
        end
    end

    
    def trio(title, plot1, plot2, plot3)
        show_model_number
        t.rescale(@subplot_scale)
        if @xaxis_by == 'mass'
          xlabel = @mass_xlabel
        elsif @xaxis_by == 'logP'
          xlabel = @logP_xlabel
        else
          xlabel = @grid_xlabel
        end
        num_plots = 3
        t.legend_scale = 0.6
        t.legend_text_ystart = 0.95
        t.legend_text_dy = 1.2
        row_margin = 0.02
        title = get_star_age_title(title)
        @doing_multiple = true
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot1, title)
        end
        t.reset_legend_info
        t.top_edge_type = AXIS_LINE_ONLY
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot2, nil)
        end
        t.reset_legend_info
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_plot_by(plot3, nil)
        end
        @doing_multiple = false
    end

    
    def duo(title, plot1, plot2)
        show_model_number
        t.rescale(@subplot_scale)
        if @xaxis_by == 'mass'
          xlabel = @mass_xlabel
        elsif @xaxis_by == 'logP'
          xlabel = @logP_xlabel
        else
          xlabel = @grid_xlabel
        end
        num_plots = 2
        t.legend_scale = 0.6
        t.legend_text_ystart = 0.95
        t.legend_text_dy = 1.2
        row_margin = 0.02
        title = get_star_age_title(title)
        @doing_multiple = true
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            show_plot_by(plot1)
        end
        t.reset_legend_info
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            show_plot_by(plot2)
        end
        @doing_multiple = false
    end

    
    def show_plot_by(plot)
      if @xaxis_by == 'mass'
          show_by_mass(plot)
      elsif @xaxis_by == 'tau'
          show_by_tau(plot)
      elsif @xaxis_by == 'logP'
          show_by_logP(plot)
      elsif @xaxis_by == 'log_column_depth'
          show_by_log_column_depth(plot)
      elsif @xaxis_by == 'logT'
          show_by_logT(plot)
      elsif @xaxis_by == 'radius'
          show_by_radius(plot)
      elsif @xaxis_by == 'logR'
          show_by_logR(plot)
      elsif @xaxis_by == 'logxm'
          show_by_logxm(plot)
      elsif @xaxis_by == 'logxq'
          show_by_logxq(plot)
      else
          show_by_grid(plot)
      end
    end

end
