# profile_data.rb


load $MESA_DIR + "/utils/image_plot.rb"


class ProfileData

    include Math
    include Tioga
    include FigureConstants
    
    attr_reader :columns, :column_names
    attr_reader :msolar, :rsolar, :lsolar
    
    
    # scalar values from file

    attr_accessor :model_number
    attr_accessor :num_zones
    attr_accessor :initial_mass
    attr_accessor :initial_z
    attr_accessor :star_age
    attr_accessor :time_step
    attr_accessor :Teff
    attr_accessor :photosphere_L
    attr_accessor :photosphere_r
    attr_accessor :photosphere_mass
    attr_accessor :photosphere_lgT
    attr_accessor :photosphere_lgP
    attr_accessor :photosphere_lgRho
    attr_accessor :log_surface_L
    attr_accessor :log_surface_radius
    attr_accessor :log_surface_temp
    attr_accessor :log_center_temp
    attr_accessor :log_center_density
    attr_accessor :log_center_P
    attr_accessor :mixing_type, :conv_mixing_type
    attr_accessor :center_eta
    attr_accessor :center_h1
    attr_accessor :center_he3
    attr_accessor :center_he4
    attr_accessor :center_c12
    attr_accessor :center_n14
    attr_accessor :center_o16
    attr_accessor :center_ne20
    attr_accessor :star_mass
    attr_accessor :star_mdot
    attr_accessor :star_mass_h1
    attr_accessor :star_mass_he3
    attr_accessor :star_mass_he4
    attr_accessor :star_mass_c12
    attr_accessor :star_mass_n14
    attr_accessor :star_mass_o16
    attr_accessor :star_mass_ne20
    attr_accessor :h1_boundary_mass
    attr_accessor :he4_boundary_mass
    attr_accessor :c12_boundary_mass
    attr_accessor :tau10_mass
    attr_accessor :tau10_radius
    attr_accessor :tau100_mass
    attr_accessor :tau100_radius
    attr_accessor :dynamic_time
    attr_accessor :kh_timescale
    attr_accessor :nuc_timescale
    attr_accessor :power_nuc_burn
    attr_accessor :power_h_burn
    attr_accessor :power_he_burn
    attr_accessor :power_c_burn
    attr_accessor :power_neu
    attr_accessor :power_pp
    attr_accessor :power_cno
    attr_accessor :power_3_alpha
    attr_accessor :power_c_alpha
    attr_accessor :power_n_alpha
    attr_accessor :power_o_alpha
    attr_accessor :power_alpha
    attr_accessor :power_cc
    attr_accessor :power_co
    attr_accessor :power_oo
    attr_accessor :power_rp
    attr_accessor :power_photo
    attr_accessor :power_plas_neu
    attr_accessor :power_brem_neu
    attr_accessor :power_pair_neu
    attr_accessor :power_photo_neu
    attr_accessor :h1_boundary_limit
    attr_accessor :he4_boundary_limit
    attr_accessor :c12_boundary_limit
    attr_accessor :burn_min1
    attr_accessor :burn_min2
    attr_accessor :velocity_km_per_sec


    attr_accessor :h1
    attr_accessor :h2
    attr_accessor :he3
    attr_accessor :he4
    attr_accessor :li7
    attr_accessor :be7
    attr_accessor :b8
    attr_accessor :c12
    attr_accessor :c13
    attr_accessor :c14
    attr_accessor :n13
    attr_accessor :n14
    attr_accessor :n15
    attr_accessor :o14
    attr_accessor :o15
    attr_accessor :o16
    attr_accessor :o17
    attr_accessor :o18
    attr_accessor :f17
    attr_accessor :f18
    attr_accessor :f19
    attr_accessor :ne18
    attr_accessor :ne19
    attr_accessor :ne20
    attr_accessor :ne21
    attr_accessor :ne22
    attr_accessor :na23
    attr_accessor :mg22
    attr_accessor :mg24
    attr_accessor :mg25
    attr_accessor :mg26
    attr_accessor :al26
    attr_accessor :al27
    attr_accessor :si28
    attr_accessor :si29
    attr_accessor :si30
    attr_accessor :p30
    attr_accessor :p31
    attr_accessor :s30
    attr_accessor :s31
    attr_accessor :s32
    attr_accessor :s33
    attr_accessor :s34
    attr_accessor :s36
    attr_accessor :cl35
    attr_accessor :cl36
    attr_accessor :cl37
    attr_accessor :ar36
    attr_accessor :ar37
    attr_accessor :ar38
    attr_accessor :ar40
    attr_accessor :k39
    attr_accessor :ca40
    attr_accessor :ca41
    attr_accessor :ca42
    attr_accessor :ca44
    attr_accessor :ca46
    attr_accessor :ca48
    attr_accessor :sc43
    attr_accessor :sc44
    attr_accessor :sc45
    attr_accessor :ti44
    attr_accessor :ti45
    attr_accessor :ti46
    attr_accessor :ti48
    attr_accessor :ti50
    attr_accessor :ti52
    attr_accessor :v47
    attr_accessor :v51
    
    attr_accessor :cr48
    attr_accessor :cr49
    attr_accessor :cr50
    attr_accessor :cr51
    attr_accessor :cr52
    attr_accessor :cr53
    attr_accessor :cr54
    attr_accessor :cr55
    attr_accessor :cr56
    
    attr_accessor :mn51
    attr_accessor :mn52
    attr_accessor :mn53
    attr_accessor :mn54
    attr_accessor :mn55
    attr_accessor :mn56
    attr_accessor :mn57
    
    attr_accessor :fe52
    attr_accessor :fe53
    attr_accessor :fe54
    attr_accessor :fe55
    attr_accessor :fe56
    attr_accessor :fe57
    attr_accessor :fe58
    attr_accessor :fe59
    attr_accessor :fe60
    attr_accessor :fe61
    attr_accessor :fe62
    attr_accessor :fe64
    attr_accessor :fe66
    attr_accessor :fe68
    attr_accessor :fe70
    
    attr_accessor :co55
    attr_accessor :co56
    attr_accessor :co57
    attr_accessor :co58
    attr_accessor :co59
    attr_accessor :co60
    attr_accessor :co61
    attr_accessor :co62
    
    attr_accessor :ni53
    attr_accessor :ni55
    attr_accessor :ni56
    attr_accessor :ni57
    attr_accessor :ni58
    attr_accessor :ni59
    attr_accessor :ni60
    attr_accessor :ni61
    attr_accessor :ni62
    attr_accessor :ni64
    attr_accessor :ni66
    
    attr_accessor :zn60
    attr_accessor :zn62
    attr_accessor :zn64
    attr_accessor :zn66
    attr_accessor :zn68
    attr_accessor :zn70
    
    attr_accessor :ge64
    attr_accessor :ge72
    attr_accessor :se68
    attr_accessor :kr72
    attr_accessor :sr76
    attr_accessor :sn104
    attr_accessor :neut
    attr_accessor :prot
    
    attr_accessor :pp
    attr_accessor :cno
    attr_accessor :tri_alfa
    
    attr_accessor :burn_c
    attr_accessor :burn_n
    attr_accessor :burn_o
    attr_accessor :burn_ne
    attr_accessor :burn_na
    attr_accessor :burn_mg
    attr_accessor :burn_si
    attr_accessor :burn_s
    attr_accessor :burn_ar
    attr_accessor :burn_ca
    attr_accessor :burn_ti
    attr_accessor :burn_cr
    attr_accessor :burn_fe
    
    attr_accessor :c12_c12
    attr_accessor :c12_o16
    attr_accessor :o16_o16
    
    attr_accessor :pnhe4
    attr_accessor :photo
    attr_accessor :other

    attr_accessor :non_nuc_neu
    attr_accessor :nonnucneu_plas
    attr_accessor :nonnucneu_brem
    attr_accessor :nonnucneu_phot
    attr_accessor :nonnucneu_pair
    attr_accessor :nonnucneu_reco

    attr_accessor :ye

    attr_accessor :dL_dm
    attr_accessor :eps_nuc
    attr_accessor :logT
    attr_accessor :logRho

    attr_accessor :logL
    attr_accessor :luminosity

    attr_accessor :tau
    attr_accessor :logtau

    attr_accessor :cp
    attr_accessor :cv
    attr_accessor :grada
    attr_accessor :gradT
    attr_accessor :gradr
    
    attr_accessor :v_div_r
    
    attr_accessor :eps_grav
    attr_accessor :env_eps_grav
    attr_accessor :net_nuclear_energy

    attr_accessor :edv_h1
    attr_accessor :edv_he4
    attr_accessor :edv_c12
    attr_accessor :edv_n14
    attr_accessor :edv_o16
    attr_accessor :edv_mg24

    
    attr_accessor :grid
    attr_accessor :zone
    attr_accessor :q
    attr_accessor :logP
    attr_accessor :logR
    attr_accessor :log_radial_depth
    attr_accessor :log_column_depth
    attr_accessor :logxq
    attr_accessor :logxm
    attr_accessor :mass
    attr_accessor :mmid
    attr_accessor :radius
    attr_accessor :r_div_R
    attr_accessor :acoustic_radius
    attr_accessor :acoustic_r_div_R_phot
    attr_accessor :csound
    attr_accessor :cs_at_cell_bdy
    
    
    attr_accessor :brunt_N_dimensionless
    attr_accessor :brunt_frequency
    attr_accessor :log_brunt_N2
    attr_accessor :log_brunt_nu
    attr_accessor :log_lamb_Sl1
    attr_accessor :log_lamb_Sl2
    attr_accessor :log_lamb_Sl3
    attr_accessor :log_lamb_Sl10
    
    attr_accessor :brunt_N2
    attr_accessor :brunt_N2_structure_term
    attr_accessor :brunt_N2_composition_term
    
    attr_accessor :brunt_nu
    attr_accessor :lamb_Sl1
    attr_accessor :lamb_Sl2
    attr_accessor :lamb_Sl3
    attr_accessor :lamb_Sl10
    
    attr_accessor :log_J_inside
    attr_accessor :log_j_rot
    attr_accessor :log_i_rot
    attr_accessor :i_rot
    
    attr_accessor :omega
    attr_accessor :dynamo_log_B_phi
    attr_accessor :dynamo_log_B_r
    attr_accessor :log_abs_shear
    attr_accessor :v_rot
        
    attr_accessor :log_D_mix, :log_D_conv, :log_D_semi, :log_D_ovr, :log_D_anon
    attr_accessor :log_D_th, :log_D_minimum
    attr_accessor :am_log_D_visc, :am_log_D_DSI, :am_log_D_SH
    attr_accessor :am_log_D_SSI, :am_log_D_ES, :am_log_D_GSF, :am_log_D_ST
    
    attr_accessor :star_age_str, :initial_z_str, :initial_mass_str
    attr_accessor :opacity_max, :entropy_max, :luminosity_max, :energy_max, :density_max, :temp_max
    attr_accessor :shell_with_max_temp, :first_shell, :last_shell, :surface_shell, :center_shell
    
    def read(filename, logxq_cutoff) # for mesa/star profiles
      f = File.open(filename)
      f.readline # skip line 1
      val_names = f.readline.split # names of scalars in line 2
      f.readline # skip line 3
      f.readline # skip line 4
      f.readline # skip line 5
      vec_names = f.readline.split # names of vectors in line 6
      f.close

      @velocity = nil
      @velocity_km_per_sec = nil
      @dr_dt = nil
      
      vals = Dvector.read_row(filename,3) # scalar values are in line 3
      assign_to_names(val_names,vals,false)
      
      @column_names = vec_names
      @columns = Array.new(vec_names.size) { Dvector.new(@num_zones) }
      Dvector.read(filename,@columns,7,@num_zones) # vector values start on line 7
      assign_to_names(vec_names,@columns,false)
      set_derived_values(logxq_cutoff)
      
      #puts "columns size #{@columns.size}"
      save_FYI_track_info
      
    end
     
    
    def save_FYI_track_info
      return
      puts "write FYI track info"
      filename = 'fyi.data'
      f = File.open(filename,"w")
      n = @mass.length
      brunt_N = @log_brunt_N2.dup.div(2).exp10
      brunt_integral = @mass.dup
      ln10 = log(10)
      n.times { |i|
         j = n-i-1
         if i == 0
            brunt_integral[j] = 0
         else
            #puts "i #{i} j #{j}  n #{n}"
            #puts "brunt_N[j] #{brunt_N[j]}"
            #puts "brunt_N[j+1] #{brunt_N[j+1]}"
            brunt_n_avg = 0.5*(brunt_N[j] + brunt_N[j+1])
            dlnR = (@logR[j] - @logR[j+1]) * ln10
            brunt_integral[j] = brunt_integral[j+1] + brunt_n_avg*dlnR
         end
      }
      n.times { |i|
          f.printf("%25.15e", @radius[i])
          #f.printf("%25.15e", @mass[i])
          #f.printf("%25.15e", 10.0**@logRho[i])
          #f.printf("%25.15e", 10.0**@logT[i])
          f.printf("%25.15e", brunt_integral[i])
          f.puts
      }
      f.close
    end

    def assign_to_names(names,vals,show_cmds)
      unless names.size == vals.size
         puts "mismatch in lengths for profile data assign to names"
         puts "names.size = #{names.size}"
         puts "vals.size = #{vals.size}"
         raise "error in reading profile"
      end
      cmd = ''
      begin
         names.size.times do |i|
            cmd = sprintf("@%s = vals[%i]",names[i],i)
            eval(cmd)
            puts "#{cmd}" if show_cmds
         end
      rescue
         puts "assign_to_names: all names must be declared as attributes for profile data"
         puts "failure for #{cmd}"
         raise "error in reading profile"
      end
    end

    
    def set_derived_values(logxq_cutoff)
      if @q == nil && @mass == nil
        puts "profile data needs to have either q or mass info"
      end
      @q = @mass.div(@star_mass) if @q == nil
      @mass = @q.mul(@star_mass) if @mass == nil

      @grid = @zone
      @logP = @pressure.safe_log10 if @logP == nil
      @pressure = @logP.exp10 if @pressure == nil
      @logR = @radius.safe_log10 if @logR == nil
      @radius = @logR.exp10 if @radius == nil
      @r_div_R = @radius.dup.div(@radius.max) if @r_div_R == nil
      @velocity_km_per_sec = @velocity.dup.div(1e5) unless @velocity == nil
      
      #puts "logxq_cutoff #{logxq_cutoff}"
      logxq_cutoff = -7 if logxq_cutoff == nil
      if @logxq == nil
          @xm = @mass.sub(@star_mass).neg if @xm == nil    
          cutoff = @xm.min_gt(10**logxq_cutoff)
          cutoff = 0 if cutoff == nil
          @logxq = @xm.dup.map! { |x| (x > cutoff)? x : cutoff }
          @logxq.safe_log10!
      else
          @logxq = @logxq.dup.map! { |x| (x > logxq_cutoff)? x : logxq_cutoff }
      end
      @logxm = @logxq.add(log10(@star_mass)) if @logxm == nil

      @msolar = 1.9892e33 # solar mass (g)
      @rsolar = 6.9598e10 # solar radius (cm)
      @lsolar = 3.8418e33 # solar luminosity (erg s^-1)
                  
      if @star_age > 1e9
         @star_age_str = sprintf("%0.3f (Gyr)", @star_age * 1e-9)
      elsif @star_age > 1e6
         @star_age_str = sprintf("%0.3f (Myr)", @star_age * 1e-6)
      elsif @star_age > 1e3
         @star_age_str = sprintf("%0.3f (Kyr)", @star_age * 1e-3)
      elsif @star_age > 1e-1
         @star_age_str = sprintf("%0.3f (yr)", @star_age)
      else
         secyer  = 3.1558149984e7
         age_secs = @star_age*secyer
         age_minutes = age_secs/60.0
         age_hours = age_minutes/60.0
         age_days = age_hours/24.0
         if age_days > 1
            @star_age_str = sprintf("%0.2f (days)", age_days)
         elsif age_hours > 1
            @star_age_str = sprintf("%0.2f (hours)", age_hours)
         elsif age_minutes > 1
            @star_age_str = sprintf("%0.2f (minutes)", age_minutes)
         else
            @star_age_str = sprintf("%0.3f (seconds)", age_secs)
        end
      end
      
      if @initial_mass >= 10 and @initial_mass == @initial_mass.floor
         @initial_mass_str = sprintf('$M_{init}$=%0.1f', @initial_mass)
      else
         @initial_mass_str = sprintf('$M_{init}$=%0.2f', @initial_mass)
      end

      if @initial_z >= 0.01
         @initial_z_str = sprintf('$Z_{init}$=%0.2f', @initial_z)
      elsif @initial_z >= 0.001
         @initial_z_str = sprintf('$Z_{init}$=%0.3f', @initial_z)
      elsif @initial_z >= 0.0001
         @initial_z_str = sprintf('$Z_{init}$=%0.4f', @initial_z)
      else
         @initial_z_str = sprintf('$Z_{init}$=%0.5g', @initial_z)
      end

      @first_shell = 0
      @last_shell = @radius.size - 1
      @surface_shell = @radius.where_max
      @center_shell = @radius.where_min
      
      if @logxq == nil
          xq = @q.sub(1).neg
          logxq_cutoff = -7 if logxq_cutoff == nil
          cutoff = xq.min_gt(10**logxq_cutoff)
          cutoff = 0 if cutoff == nil
          @logxq = xq.dup.map! { |x| (x > cutoff)? x : cutoff }
          @logxq.safe_log10!
      end

    end

end



class HR_History_Data

    include Math
    include Tioga
    include FigureConstants

    # file data
    attr_accessor :model_number, :star_age, :star_mass, :log_luminosity, :log_surface_temp
    attr_accessor :log_center_density, :log_center_temp, :log_surface_gravity
    
    # extras 
    attr_accessor :star_age_str

    def read_history(filename, mod_num)
      f = File.open(filename)
      f.readline # skip line 1
      names = f.readline.split # names in line 2
      f.close
      @dest = Array.new(names.size) { Dvector.new }
      Dvector.read(filename,@dest,3) # vector values start on line 3
      assign_to_names(names,@dest)
      # remove backups
      while true
         i = -1
         n = @model_number.size
         (n-1).times do |k|
            if @model_number[k] >= @model_number[k+1]
               i = k
               break
            end
         end
         break if i < 0
         @dest.each { |vec| vec.delete_at(i) }
      end
      # remove models beyond the current one
      k = @model_number.where_first_ge(mod_num)
      if k != nil
         @log_luminosity = @log_luminosity[0..k]
         @log_surface_temp = @log_surface_temp[0..k]
         @log_center_density = @log_center_density[0..k]
         @log_center_temp = @log_center_temp[0..k]
         @log_surface_gravity = @log_surface_gravity[0..k]
      end
      set_derived_values
    end


    def assign_to_names(names,vals)
      unless names.size == vals.size
         puts "mismatch in lengths for profile data assign to names"
         puts "names.size = #{names.size}"
         puts "vals.size = #{vals.size}"
         raise "error in reading profile"
      end
      cmd = ''
      begin
         names.size.times do |i|
            cmd = sprintf("@%s = vals[%i]",names[i],i)
            eval(cmd)
         end
      rescue
         puts "assign_to_names: all names must be declared as attributes for profile data"
         puts "failure for #{cmd}"
         raise "error in reading profile"
      end
    end

    
    def set_derived_values
        age = @star_age[-1]
        if age > 1e9
            @star_age_str = sprintf("%0.3f (Gyr)", age * 1e-9)
        elsif age > 1e6
            @star_age_str = sprintf("%0.3f (Myr)", age * 1e-6)
        else
            @star_age_str = sprintf("%0.3f (Kyr)", age * 1e-3)
        end
    end
    
end
