# history.rb

$MESA_DIR = '../../..' if $MESA_DIR == nil

class StarHistory

    include Math
    include Tioga
    include FigureConstants
    
    attr_reader :age_Units, :age_Scale, :age_String, :mesh_lines_flag
    
    def initialize(dict)
        
        which = dict['which']
        @filenames = dict['filenames']
        if which == nil
            if @filenames == nil
                which = 'all'
            else
                which = 'none'
            end
        end

        @logs_index = dict['logs_index']
        @logs_index = 0 if @logs_index == nil

        @log_dir = dict['log_dir']
        @log_dir = $MESA_DIR + '/star/test/LOGS' if @log_dir == nil
        
        @xaxis = dict['xaxis']
        @xaxis = 'by_step' if @xaxis == nil
        
        # discard log entries with model_number < first_model
        @first_model = dict['first_model']
        @first_model = 0 if @first_model == nil
        
        # discard log entries with model_number > last_model
        # if last_model == 0, then ignore this parameter.
        # if last_model < 0, then last_model = model_number.max + last_model
        @last_model = dict['last_model']
        @last_model = 0 if @last_model == nil
        
        # shift all ages by this amount (in years)
        @age_shift = dict['age_shift']
        @age_shift = 0 if @age_shift == nil
        
        # discard log entries for models with ages < first_age
        # first_age = -1 means use first log to set first_age
        @first_age = dict['first_age']
        @first_age = -1 if @first_age == nil
        
        # discard log entries for models with ages > last_age
        # last_age = -1 means use final log to set last_age
        @last_age = dict['last_age']
        @last_age = -1 if @last_age == nil
        
        # revise last_age by subtracting discard_span
        @discard_span = dict['discard_span']
        @discard_span = 0 if @discard_span == nil
        
        # discard log entries for model with ages < last_age - age_span
        # age_span = -1 means set age_span = last_age
        @age_span = dict['age_span']
        @age_span = -1 if @age_span == nil
        
        # prune set of models if there are more than max_models
        # max_models = -1 mean there is no max.
        @max_models = dict['max_models']
        @max_models = -1 if @max_models == nil
        
        @max_mass = dict['max_mass']
        @min_mass = dict['min_mass']
        
        # if need to prune models because exceed max_models,
        # start pruning by discarding models with
        # model number < previous model number + log_cnt.
        @log_cnt = dict['log_cnt']
        @log_cnt = -1 if @log_cnt == nil
        
        # if still need to discard models after have pruned using log_cnt,
        # then discard the models with the smallest model_numbers until
        # only have max_models remaining.
        
        @legend_left_margin = dict['legend_left_margin']
        @legend_top_margin = dict['legend_top_margin']
        @legend_right = dict['legend_right']

        @kipp_legend = dict['kipp_legend']
        @kipp_legend = false if @kipp_legend == nil

        @kipp_xright_for_by_lg_ybp = dict['kipp_xright_for_by_lg_ybp']
        @kipp_xright_for_by_lg_ybp = -4 if @kipp_xright_for_by_lg_ybp == nil

        @kipp_xleft_for_by_lg_ybp = dict['kipp_xleft_for_by_lg_ybp']

        @kipp_xright_for_by_lg_sbp = dict['kipp_xright_for_by_lg_sbp']
        @kipp_xright_for_by_lg_sbp = 6 if @kipp_xright_for_by_lg_sbp == nil

        @kipp_xleft_for_by_lg_sbp = dict['kipp_xleft_for_by_lg_sbp']
        
        @num_abundance_line_labels = dict['num_abundance_line_labels']
        @num_abundance_line_labels = 9 if @num_abundance_line_labels == nil

        @labels_T_RHO = dict['labels_T_RHO']
        @labels_T_RHO_dx = dict['labels_T_RHO_dx']
        @labels_T_RHO_dy = dict['labels_T_RHO_dy']

        @labels_HR = dict['labels_HR']
        @labels_HR_dx = dict['labels_HR_dx']
        @labels_HR_dy = dict['labels_HR_dy']
        

        @conv_color = dict['conv_color']
        @conv_color = LightSkyBlue if @conv_color == nil

        @conv_width = dict['conv_width']
        @conv_width = 2 if @conv_width == nil
        
        @overshoot_color = dict['overshoot_color']
        @overshoot_color = @conv_color if @overshoot_color == 'conv_color'
        @overshoot_color = Tan if @overshoot_color == nil

        @overshoot_width = dict['overshoot_width']
        @overshoot_width = 2 if @overshoot_width == nil
        
        @semiconvection_color = dict['semiconvection_color']
        @semiconvection_color = @conv_color if @semiconvection_color == 'conv_color'
        @semiconvection_color = SlateGray if @semiconvection_color == nil

        @semiconvection_width = dict['semiconvection_width']
        @semiconvection_width = 2 if @semiconvection_width == nil
        
        @thermohaline_color = dict['thermohaline_color']
        @thermohaline_color = @conv_color if @thermohaline_color == 'conv_color'
        @thermohaline_color = Lilac if @thermohaline_color == nil

        @thermohaline_width = dict['thermohaline_width']
        @thermohaline_width = 2 if @thermohaline_width == nil

        
        @rotation_color = dict['rotation_color']
        @rotation_color = @conv_color if @rotation_color == 'conv_color'
        @rotation_color = Teal if @rotation_color == nil

        @rotation_width = dict['rotation_width']
        @rotation_width = 2 if @rotation_width == nil
        
        @force_ymin = dict['force_ymin']
        @force_ymax = dict['force_ymax']            
        
        @conv_logxm_yaxis = dict['conv_logxm_yaxis']
        
        @plot_stroke_color = dict['plot_stroke_color']
        @plot_stroke_color = Black if @plot_stroke_color == nil
        
        @no_frills = dict['no_frills']

        @ymin_TRHO = dict['ymin_TRHO']
        @ymax_TRHO = dict['ymax_TRHO']
        @xmin_TRHO = dict['xmin_TRHO']
        @xmax_TRHO = dict['xmax_TRHO']

        @ymin_HRD = dict['ymin_HRD']
        @ymax_HRD = dict['ymax_HRD']
        @xmin_HRD = dict['xmin_HRD']
        @xmax_HRD = dict['xmax_HRD']
        
        @mini_log_names = dict['mini_log_names']

        @burn_max = 20
        @burn_min = -20

        @figure_maker = FigureMaker.default

        t.def_eval_function { |str| eval(str) }

        @have_setup_data = false
        setup_data 
        
        unless which == 'none'

            t.def_figure("H_R") { plot_H_R }
            t.def_figure("H_R_2") { plot_H_R_2 }
            t.def_figure("H_R_3") { plot_H_R_3 }
            t.def_figure("T_Rho") { plot_T_RHO }

            #t.def_figure("H_R_RGB") { plot_H_R_RGB }

            t.def_figure("Internal_Structure_lite") { internal_structure_lite }
            t.def_figure("Internal_no_burn_lite") { internal_no_burn_lite }
        
            t.def_figure("H_R_T_Rho") { plot_H_R_T_RHO }
            t.def_figure("H_R_2_T_Rho") { plot_H_R_2_T_RHO }
            t.def_figure("H_R_3_T_Rho") { plot_H_R_3_T_RHO }
            t.def_figure("lgg_lgTeff") { plot_lgg_lgTeff }
            t.def_figure("lgg_Teff") { plot_lgg_Teff }
            t.def_figure("surf_He4_logL") { plot_surf_He4_logL }
            t.def_figure("core_mass_luminosity") { plot_core_mass_luminosity }
            t.def_figure("luminosity_Tcentral") { plot_luminosity_Tcentral }
            t.def_figure("lgTeff_central_He") { plot_lgTeff_central_He }
            
            if d.delta_nu != nil
               t.def_figure("nu_max_Teff") { plot_nu_max_Teff }
               t.def_figure("delta_Pg_delta_nu") { plot_delta_Pg_delta_nu }
               t.def_figure("nu_max_delta_nu") { plot_nu_max_delta_nu }
               t.def_figure("delta_wkb_delta_nu") { plot_delta_wkb_delta_nu }
               t.def_figure("log_inertia_ratio_delta_nu") { log_inertia_ratio_delta_nu }
               t.def_figure("delta_log_Pg_delta_nu") { plot_log_delta_Pg_delta_nu }
               t.def_figure("nu_max_34_div_delta_nu_vs_delta_nu") { plot_nu_max_34_div_delta_nu_vs_delta_nu }
               t.def_figure("luminosity_kr_integral") { plot_luminosity_kr_integral }
               t.def_figure("l0_modes") { plot_l0_modes } if d.l0_model_1 != nil
               t.def_figure("l1_modes") { plot_l1_modes } if d.l1_model_1 != nil
               t.def_figure("l2_modes") { plot_l2_modes } if d.l2_model_1 != nil
               t.def_figure("l1_l0_offset") { plot_l1_l0_offset } if d.l1_model_1 != nil
               t.def_figure("avg_delta_nu_offset") { plot_avg_delta_nu_offset } if d.avg_delta_nu_model != nil
               t.def_figure("Internal_Structure_Pulse_Info") { internal_structure_pulse_info }
            end 

            t.def_figure("Internal_Structure") { internal_structure }
            t.def_figure("Internal_Structure_Lars") { internal_structure_lars }
            t.def_figure("Internal_no_burn") { internal_no_burn }

            t.def_figure("kipp_burn_mix") { kippenhahn_burn_mix }
            t.def_figure("kipp_no_burn") { kippenhahn_no_burn }
            t.def_figure("kipp_no_mix") { kippenhahn_no_mix }
            t.def_figure("kipp_mix_only") { kippenhahn_mix_only }
            t.def_figure("kipp_abundances") { kippenhahn_abundances }
            t.def_figure("kipp_no_abund") { kippenhahn_no_abundances }
            t.def_figure("kipp_burn_mix_no_abund") { kippenhahn_burn_mix_no_abundances }
            t.def_figure("kipp_burn_mix_simple") { kippenhahn_burn_mix_simple }
            t.def_figure("kipp_luminosities") { kippenhahn_luminosities }

            t.def_figure("teff_L") { teff_L }
            t.def_figure("teff_L_T_Rho") { teff_L_T_Rho }
        
        
        
            t.def_figure("lgLHe") { lgLHe }
            t.def_figure("logLs") { logLs }
            t.def_figure("min_log_L_Ledd") { min_log_L_Ledd }

            t.def_figure("center_entropy_ye") { center_entropy_ye }

            t.def_figure("code3") { code3 }

            t.def_figure("deltaM_Teff_L") { deltaM_Teff_L }

            t.def_figure("dt_nz") { dt_nz }
            t.def_figure("dt_nz_retries") { dt_nz_retries }
            t.def_figure("dt_nz_LHe") { dt_nz_LHe }
            t.def_figure("dt_nz_L") { dt_nz_L }
            t.def_figure("dt_nz_retries_L") { dt_nz_retries_L }
        
            t.def_figure("dt_min_Pgas_div_P") { dt_min_Pgas_div_P } if d.min_Pgas_div_P != nil
        

            t.def_figure("kipp_dt_nz_retries") { kipp_dt_nz_retries }
            t.def_figure("kipp_dt_nz") { kipp_dt_nz }
        
            t.def_figure("logT_logRho_logLneu") { logT_logRho_logLneu }
            t.def_figure("log_L_info") { log_L_info }

            t.def_figure("cz_max_abundances") { plot_cz_max_abundances }
            t.def_figure("cz_dt_dlnTbase_vs_Tbase") { plot_cz_dt_dlnTbase }
            t.def_figure("cz_dt_dlnTbase") { cz_dt_dlnTbase }

            t.def_figure("max_eps_nuc_dt_dlnT") { max_eps_nuc_dt_dlnT }

            t.def_figure('plot_rlo_Porb_mdot') { plot_rlo_Porb_mdot }
            t.def_figure('plot_rlo_Porb_mass') { plot_rlo_Porb_mass }
            t.def_figure('plot_rlo_R_vs_M') { plot_rlo_R_vs_M }


            t.def_figure('gradT_excess_track') { 
               plot_gradT_excess_track } if 
                  (d.gradT_excess_min_beta != nil) && (d.gradT_excess_max_lambda != nil)

        end
        
        save_dir = dict['save_dir']
        save_dir = 'history_out' if save_dir == nil
        t.save_dir = save_dir

        @margin = 0.1
        @xreversed = (@xaxis == 'by_lg_ybp') || (@xaxis == 'by_lg_sbp') || 
                     (@xaxis == 'by_center_h1')
                
        t.def_enter_page_function { enter_page }
        
        t.tex_preamble = t.tex_preamble + "\n\t\\usepackage{txfonts}\n"
        

        @atol = 1e-30
        @rtol = 1e-8
        d = @logs[@logs_index]
        @num_columns = d.columns.length
        @plot_info = Array.new(@num_columns)
        @column_dict = Hash.new 
        @num_columns.times {|i|
            @column_dict[d.column_names[i]] = d.columns[i]
            get_min_max_for_plot(i)
            @plot_info[i] = [d.column_names[i], @plot_ymin, @plot_ymax]
        }
        skip = [
              'conv_mx1_top',
              'conv_mx1_bot',
              'conv_mx2_top',
              'conv_mx2_bot',
              'mx1_top',
              'mx1_bot',
              'mx2_top',
              'mx2_bot',
              'epsnuc_M_1',
              'epsnuc_M_2',
              'epsnuc_M_3',
              'epsnuc_M_4',
              'epsnuc_M_5',
              'epsnuc_M_6',
              'epsnuc_M_7',
              'epsnuc_M_8'
            ]

        @num_columns.times do |i|
            unless skip.include?(d.column_names[i]) || d.column_names[i][0,4] == 'mix_' || 
                    d.column_names[i][0,5] == 'burn_'
               name = @plot_info[i][0]
               if which == 'all' || which == name
                  eval(sprintf("t.def_figure(\"%s\") { plot_column %i }", name, i))
               end
            end
        end
        
        @ymin_dict = Hash.new
        @ymax_dict = Hash.new
        
        fyi = $MESA_DIR + '/data/star_data/plot_info/'
        Dvector.read(fyi +'gamma_4_thirds.data', 
               @gamma_4_thirds = [Dvector.new, Dvector.new])

        @image_right_margin = 0.05
        
        set_track_for_plots
        k = d.log_LH.sub(d.log_L).where_first_gt(0)
        puts "LH > L at model #{d.model_number[k].to_i}" unless k == nil
            
    end



    def enter_page
        t.page_setup(11*72/2,8.5*72/2)
        t.set_frame_sides(0.15,0.85,0.85,0.15) # left, right, top, bottom in page coords        
        t.rescale(1.3)
    end


    def d
        @logs[@logs_index]
    end
    
    
    def get_min_max_for_plot(n)
        ys = get_ys_for_plot(n)
        if ys == nil
           puts "failed to find ys for column #{n}  #{d.column_names[n]}"
           return
        end
        ymin = ys.min
        ymin = ys.min_gt(-99) if ymin == -99
        ymin = -99 if ymin == nil
        ymax = ys.max
        dy = ymax - ymin
        ytol = [ymin.abs, ymax.abs].max
        dy = ytol * @rtol + @atol if dy < ytol * @rtol + @atol
        ymargin = 0.02
        @plot_ymax = ymax + ymargin * dy
        @plot_ymin = ymin - ymargin * dy
        @plot_ymax = @force_ymax unless @force_ymax == nil
        @plot_ymin = @force_ymin unless @force_ymin == nil
    end
    
    
    def get_ys_for_plot(col_num)
      return d.columns[col_num]
    end
    


    def plot_column(n)    
       #puts "plot column #{n}"
       ys = get_ys_for_plot(n)
       xreversed = false
       plot_results(n, ys[@track_first .. @track_last], d.column_names[n], xreversed)
    end
    
    def get_by_name(name)
       n = d.column_names.index(name)
       ys = get_ys_for_plot(n)
       ys[@track_first .. @track_last]
    end
    
    def get_ys_by_name(name)
       n = d.column_names.index(name)
       ys = get_ys_for_plot(n)
       ys[@track_first .. @track_last]
    end


    def plot_column_by_name(name, ylabel, xreversed = false)   
       n = d.column_names.index(name)
       ys = get_ys_for_plot(n)
       plot_results(n, ys, ylabel, xreversed)
    end
    
    
    def plot_results(n, ys, ylabel, xreversed)
        plt = @plot_info[n]
        title = plt[0].tr("_", " ") + 
            sprintf("  \\scriptsize (M %0.3g)", d.star_mass[-1])
        ymin = @ymin_dict[plt[0]]
        ymin = plt[1] if ymin == nil
        ymax = @ymax_dict[plt[0]]
        ymax = plt[2] if ymax == nil
        ylabel = ylabel.tr("_", " ") if ylabel == plt[0]
        plot_data(title, ylabel, ymin, ymax, ys)
    end
    
    
    def plot_data(title, ylabel, ymin, ymax, ys, post_decorate=nil, pre_decorate=nil)
        xs = @xaxis_data[@track_first .. @track_last]
        t.rescale(0.8)
        t.set_subframe('right_margin' => 0.12, 'left_margin' => 0.0)
        background
        #t.show_title(title)
        t.show_ylabel(ylabel)
        t.show_xlabel(@xaxis_label)
        ymargin = 0.1
        ythresh = @atol
        height = (ymax < ymin+ythresh)? ythresh : ymax - ymin
        top_boundary = ymax + ymargin * height
        bottom_boundary = ymin - ymargin * height
        boundaries = [ @xaxis_left, @xaxis_right, top_boundary, bottom_boundary ]
        t.show_plot(boundaries) { 
            eval(pre_decorate) unless pre_decorate == nil
            t.show_polyline(xs,ys,@plot_stroke_color)
            eval(post_decorate) unless post_decorate == nil
        }        
    end


    def plot_logs_column(
            n, ylabel = nil, show_mass = true, decorate_line = nil, legend_proc = nil,
            xmin = nil, xmax = nil, ymin = nil, ymax = nil, 
            exp_ys = false, max_rel_ys = false, log_ys = false, xaxis = nil)    
        xs_ary = []
        ys_ary = []
        xaxis = @xaxis if xaxis == nil
        @logs.each { |d| 
            if xaxis == 'by_period_hr'
                xs_ary << d.period_hr
            elsif xaxis == 'by_age'
                xs_ary << d.star_age
            else
                xs_ary << d.lg_age
            end
            if exp_ys
                ys_ary << d.columns[n].exp10
            elsif log_ys
                ys_ary << d.columns[n].log10
            else
                ys_ary << d.columns[n]
            end
            if max_rel_ys
                ys_ary[-1] = ys_ary[-1].sub(ys_ary[-1].max)
            end
        }
        xreversed = false
        ylabel = @logs[0].column_names[n] if ylabel == nil
        t.rescale(0.8)
        t.set_subframe('right_margin' => 0.12, 'left_margin' => 0.0)
        background
        plt = @plot_info[n]
        title = plt[0]
        #t.show_title(title)
        t.show_ylabel(ylabel)#.tr("_", " "))
        t.show_xlabel(@xaxis_label)
        xmax = max_of_many(xs_ary) if xmax == nil
        xmin = min_of_many(xs_ary) if xmin == nil
        ymax = @force_ymax if ymax == nil
        ymin = @force_ymin if ymin == nil
        ymax = max_of_many(ys_ary) if ymax == nil
        ymin = min_of_many(ys_ary) if ymin == nil
        ymargin = 0.05
        ythresh = @atol
        height = (ymax < ymin+ythresh)? ythresh : ymax - ymin
        top_boundary = ymax + ymargin * height
        bottom_boundary = ymin - ymargin * height
        boundaries = [ xmin, xmax, top_boundary, bottom_boundary ]
        x_age = xmax
        n = xs_ary.length
        t.show_plot(boundaries) { 
            n.times { |i| 
                xs = xs_ary[i]
                ys = ys_ary[i]
                color = Black
                t.line_width = 0.6
                t.show_polyline(xs,ys,color) 
                eval(sprintf("%s(%i, %i)", decorate_line, i, n)) unless decorate_line == nil
            }
            if show_mass
                n.times { |i| 
                    xs = xs_ary[i]
                    ys = ys_ary[i]
                    k = xs.where_first_closest(x_age)
                    dy = t.default_text_height_dy*0.0
                    d = @logs[i]
                    mass = d.star_mass[-1]
                    t.show_text(
                        'x' => x_age+0.08, 'y' => ys[k]+dy, 'scale' => 1, 
                        'alignment' => ALIGNED_AT_BASELINE, 
                        'justification' => LEFT_JUSTIFIED,
                        'text' => sprintf("\\scriptsize %0.3f",mass)) 
                }
            end
            eval(legend_proc) unless legend_proc == nil
        }      
    end
    
    
    def t
        @figure_maker
    end

    def read_PSIs(path = nil)
        path = '' if path == nil
        path = path + '/' if path.length > 0 && path[-1..-1] != '/'
        Dvector.read(path + 'psi4.data', @psi4 = [Dvector.new, Dvector.new])
    end
    
    def read_ZAMS(path = nil)
        path = '' if path == nil
        path = path + '/' if path.length > 0 && path[-1..-1] != '/'
        @zams = [
            @zams_log_Center_Density = Dvector.new,
            @zams_log_Center_Temp = Dvector.new,
            @zams_log_Luminosity = Dvector.new,
            @zams_log_Surface_Temp = Dvector.new,
            @zams_Ms = Dvector.new ]
        Dvector.read(path + 'ZAMS.data', @zams, 2)
    end
    
    def setup_data
        return if @have_setup_data 
        
        if @filenames == nil
            @num_logs = 1
            @logs = []
            filename = @log_dir + '/history.data'
            @logs << HistoryLogs.new(
                    filename, @first_model, @last_model, 
                    @age_shift, @first_age, @last_age, 
                    @discard_span, @age_span, @max_models, @log_cnt, @xaxis, @mini_log_names)
        else
            @num_logs = @filenames.length
            @logs = []
            @filenames.each { |filename| @logs << HistoryLogs.new(
                    filename, @first_model, @last_model, 
                    @age_shift, @first_age, @last_age, 
                    @discard_span, @age_span, @max_models, @log_cnt, @xaxis, @mini_log_names) }
        end       
                
        data_dir = $MESA_DIR + '/star/test/star_history/star_data'
        read_PSIs(data_dir)
        read_ZAMS(data_dir)
        adjust_ages
        find_He_gap
        if @xaxis == 'by_lg_ybp'
           @xaxis_data = d.lg_ybp
           @xaxis_label = 'log(years before end of run)'
        elsif @xaxis == 'by_lg_sbp'
           @xaxis_data = d.lg_sbp
           @xaxis_label = 'log(seconds before end of run)'
        elsif @xaxis == 'by_step'
           @xaxis_data = d.model_number
           @xaxis_label = 'model number'
        elsif @xaxis == 'by_lg_age'
           @xaxis_data = d.lg_age
           @xaxis_label = 'log age (yrs)'
        elsif @xaxis == 'by_lg_yrs_post_flash'
           @xaxis_data = d.lg_age
           @xaxis_label = 'lg years post flash'
        elsif @xaxis == 'by_center_h1'
           @xaxis_data = d.center_h1
           @xaxis_label = 'Center H Mass Fraction'
        else
           @xaxis_data = d.star_age
           @xaxis_label = @age_xlabel
        end
        @have_setup_data = true
    end
    
    
    def adjust_ages
        age = d.star_age[-1]
        if age >= 1e9
            @age_Units = "Gyrs"
            @age_Scale = 1e9
            @age_xlabel = '$10^9$ years'
            @age_String = '$10^9$'
        elsif age >= 1e6
            @age_Units = "Myrs"
            @age_Scale = 1e6
            @age_xlabel = '$10^6$ years'
            @age_String = '$10^6$'
        elsif age >= 1e5
            @age_Units = '$10^5$ years'
            @age_Scale = 1e5
            @age_xlabel = '$10^5$ years'
            @age_String = '$10^5$'
        elsif age >= 1e4  && false
            @age_Units = '$10^4$ years'
            @age_Scale = 1e4
            @age_xlabel = '$10^4$ years'
            @age_String = '$10^4$'
        elsif age >= 1e3
            @age_Units = "Kyrs"
            @age_Scale = 1e3
            @age_xlabel = '$10^3$ years'
            @age_String = '$10^3$'
        elsif age >= 1
            @age_Units = "yrs"
            @age_Scale = 1
            @age_xlabel = 'years'
            @age_String = ''
        elsif age >= 1.0/365.25
            @age_Units = "days"
            @age_Scale = 1.0/365.25
            @age_xlabel = 'days'
            @age_String = ''
        elsif age >= 1.0/(365.25*24.0)
            @age_Units = "hours"
            @age_Scale = 1.0/(365.25*24.0)
            @age_xlabel = 'hours'
            @age_String = ''
        elsif age >= 1.0/(365.25*24.0*60)
            @age_Units = "minutes"
            @age_Scale = 1.0/(365.25*24.0*60)
            @age_xlabel = 'minutes'
            @age_String = ''
        else
            @age_Units = "seconds"
            @age_Scale = 1.0/(365.25*24.0*60*60)
            @age_xlabel = 'seconds'
            @age_String = ''
        end
        d.star_age.div!(@age_Scale)
        d.age_Scale = @age_Scale

        @age_format = (d.star_age[-1] > 9.99)? '\sffamily %0.5f' : '\sffamily %0.6f'
        @mass_format = (d.star_mass[0] > 9.99)? '\sffamily %0.2f' : '\sffamily %0.3f'
        
    end
    
    def find_He_gap # may have a gap in tracks if had a helium flash
        @gap_in_tracks = false
    end
    
    def set_track_for_plots
        @track_first = 0
        @track_last = -1
        
        if @xaxis == 'by_lg_ybp'
           @track_last = -1
           while d.lg_ybp[@track_last] == d.lg_ybp[@track_last-1]
              @track_last -= 1 
           end
           @track_last -= 1
        end
        
        if @xaxis == 'by_lg_sbp'
           @track_last = -1
           while d.lg_sbp[@track_last] == d.lg_sbp[@track_last-1]
              @track_last -= 1 
           end
           @track_last -= 1
        end
        
        @start_age = d.star_age[@track_first]
        @end_age = d.star_age[@track_last]
        age_range = @end_age - @start_age
        @end_age += 0.01 * age_range
        @xaxis_left = @xaxis_data[@track_first]
        @xaxis_right = @xaxis_data[@track_last]
        dx = @xaxis_right - @xaxis_left
        if @add_xaxis_margin
         @xaxis_left = @xaxis_left - dx*0.07
         @xaxis_right = @xaxis_right + dx*0.07
        end
        @xaxis_right = -9.1 if @xaxis == 'by_lg_ybp' && @xaxis_right < -9.1
        @xaxis_right = -1 if @xaxis == 'by_lg_sbp' && @xaxis_right < -1
        
    end
    
    def stroke_track(xs, ys, color=Black, legend=nil, type=Line_Type_Solid, track_first=@track_first)
        t.stroke_color = color
        t.line_type = type
        if @gap_in_tracks
            t.append_points_with_gaps_to_path(xs,ys,[@helium_gap_start-track_first],false)
        else
            t.append_points_to_path(xs,ys)
        end
        t.save_legend_info(legend) if legend != nil
        t.stroke
    end
    
    def mark_spot(x, y, color = Red)
        t.show_marker('x' => x, 'y' => y, 'marker' => Bullet, 'scale' => 0.5, 'color' => color);
    end
    
    def min_of_many(ary, y_limit = nil)
        return nil if ary == nil || ary.size == 0
        ymin = Dvector.min_of_many(ary)
        ymin = y_limit if y_limit != nil && ymin < y_limit
        return ymin
    end
    
    def max_of_many(ary, y_limit = nil)
        return nil if ary == nil || ary.size == 0
        ymax = Dvector.max_of_many(ary)
        ymax = y_limit if y_limit != nil && ymax > y_limit
        return ymax
    end
    
    def show_box_labels(title, xlabel=nil, ylabel=nil)
        if title != nil
            t.show_title(title); t.no_title
        end
        if xlabel != nil
            t.show_xlabel(xlabel); t.no_xlabel
        end
        if ylabel != nil
            t.show_ylabel(ylabel); t.no_ylabel
        end
    end
    
    def background
    end
    
    
    
    def logT_logRho_logLneu
        t.set_aspect_ratio(0.5)
        num_plots = 3
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('log_center_T')) 
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('log_center_Rho'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('log_Lneu'))
        end
    end
    
    
    
    def kipp_dt_nz_retries
        t.set_aspect_ratio(0.5)
        num_plots = 5
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        t.subplot(t.row_margins('num_rows' => num_plots, 'first_row' => 1, 'last_row' => 2, 
                'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            kippenhahn_abundances
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('log_dt'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 4, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('num_zones'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 5, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('num_retries'))
        end
    end
    
    
    
    def center_entropy_ye
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        t.subplot(t.row_margins('num_rows' => num_plots, 'first_row' => 1, 'last_row' => 1, 
                'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('center_entropy'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.title_visible = false
            plot_column(names.index('center_ye'))
        end
    end
    
    
    
    def teff_L
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        t.subplot(t.row_margins('num_rows' => num_plots, 'first_row' => 1, 'last_row' => 1, 
                'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            title = nil
            ylabel = 'Teff (K)'
            ys = d.log_Teff[@track_first .. @track_last].exp10
            ymin = ys.min
            ymax = ys.max
            plot_data(title, ylabel, ymin, ymax, ys)
            #plot_column(names.index('log_Lneu_div_L'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.title_visible = false
            title = nil
            ylabel = 'L/Lsun'
            ys = d.log_L[@track_first .. @track_last].exp10
            ymin = ys.min
            ymax = ys.max
            plot_data(title, ylabel, ymin, ymax, ys)
            #plot_column(names.index('log_L_div_Ledd'))
        end
    end
    
    
    
    def teff_L_T_Rho
        t.set_aspect_ratio(0.75)
        num_plots = 4
        row_margin = 0.02
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        t.xaxis_major_tick_length = 0.35
        t.xaxis_minor_tick_length = 0.15
        t.yaxis_major_tick_length = 0.35
        t.yaxis_minor_tick_length = 0.15
        t.stroke_width = 0.7
        t.subplot(t.row_margins('num_rows' => num_plots, 'first_row' => 1, 'last_row' => 1, 
                'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            title = nil
            ylabel = '$\mathrm{T_{eff}}$ (K)'
            ys = d.log_Teff[@track_first .. @track_last].exp10
            ymin = ys.min
            ymax = ys.max
            plot_data(title, ylabel, ymin, ymax, ys)
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            t.title_visible = false
            title = nil
            ylabel = 'L ($\mathrm{L_\odot}$)'
            ys = d.log_L[@track_first .. @track_last].exp10
            ymin = ys.min
            ymax = ys.max
            plot_data(title, ylabel, ymin, ymax, ys)
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            t.title_visible = false
            title = nil
            ylabel = 'log Rho central'
            ys = d.log_center_Rho[@track_first .. @track_last]
            ymin = ys.min
            ymax = ys.max
            plot_data(title, ylabel, ymin, ymax, ys)
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 4, 'row_margin' => row_margin)) do
            t.title_visible = false
            title = nil
            ylabel = 'log T central'
            ys = d.log_center_T[@track_first .. @track_last]
            ymin = ys.min
            ymax = ys.max
            plot_data(title, ylabel, ymin, ymax, ys)
        end
    end
    
    
    
    def log_L_info
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        t.subplot(t.row_margins('num_rows' => num_plots, 'first_row' => 1, 
                'last_row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            title = 'log_Lneu_div_L'
            ylabel = 'log (Lneu / L)'
            ys = d.log_Lneu_div_L[@track_first .. @track_last]
            ymin = ys.min
            ymax = ys.max
            plot_data(title, ylabel, ymin, ymax, ys)
            #plot_column(names.index('log_Lneu_div_L'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.title_visible = false
            title = 'L_div_Ledd'
            ylabel = 'L / Ledd'
            ys = d.log_L_div_Ledd[@track_first .. @track_last].exp10
            ymin = ys.min
            ymax = ys.max
            plot_data(title, ylabel, ymin, ymax, ys)
            #plot_column(names.index('log_L_div_Ledd'))
        end
    end
    
    
    
    def kipp_dt_nz
        t.set_aspect_ratio(0.5)
        num_plots = 3
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        t.subplot(t.row_margins('num_rows' => num_plots, 'first_row' => 1, 'last_row' => 1, 
                'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            kippenhahn_abundances
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('log_dt'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            plot_column(names.index('num_zones'))
        end
    end
    
    
    
    def dt_nz_retries
        t.set_aspect_ratio(0.5)
        num_plots = 3
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('log_dt'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('num_zones'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column(names.index('num_retries'))
        end
    end
    
    
    
    def dt_nz
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('log_dt'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.title_visible = false
            plot_column(names.index('num_zones'))
        end
    end
    
    
    def dt_min_Pgas_div_P
        t.set_aspect_ratio(0.5)
        num_plots = 2
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('log_dt'))
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.title_visible = false
            plot_column(names.index('min_Pgas_div_P'))
        end
    end
    
    
    
    def lgLHe
        t.set_aspect_ratio(0.85)
        num_plots = 1
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        #t.rescale(0.8)
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            plot_column_by_name('log_LHe', 'log L_{\rm He}\ ($\mathrm{L_\odot}$)')
        end
    end
    
    
    def code3
        #t.portrait
        t.set_aspect_ratio(0.5)
        t.xaxis_major_tick_length = 0.4
        t.yaxis_major_tick_length = 0.4
        t.xaxis_minor_tick_length = 0.15
        t.yaxis_minor_tick_length = 0.15
        num_plots = 3
        row_margin = 0.02
        names = d.column_names
        xmin = 0.6
        xmax = 5 # 7.5
        exp_ys = false
        max_rel_ys = false
        log_ys = false
        xaxis = 'model number'
        t.title_visible = false
        t.line_width = 0.5
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column_by_name('num_zones', 'zones')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column_by_name('log_dt', 'log dt (years)')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            plot_column_by_name('num_retries', 'retries')
        end
    end

    
    
    
    def deltaM_Teff_L
        t.set_aspect_ratio(0.5)
        num_plots = 3
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        t.line_width = 0.5
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column_by_name('delta_mass', 'M - Minit (Msun)')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column_by_name('log_Teff', 'log Teff (K)')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column_by_name('log_L', 'log L/Lsun')
        end
    end
    
    
    def dt_nz_LHe
        t.set_aspect_ratio(0.5)
        num_plots = 3
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        title = ''
        t.rescale(0.8)
        t.line_width = 0.5
        secyer = 3.1558149984e7
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column_by_name('log_LHe', 'log $\rm L_{\rm He}}$ ($\mathrm{L_\odot}$)')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            ys = d.log_dt.add(log10(secyer))
            ymin = -4
            ymax = 4
            plot_data(title, 'log timestep (s)', ymin, ymax, ys)
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column_by_name('num_zones', 'number of cells')
        end
    end
    
    
    def dt_nz_L
        t.set_aspect_ratio(0.5)
        num_plots = 3
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        title = ''
        t.rescale(0.8)
        t.line_width = 0.5
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column_by_name('log_L', 'log $\rm L}$ ($\mathrm{L_\odot}$)')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            t.top_edge_type = AXIS_HIDDEN
            ys = d.log_dt
            ymin = -5.5
            ymax = 2
            plot_data(title, 'log timestep (yr)', ymin, ymax, ys)
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            t.top_edge_type = AXIS_HIDDEN
            plot_column_by_name('num_zones', 'number of cells')
        end
    end
    
    
    def dt_nz_retries_L
        t.set_aspect_ratio(0.5)
        num_plots = 4
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        title = ''
        t.rescale(0.8)
        t.line_width = 0.5
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column_by_name('log_L', 'log $\rm L}$ ($\mathrm{L_\odot}$)')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            t.top_edge_type = AXIS_HIDDEN
            ys = d.log_dt
            ymin = -4
            ymax = 9
            plot_data(title, 'log timestep (yr)', ymin, ymax, ys)
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            t.top_edge_type = AXIS_HIDDEN
            plot_column_by_name('num_retries', 'retries')
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 4, 'row_margin' => row_margin)) do 
            t.title_visible = false
            t.top_edge_type = AXIS_HIDDEN
            plot_column_by_name('num_zones', 'number of cells')
        end
    end
    
    
    def min_log_L_Ledd
        t.set_aspect_ratio(0.85)
        num_plots = 1
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        ys = d.log_L.dup
        ys.size.times do |i|
            if d.log_L[i] > d.log_Ledd[i]
                ys[i] = d.log_Ledd[i]
            else
                ys[i] = d.log_L[i]
            end
        end
        ymax = ys.max
        ymax = d.log_L.max
        ymin = ys.min
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.context { plot_data('', 'logL logLedd', ymin, ymax, ys) }
            t.line_width = 0.3
            t.stroke_color = Grey
            plot_data('', '', ymin, ymax, d.log_L)
        end
    end

    
    def logLs
        #t.set_aspect_ratio(0.5)
        num_plots = 1
        row_margin = 0.0
        names = d.column_names
        t.title_visible = false
        t.rescale(0.8)
        t.line_width = 0.5
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            #t.xlabel_visible = false
            #t.xaxis_type = AXIS_WITH_TICKS_ONLY
            title = nil
            ylabel = 'L (10^4\ $\mathrm{L_\odot}$)'
            ys = d.log_L[@track_first .. @track_last].exp10.mul(1e-4)
            ymin = 1
            ymax = ys.max
            ymax = 9.7
            plot_data(title, ylabel, ymin, ymax, ys, 'decorate_L')
        end
        return
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            plot_column_by_name('log_LHe', 'log $\rm L_{\rm He}}$ ($\mathrm{L_\odot}$)')
        end
    end
    
    def decorate_L
        t.line_type = Line_Type_Dash
        y = 0.9149
        t.stroke_line(-1, y, 100, y)
        t.line_type = Line_Type_Dot
        y = 10
        t.stroke_line(-1, y, 100, y)
    end


      def plot_cz_dt_dlnTbase
        t.set_aspect_ratio(0.85)
        t.rescale(0.8)
        t.show_xlabel('Base T (10^8\rm K)')
        xs = d.cz_logT.exp10.mul(1e-8)
        xs = xs[@track_first .. @track_last]
        t.show_ylabel('dt/dlnT_{\!\rm base}\ \rm (s)'); t.no_ylabel
        ys = d.cz_logT.dup
        secyer = 3.15e7
        lnTs = d.cz_logT.mul(log(10.0))
        3.times { 
            lnTs = lnTs.convolve([1,2,1],1)
        }
        (ys.size - 1).times do |i|
            dlnTbase = lnTs[i+1] - lnTs[i]
            if dlnTbase > 0
                dt = d.log_dt[i+1].exp10
                ys[i] = dt*secyer/dlnTbase
            else
                ys[i] = 1e99
            end
        end
        ys[-1] = ys[-2]
        ys = ys[@track_first .. @track_last]
        ymax = 1000 #ys.max
        ymin = ys.min
        xmin = xs.min
        xmax = xs.max
        t.show_plot('boundaries' => [ xmin, xmax, ymax, ymin ]) do
            t.stroke_width = 0.7
            t.show_polyline(xs, ys)
        end
      end
      
      
      def cz_dt_dlnTbase
        title = 'dt/dlnT_{\!\rm base}\ \rm (s)'
        ylabel = nil
        ys = d.cz_logT.dup
        secyer = 3.15e7
        lnTs = d.cz_logT.mul(log(10.0))
        3.times { 
            lnTs = lnTs.convolve([1,2,1],1)
        }
        min_dt = -1; min_i = 0
        (ys.size - 1).times do |i|
            dlnTbase = lnTs[i+1] - lnTs[i]
            if dlnTbase > 0
                dt = d.log_dt[i+1].exp10
                ys[i] = dt*secyer/dlnTbase
                if (min_dt < 0) || (ys[i] < min_dt)
                    min_dt = ys[i]
                    min_i = i
                end
            else
                ys[i] = 1e99
            end
        end
        puts "min_dt #{min_dt} model_number #{d.model_number[min_i].to_i}" if min_dt > 0
        ys[-1] = ys[-2]
        ys = ys[@track_first .. @track_last]
        ymax = 1000 #ys.max
        ymin = ys.min
        plot_data(title, ylabel, ymin, ymax, ys)
      end
      
      
      def max_eps_nuc_dt_dlnT
        title = 'max eps nuc dt/dlnT\ \rm (s)'
        ylabel = nil
        ys = d.max_eps_nuc_lgT.dup
        secyer = 3.15e7
        ln10 = log(10.0)
        lnTs = d.max_eps_nuc_lgT.mul(ln10)
        3.times { 
            lnTs = lnTs.convolve([1,2,1],1)
        }
        min_dt = -1; min_i = 0
        (ys.size - 1).times do |i|
            dlnTbase = lnTs[i+1] - lnTs[i]
            if dlnTbase > 0
                dt = d.log_dt[i+1].exp10
                ys[i] = dt*secyer/dlnTbase
                if (min_dt < 0) || (ys[i] < min_dt)
                    min_dt = ys[i]
                    min_i = i
                end
            else
                ys[i] = 1e99
            end
        end
        puts "min_dt #{min_dt} model_number #{d.model_number[min_i].to_i}" if min_dt > 0
        ys[-1] = ys[-2]
        ys = ys[@track_first .. @track_last]
        ymax = ys.max
        ymax = 1000 if ys.max > 1000
        ymin = ys.min
        plot_data(title, ylabel, ymin, ymax, ys)
      end


      def plot_cz_max_abundances
        t.set_aspect_ratio(0.85)
        t.rescale(0.8)
        t.show_xlabel('Time since Peak Luminosity (s)')
        xs = @xaxis_data[@track_first .. @track_last]
        t.show_ylabel('log mass fraction'); t.no_ylabel
        ymax = 0.02
        ymin = -3.01
        n = 2
        minmax = -43 #-1.301
        puts "xs.min #{xs.min}"
        puts "xs.max #{xs.max}"
        t0 = xs.where_first_ge(-0.13); puts "t0 #{t0}  xs[t0] #{xs[t0]}"
        t1 = xs.where_first_ge(0.7); puts "t1 #{t1}"
        t2 = xs.where_first_ge(0.5); puts "t2 #{t2}"
        t3 = xs.where_first_ge(0.85); puts "t3 #{t3}"
        t.show_plot('boundaries' => [ @xaxis_left, @xaxis_right, ymax, ymin ]) do
            t.stroke_width = 0.7
            do1_cz_max(xs, d.cz_max_log_h1, Black, '$^{1}$\rm H', minmax, t1) unless d.cz_max_log_h1 == nil
            do1_cz_max(xs, d.cz_max_log_he4, Black, '$^{4}$\rm He', minmax, -t1, t0) unless d.cz_max_log_he4 == nil
            do1_cz_max(xs, d.cz_max_log_c12, Black, '$^{12}$\rm C', minmax, -t2, t0) unless d.cz_max_log_c12 == nil
            do1_cz_max(xs, d.cz_max_log_n13, Black, '$^{13}$\rm N', minmax, t1) unless d.cz_max_log_n13 == nil
            do1_cz_max(xs, d.cz_max_log_n14, Black, '$^{14}$\rm N', minmax, t1) unless d.cz_max_log_n14 == nil
            
            do1_cz_max(xs, d.cz_max_log_o16, Black, '$^{16}$\rm O', minmax, t1) unless d.cz_max_log_o16 == nil
            do1_cz_max(xs, d.cz_max_log_ne20, Black, '$^{20}$\rm Ne', minmax, t1) unless d.cz_max_log_ne20 == nil
            do1_cz_max(xs, d.cz_max_log_na23, Black, '$^{23}$\rm Na', minmax, t1) unless d.cz_max_log_na23 == nil
            do1_cz_max(xs, d.cz_max_log_mg24, Black, '$^{24}$\rm Mg', minmax, t1) unless d.cz_max_log_mg24 == nil
            do1_cz_max(xs, d.cz_max_log_al27, Black, '$^{27}$\rm Al', minmax, t1) unless d.cz_max_log_al27 == nil
            
            do1_cz_max(xs, d.cz_max_log_si28, Black, '$^{28}$\rm Si', minmax, t2) unless d.cz_max_log_si28 == nil
            do1_cz_max(xs, d.cz_max_log_p31, Black, '$^{31}$\rm P', minmax, -t3) unless d.cz_max_log_p31 == nil
            do1_cz_max(xs, d.cz_max_log_s32, Black, '$^{32}$\rm S', minmax, t3) unless d.cz_max_log_s32 == nil
            
            do1_cz_max(xs, d.cz_max_log_cl35, Black, '$^{35}$\rm Cl', minmax, t1) unless d.cz_max_log_cl35 == nil
            do1_cz_max(xs, d.cz_max_log_ar36, Black, '$^{36}$\rm Ar', minmax, -t1) unless d.cz_max_log_ar36 == nil
            do1_cz_max(xs, d.cz_max_log_k39, Black, '$^{39}$\rm K', minmax, t1) unless d.cz_max_log_k39 == nil
            do1_cz_max(xs, d.cz_max_log_ca40, Black, '$^{40}$\rm Ca', minmax, t1) unless d.cz_max_log_ca40 == nil
            do1_cz_max(xs, d.cz_max_log_sc43, Black, '$^{43}$\rm Sc', minmax, t1) unless d.cz_max_log_sc43 == nil
            do1_cz_max(xs, d.cz_max_log_ti44, Black, '$^{44}$\rm Ti', minmax, t3) unless d.cz_max_log_ti44 == nil
            do1_cz_max(xs, d.cz_max_log_v47, Black, '$^{47}$\rm V', minmax, t1) unless d.cz_max_log_v47 == nil
            do1_cz_max(xs, d.cz_max_log_cr48, Black, '$^{48}$\rm Cr', minmax, t1) unless d.cz_max_log_cr48 == nil
            do1_cz_max(xs, d.cz_max_log_cr56, Black, '$^{56}$\rm Cr', minmax, t1) unless d.cz_max_log_cr56 == nil
            do1_cz_max(xs, d.cz_max_log_mn51, Black, '$^{51}$\rm Mn', minmax, t1) unless d.cz_max_log_mn51 == nil

            do1_cz_max(xs, d.cz_max_log_fe52, Black, '$^{52}$\rm Fe', minmax, t1) unless d.cz_max_log_fe52 == nil
            do1_cz_max(xs, d.cz_max_log_fe54, Black, '$^{54}$\rm Fe', minmax, t1) unless d.cz_max_log_fe54 == nil
            do1_cz_max(xs, d.cz_max_log_fe56, Black, '$^{56}$\rm Fe', minmax, t1) unless d.cz_max_log_fe56 == nil
            do1_cz_max(xs, d.cz_max_log_ni56, Black, '$^{56}$\rm Ni', minmax, t1) unless d.cz_max_log_ni56 == nil
            
        end
      end


      
      
      def do1_cz_max(
          xs_plot, abundance, color, label, 
          minmax, k, k2 = nil)
        ys_plot = abundance[@track_first .. @track_last]
        unless minmax == nil
           return if ys_plot.max < minmax
        end
        scale = 0.7
        t.show_polyline(xs_plot, ys_plot, color)
        if k > 0
            dy = 0.02
        else
            dy = -0.11
            k = -k
        end
        x = xs_plot[k]
        y = ys_plot[k]
        t.show_label(
            'x' => x, 'y' => y+dy, 
            'text' => label, 'scale' => scale, 
            'justification' => CENTERED,
            'alignment' => ALIGNED_AT_BASELINE)
        return if k2 == nil
        x = xs_plot[k2]
        y = ys_plot[k2]
        t.show_label(
            'x' => x, 'y' => y+dy, 
            'text' => label, 'scale' => scale, 
            'justification' => CENTERED,
            'alignment' => ALIGNED_AT_BASELINE)
    end

end

