/* pdffile.c */
/*
   Copyright (C) 2010  Bill Paxton

   This is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Library Public License as published
   by the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details. 

   You should have received a copy of the GNU Library General Public License;
   if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/


/* this is a quick hack using the PDF code from Tioga */
/* the goal is very limited; just simple vector graphics */


#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <stdbool.h>
#include <math.h>





/* make the driver callable from FORTRAN */
#ifdef PG_PPU
#define PDDRIV pddriv_
#else
#define PDDRIV pddriv
#endif


/* miscellaneous constants */

// ENLARGE = the conversion factor from "big points" to output units
#define ENLARGE 10.0
#define BIG_POINTS_PER_INCH 72.0
#define INCHES_PER_MM 0.0393700787


#define DPI (ENLARGE*BIG_POINTS_PER_INCH)
#define PAGE_WIDTH 7.8      /* 7.8 inches */
#define PAGE_HEIGHT 10.5    /* 10.5 inches */
#define NCOLORS 256
#define DEFAULT_FILENAME "pgplot.pdf"



#define boolean unsigned char
#define true 1
#define false 0

#ifndef MAX
#define MAX(a,b)    (((a) > (b)) ? (a) : (b))
#endif
#ifndef MIN
#define MIN(a,b)    (((a) < (b)) ? (a) : (b))
#endif
#ifndef ABS
#define ABS(a)      ((a)<0 ? -(a) : (a))
#endif
#ifndef ROUND
#define ROUND(a)    (int)((a)<0.0 ? ((a)-0.5) : ((a)+0.5))
#endif
#ifndef SIGN
#define SIGN(a)         ((a)<0 ? -1 : 1)
#endif


#define PDF_IDENT "PGPLOT /pdf" /* used in warning messages */
static char *pdf_ident; /* should be set each time pddriv() is entered */

/* use for opcode = 1 */
//#define DEVICE_NAME "PDF     (a tiny subset of pdf for vector graphics)"
#define DEVICE_NAME "CGM (CGM file, indexed colour selection mode)"



/* taken from the GIF driver */
static float base_colors[] = {
  0,   0,   0,
  255, 255, 255,
  255, 0, 0,
  0, 255, 0,
  0, 0, 255,
  0, 255, 255,
  255, 0, 255,
  255, 255, 0,
  255, 128, 0,
  128, 255, 0,
  0, 255, 128,
  0, 128, 255,
  128, 0, 255,
  255, 0, 128,
  85, 85, 85,
  170, 170, 170,
};

/* each new device initially copies its colortable from here */
static float default_colortable[NCOLORS * 3];

/* data for a single open device */
boolean error; /* if true, we can plot no more on this device */
char filename[256];
float ctable[NCOLORS * 3];
float cindex; /* current plotting color index */
int devnum; /* this device's identifier */
// pdf info
float line_width;
float page_left;
float page_right;
float page_bottom;
float page_top;
float page_width;
float page_height;
float clip_left;
float clip_bottom;
float clip_right;
float clip_top;
float color_R;
float color_G;
float color_B;

int bbox_llx, bbox_lly, bbox_urx, bbox_ury;

int NPTS, LASTI, LASTJ, NSEG; 
bool START;
bool have_current_point = false, constructing_path = false, writing_file = false;


/* copy the default colortable to a newly-opened device's ctable entry */
static void initialize_device_ctable() {
  memcpy(ctable, default_colortable, 3 * NCOLORS * sizeof(float));
}

/* use to set the RGB components of a colortable entry */
static void set_color_rep( float ci, float r, float g, float b) {
   int i = ci;
  ctable[i*3+0] = r;
  ctable[i*3+1] = g;
  ctable[i*3+2] = b;
  //printf("driver: set color rep %i %0.3f %0.3f %0.3f\n", i, r, g, b);
}

static void get_color_rep( float ci, float *r, float *g, float *b) {
   int i = ci;
   //printf("get_color_rep ci %g  i %i\n", ci, i);
   if (i < 0 || i >= NCOLORS) {
      printf("invalid color index value %i\n", i);
      return;
   }
  *r = ctable[i*3+0];
  *g = ctable[i*3+1];
  *b = ctable[i*3+2];
  //printf("driver: get color rep %i %0.3f %0.3f %0.3f\n", i, *r, *g, *b);
  //printf("done in get_color_rep\n");
}
















typedef struct stroke_opacity_state {
   struct stroke_opacity_state *next;
   int gs_num;
   int obj_num;
   double stroke_opacity;
} Stroke_Opacity_State;

typedef struct fill_opacity_state {
   struct fill_opacity_state *next;
   int gs_num;
   int obj_num;
   double fill_opacity;
} Fill_Opacity_State;

typedef struct xobj_info {
   struct xobj_info *next;
   int xo_num;
   int obj_num;
   int xobj_subtype;
} XObject_Info;

typedef struct jpg_info {
   // start must match start of xobj_info
   struct xobj_info *next;
   int xo_num;
   int obj_num;
   int xobj_subtype;
   // remainder is for this subtype of xobj
   int width, height;
   int mask_obj_num;
   char *filename;
} JPG_Info;

typedef struct sampled_info {
   // start must match start of xobj_info
   struct xobj_info *next;
   int xo_num;
   int obj_num;
   int xobj_subtype;
   // remainder is for this subtype of xobj
   int width, height;
   int length; // number of bytes of image data
   unsigned char *image_data;
   bool interpolate;
   bool reversed; // only applies to mono images
   int mask_obj_num;
   int image_type;
   int value_mask_min;
   int value_mask_max;
   int hival;
   int lookup_len;
   unsigned char *lookup;
} Sampled_Info;

#define JPG_SUBTYPE 1
#define SAMPLED_SUBTYPE 2

#define RGB_IMAGE 0
#define HLS_IMAGE 5
#define CMYK_IMAGE 4
#define GRAY_IMAGE 1
#define MONO_IMAGE 2
#define COLORMAP_IMAGE 3

typedef struct function_info {
   struct function_info *next;
   int obj_num;
   int hival;
   int lookup_len;
   unsigned char *lookup;
} Function_Info;

typedef struct shading_info {
   struct shading_info *next;
   int shade_num;
   int obj_num;
   bool axial;
   double x0;
   double y0;
   double x1;
   double y1;
   double r0;
   double r1;
   int function;
   bool extend_start;
   bool extend_end;
} Shading_Info;

typedef struct { 
   int font_num; // for making font resource name such as /F7
   char *font_name;
   int firstChar, lastChar;  // firstChar typically is 0 and lastChar 255
   int char_width[256], char_llx[256], char_lly[256], char_urx[256], char_ury[256];
   int widths_obj_num;
   /* FontDescriptor */
   int flags; // describe the font
   int fnt_llx, fnt_lly, fnt_urx, fnt_ury;  // FontBBox
   int italicAngle, ascent, descent, capHeight, stemV;
} Font_Afm_Info;

typedef struct font_dictionary { 
   struct font_dictionary *next;
   int font_num; // for making font resource name such as /F7
   int obj_num;
   bool in_use;
   int widths_obj_num;
   int descriptor_obj_num;
   Font_Afm_Info *afm;
} Font_Dictionary;

typedef struct old_font_dictionary { 
   struct old_font_dictionary *next;
   int font_num; // for making font resource name such as /F7
   int obj_num;
   bool in_use;
   char *font_name;
   int firstChar, lastChar;  // firstChar typically is 0 and lastChar 255
   int char_width[256], char_llx[256], char_lly[256], char_urx[256], char_ury[256];
   int widths_obj_num;
   /* FontDescriptor */
   int descriptor_obj_num;
   int flags; // describe the font
   int fnt_llx, fnt_lly, fnt_urx, fnt_ury;  // FontBBox
   int italicAngle, ascent, descent, capHeight, stemV;
} Old_Font_Dictionary;

#define FixedPitchFlag 1
#define SerifFlag 2
#define SymbolicFlag 4
#define ScriptFlag 8
#define NonsymbolicFlag (1<<5)
#define ItalicFlag (1<<6)
#define AllCapFlag (1<<16)
#define SmallCapFlag (1<<17)
#define ForceBoldFlag (1<<18)

#define RADIANS_TO_DEGREES (180.0 / PI)


#define LINE_CAP_BUTT 0
#define LINE_CAP_ROUND 1
#define LINE_CAP_SQUARE 2
#define LINE_CAP_STANDARD LINE_CAP_ROUND

#define LINE_JOIN_MITER 0
#define LINE_JOIN_ROUND 1
#define LINE_JOIN_BEVEL 2
#define LINE_JOIN_STANDARD LINE_JOIN_ROUND


/* must match the font numbers in FigureConstants.rb */
char *predefined_Fonts[] = { 
   NULL,
   "Times-Roman",
   "Times-Italic", // 2
   "Times-Bold",
   "Times-BoldItalic",
   "Helvetica",
   "Helvetica-Oblique", // 6
   "Helvetica-Bold",
   "Helvetica-BoldOblique",
   "Courier",
   "Courier-Oblique",  // 10
   "Courier-Bold",
   "Courier-BoldOblique",
   "Symbol",
   "ZapfDingbats" // 14
};
int num_predefined_fonts = 14;
int num_pdf_standard_fonts = 14;

long *obj_offsets, capacity_obj_offsets, stream_start, stream_end;
long length_offset, xref_offset;
long num_objects, next_available_object_number, next_available_gs_number;
long next_available_xo_number;
long next_available_shade_number, next_available_font_number;
Stroke_Opacity_State *stroke_opacities = NULL;
Fill_Opacity_State *fill_opacities = NULL;
XObject_Info *xobj_list = NULL;
Function_Info *functions_list;
Shading_Info *shades_list = NULL;
Font_Dictionary *font_dictionaries = NULL;
Old_Font_Dictionary *old_font_dictionaries = NULL;
FILE *OF = NULL; // for the PDF file
FILE *TF = NULL; // for the temp file


#define err_buff_len 256

void RAISE_ERROR(char *str)
{
   char msg[err_buff_len];
   sprintf(msg, "RAISE_ERROR: %s", str);
   printf(msg);
   printf("\n\n");
   exit(EXIT_FAILURE);
   }


void RAISE_ERROR_s(char *fmt, char *s)
{
   char buff[err_buff_len];
   sprintf(buff, fmt, s);
   RAISE_ERROR(buff);
}


void RAISE_ERROR_ss(char *fmt, char *s1, char *s2)
{
   char buff[err_buff_len];
   sprintf(buff, fmt, s1, s2);
   RAISE_ERROR(buff);
}


void RAISE_ERROR_i(char *fmt, int x)
{
   char buff[err_buff_len];
   sprintf(buff, fmt, x);
   RAISE_ERROR(buff);
}


void RAISE_ERROR_ii(char *fmt, int x1, int x2)
{
   char buff[err_buff_len];
   sprintf(buff, fmt, x1, x2);
   RAISE_ERROR(buff);
}


void RAISE_ERROR_g(char *fmt, double x)
{
   char buff[err_buff_len];
   sprintf(buff, fmt, x);
   RAISE_ERROR(buff);
}


void RAISE_ERROR_gg(char *fmt, double x1, double x2)
{
   char buff[err_buff_len];
   sprintf(buff, fmt, x1, x2);
   RAISE_ERROR(buff);
}


void GIVE_WARNING(const char *fmt, const char *str)
{
   printf(fmt, str);
}



void set_color(float ci) {
   float r, g, b;
   float red, green, blue;
   int i;
   //printf("call get_color_rep ci %f\n", ci);
   get_color_rep(ci, &r, &g, &b);
   //printf("back from get_color_rep\n");
   red = r/255.0;
   green = g/255.0;
   blue = b/255.0;
   if (writing_file) {
      fprintf(TF, "%0.3f %0.3f %0.3f RG\n", red, green, blue); // stroke color
      fprintf(TF, "%0.3f %0.3f %0.3f rg\n", red, green, blue); // fill color
   }
   i = ci;
   //printf("done in driver: set color %i %0.3f %0.3f %0.3f\n", i, red, green, blue);
}

void update_bbox(int x, int y)
{
   if (x >= clip_left && x < bbox_llx) bbox_llx = x;
   if (y >= clip_bottom && y < bbox_lly) bbox_lly = y;
   if (x <= clip_right && x > bbox_urx) bbox_urx = x;
   if (y <= clip_top && y > bbox_ury) bbox_ury = y;
}

void c_line_width_set(float line_width) {
   //fprintf(TF, "%0.3f w\n", line_width * ENLARGE);
   //fprintf(TF, "%0.3f w\n", line_width*5);   // the factor of 5 is taken from the ps driver.
   fprintf(TF, "%0.3f w\n", line_width*3);
}

void c_moveto(int x, int y) {
   fprintf(TF, "%i %i m\n", x, y);
   update_bbox(x, y);
}

void c_lineto(int x, int y) {
   fprintf(TF, "%i %i l\n", x, y);
   update_bbox(x, y);
}

void c_close_path() {
   fprintf(TF, "h\n");
}

void c_stroke() {
   fprintf(TF, "S\n");
}

void c_fill() {
   fprintf(TF, "f\n");
}


#define Get_pdf_xoffset()  5.0
#define Get_pdf_yoffset()  5.0



void c_line_cap_set(int line_cap) {
   fprintf(TF, "%d J\n", line_cap);
}

void c_line_join_set(int line_join) {
   fprintf(TF, "%d j\n", line_join);
}

void c_miter_limit_set(double miter_limit) {
   fprintf(TF, "%0.3f M\n", miter_limit);
}





void c_set_device_pagesize(double width, double height) { 
   // sizes in units of 1/720 inch
   page_left = 0;
   page_right = width;
   page_bottom = 0;
   page_top = height;
   page_width = page_right - page_left;
   page_height = page_top - page_bottom;
   clip_left = page_left;
   clip_right = page_right;
   clip_top = page_top;
   clip_bottom = page_bottom;
}


/* PDF File Management */


void
Init_pdf()
{
   int i;
   writing_file = false;
   capacity_obj_offsets = 1000;
   num_objects = 0;
   obj_offsets = calloc(capacity_obj_offsets, sizeof(long));
   for (i=0; i < capacity_obj_offsets; i++) obj_offsets[i] = 0;
}


void
Shutdown_pdf()
{
   free(obj_offsets);
}


void
Record_Object_Offset(int obj_number)
{
   long int offset = ftell(OF);
   if (obj_number >= capacity_obj_offsets) {
      int size_increment = 50, i;
      capacity_obj_offsets = obj_number + size_increment;
      obj_offsets = realloc(obj_offsets, capacity_obj_offsets*sizeof(long));
      for (i=num_objects; i < capacity_obj_offsets; i++) obj_offsets[i] = 0;
   }
   obj_offsets[obj_number] = offset;
   if (obj_number >= num_objects) num_objects = obj_number + 1;
}



#define INFO_OBJ 1
#define PAGES_OBJ 2
#define STREAM_OBJ 3
#define PAGE_OBJ 4
#define CATALOG_OBJ 5
#define FIRST_OTHER_OBJ 6



static void
Get_pdf_name(char *ofile, char *filename, int maxlen)
{
   strncpy(ofile, filename, maxlen);
}


void
Open_pdf(char *filename)
{
   int i;
   writing_file = true;
   next_available_object_number = FIRST_OTHER_OBJ;
   next_available_font_number = num_predefined_fonts + 1;
   next_available_gs_number = 1;
   next_available_xo_number = 1;
   next_available_shade_number = 1;
   writing_file = true;
   time_t now = time(NULL);
   char ofile[300], timestring[100];
   Get_pdf_name(ofile, filename, 300);
   if ((OF = fopen(ofile, "w")) == NULL) {
      RAISE_ERROR_s("Sorry: can't open %s.\n", filename);
      return;
   }
   if ((TF = tmpfile()) == NULL) {
      RAISE_ERROR_s("Sorry: can't create temp file for writing PDF file %s.\n",
                    filename);
      return;
   }
   /* open PDF file and write header */
   fprintf(OF, "%%PDF-1.4\n");
   strcpy(timestring, ctime(&now));
   i = strlen(timestring);
   if (i > 0) timestring[i-1] = '\0';
   Record_Object_Offset(INFO_OBJ);
   fprintf(OF,
           "%i 0 obj <<\n/CreationDate (%s)\n>>\nendobj\n",
           INFO_OBJ, timestring);
   Record_Object_Offset(PAGES_OBJ);
   fprintf(OF,
           "%i 0 obj <<\n/Type /Pages\n/Kids [%i 0 R]\n/Count 1\n>> endobj\n",
           PAGES_OBJ, PAGE_OBJ);
   Record_Object_Offset(STREAM_OBJ);
//   if (FLATE_ENCODE)
//      fprintf(OF, "%i 0 obj <<\t/Filter /FlateDecode   /Length ", STREAM_OBJ);
//   else
      fprintf(OF, "%i 0 obj <<\t/Length ", STREAM_OBJ);
   length_offset = ftell(OF);
   fprintf(OF, "             \n>>\nstream\n");
   stream_start = ftell(OF);
   fprintf(TF, "%.2f 0 0 %.2f %.2f %.2f cm\n", 1.0/ENLARGE, 1.0/ENLARGE,
           Get_pdf_xoffset(), Get_pdf_yoffset());
   c_line_width_set(1.2);
   c_line_cap_set(LINE_CAP_STANDARD);
   c_line_join_set(LINE_JOIN_STANDARD);
   c_miter_limit_set(2.0);
   //c_line_type_set();
   set_color(1.0);
   // initialize bbox
   bbox_llx = bbox_lly = 1e5;
   bbox_urx = bbox_ury = -1e5;
}




void
Write_grestore(void)
{
   fprintf(TF, "Q\n");
}


void
End_Axis_Standard_State(void)
{
   Write_grestore();
}


void
Write_gsave(void)
{
   fprintf(TF, "q\n");
}


void
c_pdf_gsave()
{
   Write_gsave();
}


void
c_pdf_grestore()
{
   Write_grestore();
}


static void
Print_Xref(long int offset)
{
   char line[80];
   int i, len;
   snprintf(line,sizeof(line), "%li", offset);
   len = strlen(line);
   for (i=0; i < 10-len; i++) fputc('0', OF);
   fprintf(OF, "%s 00000 n \n", line);
}


static void
Write_Stream()
{
   long int len = ftell(TF);
   unsigned long int new_len = (len * 11) / 10 + 100;
   unsigned char *buffer, *dest_buffer;
   rewind(TF);
   buffer = calloc(len+1,sizeof(unsigned char));
   dest_buffer = calloc(new_len+1,sizeof(unsigned char));
   fread(buffer, 1, len, TF);
   fclose(TF);
//   if (FLATE_ENCODE) {
//      if (do_flate_compress(dest_buffer, &new_len, buffer, len) != FLATE_OK) {
//         free(buffer); free(dest_buffer);
//         RAISE_ERROR("Error compressing PDF stream data"); 
//         return;
//      }
//      fwrite(dest_buffer, 1, new_len, OF);
//   }
//   else {
      fwrite(buffer, 1, len, OF);
//   }
   free(buffer);
   free(dest_buffer);
}


void
Close_pdf()
{
   int i;
   double llx, lly, urx, ury, xoff, yoff;
   if (!writing_file) {
      RAISE_ERROR("Sorry: cannot End_Output if not writing file.");
      return;
   }
   writing_file = false;
   if (constructing_path) {
      RAISE_ERROR("Sorry: must finish with current path before ending file");
      return;
   }
   Write_Stream();
   stream_end = ftell(OF);
   fprintf(OF, "endstream\nendobj\n");
   Record_Object_Offset(PAGE_OBJ);
   fprintf(OF, "%i 0 obj <<\n/Type /Page\n/Parent %i 0 R\n/MediaBox [ ",
           PAGE_OBJ, PAGES_OBJ);
   if (bbox_llx < page_left) bbox_llx = page_left;
   if (bbox_lly < page_bottom) bbox_lly = page_bottom;
   if (bbox_urx > page_left + page_width)
      bbox_urx = page_left + page_width;
   if (bbox_ury > page_bottom + page_height)
      bbox_ury = page_bottom + page_height;
//#define MARGIN 3
#define MARGIN 0
   xoff = Get_pdf_xoffset();
   yoff = Get_pdf_yoffset();
   llx = bbox_llx / ENLARGE + xoff - MARGIN;  // convert back to points
   lly = bbox_lly / ENLARGE + yoff - MARGIN;
   urx = bbox_urx / ENLARGE + xoff + MARGIN;
   ury = bbox_ury / ENLARGE + yoff + MARGIN;
   if (urx < llx || ury < lly) {
      RAISE_ERROR("Sorry: Empty plot!");
      return;
   }
   fprintf(OF, "%d %d %d %d", ROUND(llx), ROUND(lly), ROUND(urx), ROUND(ury));
   fprintf(OF, " ]\n/Contents %i 0 R\n/Resources << "
           "/ProcSet [/PDF /Text /ImageB /ImageC /ImageI]\n", STREAM_OBJ);
   fprintf(OF, "  >>\n"); // end of /Resources
   fprintf(OF, ">> endobj\n");
   Record_Object_Offset(CATALOG_OBJ);
   fprintf(OF, "%i 0 obj <<\n/Type /Catalog\n/Pages %i 0 R\n>> endobj\n",
           CATALOG_OBJ, PAGES_OBJ);
   xref_offset = ftell(OF);
   fprintf(OF, "xref\n0 %li\n0000000000 65535 f \n", num_objects);
   for (i = 1; i < num_objects; i++)
      Print_Xref(obj_offsets[i]); // NB: DONT USE OBJECT 0
   fprintf(OF, "trailer\n<<\n/Size %li\n/Root %i 0 R\n/Info %i 0 "
           "R\n>>\nstartxref\n%li\n%%%%EOF\n",
           num_objects, CATALOG_OBJ, INFO_OBJ, xref_offset);
   fseek(OF, length_offset, SEEK_SET);
   fprintf(OF, "%li", stream_end - stream_start);
   fclose(OF);
}


void
Rename_pdf(char *oldname, char *newname)
{
   char old_ofile[300], new_ofile[300];
   Get_pdf_name(old_ofile, oldname, 300);
   Get_pdf_name(new_ofile, newname, 300);
   rename(old_ofile, new_ofile); // from stdio.h
}







//**************************************








static void swap_coords(int *x1, int *y1, int *x2, int *y2) {
  int tmp;

  tmp = *x1;
  *x1 = *x2;
  *x2 = tmp;

  tmp = *y1;
  *y1 = *y2;
  *y2 = tmp;

}

static void start_plot( float w, float h) {
   double width, height;
   width = w;
   height = h;
   c_set_device_pagesize(width, height);
   Open_pdf(filename);
}

static void end_plot() {
  if (error == true)
	return;
   Close_pdf();
}

static void make_device_active(float devnum) {
}


static void initialize_default_colortable(void) {
  int i;
  float half_colors[] = { 128, 128, 128 };

  memcpy(default_colortable, base_colors, 3 * 16 * sizeof(float) );

  for (i=16; i<NCOLORS; i++)
	memcpy( default_colortable + (i * 3), half_colors, 3 * sizeof(float) );
}


static void draw_line( int I0, int J0, int I1, int J1) {
   //printf("driver draw_line i1 %i  j0 %i i0 %i  j1 %i\n", I0, J0, I1, J1);
  if (error == true)
	return;
	if (I0 == LASTI && J0 == LASTJ) {
      if (I0 == I1 && J0 == J1) return;
      NSEG++;
   } else {
      NSEG = 1;
      c_moveto(I0, J0);
   }
   c_lineto(I1, J1);
   c_stroke();
   c_moveto(I1, J1);
   LASTI = I1;
   if (NSEG > 200) LASTI = -1;
   LASTJ = J1;
}


static void fill_dot( float I1, float J1) {
   //printf("driver fill dot i1 %i  j1 %i\n", I1, J1);
   c_line_cap_set(LINE_CAP_ROUND);
   draw_line(I1, J1, I1, J1);
   c_line_cap_set(LINE_CAP_STANDARD);
}


static void polygon_fill( int I0, int J0 ) {
  if (NPTS == 0) {
     NPTS = I0;
     START = true;
  } else {
     NPTS--;
     if (START) {
        c_moveto(I0, J0);
        START = false;
        LASTI = I0;
        LASTJ = J0;
     } else if (NPTS == 0) {
        c_lineto(I0, J0);
        c_close_path();
        c_fill();
        LASTI = -1;
        LASTJ = -1;
     } else {
        c_lineto(I0, J0);
        LASTI = I0;
        LASTJ = J0;
     }
  }
}


static void open_new_device(char *file, int length, float *id, float *err, int mode) {
   if (writing_file) {
      printf("Sorry -- can only write one pdf file at a time\n");
      *err = 0.0;
      return;
   }
   writing_file = true;
  filename[length] = '\0';
  strncpy(filename,file,length);
  initialize_device_ctable();
  NPTS = 0;
  *id = 1.0;
  *err = 1.0;
   Init_pdf();
}

static void close_device(  ) {
   writing_file = false;
  Shutdown_pdf();
}

void PDF_driver(int *opcode, float *rbuf, int *nbuf, char *chr, int *lchr, int *mode, int len) {

  static int firsttime = 1;

  /* text used in warning messages */
  pdf_ident = PDF_IDENT;

  if (firsttime) {
	initialize_default_colortable();
	firsttime = 0;
  }

  //printf("opcode %i\n", *opcode);

  switch (*opcode) {

	/* Return device name */
  case 1:
	{
	  int i;
	  char *name;
	  name = DEVICE_NAME;

	  strncpy(chr,name,len);
	  *lchr = strlen(name);
	  for (i=*lchr; i<len; i++)
		chr[i] = ' ';
	};
	break;

	/* min and max dimensions of plot device, color indicies */
  case 2:
     rbuf[0] = 0.0;
     rbuf[1] = -1.0;
     rbuf[2] = 0.0;
     rbuf[3] = -1.0;
     rbuf[4] = 0.0;
     rbuf[5] = 255.0;
     *nbuf = 6;
	break;

	/* return device scale */
  case 3:
        rbuf[0] = DPI;   /* device coordinates per inch */
        rbuf[1] = DPI;
        rbuf[2] = 1.0;		/* Device coordinates per pixel */
        *nbuf = 3;
	*nbuf = 3;
	break;

	/* return device capabilities */
  case 4:
     chr[0] = 'H'; /* Hardcopy device */
     chr[1] = 'N'; /* Cursor is not available */
     chr[2] = 'N'; /* no dashed lines */
     chr[3] = 'A'; /* polygon fill available */
     chr[4] = 'T'; /* Thick lines available*/
     chr[5] = 'N'; /* no rectangle fill; do polygon instead */
     chr[6] = 'Q'; /* images (same as psdriv) */
     chr[7] = 'N'; /* Do not prompt user on close */
     chr[8] = 'Y'; /* Can return color representation */
     chr[9] = 'N'; /* no markers */                     // M if support markers
     chr[10] = 'N'; /* no scrolling */
     *lchr = 11;
	break;

	/* return default device filename */
  case 5:
	*lchr = strlen(DEFAULT_FILENAME);
	strncpy(chr,DEFAULT_FILENAME,*lchr);
	break;

	/* default edge coordinates of view surface */
  case 6:
	{
      rbuf[0] = 0.0;
      rbuf[1] = PAGE_WIDTH*DPI;
      rbuf[2] = 0.0;
      rbuf[3] = PAGE_HEIGHT*DPI;
      *nbuf = 4;
	};
	break;

	/* scale factor of obsolete character set */
  case 7:
	rbuf[0] = 1.0;
	*nbuf = 1;
	break;

	/* select active device */
  case 8:
	make_device_active(rbuf[0]);
	break;

	/* open device */
  case 9:
	open_new_device(chr,*lchr,&rbuf[0],&rbuf[1],*mode);
	break;

	/* close device */
  case 10:
	close_device();
	break;

	/* begin picture */
  case 11:
	start_plot(rbuf[0] + 1, rbuf[1] + 1);
	break;

	/* draw a line */
  case 12:
	draw_line(
			  (int)rbuf[0],
			  (int)rbuf[1],
			  (int)rbuf[2],
			  (int)rbuf[3]
			  );
	break;

	/* fill dot */
  case 13:
	fill_dot((int)rbuf[0], (int)rbuf[1]);
	break;

	/* end picture */
  case 14:
	end_plot();
	break;

	/* set current color */
  case 15:
   //printf("driver: set color %g\n", rbuf[0]);
   set_color(rbuf[0]);
	break;

	/* flush buffer */
  case 16:
	break;

	/* erase alpha (text) screen */
  case 18:
	break;

	/* polygon fill */
  case 20:
	polygon_fill((int)rbuf[0], (int)rbuf[1]);
	break;

	/* set color representation */
  case 21:
	set_color_rep(
				  rbuf[0],
				  rbuf[1]*255.0,
				  rbuf[2]*255.0,
				  rbuf[3]*255.0
				  );
	break;

	/* Set line width */
  case 22:
   c_line_width_set(rbuf[0]);
	break;

	/* escape function */
  case 23:
	break;

	/* Image */
  case 26:
     printf("sorry: images not supported.\n");
	break;

	/* query color representation */
  case 29:
	{
	  float r, g, b;
	  get_color_rep(rbuf[0], &r, &g, &b );

	  rbuf[1] = r / 255.0;
	  rbuf[2] = g / 255.0;
	  rbuf[3] = b / 255.0;
	  *nbuf = 4;
	};
	break;

  default:
	fprintf(stderr,"%s: unhandled opcode = %d\n", pdf_ident, *opcode);

  }
}




// ---------------------------------------------------------------





// ---------------------------------------------------------------






#ifdef _WIN32
#define CGDRIV CGDRIV
#elif defined(PG_PPU)
#define CGDRIV cgdriv_
#else
#define CGDRIV cgdriv
#endif


void CGDRIV(ifunc, rbuf, nbuf, chr, lchr, mode, len)
    int *ifunc, *nbuf, *lchr, *mode;
    int len;
    float rbuf[];
    char *chr;
{
   PDF_driver(ifunc, rbuf, nbuf, chr, lchr, mode, len);
}

