

! Q: A lot of names are "log" something -- natural or base 10 log?
! A: I use "ln" for natural log and "log" for base 10.


! Q: How is the "center" of a zone defined? is it the center by mass, or radius?
! A: Extensive variables (R, L, M, velocity) are defined at cell boundaries.
!    Intensive variables (T, rho, composition) are cell averages. 
!    We use the cell average as an estimate of the point value at the cell center by mass.
!   The themo variables are cell averages since they are calculated as functions of the cell averages.
!   Note however that gradr and gradT are not thermo variables.
!   They are defined at cell boundaries and depend on both neighbors.
!   If you are comparing to results from a model that doesn't have staggered variables
!   (i.e., one that defines the intensive and extensive vars at the same place), then
!   you will need to adjust.   For example, to compare our sound speed profile to
!   published results, we interpolate the cell centered csound values to find values
!   at cell boundaries.


! Q: Is there a simple way of defining a "good" derivative on this grid?
!    In particular, how should I compute dT/dr and dP/dr?
! A: in star_data.inc, we find these
!           ! gradients for structure equations
!         real(dp), pointer :: dlnP_dm(:)
!         real(dp), pointer :: dlnT_dm(:)
!      These are defined at cell boundaries:
!      	dlnP_dm(k) relates lnP(k-1) to lnP(k) at the boundary between cells k-1 (outer) and k (inner).
!
!    to convert this to dP_dm(k), we need P at the cell boundary k.
!    but since pressure is cell centered, this means we need to interpolate in m.
!    The code actually uses q == m/Mstar, and this is the interpolation equation:
!                 alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
!               P = alfa*s% P(k) + (1-alfa)*s% P(k-1)
!      so now we have dP_dm(k) = P*dlnP_dm(k)
!      Next we need dm_dr at cell boundary k, so we need
!    the density at the boundary, which we get by doing the same
!    interpolation as we just did for P.
!
!               rho = alfa*s% rho(k) + (1-alfa)*s% rho(k-1)
!      Since r is defined at cell boundaries, we can just use r(k) -- no interpolation.
!
!    		dm_dr(k) = 4*pi*r(k)**2*rho
!      combining all this gives
!      	dP_dr(k) = P*dlnP_dm(k)*dm_dr(k)
!      and we haven't used any differences -- that should help the quality of the results.
!      For dT/dr, ... well you get it.


   ! the number of models that are currently stored
      integer :: generations ! between 1 and 3
      ! 1 means have current model only
      ! 2 means have current and old
      ! 3 means have current, old, and older

   ! state variables for up to 3 generations
   
      ! the number of zones in the model
      integer :: nz, nz_old, nz_older

      ! lnPgas_flag is true if we are using lnPgas variables in place of lnd
      logical :: lnPgas_flag

      ! v_flag is true if we are using velocity variables
      logical :: v_flag

      ! rotation_flag is true if we are using rotation variables
      logical :: rotation_flag

      integer :: nvar_hydro ! the number of structure variables per zone
      
      integer :: species ! species = number of isotopes in the current net

      integer :: nvar_chem ! the number of chemical abundance variables per zone
      ! = 0 for operator splitting with separate evolution of abundances
      ! = species to solve for all abundances along with structure

      integer :: nvar ! = nvar_hydro + nvar_chem
      
      ! save a few things from start of step that will need later
      real(dp) :: prev_Lmax, prev_Ledd, surf_r_equatorial, surf_opacity, surf_csound, surf_rho

      ! nuclear reaction net info

      ! net_name identifies which net we are currently using
      character (len=net_name_len) :: net_name

      ! num_reactions = number of reactions in the current net
      integer :: num_reactions
   
      integer :: model_number, model_number_old, model_number_older
   
      ! total baryonic mass (grams)
      real(dp) :: mstar, mstar_old, mstar_older
      real(dp) :: mstar_dot, mstar_dot_old, mstar_dot_older ! (gm/second)
   
      ! total angular momentum
      real(dp) :: total_angular_momentum, total_angular_momentum_old, total_angular_momentum_older
      real(dp) :: angular_momentum_removed ! total removed by mass loss in the current step

      
      ! center boundary
      
      ! typically, mesa/star models an entire star and the center boundary has M = 0, R = 0, and L = 0.
      ! however, in some situations it is useful to model only the envelope
      ! and use a nonzero center boundary.
      real(dp) :: M_center ! total baryonic mass (g) interior to innermost cell (usually 0)
      real(dp) :: R_center ! radius (cm) of inner edge of innermost cell (usually 0)
      real(dp) :: L_center ! luminosity (erg/s) at inner edge of innermost cell (usually 0)
      ! the main reason for using a nonzero center boundary is to resolve small features in the envelope.
      ! the full model (with 0 center boundary) includes the envelope of course, but because of the
      ! limits of real(dp) we are limited to resolving envelope cell masses of about 10^-14*mstar.
      ! If that is adequate for your problem, then there is no reason to use a nonzero center boundary.
      ! But if you need to resolve smaller cell masses and you can ignore the center details,
      ! then a nonzero center can be the answer.  The code is written so that it can resolve cell masses
      ! down to about 10^-14*xmstar, where xmstar is the mass of the envelope exterior to M_center.
      ! So, for example, if xmstar = 10^-16*mstar,
      ! you can now resolve down to 10^-30*mstar in the envelope.
      ! The nonzero center boundaries are a big help with mass resolution,
      ! but they do not improve radial resolution -- that is still limited to about 10^-14*R.
      ! The remeshing algorithm knows that it must not split cells that are near that limit.
      ! In practice, this limit on radial resolution has not been a issue.
      ! For example, with R of 10 km for a neutron star, the ability to resolve 10^-8 cm is quite enough!
      ! However for mass resolution with a 1.6 Msun neutron star, we need to resolve small mass cells
      ! down to 10^10 g or less while the smallest amount of the total mass we can resolve is ~ 10^19 g.
      ! So for the NS envelope case, we need to use a nonzero center boundary.

      real(dp) :: xmstar, xmstar_old, xmstar_older ! baryonic mass external to M_center
         ! thus mstar = M_center + xmstar

      ! elapsed time since start of run (seconds)
      real(dp) :: time, time_old, time_older  

      ! timesteps between generations (seconds)
      real(dp) :: dt, dt_old

      ! dq(k) is the fraction of xmstar between points k and k+1
      ! NOTE: near the surface, dq's can be extremely small
      ! the code should only use q(k)-q(k+1) in cases where 0 is an okay value.
      real(dp), pointer, dimension(:) :: dq, dq_old, dq_older

      ! q is defined in terms of dq: q(1) = 1.0; for nz > k > 1, q(k) = q(k-1) - dq(k-1); q(nz) = dq(nz)
      real(dp), pointer, dimension(:) :: q, q_old, q_older
   
      ! composition variables: xa(j,k) holds mass fraction of species j for location k
      ! j is an integer from 1 to s% species
      ! chem_id(j) gives the iso identifier from chem_def
      ! net_iso(chem_id(j)) = j
      real(dp), pointer, dimension(:,:) :: xa, xa_old, xa_older
        
      ! vectors of structure variables: xh(i,k) holds hydro structure variable i for point k.
         ! note: abundances are in xa, not in xh
      real(dp), pointer, dimension(:,:) :: xh, xh_old, xh_older ! (nvar,nz)
      
      ! luminosities
      real(dp) :: L_phot, L_phot_old, L_phot_older ! photosphere luminosity (Lsun units)
         
      ! power emitted from nuclear reactions after subtracting neutrinos (Lsun units)
      real(dp) :: L_nuc_burn_total, L_nuc_burn_total_old, L_nuc_burn_total_older
      real(dp), dimension(num_categories) :: L_by_category, L_by_category_old, L_by_category_older
      real(dp), dimension(num_categories) :: &
         center_eps_burn, center_eps_burn_dT, center_eps_burn_dRho
   
      ! variation control value for hydro timestep controller
      real(dp) :: varcontrol, varcontrol_old, varcontrol_older
      
      ! surface radial velocity
      real(dp) :: v_surf, v_surf_old, v_surf_older ! (cm/second)
      
      ! surface 
      
      real(dp) :: omega_avg_surf
      real(dp) :: omega_crit_avg_surf
      real(dp) :: omega_div_omega_crit_avg_surf
      real(dp) :: j_rot_avg_surf
      real(dp) :: v_rot_avg_surf ! cm/s
      real(dp) :: v_crit_avg_surf
      real(dp) :: v_div_v_crit_avg_surf
      
      real(dp) :: opacity_avg_surf
      real(dp) :: logT_avg_surf
      real(dp) :: logRho_avg_surf
      
      ! for time smoothing and other things that need value from previous step
      real(dp) :: L_div_Ledd_avg_surf, L_div_Ledd_avg_surf_old, L_div_Ledd_avg_surf_older
      real(dp), pointer, dimension(:) :: conv_vel_old, conv_vel_older
      real(dp), pointer, dimension(:) :: Del_T_old, Del_T_older
      real(dp), pointer, dimension(:) :: eps_grav_old, eps_grav_older
      real(dp), pointer, dimension(:) :: nu_ST_old, nu_ST_older
      real(dp), pointer, dimension(:) :: D_ST_old, D_ST_older
      real(dp), pointer, dimension(:) :: D_DSI_old, D_DSI_older
      real(dp), pointer, dimension(:) :: D_SH_old, D_SH_older
      real(dp), pointer, dimension(:) :: D_SSI_old, D_SSI_older
      real(dp), pointer, dimension(:) :: D_ES_old, D_ES_older
      real(dp), pointer, dimension(:) :: D_GSF_old, D_GSF_older
      

      ! tau surface
      real(dp) :: tau_base, tau_factor
         ! tau_base is the default optical depth at the base of the atmosphere
         ! value depends on the selected option for the atmosphere model
         ! actual optical depth at base of atmosphere is tau_factor*tau_base
         ! tau_factor is usually 1.0,
            ! but can be changed to be larger or smaller for special applications.
         ! if tau_factor is < 1,
            ! then the code will automatically switch to integration using eddington gray
         ! if tau_factor is > 1, then use simple eddington grey approximation
         ! if tau_factor == 1, 
            ! use the specified atmosphere model to find T and P at optical depth = tau_base
            
      real(dp) :: T_surf, P_surf ! results from atm

   
      ! summary information updated at the end of each step

      ! age
      real(dp) :: star_age ! age (in years)
      real(dp) :: time_step ! timestep since previous model (in years)
   
   
      ! surface (outermost cell in model)
      real(dp) :: log_surface_luminosity ! log10(luminosity in Lsun units)
      real(dp) :: log_surface_radius ! log10(total radius in Rsun units)
      real(dp) :: log_surface_temperature ! log10(temperature)   ---- note: not necessarily Teff
      real(dp) :: log_surface_pressure ! log10(pressure)
      real(dp) :: log_surface_density ! log10(density)
      
      ! fractional abundances by mass near surface
         ! averages over a fraction = 'surface_avg_abundance_dq' of star mass
      real(dp) :: surface_h1
      real(dp) :: surface_he3
      real(dp) :: surface_he4
      real(dp) :: surface_c12
      real(dp) :: surface_n14
      real(dp) :: surface_o16
      real(dp) :: surface_ne20
   
      ! photosphere
      real(dp) :: photosphere_L ! Lsun units
      real(dp) :: photosphere_r ! Rsun units
      real(dp) :: photosphere_acoustic_r ! sound travel time from center to photosphere (seconds)
      real(dp) :: Teff

      ! center region -- averages over center_avg_value_dq
      real(dp) :: log_max_temperature ! log10(max temperature in kelvin)
      real(dp) :: log_center_temperature ! log10(center temperature in kelvin)
      real(dp) :: log_center_density ! log10(center density in g/cm^3)
      real(dp) :: log_center_pressure ! log10(center pressure in dynes/cm^2)
      real(dp) :: center_degeneracy 
         ! the electron chemical potential in units of k*T
      real(dp) :: center_gamma 
      real(dp) :: center_abar 
      real(dp) :: center_zbar 
      real(dp) :: center_mu 
      real(dp) :: center_ye 
      real(dp) :: center_eps_grav 
      real(dp) :: center_eps_nuc, d_center_eps_nuc_dlnT, d_center_eps_nuc_dlnd
      
      real(dp) :: center_non_nuc_neu 
      real(dp) :: center_dL_dm 
      real(dp) :: center_entropy ! in kerg per baryon 
      real(dp) :: iron_core_infall 
         ! max infall velocity at any location with Ye < 0.485 (cm/sec)
      real(dp) :: oxygen_core_infall 
         ! max infall velocity at any location inside where 1st have o16 > 0.5 by mass
      real(dp) :: center_omega
      real(dp) :: center_omega_div_omega_crit 
         
      real(dp) :: center_h1  ! fractional abundance by mass
      real(dp) :: center_he3
      real(dp) :: center_he4
      real(dp) :: center_c12
      real(dp) :: center_n14
      real(dp) :: center_o16
      real(dp) :: center_ne20
      real(dp) :: center_si28

      ! total baryonic mass
      real(dp) :: star_mass ! total stellar baryonic mass (Msun)
      real(dp) :: star_mdot ! d(star_mass)/dt (Msun/year)
      real(dp) :: star_mass_h1
      real(dp) :: star_mass_he3
      real(dp) :: star_mass_he4
      real(dp) :: star_mass_c12
      real(dp) :: star_mass_n14
      real(dp) :: star_mass_o16
      real(dp) :: star_mass_ne20
            
      real(dp) :: mass_conv_core ! (Msun) 0 if core is radiative

      ! element boundaries
      
      real(dp) :: he_core_mass ! baryonic (Msun)
      real(dp) :: he_core_radius ! Rsun
      real(dp) :: he_core_lgT
      real(dp) :: he_core_lgRho
      real(dp) :: he_core_L ! Lsun
      real(dp) :: he_core_v
      real(dp) :: he_core_omega ! (s^-1)
      real(dp) :: he_core_omega_div_omega_crit
      
      real(dp) :: c_core_mass ! baryonic (Msun)
      real(dp) :: c_core_radius ! Rsun
      real(dp) :: c_core_lgT
      real(dp) :: c_core_lgRho
      real(dp) :: c_core_L ! Lsun
      real(dp) :: c_core_v
      real(dp) :: c_core_omega ! (s^-1)
      real(dp) :: c_core_omega_div_omega_crit
      
      real(dp) :: o_core_mass ! baryonic (Msun)
      real(dp) :: o_core_radius ! Rsun
      real(dp) :: o_core_lgT
      real(dp) :: o_core_lgRho
      real(dp) :: o_core_L ! Lsun
      real(dp) :: o_core_v
      real(dp) :: o_core_omega ! (s^-1)
      real(dp) :: o_core_omega_div_omega_crit
      
      real(dp) :: si_core_mass ! baryonic (Msun)
      real(dp) :: si_core_radius ! Rsun
      real(dp) :: si_core_lgT
      real(dp) :: si_core_lgRho
      real(dp) :: si_core_L ! Lsun
      real(dp) :: si_core_v
      real(dp) :: si_core_omega ! (s^-1)
      real(dp) :: si_core_omega_div_omega_crit

      real(dp) :: fe_core_mass ! baryonic (Msun)
      real(dp) :: fe_core_radius ! Rsun
      real(dp) :: fe_core_lgT
      real(dp) :: fe_core_lgRho
      real(dp) :: fe_core_L ! Lsun
      real(dp) :: fe_core_v
      real(dp) :: fe_core_omega ! (s^-1)
      real(dp) :: fe_core_omega_div_omega_crit
      
      
      ! values at trace_mass_location
      real(dp) :: trace_mass_radius
      real(dp) :: trace_mass_lgT
      real(dp) :: trace_mass_lgRho
      real(dp) :: trace_mass_L
      real(dp) :: trace_mass_v
      real(dp) :: trace_mass_omega ! (s^-1)
      real(dp) :: trace_mass_omega_div_omega_crit
      real(dp) :: trace_mass_lgP
      real(dp) :: trace_mass_g
      real(dp) :: trace_mass_X
      real(dp) :: trace_mass_Y
      real(dp) :: trace_mass_edv_H
      real(dp) :: trace_mass_edv_He
      real(dp) :: trace_mass_scale_height
      real(dp) :: trace_mass_dlnX_dr
      real(dp) :: trace_mass_dlnY_dr
      real(dp) :: trace_mass_dlnRho_dr
      
      ! values at location of max T
      real(dp) :: max_T_lgT
      real(dp) :: max_T_mass ! baryonic (Msun)
      real(dp) :: max_T_radius ! (Rsun)
      real(dp) :: max_T_lgRho
      real(dp) :: max_T_L ! (Lsun)
      real(dp) :: max_T_eps_nuc ! (erg/g/s)
      
      
      real(dp) :: tau10_mass ! baryonic mass in solar units where optical depth = 10
      real(dp) :: tau10_radius ! radius in solar units where optical depth = 10
      real(dp) :: tau10_lgP ! estimate for log10(P) at tau = 10
      real(dp) :: tau10_lgT ! estimate for log10(T) at tau = 10
      real(dp) :: tau10_lgRho ! estimate for log10(density) at tau = 10
      real(dp) :: tau10_L ! estimate for L/Lsun at tau = 10
      
      real(dp) :: tau100_mass ! baryonic location in solar units where optical depth = 100
      real(dp) :: tau100_radius ! location in solar units where optical depth = 100
      real(dp) :: tau100_lgP ! estimates for values at tau = 100
      real(dp) :: tau100_lgT
      real(dp) :: tau100_lgRho
      real(dp) :: tau100_L

      ! timescales
      real(dp) :: dynamic_timescale 
         ! dynamic timescale (seconds) -- estimated by 2*pi*sqrt(r^3/(G*m))
      real(dp) :: kh_timescale 
         ! kelvin-helmholtz timescale (years) -- proportional to graviational energy divided by luminosity
      real(dp) :: nuc_timescale 
         ! nuclear timescale (years) -- proportional to mass divided by luminosity
      real(dp) :: chem_timescale 
         ! burn+mix timescale (years)
         ! approximately min over all cells k and species i of x(i,k)/abs(dxdt_mix + dxdt_burn)
      real(dp) :: cell_collapse_timescale 
         ! approximately min over all cells k of (r(k) - r(k+1))/max(1d-99, v(k+1) - v(k))
         ! i.e., time required for cell thickness to go to zero for current velocities
   
      ! power (in Lsun units)
      real(dp) :: power_nuc_burn ! total thermal power excluding photodisintegrations
      real(dp) :: power_h_burn ! total thermal power from PP and CNO
      real(dp) :: power_he_burn ! total thermal power from triple-alpha
      real(dp) :: power_neutrinos ! power emitted in neutrinos (nuclear + thermal)
      
      
      ! mixing regions (from surface inward)
      ! a "mixing region" is a run of points with the same value of mixing_type
      ! i.e., a region of convective_mixing, overshoot_mixing,
      ! semiconvective_mixing, or thermo_haline_mixing
      integer :: num_mixing_regions ! the number of different regions in the current model
         ! from 0 to max_num_mixing_regions
      integer :: mixing_region_type(max_num_mixing_regions) ! values from mlt_def
      integer :: mixing_region_top(max_num_mixing_regions) ! index of point at top of region
      integer :: mixing_region_bottom(max_num_mixing_regions) ! index of point at bottom of region
      integer :: largest_conv_mixing_region ! index into the mixing_region arrays
      
      
      ! info about the max burning locations for hydrogen, helium, and metals
      
      ! info about location where have max rate of hydrogen burning by PP and CNO
      real(dp) :: max_eps_h ! erg/g/s
      real(dp) :: max_eps_h_m ! baryonic mass coordinate at location of max burn (Msun units)
      real(dp) :: max_eps_h_lgT ! log10 temperature at location of max burn
      real(dp) :: max_eps_h_lgRho ! log10 density at location of max burn
      real(dp) :: max_eps_h_lgR ! log10 radius at location of max burn
      real(dp) :: max_eps_h_lgP ! log10 pressure at location of max burn
      real(dp) :: max_eps_h_opacity ! opacity at location of max burn
      real(dp) :: max_eps_h_cp ! cp at location of max burn
      integer :: max_eps_h_k ! cell number
      
      ! info about location where have max rate of helium burning by triple alpha
      real(dp) :: max_eps_he ! erg/g/s
      real(dp) :: max_eps_he_m ! baryonic mass coordinate at location of max burn (Msun units)
      real(dp) :: max_eps_he_lgT ! log10 temperature at location of max burn
      real(dp) :: max_eps_he_lgRho ! log10 density at location of max burn
      real(dp) :: max_eps_he_lgR ! log10 radius at location of max burn
      real(dp) :: max_eps_he_lgP ! log10 pressure at location of max burn
      real(dp) :: max_eps_he_opacity ! opacity at location of max burn
      real(dp) :: max_eps_he_cp ! cp at location of max burn
      integer :: max_eps_he_k ! cell number
      
      ! info about location where have max rate for burning other than h and he burning
      real(dp) :: max_eps_z ! erg/g/s
      real(dp) :: max_eps_z_m ! baryonic mass coordinate at location of max burn (Msun units)      
      real(dp) :: max_eps_z_lgT ! log10 temperature at location of max burn
      real(dp) :: max_eps_z_lgRho ! log10 density at location of max burn
      real(dp) :: max_eps_z_lgR ! log10 radius at location of max burn
      real(dp) :: max_eps_z_lgP ! log10 pressure at location of max burn
      real(dp) :: max_eps_z_opacity ! opacity at location of max burn
      real(dp) :: max_eps_z_cp ! cp at location of max burn
      integer :: max_eps_z_k ! cell number
      
      ! info about location where have max rate for burning of all kinds
      real(dp) :: max_eps_nuc ! erg/g/s
      real(dp) :: max_eps_nuc_m ! baryonic mass coordinate at location of max burn (Msun units)      
      real(dp) :: max_eps_nuc_lgT ! log10 temperature at location of max burn
      real(dp) :: max_eps_nuc_lgRho ! log10 density at location of max burn
      real(dp) :: max_eps_nuc_lgR ! log10 radius at location of max burn
      real(dp) :: max_eps_nuc_lgP ! log10 pressure at location of max burn
      real(dp) :: max_eps_nuc_opacity ! opacity at location of max burn
      real(dp) :: max_eps_nuc_cp ! cp at location of max burn
      integer :: max_eps_nuc_k ! cell number
      
      ! top & bottom q of 2 largest mixing regions (mx1 the largest; mx2 the 2nd largest)
      ! conv_mx for convective mixing region; mx for any kind of mixing
      real(dp) :: & ! in Mstar units --- i.e., q values, m/Mstar
         conv_mx1_top, conv_mx1_bot, conv_mx2_top, conv_mx2_bot, &
         mx1_top, mx1_bot, mx2_top, mx2_bot
      ! top & bottom r (in Rsun units) of 2 largest mixing regions
      real(dp) :: &
         conv_mx1_top_r, conv_mx1_bot_r, conv_mx2_top_r, conv_mx2_bot_r, &
         mx1_top_r, mx1_bot_r, mx2_top_r, mx2_bot_r
      
      ! burn zones
         ! up to 3 zones where eps_nuc > burn_min1 erg/g/s
         ! for each zone have 4 numbers: start1, start2, end2, end1
         ! start1 is mass of inner edge where first goes > burn_min1 (or -20 if none such)
         ! start2 is mass of inner edge where first zone reaches burn_min2 erg/g/sec (or -20 if none such)
         ! end2 is mass of outer edge where first zone drops back below burn_min2 erg/g/s
         ! end1 is mass of outer edge where first zone ends (i.e. eps_nuc < burn_min1)
         ! similar for second and third zones
      real(dp) :: burn_zone_mass(4,3)
      

      ! times
      real(dp) :: dt_next
      real(dp) :: dt_start ! dt during first try for step
      
      integer, pointer :: chem_id(:) ! maps species to chem id
         ! index from 1 to species in current net
         ! value is between 1 and num_chem_isos         
      integer, pointer :: net_iso(:) ! maps chem id to species number
         ! index from 1 to num_chem_isos
         ! value is 0 if the iso is not in the current net
         ! else is value between 1 and number of species in current net

        
      ! lagrangian time derivatives
      real(dp), pointer :: dlnd_dt(:) ! only used when lnPgas_flag is false
      real(dp), pointer :: dlnPgas_dt(:) ! only used when lnPgas_flag is true
      real(dp), pointer :: dlnT_dt(:)
      real(dp), pointer :: dlnR_dt(:)
      real(dp), pointer :: dv_dt(:)

      ! structure variables
      real(dp), pointer :: m(:) ! baryonic mass coord
         ! m(k) is enclosed baryonic mass at outer edge of cell k
         ! m(k) = s% M_center + s% q(k)*s% xmstar
      real(dp), pointer :: dm(:)
         ! dm(k) is baryonic mass of cell k
         ! dm(k) = s% dq(k)*s% xmstar
      
      real(dp), pointer :: dm_bar(:) 
         ! baryonic mass associated with cell edge
         ! for k=2,nz-1 dm_bar(k) = (dm(k-1)+dm(k))/2
         ! dm_bar(1) = dm(1)/2
         ! dm_bar(nz) = dm(nz-1)/2 + dm(nz)
      
      real(dp), pointer :: T(:) ! Temperature
      real(dp), pointer :: lnT(:) ! log(T)
      real(dp), pointer :: rho(:) ! density
      real(dp), pointer :: lnd(:) ! log(rho)
      real(dp), pointer :: Pgas(:) ! gas pressure (Pgas = Pion + Pelec)
      real(dp), pointer :: lnPgas(:) ! log(Pgas)
      real(dp), pointer :: L(:)
         ! L(k) is luminosity at outer edge of cell k
      real(dp), pointer :: r(:)
         ! r(k) is radius at outer edge of cell k
      real(dp), pointer :: lnR(:)
         ! lnR(k) is log(r(k))
      real(dp), pointer :: v(:) ! only used when v_flag is true
         ! v(k) is velocity at outer edge of cell k

      ! composition information -- cell average values
      
      real(dp), pointer :: X(:) ! mass fraction hydrogen
      real(dp), pointer :: Y(:) ! mass fraction helium
      real(dp), pointer :: abar(:) ! average atomic weight
      real(dp), pointer :: zbar(:) ! average charge per nucleus
      real(dp), pointer :: z2bar(:) ! average charge squared per nucleus
      real(dp), pointer :: ye(:)
         ! ye is mean number free electrons per nucleon, assuming complete ionization
         
      ! gravitational vs baryonic mass
      real(dp), pointer :: mass_correction(:) ! = dm_gravitational/dm_baryonic
         ! calculated by chem module
      real(dp), pointer :: m_grav(:) 
         ! enclosed gravitational mass at cell outer edge
         ! the variable called "m" is the enclosed baryonic mass
            ! = number of enclosed baryons * atomic mass unit
         ! NOTE: if use_mass_corrections = .false., 
         ! then m_grav = baryonic mass

      
      ! eos variables -- cell average values
      
      real(dp), pointer :: Prad(:) ! radiation pressure
      real(dp), pointer :: P(:) ! total pressure (P = Prad + Pgas)
      real(dp), pointer :: lnP(:) ! log(P)
      real(dp), pointer :: lnE(:)     ! log(specific internal energy)
      real(dp), pointer :: grada(:)      ! d ln T / d ln P | adiabatic
      real(dp), pointer :: dE_dRho(:) ! dE_dRho at constant T&abundances
      real(dp), pointer :: Cv(:)         ! dE/dT at constant density&abundances
      real(dp), pointer :: Cp(:)         ! heat capacity at constant pressure&abundances
         ! Cp := dh_dT at constant P, specific heat at constant pressure
         ! where h is enthalpy, h = E + P/Rho
      real(dp), pointer :: lnS(:)   ! log(specific entropy)
      real(dp), pointer :: d_entropy_dlnd(:)
      real(dp), pointer :: d_entropy_dlnT(:)
      real(dp), pointer :: gamma1(:)      ! adiabatic coefficient.
         ! (d_ln(p)/d_ln(rho)) at constant entropy.
      real(dp), pointer :: gamma3(:)      ! gamma3 := 1 + dlnT_dlnRho at constant S, 
      real(dp), pointer :: eta(:)        ! electron degeneracy parameter
      real(dp), pointer :: theta_e(:)    ! electron degeneracy factor for graboske screening
      real(dp), pointer :: gam(:)        ! plasma interaction parameter
      real(dp), pointer :: mu(:)         
         ! mean molecular weight per particle (ions + free electrons)
      real(dp), pointer :: lnfree_e(:) ! log(number of free electrons per nucleon)
      real(dp), pointer :: chiRho(:) ! dlnP_dlnRho at constant T
      real(dp), pointer :: chiT(:) ! dlnP_dlnT at constant Rho
      real(dp), pointer :: d_eos_dlnd(:,:) ! (num_eos_basic_results,nz)
      real(dp), pointer :: d_eos_dlnT(:,:) ! (num_eos_basic_results,nz)
      real(dp), pointer :: d_eos_dabar(:,:) ! (num_eos_basic_results,nz)
      real(dp), pointer :: d_eos_dzbar(:,:) ! (num_eos_basic_results,nz)
      real(dp), pointer :: dlnRho_dlnPgas_const_T(:)    
      real(dp), pointer :: dlnRho_dlnT_const_Pgas(:)  
      ! recall: dF_dlnPgas_const_T(k) = dF_dlnd(k)*s% dlnRho_dlnPgas_const_T(k)
      !         dF_dlnT_const_Pgas(k) = dF_dlnT(k) + dF_dlnd(k)*s% dlnRho_dlnT_const_Pgas(k)
      
      ! other model variables
      real(dp), pointer :: velocity(:)
      real(dp), pointer :: csound(:)
      real(dp), pointer :: csound_at_face(:)
      real(dp), pointer :: rho_face(:) ! rho at cell boundary
      real(dp), pointer :: scale_height(:) ! cm
      real(dp), pointer :: v_div_csound(:)
      real(dp), pointer :: entropy(:) ! NOTE: kerg/baryon units
      real(dp), pointer :: grav(:) ! uses gravitational mass
      real(dp), pointer :: tau(:)
      real(dp), pointer :: dr_div_csound(:) ! sound crossing time for cell
      
      real(dp), pointer :: rmid(:) ! radius at center by mass
      real(dp), pointer :: area(:) ! at face (cm^2)
      real(dp), pointer :: Amid(:) ! at center by mass of cell (cm^2)
         ! Amid only used with artificial viscosity. not evaluated otherwise.
      real(dp), pointer :: dAmid_dlnR00(:) ! partial of Amid(k) wrt lnR(k)
      real(dp), pointer :: dAmid_dlnRp1(:) ! partial of Amid(k) wrt lnR(k+1)
      
      real(dp), pointer :: csound_init(:) ! for use in normalizing the velocity equation
         ! evaluated at start of newton iterations
      
      ! rotation variables only calculated if rotatation_flag is true
      real(dp), pointer, dimension(:) :: omega, omega_old, omega_older
         ! angular velocity at cell boundary
      real(dp), pointer :: i_rot(:) ! specific moment of interia at cell boundary
      real(dp), pointer :: j_rot(:) ! only used when rotation_flag is true
         ! j_rot(k) is specific angular momentum at outer edge of cell k; = i_rot*omega
      real(dp), pointer :: fp_rot(:) ! rotation factor for momentum
      real(dp), pointer :: ft_rot(:) ! rotation factor for temperature
      real(dp), pointer :: r_polar(:) ! radius in polar direction
      real(dp), pointer :: r_equatorial(:) ! radius in equatorial direction         
      real(dp), pointer :: am_nu(:) ! rotational viscosity at cell boundary
      real(dp), pointer :: am_sig(:) ! defined at cell centers
         ! = (4 pi r^2 rho)^2 * am_nu / dm
      real(dp) :: rotational_mdot_boost
      
      real(dp), pointer :: domega_dlnR(:) ! defined at cell boundaries
      real(dp), pointer :: richardson_number(:) ! defined at cell boundaries
      
      ! diffusion coefs defined at cell boundaries (cm^2/s)

      real(dp), pointer :: D_mix_non_rotation(:) ! from non-rotation sources
         ! convection, semiconvection, thermohaline, overshooting
         
      real(dp), pointer :: D_visc(:) ! kinematic shear viscosity
      real(dp), pointer :: D_DSI(:) ! dynamical shear instability 
      real(dp), pointer :: D_SH(:) ! Solberg-Hoiland instability
      real(dp), pointer :: D_SSI(:) ! secular shear instability 
      real(dp), pointer :: D_ES(:) ! Eddington-Sweet circulation 
      real(dp), pointer :: D_GSF(:) ! Goldreich-Schubert-Fricke instability
       
      real(dp), pointer :: D_ST(:) ! Spruit dynamo mixing diffusivity
      real(dp), pointer :: nu_ST(:) ! Spruit dynamo effective viscosity 
      real(dp), pointer :: omega_shear(:) ! max(1d-30,min(1d30,abs(domega_dlnr(k)/omega(k))))

      real(dp), pointer :: dynamo_B_r(:) ! magnetic field (Gauss)
      real(dp), pointer :: dynamo_B_phi(:) ! magnetic field (Gauss)


      ! kap
      real(dp), pointer :: opacity(:) ! this is opacity -- [note: this is not log(opacity)]
      real(dp), pointer :: d_opacity_dlnd(:)
         ! this is partial of opacity wrt natural log of density at constant T
      real(dp), pointer :: d_opacity_dlnT(:)
         ! this is partial of opacity wrt natural log of T at constant density
      real(dp), pointer :: kap_frac_Type2(:)
      real, pointer, dimension(:) :: & ! working storage
         op_mono_umesh1, op_mono_ff1, op_mono_rs1, op_mono_s1
      integer :: op_mono_nptot, op_mono_ipe, op_mono_nrad, op_mono_n
   
      ! net
      integer :: screening_mode_value ! values defined in screen_def
         ! screening_mode matches the screening_mode control parameter
      real(dp), pointer :: rate_factors(:) ! (num_reactions)
         ! when reaction rates are calculated, they are multiplied by the
         ! corresponding values in this array.  
      procedure (), pointer, nopass :: set_rate_factors => null()

      real(dp), pointer :: category_factors(:) ! (num_categories)
         ! when reaction rates are calculated, they are multiplied by the
         ! corresponding values in this array.  
         
      integer, pointer :: which_rates(:) ! (rates_reaction_id_max)  controls for rates
      procedure (), pointer, nopass :: set_which_rates => null()

      real(dp), pointer :: eps_nuc(:) ! sum of reaction_eps_nuc for all reactions in net
         ! thermal ergs per gram per second from nuclear reactions
         ! (nuclear reaction neutrinos have been subtracted)
      real(dp), pointer :: eps_nuc_categories(:,:,:) ! (num_rvs, num_catgories, nz)
      real(dp), pointer :: d_epsnuc_dlnd(:) ! partial wrt density
      real(dp), pointer :: d_epsnuc_dlnT(:) ! partial wrt temperature
      real(dp), pointer :: d_epsnuc_dx(:,:)  ! (species,nz)
         ! d_ex_dx(j, k) is partial of eps_nuc(k) wrt species(j)

      real(dp), pointer :: eps_nuc_neu_total(:) ! erg/gm/sec as neutrinos from nuclear reactions

      real(dp), pointer :: reaction_eps_nuc(:,:,:) ! (num_rvs, num_reactions, nz)

      real(dp), pointer :: dxdt_nuc(:,:) ! (species,nz)
         ! rate of change of mass fractions from nuclear reactions
         ! dxdt_nuc(j,k) is time derivative from nuclear burning of species(j) at point(k)
      real(dp), pointer :: dxdt_dRho(:,:) ! (species,nz)
         ! dxdt_dRho(j, k) is partial of dxdt_nuc(k) wrt density
      real(dp), pointer :: dxdt_dT(:,:) ! (species,nz)
         ! dxdt_dRho(j, k) is partial of dxdt_nuc(k) wrt temperature
      real(dp), pointer :: d_dxdt_dx(:,:,:) ! (species,species,nz)
         ! d_dxdt_dx(i,j,k) is partial of dxdt_nuc for species i
         ! wrt abundance of species j at location k

      real(dp), pointer :: rate_screened(:,:,:)
      real(dp), pointer :: rate_raw(:,:,:)

      ! integrated eps_nuc_categories (ergs/sec)
      real(dp), pointer :: luminosity_by_category(:,:) ! (num_catgories, nz)
      
      ! selected solar neutrino fluxes at 1 AU 
      real(dp) :: flux_pp, flux_pep, flux_be7ec, flux_b8ep ! cm^-2 s^-1
      
      
      ! nse
      real(dp), pointer :: nse_fraction(:)
      

      ! non-nuclear-reaction neutrino losses
      real(dp), pointer :: non_nuc_neu(:)
      real(dp), pointer :: d_nonnucneu_dlnd(:)
      real(dp), pointer :: d_nonnucneu_dlnT(:)
   
      real(dp), pointer :: nonnucneu_plas(:)
      real(dp), pointer :: nonnucneu_brem(:)
      real(dp), pointer :: nonnucneu_phot(:)
      real(dp), pointer :: nonnucneu_pair(:) 
      real(dp), pointer :: nonnucneu_reco(:) 
            

      ! irradiation heat profile
      real(dp), pointer :: irradiation_heat(:) ! (ergs/gm/sec)
      
      
      ! extra heat profile (can be set by user)
      real(dp), pointer :: extra_heat(:) ! (ergs/gm/sec)
      real(dp), pointer, dimension(:) :: &
         d_extra_heat_dlnd, d_extra_heat_dlnT, &
         d_extra_heat_dv00, d_extra_heat_dvp1, &
         d_extra_heat_dlnR00, d_extra_heat_dlnRp1
      
      
      ! extra factor for opacities (can be set by user)
      real(dp), pointer, dimension(:) :: extra_opacity_factor
      
      
      ! extra torque (can be set by user)
      real(dp), pointer, dimension(:) :: &
         extra_jdot, extra_omegadot, &
         d_extra_jdot_domega_m1, d_extra_omegadot_domega_m1, &
         d_extra_jdot_domega_00, d_extra_omegadot_domega_00, &
         d_extra_jdot_domega_p1, d_extra_omegadot_domega_p1
      
      
      ! gravitational constant (can be set by user)
      real(dp), pointer :: cgrav(:)
      

      ! -T*Ds/Dt energy generation rate related to change in entropy
      real(dp), pointer :: eps_grav(:)
      ! partials
      real(dp), pointer, dimension(:) :: &
         d_eps_grav_dlndm1, d_eps_grav_dlnd00, d_eps_grav_dlndp1
      real(dp), pointer, dimension(:) :: &
         d_eps_grav_dlnTm1, d_eps_grav_dlnT00, d_eps_grav_dlnTp1
      real(dp), pointer, dimension(:) :: &
         d_eps_grav_dlnR00, d_eps_grav_dL00
      real(dp), pointer, dimension(:) :: &
         d_eps_grav_dlnPgasm1_const_T, &
         d_eps_grav_dlnPgas00_const_T, &
         d_eps_grav_dlnPgasp1_const_T
      real(dp), pointer, dimension(:) :: &
         d_eps_grav_dlnTm1_const_Pgas, &
         d_eps_grav_dlnT00_const_Pgas, &
         d_eps_grav_dlnTp1_const_Pgas
      real(dp), pointer, dimension(:) :: &
         d_eps_grav_dlnRp1, d_eps_grav_dv00, d_eps_grav_dvp1
         
      real(dp), pointer :: eps_grav_composition_term(:) 
         ! only set when include_composition_in_eps_grav is true
      
      ! info from start of step, saved before remesh but after set_vars
      integer :: prev_mesh_nz
      logical :: have_prev_mesh_lnS_interpolant
      real(dp), pointer, dimension(:) :: &
         prev_mesh_xm, prev_mesh_f1_lnS
      real(dp), pointer, dimension(:,:) :: prev_mesh_f_lnS

      ! results from mlt
      real(dp), pointer :: mlt_mixing_length(:) 
         ! mixing_length_alpha * scale_height as computed by mlt
      integer, pointer :: mlt_mixing_type(:) ! as defined in mesa/const
         ! this is the value from mlt, before overshooting has been added.
      real(dp), pointer :: mlt_D(:) 
         ! local mlt estimate for diffusion coeff. (cm^2/sec)
      real(dp), pointer :: mlt_vc(:) 
         ! local mlt estimate for convection velocity
      real(dp), pointer :: mlt_Gamma(:) 
         ! local mlt convection efficiency.
      real(dp), pointer :: gradT_sub_grada(:) ! difference at cell boundary
      real(dp), pointer :: grada_at_face(:) ! grada evaluated at cell boundary
      
      real(dp) :: gradT_excess_alpha ! 1 for full efficiency boost; 0 for no boost
      real(dp) :: gradT_excess_alpha_old, gradT_excess_alpha_older
      real(dp) :: gradT_excess_min_beta
      real(dp) :: gradT_excess_max_lambda

      real(dp), pointer :: mlt_cdc(:) 
      real(dp), pointer :: Del_T(:) ! for Unno time dependent convection   
      real(dp), pointer :: cdc(:) ! convective diffusion coeff. (gm^2/sec)

      real(dp), pointer :: adjust_mlt_gradT_fraction(:)
         ! this vector can be set by other_mixing routine. defaults to -1.
         ! if f(k) k is >= 0 and <= 1, then
         ! gradT(k) from mlt is replaced by f(k)*grada_at_face(k) + (1-f(k))*gradr(k)

      ! adjusted (non-local) mixing info
      real(dp), pointer :: D_mix(:) ! diffusion coeff. (cm^2/sec)
      real(dp), pointer :: conv_vel(:) ! convection velocity   
      integer, pointer :: mixing_type(:) ! mixing types are defined in mesa/mlt  
      real(dp), pointer :: mixing_type_change_dq(:) 
         ! for cases with mixing_type(k) /= mixing_type(k+1)
         ! gives dq distance inward from outer edge of cell(k) where the transition happens.
         ! i.e., q location of transition is q(k) - mixing_type_change_dq(k)
      logical, pointer :: newly_nonconvective(:) 
      
      integer :: num_conv_boundaries ! boundaries of regions with mixing_type = convective_mixing
      ! boundaries are ordered from center to surface
      real(dp), pointer :: conv_bdy_q(:) ! (num_conv_boundaries)
         ! subcell location of boundary
      logical, pointer :: top_conv_bdy(:) ! (num_conv_boundaries)
      integer, pointer :: conv_bdy_loc(:) ! (num_conv_boundaries)
         ! if top_conv_bdy, top  of region is between loc and loc+1
         ! else bottom of region is between loc and loc-1
      logical, pointer :: burn_h_conv_region(:) ! (num_conv_boundaries)
      logical, pointer :: burn_he_conv_region(:) ! (num_conv_boundaries)
      logical, pointer :: burn_z_conv_region(:) ! (num_conv_boundaries)
      
      integer :: num_mix_boundaries ! boundaries of regions with mixing_type /= no_mixing
      ! in order from center to surface
      ! no bottom boundary at loc=nz included if center is mixed
      ! however, do include top boundary at loc=1 if surface is mixed
      integer :: num_mix_regions ! number of mix boundaries with top_mix_bdy = true
      real(dp), pointer :: mix_bdy_q(:) ! (num_mix_boundaries)
      integer, pointer :: mix_bdy_loc(:) ! (num_mix_boundaries)
      logical, pointer :: top_mix_bdy(:) ! (num_conv_boundaries)
      logical, pointer :: burn_h_mix_region(:) ! (num_conv_boundaries)
      logical, pointer :: burn_he_mix_region(:) ! (num_conv_boundaries)
      logical, pointer :: burn_z_mix_region(:) ! (num_conv_boundaries)
         
      real(dp), pointer :: actual_gradT(:) ! actual dlnT/dlnP (at boundary)
         ! (dT/T_face)/(dP/P_face) with face values intepolated by mass
         
      real(dp), pointer :: gradT(:) ! dlnT/dlnP (at boundary)
      real(dp), pointer :: d_gradT_dlnd00(:)
      real(dp), pointer :: d_gradT_dlnT00(:)
      real(dp), pointer :: d_gradT_dlndm1(:)
      real(dp), pointer :: d_gradT_dlnTm1(:)
      real(dp), pointer :: d_gradT_dlnR(:)
      real(dp), pointer :: d_gradT_dL(:)
   
      real(dp), pointer :: gradr(:) ! grad_rad (at boundary)
      real(dp), pointer :: d_gradr_dlnd00(:)
      real(dp), pointer :: d_gradr_dlnT00(:)
      real(dp), pointer :: d_gradr_dlndm1(:)
      real(dp), pointer :: d_gradr_dlnTm1(:)
      real(dp), pointer :: d_gradr_dlnR(:)
      real(dp), pointer :: d_gradr_dL(:)
   
      real(dp), pointer :: conv_dP_term(:)
      real(dp), pointer :: d_conv_dP_term_dlnd00(:)
      real(dp), pointer :: d_conv_dP_term_dlnT00(:)
      real(dp), pointer :: d_conv_dP_term_dlndm1(:)
      real(dp), pointer :: d_conv_dP_term_dlnTm1(:)
      real(dp), pointer :: d_conv_dP_term_dlnR(:)
      real(dp), pointer :: d_conv_dP_term_dL(:)
         
      real(dp), pointer :: grad_density(:) ! smoothed dlnRho/dlnP at cell boundary
      real(dp), pointer :: grad_temperature(:) ! smoothed dlnT/dlnP at cell boundary
      real(dp), pointer :: gradL(:) ! Ledoux gradient at cell boundary
         ! gradL = grada + brunt_B
      real(dp), pointer :: gradL_composition_term(:)
      integer, pointer :: dominant_iso_for_thermohaline(:)
         ! chem id for dominant isotope in setting chemical gradient term.
         ! i.e. iso with largest local dX*(1+Z)/A
      
      real(dp), pointer :: sig(:) ! sig(k) is mixing flow across face k in (gm sec^1)
         ! sig(k) is D_mix*(4*pi*r(k)**2*rho_face)**2/dmavg
      real(dp), pointer :: sig_div_siglim(:) 
         ! this is raw_sig(k)/siglimit(k)
         ! where siglimit(k) = sig_term_limit*min(dm(k),dm(k-1))/dt
         ! and raw_sig(k) is sig(k) before it is set to min(siglimit(k),raw_sig(k))

      real(dp), pointer :: am_sig_div_am_siglim(:) 

      real(dp), pointer :: cdc_prev(:) ! (nz)  previous cdc interpolated to new grid
   
      ! element diffusion
      real(dp), pointer :: diffusion_D_self(:,:) ! self coeff for chem diffusion. [cm^2/sec]
      real(dp), pointer :: edv(:,:) ! velocity [cm/s] (species,nz) -- excludes chem diffusion.
      real(dp), pointer :: v_rad(:,:) ! velocity from radiative levitation [cm/s] (species,nz)
      real(dp), pointer :: g_rad(:,:) ! acceleration from radiative levitation [cm/s^2] (species,nz)
      real(dp), pointer :: typical_charge(:,:) ! (species,nz)
      real(dp), pointer :: diffusion_dX(:,:) ! (species,nz)
      real(dp), pointer :: E_field(:) ! (nz)
            

      ! gradients
      real(dp), pointer :: dlnP_dm(:) ! at outer boundary of cell
      real(dp), pointer :: dlnT_dm(:) ! at outer boundary of cell
      real(dp), pointer :: dL_dm(:) ! at cell center
   
      ! viscosity pressure
      real(dp), pointer :: Pvisc(:)
      real(dp), pointer :: dPvisc_dlnd(:)
      real(dp), pointer :: dPvisc_dlnT(:)
      real(dp), pointer :: dPvisc_dlnR00(:)
      real(dp), pointer :: dPvisc_dvel00(:)
      real(dp), pointer :: dPvisc_dlnRp1(:)
      real(dp), pointer :: dPvisc_dvelp1(:)

      ! power from nuclear reactions integrated from center out (ergs/second)
      real(dp), pointer :: L_nuc_burn(:) ! excluding photodisintegrations
      real(dp), pointer :: L_nuc_photo(:) ! photodisintegrations
      real(dp), pointer :: L_nuc_by_category(:,:) 
      
      ! power from thermal neutrinos (excludes nuclear reaction neutrinos)
      real(dp), pointer :: L_neutrinos(:) ! integrated non_nuc_neu (ergs/second)
      
      ! power from gravitational contraction
      real(dp), pointer :: L_grav(:) ! integrated eps_grav (ergs/second)
      
      ! Brunt-Vaisala information
      real(dp), pointer :: brunt_N2(:) ! at outer boundary of cell
      real(dp), pointer :: brunt_N2_composition_term(:)
      real(dp), pointer :: brunt_B(:) ! (at points)
         ! this is the Brassard et al B := -(chiY/chiT)*(dlnY/dlnP)
         
      real(dp) :: delta_Pg ! g-mode period spacing for l=1 (seconds)
         ! sqrt(2)*pi^2/(integral of brunt_N/r dr)
      real(dp) :: nu_max ! estimate from scaling relation (microHz)
         ! nu_max = nu_max_sun * M/Msun / ((R/Rsun)^3 (Teff/Teff_sun)^0.5)
         ! with nu_max_sun = 3100 microHz, Teff_sun = 5777
      real(dp) :: acoustic_cutoff ! 1e6*0.5*g*sqrt(gamma1*rho/P) at surface (micro Hz)
      ! asteroseismology
      real(dp) :: delta_nu ! in micro Hz.  = 10^6/(2*photosphere_acoustic_r)
      
      
      ! atmosphere structure information (only created on demand)
      integer :: atm_structure_num_pts
      real(dp), pointer :: atm_structure(:,:) ! (num_results_for_create_atm,atm_structure_num_pts)
         ! defined at points in atmosphere
         ! atm_structure(:,1) is base of atmosphere
         ! atm_structure(:,atm_structure_num_pts) is top of atmosphere
         
         
      real(dp) :: err_ratio_max_hydro, err_ratio_max_hydro_old, err_ratio_max_hydro_older
      real(dp) :: err_ratio_avg_hydro, err_ratio_avg_hydro_old, err_ratio_avg_hydro_older
         


      real(dp), pointer :: d_abar_dlnd(:)
      real(dp), pointer :: d_abar_dlnT(:)
      real(dp), pointer :: d_zbar_dlnd(:)
      real(dp), pointer :: d_zbar_dlnT(:)


   
      ! for solvers
      
         integer, pointer :: ipar_decsol(:), hydro_iwork(:)
         real(dp), pointer :: rpar_decsol(:), hydro_work(:)
         
         integer, dimension(:), pointer :: num_steps, mtx_solve, mtx_factor
         real(dp), dimension(:), pointer :: avg_order
         
         ! *_for_d_dt(k) has value from previous model interpolated to new m(k)
         ! for calculating Lagrangian time derivatives
         real(dp), pointer :: lnd_for_d_dt(:) ! only used when lnPgas_flag is false
         real(dp), pointer :: lnPgas_for_d_dt(:) ! only used when lnPgas_flag is true
         real(dp), pointer :: lnT_for_d_dt(:)
         real(dp), pointer :: lnR_for_d_dt(:)
         real(dp), pointer :: v_for_d_dt(:) ! only used when v_flag is true
         
         real(dp), pointer :: xh_pre(:,:) ! (nvar_hydro,nz)
         real(dp), pointer :: xa_pre(:,:) ! (species,nz)         
         real(dp), pointer :: lnS_pre(:) ! (nz)   for eps_grav dlnS/dt
         
         real(dp), pointer :: lnd_start(:) ! (nz)
         real(dp), pointer :: lnP_start(:) ! (nz)
         real(dp), pointer :: lnT_start(:) ! (nz)
         real(dp), pointer :: lnR_start(:) ! (nz)
         real(dp), pointer :: L_start(:) ! (nz)
         real(dp), pointer :: omega_start(:) ! (nz)
         real(dp), pointer :: Z_start(:) ! (nz)
         real(dp), pointer :: ye_start(:) ! (nz)
         real(dp), pointer :: i_rot_start(:) ! (nz)
         real(dp), pointer :: P_div_rho_start(:)
         real(dp), pointer :: mass_correction_start(:)
         real(dp), pointer :: dxdt_nuc_start(:,:)
         real(dp), pointer :: luminosity_by_category_start(:,:) ! (num_catgories,nz)
         real(dp), pointer :: gradr_start(:) ! (nz)
         real(dp), pointer :: gradL_start(:) ! (nz)
         real(dp), pointer :: gradL_composition_term_start(:)
         real(dp), pointer :: gradT_start(:) ! (nz)
         real(dp), pointer :: actual_gradT_start(:) ! (nz)
         real(dp), pointer :: grada_at_face_start(:) ! (nz)
         real(dp), pointer :: grada_start(:) ! (nz)
         real(dp), pointer :: chiT_start(:) ! (nz)
         real(dp), pointer :: chiRho_start(:) ! (nz)
         real(dp), pointer :: cp_start(:) ! (nz)
         real(dp), pointer :: cv_start(:) ! (nz)
         real(dp), pointer :: T_start(:) ! (nz)
         real(dp), pointer :: mu_start(:) ! (nz)
         real(dp), pointer :: abar_start(:) ! (nz)
         real(dp), pointer :: zbar_start(:) ! (nz)
         real(dp), pointer :: eps_nuc_start(:) ! (nz)
         real(dp), pointer :: non_nuc_neu_start(:) ! (nz)
         
         logical :: have_prev_step_info
         real(dp), pointer :: dlnd_dt_prev_step(:) ! (nz)
         real(dp), pointer :: dlnPgas_dt_prev_step(:) ! (nz)
         real(dp), pointer :: dlnT_dt_prev_step(:) ! (nz)
         
         ! for Lagrangian time derivatives where k < k_below_just_added
         real(dp) :: surf_lnT, surf_lnd, surf_lnPgas, surf_lnR, surf_v, surf_lnS
         integer :: num_surf_revisions
         
         ! working storage for burn+mix
         real(dp), pointer, dimension(:) :: & ! (species*nz)
            avg_burn_dxdt1, avg_mix_dxdt1
         real(dp), pointer, dimension(:,:) :: &
            avg_burn_dxdt, avg_mix_dxdt
            
         real(dp), pointer :: newton_dx(:,:)
            
         real(dp), pointer, dimension(:) :: & ! (nz)
            max_burn_correction, avg_burn_correction
         
         integer :: k_below_just_added 
            ! "just added" == added during the current step
            ! i.e., largest k s.t. for all k' < k, cell k' has material just added this step.
            ! equivalently, this is the smallest k st. for all k' > k,
            !      material below k' was already in the star at the start of this step.
            ! if no mass was added, then this == 1.
            
         integer :: k_below_recently_added
            ! i.e., largest k s.t. for all k' < k, cell k' has recently accreted material.
            ! "recently added" is defined to be a multiple of the "just added"
            ! the factor is given by "factor_for_recently_added" in contros.defaults

         real(dp), pointer :: del_t_for_just_added(:)


            
         integer :: k_thermaltime_eq_accretiontime
            ! i.e., largest k s.t. for all k' < k, cell k' has Cp(k')*T(k')*abs(mstar_dot) < L_star.
            


         integer :: k_const_mass


         integer :: k_below_mdot_eps_grav

                    
         real(dp), pointer :: xa_extra1(:,:) ! (species,nz) -- for use by op split
         real(dp), pointer :: xa_extra2(:,:) ! (species,nz) -- for use by op split
                  

         ! index definitions for the variables (= 0 if variable not in use)
         integer :: i_xlnd ! ln(cell density average by mass) + lnd_offset to keep value > 1
         integer :: i_lnT ! ln cell temperature average by mass
         integer :: i_lnR ! ln radius at outer face of cell
         integer :: i_lum ! luminosity at outer face of cell
         integer :: i_lnPgas ! ln cell gas pressure average by mass
         integer :: i_vel ! Lagrangian velocity at outer face of cell
         
         ! chem variables, if any, follow the structure variables
         integer :: i_chem1 ! index of the first chem variable (0 means no chem variables)
            ! if i_chem1 /= 0, then i_chem1 == nvar_hydro + 1
         integer :: i_dchem1 ! index of the first delta chem variable (0 means no delta chem variables)

         ! index definitions for the equations (= 0 if equation not in use)
         integer :: equP ! momentum conservation
         integer :: equT ! energy transport
         integer :: equR ! mass-volume-density relation
         integer :: equL ! energy conservation
         integer :: equv ! velocity = radius time derivative
         
         ! chem equations follow the structure equations
         integer :: equchem1 ! index of the first chem equation (0 means no chem equations)
            ! if equchem1 /= 0, then equchem1 == nvar_hydro + 1
         integer :: equdchem1 ! index of the first delta chem equation (0 means no delta chem equations)

      ! history data
      
         character (len=strlen) :: &
            model_profile_filename, model_controls_filename, model_data_filename
         character (len=strlen) :: &
            most_recent_profile_filename, most_recent_controls_filename, &
            most_recent_model_data_filename
            
         logical :: helium_ignition
         logical :: carbon_ignition
         integer :: recent_log_header
         integer :: phase_of_evolution
         real(dp) :: prev_Tcntr1         
         real(dp) :: prev_age1         
         real(dp) :: prev_Tcntr2         
         real(dp) :: prev_age2         
         real(dp) :: prev_Tsurf         
         real(dp) :: prv_log_luminosity         
         real(dp) :: prv_log_surface_temp         
         real(dp) :: prv_log_center_temp         
         real(dp) :: prv_log_center_density         
         real(dp) :: profile_age         
         real(dp) :: post_he_age         
         real(dp) :: prev_luminosity         
         real(dp) :: ignition_center_xhe         
         real(dp) :: he_luminosity_limit         
         real(dp) :: prev_cntr_rho         
         real(dp) :: next_cntr_rho
         
      ! pgstar history info
         type (pgstar_hist_node), pointer :: pgstar_hist 
      
      ! hydro matrix info
         integer :: hydro_matrix_type
         type (star_info), pointer :: other_star_info 
            ! holds pointers to "other" allocated arrays
            ! these are used during remesh
      
      ! thermal pulse info
         logical :: have_done_TP
         integer :: TP_state, TP_state_old, TP_state_older
         integer :: TP_count, TP_count_old, TP_count_older
         real(dp) :: TP_M_H_on, TP_M_H_on_old, TP_M_H_on_older
         real(dp) :: TP_M_H_min, TP_M_H_min_old, TP_M_H_min_older
      
      
      ! other stuff        

         integer :: total_num_jacobians
         integer :: num_jacobians
         integer :: num_solves
         integer :: num_retries
         integer :: num_backups
         integer :: number_of_backups_in_a_row
         integer :: last_backup
         integer :: timestep_hold
         integer :: model_number_for_last_jacobian
         integer :: model_number_for_last_retry
         integer :: mesh_call_number
         integer :: burn_and_mix_call_number
         integer :: hydro_call_number
         integer :: diffusion_call_number
         integer :: num_diffusion_solver_iters
         integer :: num_diffusion_solver_steps
         integer :: num_burn_max_iters

         real(dp) :: initial_timestep         
         real(dp) :: hydro_dt_next

         real(dp) :: super_eddington_wind_mdot

         logical :: just_wrote_terminal_header
         logical :: doing_first_model_of_run
         logical :: doing_first_model_after_restart

         real(dp) :: h1_czb_mass, h1_czb_mass_old, h1_czb_mass_older ! baryonic (Msun)
         real(dp) :: h1_czb_mass_prev ! set in finish_step.
            ! we need this to keep track of the last value calculated by the overshoot routine
            ! from the previous step which is not the same as the first value calculated in this step.

         real(dp) :: he_core_mass_old, he_core_mass_older
         real(dp) :: c_core_mass_old, c_core_mass_older
         
         real(dp) :: Teff_old, Teff_older
         real(dp) :: center_eps_nuc_old, center_eps_nuc_older
         
         integer :: n_conv_regions, n_conv_regions_old, n_conv_regions_older
         real(dp), dimension(max_num_mixing_regions) :: &
            cz_bot_mass, cz_bot_mass_old, cz_bot_mass_older, &
            cz_top_mass, cz_top_mass_old, cz_top_mass_older
         ! see also num_mix_boundaries and friends
      
         integer :: Tlim_dXnuc_species
         integer :: Tlim_dXnuc_cell
         
         integer :: Tlim_dXnuc_drop_species
         integer :: Tlim_dXnuc_drop_cell
         
         integer :: Tlim_dX_species
         integer :: Tlim_dX_cell
         
         integer :: Tlim_dX_div_X_species
         integer :: Tlim_dX_div_X_cell
         
         integer :: Tlim_dlgL_nuc_category
         integer :: Tlim_dlgL_nuc_cell
   
         integer :: why_Tlim ! for the most recent timestep
         integer :: result_reason ! code for why the result
   
         logical :: need_to_update_history_now
         logical :: need_to_save_profiles_now
         integer :: save_profiles_model_priority
   

         ! extras -- for saving extra info along with model data for restarts
         ! old and older are for backups.  old and older are same length as current.
         ! requires a procedure call to change lengths (see "star_alloc_extras" in star_lib)
         integer :: len_extra_iwork
         integer :: len_extra_work
         integer, pointer, dimension(:) :: extra_iwork, extra_iwork_old, extra_iwork_older
         real(dp), pointer, dimension(:) :: extra_work, extra_work_old, extra_work_older
         
         ! more extras -- for run_star_extras applications with state.
         ! if your extra state info is all held in the following variables,
         ! restarts, retries, and backups will work without doing anything special.
         integer :: &
            ixtra1, ixtra2, ixtra3, ixtra4, ixtra5, &
            ixtra6, ixtra7, ixtra8, ixtra9, ixtra10, &
            ixtra11, ixtra12, ixtra13, ixtra14, ixtra15, &
            ixtra16, ixtra17, ixtra18, ixtra19, ixtra20, &
            ixtra21, ixtra22, ixtra23, ixtra24, ixtra25, &
            ixtra26, ixtra27, ixtra28, ixtra29, ixtra30
         real(dp) :: &
            xtra1, xtra2, xtra3, xtra4, xtra5, &
            xtra6, xtra7, xtra8, xtra9, xtra10, &
            xtra11, xtra12, xtra13, xtra14, xtra15, &
            xtra16, xtra17, xtra18, xtra19, xtra20, &
            xtra21, xtra22, xtra23, xtra24, xtra25, &
            xtra26, xtra27, xtra28, xtra29, xtra30
         logical :: &
            lxtra1, lxtra2, lxtra3, lxtra4, lxtra5, &
            lxtra6, lxtra7, lxtra8, lxtra9, lxtra10, &
            lxtra11, lxtra12, lxtra13, lxtra14, lxtra15, &
            lxtra16, lxtra17, lxtra18, lxtra19, lxtra20, &
            lxtra21, lxtra22, lxtra23, lxtra24, lxtra25, &
            lxtra26, lxtra27, lxtra28, lxtra29, lxtra30

         integer :: &
            ixtra1_old, ixtra2_old, ixtra3_old, ixtra4_old, ixtra5_old, &
            ixtra6_old, ixtra7_old, ixtra8_old, ixtra9_old, ixtra10_old, &
            ixtra11_old, ixtra12_old, ixtra13_old, ixtra14_old, ixtra15_old, &
            ixtra16_old, ixtra17_old, ixtra18_old, ixtra19_old, ixtra20_old, &
            ixtra21_old, ixtra22_old, ixtra23_old, ixtra24_old, ixtra25_old, &
            ixtra26_old, ixtra27_old, ixtra28_old, ixtra29_old, ixtra30_old
         real(dp) :: &
            xtra1_old, xtra2_old, xtra3_old, xtra4_old, xtra5_old, &
            xtra6_old, xtra7_old, xtra8_old, xtra9_old, xtra10_old, &
            xtra11_old, xtra12_old, xtra13_old, xtra14_old, xtra15_old, &
            xtra16_old, xtra17_old, xtra18_old, xtra19_old, xtra20_old, &
            xtra21_old, xtra22_old, xtra23_old, xtra24_old, xtra25_old, &
            xtra26_old, xtra27_old, xtra28_old, xtra29_old, xtra30_old
         logical :: &
            lxtra1_old, lxtra2_old, lxtra3_old, lxtra4_old, lxtra5_old, &
            lxtra6_old, lxtra7_old, lxtra8_old, lxtra9_old, lxtra10_old, &
            lxtra11_old, lxtra12_old, lxtra13_old, lxtra14_old, lxtra15_old, &
            lxtra16_old, lxtra17_old, lxtra18_old, lxtra19_old, lxtra20_old, &
            lxtra21_old, lxtra22_old, lxtra23_old, lxtra24_old, lxtra25_old, &
            lxtra26_old, lxtra27_old, lxtra28_old, lxtra29_old, lxtra30_old

         integer :: &
            ixtra1_older, ixtra2_older, ixtra3_older, ixtra4_older, ixtra5_older, &
            ixtra6_older, ixtra7_older, ixtra8_older, ixtra9_older, ixtra10_older, &
            ixtra11_older, ixtra12_older, ixtra13_older, ixtra14_older, ixtra15_older, &
            ixtra16_older, ixtra17_older, ixtra18_older, ixtra19_older, ixtra20_older, &
            ixtra21_older, ixtra22_older, ixtra23_older, ixtra24_older, ixtra25_older, &
            ixtra26_older, ixtra27_older, ixtra28_older, ixtra29_older, ixtra30_older
         real(dp) :: &
            xtra1_older, xtra2_older, xtra3_older, xtra4_older, xtra5_older, &
            xtra6_older, xtra7_older, xtra8_older, xtra9_older, xtra10_older, &
            xtra11_older, xtra12_older, xtra13_older, xtra14_older, xtra15_older, &
            xtra16_older, xtra17_older, xtra18_older, xtra19_older, xtra20_older, &
            xtra21_older, xtra22_older, xtra23_older, xtra24_older, xtra25_older, &
            xtra26_older, xtra27_older, xtra28_older, xtra29_older, xtra30_older
         logical :: &
            lxtra1_older, lxtra2_older, lxtra3_older, lxtra4_older, lxtra5_older, &
            lxtra6_older, lxtra7_older, lxtra8_older, lxtra9_older, lxtra10_older, &
            lxtra11_older, lxtra12_older, lxtra13_older, lxtra14_older, lxtra15_older, &
            lxtra16_older, lxtra17_older, lxtra18_older, lxtra19_older, lxtra20_older, &
            lxtra21_older, lxtra22_older, lxtra23_older, lxtra24_older, lxtra25_older, &
            lxtra26_older, lxtra27_older, lxtra28_older, lxtra29_older, lxtra30_older

      
         
      ! names of variables and equations
         character (len=name_len), dimension(:), pointer :: nameofvar, nameofequ ! (nvar)
         
      ! differential or algebraic
         ! if u is a differential variable, then ODE eqn is of form du/dt = f(vars)
         ! if u is an algebraic variable, then constraint eqn is of form 0 = g(vars)
         logical, dimension(:), pointer :: ode_var ! (nvar)

      ! equation residuals
         ! equ(i,k) is residual for equation i of cell k            
         real(dp), dimension(:,:), pointer :: equ ! (nvar,nz);  equ => equ1
         real(dp), dimension(:), pointer :: equ1 ! (nvar*nz); data for equ
         
      ! equation Jacobian
         integer :: idiag
         real(dp), dimension(:,:), pointer :: jacobian
            ! NOT ALLOCATED. pointer set by enter_setmatrix
         ! these are pointers into the jacobian when it is block triangular
         real(dp), dimension(:,:,:), pointer :: ublk, dblk, lblk ! (nvar,nvar,nz) 
         real(dp), pointer, dimension(:) :: AF1 ! for factored jacobian
            ! will be allocated or reallocated as necessary by newton.
            ! deallocate when free_star

      ! profile info
         integer, pointer :: profile_column_spec(:)
         
      ! extras
         real(dp), dimension(:,:), pointer :: profile_extra ! (nz,max_num_profile_extras)
         character (len=64) :: profile_extra_name(max_num_profile_extras)
         real(dp), dimension(:,:), pointer :: history_extra ! (nz,max_num_history_extras)
         character (len=64) :: history_extra_name(max_num_history_extras)
      
      ! history info
         integer :: number_of_history_columns
            ! number_of_history_columns = num_standard_history_columns + num extra history columns
         real(dp), pointer :: history_values(:) ! (number_of_history_columns)
         logical, pointer :: history_value_is_integer(:) ! (number_of_history_columns)
         character (len=maxlen_history_column_name), pointer :: &
            history_names(:) ! (number_of_history_columns)
         type (integer_dict), pointer :: history_names_dict
            ! maps names to index in history_values
         integer :: model_number_of_history_values
         logical :: need_to_set_history_names_etc
         logical :: need_to_write_history
         
         ! the following does not include "extra" columns
         integer, pointer :: history_column_spec(:) ! (num_standard_history_columns)
         
      ! sizeB info
         integer :: max_zone1, max_var1, max_zone2, max_var2
         real(dp) :: abs_max_corr1, abs_max_corr2
      
      ! bcyclic klu sparse data
         type(sparse_info), pointer :: bcyclic_klu_storage(:)
         type(ulstore), pointer :: bcyclic_odd_storage(:)
         character (len=net_name_len) :: bcyclic_sprs_shared_net_name
         integer :: bcyclic_shared_sprs_nonzeros, bcyclic_nvar_hydro
         integer, pointer :: bcyclic_sprs_shared_ia(:) ! (species+1)
         integer, pointer :: bcyclic_sprs_shared_ja(:) ! (sprs_nonzeros)
         integer :: bcyclic_sprs_shared_ipar8_decsol(num_klu_ipar_decsol)
      
      ! sparse split burn data
         type(sparse_info), pointer :: burn_klu_storage(:) ! (1:nz)
         character (len=net_name_len) :: burn_sprs_shared_net_name
         integer :: burn_shared_sprs_nonzeros
         integer, pointer :: burn_sprs_shared_ia(:) ! (species+1)
         integer, pointer :: burn_sprs_shared_ja(:) ! (sprs_nonzeros)
         integer :: burn_sprs_shared_ipar8_decsol(num_klu_ipar_decsol)
      
      ! misc
         integer :: termination_code
         logical :: numerical_jacobian, doing_numerical_jacobian
         integer :: boost_mlt_alfa
         real(dp) :: save1, save2, save3, save4, save5, save6, save7
         integer :: burn_nstep_max, burn_nfcn_total
         real(dp) :: dVARdot_dVAR
         real(dp) :: jrot_surf_old
         real(dp) :: surf_accel_grav_ratio
         logical :: just_did_backup
         logical :: doing_flash_wind, doing_rlo_wind, doing_nova_wind
         real(dp) :: op_split_diff, Lnuc_1, Ltot_1, Lnucneu_1, Lphoto_1
         real(dp) :: prev_create_atm_R0_div_R
         logical :: have_previous_rotation_info
         logical :: have_previous_conv_vel
         real(dp) :: max_x_expected, min_x_expected
         
         character (len=256) :: most_recent_photo_name
         

      ! system clock 
         integer :: starting_system_clock_time
         integer :: current_system_clock_time
         integer :: system_clock_rate
         real(dp) :: total_elapsed_time ! (current - starting)/rate
      
      
      ! timing information
         logical :: doing_timing
         integer :: system_clock_at_start_of_step
         
         ! change total_times function in star_utils if change this list
      
         real(dp) :: time_eos
         real(dp) :: time_kap
         real(dp) :: time_net
         real(dp) :: time_neu
      
         real(dp) :: time_do_mesh
         real(dp) :: time_do_mesh_plan
         real(dp) :: time_do_mesh_adjust

         real(dp) :: time_newton_xscale
         real(dp) :: time_newton_eval_eqn
         real(dp) :: time_newton_size_equ
         real(dp) :: time_newton_size_B
         real(dp) :: time_newton_enter_setmatrix

         real(dp) :: time_do_adjust_mass
         real(dp) :: time_do_report
         real(dp) :: time_next_timestep
         real(dp) :: time_write_profile
         real(dp) :: time_write_log
         real(dp) :: time_write_photo
         real(dp) :: time_pgstar
      
         real(dp) :: time_set_basic_vars
         real(dp) :: time_set_rotation_vars
         real(dp) :: time_set_mlt_vars
         real(dp) :: time_set_eqn_vars
         real(dp) :: time_eval_eqns
         real(dp) :: time_set_newton_vars
         real(dp) :: time_newton_mtx
         real(dp) :: time_newton_self
         real(dp) :: time_newton_test
         real(dp) :: time_solve_burn_non_net
         real(dp) :: time_solve_burn_in_net
         real(dp) :: time_solve_mix
         real(dp) :: time_op_split_control

         real(dp) :: time_check_model
         real(dp) :: time_prep_new_step
         real(dp) :: time_prep_new_try
         real(dp) :: time_prep_for_retry
         real(dp) :: time_set_mdot
         real(dp) :: time_save_for_d_dt
         real(dp) :: time_diffusion
         real(dp) :: time_evolve_set_vars
         real(dp) :: time_save_start
         real(dp) :: time_check_newly_non_conv
         real(dp) :: time_struct_burn_mix

         ! change total_times function in star_utils if change this list
      
         real(dp) :: time_total
                  
      ! "other" procedures
         
         procedure(other_adjust_mdot_interface), pointer, nopass :: &
            other_adjust_mdot => null()
         procedure(other_wind_interface), pointer, nopass :: &
            other_wind => null()
         procedure (other_mixing_interface), pointer, nopass :: &
            other_mixing => null()
         procedure (other_am_mixing_interface), pointer, nopass :: &
            other_am_mixing => null()
         procedure (other_energy_interface), pointer, nopass :: &
            other_energy => null()
         procedure (other_energy_implicit_interface), pointer, nopass :: &
            other_energy_implicit => null()
         procedure (other_torque_interface), pointer, nopass :: &
            other_torque => null()            
         procedure (other_torque_implicit_interface), pointer, nopass :: &
            other_torque_implicit => null()            
         procedure (other_brunt_interface), pointer, nopass :: &
            other_brunt => null()            
         procedure (other_cgrav_interface), pointer, nopass :: &
            other_cgrav => null()
         procedure (other_opacity_factor_interface), pointer, nopass :: &
            other_opacity_factor => null()
         procedure (other_viscosity_pressure_interface), pointer, nopass :: &
            other_viscosity_pressure => null()
         procedure (other_eps_grav_interface), pointer, nopass :: &
            other_eps_grav => null()
         
         procedure (other_mlt_interface), pointer, nopass :: &
            other_mlt => null()
         procedure (other_neu_interface), pointer, nopass :: &
            other_neu => null()
         procedure (other_atm_interface), pointer, nopass :: &
            other_atm => null()
         procedure (other_diffusion_interface), pointer, nopass :: &
            other_diffusion => null()

         procedure (other_eosDT_get_interface), pointer, nopass :: &
            other_eosDT_get => null()
         procedure (other_eosDT_get_T_interface), pointer, nopass :: &
            other_eosDT_get_T => null()
         procedure (other_eosDT_get_Rho_interface), pointer, nopass :: &
            other_eosDT_get_Rho => null()

         procedure (other_eosPT_get_interface), pointer, nopass :: &
            other_eosPT_get => null()
         procedure (other_eosPT_get_T_interface), pointer, nopass :: &
            other_eosPT_get_T => null()
         procedure (other_eosPT_get_Pgas_interface), pointer, nopass :: &
            other_eosPT_get_Pgas => null()
         procedure (other_eosPT_get_Pgas_for_Rho_interface), pointer, nopass :: &
            other_eosPT_get_Pgas_for_Rho => null()

         procedure (other_kap_get_Type1_interface), pointer, nopass :: &
            other_kap_get_Type1 => null()
         procedure (other_kap_get_Type2_interface), pointer, nopass :: &
            other_kap_get_Type2 => null()

         procedure (other_pgstar_plots_info_interface), pointer, nopass :: &
            other_pgstar_plots_info => null()
         procedure (how_many_other_mesh_fcns_interface), pointer, nopass :: &
            how_many_other_mesh_fcns => null()
         procedure (other_mesh_fcn_data_interface), pointer, nopass :: &
            other_mesh_fcn_data => null()

         procedure (other_write_pulsation_info_interface), pointer, nopass :: &
            other_write_pulsation_info => null()
         procedure (other_get_pulsation_info_interface), pointer, nopass :: &
            other_get_pulsation_info => null()

         procedure (other_history_data_initialize_interface), pointer, nopass :: &
            other_history_data_initialize => null()
         procedure (other_history_data_add_model_interface), pointer, nopass :: &
            other_history_data_add_model => null()

