! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module winds

      use star_private_def
      use const_def
      use chem_def, only: ih1, ihe4, ic12, ic13, in14, io16

      implicit none


      contains
      
      
      
      ! NOTE: don't assume that vars are all set when doing winds.
      ! use values from s% xh(:,:) and s% xa(:,:) only.
      ! e.g., must use s% xh(s% i_lnT,1) instead of s% lnT(1)
      
      
      real(dp) function eval_stern51_wind(s, L, M, R, Teff, X, Y, Z) result(dms)
         use chem_def
         use mod_kuma, only: kuma
         type (star_info), pointer :: s
         real(dp), intent(in) :: L, M, R, Teff, X, Y, Z ! cgs
         
         real(dp) :: xhc, xhs, xhs2, znow, fHeavy, ZHeavy, Zlight, Zcno, &
            dms1, dms2, fff, logl, logm, logmd, logr, logz, &
            lteff,vinfkm, xxx, zfac, XC, XN, XO, XNe, XMg, &
            Zbase, ZFe, fC, fN, fO, fNe, fFe
         integer :: i, j, cid, h1, c12, n14, o16, ne20, mg24
         
         include 'formats'
         
         h1 = s% net_iso(ih1)
         Zbase = s% Zbase
         if (Zbase < 0) then
            fC = s% base_fC
            fN = s% base_fN
            fO = s% base_fO
            fNe = s% base_fNe
            fHeavy = 1d0 - fC - fN - fO - fNe
            XC = 0; XN = 0; XO = 0; XNe = 0; XMg = 0
            c12 = s% net_iso(ic12)
            n14 = s% net_iso(in14)
            o16 = s% net_iso(io16)
            ne20 = s% net_iso(ine20)
            mg24 = s% net_iso(img24)
            if (c12 > 0) XC = s% xa(c12,1)
            if (n14 > 0) XN = s% xa(n14,1)
            if (o16 > 0) XO = s% xa(o16,1)
            if (ne20 > 0) XNe = s% xa(ne20,1)
            if (mg24 > 0) XMg = s% xa(mg24,1)
            ZLight = max(0d0, min(1d0, XC + XN + XO + XNe))
            ZHeavy = max(0d0, min(1d0, Z - ZLight))
            Zbase = ZHeavy/fHeavy
         end if
         fFe = 0.07d0 ! approx metal fraction by mass of Fe
         ZFe = Zbase*fFe
         Zcno = Z - ZFe ! this is the Stern definition

         if (h1 > 0) then
            xhs = s% xa(h1,1)
            xhc = s% xa(h1,s% nz)
         else
            xhs = X
            xhc = 0
         end if
         xhs2 = X ! surface avg
                  
         if (xhs2 > 0.3d0) then ! still have H in envelope
         
            znow = max(0d0, min(1d0, &
               0.02*(0.5*(ZFe/1.32d-3)+0.5*(Zcno/0.01868))))
               
            if (xhc < 1d-6) then ! center h gone
               logr = log10(R/Rsun)
               logl = log10(L/Lsun)
               logm = log10(M/Msun)
               logz = log10(znow/0.02)
               logmd = 1.24*logl + 0.16*logm + 0.81*logr - 14.016
               dms=10.**logmd            
            else ! if (xhc >= 1d-6) then ! still have H at center
               logl = log10(L/Lsun)
               lteff = log10(Teff)
               call kuma(logl, lteff, M/Msun, xhs, dms, vinfkm)
            end if

            dms = dms*(znow/0.02)**0.69

            if (xhs2 <= 0.45) then ! envelope H is partially depleted
               znow = max(Zbase, 1d-6)
               dms1 = dms
               call wr_hamann(L,xhs,znow,dms2)
               dms2 = dms2/10.
               xxx = (xhs-0.375)*12.
               fff = 0.25*(2.-3.*xxx+xxx**3)
               dms = (1.-fff)*dms1+fff*dms2
               dms = max(dms1,dms)
            end if
         
         else ! if (xhs2 <= 0.3d0) then ! envelope H is gone or going
            
            znow = max(Zbase, 1d-6)
            call wr_hamann(L,xhs,znow,dms)
            dms = dms/10.
            if (Zcno >= znow) then
               zfac = 1. + 19.*(Zcno-znow)/(1.-znow)
               dms = dms*zfac
            end if
         
         end if
               
      end function eval_stern51_wind
      
      
      subroutine wr_hamann(xsl,xhs,znow,dms)
         real(dp) :: xsl, xhs, znow, dms
         !-------------------------------------------------------------------------         
         !     l and x dep. mass loss for WNL and WNE/WC/WO stars
         !     see Hamann et al. 1995, A&A 299, 151
         !     for log(L)<4.5, mass loss is strongly reduced according
         !     to Hamann et al. 1982, A&A 116, 273 (very low rates for
         !     Helium stars with low L)
         !--------------------------------------------------------------------------
         real(dp) :: xlogl, xlogmd, logz
         include 'formats'
         xlogl = log10(xsl/Lsun)
         logz = log10(max(1d-20,znow)/0.02)
         if (xlogl > 4.5) then
            xlogmd=1.5d0*xlogl-2.85d0*xhs-11.95d0+0.86*logz
         else
            xlogmd=6.8d0*xlogl-2.85d0*xhs-35.8d0+0.86*logz
         end if
         dms = 10**xlogmd
         return
         write(*,1) 'wr_hamann dms', dms, xlogmd
         write(*,1) 'wr_hamann xlogl', xlogl
         write(*,1) 'wr_hamann xhs', xhs
         write(*,1) 'wr_hamann znow', znow
         write(*,1) 'wr_hamann logz', logz
      end subroutine wr_hamann
      
      
      subroutine eval_super_eddington_wind(s, L, M, R, ierr)
         type (star_info), pointer :: s
         real(dp), intent(in) :: L, M, R
         integer, intent(out) :: ierr

         real(dp) :: Ledd, Leff, vesc2         
         include 'formats'
         
         ierr = 0
         s% super_eddington_wind_mdot = 0 
         if (s% super_eddington_wind_eta <= 0) return 

         Ledd = s% prev_Ledd
         Leff = L/s% super_eddington_wind_Ledd_factor
         if (Leff <= Ledd) return
         vesc2 = s% cgrav(1)*M/R
         s% super_eddington_wind_mdot = s% super_eddington_wind_eta*(Leff - Ledd)/vesc2
         write(*,'(a60,i12,992f12.5)') 'super eddington wind: lg_Mdot, L/Ledd', &
            s% model_number, log10(s% super_eddington_wind_mdot/(Msun/secyer)), L/Ledd
         
      end subroutine eval_super_eddington_wind
      
      
      ! NOTE: don't assume that vars are all set at this point.
      ! use values from s% xh(:,:) and s% xa(:,:) only.
      real(dp) function eval_nova_wind(s, L_surf, R, Teff, ierr) ! value in Msun/year
         type (star_info), pointer :: s
         real(dp), intent(in) :: L_surf, R, Teff ! Lsun, Rsun, K
         integer, intent(out) :: ierr   
         real(dp) :: roche_lobe_radius ! Rsun    
         real(dp) :: mdot
         include 'formats'
         ierr = 0
         eval_nova_wind = 0
         if (s% nova_wind_eta <= 0) return
         if (L_surf < s% nova_wind_min_L) return
         if (Teff > s% nova_wind_max_Teff) return
         
         roche_lobe_radius = s% nova_roche_lobe_radius
         if (R >= roche_lobe_radius) then
            mdot = s% nova_RLO_mdot*Msun/secyer ! in gm per second
            write(*,1) 'nova RLO log mdot', log10(s% nova_wind_eta*mdot/(Msun/secyer))
         else ! eqn 23 of Kato and Hachisu, ApJ 437:802-826, 1994
            mdot = 10d0**(1.49d0*log10(Teff/1d5) + s% nova_wind_b) ! in gm per second
            write(*,1) 'nova wind log mdot', log10(s% nova_wind_eta*mdot/(Msun/secyer))
         end if
                
         eval_nova_wind = s% nova_wind_eta*mdot/(Msun/secyer)
         
      end function eval_nova_wind
      
      
      ! NOTE: don't assume that vars are all set at this point.
      ! use values from s% xh(:,:) and s% xa(:,:) only.
      real(dp) function eval_rlo_wind(s, L_surf, R, Teff, xfer_ratio, ierr) ! value in Msun/year
         type (star_info), pointer :: s
         real(dp), intent(in) :: L_surf, R, Teff ! Lsun, Rsun, K
         real(dp), intent(inout) :: xfer_ratio   
         integer, intent(out) :: ierr   
         real(dp) :: roche_lobe_radius ! Rsun    
         real(dp) :: ratio, rho, p, grav, hp, v_th, h, rho_exponent, rho_rl, rho_rl0, mdot
         include 'formats'
         ierr = 0
         eval_rlo_wind = 0
         if (s% rlo_wind_eta <= 0) return
         if (L_surf < s% rlo_wind_min_L) return
         if (Teff > s% rlo_wind_max_Teff) return
         if (s% rlo_wind_scale_height <= 0) return
         roche_lobe_radius = s% rlo_wind_roche_lobe_radius
         ratio = R/roche_lobe_radius
         if (ratio < 1) then
            ! check for reduction in transfer ratio for almost full Roche lobe
            if (ratio < s% roche_lobe_xfer_full_on) return
            if (ratio > s% roche_lobe_xfer_full_off) then
               xfer_ratio = 0
               return
            end if
            xfer_ratio = (s% roche_lobe_xfer_full_off - ratio) / &
               (s% roche_lobe_xfer_full_off - s% roche_lobe_xfer_full_on)
            xfer_ratio = 0.5d0*(1 - cos(pi*xfer_ratio))
            return
         end if
         mdot = s% rlo_wind_base_mdot* &
            exp(min(6*ln10,(R - roche_lobe_radius)/s% rlo_wind_scale_height))
         eval_rlo_wind = s% rlo_wind_eta*mdot ! Msun/year
         
         write(*,1) 'log rlo mdot Msun/yr, R/R_L', log10(eval_rlo_wind), &
            R/roche_lobe_radius
         if (.false. .and. (R - roche_lobe_radius)/s% rlo_wind_scale_height > 100) then
            write(*,1) 'ratio', (R - roche_lobe_radius)/s% rlo_wind_scale_height
            write(*,1) 's% rlo_wind_scale_height', s% rlo_wind_scale_height
            write(*,1) 'R - roche_lobe_radius', R - roche_lobe_radius
            write(*,1) 'R', R
            write(*,1) 'exp(s% xh(s% i_lnR,1))/Rsun', &
               exp(s% xh(s% i_lnR,1))/Rsun
            write(*,1) 'roche_lobe_radius', roche_lobe_radius
            stop 'eval_rlo_wind'
         end if
         
      end function eval_rlo_wind
      
      
      ! NOTE: don't assume that vars are all set at this point.
      ! use values from s% xh(:,:) only.
      ! set_mdot can reduce s% dt if necessary to limit mass loss per step
      subroutine set_mdot(s, L_phot, M_phot, T_phot, ierr)
         use utils_lib, only: is_bad_num
         use chem_def
         use num_lib, only: safe_log10
         type (star_info), pointer :: s
         real(dp), intent(in) :: L_phot, M_phot, T_phot ! photosphere values (cgs)
         integer, intent(out) :: ierr
         
         integer :: k, j, h1, he4, nz, base
         real(dp) :: max_ejection_mass, wind_mdot, env_mass, wind, alfa, total_H, &
            X, Y, Z, w1, w2, T_high, T_low, L1, M1, R1, T1, &
            log_dtyr, log_dtyr_full_off, log_dtyr_full_on, &
            center_h1, center_he4, surface_h1, surface_he4, mdot, xfer_ratio, &
            L_div_Ledd, full_off, full_on, max_boost, super_eddington_boost
         character (len=256) :: message
         character (len=32) :: scheme
         logical :: is_infalling, using_wind_scheme_mdot, use_other
         real(dp), parameter :: Zsolar = 0.019d0 ! for Vink et al formula

         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         if (dbg) write(*,1) 'enter set_mdot mass_change', s% mass_change
         
         ierr = 0
         
         xfer_ratio = 1d0
         
         L1 = L_phot
         M1 = M_phot
         T1 = T_phot
         R1 = sqrt(L1/(pi*crad*clight*T1**4)) ! assume L1 and T1 for photosphere

         h1 = s% net_iso(ih1)
         he4 = s% net_iso(ihe4)
         nz = s% nz
         wind_mdot = 0
         using_wind_scheme_mdot = .false.
         use_other = (s% use_other_wind .or. scheme == 'other')
         
         call eval_super_eddington_wind(s, L1, M1, R1, ierr)
         if (ierr /= 0) then
            if (dbg .or. s% report_ierr) write(*, *) 'set_mdot: eval_super_eddington_wind ierr'
            return
         end if
         
         if (s% super_eddington_wind_mdot > wind_mdot) then
            wind_mdot = s% super_eddington_wind_mdot
            if (dbg) write(*,1) 'super eddington wind mdot', &
               log10(s% super_eddington_wind_mdot/(Msun/secyer))
         end if
         
         if (dbg) write(*,1) 's% remove_H_wind_mdot', s% remove_H_wind_mdot
         if (s% remove_H_wind_mdot > 0 .and. h1 > 0) then
            total_H = dot_product(s% xa(h1,1:nz), s% dm(1:nz))/Msun
            write(*,1) 'total_H', total_H
            if (dbg) write(*,1) 's% remove_H_wind_H_mass_limit', s% remove_H_wind_H_mass_limit
            if (total_H > s% remove_H_wind_H_mass_limit) then
               if (s% remove_H_wind_mdot*Msun/secyer > wind_mdot) then
                  wind_mdot = s% remove_H_wind_mdot*Msun/secyer
                  if (dbg) write(*,1) 'use remove H wind mdot', &
                     log10(s% remove_H_wind_mdot/(Msun/secyer))
               end if
            end if
         end if
            
         if (s% flash_wind_mdot > 0) then
            if (s% doing_flash_wind) then
               if (R1/Rsun <= s% flash_wind_full_off) then
                  s% doing_flash_wind = .false.
                  write(*,*) 's% doing_flash_wind = .false.'
               end if
            else if (R1/Rsun >= s% flash_wind_starts) then
               s% doing_flash_wind = .true.
               write(*,*) 's% doing_flash_wind = .true.'
            end if
            if (s% doing_flash_wind .and. s% flash_wind_mdot*Msun/secyer > wind_mdot) then
               if (R1/Rsun < s% flash_wind_declines) then
                  s% flash_wind_mdot = s% flash_wind_mdot * &
                     (s% flash_wind_declines - R1/Rsun) / &
                        (s% flash_wind_declines - s% flash_wind_full_off)
               else if (R1/Rsun > s% flash_wind_starts) then
                  s% flash_wind_mdot = s% flash_wind_mdot*(2d0**((R1/Rsun)/s% flash_wind_starts - 1d0))
               end if
               if (s% flash_wind_mdot*Msun/secyer > wind_mdot) then
                  wind_mdot = s% flash_wind_mdot*Msun/secyer
                  write(*,1) 'flash wind r', R1/Rsun
               end if
            end if
         end if
         
         mdot = eval_rlo_wind(s, L1/Lsun, R1/Rsun, T1, xfer_ratio, ierr) ! Msun/year
         mdot = mdot*Msun/secyer
         if (ierr /= 0) then
            if (dbg .or. s% report_ierr) write(*, *) 'set_mdot: eval_rlo_wind ierr'
            return
         end if
         s% doing_rlo_wind = (mdot /= 0)
         
         if (s% doing_rlo_wind .and. mdot > wind_mdot) then
            if (dbg) write(*,1) 's% doing_rlo_wind mdot', log10(mdot/(Msun/secyer))
            wind_mdot = mdot
         end if
         
         if (s% nova_wind_eta /= 0 .and. L1/Lsun > s% nova_wind_min_L .and. &
               T1 < s% nova_min_Teff_for_accretion .and. s% mass_change > 0) then
            mdot = 0
         else
            mdot = eval_nova_wind(s, L1/Lsun, R1/Rsun, T1, ierr)
            if (ierr /= 0) then
               if (dbg .or. s% report_ierr) write(*, *) 'set_mdot: eval_nova_wind ierr'
               return
            end if
         end if
         s% doing_nova_wind = (mdot /= 0)
         if (s% doing_nova_wind .and. mdot*Msun/secyer > wind_mdot) then
            if (dbg) write(*,1) 's% doing_nova_wind mdot', mdot
            wind_mdot = mdot*Msun/secyer
         end if
         
         if (h1 > 0) then
            center_h1 = s% xa(h1,nz)
            surface_h1 = s% xa(h1,1)
         else
            center_h1 = 0
            surface_h1 = 0
         end if
         if (he4 > 0) then
            center_he4 = s% xa(he4,nz)
            surface_he4 = s% xa(he4,1)
         else
            center_he4 = 0
            surface_he4 = 0
         end if
         
         if (center_h1 < 0.01d0 .and. center_he4 < s% RGB_to_AGB_wind_switch) then
            scheme = s% AGB_wind_scheme
            if (dbg) write(*,1) 'using AGB_wind_scheme: "' // trim(scheme) // '"', &
                        center_h1, center_he4, s% RGB_to_AGB_wind_switch
         else
            scheme = s% RGB_wind_scheme
            if (dbg) write(*,*) 'using RGB_wind_scheme: "' // trim(scheme) // '"'
         end if
                     
         if ((s% mass_change > 0 .and. wind_mdot == 0) .or. &
             (s% mass_change < 0 .and. -s% mass_change*Msun/secyer > wind_mdot)) then
            if (dbg) write(*,*) 'mass_change mdot', s% mass_change
            wind_mdot = -s% mass_change*Msun/secyer
            if (s% mass_change > 0 .and. xfer_ratio < 1d0) then
               write(*,1) 'almost full roche lobe: reduce mdot by xfer_ratio', xfer_ratio
               wind_mdot = wind_mdot*xfer_ratio
            end if
         end if
         
         if (use_other .or. len_trim(scheme) > 0) then
            env_mass = s% xmstar
            do k=1,nz
               if (s% xa(h1,k) <= s% h1_boundary_limit) then
                  env_mass = s% xmstar*(1 - s% q(k))
                  exit
               end if
            end do
            if (dbg) then
               write(*,1) 'env_mass', env_mass/Msun
               write(*,1) 's% wind_envelope_limit', s% wind_envelope_limit
            end if
            if (s% wind_envelope_limit > 0 .and. env_mass < s% wind_envelope_limit*Msun) then
               wind = 0
            else
               wind = 4d-13*(L1*R1/M1)/(Lsun*Rsun/Msun) ! in Msun/year
               if (dbg) write(*,1) 'wind', wind
               if (wind <= 0 .or. is_bad_num(wind)) then
                  ierr = -1
                  write(*,*) 'bad value for wind :', wind,L1,R1,M1
                  if (dbg) stop 'debug: bad value for wind'
                  return
               end if
               X = surface_h1
               Y = surface_he4
               Z = 1 - (X + Y)
               
               if (use_other) then
                  if (dbg) write(*,*) 'call other_wind'
                  call s% other_wind(s% id, L1, M1, R1, T1, wind, ierr)
                  if (ierr /= 0) return
               else if (scheme == 'Dutch') then
                  T_high = 11000
                  T_low = 10000
                  if (s% Dutch_wind_eta == 0) then
                     wind = 0
                  else if (T1 <= T_low) then
                     call eval_lowT_Dutch(wind)
                  else if (T1 >= T_high) then
                     call eval_highT_Dutch(wind)
                  else ! transition
                     call eval_lowT_Dutch(w1)
                     call eval_highT_Dutch(w2)
                     alfa = (T1 - T_low)/(T_high - T_low)
                     wind = (1-alfa)*w1 + alfa*w2
                  end if
                  wind = s% Dutch_wind_eta * wind
               else if (scheme == 'Stern51') then
                  wind = eval_stern51_wind(s, L1, M1, R1, T1, X, Y, Z)
                  wind = wind * s% Stern51_wind_eta
                  if (dbg) write(*,1) 'Stern51_wind', wind
               else if (scheme == 'Reimers') then
                  wind = wind * s% Reimers_wind_eta
                  if (dbg) then
                     write(*,1) 's% Reimers_wind_eta', s% Reimers_wind_eta
                     write(*,1) 'Reimers_wind', wind
                     write(*,1) 'L1/Lsun', L1/Lsun
                     write(*,1) 'R1/Rsun', R1/Rsun
                     write(*,1) 'M1/Msun', M1/Msun
                     write(*,1) 'Reimers_wind_etaReimers_wind_eta', s% Reimers_wind_eta
                     write(*,1) 'wind', wind
                     write(*,1) 'log10 wind', log10(wind)
                     write(*,*)
                     stop 'debug: winds'
                  end if
               else if (scheme == 'Vink') then
                  call eval_vink_wind(wind)
                  wind = wind * s% Vink_wind_eta
                  if (dbg) write(*,1) 'Vink_wind', wind
               else if (scheme == 'Kudritzki') then
                  call eval_Kudritzki_wind(wind)
                  if (dbg) write(*,1) 'Kudritzki_wind', wind
               else if (scheme == 'Blocker') then
                  call eval_blocker_wind(wind)
                  if (dbg) write(*,1) 'Blocker_wind', wind
               else if (scheme == 'de Jager') then
                  call eval_de_Jager_wind(wind)
                  wind = s% de_Jager_wind_eta * wind
                  if (dbg) write(*,1) 'de_Jager_wind', wind
               else if (scheme == 'van Loon') then
                  call eval_van_Loon_wind(wind)
                  wind = s% van_Loon_wind_eta * wind
                  if (dbg) write(*,1) 'van_Loon_wind', wind
               else if (scheme == 'Nieuwenhuijzen') then
                  call eval_Nieuwenhuijzen_wind(wind)
                  wind = s% Nieuwenhuijzen_wind_eta * wind
                  if (dbg) write(*,1) 'Nieuwenhuijzen_wind', wind
               else 
                  ierr = -1
                  write(*,*) 'unknown name for wind scheme : ' // trim(scheme)
                  if (dbg) stop 'debug: bad value for wind scheme'
                  return
               end if
               
               if (wind*Msun/secyer > abs(wind_mdot)) then
                  using_wind_scheme_mdot = .true.
                  wind_mdot = wind*Msun/secyer
                  if (dbg) write(*,1) 'use wind scheme mdot', wind_mdot
               end if
               
            end if
         end if
         
         if (using_wind_scheme_mdot) then
            if (s% no_wind_if_no_rotation .and. .not. s% rotation_flag) then
               s% mstar_dot = 0
               if (s% trace_dt_control_mass_change) &
                  write(*,1) 'no_wind_if_no_rotation'
               return
            end if
            if (s% dt > 0 .and. s% dt < s% mass_change_full_on_dt) then
               if (s% dt <= s% mass_change_full_off_dt) then
                  s% mstar_dot = 0
                  if (s% trace_dt_control_mass_change) &
                     write(*,1) 'no wind: dt <= mass_change_full_off_dt'
                  return
               end if
               !log_dtyr = log10(s% dt/secyer)
               !log_dtyr_full_on = log10(s% mass_change_full_on_dt/secyer)
               !log_dtyr_full_off = log10(s% mass_change_full_off_dt/secyer)
               !alfa = (log_dtyr - log_dtyr_full_off) / &
               !         (log_dtyr_full_on - log_dtyr_full_off)
               alfa = (s% dt - s% mass_change_full_off_dt)/ &
                        (s% mass_change_full_on_dt - s% mass_change_full_off_dt)
               if (s% trace_dt_control_mass_change) &
                  write(*,1) 'reduce wind: dt <= mass_change_full_on_dt', alfa
               wind_mdot = wind_mdot*alfa
            end if
         end if
         
         if (wind_mdot >= 0 .and. s% super_eddington_wind_eta <= 0) then 
            ! check for super eddington boost to wind
            L_div_Ledd = L1 / s% prev_Ledd
            full_off = s% wind_boost_full_off_L_div_Ledd
            if (L_div_Ledd > full_off) then
               full_on = s% wind_boost_full_on_L_div_Ledd
               max_boost = s% super_eddington_wind_max_boost
               if (L_div_Ledd >= full_on) then
                  super_eddington_boost = max_boost
               else
                  super_eddington_boost = &
                     1 + (max_boost-1)*(L_div_Ledd - full_off)/(full_on - full_off)
               end if
               wind_mdot = wind_mdot*super_eddington_boost
               if (s% trace_super_eddington_wind_boost) then
                  write(*,1) 'super eddington wind boost factor, L_div_Ledd', &
                     super_eddington_boost, L_div_Ledd
                  write(*,*)
               end if
            end if
         end if
         
         if (wind_mdot >= 0 .and. s% min_wind > 0 .and. &
               wind_mdot < s% min_wind*Msun/secyer) then
            if (dbg) write(*,1) 'use s% min_wind', s% min_wind
            wind_mdot = s% min_wind*Msun/secyer
         end if

         if (wind_mdot >= 0 .and. s% max_wind > 0 .and. &
               wind_mdot > s% max_wind*Msun/secyer) then
            if (dbg) write(*,1) 'use s% max_wind', s% max_wind
            wind_mdot = s% max_wind*Msun/secyer
         end if
         
         s% mstar_dot = -wind_mdot
         if (s% mstar_dot < 0 .and. &
               (s% min_wind > 0 .or. &
                  using_wind_scheme_mdot .or. &
                     s% v_div_v_crit_avg_surf > 0.8)) then
            call rotation_enhancement(ierr)
            if (ierr /= 0) then
               if (dbg .or. s% report_ierr) write(*, *) 'set_mdot: rotation_enhancement ierr'
               return
            end if
         end if         
         
         if (dbg) then
            write(*,1) 'final lg s% mstar_dot/(Msun/secyer)', safe_log10(abs(s% mstar_dot/(Msun/secyer)))
            write(*,*)
         end if


         contains
         
         
         subroutine rotation_enhancement(ierr)
            use star_utils, only: eval_kh_timescale
            integer, intent(out) :: ierr
            ! as in Heger, Langer, and Woosley, 2000, ApJ, 528:368-396.  section 2.6
            ! Mdot = Mdot_no_rotation/(1 - Osurf/Osurf_crit)^mdot_omega_power
            ! where Osurf = angular velocity at surface
            !       Osurf_crit^2 = (1 - Gamma_edd)*G*M/R_equatorial^3
            !       Gamma_edd = kappa*L/(4 pi c G M), Eddington factor
            real(dp) :: enhancement, wind_mdot, &
               kh_timescale, mdot_lim, wind_mdot_prev, dmsfac, dmskhf, &
               wind_mdot_lim, v_div_v_crit_full_on, v_div_v_crit_full_off

            include 'formats'
            
            ierr = 0
            
            if (.not. s% rotation_flag) return
            if (s% mdot_omega_power <= 0) return
            if (s% mstar_dot >= 0) return

            wind_mdot = -s% mstar_dot
            
            kh_timescale = eval_kh_timescale(s% cgrav(1), M1, R1, L1)
            dmskhf = s% rotational_mdot_kh_fac
            dmsfac = s% rotational_mdot_boost_fac
            wind_mdot_lim = min(dmskhf*M1/kh_timescale, wind_mdot*dmsfac)
            
            if (.false. .and. s% v_div_v_crit_avg_surf > 0.99d0) then
               enhancement = wind_mdot_lim/wind_mdot
            else if (.false.) then
               v_div_v_crit_full_on = 10
               v_div_v_crit_full_off = 0.9
               
               
               if (s% v_div_v_crit_avg_surf < v_div_v_crit_full_off) then
                  enhancement = 1
               else if (s% v_div_v_crit_avg_surf > v_div_v_crit_full_on) then
                  enhancement = s% max_rotational_mdot_boost
               else
                  enhancement = s% max_rotational_mdot_boost * &
                     (s% v_div_v_crit_avg_surf - v_div_v_crit_full_off)/&
                        (v_div_v_crit_full_on - v_div_v_crit_full_off)
               end if
            else
               enhancement = max(1d-22, 1d0 - s% v_div_v_crit_avg_surf)**(-s% mdot_omega_power)
               if (s% max_rotational_mdot_boost > 0 .and. &
                     enhancement > s% max_rotational_mdot_boost) then
                  enhancement = s% max_rotational_mdot_boost
               end if
            end if
            if (enhancement > s% lim_trace_rotational_mdot_boost) then
               if (.false.) write(*,1) &
                  'mdot rotation enhancement factor for mdot, v_div_v_crit_avg_surf', &
                  enhancement, s% v_div_v_crit_avg_surf
            end if
            
            if (wind_mdot*enhancement < wind_mdot_lim) then
               wind_mdot = wind_mdot*enhancement
               if (.false.) write(*,2) 'wind_mdot = wind_mdot*enhancement', &
                  s% model_number, enhancement, &
                  log10(wind_mdot/(Msun/secyer)), log10(wind_mdot_lim/(Msun/secyer))
            else
               enhancement = wind_mdot_lim/wind_mdot
               wind_mdot = wind_mdot_lim
               if (.false.) write(*,2) 'wind_mdot = wind_mdot_lim', &
                  s% model_number, log10(wind_mdot/(Msun/secyer))
            end if
            
            wind_mdot_prev = -s% mstar_dot_old
            if (s% generations > 2 .and. wind_mdot > 0 .and. wind_mdot_prev > 0) &
               wind_mdot = min(wind_mdot, s% max_mdot_jump_for_rotation*wind_mdot_prev)
            s% mstar_dot = -wind_mdot
            s% rotational_mdot_boost = enhancement
            
            !write(*,2) 'v_div_v_crit_avg_surf', s% model_number, s% v_div_v_crit_avg_surf
            !write(*,2) 'mdot_omega_power', s% model_number, s% mdot_omega_power
            !write(*,2) 'max_rotational_mdot_boost', s% model_number, s% max_rotational_mdot_boost
            !write(*,2) 'rotational_mdot_boost', s% model_number, s% rotational_mdot_boost

         end subroutine rotation_enhancement
         
         
         subroutine eval_vink_wind(w)
            real(dp), intent(inout) :: w
            real(dp) :: alfa, w1, w2, Teff_jump, logMdot, dT, vinf_div_vesc

            ! alfa = 1 for hot side, = 0 for cool side
            if (T1 > 27500d0) then
               alfa = 1
            else if (T1 < 22500d0) then
               alfa = 0
            else ! use Vink et al 2001, eqns 14 and 15 to set "jump" temperature
               Teff_jump = 1d3*(61.2d0 + 2.59d0*(-13.636d0 + 0.889d0*log10(Z/Zsolar)))
               dT = 100d0
               if (T1 > Teff_jump + dT) then
                  alfa = 1
               else if (T1 < Teff_jump - dT) then
                  alfa = 0
               else
                  alfa = (T1 - (Teff_jump - dT)) / (2*dT)
               end if
            end if
            
            if (alfa > 0) then ! eval hot side wind (eqn 24)
               vinf_div_vesc = 2.6d0 ! this is the hot side galactic value
               vinf_div_vesc = vinf_div_vesc*(Z/Zsolar)**0.13d0 ! corrected for Z
               logMdot = &
                  - 6.697d0 &
                  + 2.194d0*log10(L1/Lsun/1d5) &
                  - 1.313d0*log10(M1/Msun/30) &
                  - 1.226d0*log10(vinf_div_vesc/2d0) &
                  + 0.933d0*log10(T1/4d4) &
                  - 10.92d0*log10(T1/4d4)**2 &
                  + 0.85d0*log10(Z/Zsolar)
               w1 = 10**logMdot
            else
               w1 = 0
            end if
            
            if (alfa < 1) then ! eval cool side wind (eqn 25)
               vinf_div_vesc = 1.3d0 ! this is the cool side galactic value
               vinf_div_vesc = vinf_div_vesc*(Z/Zsolar)**0.13d0 ! corrected for Z
               logMdot = &
                  - 6.688d0 &
                  + 2.210d0*log10(L1/Lsun/1d5) &
                  - 1.339d0*log10(M1/Msun/30) &
                  - 1.601d0*log10(vinf_div_vesc/2d0) &
                  + 1.07d0*log10(T1/2d4) &
                  + 0.85d0*log10(Z/Zsolar)
               w2 = 10**logMdot
            else
               w2 = 0
            end if
            
            w = alfa*w1 + (1 - alfa)*w2
            
            if (dbg) write(*,*) 'vink wind', w
            
         end subroutine eval_vink_wind
         
         
         subroutine eval_Kudritzki_wind(w)
            use mod_kuma, only: kuma
            real(dp), intent(inout) :: w
            real(dp) :: xlogl, xlteff, stmass, xsurf, xmdfic, vinfkm
            xlogl = log10(L1/Lsun)
            xlteff = log10(T1)
            stmass = M1/Msun
            xsurf = surface_h1
            ! output xmdfic = mass loss rate [Msun/yr]
            ! output vinfkm = v_infinity
            call kuma(xlogl,xlteff,stmass,xsurf,xmdfic,vinfkm)
            w = xmdfic*s% Kudritzki_wind_eta
            if (dbg) write(*,*) 'lg eval_Kudritzki_wind', log10(w)
         end subroutine eval_Kudritzki_wind

         
         subroutine eval_blocker_wind(w)
            real(dp), intent(inout) :: w
            w = w * s% Blocker_wind_eta * &
               4.83d-9 * ((M1/Msun)**(-2.1d0)) * ((L1/Lsun)**2.7d0)
            if (dbg) write(*,*) 'blocker wind', w
         end subroutine eval_blocker_wind
         
         
         subroutine eval_highT_Dutch(w)
            real(dp), intent(out) :: w
            include 'formats'
            if (surface_h1 < 0.4d0) then ! helium rich Wolf-Rayet star: Nugis & Lamers
               w = 1d-11 * (L1/Lsun)**1.29d0 * Y**1.7d0 * Z**0.5d0
               if (dbg) write(*,1) 'Dutch_wind = Nugis & Lamers', log10(wind)
            else             
               call eval_vink_wind(w)
            end if
         end subroutine eval_highT_Dutch
         
         
         subroutine eval_lowT_Dutch(w)
            real(dp), intent(out) :: w
            include 'formats'
            if (s% Dutch_wind_lowT_scheme == 'de Jager') then
               call eval_de_Jager_wind(w)
               if (dbg) write(*,1) 'Dutch_wind = de Jager', log10(wind), T1, T_low, T_high
            else if (s% Dutch_wind_lowT_scheme == 'van Loon') then
               call eval_van_Loon_wind(w)
               if (dbg) write(*,1) 'Dutch_wind = van Loon', log10(wind), T1, T_low, T_high
            else if (s% Dutch_wind_lowT_scheme == 'Nieuwenhuijzen') then
               call eval_Nieuwenhuijzen_wind(w)
               if (dbg) write(*,1) 'Dutch_wind = Nieuwenhuijzen', log10(wind), T1, T_low, T_high
            else
               write(*,*) 'unknown value for Dutch_wind_lowT_scheme ' // &
                  trim(s% Dutch_wind_lowT_scheme)
               w = 0
            end if
         end subroutine eval_lowT_Dutch
         
         
         subroutine eval_de_Jager_wind(w)
            ! de Jager, C., Nieuwenhuijzen, H., & van der Hucht, K. A. 1988, A&AS, 72, 259.
            real(dp), intent(out) :: w
            real(dp) :: log10w
            include 'formats'
            log10w = 1.769d0*log10(L1/Lsun) - 1.676d0*log10(T1) - 8.158d0
            w = 10d0**log10w
            if (dbg) then
               write(*,1) 'de_Jager log10(wind)', log10w
            end if
         end subroutine eval_de_Jager_wind
         
         
         subroutine eval_van_Loon_wind(w)
            ! van Loon et al. 2005, A&A, 438, 273
            real(dp), intent(out) :: w
            real(dp) :: log10w
            include 'formats'
            log10w = -5.65d0 + 1.05*log10(L1/(1d4*Lsun)) - 6.3d0*log10(T1/35d2)
            w = 10d0**log10w
         end subroutine eval_van_Loon_wind
         
         
         subroutine eval_Nieuwenhuijzen_wind(w)
            ! Nieuwenhuijzen, H.; de Jager, C. 1990, A&A, 231, 134 (eqn 2)
            real(dp), intent(out) :: w
            real(dp) :: log10w
            include 'formats'
            log10w = -14.02d0 + &
                     1.24d0*log10(L1/Lsun) + &
                     0.16d0*log10(M1/Msun) + &
                     0.81d0*log10(R1/Rsun)
            w = 10d0**log10w
            if (dbg) then
               write(*,1) 'Nieuwenhuijzen log10(wind)', log10w
            end if
         end subroutine eval_Nieuwenhuijzen_wind
         
         
         
         
         
      end subroutine set_mdot




      end module winds
