! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module timestep
      
      use star_private_def
      use const_def
      use utils_lib, only:is_bad_num

      implicit none

      logical, parameter :: dbg_timestep = .false.
      real(dp) :: max_dt
      integer :: why_Tlim

      contains


      integer function timestep_controller(s, max_timestep)
         ! if don't return keep_going, then set result_reason to say why.
         type (star_info), pointer :: s
         real(dp), intent(in) :: max_timestep
         
         include 'formats'
                  
         timestep_controller = keep_going
         s% dt_next = s% hydro_dt_next 
         
         max_dt = max_timestep
         if (max_dt <= 0) max_dt = 1d99
         if (s% dt_next > max_dt) then
            s% dt_next = max_dt
            s% why_Tlim = Tlim_max_timestep
            if (s% report_why_dt_limits) write(*, *) 'dt_next > max_timestep', max_dt
         end if
         
         if (s% timestep_hold >= s% model_number .and. s% dt_next > s% dt) then
            s% dt_next = min(max_dt, s% dt)
            s% why_Tlim = Tlim_timestep_hold
            if (s% report_why_dt_limits) write(*, *) 'timestep_hold >= model_number'
         end if

         if (s% doing_first_model_of_run) return
                  
         ! check for special situations that might require reduced timestep         
         timestep_controller = do_timestep_limits(s, s% dt, s% dt_next)
         
         if (timestep_controller /= keep_going) then
            s% result_reason = timestep_limits
         end if
         
         s% dt_next = min(max_dt, max(s% dt_next, s% dt*s% timestep_factor_for_backups))
         
         if (is_bad_num(s% dt_next)) then
            write(*, *) 'timestep_controller: dt_next', s% dt_next
            timestep_controller = terminate
            s% termination_code = t_timestep_controller
            return
         end if
         
         if (dbg_timestep) then
            write(*,*) 'final result from timestep_controller for model_number', &
               timestep_controller, s% model_number
            write(*,1) 'lg dt/secyer', log10(s% dt/secyer)
            write(*,1) 'lg dt_next/secyer', log10(s% dt_next/secyer)
            write(*,1) 'dt_next/dt', s% dt_next/s% dt
            write(*,*)
            write(*,*)
         end if
         
      end function timestep_controller
      
      
      integer function do_timestep_limits(s, dt, dt_next)
         use rates_def, only: i_rate
         use chem_def, only: chem_isos, ipp, icno, category_name
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt ! timestep just completed
         real(dp), intent(inout) :: dt_next 
            ! input: next timestep from structure
            ! output: the desired new timestep

         real(dp) :: Tlimit(numTlim), dt_limit
         integer :: i_limit, nz, ierr
         logical :: skip_hard_limit


         integer :: num_mix_boundaries ! boundaries of regions with mixing_type /= no_mixing
         real(dp), pointer :: mix_bdy_q(:) ! (num_mix_boundaries)
         integer, pointer :: mix_bdy_loc(:) ! (num_mix_boundaries)

         
         include 'formats'
                  
         skip_hard_limit = ((s% relax_hard_limits_after_backup) .and. &
                           (s% timestep_hold >= s% model_number)) .or. &
                           ((s% relax_hard_limits_after_retry) .and. &
                           (s% model_number_for_last_retry == s% model_number)) 
         !skip_hard_limit = .false.
         
         nz = s% nz
         
         ! NOTE: when we get here, complete_model has called the report routine, 
         ! so we can use information that it has calculated

         ierr = 0
         
         num_mix_boundaries = s% num_mix_boundaries
         mix_bdy_q => s% mix_bdy_q
         mix_bdy_loc => s% mix_bdy_loc
                  
         Tlimit(:) = 1d99

         do_timestep_limits = check_jacobian_limit( &
            s, skip_hard_limit, dt, Tlimit(Tlim_num_jacobians))
         if (return_now(Tlim_num_jacobians)) return
         
         do_timestep_limits = check_diffusion_steps_limit( &
               s, skip_hard_limit, dt, Tlimit(Tlim_num_diff_solver_steps))
         if (return_now(Tlim_num_diff_solver_steps)) return
         
         do_timestep_limits = check_diffusion_iters_limit( &
               s, skip_hard_limit, dt, Tlimit(Tlim_num_diff_solver_iters))
         if (return_now(Tlim_num_diff_solver_iters)) return
         
         do_timestep_limits = check_v_div_v_crit( &
               s, skip_hard_limit, dt, Tlimit(Tlim_v_div_v_crit))
         if (return_now(Tlim_v_div_v_crit)) return
         
         do_timestep_limits = check_min_x_expected_limit( &
               s, skip_hard_limit, dt, Tlimit(Tlim_min_x_expected))
         if (return_now(Tlim_min_x_expected)) return
         
         do_timestep_limits = check_max_x_expected_limit( &
               s, skip_hard_limit, dt, Tlimit(Tlim_max_x_expected))
         if (return_now(Tlim_max_x_expected)) return
         
         do_timestep_limits = check_burn_max_iters_limit( &
               s, skip_hard_limit, dt, Tlimit(Tlim_num_burn_max_iters))
         if (return_now(Tlim_num_burn_max_iters)) return
         
         do_timestep_limits = check_dX(s, 0, skip_hard_limit, dt, &
            num_mix_boundaries, mix_bdy_loc, mix_bdy_q, &
            Tlimit(Tlim_dH), Tlimit(Tlim_dH_div_H))
         if (return_now(0)) return
         
         do_timestep_limits = check_dX(s, 1, skip_hard_limit, dt, &
            num_mix_boundaries, mix_bdy_loc, mix_bdy_q, &
            Tlimit(Tlim_dHe), Tlimit(Tlim_dHe_div_He))
         if (return_now(0)) return
         
         do_timestep_limits = check_dX(s, 2, skip_hard_limit, dt, &
            num_mix_boundaries, mix_bdy_loc, mix_bdy_q, &
            Tlimit(Tlim_dHe3), Tlimit(Tlim_dHe3_div_He3))
         if (return_now(0)) return
         
         do_timestep_limits = check_dX(s, -1, skip_hard_limit, dt, &
            num_mix_boundaries, mix_bdy_loc, mix_bdy_q, &
            Tlimit(Tlim_dX), Tlimit(Tlim_dX_div_X))
         if (return_now(0)) return

         do_timestep_limits = check_dL_div_L( &
            s, skip_hard_limit, dt, Tlimit(Tlim_dL_div_L))
         if (return_now(Tlim_dL_div_L)) return
         
         do_timestep_limits = check_dlgP_change( &
               s, skip_hard_limit, Tlimit(Tlim_dlgP))
         if (return_now(Tlim_dlgP)) return
         
         do_timestep_limits = check_dlgRho_change( &
               s, skip_hard_limit, Tlimit(Tlim_dlgRho))
         if (return_now(Tlim_dlgRho)) return
         
         do_timestep_limits = check_dlgT_change( &
               s, skip_hard_limit, Tlimit(Tlim_dlgT))
         if (return_now(Tlim_dlgT)) return
         
         do_timestep_limits = check_dlgR_change( &
               s, skip_hard_limit, Tlimit(Tlim_dlgR))
         if (return_now(Tlim_dlgR)) return
         
         do_timestep_limits = check_d_deltaR_shrink( &
               s, skip_hard_limit, Tlimit(Tlim_d_deltaR_shrink))
         if (return_now(Tlim_d_deltaR_shrink)) return
         
         do_timestep_limits = check_d_deltaR_grow( &
               s, skip_hard_limit, Tlimit(Tlim_d_deltaR_grow))
         if (return_now(Tlim_d_deltaR_grow)) return
         
         do_timestep_limits = check_lgL_nuc_cat_change( &
            s, num_mix_boundaries, mix_bdy_q, skip_hard_limit, Tlimit(Tlim_dlgL_nuc_cat))
         if (return_now(Tlim_dlgL_nuc_cat)) return

         do_timestep_limits = check_lgL_H_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlgL_H))
         if (return_now(Tlim_dlgL_H)) return
         
         do_timestep_limits = check_lgL_He_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlgL_He))
         if (return_now(Tlim_dlgL_He)) return
         
         do_timestep_limits = check_lgL_z_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlgL_z))
         if (return_now(Tlim_dlgL_z)) return
         
         do_timestep_limits = check_lgL_photo_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlgL_photo))
         if (return_now(Tlim_dlgL_photo)) return

         do_timestep_limits = check_lgL_nuc_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlgL_nuc))
         if (return_now(Tlim_dlgL_nuc)) return
         
         do_timestep_limits = check_dlgTeff_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlgTeff))
         if (return_now(Tlim_dlgTeff)) return
         
         do_timestep_limits = check_dlgRho_cntr_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlgRho_cntr))
         if (return_now(Tlim_dlgRho_cntr)) return
         
         do_timestep_limits = check_dlgT_cntr_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlgT_cntr))
         if (return_now(Tlim_dlgT_cntr)) return  
         
         do_timestep_limits = check_dlgT_max_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlgT_max))
         if (return_now(Tlim_dlgT_max)) return  
         
         do_timestep_limits = check_dlgRho_max_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlgRho_max))
         if (return_now(Tlim_dlgRho_max)) return  
         
         do_timestep_limits = check_dlog_eps_nuc_cntr_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlog_eps_nuc_cntr))
         if (return_now(Tlim_dlog_eps_nuc_cntr)) return   

         do_timestep_limits = check_dYe_change( &
               s, skip_hard_limit, Tlimit(Tlim_delta_Ye)) 
         if (return_now(Tlim_delta_Ye)) return

         do_timestep_limits = check_dYe_highT_change( &
               s, skip_hard_limit, Tlimit(Tlim_delta_Ye_highT)) 
         if (return_now(Tlim_delta_Ye_highT)) return
         
         do_timestep_limits = check_dlog_eps_nuc_change( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dlog_eps_nuc))
         if (return_now(Tlim_dlog_eps_nuc)) return                  
                  
         do_timestep_limits = check_lg_XH_cntr( &
               s, skip_hard_limit, Tlimit(Tlim_lg_XH_cntr))
         if (return_now(Tlim_lg_XH_cntr)) return
         
         do_timestep_limits = check_lg_XHe_cntr( &
               s, skip_hard_limit, Tlimit(Tlim_lg_XHe_cntr))
         if (return_now(Tlim_lg_XHe_cntr)) return

         do_timestep_limits = check_delta_mstar( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dmstar))
         if (return_now(Tlim_dmstar)) return

         do_timestep_limits = check_delta_mdot( &
               s, skip_hard_limit, dt, Tlimit(Tlim_del_mdot))
         if (return_now(Tlim_del_mdot)) return

         do_timestep_limits = check_delta_ang_momentum( &
               s, skip_hard_limit, dt, Tlimit(Tlim_deltaJ))
         if (return_now(Tlim_deltaJ)) return

         do_timestep_limits = check_mdot_eps_grav( &
               s, skip_hard_limit, dt, Tlimit(Tlim_mdot_eps_grav))
         if (return_now(Tlim_mdot_eps_grav)) return
         
         do_timestep_limits = check_delta_lgL( &
               s, skip_hard_limit, Tlimit(Tlim_lgL))
         if (return_now(Tlim_lgL)) return
         
         do_timestep_limits = check_op_split_diff( &
               s, skip_hard_limit, Tlimit(Tlim_op_split))
         if (return_now(Tlim_op_split)) return
         
         do_timestep_limits = check_delta_HR( &
               s, skip_hard_limit, Tlimit(Tlim_delta_HR))
         if (return_now(Tlim_delta_HR)) return
         
         do_timestep_limits = check_dX_nuc( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dX_nuc))
         if (return_now(Tlim_dX_nuc)) return
         
         do_timestep_limits = check_dX_nuc_drop( &
               s, skip_hard_limit, dt, Tlimit(Tlim_dX_nuc_drop))
         if (return_now(Tlim_dX_nuc_drop)) return
         
         s% why_Tlim = Tlim_struc
         i_limit = minloc(Tlimit(1:numTlim), dim=1)
         
         dt_limit = Tlimit(i_limit)*s% timestep_limit_factor
         if (dt_limit < dt_next) then
            dt_next = dt_limit
            !if (i_limit == Tlim_struc) i_limit = s% why_Tlim
            s% why_Tlim = i_limit
            if (s% report_why_dt_limits) then ! report reason for lowering timestep
               if (i_limit == Tlim_dX) then
                  write(*, '(a, 2i6, 2f16.9)') &
                     'reduce dt because of excessive dX  ' // &
                     trim(chem_isos% name(s% chem_id(s% Tlim_dX_species))), &
                     s% Tlim_dX_cell, s% model_number, s% star_mass*s% q(s% Tlim_dX_cell), &
                     log10(dt_next/secyer)
               else if (i_limit == Tlim_dX_nuc) then
                  write(*, '(a, 2i6, 2f16.9)') &
                     'reduce dt because of excessive dX_nuc  ' // &
                     trim(chem_isos% name(s% chem_id(s% Tlim_dXnuc_species))), &
                     s% Tlim_dXnuc_cell, s% model_number, s% star_mass*s% q(s% Tlim_dXnuc_cell), &
                     log10(dt_next/secyer)
               else if (i_limit == Tlim_dX_nuc_drop) then
                  write(*, '(a, 2i6, 99f16.9)') &
                     'reduce dt because of excessive dX_nuc_drop  ' // &
                     trim(chem_isos% name(s% chem_id(s% Tlim_dXnuc_drop_species))), &
                     s% Tlim_dXnuc_drop_cell, s% model_number, &
                     s% xa(s% Tlim_dXnuc_drop_species, s% Tlim_dXnuc_drop_cell) - &
                        s% xa_old(s% Tlim_dXnuc_drop_species, s% Tlim_dXnuc_drop_cell), &
                     s% xa(s% Tlim_dXnuc_drop_species, s% Tlim_dXnuc_drop_cell), &
                     s% xa_old(s% Tlim_dXnuc_drop_species, s% Tlim_dXnuc_drop_cell), &
                     s% star_mass*s% q(s% Tlim_dXnuc_drop_cell), &
                     log10(dt_next/secyer)
               else if (i_limit == Tlim_dX_div_X) then
                  write(*, '(a, 2i6, 2f16.9)') &
                     'reduce dt because of excessive dX_div_X  ' // &
                     trim(chem_isos% name(s% chem_id(s% Tlim_dX_div_X_species))), &
                     s% Tlim_dX_div_X_cell, s% model_number, s% star_mass*s% q(s% Tlim_dX_div_X_cell), &
                     log10(dt_next/secyer)
               else if (i_limit == Tlim_dH) then
                  write(*, '(a, 2i6, 2f16.9)') &
                     'reduce dt because of excessive dH', &
                     s% Tlim_dX_cell, s% model_number, s% star_mass*s% q(s% Tlim_dX_cell), &
                     log10(dt_next/secyer)
               else if (i_limit == Tlim_dH_div_H) then
                  write(*, '(a, 2i6, 2f16.9)') &
                     'reduce dt because of excessive dH_div_H', &
                     s% Tlim_dX_div_X_cell, s% model_number, s% star_mass*s% q(s% Tlim_dX_div_X_cell), &
                     log10(dt_next/secyer)
               else if (i_limit == Tlim_dHe) then
                  write(*, '(a, 2i6, 2f16.9)') &
                     'reduce dt because of excessive dHe', &
                     s% Tlim_dX_cell, s% model_number, s% star_mass*s% q(s% Tlim_dX_cell), &
                     log10(dt_next/secyer)
               else if (i_limit == Tlim_dHe_div_He) then
                  write(*, '(a, 2i6, 2f16.9)') &
                     'reduce dt because of excessive dHe_div_He', &
                     s% Tlim_dX_div_X_cell, s% model_number, s% star_mass*s% q(s% Tlim_dX_div_X_cell), &
                     log10(dt_next/secyer)
               else if (i_limit == Tlim_dHe3) then
                  write(*, '(a, 2i6, 2f16.9)') &
                     'reduce dt because of excessive dHe3', &
                     s% Tlim_dX_cell, s% model_number, s% star_mass*s% q(s% Tlim_dX_cell), &
                     log10(dt_next/secyer)
               else if (i_limit == Tlim_dHe3_div_He3) then
                  write(*, '(a, 2i6, 2f16.9)') &
                     'reduce dt because of excessive dHe3_div_He3', &
                     s% Tlim_dX_div_X_cell, s% model_number, s% star_mass*s% q(s% Tlim_dX_div_X_cell), &
                     log10(dt_next/secyer)
               else if (i_limit == Tlim_dlgL_nuc_cat) then
                  write(*, '(a, 2i6, 2f16.9)') &
                     'reduce dt because of excessive dlgL ' // &
                     trim(category_name(s% Tlim_dlgL_nuc_category)), &
                     s% Tlim_dlgL_nuc_cell, s% model_number, s% star_mass*s% q(s% Tlim_dlgL_nuc_cell), &
                     log10(dt_next/secyer)
               else if (i_limit == Tlim_neg_X) then
                  write(*, '(a, i6, f16.9)') &
                     'reduce dt because of negative mass fraction', &
                     s% model_number, log10(dt_next/secyer)
               else if (i_limit == Tlim_bad_Xsum) then
                  write(*, '(a, i6, f16.9)') &
                     'reduce dt because of bad sum of mass fractions', &
                     s% model_number, log10(dt_next/secyer)
               else
                  write(*, '(a, i6)') 'reduce dt because of ' // &
                     trim(dt_why_str(i_limit)), s% model_number
               end if
            end if
            
         end if
         
         
         contains
         
         logical function return_now(i_limit)
            integer, intent(in) :: i_limit
            if (do_timestep_limits == keep_going) then
               return_now = .false.
               return
            end if
            if (s% report_why_dt_limits) then ! report reason for lowering timestep
               if (i_limit == Tlim_dX) then
                  write(*, '(a, 2i6)') &
                     'retry because of excessive dX  ' // &
                     trim(chem_isos% name(s% chem_id(s% Tlim_dX_species))), &
                     s% Tlim_dX_cell, s% model_number
               else if (i_limit == Tlim_dX_div_X) then
                  write(*, '(a, 2i6)') &
                     'retry because of excessive dX_div_X  ' // &
                     trim(chem_isos% name(s% chem_id(s% Tlim_dX_div_X_species))), &
                     s% Tlim_dX_div_X_cell, s% model_number
               else if (i_limit == Tlim_dH) then
                  write(*, '(a, 2i6)') &
                     'reduce dt because of excessive dH', &
                     s% Tlim_dX_cell, s% model_number
               else if (i_limit == Tlim_dH_div_H) then
                  write(*, '(a, 2i6)') &
                     'reduce dt because of excessive dH_div_H', &
                     s% Tlim_dX_div_X_cell, s% model_number
               else if (i_limit == Tlim_dHe) then
                  write(*, '(a, 2i6)') &
                     'reduce dt because of excessive dHe', &
                     s% Tlim_dX_cell, s% model_number
               else if (i_limit == Tlim_dHe_div_He) then
                  write(*, '(a, 2i6)') &
                     'reduce dt because of excessive dHe_div_He', &
                     s% Tlim_dX_div_X_cell, s% model_number
               else if (i_limit == Tlim_dHe3) then
                  write(*, '(a, 2i6)') &
                     'reduce dt because of excessive dHe3', &
                     s% Tlim_dX_cell, s% model_number
               else if (i_limit == Tlim_dHe3_div_He3) then
                  write(*, '(a, 2i6)') &
                     'reduce dt because of excessive dHe3_div_He3', &
                     s% Tlim_dX_div_X_cell, s% model_number
               else if (i_limit == Tlim_dlgL_nuc_cat) then
                  write(*, '(a, 2i6)') &
                     'retry because of excessive change in lgL ' // &
                     trim(category_name(s% Tlim_dlgL_nuc_category)), &
                     s% Tlim_dlgL_nuc_cell, s% model_number
               else
                  write(*, '(a, i6)') 'retry because of ' // &
                     trim(dt_why_str(i_limit)), s% model_number
               end if
            end if
            if (i_limit > 0) s% why_Tlim = i_limit
            return_now = .true.
         end function return_now
         
      end function do_timestep_limits
      
      
      integer function check_error_limit( &
            s, limit, hard_limit, value, msg, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         real(dp), intent(in) :: limit, hard_limit, value
         character (len=*), intent(in) :: msg
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         if (value > hard_limit .and. hard_limit > 0 .and. (.not. skip_hard_limit)) then
            if (s% report_all_dt_limits) &
               write(*,'(a,3x,99e16.6)') trim(msg) // ' hard limit', hard_limit, value
            check_error_limit = retry
            return
         end if
         check_error_limit = keep_going
         if (value > limit .and. limit > 0) then
            dt_limit = dt*0.5d0*(1d0 + dble(limit)/value)
            if (s% report_all_dt_limits) write(*,*) trim(msg), limit, value
         end if
      end function check_error_limit
      
      
      integer function check_integer_limit( &
            s, limit, hard_limit, value, msg, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         integer, intent(in) :: limit, hard_limit, value
         character (len=*), intent(in) :: msg
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         if (value > hard_limit .and. hard_limit > 0 .and. (.not. skip_hard_limit)) then
            if (s% report_all_dt_limits) write(*,*) trim(msg) // ' hard limit', hard_limit, value
            check_integer_limit= retry
            return
         end if
         check_integer_limit = keep_going
         if (value > (limit*7)/10) then
            dt_limit = dt*0.5d0*(1d0 + dble(limit)/value)
            if (s% report_all_dt_limits) write(*,*) trim(msg), limit, value
         end if
      end function check_integer_limit
      
      
      integer function check_jacobian_limit(s, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         check_jacobian_limit = check_integer_limit( &
           s, s% jacobian_limit, s% jacobian_hard_limit, s% num_jacobians,  &
           'num_jacobians', skip_hard_limit, dt, dt_limit)
      end function check_jacobian_limit
         
            
      integer function check_diffusion_steps_limit(s, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         if (.not. s% do_element_diffusion) then
            check_diffusion_steps_limit = keep_going; return
         end if
         check_diffusion_steps_limit = check_integer_limit( &
           s, s% diffusion_steps_limit, s% diffusion_steps_hard_limit, &
           s% num_diffusion_solver_steps,  &
           'num_diffusion_solver_steps', skip_hard_limit, dt, dt_limit)
      end function check_diffusion_steps_limit
         
            
      integer function check_diffusion_iters_limit(s, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         if (.not. s% do_element_diffusion) then
            check_diffusion_iters_limit = keep_going; return
         end if
         check_diffusion_iters_limit = check_integer_limit( &
           s, s% diffusion_iters_limit, s% diffusion_iters_hard_limit, &
           s% num_diffusion_solver_iters,  &
           'num_diffusion_solver_iters', skip_hard_limit, dt, dt_limit)
      end function check_diffusion_iters_limit


      integer function check_burn_max_iters_limit(s, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         if (s% operator_coupling_choice == 0) then
            check_burn_max_iters_limit = keep_going; return
         end if
         check_burn_max_iters_limit = check_integer_limit( &
           s, s% burn_max_iters_limit, s% burn_max_iters_hard_limit, &
           s% num_burn_max_iters,  &
           'num_burn_max_iters', skip_hard_limit, dt, dt_limit)
      end function check_burn_max_iters_limit
         
         

      integer function check_dX(s, which, skip_hard_limit, dt, &
            n_mix_bdy, mix_bdy_loc, mix_bdy_q, dX_dt_limit, dX_div_X_dt_limit)
         use chem_def
         use num_lib, only: binary_search
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         integer, intent(in) :: which, n_mix_bdy, mix_bdy_loc(:)
         real(dp), intent(in) :: dt
         real(dp), intent(in), pointer :: mix_bdy_q(:)
         real(dp), intent(inout) :: dX_dt_limit, dX_div_X_dt_limit
         
         real(dp) :: X, X_old, delta_dX, delta_dX_div_X, max_dX, max_dX_div_X, &
            bdy_dist_dm, max_dX_bdy_dist_dm, max_dX_div_X_bdy_dist_dm, cz_dist_limit
         integer :: j, k, cid, bdy, max_dX_j, max_dX_k, max_dX_div_X_j, max_dX_div_X_k
         real(dp) :: D_mix_cutoff, dX_limit_min_X, dX_limit, dX_hard_limit
         real(dp) :: dX_div_X_limit_min_X, dX_div_X_limit, dX_div_X_hard_limit
         logical :: decreases_only
         
         include 'formats'
         
         check_dX = keep_going
         
         if (which == 0) then ! hydrogen
            dX_limit_min_X = s% dH_limit_min_H
            dX_limit = s% dH_limit
            dX_hard_limit = s% dH_hard_limit
            dX_div_X_limit_min_X = s% dH_div_H_limit_min_H
            dX_div_X_limit = s% dH_div_H_limit
            dX_div_X_hard_limit = s% dH_div_H_hard_limit
            decreases_only = s% dH_decreases_only
         else if (which == 1) then ! helium
            dX_limit_min_X = s% dHe_limit_min_He
            dX_limit = s% dHe_limit
            dX_hard_limit = s% dHe_hard_limit
            dX_div_X_limit_min_X = s% dHe_div_He_limit_min_He
            dX_div_X_limit = s% dHe_div_He_limit
            dX_div_X_hard_limit = s% dHe_div_He_hard_limit
            decreases_only = s% dHe_decreases_only
         else if (which == 2) then ! He3
            dX_limit_min_X = s% dHe3_limit_min_He3
            dX_limit = s% dHe3_limit
            dX_hard_limit = s% dHe3_hard_limit
            dX_div_X_limit_min_X = s% dHe3_div_He3_limit_min_He3
            dX_div_X_limit = s% dHe3_div_He3_limit
            dX_div_X_hard_limit = s% dHe3_div_He3_hard_limit
            decreases_only = s% dHe3_decreases_only
         else ! metals
            dX_limit_min_X = s% dX_limit_min_X
            dX_limit = s% dX_limit
            dX_hard_limit = s% dX_hard_limit
            dX_div_X_limit_min_X = s% dX_div_X_limit_min_X
            dX_div_X_limit = s% dX_div_X_limit
            dX_div_X_hard_limit = s% dX_div_X_hard_limit
            decreases_only = s% dX_decreases_only
         end if
         
         if (  dX_limit_min_X >= 1 .and. &
               dX_limit >= 1 .and. &
               dX_hard_limit >= 1 .and. &
               dX_div_X_limit_min_X >= 1 .and. &
               dX_div_X_limit >= 1 .and. &
               dX_div_X_hard_limit >= 1) then
            if (.false. .and. which == 2) then ! He3
               write(*,1) 'dX_limit', dX_limit
               write(*,1) 'dX_div_X_hard_limit', dX_div_X_hard_limit
               write(*,1) 'dX_limit_min_X', dX_limit_min_X
               write(*,1) 'dX_div_X_limit_min_X', dX_div_X_limit_min_X
            end if
            return
         end if
         
         max_dX = -1; max_dX_j = -1; max_dX_k = -1
         max_dX_div_X = -1; max_dX_div_X_j = -1; max_dX_div_X_k = -1
         bdy = 0
         max_dX_bdy_dist_dm = 0
         max_dX_div_X_bdy_dist_dm = 0
         cz_dist_limit = s% dX_mix_dist_limit*Msun

         if (s% set_min_D_mix .and. s% ye(s% nz) >= s% min_center_Ye_for_min_D_mix) then
            D_mix_cutoff = s% min_D_mix
         else
            D_mix_cutoff = 0
         end if

         if (.false. .and. which == 2) write(*,1) 'check all He3'
         
         do k = 1, s% nz
         
            if (s% D_mix(k) > D_mix_cutoff) then
               if (.false. .and. which == 2) write(*,2) 'check He3: s% D_mix(k) > D_mix_cutoff', &
                  k, s% D_mix(k), D_mix_cutoff
               cycle
            end if
            if (k < s% nz) then
               if (s% D_mix(k+1) > D_mix_cutoff) then
                  if (.false. .and. which == 2) write(*,2) 'check He3: s% D_mix(k+1) > D_mix_cutoff', &
                     k, s% D_mix(k+1), D_mix_cutoff
                  cycle
               end if
            end if
            if (s% newly_nonconvective(k)) then
               cycle
            end if
            
            ! find the nearest mixing boundary
            bdy = binary_search(n_mix_bdy, mix_bdy_q, bdy, s% q(k))
            ! don't check cells near a mixing boundary
            if (bdy > 0 .and. bdy < n_mix_bdy) then
               bdy_dist_dm = s% xmstar*abs(s% q(k) - mix_bdy_q(bdy))
               if (bdy_dist_dm < cz_dist_limit) cycle
            else
               bdy_dist_dm = 0
            end if
            
            if (.false. .and. which == 2) write(*,2) 'check He3', k

            do j = 1, s% species
               
               cid = s% chem_id(j)
               if (which == 0) then ! hydrogen
                  if (cid /= ih1) cycle
               else if (which == 1) then ! helium
                  if (cid /= ihe4) cycle
               else if (which == 2) then ! he3
                  if (cid /= ihe3) cycle
               else ! other
                  if (chem_isos% Z(cid) <= 2) cycle
               end if
               
               X = s% xa(j,k)
               X_old = s% xa_old(j,k)
               delta_dX = X_old - X ! decrease in abundance
               if (.false. .and. which == 2) write(*,2) 'delta_dHe3', k, delta_dX

               if ((.not. decreases_only) .and. delta_dX < 0) delta_dX = -delta_dX
               
               if (X >= dX_limit_min_X) then
                  if ((.not. skip_hard_limit) .and. delta_dX > dX_hard_limit) then
                     dX_dt_limit = 0
                     check_dX= retry
                     s% why_Tlim = Tlim_dX
                     s% Tlim_dX_species = j
                     s% Tlim_dX_cell = k
                     if (s% report_why_dt_limits) then
                        write(*, '(a30, i5, 99(/,a30,e20.10))') &
                           'dX ' // trim(chem_isos% name(s% chem_id(j))), &
                           k, 'delta_dX', delta_dX, 'dX_hard_limit', dX_hard_limit
                     end if
                     return
                  end if
                  if (delta_dX > max_dX) then
                     max_dX = delta_dX
                     max_dX_j = j
                     max_dX_k = k
                     max_dX_bdy_dist_dm = bdy_dist_dm
                  end if
               end if
               if (X >= dX_div_X_limit_min_X) then
                  delta_dX_div_X = delta_dX/X
                  if ((.not. skip_hard_limit) .and. delta_dX_div_X > dX_div_X_hard_limit) then
                     dX_div_X_dt_limit = 0
                     check_dX= retry
                     s% why_Tlim = Tlim_dX_div_X
                     s% Tlim_dX_div_X_species = j
                     s% Tlim_dX_div_X_cell = k
                     if (s% report_why_dt_limits) then
                        write(*, '(a30, i5, 99(/,a30,e20.10))') &
                           'delta_dX_div_X ' // trim(chem_isos% name(s% chem_id(j))), &
                           k, 'delta_dX_div_X', delta_dX_div_X, 'dX_div_X_hard_limit', dX_div_X_hard_limit
                     end if
                     return
                  end if
                  if (delta_dX_div_X > max_dX_div_X) then
                     max_dX_div_X = delta_dX_div_X
                     max_dX_div_X_j = j
                     max_dX_div_X_k = k
                     max_dX_div_X_bdy_dist_dm = bdy_dist_dm
                  end if
               end if
            end do
         end do

         if (max_dX > dX_limit) then
            dX_dt_limit = dt*dX_limit/max_dX
            j = max_dX_j
            k = max_dX_k
            s% Tlim_dX_species = j
            s% Tlim_dX_cell = k
            write(*, '(a30, i5, 99e20.10)') &
               'dX ' // trim(chem_isos% name(s% chem_id(j))), &
               k, max_dX, dX_limit, (s% M_center + s% xmstar*(s% q(k) - s% dq(k)/2))/Msun, &
               max_dX_bdy_dist_dm/Msun
            if (s% report_all_dt_limits) then
               write(*, 2) 's% D_mix(k)', k, s% D_mix(k)
               X_old = s% xa_old(j,k)
               write(*, 2) 'X_old', k, X_old
               write(*, 2) 's% xa(j,k)', k, s% xa(j,k)
               write(*, 2) 'dX', k, s% xa(j,k) - X_old
               do j=k-5,k+5
                  write(*,2) 'D_mix', j, s% D_mix(j), s% q(j) - s% q(k)
               end do
               write(*,*)
               !stop 'check_dX'
            end if
         end if

         if (max_dX_div_X > dX_div_X_limit) then
            dX_div_X_dt_limit = dt*dX_div_X_limit/max_dX_div_X
            s% Tlim_dX_div_X_species = max_dX_div_X_j
            s% Tlim_dX_div_X_cell = max_dX_div_X_k
            if (which > 2) write(*, '(a30, i5, 99e20.10)') &
               'limit dt because of large dX_div_X ' // &
                  trim(chem_isos% name(s% chem_id(max_dX_div_X_j))) // &
                  ' k, max, lim, m ', &
               max_dX_div_X_k, max_dX_div_X, dX_div_X_limit, &
               max_dX_div_X_bdy_dist_dm/Msun
            if (s% report_all_dt_limits) then
               write(*,*)
               k = max_dX_div_X_k
               j = max_dX_div_X_j
               X = s% xa(j,k)
               X_old = s% xa_old(j,k)
               write(*,'(a)') &
                  '                                                X, X_old, X-X_old, (X-X_old)/X'
               do j = 1, s% species
                  write(*,2) trim(chem_isos% name(s% chem_id(j))), k, X, X_old, X - X_old, (X - X_old)/X
               end do
               write(*,2) 'which', which
               write(*,*) 's% report_all_dt_limits', s% report_all_dt_limits
               bdy = binary_search(n_mix_bdy, mix_bdy_q, 0, s% q(k))
               if (bdy > 0) then
                  write(*,2) 'q', k, s% q(k)
                  write(*,2) 'nearest cz bdy', mix_bdy_loc(bdy), mix_bdy_q(bdy)
                  write(*,2) 'cz_dq', mix_bdy_loc(bdy), s% q(k) - mix_bdy_q(bdy)
                  write(*,*)
               end if
               write(*,*)
            end if
         end if
         
         if (.false. .and. which == 2) then ! He3
            write(*,1) 'max_dX_div_X', max_dX_div_X
            write(*,1) 'dX_div_X_limit', dX_div_X_limit
            stop 'check_dX'
         end if
         
      end function check_dX


      integer function check_dL_div_L(s, skip_hard_limit, dt, dL_div_L_dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dL_div_L_dt_limit
         
         real(dp) :: L, abs_dL, abs_dL_div_L, max_dL_div_L
         integer :: k, max_dL_div_L_k
         real(dp) :: dL_div_L_limit_min_L, dL_div_L_limit, dL_div_L_hard_limit
         
         check_dL_div_L = keep_going
         
         dL_div_L_limit_min_L = Lsun*s% dL_div_L_limit_min_L
         dL_div_L_limit = s% dL_div_L_limit
         dL_div_L_hard_limit = s% dL_div_L_hard_limit
         
         if (dL_div_L_limit_min_L <= 0) return
         if (dL_div_L_limit <= 0 .and. dL_div_L_hard_limit <= 0) return
         
         max_dL_div_L = -1
         max_dL_div_L_k = -1
         abs_dL_div_L = 0; L=0 ! to quiet gfortran
         
         do k = 1, s% nz
            L = s% L(k)
            abs_dL = abs(L - s% L_start(k))
            if (L >= dL_div_L_limit_min_L) then
               abs_dL_div_L = abs_dL/L
               if (dL_div_L_hard_limit > 0 .and. (.not. skip_hard_limit) &
                     .and. abs_dL_div_L > dL_div_L_hard_limit) then
                  dL_div_L_dt_limit = 0
                  check_dL_div_L= retry
                  if (s% report_all_dt_limits) write(*, '(a30, i5, 99e20.10)') &
                     'dL_div_L too large at', k, L, s% L_start(k), abs_dL_div_L, dL_div_L_limit
                  return
               end if
               if (abs_dL_div_L > max_dL_div_L) then
                  max_dL_div_L = abs_dL_div_L
                  max_dL_div_L_k = k
               end if
            end if
         end do

         if (dL_div_L_limit > 0 .and. max_dL_div_L > dL_div_L_limit) then
            dL_div_L_dt_limit = dt*dL_div_L_limit/max_dL_div_L
            if (s% report_all_dt_limits) write(*, '(a30, i5, 99e20.10)') &
               'dL_div_L too large at', max_dL_div_L_k, max_dL_div_L, dL_div_L_limit
         end if
         
      end function check_dL_div_L
      
      
      integer function check_change( &
            s, delta_value, lim, hard_lim, i, msg, skip_hard_limit, dt_limit, relative_excess)
         use const_def, only:ln10
         type (star_info), pointer :: s
         real(dp), intent(in) :: delta_value, lim, hard_lim
         integer, intent(in) :: i
         character (len=*), intent(in) :: msg
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp), intent(out) :: relative_excess
         real(dp) :: abs_change
         check_change = keep_going
         abs_change = abs(delta_value)
         if (hard_lim > 0 .and. abs_change > hard_lim .and. (.not. skip_hard_limit)) then
            if (s% report_all_dt_limits) &
               write(*, '(a30, f20.10, i5, 99e20.10)') trim(msg) // ' hard limit', &
                  log10(dt_limit/secyer), i, delta_value, hard_lim
            check_change= retry
            return
         end if
         if (lim <= 0) return
         relative_excess = (abs_change - lim) / lim
         if (relative_excess > 0) then
            dt_limit = s% dt * s% timestep_dt_factor**relative_excess
            if (s% report_all_dt_limits) write(*, '(a30, f20.10, i5, 99e20.10)') trim(msg), &
               log10(dt_limit/secyer), i, delta_value, relative_excess, abs_change, lim
         end if
      end function check_change
         
            
      integer function check_max_x_expected_limit(s, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         check_max_x_expected_limit = keep_going
         if (s% max_x_expected <= s% max_x_expected_max_limit) return
         if ((.not. skip_hard_limit) .and. &
               s% max_x_expected > s% max_x_expected_max_hard_limit) then
            check_max_x_expected_limit = retry
            if (s% report_all_dt_limits) &
               write(*, '(a30, f20.10, 99e20.10)') 'max_x_expected hard limit', &
                  log10(dt_limit/secyer), s% max_x_expected, s% max_x_expected_max_hard_limit
            return
         end if
         dt_limit = dt*s% max_x_expected_max_limit/s% max_x_expected
         if (s% report_all_dt_limits) write(*, '(a30, f20.10, 99e20.10)') &
            'max_x_expected: log dt, max, limit', log10(dt_limit/secyer), &
            s% max_x_expected, s% max_x_expected_max_limit
      end function check_max_x_expected_limit
         
            
      integer function check_min_x_expected_limit(s, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         check_min_x_expected_limit = keep_going
         if (s% min_x_expected >= s% min_x_expected_min_limit) return
         if ((.not. skip_hard_limit) .and. &
               s% min_x_expected < s% min_x_expected_min_hard_limit) then
            check_min_x_expected_limit = retry
            if (s% report_all_dt_limits) &
               write(*, '(a30, f20.10, 99e20.10)') 'min_x_expected hard limit', &
                  log10(dt_limit/secyer), s% min_x_expected, s% min_x_expected_min_hard_limit
            return
         end if
         dt_limit = dt*s% min_x_expected_min_limit/s% min_x_expected
         if (s% report_all_dt_limits) write(*, '(a30, f20.10, 99e20.10)') &
            'min_x_expected: log dt, min, limit', log10(dt_limit/secyer), &
            s% min_x_expected, s% min_x_expected_min_limit
      end function check_min_x_expected_limit
      
      
      integer function check_dlgP_change(s, skip_hard_limit, dt_limit) ! check max change in log10(pressure)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, max_diff
         integer :: i, nz
         nz = s% nz
         include 'formats'
         i = maxloc(abs(s% lnP(1:nz) - s% lnP_start(1:nz)), dim=1)
         max_diff = s% lnP(i) - s% lnP_start(i)
         check_dlgP_change = check_change(s, max_diff/ln10, &
            s% delta_lgP_limit, s% delta_lgP_hard_limit, &
            i, 'check_dlgP_change', skip_hard_limit, dt_limit, relative_excess)
      end function check_dlgP_change
      
      
      integer function check_dlgRho_change(s, skip_hard_limit, dt_limit) 
      ! check max change in log10(density)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, max_diff
         integer :: i, nz
         nz = s% nz
         i = maxloc(abs(s% lnd(1:nz) - s% lnd_start(1:nz)), dim=1)
         max_diff = s% lnd(i) - s% lnd_start(i)
         check_dlgRho_change = check_change(s, max_diff/ln10, &
            s% delta_lgRho_limit, s% delta_lgRho_hard_limit, &
            i, 'check_dlgRho_change', skip_hard_limit, dt_limit, relative_excess)
      end function check_dlgRho_change
      
      
      integer function check_dlgT_change(s, skip_hard_limit, dt_limit) 
      ! check max change in log10(temperature)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, max_diff
         integer :: i, nz
         include 'formats'
         nz = s% nz
         i = maxloc(abs(s% lnT(1:nz) - s% lnT_start(1:nz)), dim=1)
         max_diff = s% lnT(i) - s% lnT_start(i)
         check_dlgT_change = check_change(s, max_diff/ln10, &
            s% delta_lgT_limit, s% delta_lgT_hard_limit, &
            i, 'check_dlgT_change', skip_hard_limit, dt_limit, relative_excess)
      end function check_dlgT_change
      
      
      integer function check_dlgR_change(s, skip_hard_limit, dt_limit)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, max_diff
         integer :: i, nz
         nz = s% nz
         i = maxloc(abs(s% lnR(1:nz) - s% lnR_start(1:nz)), dim=1)
         max_diff = s% lnR(i) - s% lnR_start(i)
         check_dlgR_change = check_change(s, max_diff/ln10, &
            s% delta_lgR_limit, s% delta_lgR_hard_limit, &
            i, 'check_dlgR_change', skip_hard_limit, dt_limit, relative_excess)
      end function check_dlgR_change
      
      
      integer function check_d_deltaR_shrink(s, skip_hard_limit, dt_limit)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: Rm1, R00, dR_pre, dR, diff, relative_excess, max_diff, &
            max_Rm1, max_R00
         integer :: k, kmax
         include 'formats'
         R00 = exp(s% lnR_start(1))
         max_diff = 0; max_R00 = 0; max_Rm1 = 0
         do k=2, s% nz
            Rm1 = R00
            R00 = exp(s% lnR_start(k))
            dR_pre = Rm1 - R00
            dR = s% r(k-1) - s% r(k)
            diff = max(0d0,dR_pre - dR) / max(1d-99,dR_pre) ! only consider reductions in thickness
            if (diff > max_diff) then
               max_Rm1 = Rm1; max_R00 = R00; max_diff = diff; kmax = k
            end if
         end do
         
         if (s% report_all_dt_limits .and. max_diff > s% d_deltaR_shrink_limit &
               .and. s% d_deltaR_shrink_limit > 0) then
            k = kmax
            dR = s% r(k-1) - s% r(k)
            dR_pre = max_Rm1 - max_R00
            write(*,2) 'max_diff', k, max_diff
            write(*,2) 'dR', k, dR
            write(*,2) 'dR_pre', k, dR_pre
            write(*,2) 'dR - dR_pre', k, dR - dR_pre
            write(*,2) 's% d_deltaR_shrink_limit', k, s% d_deltaR_shrink_limit
            write(*,*)
         end if
         
         check_d_deltaR_shrink = check_change(s, max_diff, &
            s% d_deltaR_shrink_limit, s% d_deltaR_shrink_hard_limit, &
            kmax, 'check_d_deltaR_shrink', skip_hard_limit, dt_limit, relative_excess)
      end function check_d_deltaR_shrink
      
      
      integer function check_d_deltaR_grow(s, skip_hard_limit, dt_limit)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: Rm1, R00, dR_pre, dR, diff, relative_excess, max_diff, &
            max_Rm1, max_R00
         integer :: k, kmax
         include 'formats'
         R00 = exp(s% lnR_start(1))
         max_diff = 0; max_R00 = 0; max_Rm1 = 0
         do k=2, s% nz
            Rm1 = R00
            R00 = exp(s% lnR_start(k))
            dR_pre = Rm1 - R00
            dR = s% r(k-1) - s% r(k)
            diff = max(0d0,dR - dR_pre) / max(1d-99,dR_pre) ! only consider increases in thickness
            if (diff > max_diff) then
               max_Rm1 = Rm1; max_R00 = R00; max_diff = diff; kmax = k
            end if
         end do
         
         if (s% report_all_dt_limits .and. max_diff > s% d_deltaR_grow_limit &
               .and. s% d_deltaR_grow_limit > 0) then
            k = kmax
            dR = s% r(k-1) - s% r(k)
            dR_pre = max_Rm1 - max_R00
            write(*,2) 'max_diff', k, max_diff
            write(*,2) 'dR', k, dR
            write(*,2) 'dR_pre', k, dR_pre
            write(*,2) 'dR - dR_pre', k, dR - dR_pre
            write(*,2) 's% d_deltaR_grow_limit', k, s% d_deltaR_grow_limit
            write(*,*)
         end if
         
         check_d_deltaR_grow = check_change(s, max_diff, &
            s% d_deltaR_grow_limit, s% d_deltaR_grow_hard_limit, &
            kmax, 'check_d_deltaR_grow', skip_hard_limit, dt_limit, relative_excess)
      end function check_d_deltaR_grow

      
      integer function check_lgL_H_change(s, skip_hard_limit, dt, dt_limit)
         use chem_def, only: ih1
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         check_lgL_H_change = check_lgL( &
            s, ih1, 'check_lgL_H_change', skip_hard_limit, dt, dt_limit)
      end function check_lgL_H_change
      
      
      integer function check_lgL_He_change(s, skip_hard_limit, dt, dt_limit)
         use chem_def, only: ihe4
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         check_lgL_He_change = check_lgL( &
            s, ihe4, 'check_lgL_H_change', skip_hard_limit, dt, dt_limit)
      end function check_lgL_He_change

            
      integer function check_lgL_z_change(s, skip_hard_limit, dt, dt_limit)
         use chem_def, only: isi28
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         check_lgL_z_change = check_lgL( &
            s, isi28, 'check_lgL_H_change', skip_hard_limit, dt, dt_limit)
      end function check_lgL_z_change

            
      integer function check_lgL_photo_change(s, skip_hard_limit, dt, dt_limit)
         use chem_def, only: iprot
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         check_lgL_photo_change = check_lgL( &
            s, iprot, 'check_lgL_photo_change', skip_hard_limit, dt, dt_limit)
      end function check_lgL_photo_change

      
      integer function check_lgL_nuc_change(s, skip_hard_limit, dt, dt_limit)
         use chem_def, only: ineut
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         check_lgL_nuc_change = check_lgL( &
            s, ineut, 'check_lgL_nuc_change', skip_hard_limit, dt, dt_limit)
      end function check_lgL_nuc_change
      
      
      integer function check_lgL( &
            s, iso, msg, skip_hard_limit, dt, dt_limit)
         use num_lib, only: safe_log10
         use chem_def
         type (star_info), pointer :: s         
         integer, intent(in) :: iso
         character (len=*), intent(in) :: msg
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit

         real(dp) :: &
            sum_LH, sum_LHe, sum_Lz, &
            new_L, max_other_L, old_L, lim, hard_lim, lgL_min, &
            drop_factor, relative_limit, lgL, max_other_lgL, lgL_old, &
            abs_change, relative_excess
         logical, parameter :: dbg = .false.
         include 'formats'
         check_lgL = keep_going

         sum_LH = s% L_by_category(ipp) + s% L_by_category(icno)
         sum_LHe = s% L_by_category(i3alf)
         sum_Lz = sum(s% L_by_category) - &
            (s% L_by_category(ipp) + s% L_by_category(icno) &
               + s% L_by_category(i3alf) + s% L_by_category(iphoto))

         if (iso == ih1) then
            new_L = sum_LH
            max_other_L = max(sum_LHe, sum_Lz)
            old_L = s% L_by_category_old(ipp) + s% L_by_category_old(icno)
            lim = s% delta_lgL_H_limit
            hard_lim = s% delta_lgL_H_hard_limit
            lgL_min = s% lgL_H_burn_min
            drop_factor = s% lgL_H_drop_factor
            relative_limit = s% lgL_H_burn_relative_limit
         else if (iso == ihe4) then
            new_L = sum_LHe
            max_other_L = max(sum_LH, sum_Lz)
            old_L = s% L_by_category_old(i3alf)
            lim = s% delta_lgL_He_limit
            hard_lim = s% delta_lgL_He_hard_limit
            lgL_min = s% lgL_He_burn_min
            drop_factor = s% lgL_He_drop_factor
            relative_limit = s% lgL_He_burn_relative_limit
         else if (iso == isi28) then
            new_L = sum_Lz
            max_other_L = max(sum_LH, sum_LHe)
            old_L = sum(s% L_by_category_old) - &
               (s% L_by_category_old(ipp) + s% L_by_category_old(icno) &
                  + s% L_by_category_old(i3alf) + s% L_by_category_old(iphoto))
            lim = s% delta_lgL_z_limit
            hard_lim = s% delta_lgL_z_hard_limit
            lgL_min = s% lgL_z_burn_min
            drop_factor = s% lgL_z_drop_factor
            relative_limit = s% lgL_z_burn_relative_limit
         else if (iso == iprot) then ! check_lgL_photo_change
            new_L = abs(s% L_by_category(iphoto))
            max_other_L = 0d0
            old_L = abs(s% L_by_category_old(iphoto))
            lim = s% delta_lgL_photo_limit
            hard_lim = s% delta_lgL_photo_hard_limit
            lgL_min = s% lgL_photo_burn_min
            drop_factor = s% lgL_photo_drop_factor 
            relative_limit = 0d0
         else if (iso == ineut) then ! check_lgL_nuc_change
            new_L = sum(s% L_by_category) - s% L_by_category(iphoto)
            max_other_L = 0d0
            old_L = sum(s% L_by_category_old) - s% L_by_category_old(iphoto)
            lim = s% delta_lgL_nuc_limit
            hard_lim = s% delta_lgL_nuc_hard_limit
            lgL_min = s% lgL_nuc_burn_min
            drop_factor = s% lgL_nuc_drop_factor 
            relative_limit = 0d0
         else
            stop 'bad iso arg for check_lgL'
         end if
         
         if (new_L < old_L) then
            lim = lim*drop_factor
            hard_lim = hard_lim*drop_factor
         end if

         if (dbg) write(*,*)
         if (dbg) write(*,1) trim(msg) // ' new_L', new_L
         if (dbg) write(*,1) 'old_L', old_L
         if (new_L <= 0 .or. old_L <= 0) return
         
         lgL = safe_log10(new_L)
         if (dbg) write(*,1) 'lgL', lgL
         if (lgL < lgL_min) return
         
         if (max_other_L > 0) then
            max_other_lgL = safe_log10(max_other_L)
            if (dbg) write(*,1) 'max_other_lgL', max_other_lgL
            if (max_other_lgL - relative_limit > lgL) return
         end if
         
         lgL_old = safe_log10(old_L)
         if (dbg) write(*,1) 'lgL_old', lgL_old
         abs_change = abs(lgL - lgL_old)
         if (dbg) write(*,1) 'abs_change', abs_change
         if (dbg) write(*,1) 'hard_lim', hard_lim
         if (hard_lim > 0 .and. abs_change > hard_lim .and. (.not. skip_hard_limit)) then
            if (s% report_all_dt_limits) write(*, '(a30, f20.10, 99e20.10)') trim(msg), &
               log10(dt_limit/secyer), lgL - lgL_old, hard_lim, lgL, lgL_old
            check_lgL = retry
            return
         end if
         
         if (dbg) write(*,1) 'lim', lim
         if (lim <= 0) return
         
         relative_excess = (abs_change - lim) / lim
         if (dbg) write(*,1) 'relative_excess', relative_excess
         if (relative_excess > 0) then
            dt_limit = dt * s% timestep_dt_factor**relative_excess
            if (s% report_all_dt_limits) write(*, '(a30, f20.10, 99e20.10)') trim(msg), &
               log10(dt_limit/secyer), lgL - lgL_old, lim, lgL, lgL_old, relative_excess
         end if
         
      end function check_lgL
      
      
      integer function check_lgL_nuc_cat_change( &
            s, n_mix_bdy, mix_bdy_q, skip_hard_limit, dt_limit)
         use rates_def
         use chem_def
         use num_lib, only: binary_search
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         integer, intent(in) :: n_mix_bdy
         real(dp), intent(in), pointer :: mix_bdy_q(:)
         real(dp), intent(inout) :: dt_limit
         
         integer :: k, max_j, max_k, bdy
         real(dp) :: max_lgL_diff, relative_excess, max_diff, cat_burn_min
         
         include 'formats'

         check_lgL_nuc_cat_change = keep_going
         
         if (s% delta_lgL_nuc_cat_limit <= 0 .and. s% delta_lgL_nuc_cat_hard_limit <= 0) return

         cat_burn_min = 10**s% lgL_nuc_cat_burn_min
         max_diff = 0
         max_j = -1
         max_k = -1
         bdy = -1
         
         do k = 1, s% nz
            
            ! find the nearest mixing boundary
            bdy = binary_search(n_mix_bdy, mix_bdy_q, bdy, s% q(k))
            ! don't check cells near a mixing boundary
            if (bdy > 0) then
               if (abs(s% q(k) - mix_bdy_q(bdy)) < s% lgL_nuc_mix_dist_limit) cycle
            end if

            if (s% check_delta_lgL_pp) call do1_category(ipp,k)
            if (s% check_delta_lgL_cno) call do1_category(icno,k)
            if (s% check_delta_lgL_3alf) call do1_category(i3alf,k)
            
            if (s% check_delta_lgL_burn_c) call do1_category(i_burn_c,k)
            if (s% check_delta_lgL_burn_n) call do1_category(i_burn_n,k)
            if (s% check_delta_lgL_burn_o) call do1_category(i_burn_o,k)
            if (s% check_delta_lgL_burn_ne) call do1_category(i_burn_ne,k)
            if (s% check_delta_lgL_burn_na) call do1_category(i_burn_na,k)
            if (s% check_delta_lgL_burn_mg) call do1_category(i_burn_mg,k)
            if (s% check_delta_lgL_burn_si) call do1_category(i_burn_si,k)
            if (s% check_delta_lgL_burn_s) call do1_category(i_burn_s,k)
            if (s% check_delta_lgL_burn_ar) call do1_category(i_burn_ar,k)
            if (s% check_delta_lgL_burn_ca) call do1_category(i_burn_ca,k)
            if (s% check_delta_lgL_burn_ti) call do1_category(i_burn_ti,k)
            if (s% check_delta_lgL_burn_cr) call do1_category(i_burn_cr,k)
            if (s% check_delta_lgL_burn_fe) call do1_category(i_burn_fe,k)
            
            if (s% check_delta_lgL_cc) call do1_category(icc,k)
            if (s% check_delta_lgL_co) call do1_category(ico,k)
            if (s% check_delta_lgL_oo) call do1_category(ioo,k)
            
         end do
         

         if (max_diff <= 0) return
         
         max_lgL_diff = log10(max_diff/Lsun)
         s% Tlim_dlgL_nuc_category = max_j
         s% Tlim_dlgL_nuc_cell = max_k
         
         check_lgL_nuc_cat_change = check_change(s, max_lgL_diff, &
            s% delta_lgL_nuc_cat_limit, s% delta_lgL_nuc_cat_hard_limit, &
            max_j, 'check_lgL_nuc_cat_change', skip_hard_limit, dt_limit, relative_excess)
         
         contains         
         
         subroutine do1_category(j, k)
            integer, intent(in) :: j, k
            real(dp) :: diff, abs_diff
            if (s% luminosity_by_category(j,k) < cat_burn_min) return
            if (s% luminosity_by_category_start(j,k) < cat_burn_min) return
            diff = s% luminosity_by_category(j,k) - s% luminosity_by_category_start(j,k)
            abs_diff = abs(diff)
            if (abs_diff <= max_diff) return
            max_diff = abs_diff
            max_j = j
            max_k = k
         end subroutine do1_category
         
      end function check_lgL_nuc_cat_change

         
      integer function check_delta_ang_momentum(s, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: delta_lg_J, relative_excess
         check_delta_ang_momentum = keep_going
         if (s% total_angular_momentum_old <= 0 .or. s% total_angular_momentum <= 0) return
         delta_lg_J = log10(s% total_angular_momentum/s% total_angular_momentum_old)
         check_delta_ang_momentum = check_change(s, delta_lg_J, &
            s% delta_lg_total_J_limit, s% delta_lg_total_J_hard_limit, &
            1, 'check_delta_ang_momentum', skip_hard_limit, dt_limit, relative_excess)
      end function check_delta_ang_momentum
      
      
      integer function check_dlgTeff_change(s, skip_hard_limit, dt, dt_limit)
         use num_lib, only: safe_log10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess
         include 'formats'
         if (s% Teff_old <= 0 .or. s% Teff <= 0)  then
            check_dlgTeff_change = keep_going
            return
         end if
         check_dlgTeff_change = check_change(s, safe_log10(s% Teff/s% Teff_old), &
            s% delta_lgTeff_limit, s% delta_lgTeff_hard_limit, &
            1, 'check_dlgTeff_change', skip_hard_limit, dt_limit, relative_excess)
            
         return
         write(*,1) 's% Teff', s% Teff
         write(*,1) 's% Teff_old', s% Teff_old
         write(*,1) 's% Teff/s% Teff_old', s% Teff/s% Teff_old
         write(*,1) 'safe_log10(s% Teff/s% Teff_old)', safe_log10(s% Teff/s% Teff_old)
         write(*,1) 's% delta_lgTeff_limit', s% delta_lgTeff_limit
         write(*,1) 's% delta_lgTeff_hard_limit', s% delta_lgTeff_hard_limit
         write(*,*)
         
      end function check_dlgTeff_change


      integer function check_dlgRho_cntr_change(s, skip_hard_limit, dt, dt_limit)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, dlgRho_cntr
         integer :: nz
         nz = s% nz
         if (s% lnPgas_flag) then
            dlgRho_cntr = (s% lnd(nz) - s% lnd_start(nz))/ln10
         else
            dlgRho_cntr = dt*s% dlnd_dt(nz)/ln10
         end if
         check_dlgRho_cntr_change = check_change(s, dlgRho_cntr, &
            s% delta_lgRho_cntr_limit, s% delta_lgRho_cntr_hard_limit, &
            nz, 'check_dlgRho_cntr_change', skip_hard_limit, dt_limit, relative_excess)
      end function check_dlgRho_cntr_change
      
      
      integer function check_dYe_change(s, skip_hard_limit, dt_limit) ! check max change in Ye
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, max_diff
         integer :: i, nz
         nz = s% nz
         include 'formats'
         i = maxloc(abs(s% ye(1:nz) - s% ye_start(1:nz)), dim=1)
         max_diff = s% ye(i) - s% ye_start(i)
         
         !write(*,2) 'dYe', i, max_diff, s% ye_start(i), s% ye(i), &
         !   s% delta_Ye_limit, s% delta_Ye_hard_limit
         !write(*,*) 'skip_hard_limit', skip_hard_limit
         !write(*,*)
         
         check_dYe_change = check_change(s, max_diff, &
            s% delta_Ye_limit, s% delta_Ye_hard_limit, &
            i, 'check_dYe_change', .false., dt_limit, relative_excess)
         !   i, 'check_dYe_change', skip_hard_limit, dt_limit, relative_excess)
      end function check_dYe_change
      
      
      integer function check_dYe_highT_change(s, skip_hard_limit, dt_limit) ! check max change in Ye
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, max_diff, Tlimit, ye_diff
         integer :: i, k
         include 'formats'
         i = 0
         max_diff = 0
         Tlimit = s% minT_for_highT_Ye_limit
         do k=1, s% nz
            if (s% T(k) < Tlimit) cycle
            ye_diff = abs(s% ye(k) - s% ye_start(k))
            if (ye_diff <= max_diff) cycle
            max_diff = ye_diff
            i = k
         end do
         !if (max_diff > 0d0) write(*,2) 'check_dYe_highT_change max_diff', i, max_diff
         check_dYe_highT_change = check_change(s, max_diff, &
            s% delta_Ye_highT_limit, s% delta_Ye_highT_hard_limit, &
            i, 'check_dYe_highT_change', .false., dt_limit, relative_excess)
      end function check_dYe_highT_change


      integer function check_dlgT_cntr_change(s, skip_hard_limit, dt, dt_limit)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess
         check_dlgT_cntr_change = check_change(s, dt*s% dlnT_dt(s% nz)/ln10, &
            s% delta_lgT_cntr_limit, s% delta_lgT_cntr_hard_limit, &
            s% nz, 'check_dlgT_cntr_change', skip_hard_limit, dt_limit, relative_excess)
      end function check_dlgT_cntr_change


      integer function check_dlgT_max_change(s, skip_hard_limit, dt, dt_limit)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, dlgT_max
         dlgT_max = dt*maxval(s% dlnT_dt(1:s% nz))/ln10
         check_dlgT_max_change = check_change(s, dlgT_max, &
            s% delta_lgT_max_limit, s% delta_lgT_max_hard_limit, &
            s% nz, 'check_dlgT_max_change', skip_hard_limit, dt_limit, relative_excess)
      end function check_dlgT_max_change


      integer function check_dlgRho_max_change(s, skip_hard_limit, dt, dt_limit)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, dlgRho_max
         integer :: nz
         nz = s% nz
         if (s% lnPgas_flag) then
            dlgRho_max = maxval(s% lnd(1:nz) - s% lnd_start(1:nz))/ln10
         else
            dlgRho_max = dt*maxval(s% dlnd_dt(1:nz))/ln10
         end if
         check_dlgRho_max_change = check_change(s, dlgRho_max, &
            s% delta_lgRho_max_limit, s% delta_lgRho_max_hard_limit, &
            s% nz, 'check_dlgRho_max_change', skip_hard_limit, dt_limit, relative_excess)
      end function check_dlgRho_max_change


      integer function check_dlog_eps_nuc_cntr_change(s, skip_hard_limit, dt, dt_limit)
         use mlt_def, only: convective_mixing
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, delta
         delta = 0
         if (s% mixing_type(s% nz) /= convective_mixing .and. &
             s% center_eps_nuc > s% center_eps_nuc_old .and. &
             s% center_eps_nuc_old > 1) then
            delta = log10(s% center_eps_nuc / s% center_eps_nuc_old)
         end if
         check_dlog_eps_nuc_cntr_change = check_change(s, delta, &
            s% delta_log_eps_nuc_cntr_limit, s% delta_log_eps_nuc_cntr_hard_limit, &
            s% nz, 'check_dlog_eps_nuc_cntr_change', &
            skip_hard_limit, dt_limit, relative_excess)
      end function check_dlog_eps_nuc_cntr_change


      integer function check_dlog_eps_nuc_change(s, skip_hard_limit, dt, dt_limit)
         use mlt_def, only: convective_mixing
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, max_ratio, ratio, delta, &
            limit_ratio, delta_r
         integer :: k, j, nz, k_max
         include 'formats'
         nz = s% nz
         limit_ratio = 10d0**s% delta_log_eps_nuc_limit
         max_ratio = limit_ratio
         k_max = 0
         zoneloop: do k=1,nz
            if (s% eps_nuc_start(k) < 1) cycle zoneloop
            ratio = s% eps_nuc(k)/s% eps_nuc_start(k)
            if (s% mixing_type(k) /= convective_mixing .and. &
                s% mixing_type(min(nz,k+1)) /= convective_mixing .and. &
                ratio > max_ratio) then
               do j = 1, s% num_conv_boundaries
                  delta_r = abs(s% r(s% conv_bdy_loc(j)) - s% r(k))
                  if (delta_r <= s% scale_height(k)) then
                     cycle zoneloop ! skip ones that are too close to convection zone
                  end if
               end do
               max_ratio = ratio
               k_max = k
            end if
         end do zoneloop
         if (k_max > 0) then
            delta = log10(max_ratio)
            if (s% report_all_dt_limits) write(*,4) 'log eps nuc ratio', &
               k_max, nz, s% model_number, &
               delta/s% delta_log_eps_nuc_limit, delta, &
               s% eps_nuc(k_max), s% eps_nuc_start(k_max)
         else
            delta = 0
         end if
         check_dlog_eps_nuc_change = check_change(s, delta, &
            s% delta_log_eps_nuc_limit, s% delta_log_eps_nuc_hard_limit, &
            nz, 'check_dlog_eps_nuc_change', &
            skip_hard_limit, dt_limit, relative_excess)
      end function check_dlog_eps_nuc_change

      
      integer function check_lg_XH_cntr(s, skip_hard_limit, dt_limit)
         use chem_def, only: ih1
         use num_lib, only: safe_log10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, lg_XH_cntr, lg_XH_cntr_old
         integer :: h1, nz
         include 'formats'
         h1 = s% net_iso(ih1)
         nz = s% nz
         check_lg_XH_cntr = keep_going
         if (s% xa(h1,nz) < 1d-10) return
         lg_XH_cntr = log10(s% xa(h1,nz))
         if (lg_XH_cntr > s% delta_lg_XH_cntr_max) return
         if (lg_XH_cntr < s% delta_lg_XH_cntr_min) return
         lg_XH_cntr_old = safe_log10(s% xa_old(h1,nz))
         check_lg_XH_cntr = check_change(s, lg_XH_cntr - lg_XH_cntr_old, &
            s% delta_lg_XH_cntr_limit, s% delta_lg_XH_cntr_hard_limit, &
            1, 'check_lg_XH_cntr', skip_hard_limit, dt_limit, relative_excess)
      end function check_lg_XH_cntr

      
      integer function check_lg_XHe_cntr(s, skip_hard_limit, dt_limit) 
         use chem_def, only: ihe4
         use num_lib, only: safe_log10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, lg_XHe_cntr, lg_XHe_cntr_old
         integer :: he4, nz
         include 'formats'
         he4 = s% net_iso(ihe4)
         nz = s% nz
         check_lg_XHe_cntr = keep_going
         if (s% xa(he4,nz) < 1d-10) return
         lg_XHe_cntr = log10(s% xa(he4,nz))
         if (lg_XHe_cntr > s% delta_lg_XHe_cntr_max) return
         if (lg_XHe_cntr < s% delta_lg_XHe_cntr_min) return
         lg_XHe_cntr_old = safe_log10(s% xa_old(he4,nz))
         check_lg_XHe_cntr = check_change(s, lg_XHe_cntr - lg_XHe_cntr_old, &
            s% delta_lg_XHe_cntr_limit, s% delta_lg_XHe_cntr_hard_limit, &
            1, 'check_lg_XHe_cntr', skip_hard_limit, dt_limit, relative_excess)
      end function check_lg_XHe_cntr
         
            
      integer function check_v_div_v_crit(s, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: v_div_v_crit
         check_v_div_v_crit = keep_going
         v_div_v_crit = s% v_div_v_crit_avg_surf
         if ((.not. skip_hard_limit) .and. s% v_div_v_crit_hard_limit > 0 .and. &
               v_div_v_crit > s% v_div_v_crit_hard_limit) then
            check_v_div_v_crit = retry
            if (s% report_all_dt_limits) &
               write(*, '(a30, f20.10, 99e20.10)') 'v_div_v_crit hard limit', &
                  log10(dt_limit/secyer), v_div_v_crit, s% v_div_v_crit_hard_limit
            return
         end if
         if (s% v_div_v_crit_limit <= 0) return
         if (v_div_v_crit <= s% v_div_v_crit_limit) return
         dt_limit = dt*s% v_div_v_crit_limit/v_div_v_crit
         if (s% report_all_dt_limits) write(*, '(a30, f20.10, 99e20.10)') &
            'v_div_v_crit: log dt, v_div_v_crit, limit', log10(dt_limit/secyer), &
            v_div_v_crit, s% v_div_v_crit_limit
      end function check_v_div_v_crit


      integer function check_delta_mdot(s, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: mdot, mdot_old, delta_mdot
         check_delta_mdot = keep_going
         mdot = s% mstar_dot
         mdot_old = s% mstar_dot_old
         delta_mdot = abs(mdot - mdot_old)/ &
            (s% delta_mdot_atol*Msun/secyer + &
               s% delta_mdot_rtol*max(abs(mdot),abs(mdot_old)))
         if (s% delta_mdot_hard_limit > 0 .and. (.not. skip_hard_limit)) then
            if (delta_mdot > s% delta_mdot_hard_limit) then
               if (s% report_all_dt_limits) &
                  write(*, '(a30, f20.10, 99e20.10)') 'delta_mdot_hard_limit', &
                     delta_mdot, s% delta_mdot_hard_limit
                  check_delta_mdot= retry
               return
            end if
         end if
         if (delta_mdot > s% delta_mdot_limit .and. s% delta_mdot_limit > 0) then
            dt_limit = dt*s% delta_mdot_limit/delta_mdot
            if (s% report_all_dt_limits) &
               write(*, '(a30, f20.10, 99e20.10)') 'delta_mdot_limit', &
                  delta_mdot, s% delta_mdot_limit
         end if
      end function check_delta_mdot

         
      integer function check_delta_mstar(s, skip_hard_limit, dt, dt_limit) ! check magnitude of mdot
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: mstar, delta_lg_star_mass
         mstar = s% mstar
         delta_lg_star_mass = abs(s% mstar_dot)*dt / mstar
         if (s% delta_lg_star_mass_hard_limit > 0 .and. (.not. skip_hard_limit)) then
            if (delta_lg_star_mass > s% delta_lg_star_mass_hard_limit) then
               if (s% report_all_dt_limits) &
                  write(*, '(a30, f20.10, 99e20.10)') 'delta_lg_star_mass_hard_limit', &
                     delta_lg_star_mass, s% delta_lg_star_mass_hard_limit
                  check_delta_mstar= retry
               return
            end if
         end if
         check_delta_mstar = keep_going
         if (delta_lg_star_mass > s% delta_lg_star_mass_limit) then
            dt_limit = s% delta_lg_star_mass_limit*mstar/max(1d0,abs(s% mstar_dot))
            if (s% report_all_dt_limits) &
               write(*, '(a30, f20.10, 99e20.10)') 'delta_lg_star_mass_limit', &
                  delta_lg_star_mass, s% delta_lg_star_mass_limit
         end if
      end function check_delta_mstar

         
      integer function check_mdot_eps_grav( &
            s, skip_hard_limit, dt, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         
         real(dp) :: ratio
         integer :: k
         include 'formats'
         
         check_mdot_eps_grav = keep_going
         
         k = s% k_below_mdot_eps_grav
         if (k <= 1 .or. s% L(1) <= 0) return
         ratio = s% Cp(k)*s% T(k)*abs(s% mstar_dot)/s% L(1)
         !write(*,1) 'ratio for mdot_eps_grav', ratio

         if (s% mdot_eps_grav_hard_limit > 0 .and. &
               (.not. skip_hard_limit) .and. &
               ratio > s% mdot_eps_grav_hard_limit) then
            if (s% report_all_dt_limits) &
               write(*, '(a30, 99e20.10)') 'mdot_eps_grav_hard_limit', &
                  ratio, s% mdot_eps_grav_hard_limit
               check_mdot_eps_grav = retry
            return
         end if
         
         if (s% mdot_eps_grav_limit <= 0) return
         if (ratio <= s% mdot_eps_grav_limit) return
                  
         dt_limit = dt*s% mdot_eps_grav_limit/ratio
         if (s% report_all_dt_limits) &
            write(*, '(a30, 99e20.10)') 'mdot_eps_grav_limit', &
               ratio, s% mdot_eps_grav_limit
         
      end function check_mdot_eps_grav

         
      integer function check_delta_lgL(s, skip_hard_limit, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: dlgL, relative_excess
         if (s% L_phot < s% delta_lgL_limit_L_min) then
            dlgL = 0
         else
            dlgL = log10(s% L_phot/s% L_phot_old)
         end if
         check_delta_lgL = check_change(s, dlgL, &
            s% delta_lgL_limit, s% delta_lgL_hard_limit, &
            1, 'check_delta_lgL', skip_hard_limit, dt_limit, relative_excess)
      end function check_delta_lgL

         
      integer function check_op_split_diff(s, skip_hard_limit, dt_limit)
         use chem_def, only: iphoto
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: relative_excess, sum_Lnuc
         check_op_split_diff = keep_going
         if (s% center_h1 >= s% delta_op_split_center_h1_limit) return
         sum_Lnuc = sum(s% L_by_category) - s% L_by_category(iphoto)
         if (sum_Lnuc <= s% delta_op_split_Lnuc_limit) return
         check_op_split_diff = check_change(s, s% op_split_diff, &
            s% delta_op_split_limit, s% delta_op_split_hard_limit, &
            1, 'check_op_split_diff', skip_hard_limit, dt_limit, relative_excess)
      end function check_op_split_diff

         
      integer function check_delta_HR(s, skip_hard_limit, dt_limit)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit
         real(dp) :: dlgL, dlgTeff, dHR, relative_excess
         check_delta_HR = keep_going
         if (s% L_phot_old <= 0 .or. s% Teff_old <= 0) return
         dlgL = log10(s% L_phot/s% L_phot_old)
         dlgTeff = log10(s% Teff/s% Teff_old)
         dHR = sqrt((s% delta_HR_ds_L*dlgL)**2 + (s% delta_HR_ds_Teff*dlgTeff)**2)
         check_delta_HR = check_change(s, dHR, &
            s% delta_HR_limit, s% delta_HR_hard_limit, &
            1, 'check_delta_HR', skip_hard_limit, dt_limit, relative_excess)
      end function check_delta_HR
            
      
      integer function check_dX_nuc(s, skip_hard_limit, dt, dt_limit)
         use chem_def
         use rates_def, only: i_rate
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         real(dp) :: dX_timescale, max_dXdt, dX, max_dX, &
            del, delnuc, dX_nuc_min_dt_limit, D_mix_cutoff
         integer :: max_k, max_j, max_category, dxdt_loc(2), i, j, k, nz
         
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         check_dX_nuc = keep_going
         nz = s% nz
         
         dxdt_loc(1:2) = maxloc(s% eps_nuc_categories(i_rate,:,1:nz))
         max_category = dxdt_loc(1) 
         max_k = dxdt_loc(2)            
         if (max_category == ipp .or. max_category == icno .or. max_category == i3alf) return
         if (max_category == i_burn_c .and. s% xa(s% net_iso(ihe4), max_k) > 1d-4) return
         
         max_dX = -1
         
         if (s% set_min_D_mix .and. s% ye(nz) >= s% min_center_Ye_for_min_D_mix) then
            D_mix_cutoff = s% min_D_mix
         else
            D_mix_cutoff = 0
         end if
         
         zoneloop: do k = 1, nz
            if (dbg .and. k == 1387) write(*,2) 'check max_dX', k
            if (s% D_mix(k) > D_mix_cutoff) then ! check if near jump in D_mix, and skip if is.
               do i = max(1, k-3), min(s% nz, k+3)
                  if (abs(s% D_mix(i) - s% D_mix(k)) > 1d-2*max(s% D_mix(i), s% D_mix(k))) then
                     if (dbg .and. k == 1387) write(*,2) 'too close', i, s% D_mix(i), s% D_mix(k)
                     cycle zoneloop
                  end if
               end do
            end if
            do j=1,s% species
               if (s% xa(j,k) < s% dprctd_dX_nuc_min_X_limit) then
                  if (dbg .and. k == 1387) &
                     write(*,2) 'dX_nuc_min_X_limit ' // trim(chem_isos% name(s% chem_id(j))), &
                        k, s% xa(j,k), s% dprctd_dX_nuc_min_X_limit
                  cycle
               end if
               if (chem_isos% W(s% chem_id(j)) > s% dprctd_dX_nuc_max_A_limit) then
                  if (dbg .and. k == 1387) &
                     write(*,2) 'dX_nuc_max_A_limit ' // trim(chem_isos% name(s% chem_id(j))), &
                        k, s% xa(j,k), s% dprctd_dX_nuc_max_A_limit
                  cycle
               end if
               if (chem_isos% Z(s% chem_id(j)) <= 2) then
                  cycle ! skip the little guys
               end if
               del = s% xa_old(j,k) - s% xa(j,k)
               if (del < 0) then
                  if (dbg .and. k == 1387) &
                     write(*,2) 'increased ' // trim(chem_isos% name(s% chem_id(j))), &
                        k, s% xa(j,k)
                  cycle ! skip if increased in abundance
               end if
               if (s% xa(j,k) > s% dprctd_dX_nuc_X_limit .and. &
                   s% xa(j,k) > s% dprctd_dX_nuc_dX_div_del_limit*del) then
                  if (dbg .and. k == 1387) &
                     write(*,2) 'too much? ' // trim(chem_isos% name(s% chem_id(j))), &
                        k, s% xa(j,k)
                  cycle
               end if
               delnuc = -s% dxdt_nuc(j,k)*s% dt
               if (del < delnuc .and. s% dprctd_dX_nuc_factor > 1) then
                  dX = max(delnuc/s% dprctd_dX_nuc_factor, del)
               else
                  dX = delnuc
               end if
               if (dbg .and. k == 1387) &
                  write(*,3) 'dX ' // trim(chem_isos% name(s% chem_id(max_j))), k, max_k, max_dX
               if (dX > max_dX) then
                  max_dX = dX; max_j = j; max_k = k
                  !write(*,3) 'max_dX ' // trim(chem_isos% name(s% chem_id(max_j))), k, max_k, max_dX
               end if
            end do
         end do zoneloop
         if (max_dX <= 0) return
         
         max_dXdt = max_dX/s% dt

         if (chem_isos% Z(s% chem_id(max_j)) <= 2) return ! doesn't apply to H or He

         s% Tlim_dXnuc_cell = max_k
         s% Tlim_dXnuc_species = max_j
         
         dX_timescale = 1/max(1d-99,max_dXdt)
         dX_nuc_min_dt_limit = secyer*s% dprctd_dX_nuc_min_yrs_for_dt

         if (s% dprctd_dX_nuc_hard_limit > 0 .and. (.not. skip_hard_limit) .and. &
               max(s% dprctd_dX_nuc_hard_limit*dX_timescale,dX_nuc_min_dt_limit) < dt) then
            if (s% report_all_dt_limits) then
               write(*, '(a30, i6, f20.10, 99e20.10)') &
                  'dX_nuc_hard_limit ' // trim(chem_isos% name(s% chem_id(max_j))), &
                  max_k, log10(dX_timescale/secyer), dX_timescale, s% dprctd_dX_nuc_hard_limit, &
                  max_dX, max_dXdt, s% eps_nuc_categories(i_rate,max_category,max_k)
            end if
            check_dX_nuc= retry
            return
         end if
         if (s% dprctd_dX_nuc_limit <= 0) return
         
         dt_limit = max(s% dprctd_dX_nuc_limit*dX_timescale, dX_nuc_min_dt_limit)
            
         if (s% report_all_dt_limits) then
            !write(*,*) 'check_dX_nuc'; write(*,*)
            !call report_dels; write(*,*)
            write(*,2) 'dX_nuc dt limit'
            write(*,2) 'max_k', max_k
            write(*,2) 'max_j ' // trim(chem_isos% name(s% chem_id(max_j))), max_j
            write(*,2) 'max_category ' // trim(category_Name(max_category)), max_category
            write(*,1) 'max delnuc', -s% dxdt_nuc(max_j, max_k)*s% dt/max(1d0,s% dprctd_dX_nuc_factor)
            write(*,1) 'max del', s% xa_old(max_j,max_k) - s% xa(max_j,max_k)
            write(*,1) 'max_dX', max_dX
            write(*,1) 'max_dXdt', max_dXdt
            write(*,1) 'dX_timescale', dX_timescale
            write(*,1) 's% dX_nuc_limit*dX_timescale', s% dprctd_dX_nuc_limit*dX_timescale
            write(*,1) 'dX_nuc_min_dt_limit', dX_nuc_min_dt_limit
            write(*,1) 'dt_limit', dt_limit
            write(*,1) 'log10(dt_limit/secyer)', log10(dt_limit/secyer)
            write(*,*)
            write(*,1) 'dX_nuc_limit/max_dX', s% dprctd_dX_nuc_limit/max_dX
            write(*,1) 'dt_limit/dt', dt_limit/s% dt
            write(*,*)
            !if (dt_limit < s% dt) stop
         end if
         
         if (dt_limit < s% dt) then
            call write_max(6)
            if (s% extra_terminal_iounit > 0) call write_max(s% extra_terminal_iounit)
         end if

         contains
         
         subroutine write_max(io)
            integer, intent(in) :: io
            write(io,'(a,a10,i6,99(1x,a,1x,1pe16.8))') &
               'dX_nuc max decrease: ', trim(chem_isos% name(s% chem_id(max_j))), &
               max_k, &
               'max_dX', max_dX, &
               'limit', s% dprctd_dX_nuc_limit, &
               'delnuc', -s% dxdt_nuc(max_j, max_k)*s% dt/max(1d0,s% dprctd_dX_nuc_factor), &
               'delX', s% xa_old(max_j,max_k) - s% xa(max_j,max_k), &
               'new', s% xa(max_j,max_k), &
               'm', s% star_mass*s% q(max_k)
         end subroutine write_max
         
         subroutine report_dels
            integer :: j
            include 'formats'
            do j = 1, s% species
               write(*,1) 'delnuc del ' // trim(chem_isos% name(s% chem_id(j))), -s% dxdt_nuc(j, max_k)*s% dt, &
                  s% xa_old(j,max_k) - s% xa(j,max_k), max_dX
            end do
            write(*,*)
         end subroutine report_dels
                       
      end function check_dX_nuc
            
      
      integer function check_dX_nuc_drop(s, skip_hard_limit, dt, dt_limit)
         use chem_def
         use rates_def, only: i_rate
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit
         
         integer :: k, nz, max_k, max_j
         real(dp), pointer, dimension(:) :: sig
         real(dp) :: max_dx_nuc_drop, X_limit, A_limit
         
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         check_dX_nuc_drop = keep_going
         
         X_limit = s% dX_nuc_drop_min_X_limit
         A_limit = s% dX_nuc_drop_max_A_limit
         nz = s% nz
         sig => s% sig
         
         max_dx_nuc_drop = 0
         max_k = 0
         max_j = 0
         do k = 1, nz
            if (s% T(k) > s% T_NSE_full_off) cycle ! skip NSE
            call do1(k)
         end do
         
         s% Tlim_dXnuc_drop_cell = max_k
         s% Tlim_dXnuc_drop_species = max_j

         if (s% dX_nuc_drop_hard_limit > 0 .and. (.not. skip_hard_limit) .and. &
               max_dx_nuc_drop > s% dX_nuc_drop_hard_limit) then
            if (s% report_all_dt_limits) then
               write(*, '(a30, i6, f20.10, 99e20.10)') &
                  'dX_nuc_drop_hard_limit ' // trim(chem_isos% name(s% chem_id(max_j))), &
                  max_k, max_dx_nuc_drop, s% dX_nuc_drop_hard_limit
            end if
            check_dX_nuc_drop= retry
            return
         end if
         if (s% dX_nuc_drop_limit <= 0 .or. max_dx_nuc_drop <= 0) return
         
         dt_limit = max(s% dX_nuc_drop_limit*s% dt/max_dx_nuc_drop, &
                        secyer*s% dX_nuc_drop_min_yrs_for_dt)
         
         if (max_dx_nuc_drop > s% dX_nuc_drop_limit .and. max_j /= 0) then
            write(*,2) 'dt limit: dX_nuc_drop ' // trim(chem_isos% name(s% chem_id(max_j))), &
               max_k, max_dx_nuc_drop, s% m(max_k)/Msun
         end if
         
         if (s% report_all_dt_limits) then
            !write(*,*) 'check_dX_nuc_drop'; write(*,*)
            write(*,2) 'dX_nuc_drop dt limit T', max_k, s% T(max_k)
            write(*,2) 'max_k', max_k
            if (max_j /= 0) then
               write(*,2) 'max_j ' // trim(chem_isos% name(s% chem_id(max_j))), max_j
               write(*,1) 'xa', s% xa(max_j,max_k)
               write(*,1) 'xa_prev', s% xa_old(max_j,max_k)
               write(*,1) 'xa - xa_prev', s% xa(max_j,max_k) - s% xa_old(max_j,max_k)
            end if
            write(*,1) 'max_dx_nuc_drop', max_dx_nuc_drop
            write(*,1) 's% dX_nuc_drop_min_yrs_for_dt', s% dX_nuc_drop_min_yrs_for_dt
            write(*,1) 's% dX_nuc_drop_limit', s% dX_nuc_drop_limit
            write(*,1) 'dt_limit', dt_limit
            write(*,1) 'log10(dt_limit/secyer)', log10(dt_limit/secyer)
            write(*,*)
            write(*,1) 'dt_limit/dt', dt_limit/dt
            write(*,*)
            !if (dt_limit < dt) stop
         end if

         if (dt_limit < dt .or. dt_limit == secyer*s% dX_nuc_drop_min_yrs_for_dt) then
            call write_max(6)
            if (s% extra_terminal_iounit > 0) call write_max(s% extra_terminal_iounit)
            !stop 'dX_nuc_drop'
         end if
         

         contains
         
         subroutine write_max(io)
            integer, intent(in) :: io
            if (.not. s% report_dX_nuc_drop_dt_limits) return
            write(io,'(a,i6,a20,i6,99(1x,a,1x,1pe10.3))') &
               'dt limit dX_nuc_drop:  model ', s% model_number, &
               '  ' // trim(chem_isos% name(s% chem_id(max_j))) // '  at cell', &
               max_k, '  drop', max_dx_nuc_drop, &
               '  before', s% xa(max_j,max_k)+max_dx_nuc_drop, &
               '  after', s% xa(max_j,max_k), &
               '  lg_dt_yr', log10(dt_limit/secyer), '  m', s% m(max_k)/Msun
         end subroutine write_max
         

         subroutine do1(k)
            integer, intent(in) :: k

            integer :: j, jj, ii
            real(dp) :: dx, dx_drop, dm, dt_dm, dx_burning, dx_inflow, dxdt_nuc
            real(dp) ::dx00, dxp1, sig00, sigp1, flux00, fluxp1

            include 'formats'
            
            dm = s% dm(k)  
            dt_dm = dt/dm

            do j=1, s% species

               if (chem_isos% W(s% chem_id(j)) > A_limit) then
                  if (dbg .and. k == 1387) &
                     write(*,2) 'dX_nuc_max_A_limit ' // trim(chem_isos% name(s% chem_id(j))), &
                        k, s% xa(j,k), chem_isos% W(s% chem_id(j)), A_limit
                  cycle
               end if

               if (chem_isos% Z(s% chem_id(j)) <= 2) then
                  cycle ! skip the little guys
               end if
            
               if (s% xa(j,k) < X_limit) then
                  if (dbg .and. k == 1387) &
                     write(*,2) &
                        'dX_nuc_drop_min_X_limit ' // trim(chem_isos% name(s% chem_id(j))), &
                        k, s% xa(j,k), X_limit
                  cycle
               end if
            
               dxdt_nuc = s% dxdt_nuc(j,k)
               if (dxdt_nuc >= 0) cycle
               
               dx_burning = dxdt_nuc*dt
               sig00 = sig(k)
         
               if (k < s% nz) then
                  sigp1 = sig(k+1)
               else
                  sigp1 = 0
               end if
         
               if (k > 1) then
                  dx00 = s% xa(j,k-1) - s% xa(j,k)
                  flux00 = -sig00*dx00
               else
                  flux00 = 0
               end if
         
               if (k < s% nz) then
                  dxp1 = s% xa(j,k) - s% xa(j,k+1)
                  fluxp1 = -sigp1*dxp1
               else
                  fluxp1 = 0
               end if

               dx_inflow = max(0d0, fluxp1, -flux00)*dt_dm
               dx_drop = -(dx_burning + dx_inflow) ! dx_burning < 0 for drop
               
               dx = s% xa_old(j,k) - s% xa(j,k) ! the actual drop
               if (dx < dx_drop) dx_drop = dx
               
               if (dx_drop > max_dx_nuc_drop) then
                  max_dx_nuc_drop = dx_drop
                  max_k = k
                  max_j = j
               end if

            end do
         

         end subroutine do1
         
                       
      end function check_dX_nuc_drop


      subroutine eval_varcontrol(s, ierr)
         use utils_lib, only: is_bad_num, alloc_iounit, free_iounit
         use chem_def
         use num_lib, only: safe_log10
         use star_utils, only: std_write_internals_to_file
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: j, nterms, nvar_hydro, nz, k, &
            skip1, skip2, i_lnPgas, i_xlnd, iounit
         real(dp) :: sumj, sumvar, sumscales, sumterm(s% nvar)
         real(dp), pointer :: vc_data(:,:)
         logical :: dbg
         real(dp), parameter :: xscale_min = 1
         
         include 'formats'
         
         ierr = 0
         
         dbg = (s% write_varcontrol_internals >= 0)
         nvar_hydro = s% nvar_hydro
         nz = s% nz
         
         if (dbg) then
            allocate(vc_data(nvar_hydro,nz),stat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in alloca in eval_varcontrol'
               return
            end if
         end if
         
         if (s% include_L_in_error_est) then
            skip1 = 0
         else
            skip1 = s% i_lum
         end if
         
         if (s% include_v_in_error_est) then
            skip2 = 0
         else
            skip2 = s% i_vel
         end if
         
         i_xlnd = s% i_xlnd
         i_lnPgas = s% i_lnPgas
         
         nterms = 0
         sumvar = 0
         sumscales = 0
         sumterm(:) = 0
         
         ! use differences in smoothed old and new to filter out high frequency noise.
         do j = 1, nvar_hydro
                  
            if (j == skip1 .or. &
                j == skip2) cycle
            
            nterms = nterms + nz
            do k = 3, nz-2
               sumj = abs(sum(s% xh(j,k-2:k+2)) - sum(s% xh_old(j,k-2:k+2)))/5
               sumterm(j) = sumterm(j) + sumj
            end do
            sumterm(j) = sumterm(j) + &
               abs((2*s% xh(j,1) + s% xh(j,2)) - (2*s% xh_old(j,1) + s% xh_old(j,2)))/3 + &
               abs((2*s% xh(j,nz) + s% xh(j,nz-1)) - (2*s% xh_old(j,nz) + s% xh_old(j,nz-1)))/3
            k = 2
            sumj = abs(sum(s% xh(j,k-1:k+1)) - sum(s% xh_old(j,k-1:k+1)))/3
            sumterm(j) = sumterm(j) + sumj
            k = nz-1
            sumj = abs(sum(s% xh(j,k-1:k+1)) - sum(s% xh_old(j,k-1:k+1)))/3
               
            if (j == i_xlnd) then 
               sumterm(j) = sumterm(j)/3
            end if
               
            if (j == i_lnPgas) then
               sumterm(j) = sumterm(j)/5
            end if
            
            sumvar = sumvar + sumterm(j)
            sumscales = sumscales + max(xscale_min, abs(s% xh_old(j,1)))
            
         end do

         sumterm(:) = sumterm(:)/sumscales
         sumvar = sumvar/sumscales

         if (dbg) then
            call show_info            
            if (s% write_varcontrol_internals > 0) &
               call std_write_internals_to_file(s% id, s% write_varcontrol_internals)            
            call std_write_internals_to_file(s% id, 0)
            
            deallocate(vc_data)
            stop 'debug: timestep'
         end if
         
         s% varcontrol = sumvar/nterms
         
         contains
         
         subroutine show_info
            character (len=64) :: filename
            real(dp) :: newterm
            write(*,*)
            write(*, *) 'sumvar', sumvar
            write(*, *) 'nterms', nterms
            write(*, *) 'sumvar/nterms', sumvar/nterms
            write(*,*)
            do j=1, nvar_hydro
               if (j == skip1 .or. &
                   j == skip2) cycle
               write(*, '(a40, e26.16)') &
                  'varcontrol fraction for ' // trim(s% nameofvar(j)) // ' = ', sumterm(j)/sumvar
            end do
            write(*,*)
            do j=1, s% species
               if (sumterm(nvar_hydro + j) /= 0) &
                  write(*, '(a40, e26.16)') 'varcontrol fraction for ' // &
                        trim(chem_isos% name(s% chem_id(j))) // ' = ', &
                        sumterm(nvar_hydro + j)/sumvar
            end do
            if (s% write_varcontrol_internals >= 0) then
               iounit = alloc_iounit(ierr); if (ierr /= 0) return
               filename = 'plot_data/varcontrol.data'
               write(*,*) 'write ' // trim(filename)
               open(iounit, file=trim(filename), action='write', status='replace', iostat=ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed to open file for varcontrol info'
                  call free_iounit(iounit)
                  return
               end if
               write(iounit, fmt='(a26)', advance='no') 'k'
               do j=1, nvar_hydro
               if (j == skip1 .or. &
                   j == skip2) cycle
                  write(iounit, fmt='(a26)', advance='no') trim(s% nameofvar(j))
               end do
               write(iounit,*)
               do k=1,nz
                  write(iounit, fmt='(i26)', advance='no') k
                  do j=1, nvar_hydro
                     if (j == skip1 .or. &
                         j == skip2) cycle
                     if (k <= 2) then
                        newterm = &
                           abs((2*s% xh(j,1) + s% xh(j,2)) - (2*s% xh_old(j,1) + s% xh_old(j,2)))/3
                     else if (k >= nz-1) then
                        newterm = &
                           abs((2*s% xh(j,nz) + s% xh(j,nz-1)) - (2*s% xh_old(j,nz) + s% xh_old(j,nz-1)))/3
                     else
                        newterm = abs(sum(s% xh(j,k-2:k+2)) - sum(s% xh_old(j,k-2:k+2)))/5
                     end if
                     write(iounit, fmt='(e26.16)', advance='no') newterm
                  end do
                  write(iounit,*)
               end do
               close(iounit)
               call free_iounit(iounit)
            end if
         end subroutine show_info
         
         
      end subroutine eval_varcontrol


      integer function hydro_timestep(s)
         ! H211b "low pass" controller.
         ! Soderlind Wang, J of Computational and Applied Math 185 (2006) 225 – 243.
         use num_def
         use utils_lib, only:is_bad_num
         type (star_info), pointer :: s
         
         real(dp), parameter :: order = 1d0
         real(dp), parameter :: beta1 = 0.25d0/order
         real(dp), parameter :: beta2 = 0.25d0/order
         real(dp), parameter :: alpha2 = 0.25d0
         real(dp) :: vcratio, vcratio_prev, limtr, vc_target
         integer :: ierr
         
         include 'formats'
                  
         ierr = 0
         hydro_timestep = keep_going
         
         s% hydro_dt_next = 1d99
         
         call eval_varcontrol(s, ierr)
         if (ierr /= 0) then
            hydro_timestep = retry
            if (s% report_ierr) write(*, *) 'hydro_timestep: eval_varcontrol ierr', ierr
            s% result_reason = nonzero_ierr
            return
         end if

         vc_target = s% varcontrol_target
         if (s% DUP_varcontrol_factor > 0 .and. s% TP_state == 2) &
            vc_target = vc_target * s% DUP_varcontrol_factor

         if (s% report_hydro_dt_info) then
            write(*, 1) 's% varcontrol_target', s% varcontrol_target
            write(*, 1) 's% DUP_varcontrol_factor', s% DUP_varcontrol_factor
            write(*, 2) 's% TP_state', s% TP_state
            write(*, 1) 'vc_target', vc_target
            write(*, 1) 's% varcontrol', s% varcontrol
            write(*, 1) 's% varcontrol_old', s% varcontrol_old
            write(*, 1) 's% dt_old', s% dt_old
            write(*, *)
         end if

         if (s% varcontrol_old > 0 .and. s% dt_old > 0) then ! use 2 values to do "low pass" controller
            vcratio = limiter(vc_target/s% varcontrol)
            vcratio_prev = limiter(vc_target/s% varcontrol_old)    
            limtr = limiter(vcratio**beta1 * vcratio_prev**beta2 * (s% dt/s% dt_old)**(-alpha2))
            s% hydro_dt_next = min(s% hydro_dt_next, s% dt*limtr)
            if (s% report_hydro_dt_info) then
               write(*, 1) 'limtr', limtr
            end if
         else ! no history available, so fall back to the basic controller
            s% hydro_dt_next = min(s% hydro_dt_next, s% dt*vc_target/s% varcontrol)
            if (s% report_hydro_dt_info) then
               write(*, 1) 'vc_target', vc_target
               write(*, 1) 's% varcontrol', s% varcontrol
               write(*, 1) 'vc_target/s% varcontrol', vc_target/s% varcontrol
            end if
         end if

         if (s% max_timestep_factor > 0 .and. s% hydro_dt_next > s% max_timestep_factor*s% dt) then
            s% hydro_dt_next = s% max_timestep_factor*s% dt
            if (s% report_hydro_dt_info) then
               write(*, 1) 's% max_timestep_factor', s% max_timestep_factor
               write(*, 1) 'hydro_dt_next', s% hydro_dt_next
               write(*, 1) 'lg hydro_dt_next/secyer', log10(s% hydro_dt_next/secyer)
            end if
         end if

         if (s% min_timestep_factor > 0 .and. s% hydro_dt_next < s% min_timestep_factor*s% dt) then
            s% hydro_dt_next = s% min_timestep_factor*s% dt
         end if

         if (s% report_hydro_dt_info .or. is_bad_num(s% hydro_dt_next)) then
            write(*, *) 'model_number', s% model_number
            write(*, 1) 'dt', s% dt
            write(*, 1) 'lg dt/secyer', log10(s% dt/secyer)
            write(*, 1) 'hydro_dt_next', s% hydro_dt_next
            write(*, 1) 'lg hydro_dt_next/secyer', log10(s% hydro_dt_next/secyer)
            write(*, 1) 'hydro_dt_next/dt', s% hydro_dt_next/s% dt
            write(*, *)
         end if
         
         if (s% report_hydro_dt_info) write(*, *)
         
         
         contains
         
         real(dp) function limiter(x)
            real(dp), intent(in) :: x
            real(dp), parameter :: kappa = 2
            ! for x >= 0 and kappa = 2, limiter value is between 0.07 and 4.14
            ! for x = 1, limiter = 1
            limiter = 1 + kappa*ATAN((x-1)/kappa)
         end function limiter
         
      end function hydro_timestep
      

      end module timestep


