! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module struct_burn_mix

      use star_private_def
      use const_def

      implicit none

      contains


      integer function do_struct_burn_mix(s, skip_global_corr_coeff_limit, dt) 
         use solve_hydro, only: do_hydro_converge
         use utils_lib, only:is_bad_num
         use hydro_vars, only: check_rs, set_vars
         use rates_def, only: num_rvs
         use chem_def, only: num_categories, ih1, isi28
         
         type (star_info), pointer :: s
         logical, intent(in) :: skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt
         
         integer :: ierr

         include 'formats'
         
         ierr = 0
         
         do_struct_burn_mix = retry
         s% op_split_diff = 0
         
         if (s% operator_coupling_choice == -2) then
            call do_new_solve2(s, skip_global_corr_coeff_limit, dt, ierr)
            if (ierr == 0) do_struct_burn_mix = keep_going
            return
         end if
         
         if (s% operator_coupling_choice == -1) then
            call do_new_solve1(s, skip_global_corr_coeff_limit, dt, ierr)
            if (ierr == 0) do_struct_burn_mix = keep_going
            return
         end if

         if (s% operator_coupling_choice < 0) then
            write(*,2) 'invalid value for operator_coupling_choice', &
               s% operator_coupling_choice
            stop 'struct_burn_mix'
         end if
               
         call save_pre_values(s, ierr)
         if (ierr /= 0) return
         
         s% do_struct_hydro = .true.
         s% do_struct_thermo = .true.
         s% do_burn = .true.
         s% do_mix = .true.
         
         do_struct_burn_mix = &
            do_hydro_converge( &
               s, s% newton_itermin, skip_global_corr_coeff_limit, dt)
         if (do_struct_burn_mix /= keep_going) return
         
         do_struct_burn_mix = do_smooth_conv_bdy_and_mix_omega(s,dt)

      end function do_struct_burn_mix

      
      subroutine save_pre_values(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr         
         integer :: k, j
         include 'formats'    
         ierr = 0         
         do k=1,s% nz
            do j=1,s% nvar_hydro
               s% xh_pre(j,k) = s% xh(j,k)
            end do
            do j=1,s% species
               s% xa_pre(j,k) = s% xa(j,k)
            end do
            s% lnS_pre(k) = s% lnS(k)
         end do    
      end subroutine save_pre_values


      subroutine do_new_solve1(s, skip_global_corr_coeff_limit, dt, ierr)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         ! do_burn false means do burn separately
         ! do_mix false means do mix separately
         logical, parameter :: do_burn = .false., do_mix = .false.
         call do_new_solve( &
            s, skip_global_corr_coeff_limit, do_burn, do_mix, dt, ierr)
      end subroutine do_new_solve1


      subroutine do_new_solve2(s, skip_global_corr_coeff_limit, dt, ierr)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         ! do_burn true means do burn coupled
         ! do_mix false means do mix separately
         logical, parameter :: do_burn = .true., do_mix = .false.
         
         stop 'op_split = -2 not currently supported: do_new_solve2'
         
         call do_new_solve( &
            s, skip_global_corr_coeff_limit, do_burn, do_mix, dt, ierr)
      end subroutine do_new_solve2


      subroutine do_new_solve( &
            s, skip_global_corr_coeff_limit, do_burn, do_mix, dt, ierr)
         use star_utils, only: update_time, total_times, center_value
         use solve_hydro, only: do_hydro_converge

         type (star_info), pointer :: s
         logical, intent(in) :: skip_global_corr_coeff_limit, do_burn, do_mix
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         integer :: i, j, k, nz, species, &
            do_burn_mix_before_struct, do_burn_mix_after_struct
         integer :: time0, clock_rate, res
         real(dp) :: total_all_before, err, limit_T, dt_burn_and_mix, &
            op_split_Ltot_atol, op_split_Ltot_rtol, &
            Lnuc1, Lphoto1, Lnucneu1, Ltot1, &
            Lnuc2, Lphoto2, Lnucneu2, Ltot2
         logical :: trace, half_and_half  
         
         
         include 'formats.dek'
         
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
                           
         ierr = 0
         nz = s% nz
         species = s% species
         
         call do_alloc(ierr)
         if (ierr /= 0) then
            write(*,*) 'allocate failed in do_op_split'
            return
         end if

         trace = s% op_split_burn_mix_trace 
!         s% chem_timescale = eval_chem_timescale(s)

         s% do_struct_hydro = .true.
         s% do_struct_thermo = .true.
         s% do_burn = do_burn
         s% do_mix = do_mix

         if (.true. .or. s% do_mix) then
            s% avg_burn_dxdt(:,1:s% nz) = 0d0
            s% avg_mix_dxdt(:,1:s% nz) = 0d0
         else ! set initial avg_burn_dxdt for use by mixing
            write(*,2) 'set initial avg_burn_dxdt', s% model_number
            do k=1,nz
               do j=1,species
                  s% avg_burn_dxdt(j,k) = s% dxdt_nuc(j,k)
               end do
            end do
         end if

         s% num_burn_max_iters = 0

         call save_pre_values(s, ierr)
         if (ierr /= 0) then
            call dealloc
            return
         end if

         if (trace) write(*,2) 'call do_hydro_converge', s% model_number
         res = do_hydro_converge( &
                  s, s% newton_itermin, skip_global_corr_coeff_limit, dt)
         if (res /= keep_going) then
            if (trace) write(*,*) 'struct failed to converge'
            if (s% report_ierr) write(*,*) 'struct failed in do_op_split'
            ierr = -1            
            write(*,3) 'split hydro failed for model', s% model_number
            write(*,*)
            call dealloc
            s% xa(:,1:s% nz) = s% xa_pre(:,1:s% nz)
            return
         end if

         call dealloc
                     
         
         contains
         
         
         subroutine do_alloc(ierr)
            use alloc
            integer, intent(out) :: ierr
            integer :: sz, sz_extra
            sz = nz*species
            sz_extra = nz_alloc_extra*species
            
            call non_crit_get_work_array( &
               s, s% avg_burn_dxdt1, sz, sz_extra, 'solve_split_burn_mix', ierr)
            if (ierr /= 0) return            
            s% avg_burn_dxdt(1:species,1:nz) => s% avg_burn_dxdt1(1:sz)
            
            call non_crit_get_work_array( &
               s, s% avg_mix_dxdt1, sz, sz_extra, 'solve_split_burn_mix', ierr)
            if (ierr /= 0) return            
            s% avg_mix_dxdt(1:species,1:nz) => s% avg_mix_dxdt1(1:sz)
            
         end subroutine do_alloc

            
         subroutine dealloc
            use alloc
            call non_crit_return_work_array(s, s% avg_burn_dxdt1, 'solve_split_burn_mix')
            call non_crit_return_work_array(s, s% avg_mix_dxdt1, 'solve_split_burn_mix')
            nullify( &
               s% avg_burn_dxdt1, s% avg_mix_dxdt1, &
               s% avg_burn_dxdt, s% avg_mix_dxdt)
         end subroutine dealloc


      end subroutine do_new_solve




      subroutine do_old_solve( &
            s, skip_global_corr_coeff_limit, dt, ierr)
         use star_utils, only: update_time, total_times, center_value
         use solve_hydro, only: do_hydro_converge

         type (star_info), pointer :: s
         logical, intent(in) :: skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         integer :: i, j, k, nz, species, &
            do_burn_mix_before_struct, do_burn_mix_after_struct
         integer :: time0, clock_rate, res
         real(dp) :: total_all_before, err, limit_T, dt_burn_and_mix, &
            op_split_Ltot_atol, op_split_Ltot_rtol, &
            Lnuc1, Lphoto1, Lnucneu1, Ltot1, &
            Lnuc2, Lphoto2, Lnucneu2, Ltot2
         logical :: trace, half_and_half  
         
         
         include 'formats.dek'
         
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
                           
         ierr = 0
         nz = s% nz
         species = s% species
         
         call do_alloc(ierr)
         if (ierr /= 0) then
            write(*,*) 'allocate failed in do_op_split'
            return
         end if

         trace = s% op_split_burn_mix_trace 
!         s% chem_timescale = eval_chem_timescale(s)

         s% do_struct_hydro = .true.
         s% do_struct_thermo = .true.
         s% do_burn = .false.
         s% do_mix = .false.

         s% avg_burn_dxdt(:,1:s% nz) = 0d0
         s% avg_mix_dxdt(:,1:s% nz) = 0d0

         s% num_burn_max_iters = 0

         call save_pre_values(s, ierr)
         if (ierr /= 0) then
            call dealloc
            return
         end if
         
         call burn_and_mix(ierr)
         if (ierr /= 0) then
            call dealloc
            return
         end if

         if (trace) write(*,2) 'call do_hydro_converge', s% model_number
         res = do_hydro_converge( &
                  s, s% newton_itermin, skip_global_corr_coeff_limit, dt)
         if (res /= keep_going) then
            if (trace) write(*,*) 'struct failed to converge'
            if (s% report_ierr) write(*,*) 'struct failed in do_op_split'
            ierr = -1            
            write(*,3) 'split hydro failed for model', s% model_number
            write(*,*)
            call dealloc
            s% xa(:,1:s% nz) = s% xa_pre(:,1:s% nz)
            return
         end if

         call dealloc
                     
         
         contains
         
         
         subroutine burn_and_mix(ierr)
            use micro, only: do_mix_burn
            use mix_info, only: get_convection_sigmas
            use hydro_rotation, only: get_rotation_sigmas
            use mix_smoothing, only: smooth_newly_non_conv
            use hydro_vars, only: set_vars
            
            integer, intent(out) :: ierr
            
            include 'formats'
   
            call get_convection_sigmas(s, dt, ierr)
            if (ierr /= 0) then
               if (s% report_ierr) write(*,*) 'get_convection_sigmas failed'
               return
            end if
      
            if (s% rotation_flag) then
               call get_rotation_sigmas(s, 1, nz, dt, ierr)
               if (ierr /= 0) return
            end if
   
            if (trace) write(*,2) 'call do_solve_split_burn_mix', s% model_number
            call do_mix_burn(s, dt, ierr) 
            if (ierr /= 0) then
               if (s% report_ierr) write(*,*) 'burn_mix failed'
               return
            end if

            call set_vars(s, dt, ierr) ! this does set_mixing_info too
            if (ierr /= 0) return
      
            if (s% smooth_convective_bdy) then
               ierr = 0
               if (trace) write(*,2) 'call smooth_newly_non_conv', s% model_number
               call smooth_newly_non_conv(s, ierr)
               if (ierr /= 0) then
                  if (s% report_ierr) then
                     write(*, *) 'smooth_newly_non_conv ierr: retry', s% model_number
                  end if
                  return
               end if
            end if
         
         end subroutine burn_and_mix
         
         
         subroutine do_alloc(ierr)
            use alloc
            integer, intent(out) :: ierr
            integer :: sz, sz_extra
            sz = nz*species
            sz_extra = nz_alloc_extra*species
            
            call non_crit_get_work_array( &
               s, s% avg_burn_dxdt1, sz, sz_extra, 'solve_split_burn_mix', ierr)
            if (ierr /= 0) return            
            s% avg_burn_dxdt(1:species,1:nz) => s% avg_burn_dxdt1(1:sz)
            
            call non_crit_get_work_array( &
               s, s% avg_mix_dxdt1, sz, sz_extra, 'solve_split_burn_mix', ierr)
            if (ierr /= 0) return            
            s% avg_mix_dxdt(1:species,1:nz) => s% avg_mix_dxdt1(1:sz)
            
         end subroutine do_alloc

            
         subroutine dealloc
            use alloc
            call non_crit_return_work_array(s, s% avg_burn_dxdt1, 'solve_split_burn_mix')
            call non_crit_return_work_array(s, s% avg_mix_dxdt1, 'solve_split_burn_mix')
            nullify( &
               s% avg_burn_dxdt1, s% avg_mix_dxdt1, &
               s% avg_burn_dxdt, s% avg_mix_dxdt)
         end subroutine dealloc


      end subroutine do_old_solve
      
      
      integer function do_smooth_conv_bdy_and_mix_omega(s, dt)
         use solve_omega_mix, only: do_solve_omega_mix
         use mix_smoothing, only: smooth_newly_non_conv

         type (star_info), pointer :: s
         real(dp), intent(in) :: dt
         
         integer :: ierr
      
         do_smooth_conv_bdy_and_mix_omega = keep_going
         
         if (s% smooth_convective_bdy) then
            ierr = 0
            call smooth_newly_non_conv(s, ierr) ! do this after do_struct_burn_mix
            if (ierr /= 0) then
               if (s% report_ierr) then
                  write(*, *) 'smooth_newly_non_conv ierr: retry', s% model_number
               end if
               do_smooth_conv_bdy_and_mix_omega = retry
               s% result_reason = nonzero_ierr
               return
            end if
         end if
      
         if (s% rotation_flag) then
            do_smooth_conv_bdy_and_mix_omega = do_solve_omega_mix(s, dt)
            if (do_smooth_conv_bdy_and_mix_omega /= keep_going) return
         end if      
      
      end function do_smooth_conv_bdy_and_mix_omega


      end module struct_burn_mix


