! ***********************************************************************
!
!   Copyright (C) 2013  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************



      module mod_star_sparse
      
      use star_private_def
      use const_def
      use num_def
      use utils_lib, only: fill_with_NaNs, fill_quad_with_NaNs
      use mtx_lib, only: num_klu_ipar_decsol, num_klu_rpar_decsol, &
         klu_dble_decsols_nrhs_0_based, klu_quad_decsols_nrhs_0_based, &
         dense_to_col_with_diag_0_based, quad_dense_to_col_with_diag_0_based
         
      implicit none
      
      
      
      logical, parameter :: do_fill_with_NaNs = .false.
      

      integer, parameter :: lid = num_klu_ipar_decsol
      integer, parameter :: lrd = num_klu_rpar_decsol
      

      
      logical :: keep_sprs_statistics = .false.
      
      integer :: sprs_num_alloc_klu_storage = 0
      integer :: sprs_num_clear_klu_storage = 0
      integer :: sprs_num_analyze = 0
      integer :: sprs_num_factor = 0
      integer :: sprs_num_refactor = 0
      integer :: sprs_num_solve = 0
      integer :: sprs_num_free_numeric = 0
      integer :: sprs_num_free_symbolic = 0
      integer :: sprs_num_alloc_klu_factors = 0
      integer :: sprs_num_free_klu_factors = 0

      
      contains

      
      
            
      subroutine star_alloc_klu_storage(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         integer :: nz, k, old_sz, new_sz
         type(sparse_info), pointer :: ks(:)
         
         nz = s% nz
         if (associated(s% bcyclic_klu_storage)) then
            ks => s% bcyclic_klu_storage
            old_sz = size(s% bcyclic_klu_storage)
            if (old_sz >= nz) then
               return
            end if
         else  
            old_sz = 0
         end if
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_alloc_klu_storage = sprs_num_alloc_klu_storage + 1
!$omp end critical (sprs_statistics)
         end if
         
         new_sz = nz*2 + 1000
         allocate (s% bcyclic_klu_storage(new_sz), stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'alloc failed for klu_storage in star_sparse'
            return
         end if
         
         if (old_sz > 0) then
            do k = 1, old_sz
               s% bcyclic_klu_storage(k) = ks(k)
            end do
         end if
         
         do k = old_sz+1, new_sz
            s% bcyclic_klu_storage(k)% sprs_nonzeros = -1
            s% bcyclic_klu_storage(k)% ia => null()
            s% bcyclic_klu_storage(k)% ja => null()
            s% bcyclic_klu_storage(k)% values => null()
            s% bcyclic_klu_storage(k)% rpar_decsol = 0
            s% bcyclic_klu_storage(k)% ipar8_decsol = 0
            s% bcyclic_klu_storage(k)% have_Numeric = .false.
         end do
         
         if (old_sz > 0) then
            write(*,*) 'free old klu_storage'
            deallocate(ks)
         end if
            
      end subroutine star_alloc_klu_storage
      
      
      subroutine star_sparse_setup_shared(s, k, k_shared, nvar, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, k_shared, nvar
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)

         include 'formats'
         
         ierr = 0
         
         if (s% bcyclic_shared_sprs_nonzeros <= 0) then
            write(*,3) 's% bcyclic_shared_sprs_nonzeros', &
               s% bcyclic_shared_sprs_nonzeros, s% model_number
            stop 'sparse_setup_shared: burn1_BE'
         end if
      
         ks => s% bcyclic_klu_storage
         
         if (ks(k)% have_Numeric) return ! already setup from last time
                  
         ! check private values array
         if (associated(ks(k)% values)) then
            if (size(ks(k)% values,dim=1) < s% bcyclic_shared_sprs_nonzeros) then
               deallocate(ks(k)% values)
               allocate(ks(k)% values(s% bcyclic_shared_sprs_nonzeros + 100), stat=ierr)
            end if
         else 
            allocate(ks(k)% values(s% bcyclic_shared_sprs_nonzeros + 100), stat=ierr)
         end if
         if (ierr /= 0) then
            write(*,2) 'burn1_BE: allocate sparse matrix values failed', &
               s% bcyclic_shared_sprs_nonzeros
            stop
            return
         end if
         if (do_fill_with_NaNs) call fill_with_NaNs(ks(k)% values)
         
         if (k == k_shared) return
         ! copy shared stuff
         
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b => b_ary
         
         ks(k)% sprs_nonzeros = s% bcyclic_shared_sprs_nonzeros
         ks(k)% ia => s% bcyclic_sprs_shared_ia
         ks(k)% ja => s% bcyclic_sprs_shared_ja
         ks(k)% ipar8_decsol = s% bcyclic_sprs_shared_ipar8_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_alloc_klu_factors = sprs_num_alloc_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if

         call klu_dble_decsols_nrhs_0_based( &  ! copy klu factors array
            -6, 0, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
         if (ierr /= 0) return
      
      
      end subroutine star_sparse_setup_shared


      subroutine star_sparse_matrix_info( &
            s, k, nvar, species, net_lwork, mtx, sprs_nonzeros, ierr)   

         use eos_def, only : i_eta
         use net_lib, only: net_get_symbolic_d_dxdt_dx
         use rates_def, only: std_reaction_Qs, std_reaction_neuQs, num_rvs
         
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar, species, net_lwork
         real(dp), pointer, intent(inout) :: mtx(:,:) ! (nvar,nvar)
         integer, intent(out) :: ierr, sprs_nonzeros
      
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: values(:), b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         integer :: nonzero_cnt, j, i, nvar_hydro, i1, i2
         
         real(dp) :: d_epsnuc_dx(species) 
         real(dp) :: dxdt_nuc(species), dxdt_dRho(species), dxdt_dT(species)
         real(dp) :: d_dxdt_dRho(species)
         real(dp) :: d_dxdt_dT(species)
         real(dp) :: d_dxdt_dx(species, species)            
         real(dp) :: reaction_eps_nuc(num_rvs, s% num_reactions)
         real(dp) :: eps_nuc_categories(num_rvs, num_categories)
         real(dp) :: eps_nuc_neu_total, eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT
         real(dp), dimension(num_rvs, s% num_reactions) :: rate_screened, rate_raw
         
         real(dp), target :: net_work_ary(net_lwork)
         real(dp), pointer :: net_work(:)
         integer, parameter :: screening_mode = 0
          
         include 'formats'
         
         ierr = 0
         net_work => net_work_ary
         
         call net_get_symbolic_d_dxdt_dx( &
            s% net_handle, species, s% num_reactions, s% xa(1:species,k), &
            s% T(k), s% lnT(k)/ln10, s% rho(k), s% lnd(k)/ln10, &
            s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
            s% eta(k), s% d_eos_dlnd(i_eta,k), s% d_eos_dlnT(i_eta,k), &
            s% rate_factors, s% category_factors, &
            std_reaction_Qs, std_reaction_neuQs, &
            eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, d_epsnuc_dx, & 
            dxdt_nuc, dxdt_dRho, dxdt_dT, d_dxdt_dx, &
            screening_mode, s% theta_e(k), &
            rate_screened, rate_raw, &
            reaction_eps_nuc, eps_nuc_categories, eps_nuc_neu_total, &
            net_lwork, net_work, ierr)
         if (ierr /= 0) then
            write(*,2) 'net_get_symbolic_d_dxdt_dx failed', k
            stop
            return
         end if
         
         mtx(:,:) = 0
         nvar_hydro = nvar - species
         do j=1,species
            do i=1,species
               mtx(i+nvar_hydro,j+nvar_hydro) = d_dxdt_dx(i,j)
            end do
            mtx(j+nvar_hydro,j+nvar_hydro) = 1 ! always include diagonal
         end do
         
         do j=1,nvar_hydro
            do i=1,nvar_hydro
               mtx(i,j) = 1
            end do
         end do
         
         i1 = max(s% i_xlnd, s% i_lnPgas) ! only one is nonzero
         i2 = s% i_lnT
         do j=nvar_hydro+1,nvar
            mtx(s% equL,j) = 1 ! luminosity eqn depends on abundances
            mtx(j,i1) = 1 ! abundance eqns depend on density or pressure
            mtx(j,i2) = 1 ! and on temperature
         end do
         
         ks => s% bcyclic_klu_storage
      
         nonzero_cnt = 0
         do j=1,nvar
            do i=1,nvar
               if (mtx(i,j) == 0) cycle
               nonzero_cnt = nonzero_cnt + 1
            end do
         end do
         
         if (associated(ks(k)% ia)) then
            if (size(ks(k)% ia,dim=1) < nvar+1) then
               deallocate(ks(k)% ia)
               allocate(ks(k)% ia(nvar + 10), stat=ierr)
            end if
         else 
            allocate(ks(k)% ia(nvar + 10), stat=ierr)
         end if
         if (do_fill_with_NaNs) ks(k)% ia = -9999999
         if (ierr /= 0) then
            write(*,2) 'allocate failed for ia', nvar+1
            stop
            return
         end if
         
         if (associated(ks(k)% ja)) then
            if (size(ks(k)% ja,dim=1) < nonzero_cnt) then
               deallocate(ks(k)% ja)
               allocate(ks(k)% ja(nonzero_cnt + 100), stat=ierr)
            end if
         else 
            allocate(ks(k)% ja(nonzero_cnt + 100), stat=ierr)
         end if
         if (do_fill_with_NaNs) ks(k)% ja = -9999999
         if (ierr /= 0) then
            write(*,2) 'allocate failed for ja', nonzero_cnt
            stop
            return
         end if
         
         if (associated(ks(k)% values)) then
            if (size(ks(k)% values,dim=1) < nonzero_cnt) then
               deallocate(ks(k)% values)
               allocate(ks(k)% values(nonzero_cnt + 100), stat=ierr)
            end if
         else 
            allocate(ks(k)% values(nonzero_cnt + 100), stat=ierr)
         end if
         if (do_fill_with_NaNs) call fill_with_NaNs(ks(k)% values)
         if (ierr /= 0) then
            write(*,2) 'allocate failed for values', nonzero_cnt
            stop
            return
         end if
         
         values => ks(k)% values
         
         ! compressed_format is compressed_col_sparse_0_based for KLU
         call dense_to_col_with_diag_0_based( &
            nvar, nvar, mtx, nonzero_cnt, sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, values, ierr)
         if (ierr /= 0) then
            write(*,*) 'solve burn sparse failed in converting from dense to sparse'
            return
         end if
         if (sprs_nonzeros /= nonzero_cnt) then
            write(*,*) &
               'solve burn sparse failed in converting dense to sparse: bad sprs_nonzeros'
            ierr = -1
            return
         end if
         ks(k)% sprs_nonzeros = sprs_nonzeros
      
      end subroutine star_sparse_matrix_info
      
         
      subroutine star_sparse_store_new_values(s, k, nvar, mtx, dbg, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         real(dp), pointer, intent(inout) :: mtx(:,:)
         logical, intent(in) :: dbg
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: values(:)
         integer, pointer :: colptr(:) ! (n+1)
         integer, pointer :: rowind(:) ! (nz)
         integer :: j, i, nz, c0, c1, c, sprs_nonzeros
         
         include 'formats'

         ierr = 0
         ks => s% bcyclic_klu_storage
         values => ks(k)% values
         colptr => ks(k)% ia
         rowind => ks(k)% ja
         sprs_nonzeros = ks(k)% sprs_nonzeros
         nz = 0
         do j=1,nvar ! column index
            c0 = colptr(j)+1 ! 1st rowind
            c1 = colptr(j+1) ! last rowind
            do c=c0,c1
               i = rowind(c)+1 ! row index
               nz = nz+1
               if (nz > sprs_nonzeros) then
                  write(*,5) 'bad value for nz', nz, k, i, j
                  stop 'sparse_store_new_values'
                  ierr = -1
                  return
               end if
               if (i < 1 .or. i > nvar) then
                  write(*,5) 'bad value for i', nz, k, i, j
                  stop 'sparse_store_new_values'
                  ierr = -1
                  return
               end if
               if (dbg) write(*,5) 'values(nz) = mtx(i,j)', nz, i, j, k, mtx(i,j)
               values(nz) = mtx(i,j)
            end do
         end do
         if (nz /= sprs_nonzeros) then
            ierr = -1
            write(*,4) 'nz /= ks(k)% sprs_nonzeros', &
               k, nz, ks(k)% sprs_nonzeros
            stop 'sparse_store_new_values'
            return
         end if

      end subroutine star_sparse_store_new_values
         
         
      subroutine star_sparse_analyze_and_factor(s, k, nvar, mtx, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         
         include 'formats'

         ierr = 0
         ks => s% bcyclic_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if

         call star_sparse_free_numeric(s, k, nvar, ierr)
         if (ierr /= 0) return
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_analyze = sprs_num_analyze + 1
            sprs_num_factor = sprs_num_factor + 1
            sprs_num_alloc_klu_factors = sprs_num_alloc_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if
         
         call klu_dble_decsols_nrhs_0_based( & ! klu_analyze and klu_factor
            0, nvar, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
            
         ks(k)% have_Numeric = .true.
         
         if (ierr /= 0) then
            write(*,2) 'do_klu_dble_decsols_nrhs_0_based failed factor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if
      
      end subroutine star_sparse_analyze_and_factor
         
         
      subroutine star_sparse_analyze(s, k, nvar, mtx, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         
         include 'formats'

         ierr = 0
         ks => s% bcyclic_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_analyze = sprs_num_analyze + 1
            sprs_num_alloc_klu_factors = sprs_num_alloc_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if

         call klu_dble_decsols_nrhs_0_based( & ! klu_analyze
            -2, nvar, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
            
         if (ierr /= 0) then
            write(*,2) 'klu_dble_decsols_nrhs_0_based failed factor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if
      
      end subroutine star_sparse_analyze
         
         
      subroutine star_sparse_factor(s, k, nvar, mtx, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         
         include 'formats'

         ierr = 0
         ks => s% bcyclic_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         
         call star_sparse_free_numeric(s, k, nvar, ierr)
         if (ierr /= 0) return
         
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_factor = sprs_num_factor + 1
!$omp end critical (sprs_statistics)
         end if
         
         call klu_dble_decsols_nrhs_0_based( & ! klu_factor
            -1, nvar, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
            
         ks(k)% have_Numeric = .true.
            
         if (ierr /= 0) then
            write(*,2) 'klu_dble_decsols_nrhs_0_based failed factor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if
      
      end subroutine star_sparse_factor
         
         
      subroutine star_sparse_refactor(s, k, nvar, mtx, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         
         ierr = 0
         ks => s% bcyclic_klu_storage
         if (.not. ks(k)% have_Numeric) then
            write(*,3) 'called refactor without Numeric', k
            ierr = -1
            return
         end if
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_refactor = sprs_num_refactor + 1
!$omp end critical (sprs_statistics)
         end if

         call klu_dble_decsols_nrhs_0_based( & ! refactor
            3, nvar, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
            
      end subroutine star_sparse_refactor
         
         
      subroutine star_sparse_no_pivot(s, k, nvar, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         ierr = 0
         ks => s% bcyclic_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         call klu_dble_decsols_nrhs_0_based( &
            -7, nvar, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)  
      end subroutine star_sparse_no_pivot
         
         
      real(dp) function star_sparse_rcond(s, k, nvar, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         ierr = 0
         ks => s% bcyclic_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         call klu_dble_decsols_nrhs_0_based( & ! klu_rcond
            4, nvar, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)  
         star_sparse_rcond = rpar(1)    
      end function star_sparse_rcond
         
         
      real(dp) function star_sparse_rgrowth(s, k, nvar, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         
         ierr = 0
         ks => s% bcyclic_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         call klu_dble_decsols_nrhs_0_based( & ! klu_rgrowth
            5, nvar, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)  
         star_sparse_rgrowth = rpar(1)    
      end function star_sparse_rgrowth
         
         
      real(dp) function star_sparse_condest(s, k, nvar, ierr)  
         ! accurate condition number estimate,
         ! but more expensive to compute than rgrowth
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         ierr = 0
         ks => s% bcyclic_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         call klu_dble_decsols_nrhs_0_based( & ! klu_condest
            6, nvar, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)  
         star_sparse_condest = rpar(1)    
      end function star_sparse_condest


      subroutine star_sparse_solve(s, k, nrhs, nvar, b, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, nrhs, nvar
         real(dp), pointer :: b(:)
         integer, intent(out) :: ierr
         real(dp), pointer :: rpar(:)
         type(sparse_info), pointer :: ks(:)
         integer, pointer :: ipar(:)
         include 'formats'

         ks => s% bcyclic_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_solve = sprs_num_solve + 1
!$omp end critical (sprs_statistics)
         end if

         call klu_dble_decsols_nrhs_0_based( & ! klu_solve
            1, nrhs, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
            
         if (ierr /= 0) then
            write(*,2) 'do_klu_dble_decsols_nrhs_0_based failed refactor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if

      end subroutine star_sparse_solve


      subroutine star_sparse_free_all(s, nvar, k_min, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: nvar, k_min
         integer, intent(out) :: ierr
         
         integer :: k
         type(sparse_info), pointer :: ks(:)
         
         include 'formats'
         ierr = 0
         ks => s% bcyclic_klu_storage
         do k = k_min, size(ks)
            if (ks(k)% sprs_nonzeros <= 0) cycle
            call star_sparse_free_numeric(s, k, nvar, ierr)
            if (ierr /= 0) exit
            call star_sparse_free_klu_factors(s, k, nvar, ierr)
            if (ierr /= 0) exit
            ks(k)% sprs_nonzeros = -1
         end do
      end subroutine star_sparse_free_all
      
      
      subroutine star_sparse_free_klu_factors(s, k, nvar, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         
         include 'formats'
         ierr = 0
         ks => s% bcyclic_klu_storage
         b(1:1) => b_ary(1:1)
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol


         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_free_klu_factors = sprs_num_free_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if

         ! free klu factors array (this is also done by klu_free_symbolic)
         call klu_dble_decsols_nrhs_0_based( & 
            -5, 0, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
               
      end subroutine star_sparse_free_klu_factors
      
      
      subroutine star_sparse_free_symbolic(s, k, nvar, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         
         include 'formats'
         ierr = 0
         ks => s% bcyclic_klu_storage
         b(1:1) => b_ary(1:1)
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_free_symbolic = sprs_num_free_symbolic + 1
            sprs_num_free_klu_factors = sprs_num_free_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if

         call klu_dble_decsols_nrhs_0_based( & 
            -4, 0, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
               
      end subroutine star_sparse_free_symbolic
      
      
      subroutine star_sparse_free_numeric(s, k, nvar, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)

         include 'formats'
         ierr = 0
         ks => s% bcyclic_klu_storage
         if (.not. ks(k)% have_Numeric) return
         b(1:1) => b_ary(1:1)
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_free_numeric = sprs_num_free_numeric + 1
!$omp end critical (sprs_statistics)
         end if         
         call klu_dble_decsols_nrhs_0_based( & 
            -3, 0, nvar, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
         ks(k)% have_Numeric = .false.
         
      end subroutine star_sparse_free_numeric
      
      
      subroutine star_clear_klu_storage(s, max_k_for_free_shared)
         type (star_info), pointer :: s
         integer, intent(in) :: max_k_for_free_shared
         integer :: k

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_clear_klu_storage = sprs_num_clear_klu_storage + 1
!$omp end critical (sprs_statistics)
         end if

         do k = 1, size(s% bcyclic_klu_storage)
            if (k <= max_k_for_free_shared) then
               if (associated(s% bcyclic_klu_storage(k)% ia)) &
                  deallocate(s% bcyclic_klu_storage(k)% ia)
               if (associated(s% bcyclic_klu_storage(k)% ja)) &
                  deallocate(s% bcyclic_klu_storage(k)% ja)
            end if
            if (associated(s% bcyclic_klu_storage(k)% values)) &
               deallocate(s% bcyclic_klu_storage(k)% values)
         end do
         deallocate(s% bcyclic_klu_storage)
         nullify(s% bcyclic_klu_storage)
         
      end subroutine star_clear_klu_storage

      end module mod_star_sparse



