! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************



      module solve_burn_sparse
      
      use star_private_def
      use const_def
      use num_def
      use utils_lib, only: fill_with_NaNs, fill_quad_with_NaNs
      use mtx_lib, only: num_klu_ipar_decsol, num_klu_rpar_decsol, &
         klu_dble_decsols_nrhs_0_based, klu_quad_decsols_nrhs_0_based, &
         dense_to_col_with_diag_0_based, quad_dense_to_col_with_diag_0_based
         
      implicit none
      
      
      
      logical, parameter :: do_fill_with_NaNs = .false.
      

      integer, parameter :: lid = num_klu_ipar_decsol
      integer, parameter :: lrd = num_klu_rpar_decsol
      

      
      logical :: keep_sprs_statistics = .false.
      
      integer :: sprs_num_alloc_klu_storage = 0
      integer :: sprs_num_clear_klu_storage = 0
      integer :: sprs_num_analyze = 0
      integer :: sprs_num_factor = 0
      integer :: sprs_num_refactor = 0
      integer :: sprs_num_solve = 0
      integer :: sprs_num_free_numeric = 0
      integer :: sprs_num_free_symbolic = 0
      integer :: sprs_num_alloc_klu_factors = 0
      integer :: sprs_num_free_klu_factors = 0

      
      contains

      
      
            
      subroutine alloc_klu_storage(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         integer :: nz, k, old_sz, new_sz
         type(sparse_info), pointer :: ks(:)
         
         nz = s% nz
         if (associated(s% burn_klu_storage)) then
            ks => s% burn_klu_storage
            old_sz = size(s% burn_klu_storage)
            if (old_sz >= nz) then
               return
            end if
         else  
            old_sz = 0
         end if
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_alloc_klu_storage = sprs_num_alloc_klu_storage + 1
!$omp end critical (sprs_statistics)
         end if
         
         new_sz = nz*2 + 1000
         allocate (s% burn_klu_storage(new_sz), stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'alloc failed for klu_storage in solve_burn'
            return
         end if
         
         if (old_sz > 0) then
            do k = 1, old_sz
               s% burn_klu_storage(k) = ks(k)
            end do
         end if
         
         do k = old_sz+1, new_sz
            s% burn_klu_storage(k)% sprs_nonzeros = -1
            s% burn_klu_storage(k)% ia => null()
            s% burn_klu_storage(k)% ja => null()
            s% burn_klu_storage(k)% values => null()
            s% burn_klu_storage(k)% rpar_decsol = 0
            s% burn_klu_storage(k)% ipar8_decsol = 0
            s% burn_klu_storage(k)% have_Numeric = .false.
         end do
         
         if (old_sz > 0) then
            write(*,*) 'free old klu_storage'
            deallocate(ks)
         end if
            
      end subroutine alloc_klu_storage
      
      
      subroutine sparse_setup_shared(s, k, k_shared, species, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, k_shared, species
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)

         include 'formats'
         
         ierr = 0
         
         if (s% burn_shared_sprs_nonzeros <= 0) then
            write(*,3) 's% burn_shared_sprs_nonzeros', &
               s% burn_shared_sprs_nonzeros, s% model_number
            stop 'sparse_setup_shared: burn1_BE'
         end if
      
         ks => s% burn_klu_storage
         
         if (ks(k)% have_Numeric) return ! already setup from last time
                  
         ! check private values array
         if (associated(ks(k)% values)) then
            if (size(ks(k)% values,dim=1) < s% burn_shared_sprs_nonzeros) then
               deallocate(ks(k)% values)
               allocate(ks(k)% values(s% burn_shared_sprs_nonzeros + 100), stat=ierr)
            end if
         else 
            allocate(ks(k)% values(s% burn_shared_sprs_nonzeros + 100), stat=ierr)
         end if
         if (ierr /= 0) then
            write(*,2) 'burn1_BE: allocate sparse matrix values failed', &
               s% burn_shared_sprs_nonzeros
            stop
            return
         end if
         if (do_fill_with_NaNs) call fill_with_NaNs(ks(k)% values)
         
         if (k == k_shared) return
         ! copy shared stuff
         
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b => b_ary
         
         ks(k)% sprs_nonzeros = s% burn_shared_sprs_nonzeros
         ks(k)% ia => s% burn_sprs_shared_ia
         ks(k)% ja => s% burn_sprs_shared_ja
         ks(k)% ipar8_decsol = s% burn_sprs_shared_ipar8_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_alloc_klu_factors = sprs_num_alloc_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if

         call klu_dble_decsols_nrhs_0_based( &  ! copy klu factors array
            -6, 0, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
         if (ierr /= 0) return
      
      
      end subroutine sparse_setup_shared


      subroutine sparse_net_matrix_info( &
            s, k, species, net_lwork, mtx, sprs_nonzeros, ierr)   

         use eos_def, only : i_eta

         use net_lib, only: net_get_symbolic_d_dxdt_dx
         use rates_def, only: std_reaction_Qs, std_reaction_neuQs
         
         use utils_lib, only: enlarge_integer_if_needed_1, enlarge_if_needed_1
         use rates_def, only: num_rvs
         
         type (star_info), pointer :: s
         integer, intent(in) :: k, species, net_lwork
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr, sprs_nonzeros
      
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: values(:), b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         integer :: nonzero_cnt, j, i
         
         
         real(dp) :: d_epsnuc_dx(species) 
         real(dp) :: dxdt_nuc(species), dxdt_dRho(species), dxdt_dT(species)
         real(dp) :: d_dxdt_dRho(species)
         real(dp) :: d_dxdt_dT(species)
         real(dp) :: d_dxdt_dx(species, species)            
         real(dp) :: reaction_eps_nuc(num_rvs, s% num_reactions)
         real(dp) :: eps_nuc_categories(num_rvs, num_categories)
         real(dp) :: eps_nuc_neu_total, eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT
         real(dp), dimension(num_rvs, s% num_reactions) :: rate_screened, rate_raw
         
         real(dp), target :: net_work_ary(net_lwork)
         real(dp), pointer :: net_work(:)
         integer, parameter :: screening_mode = 0
          
         include 'formats'
         
         ierr = 0
         net_work => net_work_ary
         
         mtx(:,:) = 0
         call net_get_symbolic_d_dxdt_dx( &
            s% net_handle, species, s% num_reactions, s% xa(1:species,k), &
            s% T(k), s% lnT(k)/ln10, s% rho(k), s% lnd(k)/ln10, &
            s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
            s% eta(k), s% d_eos_dlnd(i_eta,k), s% d_eos_dlnT(i_eta,k), &
            s% rate_factors, s% category_factors, &
            std_reaction_Qs, std_reaction_neuQs, &
            eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, d_epsnuc_dx, & 
            dxdt_nuc, dxdt_dRho, dxdt_dT, d_dxdt_dx, &
            screening_mode, s% theta_e(k), &
            rate_screened, rate_raw, &
            reaction_eps_nuc, eps_nuc_categories, eps_nuc_neu_total, &
            net_lwork, net_work, ierr)
         if (ierr /= 0) then
            write(*,2) 'net_get_symbolic_d_dxdt_dx failed', k
            stop
            return
         end if
         
         do j=1,species
            do i=1,species
               mtx(i,j) = d_dxdt_dx(i,j)
            end do
            mtx(j,j) = 1 ! always include diagonal
         end do
         
         ks => s% burn_klu_storage
      
         nonzero_cnt = 0
         do j=1,species
            do i=1,species
               if (i == j) then ! don't clip diagonals
                  nonzero_cnt = nonzero_cnt + 1
                  cycle
               end if
               if (mtx(i,j) == 0) cycle
               nonzero_cnt = nonzero_cnt + 1
            end do
         end do
         
         !call enlarge_integer_if_needed_1(ks(k)% ia, species+1, 10, ierr)
         if (associated(ks(k)% ia)) then
            if (size(ks(k)% ia,dim=1) < species+1) then
               deallocate(ks(k)% ia)
               allocate(ks(k)% ia(species + 10), stat=ierr)
            end if
         else 
            allocate(ks(k)% ia(species + 10), stat=ierr)
         end if
         if (do_fill_with_NaNs) ks(k)% ia = -9999999
         if (ierr /= 0) then
            write(*,2) 'allocate failed for ia', species+1
            stop
            return
         end if
         
         !call enlarge_integer_if_needed_1(ks(k)% ja, nonzero_cnt, 100, ierr)
         if (associated(ks(k)% ja)) then
            if (size(ks(k)% ja,dim=1) < nonzero_cnt) then
               deallocate(ks(k)% ja)
               allocate(ks(k)% ja(nonzero_cnt + 100), stat=ierr)
            end if
         else 
            allocate(ks(k)% ja(nonzero_cnt + 100), stat=ierr)
         end if
         if (do_fill_with_NaNs) ks(k)% ja = -9999999
         if (ierr /= 0) then
            write(*,2) 'allocate failed for ja', nonzero_cnt
            stop
            return
         end if
         
         !call enlarge_if_needed_1(ks(k)% values, nonzero_cnt, 100, ierr)
         if (associated(ks(k)% values)) then
            if (size(ks(k)% values,dim=1) < nonzero_cnt) then
               deallocate(ks(k)% values)
               allocate(ks(k)% values(nonzero_cnt + 100), stat=ierr)
            end if
         else 
            allocate(ks(k)% values(nonzero_cnt + 100), stat=ierr)
         end if
         if (do_fill_with_NaNs) call fill_with_NaNs(ks(k)% values)
         if (ierr /= 0) then
            write(*,2) 'allocate failed for values', nonzero_cnt
            stop
            return
         end if
         
         values => ks(k)% values
         
         ! compressed_format is compressed_col_sparse_0_based for KLU
         call dense_to_col_with_diag_0_based( &
            species, species, mtx, nonzero_cnt, sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, values, ierr)
         if (ierr /= 0) then
            write(*,*) 'solve burn sparse failed in converting from dense to sparse'
            return
         end if
         if (sprs_nonzeros /= nonzero_cnt) then
            write(*,*) &
               'solve burn sparse failed in converting dense to sparse: bad sprs_nonzeros'
            ierr = -1
            return
         end if
         ks(k)% sprs_nonzeros = sprs_nonzeros

      end subroutine sparse_net_matrix_info


      subroutine sparse_get_matrix_info( &
            s, k, species, mtx, sprs_nonzeros, ierr)   
         use utils_lib, only: enlarge_integer_if_needed_1, enlarge_if_needed_1
         
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: sprs_nonzeros, ierr
      
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: values(:), b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         integer :: nonzero_cnt, j, i
         
         include 'formats'
         
         ierr = 0
         ks => s% burn_klu_storage
      
         nonzero_cnt = 0
         do j=1,species
            do i=1,species
               if (i == j) then ! don't clip diagonals
                  nonzero_cnt = nonzero_cnt + 1
                  cycle
               end if
               if (mtx(i,j) == 0) cycle
               nonzero_cnt = nonzero_cnt + 1
            end do
         end do
         
         !call enlarge_integer_if_needed_1(ks(k)% ia, species+1, 10, ierr)
         if (associated(ks(k)% ia)) then
            if (size(ks(k)% ia,dim=1) < species+1) then
               deallocate(ks(k)% ia)
               allocate(ks(k)% ia(species + 10), stat=ierr)
            end if
         else 
            allocate(ks(k)% ia(species + 10), stat=ierr)
         end if
         if (do_fill_with_NaNs) ks(k)% ia = -9999999
         if (ierr /= 0) then
            write(*,2) 'allocate failed for ia', species+1
            stop
            return
         end if
         
         !call enlarge_integer_if_needed_1(ks(k)% ja, nonzero_cnt, 100, ierr)
         if (associated(ks(k)% ja)) then
            if (size(ks(k)% ja,dim=1) < nonzero_cnt) then
               deallocate(ks(k)% ja)
               allocate(ks(k)% ja(nonzero_cnt + 100), stat=ierr)
            end if
         else 
            allocate(ks(k)% ja(nonzero_cnt + 100), stat=ierr)
         end if
         if (do_fill_with_NaNs) ks(k)% ja = -9999999
         if (ierr /= 0) then
            write(*,2) 'allocate failed for ja', nonzero_cnt
            stop
            return
         end if
         
         !call enlarge_if_needed_1(ks(k)% values, nonzero_cnt, 100, ierr)
         if (associated(ks(k)% values)) then
            if (size(ks(k)% values,dim=1) < nonzero_cnt) then
               deallocate(ks(k)% values)
               allocate(ks(k)% values(nonzero_cnt + 100), stat=ierr)
            end if
         else 
            allocate(ks(k)% values(nonzero_cnt + 100), stat=ierr)
         end if
         if (do_fill_with_NaNs) call fill_with_NaNs(ks(k)% values)
         if (ierr /= 0) then
            write(*,2) 'allocate failed for values', nonzero_cnt
            stop
            return
         end if
         
         values => ks(k)% values
         
         ! compressed_format is compressed_col_sparse_0_based for KLU
         call dense_to_col_with_diag_0_based( &
            species, species, mtx, nonzero_cnt, sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, values, ierr)
         if (ierr /= 0) then
            write(*,*) 'solve burn sparse failed in converting from dense to sparse'
            return
         end if
         if (sprs_nonzeros /= nonzero_cnt) then
            write(*,*) &
               'solve burn sparse failed in converting dense to sparse: bad sprs_nonzeros'
            ierr = -1
            return
         end if
         ks(k)% sprs_nonzeros = sprs_nonzeros
      
      end subroutine sparse_get_matrix_info
      
         
      subroutine sparse_store_new_values(s, k, species, mtx, dbg, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         real(dp), pointer, intent(inout) :: mtx(:,:)
         logical, intent(in) :: dbg
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: values(:)
         integer, pointer :: colptr(:) ! (n+1)
         integer, pointer :: rowind(:) ! (nz)
         integer :: j, i, nz, c0, c1, c, sprs_nonzeros
         
         include 'formats'

         ierr = 0
         ks => s% burn_klu_storage
         values => ks(k)% values
         colptr => ks(k)% ia
         rowind => ks(k)% ja
         sprs_nonzeros = ks(k)% sprs_nonzeros
         nz = 0
         do j=1,species ! column index
            c0 = colptr(j)+1 ! 1st rowind
            c1 = colptr(j+1) ! last rowind
            do c=c0,c1
               i = rowind(c)+1 ! row index
               nz = nz+1
               if (nz > sprs_nonzeros) then
                  write(*,5) 'bad value for nz', nz, k, i, j
                  stop 'sparse_store_new_values'
                  ierr = -1
                  return
               end if
               if (i < 1 .or. i > species) then
                  write(*,5) 'bad value for i', nz, k, i, j
                  stop 'sparse_store_new_values'
                  ierr = -1
                  return
               end if
               if (dbg) write(*,5) 'values(nz) = mtx(i,j)', nz, i, j, k, mtx(i,j)
               values(nz) = mtx(i,j)
            end do
         end do
         if (nz /= sprs_nonzeros) then
            ierr = -1
            write(*,4) 'nz /= ks(k)% sprs_nonzeros', &
               k, nz, ks(k)% sprs_nonzeros
            stop 'sparse_store_new_values'
            return
         end if

      end subroutine sparse_store_new_values
         
         
      subroutine sparse_analyze_and_factor(s, k, species, mtx, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         
         include 'formats'

         ierr = 0
         ks => s% burn_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if

         call sparse_free_numeric(s, k, species, ierr)
         if (ierr /= 0) return
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_analyze = sprs_num_analyze + 1
            sprs_num_factor = sprs_num_factor + 1
            sprs_num_alloc_klu_factors = sprs_num_alloc_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if
         
         call klu_dble_decsols_nrhs_0_based( & ! klu_analyze and klu_factor
            0, species, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
            
         ks(k)% have_Numeric = .true.
         
         if (ierr /= 0) then
            write(*,2) 'do_klu_dble_decsols_nrhs_0_based failed factor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if
      
      end subroutine sparse_analyze_and_factor
         
         
      subroutine sparse_analyze(s, k, species, mtx, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         
         include 'formats'

         ierr = 0
         ks => s% burn_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_analyze = sprs_num_analyze + 1
            sprs_num_alloc_klu_factors = sprs_num_alloc_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if

         call klu_dble_decsols_nrhs_0_based( & ! klu_analyze
            -2, species, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
            
         if (ierr /= 0) then
            write(*,2) 'klu_dble_decsols_nrhs_0_based failed factor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if
      
      end subroutine sparse_analyze
         
         
      subroutine sparse_factor(s, k, species, mtx, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         
         include 'formats'

         ierr = 0
         ks => s% burn_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         
         call sparse_free_numeric(s, k, species, ierr)
         if (ierr /= 0) return
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_factor = sprs_num_factor + 1
!$omp end critical (sprs_statistics)
         end if
         
         call klu_dble_decsols_nrhs_0_based( & ! klu_factor
            -1, species, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
            
         ks(k)% have_Numeric = .true.
            
         if (ierr /= 0) then
            write(*,2) 'klu_dble_decsols_nrhs_0_based failed factor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if
      
      end subroutine sparse_factor
         
         
      subroutine sparse_refactor(s, k, species, mtx, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         ierr = 0
         ks => s% burn_klu_storage
         if (.not. ks(k)% have_Numeric) then
            write(*,3) 'called refactor without Numeric', k
            ierr = -1
            return
         end if
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_refactor = sprs_num_refactor + 1
!$omp end critical (sprs_statistics)
         end if

         call klu_dble_decsols_nrhs_0_based( & ! refactor
            3, species, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)      
      end subroutine sparse_refactor
         
         
      subroutine sparse_no_pivot(s, k, species, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         ierr = 0
         ks => s% burn_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         call klu_dble_decsols_nrhs_0_based( &
            -7, species, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)  
      end subroutine sparse_no_pivot
         
         
      real(dp) function sparse_rcond(s, k, species, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         ierr = 0
         ks => s% burn_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         call klu_dble_decsols_nrhs_0_based( & ! klu_rcond
            4, species, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)  
         sparse_rcond = rpar(1)    
      end function sparse_rcond
         
         
      real(dp) function sparse_rgrowth(s, k, species, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         ierr = 0
         ks => s% burn_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         call klu_dble_decsols_nrhs_0_based( & ! klu_rgrowth
            5, species, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)  
         sparse_rgrowth = rpar(1)    
      end function sparse_rgrowth
         
         
      real(dp) function sparse_condest(s, k, species, ierr)  
         ! accurate condition number estimate,
         ! but more expensive to compute than rgrowth
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         ierr = 0
         ks => s% burn_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b(1:1) => b_ary(1:1)
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         call klu_dble_decsols_nrhs_0_based( & ! klu_condest
            6, species, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)  
         sparse_condest = rpar(1)    
      end function sparse_condest


      subroutine sparse_solve(s, k, nrhs, species, b, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, nrhs, species
         real(dp), pointer :: b(:)
         integer, intent(out) :: ierr
         real(dp), pointer :: rpar(:)
         type(sparse_info), pointer :: ks(:)
         integer, pointer :: ipar(:)
         include 'formats'

         ks => s% burn_klu_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_solve = sprs_num_solve + 1
!$omp end critical (sprs_statistics)
         end if

         call klu_dble_decsols_nrhs_0_based( & ! klu_solve
            1, nrhs, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
            
         if (ierr /= 0) then
            write(*,2) 'do_klu_dble_decsols_nrhs_0_based failed refactor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if

      end subroutine sparse_solve


      subroutine sparse_free_all(s, species, k_min, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: species, k_min
         integer, intent(out) :: ierr
         
         integer :: k
         type(sparse_info), pointer :: ks(:)
         
         include 'formats'
         ierr = 0
         ks => s% burn_klu_storage
         do k = k_min, size(ks)
            if (ks(k)% sprs_nonzeros <= 0) cycle
            call sparse_free_numeric(s, k, species, ierr)
            if (ierr /= 0) exit
            call sparse_free_klu_factors(s, k, species, ierr)
            if (ierr /= 0) exit
            ks(k)% sprs_nonzeros = -1
         end do
      end subroutine sparse_free_all
      
      
      subroutine sparse_free_klu_factors(s, k, species, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         
         include 'formats'
         ierr = 0
         ks => s% burn_klu_storage
         b(1:1) => b_ary(1:1)
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol


         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_free_klu_factors = sprs_num_free_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if

         ! free klu factors array (this is also done by klu_free_symbolic)
         call klu_dble_decsols_nrhs_0_based( & 
            -5, 0, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
               
      end subroutine sparse_free_klu_factors
      
      
      subroutine sparse_free_symbolic(s, k, species, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         
         include 'formats'
         ierr = 0
         ks => s% burn_klu_storage
         b(1:1) => b_ary(1:1)
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_free_symbolic = sprs_num_free_symbolic + 1
            sprs_num_free_klu_factors = sprs_num_free_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if

         call klu_dble_decsols_nrhs_0_based( & 
            -4, 0, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
               
      end subroutine sparse_free_symbolic
      
      
      subroutine sparse_free_numeric(s, k, species, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)

         include 'formats'
         ierr = 0
         ks => s% burn_klu_storage
         if (.not. ks(k)% have_Numeric) return
         b(1:1) => b_ary(1:1)
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_free_numeric = sprs_num_free_numeric + 1
!$omp end critical (sprs_statistics)
         end if         
         call klu_dble_decsols_nrhs_0_based( & 
            -3, 0, species, ks(k)% sprs_nonzeros, &
            ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
            lrd, rpar, lid, ipar, ierr)
         ks(k)% have_Numeric = .false.
      end subroutine sparse_free_numeric
      
      
      subroutine clear_klu_storage(s, max_k_for_free_shared)
         type (star_info), pointer :: s
         integer, intent(in) :: max_k_for_free_shared
         integer :: k

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_clear_klu_storage = sprs_num_clear_klu_storage + 1
!$omp end critical (sprs_statistics)
         end if

         do k = 1, size(s% burn_klu_storage)
            if (k <= max_k_for_free_shared) then
               if (associated(s% burn_klu_storage(k)% ia)) &
                  deallocate(s% burn_klu_storage(k)% ia)
               if (associated(s% burn_klu_storage(k)% ja)) &
                  deallocate(s% burn_klu_storage(k)% ja)
            end if
            if (associated(s% burn_klu_storage(k)% values)) &
               deallocate(s% burn_klu_storage(k)% values)
         end do
         deallocate(s% burn_klu_storage)
         nullify(s% burn_klu_storage)
      end subroutine clear_klu_storage



      end module solve_burn_sparse


