! ***********************************************************************
!
!   Copyright (C) 2013  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module solve_burn
      use solve_burn_sparse

      use star_private_def
      use const_def
      use num_def

      implicit none
      
      
      contains

      
      integer function do_one_zone_burns( &
            s, dt_total, avg_mix_dxdt, avg_burn_dxdt, skip_partials, &
            pass, num_passes, k_bad)
         use net_lib, only: net_work_size
         use chem_def, only: chem_isos
         use nse_def, only: min_T_for_NSE

         type (star_info), pointer :: s
         real(dp), intent(in) :: dt_total
         real(dp), pointer, dimension(:,:), intent(in) :: &
            avg_mix_dxdt, avg_burn_dxdt
         logical, intent(in) :: skip_partials
         integer, intent(in) :: pass, num_passes
         integer, intent(out) :: k_bad

         integer :: net_lwork, ierr, max_num_iters_k, &
            j, k, k_nse, op_err, num_iters, species, max_num_iters_used
         real(dp) :: dlamch, sfmin, atol, rtol, sum_dxdt, sum_rates
         logical :: sparse, trace, dbg, okay, use_net_info_for_sparsity
         real(dp), parameter :: burn1_BE_alfa = 0d0
         logical, parameter :: burn1_BE_dbg = .false., &
            burn1_BE_doing_partials = .false.      
            
         real(dp) :: sfmin_dp
         
         include 'formats'
         
         do_one_zone_burns = keep_going
         if (dt_total <= 0d0) return

         trace = s% op_split_burn_trace   
         atol = s% op_split_burn_atol
         rtol = s% op_split_burn_rtol

         ierr = 0
         species = s% species
         sparse = (species >= s% op_split_burn_sparse_limit)
         use_net_info_for_sparsity = s% op_split_burn_use_net_info_for_sparsity
         
         sfmin = dlamch('S')  
         sfmin_dp = sfmin
         net_lwork = net_work_size(s% net_handle, ierr)
         if (ierr /= 0) then
            write(*,*) 'do_one_zone_burns failed in net_work_size'
            do_one_zone_burns = terminate
            s% termination_code = t_solve_burn
            return
         end if
         
         if (sparse) then
            call setup_for_sparse(ierr)
            if (ierr /= 0) then
               write(*,*) 'do_one_zone_burns failed in setup_for_sparse'
               do_one_zone_burns = terminate
               s% termination_code = t_solve_burn
               return
            end if
         end if            
         
         max_num_iters_used = 0
         max_num_iters_k = 0
         k_bad = 0
         
         dbg = .false. ! (s% model_number == 1137)
         if (dbg .or. trace) write(*,2) 'start do_one_zone_burns'   
                  
!$OMP PARALLEL DO PRIVATE(k,op_err,num_iters,j,sum_dxdt,sum_rates,okay) SCHEDULE(DYNAMIC,4)
         do k = 1, s% nz
         
            if (k_bad /= 0) cycle
            
            s% max_burn_correction(k) = 0d0
            
            do j=1,species
               s% xa(j,k) = s% xa_pre(j,k)
            end do
            
            op_err = 0
            if (s% T(k) >= s% T_NSE_full_on) then
               if (trace .or. .true.) write(*,2) 'call burn1_NSE', k
               call burn1_NSE(s, k, species, dt_total, op_err)
               num_iters = 0
               if (op_err /= 0 .and. trace) write(*,2) 'burn1_NSE failed', k
            else if (s% T(k) > s% T_NSE_full_off) then   
               if (trace .or. .true.) write(*,2) 'call burn1_partial_NSE', k
               call burn1_partial_NSE( &
                  s, k, species, net_lwork, sparse, sfmin_dp, dt_total, &
                  avg_mix_dxdt, num_iters, op_err)
               if (op_err /= 0 .and. trace) write(*,2) 'burn1_partial_NSE failed', k
            else
               call burn1_BE( &
                  s, k, species, net_lwork, sparse, sfmin_dp, dt_total, burn1_BE_alfa, &
                  avg_mix_dxdt, num_iters, burn1_BE_doing_partials, burn1_BE_dbg, op_err)                  
               if (op_err /= 0 .and. trace) then
                  write(*,2) 'burn1_BE failed', k
               end if
               if (num_iters > max_num_iters_used) then
                  max_num_iters_used = num_iters
                  max_num_iters_k = k
               end if         
            end if
            if (op_err /= 0) then
               ierr = -1
               k_bad = k
            end if
            
            do j=1,species
               avg_burn_dxdt(j,k) = &
                  (s% xa(j,k) - s% xa_pre(j,k))/dt_total - avg_mix_dxdt(j,k)
            end do
            
            if (.true.) then
               call revise_avg_burn_dxdt(s, k, species, avg_burn_dxdt, dt_total, op_err)
               if (op_err /= 0 .and. trace) then
                  write(*,2) 'burn1_BE failed', k
               end if
               if (op_err /= 0) then
                  ierr = -1
                  k_bad = k
               end if
            end if

         end do
!$OMP END PARALLEL DO

         if (max_num_iters_used > s% num_burn_max_iters) then
            s% num_burn_max_iters = max_num_iters_used
            !write(*,3) 'max_num_iters k used', max_num_iters_k, max_num_iters_used
         end if
            
         if (ierr == 0 .and. .not. skip_partials) then
            call get_partials( &
               s, species, net_lwork, sparse, sfmin_dp, dt_total, 0d0, &
               avg_mix_dxdt, num_iters, k_bad, ierr)
            if (ierr /= 0) then
               write(*,*) 'solve_burn failed in get_partials'
               do_one_zone_burns = retry
               return
            end if
         end if

         if (dbg .or. trace) write(*,2) 'done do_one_zone_burns'

         if (sparse .and. keep_sprs_statistics) then
            write(*,*)
            write(*,2) 'sprs_num_alloc_klu_storage', sprs_num_alloc_klu_storage
            write(*,2) 'sprs_num_clear_klu_storage', sprs_num_clear_klu_storage
            write(*,*)
            write(*,2) 'sprs_num_analyze', sprs_num_analyze
            write(*,2) 'sprs_num_free_symbolic', sprs_num_free_symbolic
            write(*,*)
            write(*,2) 'sprs_num_factor', sprs_num_factor
            write(*,2) 'sprs_num_free_numeric', sprs_num_free_numeric
            write(*,*)
            write(*,2) 'sprs_num_alloc_klu_factors', sprs_num_alloc_klu_factors
            write(*,2) 'sprs_num_free_klu_factors', sprs_num_free_klu_factors
            write(*,*)
            write(*,2) 'sprs_num_refactor', sprs_num_refactor
            write(*,2) 'sprs_num_solve', sprs_num_solve
            write(*,2) 'factor + refactor - solve', &
               sprs_num_factor + sprs_num_refactor - sprs_num_solve
            write(*,*)
         end if
         
         if (ierr /= 0) then
            do_one_zone_burns = retry
            return
         end if

         
         contains
         
         
         subroutine setup_for_sparse(ierr)
            integer, intent(out) :: ierr
         
            real(dp), target :: mtx_array(species*species)
            real(dp), pointer :: mtx(:,:)
            integer :: k, sprs_nonzeros
            type(sparse_info), pointer :: ks(:)
            
            include 'formats'
            
            ierr = 0
            mtx(1:species,1:species) => mtx_array(1:species*species)
            
            !write(*,*) 'setup_for_sparse'

            !write(*,*) 'call alloc_klu_storage'
            call alloc_klu_storage(s, ierr)
            !write(*,*) 'done alloc_klu_storage'
            if (ierr /= 0) return
            
            if (.not. use_net_info_for_sparsity) return 
            
            if (trim(s% burn_sprs_shared_net_name) == trim(s% net_name)) return
            
            if (len_trim(s% burn_sprs_shared_net_name) /= 0) then ! free old before get new
            
               !write(*,*) 'free old before get new'
               
               call sparse_free_symbolic(s, 1, species, ierr)
               if (ierr /= 0) return
               call sparse_free_numeric(s, 1, species, ierr)
               if (ierr /= 0) return
               call sparse_free_all(s, species, 2, ierr)
               if (ierr /= 0) return
            
               !write(*,*) 'done free old before get new'

            end if
            
            ! get nonzero pattern for net
            k = 1
            !write(*,*) 'call sparse_net_matrix_info'
            call sparse_net_matrix_info( &
               s, k, species, net_lwork, mtx, sprs_nonzeros, ierr)   
            if (ierr /= 0) then
               write(*,2) 'sparse_get_matrix failed', s% model_number
               stop 'do_one_zone_burns'
            end if
            !write(*,*) 'call sparse_analyze'
            call sparse_analyze(s, k, species, mtx, ierr)   
            if (ierr /= 0) then
               write(*,2) 'sparse_analyze failed', s% model_number
               stop 'do_one_zone_burns'
            end if
            
            ks => s% burn_klu_storage
            s% burn_sprs_shared_net_name = s% net_name
            s% burn_shared_sprs_nonzeros = sprs_nonzeros
            s% burn_sprs_shared_ia => ks(k)% ia
            s% burn_sprs_shared_ja => ks(k)% ja
            s% burn_sprs_shared_ipar8_decsol = ks(k)% ipar8_decsol

            
            !write(*,*) 'done setup_for_sparse'
         
         end subroutine setup_for_sparse
         
         
      end function do_one_zone_burns
      
      
      subroutine get_partials( &
            s, species, net_lwork, sparse, sfmin, dt, alfa, &
            avg_mix_dxdt, max_num_iters_used, k_bad, ierr)
         use alloc, only: get_integer_work_array, return_integer_work_array
         type (star_info), pointer :: s
         integer, intent(in) :: species, net_lwork
         logical, intent(in) :: sparse
         real(dp), intent(in) :: sfmin
         real(dp), intent(in) :: dt, alfa 
            ! alfa is fraction NSE for this T
         real(dp), pointer, intent(in) :: avg_mix_dxdt(:,:)
         integer, intent(out) :: max_num_iters_used, k_bad, ierr
         
         integer :: k, kk, op_err, j, jj, cnt, kmax, nz, num_iters
         real(dp) :: abs_e, abs_e_dm, abs_e_limit, &
            max_abs_e_dm, dm_limit, e_limit, lnT_max, min_lnT_for_partials
         real(dp), pointer, dimension(:) :: &
            xa1, net_dxdt_nuc1, net_d_epsnuc_dx1, &
            net_dxdt_dRho1, net_dxdt_dT1, net_d_dxdt_dx1
         real(dp), pointer, dimension(:,:) :: &
            xa, net_dxdt_nuc, net_d_epsnuc_dx, &
            net_dxdt_dRho, net_dxdt_dT
         real(dp), pointer, dimension(:,:,:) :: net_d_dxdt_dx
         integer, pointer :: ks(:)
         logical :: okay
      
         include 'formats'
         
         k_bad = 0
         max_num_iters_used = 0
         ierr = 0
         
         min_lnT_for_partials = ln10*s% burn_min_logT_for_finite_diff_partials
         nz = s% nz
         k = 1
         kmax = k
         abs_e = abs(s% eps_nuc(k))
         max_abs_e_dm = abs_e*s% dm(k)
         lnT_max = s% lnT(k)
         do k=2, nz
            abs_e = abs(s% eps_nuc(k))
            abs_e_dm = abs_e*s% dm(k)
            if (abs_e_dm > max_abs_e_dm) then
               kmax = k
               max_abs_e_dm = abs_e_dm
            end if
            if (s% lnT(k) > lnT_max) lnT_max = s% lnT(k)
         end do
         if (lnT_max < min_lnT_for_partials) return
         
         k = kmax
         abs_e = abs(s% eps_nuc(k))
         abs_e_limit = 1d-3*abs_e
         e_limit = 1d-5*s% mstar/Msun
         dm_limit = 0d0 ! 1d-4*s% L_phot*Lsun/max(1d0,abs_e)
      
         call get_integer_work_array(s, ks, nz, 0, ierr)
         if (ierr /= 0) return            
         cnt = 0
         do k=1,nz
            ks(k) = 0
            if (s% lnT(k) < min_lnT_for_partials) cycle
            if (s% dm(k) < dm_limit) cycle
            if (s% eps_nuc(k) < e_limit) cycle
            abs_e = abs(s% eps_nuc(k))
            if (abs_e < abs_e_limit) cycle
            if (s% T(k) > s% T_NSE_full_off) cycle ! DO NOT DO THIS FOR NSE AT PRESENT
            
            ! DO ALL THE REST TO CENTER
            do kk=k,nz
               ks(kk) = 1
               cnt = cnt+1
            end do
            exit
            
         end do
         if (cnt == 0) then
            call return_integer_work_array(s, ks)
            return
         end if
         
         !write(*,3) 'cnt for burn partials', s% model_number, cnt
         
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         okay = .true.
!$OMP PARALLEL DO PRIVATE(j,jj,k,num_iters,op_err)
         do j=1,cnt
            jj = 0
            do k = 1,nz
               if (.not. okay) cycle
               if (ks(k) /= 0) jj = jj+1
               if (jj < j) cycle
               op_err = 0
               call do1_burn_partials( &
                  s, k, species, net_lwork, sparse, sfmin, dt, 0d0, &
                  avg_mix_dxdt, &
                  net_dxdt_nuc(:,j), net_dxdt_dRho(:,j), net_dxdt_dT(:,j), &
                  net_d_epsnuc_dx(:,j), xa(:,j), net_d_dxdt_dx(:,:,j), &
                  num_iters, op_err)                  
               if (op_err /= 0) then
                  okay = .false.
                  k_bad = k
               end if
               max_num_iters_used = max(num_iters, max_num_iters_used)            
               exit
            end do
         end do
!$OMP END PARALLEL DO

         if (.not. okay) ierr = -1

         call dealloc
      
      
         contains
         
         
         subroutine do_alloc(ierr)
            use alloc
            integer, intent(out) :: ierr
            
            ierr = 0
            call non_crit_get_work_array(s, xa1, species*cnt, 0, 'get_partials', ierr)
            if (ierr /= 0) return            
            xa(1:species,1:cnt) => xa1(1:species*cnt)   
                  
            call non_crit_get_work_array(s, net_dxdt_nuc1, species*cnt, 0, 'get_partials', ierr)
            if (ierr /= 0) return 
            net_dxdt_nuc(1:species,1:cnt) => net_dxdt_nuc1(1:species*cnt)   
                       
            call non_crit_get_work_array(s, net_d_epsnuc_dx1, species*cnt, 0, 'get_partials', ierr)
            if (ierr /= 0) return
            net_d_epsnuc_dx(1:species,1:cnt) => net_d_epsnuc_dx1(1:species*cnt)   
                        
            call non_crit_get_work_array(s, net_dxdt_dRho1, species*cnt, 0, 'get_partials', ierr)
            if (ierr /= 0) return
            net_dxdt_dRho(1:species,1:cnt) => net_dxdt_dRho1(1:species*cnt)   
                        
            call non_crit_get_work_array(s, net_dxdt_dT1, species*cnt, 0, 'get_partials', ierr)
            if (ierr /= 0) return 
            net_dxdt_dT(1:species,1:cnt) => net_dxdt_dT1(1:species*cnt)   
                       
            call non_crit_get_work_array(s, net_d_dxdt_dx1, species**2*cnt, 0, 'get_partials', ierr)
            if (ierr /= 0) return
            net_d_dxdt_dx(1:species,1:species,1:cnt) => net_d_dxdt_dx1(1:species**2*cnt) 
                    
         end subroutine do_alloc

         
         subroutine dealloc
            use alloc
            call non_crit_return_work_array(s, xa1, 'get_partials')
            call non_crit_return_work_array(s, net_dxdt_nuc1, 'get_partials')
            call non_crit_return_work_array(s, net_d_epsnuc_dx1, 'get_partials')
            call non_crit_return_work_array(s, net_dxdt_dRho1, 'get_partials')
            call non_crit_return_work_array(s, net_dxdt_dT1, 'get_partials')
            call non_crit_return_work_array(s, net_d_dxdt_dx1, 'get_partials')
            call return_integer_work_array(s, ks)
         end subroutine dealloc

      
      end subroutine get_partials
      
         
      ! use finite differences to set d_epsnuc_dlnd and d_epsnuc_dlnT
      subroutine do1_burn_partials( &
            s, k, species, net_lwork, sparse, sfmin, dt, alfa, &
            avg_mix_dxdt, &
            net_dxdt_nuc, net_dxdt_dRho, net_dxdt_dT, &
            net_d_epsnuc_dx, xa, net_d_dxdt_dx, &
            num_iters_out, ierr)
         use net, only: show_stuff
         use chem_lib, only: basic_composition_info
         type (star_info), pointer :: s
         integer, intent(in) :: k, species, net_lwork
         logical, intent(in) :: sparse
         real(dp), intent(in) :: sfmin
         real(dp), intent(in) :: dt, alfa 
            ! alfa is fraction NSE for this T
         real(dp), pointer, intent(in) :: avg_mix_dxdt(:,:)
         real(dp), dimension(:) :: & ! work arrays
            net_dxdt_nuc, net_dxdt_dRho, net_dxdt_dT, &
            net_d_epsnuc_dx, xa
         real(dp), dimension(:,:) :: net_d_dxdt_dx
         integer, intent(out) :: num_iters_out, ierr
         
         integer :: num_iters
         real(dp) :: &
            T, lnT, rho, lnd, sumx, &
            net_X, net_Y, net_z2bar, net_ye, net_mass_correction, &
            net_eps_nuc, net_eps_nuc_neu_total, net_abar, net_zbar, &
            d_epsnuc_dlnd, d_abar_dlnd, d_zbar_dlnd, &
            d_epsnuc_dlnT, d_abar_dlnT, d_zbar_dlnT, dlnT, dlnd
         
         logical, parameter :: switch_sign = .false.

         include 'formats'
         
         ierr = 0
         
         call basic_composition_info( &
            species, s% chem_id, s% xa(1:species,k), s% X(k), s% Y(k), &
            s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
            s% mass_correction(k), sumx)

         !write(*,1) 's% op_split_partials_dlogRho', s% op_split_partials_dlogRho
         !write(*,1) 's% op_split_partials_dlogT', s% op_split_partials_dlogT
         
         call save_burn_info
         
         dlnd = ln10*s% op_split_partials_dlogRho
         if (switch_sign .and. s% lnd(k) < s% lnd_start(k)) dlnd = -dlnd
         s% lnd(k) = s% lnd(k) + dlnd
         s% rho(k) = exp(s% lnd(k))
         call burn1_BE( &
            s, k, species, net_lwork, sparse, sfmin, dt, alfa, &
            avg_mix_dxdt, num_iters, .true., .false., ierr)
         num_iters_out = num_iters
         if (ierr /= 0) then
            write(*,2) 'failed in burn1_BE for dlnd partials', k
            !stop 'do1_burn_partials'
            return
         end if

         d_epsnuc_dlnd = (s% eps_nuc(k) - net_eps_nuc)/dlnd
         d_abar_dlnd = (s% abar(k) - net_abar)/dlnd
         d_zbar_dlnd = (s% zbar(k) - net_zbar)/dlnd
         
         call restore_burn_info
         
         dlnT = ln10*s% op_split_partials_dlogT
         if (switch_sign .and. s% lnT(k) < s% lnT_start(k)) dlnT = -dlnT
         s% lnT(k) = s% lnT(k) + dlnT
         s% T(k) = exp(s% lnT(k))
         call burn1_BE( &
            s, k, species, net_lwork, sparse, sfmin, dt, alfa, &
            avg_mix_dxdt, num_iters, .true., .false., ierr)
         num_iters_out = max(num_iters, num_iters_out)
         if (ierr /= 0) then
            write(*,2) 'failed in burn1_BE for dlnT partials', k
            !stop 'do1_burn_partials'
            return
         end if
            
         d_epsnuc_dlnT = (s% eps_nuc(k) - net_eps_nuc)/dlnT
         d_abar_dlnT = (s% abar(k) - net_abar)/dlnT
         d_zbar_dlnT = (s% zbar(k) - net_zbar)/dlnT
         
         call restore_burn_info
         
         s% d_epsnuc_dlnT(k) = d_epsnuc_dlnT
         s% d_epsnuc_dlnd(k) = d_epsnuc_dlnd
         s% d_abar_dlnT(k) = d_abar_dlnT
         s% d_zbar_dlnT(k) = d_zbar_dlnT
         s% d_abar_dlnd(k) = d_abar_dlnd
         s% d_zbar_dlnd(k) = d_zbar_dlnd
         
         
         contains
         
         
         subroutine save_burn_info
            integer :: j, i
            
            T = s% T(k)
            lnT = s% lnT(k)
            rho = s% rho(k)
            lnd = s% lnd(k)
            net_eps_nuc = s% eps_nuc(k)
            net_eps_nuc_neu_total = s% eps_nuc_neu_total(k)
            
            net_X = s% X(k)
            net_Y = s% Y(k)
            net_abar = s% abar(k)
            net_zbar = s% zbar(k)
            net_z2bar = s% z2bar(k)
            net_ye = s% ye(k)
            net_mass_correction = s% mass_correction(k)

            do j=1,species
               xa(j) = s% xa(j,k)
               net_d_epsnuc_dx(j) = s% d_epsnuc_dx(j,k)
               net_dxdt_nuc(j) = s% dxdt_nuc(j,k)
               net_dxdt_dRho(j) = s% dxdt_dRho(j,k)
               net_dxdt_dT(j) = s% dxdt_dT(j,k)
               do i=1,species
                  net_d_dxdt_dx(i,j) = s% d_dxdt_dx(i,j,k)
               end do
            end do
            
         end subroutine save_burn_info
         
         
         subroutine restore_burn_info
            integer :: j, i
            
            s% T(k) = T
            s% lnT(k) = lnT
            s% rho(k) = rho
            s% lnd(k) = lnd
            s% eps_nuc(k) = net_eps_nuc
            s% eps_nuc_neu_total(k) = net_eps_nuc_neu_total
            
            s% X(k) = net_X
            s% Y(k) = net_Y
            s% abar(k) = net_abar
            s% zbar(k) = net_zbar
            s% z2bar(k) = net_z2bar
            s% ye(k) = net_ye
            s% mass_correction(k) = net_mass_correction
            
            do j=1,species
               s% xa(j,k) = xa(j)
               s% d_epsnuc_dx(j,k) = net_d_epsnuc_dx(j)
               s% dxdt_nuc(j,k) = net_dxdt_nuc(j)
               s% dxdt_dRho(j,k) = net_dxdt_dRho(j)
               s% dxdt_dT(j,k) = net_dxdt_dT(j)
               do i=1,species
                  s% d_dxdt_dx(i,j,k) = net_d_dxdt_dx(i,j)
               end do
            end do
         
            call basic_composition_info( &
               species, s% chem_id, s% xa(1:species,k), s% X(k), s% Y(k), &
               s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
               s% mass_correction(k), sumx)
            
         end subroutine restore_burn_info

         
      end subroutine do1_burn_partials
      
         
      subroutine burn1_BE( &
            s, k, species, net_lwork, sparse, sfmin, dt, alfa, &
            avg_mix_dxdt, num_iters, doing_partials, dbg_in, ierr)

         use eos_def
         use chem_def, only: chem_isos
         use chem_lib, only: composition_info
         use neu, only: do_neu_for_cell
         use net, only: do1_net
         use utils_lib, only: is_bad_num
         
         type (star_info), pointer :: s
         integer, intent(in) :: k, species, net_lwork
         logical, intent(in) :: sparse
         real(dp), intent(in) :: sfmin
         real(dp), intent(in) :: dt, alfa 
            ! alfa is fraction NSE for this T
         real(dp), pointer, intent(in) :: avg_mix_dxdt(:,:)
         logical, intent(in) :: doing_partials, dbg_in
         integer, intent(out) :: num_iters, ierr
         
         logical :: did_refactor, keep_T_fixed, &
            use_net_info_for_sparsity, dbg, use_pivoting, unchanged
         real(dp), target, dimension(species+1) :: &
            del_array, x0_array, x1_array, dx_array
         real(dp), target, dimension((species+1)*(species+1)) :: mtx_array
         real(dp), pointer, dimension(:,:) :: mtx
         real(dp), pointer, dimension(:) :: del, x0, x1, dx
         real(dp), parameter :: one = 1, zero = 0
         real(dp) :: &
            atol, max_resid, lambda, min_lambda, xsum, dx_dlnd, dx_dlnT, &
            tmp1, tmp2, tmp3, remaining_time, time, time_end
         real(dp) :: rgrowth, condest
         integer :: i, j, nz, maxiters, sprs_nonzeros, nonzeros
         integer, target :: ipivot_array(species+1)
         integer, pointer :: ipivot(:)
         type(sparse_info), pointer :: ks(:)

         real(dp) :: sumx
         real(dp), dimension(species) :: &
            xa_init, dabar_dx, dzbar_dx, dmc_dx
            
         integer, pointer :: chem_id(:)
         
         
         ! for lnT eqn
         integer :: i_dlnT, equ1, nvar 
         real(dp) :: &
            T, lnT, dlnT, undercorrection_factor, &
            xh, xhe, Cv, d_dlnTdt_dlnT, d_dlnTdt_dx, d_dxdt_dlnT, &
            lnT_start, T_start, lnT_scale, T_init, lnT_init, &
            d_dEdY_dlnT, d_del_dlnT, d_del_dx, d_dTdt_dlnT, d_dYdt_dx, &
            dL_dm, dT_dt, eps_fixed, rho, dE_dV, dV_dt, P, &
            mu, mu_start, dmu_dt, dE_dmu, d_dEdmu_dlnT, lnT_eqn_scale, &
            dmu_dx, abar, zbar, d_dEdmu_dmu, d_dEdmu_dx, d_dmu_dt_dx, &
            d_dmu_dt_dlnT, dabar_dlnT, dzbar_dlnT, dmu_dlnT
            
         real(dp) :: dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas
         real(dp) :: dlnRho_dlnT_const_P, d_epsnuc_dlnT_const_P, d_Cp_dlnT
         real(dp) :: res(num_eos_basic_results)
         real(dp) :: d_dlnRho_const_T(num_eos_basic_results) 
         real(dp) :: d_dlnT_const_Rho(num_eos_basic_results)
         real(dp), dimension(species) :: d_dxdt_dRho, d_dxdt_dT
         real(dp) :: d_dxdt_dx(species,species)
         

         include 'formats'
         
         ierr = 0    
         nz = s% nz     
         chem_id => s% chem_id
         
         dbg = .false.
         
         if (dbg) then
            write(*,*)
            write(*,2) 'enter burn', k
            do j=1,species
               if (s% xa(j,k) < 0d0) &
                  write(*,3) trim(chem_isos% name(s% chem_id(j))), &
                     j, k, s% xa(j,k)
            end do
            write(*,*)
         end if
         
         do i=1,species
            xa_init(i) = s% xa(i,k)
         end do
         unchanged = .true.
         
         
         ! for lnT eqn

         ! to avoid compiler warnings
         dmu_dt = 0 
         dE_dmu = 0
         dT_dt = 0
         dlnT = 0
         lnT_eqn_scale = 0

         T = s% T(k)
         lnT = s% lnT(k)
         T_init = T
         lnT_init = lnT
         
         abar = s% abar(k)
         zbar = s% zbar(k)
         mu_start = abar/(1 + zbar)
            ! DON'T use s% mu from eos
            ! assume complete ionization so don't have to call eos to update
         
         lnT_start = s% lnT_start(k)
         T_start = exp(lnT_start)
         lnT_scale = lnT_start
         if (k < nz) then
            dL_dm = (s% L(k) - s% L(k+1))/s% dm(k)
         else
            dL_dm = (s% L(k) - s% L_center)/s% dm(k)
         end if


         ! NOTE: cannot use extra_heat or irradiation_heat at this point
         ! since haven't been evaluated during normal sequence of execution.
         

         keep_T_fixed = .false.
         !if (dL_dm < 0) then
         !   keep_T_fixed = .true.
         !else 
         if (lnT_start/ln10 <= s% op_split_burn_logT_eqn_limit) & !then
            keep_T_fixed = .true.
         !else if (s% mixing_type(k) == convective_mixing) then
         !   keep_T_fixed = .true.
         !else if (k < nz) then
         !   if (s% mixing_type(k+1) == convective_mixing) &
         !      keep_T_fixed = .true.
         !end if
         !keep_T_fixed = .true.

         if (keep_T_fixed) then
            nvar = species        
            i_dlnT = 0
         else
            nvar = species + 1         
            i_dlnT = species + 1
            
            
            
            
            if (.not. associated(s% newton_dx)) then
               stop 'solve_burn -- needs newton_dx'
            end if
            
            if (dt <= 0d0) then
               stop 'solve_burn -- needs dt > 0'
            end if
            
            !if (.not. doing_partials) write(*,2) 'do lnT eqn with burn', k
            
         end if
         equ1 = i_dlnT

         rho = 0
         dE_dV = 0
         dV_dt = 0
         P = 0
         Cv = 0
         eps_fixed = dL_dm

         call composition_info( &
            species, chem_id, s% xa(1:species,k), s% X(k), s% Y(k), &
            s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
            s% mass_correction(k), sumx, dabar_dx, dzbar_dx, dmc_dx)  

         if (.not. keep_T_fixed) then
             
            if (s% lnPgas_flag) then
               stop 'solve burn lnT eqn relies on lnd as var'
            end if
            
            rho = s% rho(k)
            dV_dt = -s% dlnd_dt(k)/rho
            P = s% P(k)
            Cv = s% Cv(k) ! dE/dT at constant density&abundances
            dE_dV = -s% dE_dRho(k)*rho**2
            eps_fixed = eps_fixed + (dE_dV + P)*dV_dt            
            lnT_eqn_scale = dt/(Cv*T_init) ! constant to make d_del_dlnT order 1
         
            call do_neu_for_cell(s,k,ierr)
            if (ierr /= 0) then
               if (.true.) then
                  if (dbg) then
                     write(*,3) &
                        'do_neu_for_cell failed for burn', s% model_number, k
                     stop 'burn'
                     call show_stuff('do1_net failed')
                  end if
               end if
               return
            end if
            
            eps_fixed = eps_fixed + s% non_nuc_neu(k)
            
            if (dbg) then
               write(*,2) 'rho', k, rho
               write(*,2) 'dE_dV', k, dE_dV
               write(*,2) 'dV_dt', k, dV_dt
               write(*,2) 'P', k, P
               write(*,2) 'eps_fixed', k, eps_fixed
               write(*,2) 'dL_dm', k, dL_dm
            end if
            
         end if

         call do1_net( &
            s, k, species, s% num_reactions, net_lwork, dt, ierr)
         if (ierr /= 0) return
                  
         ks => s% burn_klu_storage
         ipivot => ipivot_array
         del => del_array
         x0 => x0_array
         x1 => x1_array
         dx => dx_array
         mtx(1:nvar,1:nvar) => mtx_array(1:nvar*nvar)

         use_net_info_for_sparsity = &
            s% op_split_burn_use_net_info_for_sparsity
         if (sparse .and. use_net_info_for_sparsity) then
            call sparse_setup_shared(s, k, 1, species, ierr)
            if (ierr /= 0) return
         end if

         num_iters = 0         
         maxiters = s% op_split_burn_max_iterations
         use_pivoting = s% op_split_burn_use_pivoting
         min_lambda = s% op_split_burn_min_lambda
         
         if (sparse .and. .not. use_pivoting) then
            call sparse_no_pivot(s, k, species, ierr) 
            if (ierr /= 0) return
         end if
         
         if (alfa >= 1d0) then
            ierr = -1
            return
         end if
         
         if (s% T(k) < s% op_split_burn_atol_limit_T_lo) then
            atol = s% op_split_burn_atol
         else if (s% T(k) >= s% op_split_burn_atol_limit_T_hi) then
            atol = s% op_split_burn_atol_for_T_hi
         else
            atol = s% op_split_burn_atol_for_T_lo + &
                  (s% op_split_burn_atol_for_T_hi - &
                        s% op_split_burn_atol_for_T_lo)* &
                   (s% T(k) - s% op_split_burn_atol_limit_T_lo) / &
                     (s% op_split_burn_atol_limit_T_hi - &
                        s% op_split_burn_atol_limit_T_lo)
         end if
         !if (doing_partials) atol = atol*10
      
         do j=1,species
            x0(j) = s% xa_pre(j,k)
            x1(j) = s% xa(j,k)
            dx(j) = x1(j) - x0(j)
         end do

         if (.not. keep_T_fixed) then
            j = i_dlnT
            dlnT = s% newton_dx(s% i_lnT,k)
            x0(j) = dlnT/lnT_scale
            dx(j) = zero
            x1(j) = x0(j)
            
            if (dbg) write(*,2) 'initial dlnT', k, dlnT
            
         else
            dlnT = 0
         end if
         
         if (dbg) write(*,*) 
                  
         ! solve ODEs for abundances and DAEs for constaints
         
         ! for ODEs: dx/dt = f(x)
            ! dx = cumulative change to x from all iterations (starts = 0)
            ! if had perfect dx, then dx = dt*f(x+dx)
            ! iterate to make this hold to close approximation.
            ! solve (I - dt*J)*del = dt*f(x+dx) - dx, J = df/dx(x+dx)
            ! then set dx = dx + del and repeat until del is small enough
            
         ! for DAEs: 0 = g(x)
            ! solve -J*del = g(x+dx), J = dg/dx(x+dx)
            ! then set dx = dx + del and repeat until del is small enough
            
         solve_loop: do num_iters = 1, maxiters

            do j=1,species
               tmp1 = s% dxdt_nuc(j,k) + avg_mix_dxdt(j,k)
               tmp2 = dx(j)
               del(j) = dt*tmp1 - tmp2 ! residual
               do i=1,species
                  tmp1 = s% d_dxdt_dx(j,i,k)
                  mtx(j,i) = -dt*tmp1
               end do
               mtx(j,j) = one + mtx(j,j)
            end do

            if (.not. keep_T_fixed) then
               
               ! energy conservation for cell k
               ! dL/dm = eps_nuc - non_nuc_neu - (dE_dt + P*dV_dt)
               ! dE_dt = dE_dV*dV_dt + dE_dT*dT_dt + dE_dmu*dmu_dt
               ! V = 1/rho
               ! 0 = eps_nuc - eps_fixed - dE_dT*dT_dt - dE_dmu*dmu_dt
               ! where eps_fixed = dL_dm + (dE_dV + P)*dV_dt + non_nuc_neu

               j = equ1

               abar = s% abar(k)
               zbar = s% zbar(k)
               mu = abar/(1 + zbar) ! assume complete ionization
               dmu_dt = (mu - mu_start)/dt
               dE_dmu = -1.5d0*cgas*T/mu**2
               dT_dt = T*dlnT/dt
               del(j) = (s% eps_nuc(k) - eps_fixed - Cv*dT_dt - dE_dmu*dmu_dt)*lnT_eqn_scale
                              
               d_dEdmu_dmu = -2*dE_dmu/mu
               dabar_dlnT = 0
               dzbar_dlnT = 0
               do i=1,species
                  dx_dlnT = dt*s% dxdt_dT(i,k)*T
                  dabar_dlnT = dabar_dlnT + dabar_dx(i)*dx_dlnT
                  dzbar_dlnT = dzbar_dlnT + dzbar_dx(i)*dx_dlnT
                  dmu_dx = ((1 + zbar)*dabar_dx(i) - abar*dzbar_dx(i))/(1 + zbar)**2
                  d_dEdmu_dx = d_dEdmu_dmu*dmu_dx
                  d_dmu_dt_dx = dmu_dx/dt
                  d_del_dx = (s% d_epsnuc_dx(i,k) &
                     - dE_dmu*d_dmu_dt_dx - d_dEdmu_dx*dmu_dt)*lnT_eqn_scale
                  mtx(j,i) = -d_del_dx
                  mtx(i,j) = -dx_dlnT*lnT_scale
               end do
               
               dmu_dlnT = ((1 + zbar)*dabar_dlnT - abar*dzbar_dlnT)/(1 + zbar)**2
               d_dEdmu_dlnT = dE_dmu + d_dEdmu_dmu*dmu_dlnT
               d_dmu_dt_dlnT = dmu_dlnT/dt
               d_dTdt_dlnT = T*(1d0 + dlnT)/dt
               d_del_dlnT = (s% d_epsnuc_dlnT(k) &
                  - Cv*d_dTdt_dlnT &
                  - d_dEdmu_dlnT*dmu_dt &
                  - dE_dmu*d_dmu_dt_dlnT)*lnT_eqn_scale
               mtx(j,j) = -d_del_dlnT*lnT_scale
               
               if (.false.) then
                  write(*,3) 'lnT eqn resid', num_iters, k, del(j)
                  write(*,3) 'dlogT', num_iters, k, &
                     (lnT - lnT_start)/ln10, lnT/ln10, lnT_start/ln10
                  write(*,3) 'Cv', num_iters, k, Cv
                  write(*,3) 'dt', num_iters, k, dt
                  write(*,3) 'T', num_iters, k, T
                  write(*,3) 's% eps_nuc(k)', num_iters, k, s% eps_nuc(k)
                  write(*,3) 's% d_epsnuc_dlnT(k)', num_iters, k, s% d_epsnuc_dlnT(k)
                  write(*,3) 'Cv*dT_dt', num_iters, k, Cv*dT_dt
                  write(*,3) 'abar', num_iters, k, s% abar(k)
                  write(*,3) 'eps_fixed', num_iters, k, eps_fixed
                  write(*,3) 'dL_dm', num_iters, k, dL_dm
                  write(*,3) 'lnT_scale', num_iters, k, lnT_scale
                  write(*,*)
                  if (.false.) then
                     do i=1,species
                        write(*,3) 'dabar_dx(i) for k', i, k, dabar_dx(i)
                        write(*,3) 'dY_dx(i) for k', i, k, s% d_epsnuc_dx(i,k)
                        write(*,3) 'dxdt_dT(i) for k', i, k, s% dxdt_dT(i,k)
                     end do
                     do j=1,nvar
                        write(*,3) 'rhs del(j) for k', j, k, del(j)
                        do i=1,nvar
                           write(*,4) 'mtx(i,j) for k', i, j, k, mtx(i,j)
                        end do
                     end do
                  end if
               end if
               
            end if

            if (dbg) then
               j = maxloc(abs(del(1:species)), dim=1)
               !write(*,3) 'max x resid ' // trim(chem_isos% name(s% chem_id(j))), &
               !   num_iters, k, del(j), x1(j)
               if (.not. keep_T_fixed) then ! apply correction for lnT
                  j = i_dlnT
                  if (dbg) write(*,3) 'resid lnT', num_iters, k, del(j)
                  !write(*,3) 'd_del_dlnT', num_iters, k, d_del_dlnT
               end if
            end if
         
            ! solve mtx*del = rhs (rhs is in del at start)
            if (.not. sparse) then
         
               call getf2( &
                  nvar, mtx, nvar, ipivot, sfmin, use_pivoting, ierr)    
               if (ierr /= 0) then
                  write(*,3) 'getf2 failed', k, s% model_number
                  return
               end if
            
               call getrs_1( &
                  nvar, mtx, nvar, ipivot, del, species, use_pivoting, ierr)
               if (ierr /= 0) then
                  write(*,3) 'burn_getrs failed', k, s% model_number
                  return
               end if
            
            else ! sparse
         
               if (use_net_info_for_sparsity) then
                  if (dbg) write(*,3) 'sparse_store_new_values', k, s% model_number
                  call sparse_store_new_values(s, k, species, mtx, .false., ierr)                  
                  if (ierr /= 0) then
                     write(*,3) 'sparse_store_new_values failed', k, s% model_number
                     stop 'burn1_BE'
                  end if
               
                  did_refactor = .false.
                  if (num_iters > 1 .and. ks(k)% have_Numeric) then ! try refactor
                     rgrowth = 0
                     call sparse_refactor(s, k, species, mtx, ierr) 
                     if (ierr == 0) rgrowth = sparse_rgrowth(s, k, species, ierr)
                     if (ierr /= 0 .or. rgrowth < 1d-4) then
                        ierr = 0
                     else
                        did_refactor = .true.
                     end if
                  end if
                  if (.not. did_refactor) then
                     call sparse_factor(s, k, species, mtx, ierr)  
                     if (ierr /= 0) then
                        if (dbg) then
                           write(*,3) 'sparse_factor failed', k, s% model_number
                        end if
                        return
                        stop 'burn1_BE'
                     end if
                  
                     if (.false. .and. num_iters == 1) then
                        condest = sparse_condest(s, k, species, ierr)
                        write(*,4) 'BE condest', &
                           k, num_iters, s% model_number, condest
                     end if

                  end if
               else
                  call sparse_get_matrix_info(s, k, species, mtx, nonzeros, ierr) 
                  if (ierr /= 0) then
                     write(*,3) 'sparse_get_matrix_info failed', k, s% model_number
                     stop 'burn1_BE'
                  end if
                  call sparse_analyze_and_factor(s, k, species, mtx, ierr)   
                  if (ierr /= 0) then
                     write(*,3) 'sparse_analyze_and_factor failed', k, s% model_number
                     stop 'burn1_BE'
                  end if
               end if
                           
               call sparse_solve(s, k, 1, species, del, ierr)
               if (ierr /= 0) then
                  write(*,3) 'sparse_solve failed', k, s% model_number
                  stop 'burn1_BE'
               end if
            
            end if
         
            ! set lambda for positivity
            call positivity(s, k, species, x1, del, lambda, 'BE', num_iters, dbg)
                     
            if (lambda < min_lambda) then
               lambda = min_lambda
               !if (dbg) write(*,3) 'min lambda', num_iters, k, lambda
            end if
               
            if (.not. keep_T_fixed) then ! check change in dlnT
               dlnT = (x0(j) + dx(j) + lambda*del(j))*lnT_scale
               if (abs(dlnT) > 1d-3*lnT) then
                  lambda = max(min_lambda, lambda*1d-3*lnT/abs(dlnT))
                  write(*,3) 'reduce lambda for dlnT', num_iters, k, lambda
               end if
            end if
            
            if (.not. keep_T_fixed) then ! pathetic attempt to tame stiffness
               undercorrection_factor = min(lambda,0.8)
            else
               undercorrection_factor = lambda
            end if
            
            do j=1,species ! apply the correction for abundances
               dx(j) = dx(j) + del(j)*undercorrection_factor
               x1(j) = x0(j) + dx(j)
               s% xa(j,k) = x1(j)
               if (abs(s% xa(j,k) - xa_init(j)) > 1d-14) unchanged = .false.
            end do
               
            if (.not. keep_T_fixed) then ! apply correction for lnT
               j = i_dlnT
               if (dbg) write(*,3) 'del lnT', num_iters, k, del(j), lambda*del(j)*lnT_scale
               dx(j) = dx(j) + del(j)*undercorrection_factor
               x1(j) = x0(j) + dx(j)
               dlnT = x1(j)*lnT_scale
               lnT = dlnT + lnT_start
               T = exp(lnT)
               s% T(k) = T
               s% lnT(k) = lnT
            end if

            if (dbg) then
               j = maxloc(abs(del(1:species)), dim=1)
               write(*,3) 'max x del ' // trim(chem_isos% name(s% chem_id(j))), &
                  num_iters, k, del(j), x1(j), x0(j), dx(j)
               !if (num_iters == maxiters) call show_stuff('num_iters == maxiters')
               !do j=1,species
               !   write(*,3) 'result del(j) for k', j, k, del(j)
               !end do
            end if
            
            if (.false. .and. lambda < 1) then
               do j=1,species
                  if (x1(j) > -1d-8) cycle
                  if (dbg) write(*,3) trim(chem_isos% name(s% chem_id(j))), &
                     j, k, s% xa(j,k), dt*avg_mix_dxdt(j,k)
                  if (avg_mix_dxdt(j,k) < 0d0) then
                     if (dbg) write(*,3,advance='no') 'reduce dt*avg_mix_dxdt(j,k) ' // &
                        trim(chem_isos% name(s% chem_id(j))), j, k, dt*avg_mix_dxdt(j,k)
                     avg_mix_dxdt(j,k) = &
                        min(0.1d0*avg_mix_dxdt(j,k), avg_mix_dxdt(j,k) - 2*x1(j)/dt)
                     if (dbg) write(*,'(99(1pe26.16))') dt*avg_mix_dxdt(j,k), x1(j)
                  end if
               end do
            end if
            
            if (.not. unchanged) then
            
               call composition_info( &
                  species, chem_id, s% xa(1:species,k), s% X(k), s% Y(k), &
                  s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
                  s% mass_correction(k), sumx, dabar_dx, dzbar_dx, dmc_dx)  
                  
               do j=1,species
                  s% xa(j,k) = s% xa(j,k)/sumx
               end do
               
               call do1_net( &
                  s, k, species, s% num_reactions, &
                  net_lwork, dble(dt), ierr)

               if (ierr /= 0) then
                  if (.true.) then
                     if (dbg_in .or. dbg) then
                        write(*,4) &
                           'do1_net failure', s% model_number, k, num_iters
                        write(*,2) '1 - sum(xa)', k, 1d0 - sum(s% xa(1:species,k))
                     
                        !stop 'burn'
                     
                        write(*,2) 'sum(xa)', k, sum(s% xa(1:species,k))
                        write(*,2) 'max xa', k, maxval(s% xa(1:species,k))
                        write(*,2) 'min xa', k, minval(s% xa(1:species,k))
                        write(*,*)
                        write(*,2) '1 - sum(xa_pre)', k, 1d0 - sum(s% xa_pre(1:species,k))
                        write(*,2) 'sum(xa_pre)', k, sum(s% xa_pre(1:species,k))
                        write(*,2) 'max xa_pre', k, maxval(s% xa_pre(1:species,k))
                        write(*,2) 'min xa_pre', k, minval(s% xa_pre(1:species,k))
                        write(*,*)
                        write(*,2) 'sum avg_mix_dxdt', k, sum(avg_mix_dxdt(1:species,k))
                        write(*,2) 'sum del', k, sum(del(1:species))
                        write(*,2) 'sum dx', k, sum(dx(1:species))
                        write(*,2) 'sum x0', k, sum(x0(1:species))
                        write(*,2) 'lambda', k, lambda
                        write(*,*)
                        call show_stuff('do1_net failed')
                     
                        stop 'burn'

                     end if
                     call report_failure(k, num_iters)
                  end if
                  return
               end if
            
            end if
         
            s% max_burn_correction(k) = maxval(abs(del(1:nvar)))
            s% avg_burn_correction(k) = sum(abs(del(1:nvar)))/nvar

            if (lambda == one) then ! using full correction
               ! if magnitude of max correction is small enough, consider converged.
               if (is_bad_num(s% max_burn_correction(k))) then
                  ierr = -1
                  if (.not. (dbg .or. dbg_in)) return
                  
                  do j=1,species
                     write(*,2) 'del', j, del(j)
                  end do
                  write(*,3) 'max_burn_correction', k, num_iters, s% max_burn_correction(k)
                  stop 'burn1_BE'
               end if
               if (s% max_burn_correction(k) <= atol) exit solve_loop
            end if
         
            if (num_iters == maxiters) then
               if (dbg .or. dbg_in) then
!$omp critical(dbg_solve_burn)
                  write(*,2) 'num_iters', num_iters
                  write(*,2) 'maxiters', maxiters
                  write(*,2) 's% max_burn_correction(k)', k, s% max_burn_correction(k)
                  write(*,2) 's% avg_burn_correction(k)', k, s% avg_burn_correction(k)
                  write(*,2) 's% lnd(k)/ln10', k, s% lnd(k)/ln10
                  write(*,2) 's% lnd_start(k)/ln10', k, s% lnd_start(k)/ln10
                  write(*,2) 's% lnT(k)/ln10', k, s% lnT(k)/ln10
                  write(*,2) 's% lnT_start(k)/ln10', k, s% lnT_start(k)/ln10
                  write(*,*)
                  stop 'burn1_BE: num_iters == maxiters'
!$omp end critical(dbg_solve_burn)               
               end if
               ierr = -1
               return
            end if
         
            if (dbg) write(*,*)

         end do solve_loop
         
         ! set partials of abar and zbar wrt lnd and lnT
         s% d_abar_dlnd(k) = 0
         s% d_abar_dlnT(k) = 0
         s% d_zbar_dlnd(k) = 0
         s% d_zbar_dlnT(k) = 0
         
         if (.false.) then
            do i=1,species
               dx_dlnd = dt*s% dxdt_dRho(i,k)*s% rho(k)
               dx_dlnT = dt*s% dxdt_dT(i,k)*s% T(k)
               s% d_abar_dlnd(k) = s% d_abar_dlnd(k) + dabar_dx(i)*dx_dlnd
               s% d_abar_dlnT(k) = s% d_abar_dlnT(k) + dabar_dx(i)*dx_dlnT
               s% d_zbar_dlnd(k) = s% d_zbar_dlnd(k) + dzbar_dx(i)*dx_dlnd
               s% d_zbar_dlnT(k) = s% d_zbar_dlnT(k) + dzbar_dx(i)*dx_dlnT
            end do
         end if

         if (.not. keep_T_fixed) then
            
            if (k == -1) write(*,2) 'burn dlogT new old', &
               k, (lnT-lnT_init)/ln10, lnT/ln10, lnT_init/ln10
            
            !if (k == nz .and. .not. doing_partials) write(*,2) &
            !   'burn change cntr logT', nz, (lnT - lnT_init)/ln10, lnT/ln10
            s% T(k) = T_init
            s% lnT(k) = lnT_init
            
            if (.false. .and. .not. doing_partials) then
               dlnT = 0.5d0*(dlnT + s% newton_dx(s% i_lnT,k)) 
               s% newton_dx(s% i_lnT,k) = dlnT          
               s% lnT(k) = lnT_init + dlnT
               s% T(k) = exp(s% lnT(k))
            end if

            !if (ierr /= 0) stop 'burn1_BE failed'
            if (dbg .and. num_iters > 7) stop 'burn1_BE'

         end if
                  
         
         contains
         
         
         subroutine report_failure(k, num_iters)
            integer, intent(in) :: k, num_iters
            include 'formats'
            if (.not. (s% op_split_burn_trace .or. s% report_ierr)) return
            if (sparse) then
               condest = sparse_condest(s, k, species, ierr)
               write(*,'(a,1x,3i6,99e20.10)') &
                  'BE sparse failed to converge: k, num_iters, model, atol, T, condest', &
                  k, num_iters, s% model_number, atol, s% T(k), condest
            else
               write(*,4) 'BE failed to converge: k, num_iters, model, atol, T', &
                  k, num_iters, s% model_number, atol, s% T(k)
            end if
         end subroutine report_failure
         
         
         subroutine show_stuff(str)
            character (len=*), intent(in) :: str
            include 'formats'
            
            if (sparse) then
               condest = sparse_condest(s, k, species, ierr)    
            else
               condest = 0
            end if
            
            write(*,4) trim(str) // ' T, logT', &
               num_iters, k, s% model_number, s% T(k), s% lnT(k)/ln10
            write(*,4) trim(str) // ' rho, logRho', &
               num_iters, k, s% model_number, s% rho(k), s% lnd(k)/ln10
            write(*,4) trim(str) // ' Ye, eta', &
               num_iters, k, s% model_number, s% Ye(k), s% eta(k)
            write(*,4) trim(str) // ' dt, condest', &
               num_iters, k, s% model_number, dt, condest
            write(*,4) trim(str) // ' sum(x0)', &
               num_iters, k, s% model_number, sum(x0(1:species))
            write(*,4) trim(str) // ' sum(del)', &
               num_iters, k, s% model_number, sum(del(1:species))
            write(*,4) trim(str) // ' sum(xa)', &
               num_iters, k, s% model_number, sum(s% xa(1:species,k))
               
            write(*,*)
            write(*,2) 'del'
            do j=1,species
               !if (s% xa(j,k) > 1d-16) &
                  write(*,3) 'del ' // trim(chem_isos% name(s% chem_id(j))), &
                     j, k, del(j)
            end do
               
            write(*,*)
            write(*,2) 'xa'
            do j=1,species
               !if (s% xa(j,k) > 1d-16) &
                  write(*,3) 'xa ' // trim(chem_isos% name(s% chem_id(j))), &
                     j, k, s% xa(j,k)
            end do

            
            write(*,*)
            write(*,*)
            write(*,2) 'dt', k, dt
            write(*,2) 'sum dxdt_nuc', k, sum(s% dxdt_nuc(:,k))
            write(*,2) 'dt*(sum dxdt_nuc)', k, dt*sum(s% dxdt_nuc(:,k))
            write(*,2) 'sum del', k, sum(del(:))
            write(*,2) 'sum dx', k, sum(dx(:))
            write(*,2) 'sum xa', k, sum(s% xa(:,k))
            write(*,*)
            xsum = 0


            write(*,1) 'large magnitude entries'
            write(*,*)
            write(*,2) 'abs(dt*dxdt_nuc) > 1d-4', k
            do j=1,species
               if (dt*abs(s% dxdt_nuc(j,k)) > 1d-4) &
                  write(*,3) 'dt*dxdt_nuc(j,k) ' // trim(chem_isos% name(s% chem_id(j))), &
                     j, k, dt*s% dxdt_nuc(j,k)
            end do
            write(*,*)
            write(*,2) 'abs(dt*d_dxdt_dx) > 1d12', k
            do j=1,species
               do i=1,species
                  if (dt*abs(s% d_dxdt_dx(i,j,k)) > 1d12) &
                     write(*,4) 'dt*d_dxdt_dx(i,j,k) ' // trim(chem_isos% name(s% chem_id(j))) &
                        // ' wrt ' // trim(chem_isos% name(s% chem_id(i))), &
                        i, j, k, dt*s% d_dxdt_dx(i,j,k)
               end do
            end do
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,1) 'nonzero entries'
            write(*,*)
            do j=1,species
               if (s% dxdt_nuc(j,k) /= 0d0) &
                  write(*,3) 'dxdt_nuc(j,k) ' // trim(chem_isos% name(s% chem_id(j))), &
                     j, k, s% dxdt_nuc(j,k)
            end do
            write(*,*)
            write(*,2) 'd_dxdt_dx', k
            do j=1,species
               do i=1,species
                  if (s% d_dxdt_dx(i,j,k) /= 0d0) &
                     write(*,4) 'd_dxdt_dx(i,j,k) ' // trim(chem_isos% name(s% chem_id(j))) &
                        // ' wrt ' // trim(chem_isos% name(s% chem_id(i))), &
                        i, j, k, s% d_dxdt_dx(i,j,k)
               end do
            end do
            write(*,2) 'done d_dxdt_dx', k
            write(*,*)
            do j=1,species
               write(*,3) 'sum d_dxdt_dx(:,j,k) ' // trim(chem_isos% name(s% chem_id(j))), &
                  j, k, sum(s% d_dxdt_dx(:,j,k))
            end do
            write(*,*)
            do i=1,species
               write(*,3) 'sum d_dxdt_dx(i,:,k) ' // trim(chem_isos% name(s% chem_id(i))), &
                  j, k, sum(s% d_dxdt_dx(i,:,k))
            end do

            stop 'burn1_BE'
         end subroutine show_stuff
         
      
      end subroutine burn1_BE
      
      
      subroutine revise_avg_burn_dxdt(s, k, species, avg_burn_dxdt, dt_total, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         real(dp), pointer :: avg_burn_dxdt(:,:)
         real(dp), intent(in) :: dt_total
         integer, intent(out) :: ierr
         
         integer :: j
         real(dp) :: sum_dxdt, sum_rates
         
         include 'formats'
      
         ierr = 0

         ! revise avg_burn_dxdt so sums to 0 for cell k
         sum_dxdt = sum(avg_burn_dxdt(1:species,k))

         if (sum_dxdt > 0d0) then ! scale down the rates that are > 0
            sum_rates = 0d0 ! sum of rates > 0
            do j=1,species
               if (avg_burn_dxdt(j,k) <= 0d0) cycle
               sum_rates = sum_rates + avg_burn_dxdt(j,k)
            end do
            if (sum_rates <= 0d0) then
               write(*,2) 'sum_rates should be > 0', k, sum_rates
               stop 'revise_avg_burn_dxdt'
            end if
            do j=1,species
               if (avg_burn_dxdt(j,k) <= 0d0) cycle
               avg_burn_dxdt(j,k) = &
                  avg_burn_dxdt(j,k)*(sum_rates - sum_dxdt)/sum_rates
            end do
         else if (sum_dxdt < 0d0) then ! scale down the rates that are < 0
            sum_rates = 0d0 ! sum of rates < 0
            do j=1,species
               if (avg_burn_dxdt(j,k) >= 0d0) cycle
               sum_rates = sum_rates + avg_burn_dxdt(j,k)
            end do
            if (sum_rates >= 0d0) then
               write(*,2) 'sum_rates should be < 0', k, sum_rates
               stop 'revise_avg_burn_dxdt'
            end if
            do j=1,species
               if (avg_burn_dxdt(j,k) >= 0d0) cycle
               avg_burn_dxdt(j,k) = &
                  avg_burn_dxdt(j,k)*(sum_rates - sum_dxdt)/sum_rates
            end do
         end if
         sum_dxdt = sum(avg_burn_dxdt(1:species,k))
         !if (k == 939) write(*,2) 'sum avg_burn_dxdt', k, sum_dxdt
         if (abs(sum_dxdt) > 1d-13) then
            write(*,2) 'bad sum avg_burn_dxdt', k, sum_dxdt
            write(*,2) 'sum xa', k, sum(s% xa(1:species,k))
            write(*,2) 'sum xa_pre', k, sum(s% xa_pre(1:species,k))
            write(*,2) 'dt_total', k, dt_total
            stop 'revise_avg_burn_dxdt'
         end if
         
      end subroutine revise_avg_burn_dxdt
      
         
      subroutine burn1_partial_NSE( &
            s, k, species, net_lwork, sparse, sfmin, dt, &
            avg_mix_dxdt, num_iters, ierr)

         use chem_def, only: iother
         use rates_def, only: i_rate
         use star_utils, only: get_fraction_NSE_burn

         type (star_info), pointer :: s
         integer, intent(in) :: k, species, net_lwork
         logical, intent(in) :: sparse
         real(dp), intent(in) :: sfmin
         real(dp), intent(in) :: dt
         real(dp), pointer, intent(in) :: avg_mix_dxdt(:,:)
         integer, intent(out) :: num_iters, ierr
         
         real(dp) :: eps_nuc, xa(species), alfa, beta, tmp
         integer :: j
         logical :: dbg, trace
         
         include 'formats'
         
         ierr = 0
         num_iters = 0

         dbg = .false. ! (s% model_number == 1137)
         trace = s% op_split_burn_trace   
         
         alfa = get_fraction_NSE_burn(s,k)
         beta = 1d0 - alfa
         if (alfa > 1d0 .or. alfa < 0d0) then
            ierr = -1
            return
         end if
         
         if (trace) write(*,2) 'burn1_partial_NSE alfa', &
            k, alfa, &
            (s% T(k) - s% T_NSE_full_off)/ &
               (s% T_NSE_full_on - s% T_NSE_full_off), &
            s% T(k), s% T_NSE_full_off, s% T_NSE_full_on
         
         ierr = 0
         do j=1,species
            xa(j) = s% xa(j,k)
         end do
         
         call burn1_BE( &
            s, k, species, net_lwork, sparse, sfmin, dt, alfa, &
            avg_mix_dxdt, num_iters, .false., .false., ierr)
         if (ierr /= 0) then
            write(*,3) 'burn1_partial_NSE failed in net burn', k, num_iters, alfa, s% T(k)
            return
         end if
         
         eps_nuc = s% eps_nuc(k)
         do j=1,species
            tmp = xa(j)
            xa(j) = s% xa(j,k)
            s% xa(j,k) = tmp
         end do
         
         call burn1_NSE(s, k, species, dt, ierr)         
         if (ierr /= 0) then
            write(*,2) 'burn1_partial_NSE failed in NSE burn', k, alfa, s% T(k)
            return
         end if
         
         if (trace .or. dbg) &
            write(*,2) 'burn1_partial_NSE eps_nuc combo, NSE, net, alfa', &
               k, alfa*s% eps_nuc(k) + beta*eps_nuc, s% eps_nuc(k), eps_nuc, alfa
               
         !s% eps_nuc(k) = alfa*s% eps_nuc(k) + beta*eps_nuc
         s% eps_nuc(k) = eps_nuc  ! TRY JUST USING NET VALUE FOR EPS_NUC
         do j=1,species
            s% xa(j,k) = alfa*s% xa(j,k) + beta*xa(j)
         end do
         
         s% eps_nuc_categories(i_rate,iother,k) = s% eps_nuc(k)

         if (dbg) write(*,2) 'eps_nuc combined', k, s% eps_nuc(k)
         
         if (dbg) write(*,2) 'burn1_partial_NSE', k, alfa, s% T(k)
         
      end subroutine burn1_partial_NSE

      
      subroutine burn1_NSE(s, k, species, dt, ierr)
         use chem_lib, only: composition_info
         use nse, only: do1_nse_get
         type (star_info), pointer :: s
         integer, intent(in) :: k, species
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         real(dp) :: xsum, ye_init, nse_eps_nuc, nse_eps_nuc_neu_total
         real(dp), dimension(species) :: &
            nse_dxdt_nuc, dabar_dx, dzbar_dx, dmc_dx
         integer :: i
         
         include 'formats'
         call composition_info( &
            species, s% chem_id, s% xa(1:species,k), s% X(k), s% Y(k), &
            s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
            s% mass_correction(k), xsum, dabar_dx, dzbar_dx, dmc_dx)  
         ye_init = s% ye(k)
         !write(*,*)
            
         ierr = 0
         ! this sets eps_nuc and dxdt_nuc but doesn't change xa
         call do1_nse_get( &
            s, k, species, dt, s% xa(1:species,k), &
            s% lnT(k)/ln10, s% lnd(k)/ln10, s% eta(k), s% ye(k), &
            nse_eps_nuc_neu_total, nse_eps_nuc, nse_dxdt_nuc, &
            ierr)
         if (ierr /= 0) then
            write(*,3) 'burn1_NSE FAILED', k, s% model_number, s% T(k)
            return
         end if
         
         do i=1,species
            s% xa(i,k) = s% xa(i,k) + dt*nse_dxdt_nuc(i)
         end do
         
         call composition_info( &
            species, s% chem_id, s% xa(1:species,k), s% X(k), s% Y(k), &
            s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
            s% mass_correction(k), xsum, dabar_dx, dzbar_dx, dmc_dx)  

      end subroutine burn1_NSE
      
      
      subroutine positivity(s, k, species, x, del, lambda, str, iter, dbg)
         use chem_def, only: chem_isos
         type (star_info), pointer :: s         
         integer, intent(in) :: k, species, iter
         logical, intent(in) :: dbg
         real(dp), intent(in), dimension(:) :: x, del
         character(len=*), intent(in) :: str
         real(dp), intent(out) :: lambda
      
         integer :: i, j, bad_j
         real(dp) :: alpha, new_xa, old_xa, dxa, eps
      
         include 'formats'
      
         lambda = 1
         eps = s% op_split_burn_atol ! allow this much below 0
         bad_j = 0
         do j=1,species
            old_xa = x(j)
            if (old_xa < 1d-99) cycle 
            dxa = del(j)
            new_xa = old_xa + dxa
            if (new_xa >= 0) cycle
            alpha = -(old_xa + eps)/dxa
            ! so dxa*alpha = -old_xa - eps,
            ! and therefore old_xa + alpha*dxa = -eps
            if (alpha < lambda) then
               lambda = alpha
               bad_j = j
            end if 
         end do
         if (lambda < 1) lambda = 0.8*lambda ! under correct
         if (dbg .and. lambda < 1) then
            j = bad_j
            write(*,3) 'lambda ' // trim(chem_isos% name(s% chem_id(j))), &
               iter, j, lambda, x(j) + del(j), x(j), del(j)
         end if
   
      end subroutine positivity


      subroutine getf2(m, a, lda, ipiv, sfmin, use_pivoting, info)
         integer :: info, lda, m
         logical :: use_pivoting
         integer :: ipiv(:) ! (*)
         real(dp) :: a(:,:) ! (lda,*)
         real(dp) :: sfmin
         real(dp), parameter :: one=1, zero=0
         integer :: i, j, jp, ii, jj, n, mm
         real(dp) :: tmp, da
         do j = 1, m
            info = 0
            if (use_pivoting) then
               jp = j - 1 + maxloc(abs(a(j:lda,j)),dim=1)
               ipiv( j ) = jp
            else
               jp = j
            end if
            if( a( jp, j ).ne.zero ) then
               if( jp.ne.j ) then ! swap a(j,:) and a(jp,:)
                  do i=1,m
                     tmp = a(j,i)
                     a(j,i) = a(jp,i)
                     a(jp,i) = tmp
                  end do
               end if
               if( j.lt.m ) then 
                  if( abs(a( j, j )) .ge. sfmin ) then
                     da = one / a( j, j )
                     n = m-j
                     mm = mod(n,5)
                     if (mm /= 0) then
                        do i = 1,mm
                           a(j+i,j) = da*a(j+i,j)
                        end do
                     end if
                     if (n >= 5) then
                        do i = mm + 1,n,5
                           a(j+i,j) = da*a(j+i,j)
                           a(j+i+1,j) = da*a(j+i+1,j)
                           a(j+i+2,j) = da*a(j+i+2,j)
                           a(j+i+3,j) = da*a(j+i+3,j)
                           a(j+i+4,j) = da*a(j+i+4,j)
                        end do
                     end if
                  else ! no scale
                    do i = 1, m-j 
                       a( j+i, j ) = a( j+i, j ) / a( j, j ) 
                    end do 
                  end if 
               end if 
            else if( info.eq.0 ) then
               info = j
            end if
            if( j.lt.m ) then
               !call dger( m-j, m-j, -one, a( j+1, j ), 1, a( j, j+1 ), lda, a( j+1, j+1 ), lda )
               do jj = j+1, m
                  do ii = j+1, m
                     a(ii,jj) = a(ii,jj) - a(ii,j)*a(j,jj)
                  end do
               end do
            end if
         end do
      end subroutine getf2
      
      
      subroutine getrs_1( n, a, lda, ipiv, b, ldb, use_pivoting, info )
         integer :: info, lda, ldb, n ! nrhs=1
         logical :: use_pivoting
         integer :: ipiv(:)
         real(dp) :: a(:,:)
         real(dp) :: b(:)
         real(dp), parameter :: one=1, zero=0
         integer :: i, k, n32, ix, ip
         info = 0
         if (use_pivoting) call laswp_1(b, ldb, n, ipiv )
         !call dtrsm( 'left', 'lower', 'no transpose', 'unit', n, nrhs, one, a, lda, b, ldb )
         do k = 1,n
            if (b(k).ne.zero) then
               do i = k + 1,n
                  b(i) = b(i) - b(k)*a(i,k)
               end do
            end if
         end do
         !call dtrsm( 'left', 'upper', 'no transpose', 'non-unit', n, nrhs, one, a, lda, b, ldb )
         do k = n,1,-1
            if (b(k).ne.zero) then
               b(k) = b(k)/a(k,k)
               do i = 1,k - 1
                  b(i) = b(i) - b(k)*a(i,k)
               end do
            end if
         end do
      end subroutine getrs_1
      
      
      subroutine laswp_1( a, lda, k2, ipiv )
         integer :: k2, lda ! n=1, incx=1, k1 = 1
         integer :: ipiv(:)
         real(dp) :: a(:)
         integer :: i, ip
         real(dp) :: temp
         do i = 1, k2
            ip = ipiv( i )
            if( ip.ne.i ) then
               temp = a( i )
               a( i ) = a( ip )
               a( ip ) = temp
            end if
         end do
      end subroutine laswp_1


               


      end module solve_burn

