! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module overshoot

      use const_def
      use num_lib
      use utils_lib
      use star_private_def

      implicit none

      
      contains

      
      ! exponential diffusive overshoot as described in the paper by Falk Herwig, 
      ! "The evolution of AGB stars with convective overshoot", A&A, 360, 952-968 (2000)
      subroutine add_overshooting(s, ierr)
         use mlt_def
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         real(dp) :: D0, vc0, Hp, r0, D_ov, cdc_ov, vc_ov, frac, q_edge, r_edge, &
            f_above, f_below, step_f_above, step_f_below, f0_above, f0_below, &
            f2_above, f2_below, D2_above, D2_below, &
            ov_limit, D_ov_limit, h1_czb_mass, h1_czb_dm, &
            h1_czb_mass_old, top_r, bot_r, zone_dr
         integer :: i, k, loc, k0, n_conv_bdy, nz
         real(dp), pointer, dimension(:) :: grada, gradr

         logical :: dbg, have_h1_czb
         
         include 'formats'
         
         dbg = .false.
         
         ierr = 0         
         nz = s% nz
         grada => s% grada_at_face
         gradr => s% gradr
         n_conv_bdy = s% num_conv_boundaries
         D_ov_limit = s% D_mix_ov_limit
         
         if (dbg) write(*,3) 'add_overshooting model n_conv_bdy', s% model_number, n_conv_bdy
         
         do i=1, n_conv_bdy ! from center to surface

            if (s% conv_bdy_q(i) < s% min_overshoot_q) then
               if (dbg) write(*,*) 'skip since s% conv_bdy_q(i) < min_overshoot_q', i
               cycle
            end if
            
            call set_f(i)
            
            have_h1_czb = .false.
            
            if (f_above == 0 .and. f_below == 0 .and. &
                  step_f_above == 0 .and. step_f_below == 0) cycle
            if (s% star_mass <= s% mass_for_overshoot_full_off) cycle
            if (s% star_mass <= s% mass_for_overshoot_full_on) then
               frac = (s% star_mass - s% mass_for_overshoot_full_off) / &
                      (s% mass_for_overshoot_full_on - s% mass_for_overshoot_full_off)
               frac = 0.5d0*(1 - cos(pi*frac))
               if (dbg) write(*,1) 'overshoot frac', frac
               f_above = f_above*frac
               f_below = f_below*frac
               step_f_above = step_f_above*frac
               step_f_below = step_f_below*frac
            end if
            
            if (f0_above < 0) f0_above = max(f_above, step_f_above)
            if (f0_below < 0) f0_below = max(f_below, step_f_below)
            
            loc = s% conv_bdy_loc(i)
            if (loc < 1 .or. loc > nz) then
               write(*,*) 'bad s% conv_bdy_loc(i)', i, s% conv_bdy_loc(i)
               ierr = -1
               return
            end if
            
            !write(*,*) 's% top_conv_bdy(i) q', s% top_conv_bdy(i), i, s% conv_bdy_q(i)
            
            if (s% top_conv_bdy(i)) then ! overshoot toward surface
               if (f_above + step_f_above > 0 .and. &
                     s% mixing_type(loc+1) == convective_mixing .and. loc > 1) then
                  if (i <= 1) then
                     bot_r = s% R_center
                  else
                     if (s% top_conv_bdy(i-1)) cycle
                     bot_r = s% r(s% conv_bdy_loc(i-1))
                  end if
                  top_r = s% r(loc)
                  zone_dr = top_r - bot_r
                  if (dbg) write(*,1) 'toward surface zone_dr', zone_dr
                  call overshoot_toward_surface(ierr)
                  if (ierr /= 0) return
               end if
            else ! overshoot toward center
               if (f_below + step_f_below > 0 .and. &
                     s% mixing_type(loc-1) == convective_mixing) then
                  if (i >= n_conv_bdy) then
                     top_r = s% r(1)
                  else
                     if (.not. s% top_conv_bdy(i+1)) then
                        write(*,*) 's% top_conv_bdy(i+1)', s% top_conv_bdy(i+1)
                        cycle
                     end if
                     top_r = s% r(s% conv_bdy_loc(i+1))
                  end if
                  bot_r = s% r(loc)
                  zone_dr = top_r - bot_r
                  if (.false. .or. dbg) write(*,1) 'toward center zone_dr', &
                     zone_dr, s% m(loc)/Msun, bot_r/Rsun
                  call overshoot_toward_center( &
                     s% burn_h_conv_region(i), s% burn_he_conv_region(i), &
                     s% burn_z_conv_region(i), ierr)
                  if (ierr /= 0) return
               else if (.false.) then
                  write(*,3) 'f_below', s% mixing_type(loc+2), loc+2, f_below
                  write(*,3) 'f_below', s% mixing_type(loc+1), loc+1, f_below
                  write(*,3) 'f_below', s% mixing_type(loc), loc, f_below
                  write(*,3) 'f_below', s% mixing_type(loc-1), loc-1, f_below
                  write(*,3) 'f_below', s% mixing_type(loc-2), loc-2, f_below
                  !stop
               end if
            end if
            if (dbg) write(*,*)
         end do
         
         
         !if (n_conv_bdy > 0) stop 'overshoot'
         
         
         contains
         
         
         subroutine overshoot_toward_surface(ierr)
            integer, intent(out) :: ierr
            include 'formats'
            real(dp) :: h, hstep, h2, dr, dr2, dr_limit, r_limit
            integer :: kk
            ierr = 0
            if (dbg) write(*,1) 'overshoot_toward_surface'
            ! top  of region is between loc+1 and loc, i.e. in cell k=loc
            call get_r0_vc0( &
               loc, .false., f0_above, k0, r0, vc0, D0, Hp, q_edge, r_edge, dbg, ierr)
            if (ierr /= 0) then
               write(*,*) 'overshoot toward surface failure'
               return
            end if
            if (k0 <= 0) then
               if (dbg) write(*,2) 'skip overshoot_toward_surface: k0', k0
               return
            end if

            h = f_above*Hp    
            hstep = step_f_above*Hp
            if (D2_above > 0 .and. f2_above > 0 .and. D2_above < D0) then
               dr2 = h*log(D0/D2_above)/2
               h2 = f2_above*Hp
            else
               dr2 = 1d99
            end if

            do k = k0, 1, -1
               if (s% r(k) < r0) cycle
               dr = s% r(k) - r0
               if (k == k0) then
                  ov_limit = 0
               else
                  ov_limit = D_ov_limit
               end if
               if (dr < dr2) then
                  if (dbg .and. D0 < 0) write(*,2) 'bogus D0', k, D0
                  call get_D_ov(k, dr, D0, h, hstep, D_ov, ov_limit, cdc_ov, vc_ov, dbg)
               else
                  call get_D_ov(k, dr-dr2, D2_above, h2, 0d0, D_ov, ov_limit, cdc_ov, vc_ov, dbg)
               end if
               if (cdc_ov == 0) then
                  if (dbg) write(*,2) 'finished extending upward: cdc_ov == 0', k

                  if (s% mixing_type(k+1) == overshoot_mixing) then 
                     ! set mixing_type_change_dq(k)
                     if (dr < dr2) then
                        dr_limit = max(hstep, 0.5d0*h*log(D0/ov_limit))
                     else
                        dr_limit = dr + 0.5d0*h2*log(D2_below/ov_limit)
                     end if
                     r_limit = r0 + dr_limit
                     if (s% r(k) >= r_limit .and. r_limit > s% r(k+1)) then
                        s% mixing_type_change_dq(k) = s% dq(k) * &
                           (s% r(k)**3 - r_limit**3) / (s% r(k)**3 - s% r(k+1)**3)
                     end if
                  end if
                  
                  if (k > loc) then ! undershooting
                     do kk = k, loc, -1 ! change to radiative
                        s% cdc(k) = 0
                        s% D_mix(k) = 0
                        s% conv_vel(k) = 0
                        s% mixing_type(k) = no_mixing
                     end do
                  end if

                  exit
               end if
               if (dbg) then
                  write(*,2) 'overshoot upward', k, s% m(k)/Msun, (s% r(k)-r0)/Rsun, &
                     s% r(k)/Rsun, Hp/Rsun, D_ov, s% D_mix(k), s% r(k) - r_edge
               end if
               if (s% r(k) <= r_edge) then ! leave it convective type, but change mixing coeff
                  if (dbg) write(*,2) 'leave it convective type, but change mixing coeff', k
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  s% conv_vel(k) = vc_ov
                  s% mixing_type(k) = convective_mixing
               else if (cdc_ov > s% cdc(k)) then
                  if (dbg) write(*,2) 'change to overshooting', k
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  s% conv_vel(k) = vc_ov
                  s% mixing_type(k) = overshoot_mixing
               end if
            end do
         end subroutine overshoot_toward_surface
         
         
         subroutine overshoot_toward_center(burn_h, burn_he, burn_z, ierr)
            logical, intent(in) :: burn_h, burn_he, burn_z
            integer, intent(out) :: ierr
            
            real(dp) :: h, hstep, h2, dr, dr2, dr_limit, r_limit
            logical :: dbg
            integer :: kk
            include 'formats'

            dbg = .false.

            if (dbg) write(*,*) 'overshoot_toward_center'
            ierr = 0
            ! bottom of region is between loc-1 and loc, i.e., in cell loc-1
            call get_r0_vc0( &
               loc-1, .true., f0_below, k0, r0, vc0, D0, Hp, q_edge, r_edge, dbg, ierr)
            if (ierr /= 0) return
            if (k0 <= 0) return       
            
            if (dbg) then
               write(*,2) 'loc-1', loc-1
               write(*,2) 'k0', k0
               write(*,1) 'f0_below', f0_below
               write(*,1) 'r0', r0
               write(*,1) 'Hp', Hp
            end if     
            
            if ((.not. (burn_h .or. burn_he .or. burn_z)) .and. (.not. have_h1_czb)) then
               have_h1_czb = .true.
               h1_czb_mass = s% M_center + s% xmstar*q_edge
               s% h1_czb_mass = h1_czb_mass/Msun
               if (.false.) then
                  write(*,2) 's% h1_czb_mass', k0, s% h1_czb_mass
                  write(*,2) 'q_edge', k0, q_edge
                  write(*,2) 'r_edge/Rsun', k0, r_edge/Rsun
                  write(*,2) 'Hp/Rsun', k0, Hp/Rsun
                  write(*,*)
               end if
               call check_TP
               if (dbg) write(*,1) 'f_below', f_below
            end if

            h = f_below*Hp
            hstep = step_f_below*Hp
            if (D2_below > 0 .and. f2_below > 0 .and. D2_below < D0) then
               dr2 = h*log(D0/D2_below)/2
               h2 = f2_below*Hp
               if (dbg) then
                  write(*,2) 'dr2', k0, dr2
                  if (r0 > dr2) write(*,1) 'log10((r0-dr2)/Rsun)', log10((r0-dr2)/Rsun)
                  write(*,1) 'log10(r0/Rsun)', log10(r0/Rsun)
                  write(*,1) 'D2_below', D2_below
                  write(*,1) 'f2_below', f2_below
                  write(*,1) 'f_below', f_below
                  write(*,1) 'Hp', Hp
                  write(*,1) 'h', h
                  write(*,1) 'D0', D0
                  write(*,*)
               end if
            else
               dr2 = 1d99
               h2 = 0
            end if
            
            do k=k0+1, nz
               if (s% r(k) > r0) cycle
               dr = r0 - s% r(k)
               
               if (dbg) write(*,1) 'dr/(f_below*Hp)', dr/(f_below*Hp)
               
               if (k == k0+1) then
                  ov_limit = 0
               else
                  ov_limit = D_ov_limit
               end if
               if (dr < dr2) then
                  call get_D_ov(k, dr, D0, h, hstep, D_ov, ov_limit, cdc_ov, vc_ov, dbg)
               else
                  call get_D_ov(k, dr-dr2, D2_below, h2, 0d0, D_ov, ov_limit, cdc_ov, vc_ov, dbg)
               end if
               
               if (dbg) write(*,1) 'D_ov', D_ov
               if (dbg) write(*,1) 'D_ov/D0', D_ov/D0
               
               if (cdc_ov == 0) then
                  if (dbg) write(*,2) 'finished extending downward', k
                  if (s% mixing_type(k-1) == overshoot_mixing) then 
                     ! set mixing_type_change_dq(k-1)
                     if (dr < dr2) then
                        dr_limit = max(hstep, 0.5d0*h*log(D0/ov_limit))
                     else
                        dr_limit = dr + 0.5d0*h2*log(D2_below/ov_limit)
                     end if
                     r_limit = r0 - dr_limit
                     if (s% r(k-1) >= r_limit .and. r_limit > s% r(k)) then
                        s% mixing_type_change_dq(k-1) = s% dq(k-1) * &
                           (s% r(k-1)**3 - r_limit**3) / (s% r(k-1)**3 - s% r(k)**3)
                     end if
                  end if
                  
                  if (k < loc) then ! undershooting
                     do kk = k, loc ! change to radiative
                        s% cdc(k) = 0
                        s% D_mix(k) = 0
                        s% conv_vel(k) = 0
                        s% mixing_type(k) = no_mixing
                     end do
                  end if




                  exit
               end if
               if (dbg) then
                  write(*,2) 'overshoot downward', k, s% star_mass*s% q(k), s% r(k)/Rsun
                  write(*,*)
               end if
               if (s% r(k) >= r_edge) then ! leave it convective type, but change mixing coeff
                  if (dbg) write(*,2) 'leave it convective', k
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  s% conv_vel(k) = vc_ov
                  s% mixing_type(k) = convective_mixing
               else if (cdc_ov > s% cdc(k)) then ! change to overshooting
                  if (dbg) write(*,2) 'change to overshooting', k
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  s% conv_vel(k) = vc_ov
                  s% mixing_type(k) = overshoot_mixing
               end if
            end do
            
         end subroutine overshoot_toward_center

         
         subroutine set_f(i)
            integer, intent(in) :: i
            if (s% burn_h_conv_region(i)) then
            
               if (dbg) write(*,*) 'burn_h_conv_region'

               f_above = s% overshoot_f_above_burn_h
               f_below = s% overshoot_f_below_burn_h
               step_f_above = s% step_overshoot_f_above_burn_h
               step_f_below = s% step_overshoot_f_below_burn_h
               
               f0_above = s% overshoot_f0_above_burn_h
               f0_below = s% overshoot_f0_below_burn_h
               D2_above = s% overshoot_D2_above_burn_h
               D2_below = s% overshoot_D2_below_burn_h
               f2_above = s% overshoot_f2_above_burn_h
               f2_below = s% overshoot_f2_below_burn_h
               
            else if (s% burn_he_conv_region(i)) then 
            
               if (dbg) write(*,*) 'burn_he_conv_region'
                       
               f_above = s% overshoot_f_above_burn_he
               f_below = s% overshoot_f_below_burn_he
               step_f_above = s% step_overshoot_f_above_burn_he
               step_f_below = s% step_overshoot_f_below_burn_he
               
               f0_above = s% overshoot_f0_above_burn_he
               f0_below = s% overshoot_f0_below_burn_he
               D2_above = s% overshoot_D2_above_burn_he
               D2_below = s% overshoot_D2_below_burn_he
               f2_above = s% overshoot_f2_above_burn_he
               f2_below = s% overshoot_f2_below_burn_he
               if (s% have_done_TP) then
                  f_below = f_below*s% ovr_below_burn_he_factor
                  f0_below = f0_below*s% ovr_below_burn_he_factor
                  f2_below = f2_below*s% ovr_below_burn_he_factor
               end if               
               
            else if (s% burn_z_conv_region(i)) then    
            
               if (dbg) write(*,*) 'burn_z_conv_region'
                   
               f_above = s% overshoot_f_above_burn_z
               f_below = s% overshoot_f_below_burn_z
               step_f_above = s% step_overshoot_f_above_burn_z
               step_f_below = s% step_overshoot_f_below_burn_z
               
               f0_above = s% overshoot_f0_above_burn_z
               f0_below = s% overshoot_f0_below_burn_z
               D2_above = s% overshoot_D2_above_burn_z
               D2_below = s% overshoot_D2_below_burn_z
               f2_above = s% overshoot_f2_above_burn_z
               f2_below = s% overshoot_f2_below_burn_z
               
            else  
            
               if (dbg) write(*,*) 'nonburn_conv_region'
                      
               f_above = s% overshoot_f_above_nonburn
               f_below = s% overshoot_f_below_nonburn
               step_f_above = s% step_overshoot_f_above_nonburn
               step_f_below = s% step_overshoot_f_below_nonburn
               
               f0_above = s% overshoot_f0_above_nonburn
               f0_below = s% overshoot_f0_below_nonburn               
               D2_above = s% overshoot_D2_above_nonburn
               D2_below = s% overshoot_D2_below_nonburn
               f2_above = s% overshoot_f2_above_nonburn
               f2_below = s% overshoot_f2_below_nonburn
               
            end if
         end subroutine set_f
         
         
         subroutine check_TP ! check for AGB thermal pulse dredge up
            logical, parameter :: dbg = .false.
            include 'formats'
            if (s% power_h_burn + s% power_he_burn < 0.5d0*s% power_nuc_burn) then
               if (dbg .and. s% TP_state /= 0) then
                  write(*,*) 'advanced burning: change to TP_state = 0'
                  write(*,*)
               end if
               s% TP_state = 0
            else if (s% he_core_mass > s% c_core_mass + 0.1d0) then
               if (dbg .and. s% TP_state /= 0) then
                  write(*,*) 'not yet: change to TP_state = 0'
                  write(*,1) 's% he_core_mass', s% he_core_mass
                  write(*,1) 's% c_core_mass', s% c_core_mass
                  write(*,1) 'h1 - c_core_mass', s% he_core_mass - s% c_core_mass
                  write(*,*)
               end if
               s% TP_state = 0
            else if (s% TP_state == 0) then ! not in TP
               if (s% power_he_burn > s% power_h_burn) then ! starting TP
                  s% TP_state = 1
                  s% TP_M_H_on = h1_czb_mass
                  s% TP_M_H_min = h1_czb_mass
                  if (dbg) then
                     write(*,*) 'change to TP_state = 1'
                     write(*,1) 's% TP_M_H_on', s% TP_M_H_on
                     write(*,*)
                  end if
               end if
            else
               h1_czb_dm = h1_czb_mass*1d-6
               if (s% TP_state == 1) then ! early part of TP
                  if (h1_czb_mass < s% TP_M_H_on - h1_czb_dm) then
                     if (dbg) then
                        write(*,*) 'change to TP_state = 2'
                        write(*,1) 's% TP_M_H_on', s% TP_M_H_on
                        write(*,1) 'h1_czb_dm', h1_czb_dm
                        write(*,1) 's% TP_M_H_on - h1_czb_dm', s% TP_M_H_on - h1_czb_dm
                        write(*,1) 'h1_czb_mass', h1_czb_mass
                        write(*,1) '(s% TP_M_H_on - h1_czb_dm) - h1_czb_mass', &
                           (s% TP_M_H_on - h1_czb_dm) - h1_czb_mass
                        write(*,*)
                     end if
                     s% TP_state = 2
                     s% TP_count = 0
                     s% TP_M_H_min = h1_czb_mass
                     s% have_done_TP = .true.
                  else if (s% power_h_burn > s% power_he_burn) then ! no dredge up
                     if (dbg) write(*,*) 'change to TP_state = 0: no dredge up this time'
                     s% TP_state = 0
                  end if
               else if (s% TP_state == 2) then ! dredge up part of TP
                  if (h1_czb_mass < s% TP_M_H_min) s% TP_M_H_min = h1_czb_mass
                  h1_czb_mass_old = s% h1_czb_mass_old*Msun
                  if (dbg) then
                     write(*,1) '(h1_czb_mass - h1_czb_mass_old)/Msun', &
                        (h1_czb_mass - h1_czb_mass_old)/Msun
                     write(*,1) 'h1_czb_dm/Msun', h1_czb_dm/Msun
                     write(*,*)
                  end if
                  if (h1_czb_mass > h1_czb_mass_old + h1_czb_dm .or. &
                        s% power_h_burn > s% power_he_burn) then
                     s% TP_count = s% TP_count + 1
                     if (dbg) write(*,2) '++ TP_count', s% TP_count, &
                           h1_czb_mass - (h1_czb_mass_old + h1_czb_dm), &
                           h1_czb_mass,  h1_czb_mass_old + h1_czb_dm, &
                           s% h1_czb_mass_old, h1_czb_dm
                  else if (h1_czb_mass < h1_czb_mass_old - h1_czb_dm) then
                     s% TP_count = max(0, s% TP_count - 1)
                     if (dbg) write(*,3) '-- TP_count', s% TP_count, s% TP_state
                  end if
                  if (s% TP_count > s% max_DUP_counter) then
                     if (dbg) write(*,*) 'change to TP_state = 0'
                     s% TP_state = 0
                  end if
               end if
               if (s% TP_state == 2) then
                  f_below = f_below*s% overshoot_below_noburn_factor
                  f0_below = f0_below*s% overshoot_below_noburn_factor
               end  if
            end if
         end subroutine check_TP
         
         
         subroutine get_r0_vc0( &
               k00, overshoot_toward_center, f0, k0, r0, &
               vc0, D0, Hp, q_edge, r_edge, dbg, ierr)
            integer, intent(in) :: k00 
            ! expect edge of convection somewhere in cell k00
            ! if s% use_Ledoux_criterion, then edge is where gradL == gradr
            ! else, else is where grada_at_face == gradr
            logical, intent(in) :: overshoot_toward_center
            real(dp), intent(in) :: f0
            integer, intent(out) :: k0
            real(dp), intent(out) :: r0, vc0, D0, Hp, q_edge, r_edge
            logical, intent(in) :: dbg
            ! r(k0) >= r0 >= r(k0+1)
            ! r0 is radius where start overshooting
            ! vc0 is convection velocity at r0
            ! Hp is scale height at edge of convection
            integer, intent(out) :: ierr
            
            real(dp) :: dq, ddq, dqfrac, x, x0, x1, x2, a0, a1, a2, &
               lnP_edge, P_edge, lnd_edge, rho_edge, frac, dg00, dgp1, grada_at_face         
            integer :: km1, kp1, kk
            
            include 'formats'
            
            k0 = -1
            r0 = 0
            vc0 = 0
            D0 = 0
            Hp = 0
            ierr = 0

            km1 = k00-1
            kp1 = k00+1
            
            if (kp1 > nz) then
               if (.not. overshoot_toward_center) return
               k0 = nz
               r0 = s% r(nz)
               vc0 = s% conv_vel(nz)
               D0 = s% cdc(nz)
               Hp = r0
               return
            end if

            dq = s% dq(k00)
            ddq = s% mixing_type_change_dq(k00)
            dqfrac = ddq/dq ! fractional distance from k00 to kp1 for edge of convection
            if (dqfrac > 1d0 .or. dqfrac < 0d0) then
               write(*,2) 'bad dqfrac', k00, dqfrac, ddq, dq
               stop 'get_r0_vc0'
            end if
            q_edge = s% q(k00) - ddq
            r_edge = (dqfrac*s% r(kp1)**3 + (1-dqfrac)*s% r(k00)**3)**(1d0/3d0)
            
            x0 = s% dq(km1)/2
            x1 = s% dq(km1) + s% dq(k00)/2
            x2 = s% dq(km1) + s% dq(k00) + s% dq(kp1)/2
            x = s% dq(km1) + ddq
            call two_piece_linear_coeffs(x, x0, x1, x2, a0, a1, a2, ierr)
            if (ierr /= 0) return
            
            lnP_edge = a0*s% lnP(km1) + a1*s% lnP(k00) + a2*s% lnP(kp1)
            P_edge = exp(lnP_edge)
            lnd_edge = a0*s% lnd(km1) + a1*s% lnd(k00) + a2*s% lnd(kp1)
            rho_edge = exp(lnd_edge)
            Hp = P_edge/(rho_edge*s% cgrav(k00)*(s% M_center + s% xmstar*q_edge)/r_edge**2)
            if (s% mixing_length_alpha*Hp > zone_dr) Hp = zone_dr/s% mixing_length_alpha
            if (dbg) then
               write(*,*)
               write(*,*)
               write(*,*)
               write(*,1) 'r_edge+f0*Hp', (r_edge+f0*Hp)/Rsun
               write(*,2) 'r(km1)', k00-1, exp(s% lnR(k00-1))/Rsun
               write(*,2) 'r(k00)', k00, exp(s% lnR(k00))/Rsun
               write(*,1) 'r_edge', r_edge/Rsun
               write(*,2) 'r(kp1)', kp1, exp(s% lnR(kp1))/Rsun
               write(*,2) 'r(kp2)', kp1+1, exp(s% lnR(kp1+1))/Rsun
               write(*,1) 'r_edge-f0*Hp', (r_edge-f0*Hp)/Rsun
               write(*,*)
            end if
            
            if (overshoot_toward_center) then
               ! r0 is toward surface from r_edge
               r0 = min(s% r(1), max(s% r(k00), r_edge + f0*Hp))
               if (dbg) then
                  write(*,*) 'get_r0_vc0: overshoot toward center'
                  write(*,2) 'r_edge/Rsun', k00, r_edge/Rsun
                  write(*,2) 'Hp/Rsun', k00, Hp/Rsun
                  write(*,2) 'f0/Rsun', k00, f0
                  write(*,2) 'f0*Hp/Rsun', k00, f0*Hp/Rsun
                  write(*,2) '(r_edge + f0*Hp)//Rsun', k00, (r_edge + f0*Hp)/Rsun
                  write(*,2) 'initial k0 for toward center: k00, r0', k00, r0/Rsun
               end if
               if (r0 == s% r(1)) then
                  k0 = 1
               else
                  k0 = k00
                  do kk = k00, 1, -1
                     if (s% r(kk) > r0) then
                        k0 = kk; exit
                     end if
                     if (s% mixing_type(kk) /= convective_mixing) then
                        if (dbg) write(*,3) 'gap near bottom convective zone', kk, s% model_number
                        if (dbg) write(*,*)
                        k0 = -1; return
                     end if
                  end do
               end if
               
               if (dbg) write(*,2) 'final k0, r(k0)/Rsun', k0, s% r(k0)/Rsun, s% r(k0-1)/Rsun
               if (dbg) write(*,*)
               
            else ! overshoot toward surface
               ! r0 is toward center from r_edge
               r0 = max(s% r(nz), min(s% r(k00+1), r_edge - f0*Hp))
               if (dbg) then
                  write(*,*) 'get_r0_vc0: overshoot toward surface'
                  write(*,2) 'r_edge', k00, r_edge/Rsun
                  write(*,2) 'Hp', k00, Hp/Rsun
                  write(*,2) 'f0', k00, f0
                  write(*,2) 'f0*Hp', k00, f0*Hp/Rsun
                  write(*,2) 'r_edge - f0*Hp', k00, (r_edge - f0*Hp)/Rsun
                  write(*,2) 'initial k0 for toward surface: k00, r0', k00, r0/Rsun
               end if
               k0 = k00+1
               do kk = k00+1, nz-1
                  if (dbg) write(*,2) 's% r(kk+1)', kk+1, s% r(kk+1)
                  if (s% r(kk+1) <= r0) then
                     k0 = kk; exit
                  end if
                  if (s% mixing_type(kk) /= convective_mixing) then
                     if (dbg) write(*,3) 'gap near top of convective zone', kk, s% model_number
                     if (dbg) write(*,*)
                     k0 = -1
                     return
                  end if
               end do
               !if (s% r(kk+1) < r0) r0 = s% r(kk+1)
               if (dbg) write(*,2) 'final k0', k0
               if (dbg) write(*,*)
               
            end if
            if (k0 < 0) return

            if (s% r(k0+1) > r0 .or. r0 > s% r(k0)) then
               write(*,*) 'overshoot_toward_center', overshoot_toward_center
               write(*,2) 's% r(k0) - r0', k0, s% r(k0) - r0
               write(*,2) 'r0 - s% r(k0+1)', k0, r0 - s% r(k0+1)
               write(*,*)
               write(*,2) 's% r(k00) - r0', k00, s% r(k00) - r0
               write(*,2) 'r0 - s% r(k00+1)', k00, r0 - s% r(k00+1)
               write(*,*)
               write(*,2) 'D0', k0, D0
               write(*,2) 'vc0', k0, vc0
               write(*,2) 'frac', k0, frac
               stop 'get_r0_vc0'
            end if
            
            if (s% conv_vel(k0) /= 0 .and. s% conv_vel(k0+1) /= 0) then ! AOK
               frac = (s% r(k0)**3 - r0**3)/(s% r(k0)**3 - s% r(k0+1)**3)
               s% mixing_type_change_dq(k0) = frac*s% dq(k0)
               vc0 = (1-frac)*s% conv_vel(k0) + frac*s% conv_vel(k0+1)
            else
               ! this happens if get a single point convection "zone".
               ! obviously don't want overshooting for this one.
               k0 = -1
               r0 = 0
               vc0 = 0
               D0 = 0
               Hp = 0
               return
            end if
            
            D0 = vc0*s% mixing_length_alpha*Hp/3 ! diffusion coefficient [cm^2/sec]

            if (D0 < 0) then
               ierr = -1
               if (dbg) then
                  write(*,2) 'D0', k0, D0
                  write(*,2) 'vc0', k0, vc0
                  write(*,2) 'frac', k0, frac
                  write(*,2) 's% r(k0)', k0, s% r(k0)
                  write(*,2) 'r0', k0, r0
                  write(*,2) 's% r(k0+1)', k0, s% r(k0+1)
                  stop 'get_r0_vc0'
               end if
            end if
                        
         end subroutine get_r0_vc0
         
         
         subroutine get_D_ov(k, z, D0, ov_scale, hstep, D_ov, ov_limit, cdc_ov, vc_ov, dbg)
            integer, intent(in) :: k
            real(dp), intent(in) :: D0, ov_scale, hstep, ov_limit
            real(dp), intent(out) :: D_ov, cdc_ov, vc_ov
            logical, intent(in) :: dbg
            
            real(dp) :: rho, r, z, fctr
            include 'formats'
            fctr = exp(-2*z/ov_scale)
            D_ov = D0*fctr
            if (hstep >= z) &
               D_ov = max(D_ov, s% step_overshoot_D + D0*s% step_overshoot_D0_coeff)
            if (D_ov < ov_limit) then
               cdc_ov = 0
               vc_ov = 0
               if (dbg) then
                  write(*,2) 'get_D_ov: D_ov < ov_limit', &
                     k, D_ov, ov_limit
                  write(*,2) 'D0', k, D0
                  write(*,2) 'fctr', k, fctr
                  write(*,2) 'hstep', k, hstep
                  write(*,2) 'z', k, z
                  write(*,2) 'step_f_above', k, step_f_above
                  write(*,*)
               end if
               return
            end if
            rho = s% rho(k)
            r = s% r(k)
            cdc_ov = (pi4*r**2*rho)**2*D_ov ! gm^2/sec
            vc_ov = vc0*fctr
         end subroutine get_D_ov

         
      end subroutine add_overshooting


      
      
      end module overshoot
