! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module nse
      
      use star_private_def
      use const_def
      use nse_lib
      use nse_def
      
      implicit none


      contains
      
      
      subroutine set_net_info_to_zero(s, k)
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         s% eps_nuc(k) = 0
         s% d_epsnuc_dlnd(k) = 0
         s% d_epsnuc_dlnT(k) = 0
         s% d_epsnuc_dx(:,k) = 0
         s% dxdt_nuc(:,k) = 0
         s% dxdt_dRho(:,k) = 0
         s% dxdt_dT(:,k) = 0
         s% d_dxdt_dx(:,:,k) = 0
         s% eps_nuc_neu_total(k) = 0
         s% rate_screened(:,:,k) = 0
         s% rate_raw(:,:,k) = 0
         s% reaction_eps_nuc(:,:,k) = 0
         s% eps_nuc_categories(:,:,k) = 0
      end subroutine set_net_info_to_zero
      
      
      subroutine get_nse_info( &
            s, k, species, dt, nse_eps_nuc, nse_eps_nuc_neu_total, &
            nse_d_epsnuc_dlnd, nse_d_epsnuc_dlnT, nse_d_epsnuc_dx, &
            nse_dxdt_nuc, nse_dxdt_dRho, nse_dxdt_dT, nse_d_dxdt_dx, &
            ierr)
         use chem_lib, only: get_composition_info
         
         type (star_info), pointer :: s         
         integer, intent(in) :: k, species
         real(dp), intent(in) :: dt
         real(dp), intent(out) :: &
            nse_eps_nuc, nse_eps_nuc_neu_total, &
            nse_d_epsnuc_dlnd, nse_d_epsnuc_dlnT
         real(dp), intent(out) :: nse_d_epsnuc_dx(:) ! species
         real(dp), intent(out) :: nse_dxdt_nuc(:) ! species
         real(dp), intent(out) :: nse_dxdt_dRho(:) ! species
         real(dp), intent(out) :: nse_dxdt_dT(:) ! species
         real(dp), intent(out) :: nse_d_dxdt_dx(:,:) ! species,species
         integer, intent(out) :: ierr
         
         real(dp) :: eps_nuc, eps_neu, delta, del, dYe_dx, &
            abar, zbar, z2bar, ye, mass_correction, xsum, d_eps_nuc_dYe
         real(dp), dimension(species) :: &
            dxdt_nuc, d_dxdt_dYe, dabar_dx, dzbar_dx, dmc_dx
         integer :: i, j
         
         include 'formats'
         
         ierr = 0
         
         call do1_nse_get( &
            s, k, species, dt, s% xa(1:species,k), &
            s% lnT(k)/ln10, s% lnd(k)/ln10, s% eta(k), s% ye(k), &
            nse_eps_nuc_neu_total, nse_eps_nuc, nse_dxdt_nuc, &
            ierr)
         if (ierr /= 0) return
         
         ! partials wrt lnT
         
         delta = 1d-6
         call do1_nse_get( &
            s, k, species, dt, s% xa(1:species,k), &
            (s% lnT(k) + delta)/ln10, s% lnd(k)/ln10, s% eta(k), s% ye(k), &
            eps_neu, eps_nuc, dxdt_nuc, &
            ierr)
         if (ierr /= 0) return
         nse_d_epsnuc_dlnT = (eps_nuc - nse_eps_nuc)/delta
         del = exp(s% lnT(k) + delta) - s% T(k)
         do j=1,species
            nse_dxdt_dT(j) = (dxdt_nuc(j) - nse_dxdt_nuc(j))/del
         end do
         
         ! partials wrt lnd
         
         delta = 1d-6
         call do1_nse_get( &
            s, k, species, dt, s% xa(1:species,k), &
            s% lnT(k)/ln10, (s% lnd(k) + delta)/ln10, s% eta(k), s% ye(k), &
            eps_neu, eps_nuc, dxdt_nuc, &
            ierr)
         if (ierr /= 0) return
         nse_d_epsnuc_dlnd = (eps_nuc - nse_eps_nuc)/delta
         del = exp(s% lnd(k) + delta) - s% rho(k)
         do j=1,species
            nse_dxdt_dRho(j) = (dxdt_nuc(j) - nse_dxdt_nuc(j))/del
         end do
         
         ! partials wrt ye
         
         delta = -1d-5
         call do1_nse_get( &
            s, k, species, dt, s% xa(1:species,k), &
            s% lnT(k)/ln10, s% lnd(k)/ln10, s% eta(k), s% ye(k) + delta, &
            eps_neu, eps_nuc, dxdt_nuc, &
            ierr)
         if (ierr /= 0) return
         d_eps_nuc_dYe = (eps_nuc - nse_eps_nuc)/delta
         
         do j=1,species
            d_dxdt_dYe(j) = (dxdt_nuc(j) - nse_dxdt_nuc(j))/delta
         end do         
         
         ! partials wrt x
         
         call get_composition_info( &
            species, s% chem_id, s% xa(1:species,k), s% X(k), s% Y(k), &
            abar, zbar, z2bar, ye, mass_correction, xsum, .false., &
            dabar_dx, dzbar_dx, dmc_dx)  
         
         do j=1,species
            dYe_dx = dzbar_dx(j)/abar - ye*dabar_dx(j)/abar ! ye = zbar/abar
            nse_d_epsnuc_dx(j) = d_eps_nuc_dYe*dYe_dx
            do i=1,species
               nse_d_dxdt_dx(i,j) = d_dxdt_dYe(i)*dYe_dx
            end do
         end do
      
      end subroutine get_nse_info
      
      
      subroutine do1_nse_get( &
            s, k, species, dt_in, X0, &
            logT, logRho, eta, ye_init, &
            nse_eps_nuc, nse_eps_neu, nse_dxdt_nuc, &
            ierr)
         use chem_def, only: chem_isos
         
         type (star_info), pointer :: s         
         integer, intent(in) :: k, species
         real(dp), intent(in) :: dt_in, logT, logRho, eta, ye_init
         real(dp), intent(in) :: X0(:) ! (species)
         real(dp), intent(out) :: nse_eps_nuc, nse_eps_neu
         real(dp), intent(out) :: nse_dxdt_nuc(:) ! species
         integer, intent(out) :: ierr
         
         real(dp) :: dt, ye_final, nse_eps_binding
         real(dp), dimension(:), pointer :: X_net
         integer :: i, j
         real(dp), dimension(:), pointer :: X_nse_init
         logical :: dbg, okay
         
         include 'formats'
         
         dbg = .false.
         !dbg = .true.
         !dbg = (s% model_number == 1137)
         
         if (dbg) write(*,3) 'enter do1_nse_get', k, s% model_number, s% T(k)
         !stop 'do1_nse_get'
         
         ierr = 0
         dt = max(dt_in, 1d0)
         
         call do_alloc(ierr)
         if (ierr /= 0) return

      	if (dbg) write(*,2) 'do1_nse_get: call check_xnse', k

         !write(*,2) 'do1_nse_get: call get_nse_abundances', k
         call get_nse_abundances(logT, logRho, ye_init, X_nse_init, ierr)
         !write(*,2) 'do1_nse_get: done get_nse_abundances', k
      	if (ierr /= 0) then
   	      if (dbg) then
   	         write(*,2) 'do1_nse_get: FAILED in get_nse_abundances', k
   	         stop 'do1_nse_get'
   	      end if
   	      call dealloc
      	   return
      	end if
         
         !write(*,2) 'do1_nse_get: call get_nse', k
         call get_nse( &
            s, dt, logT, logRho, ye_init, eta, X_nse_init, &
            s% species, s% net_iso, s% chem_id, &
            nse_eps_binding, nse_eps_neu, &
            X_net, ye_final, &
            k, dbg, ierr)
      	if (ierr /= 0) then
   	      if (dbg) then
   	         write(*,2) 'do1_nse_get: FAILED in get_nse', k
   	         stop 'do1_nse_get'
   	      end if
   	      call dealloc
      	   return
      	end if
   	   if (dbg) write(*,2) 'done get_nse', k
   	   
   	   ! check X_net
   	   okay = .true.
   	   do i=1,species
   	      if (X_net(i) < 0) then
   	         okay = .false.
   	         write(*,3)' X_net ' // trim(chem_isos% name(s% chem_id(i))), &
   	            i, k, X_net(i)
   	      end if
   	   end do
   	   if (.not. okay) then
   	      write(*,2) 'sum(X_net)', k, sum(X_net(1:species))
   	      stop 'do1_nse_get'
   	   end if
            
         nse_eps_nuc = nse_eps_binding - nse_eps_neu
         do i=1,species
            nse_dxdt_nuc(i) = (X_net(i) - X0(i))/dt
         end do
         
         call dealloc
         
         
         contains
         
         
         subroutine do_alloc(ierr)
            use alloc
            integer, intent(out) :: ierr
            call get_work_array(s, X_net, species, 0, 'nse', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, X_nse_init, nse_species, 0, 'nse', ierr)
            if (ierr /= 0) return                        
            ierr = 0
         end subroutine do_alloc
         
         
         subroutine dealloc
            use alloc
            call return_work_array(s, X_net, 'nse')
            call return_work_array(s, X_nse_init, 'nse')
         end subroutine dealloc


      end subroutine do1_nse_get
      
      
      subroutine get_nse( &
            s, dt, logT, logRho, ye_init, eta, X_nse_init, &
            species, net_isos, chem_id, &
            eps_binding, nse_eps_neu, X_net, ye_final, &
            k, dbg, ierr)
         type (star_info), pointer :: s         
         real(dp), intent(in) :: dt, logT, logRho, ye_init, eta
         real(dp), pointer :: X_nse_init(:)
         integer, intent(in) :: species
         integer, pointer, intent(in) :: net_isos(:), chem_id(:)
         real(dp), intent(out) :: eps_binding, nse_eps_neu, ye_final
         real(dp), pointer, intent(out) :: X_net(:)
         integer, intent(in) :: k
         logical, intent(in) :: dbg
         integer, intent(out) :: ierr
         
         real(dp), dimension(:), pointer :: X_nse_final
         
         include 'formats'
         
         ierr = 0
         eps_binding = 0
         nse_eps_neu = 0
         ye_final = 0
         
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         if (dbg) write(*,2) 'call advance_nse', k
         call advance_nse( &
            dt, X_nse_init, ye_init, logT, logRho, eta, &
            s% nse_atol, s% nse_newt_imax, s% nse_maxiters, &
            X_nse_final, eps_binding, nse_eps_neu, ye_final, &
            ierr)
      	if (ierr /= 0) then
      	   if (dbg) then
      	      write(*,2) 'failed in advance_nse', k
      	      write(*,2) 'dt', k, dt
      	      write(*,2) 'ye_init', k, ye_init
      	      write(*,2) 'logT', k, logT
      	      write(*,2) 'logRho', k, logRho
      	      write(*,2) 'eta', k, eta
      	      write(*,2) 's% nse_atol', k, s% nse_atol
      	      write(*,3) 's% nse_newt_imax', k, s% nse_newt_imax
      	      write(*,3) 's% nse_maxiters', k, s% nse_maxiters
      	      stop 'get_nse'
      	   end if
      	   call dealloc
      	   return
      	end if
      	
      	if (.false. .and. logT > 9.5 .and. k == s% nz) then
         	write(*,2) 'get_nse nz dYe, Ye_init, Ye_final', &
         	   s% model_number, ye_final - ye_init, ye_init, ye_final
      	end if
      	
      	if (dbg) then
         	write(*,*)
         	write(*,1) 'get_nse ye_init', ye_init
         	write(*,1) 'get_nse ye_final', ye_final
         	write(*,1) 'get_nse eps_binding', eps_binding
         	write(*,1) 'get_nse nse_eps_neu', nse_eps_neu
         	write(*,1) 'get_nse dt', dt
         	write(*,1) 'get_nse logT', logT
         	write(*,1) 'get_nse logRho', logRho
         	write(*,1) 'get_nse eta', eta
      	end if
         
         if (dbg) write(*,2) 'call convert_from_nse_abundances', k
         call convert_from_nse_abundances( &
            X_nse_final, species, net_isos, chem_id, &
            X_net, ierr)
      	if (ierr /= 0) then
      	   if (dbg) write(*,2) 'failed in convert_from_nse_abundances', k
      	   call dealloc
      	   return
      	end if
      	
         call dealloc
         
         
         contains
         
         subroutine do_alloc(ierr)
            use alloc
            integer, intent(out) :: ierr
            call get_work_array(s, X_nse_final, nse_species, 0, 'nse', ierr)
         end subroutine do_alloc
         
         subroutine dealloc
            use alloc
            call return_work_array(s, X_nse_final, 'nse')
         end subroutine dealloc
         
      end subroutine get_nse


      end module nse

