! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module net
      
      use star_private_def
      use const_def
      
      implicit none


      contains
      
      
      subroutine do_net(s, nzlo, nzhi, dt, ierr)
         use net_lib, only: net_work_size
         use alloc
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         logical, parameter :: use_omp = .true.
         integer :: k, op_err, net_lwork, j, jj, cnt, kmax
         real(dp) :: abs_e, abs_e_dm, abs_e_limit, max_abs_e_dm, dm_limit, e_limit
         integer, pointer :: ks(:)
         logical, parameter :: only_dlnT = .false.
         logical :: okay
         
         include 'formats'
         
         ierr = 0
         net_lwork = net_work_size(s% net_handle, ierr)

         if (nzlo == nzhi) then
            call do1_net( &
               s, nzlo, s% species, s% num_reactions, net_lwork, dt, ierr)
            return
         end if
         
         if (use_omp) then
            okay = .true.
!$OMP PARALLEL DO PRIVATE(k,op_err) SCHEDULE(STATIC,10)
            do k = nzlo, nzhi
               if (.not. okay) cycle
               op_err = 0
               call do1_net( &
                  s, k, s% species, s% num_reactions, net_lwork, dt, op_err)
               if (op_err /= 0) okay = .false.
            end do
!$OMP END PARALLEL DO
            if (.not. okay) ierr = -1
         else
            do k = nzlo, nzhi
               call do1_net( &
                  s, k, s% species, s% num_reactions, net_lwork, dt, ierr)
               if (ierr /= 0) exit
            end do
         end if
         
         
      end subroutine do_net


      subroutine do1_net( &
            s, k, species, num_reactions, net_lwork, dt, ierr)

         use rates_def
         use net_lib, only: net_get, get_net_reaction_table_ptr
         use chem_def
         use eos_def, only : i_eta
         use nse_def, only: min_T_for_NSE
         use nse, only: get_nse_info, set_net_info_to_zero
         use utils_lib,only: &
            is_bad_num, return_nan, realloc_double, realloc_double3
         type (star_info), pointer :: s         
         integer, intent(in) :: k, species, num_reactions, net_lwork
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr

         integer :: i, j, screening_mode
         real(dp) :: log10_rho, log10_T, alfa, beta, &
            d_eps_nuc_dRho, d_eps_nuc_dT, &
            category_factors(num_categories), cat_factor
            
         real(dp) :: &
            nse_eps_nuc, nse_eps_nuc_neu_total, &
            nse_d_epsnuc_dlnd, nse_d_epsnuc_dlnT
         real(dp), pointer, dimension(:) :: &
            nse_dxdt_nuc, nse_dxdt_dRho, nse_dxdt_dT, &
            nse_d_epsnuc_dx, nse_d_dxdt_dx1
         real(dp), pointer :: nse_d_dxdt_dx(:,:) ! (species,species)
            
         real(dp) :: &
            net_eps_nuc, net_eps_nuc_neu_total, &
            net_d_epsnuc_dlnd, net_d_epsnuc_dlnT
         real(dp), pointer, dimension(:) :: &
            net_dxdt_nuc, net_dxdt_dRho, net_dxdt_dT, &
            net_d_epsnuc_dx, net_d_dxdt_dx1
         real(dp), pointer :: net_d_dxdt_dx(:,:) ! (species,species)
         
         character (len=100) :: message
         integer, pointer :: net_reaction_ptr(:) 
         real(dp), pointer :: reaction_neuQs(:), net_work(:)
         real(dp), target :: net_work_ary(net_lwork)
      
         real(dp), parameter :: logT_lim1 = 7.8d0, logT_lim2 = 7.7d0
         real(dp), parameter :: logT_lim3 = 8.8d0, logT_lim4 = 8.7d0

         integer :: ra(9), sz
         real(dp) :: rfac(9), eps_nuc_factor
         
         logical, parameter :: dbg = .true.

         include 'formats'
         
         ierr = 0
            
         if (s% nse_fraction(k) < 1d0) then
         
            net_work => net_work_ary
         
            log10_rho = s% lnd(k)/ln10
            log10_T = s% lnT(k)/ln10
         
            category_factors(:) = s% category_factors(:)
         
            if (s% net_rate_factor /= 1) &
               category_factors(:) = category_factors(:)*s% net_rate_factor
         
            ! once H depleted, turn off hydrogen burning by PP where doing advanced burning
            ! else solver seems to hallucinate he3(he3,2p)he4 and other bad things
            if (s% suppress_dubious_PP_burning .and. s% net_iso(ih1) /= 0 .and. s% net_iso(ihe3) /= 0) then
               if (s% xa(s% net_iso(ih1),k) < 1d-30 .and. &
                     s% xa(s% net_iso(ihe3),k) < 1d-30 .and. &
                     category_factors(ipp) == 1) then
                  if (log10_T > logT_lim1) then
                     category_factors(ipp) = 0
                  else if (log10_T > logT_lim2) then
                     cat_factor = (logT_lim1 - log10_T)/(logT_lim1 - logT_lim2)
                     category_factors(ipp) = cat_factor
                  end if
               end if
            end if

            ! once N14 depleted, turn off n14 burning where doing advanced burning
            if (s% suppress_dubious_N_burning .and. s% net_iso(in14) /= 0) then
               if (s% xa(s% net_iso(in14),k) < 1d-30 .and. &
                     category_factors(i_burn_n) == 1) then
                  if (log10_T > logT_lim3) then
                     category_factors(i_burn_n) = 0
                  else if (log10_T > logT_lim4) then
                     cat_factor = (logT_lim3 - log10_T)/(logT_lim3 - logT_lim4)
                     category_factors(i_burn_n) = cat_factor
                  end if
               end if 
            end if
         
            if (s% net_iso(icr56) /= 0) then ! limit rate for fe56 + 2e => cr56
               call get_net_reaction_table_ptr(s% net_handle, net_reaction_ptr, ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed in do1_net call on get_net_reaction_table_ptr'
                  return
               end if
               j = net_reaction_ptr(irfe56ec_fake_to_cr56)
               if (j /= 0) then
                  if (s% xa(s% net_iso(icr56),s% nz) >= s% max_center_cr56_for_fe56ec) then
                     s% rate_factors(j) = 0
                  else
                     s% rate_factors(j) = s% rate_factor_for_fe56ec_to_cr56
                  end if
               end if
            end if
         
            if (s% abar(k) > s% max_abar_for_burning) then
               category_factors = 0
            end if

            screening_mode = get_screening_mode(s,ierr)         
            if (ierr /= 0) then
               write(*,*) 'unknown string for screening_mode: ' // trim(s% screening_mode)
            
               stop 'do1_net'
            
            
               return
            end if
         
            if (s% reaction_neuQs_factor /= 1d0) then
               sz = size(std_reaction_neuQs,dim=1)
               allocate(reaction_neuQs(sz))
               do j=1,sz
                  reaction_neuQs(j) = std_reaction_neuQs(j)*s% reaction_neuQs_factor
               end do
               !write(*,1) 'reaction_neuQs_factor', s% reaction_neuQs_factor
            else
               reaction_neuQs => std_reaction_neuQs
            end if
         
            call net_get( &
                  s% net_handle, species, num_reactions, s% xa(1:species,k), &
                  s% T(k), log10_T, s% rho(k), log10_Rho, &
                  s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
                  s% eta(k), s% d_eos_dlnd(i_eta,k), s% d_eos_dlnT(i_eta,k), &
                  s% rate_factors, category_factors, std_reaction_Qs, reaction_neuQs, &
                  s% eps_nuc(k), d_eps_nuc_dRho, d_eps_nuc_dT, s% d_epsnuc_dx(:,k), & 
                  s% dxdt_nuc(:,k), s% dxdt_dRho(:,k), s% dxdt_dT(:,k), s% d_dxdt_dx(:,:,k), &
                  screening_mode, s% theta_e(k), &
                  s% rate_screened(:,:,k), s% rate_raw(:,:,k), &
                  s% reaction_eps_nuc(:,:,k), s% eps_nuc_categories(:,:,k), s% eps_nuc_neu_total(k), &
                  net_lwork, net_work, ierr)

            if (s% reaction_neuQs_factor /= 1d0) deallocate(reaction_neuQs)
         
            if (ierr /= 0) then
               write(message,*) 'do1_net: net_get failure for cell ', k
               if (s% report_ierr) then
                  write(*,*) trim(message)
                  call show_stuff(s,k)
               end if
               return
            end if
         
         
         
         
            if (.false. .and. k == s% nz) then
               write(*,2) 'eps_nuc', k, s% eps_nuc(k)
               write(*,2) 'logT', k, log10_T
               write(*,2) 'T', k, s% T(k)
               write(*,2) 'logRho', k, log10_Rho
               write(*,2) 'Rho', k, s% rho(k)
               call show_stuff(s,k)
               write(*,*)
               stop 'do1_net'
            end if
         
         
         
         
         
         
            if (is_bad_num(s% eps_nuc(k))) then
               !write(message,*) 'do1_net: net_get for cell ', k
               !call show_stuff(s,k)
               !write(*,2) 'eps_nuc T Rho', k, s% eps_nuc(k), log10_T, log10_Rho
               !stop 'do1_net'
               ierr = -1
               return
            end if

            s% d_abar_dlnd(k) = 0
            s% d_abar_dlnT(k) = 0
            s% d_zbar_dlnd(k) = 0
            s% d_zbar_dlnT(k) = 0
      
            s% d_epsnuc_dlnd(k) = d_eps_nuc_dRho*s% rho(k)
            s% d_epsnuc_dlnT(k) = d_eps_nuc_dT*s% T(k)
            
            eps_nuc_factor = s% eps_nuc_factor
            if (eps_nuc_factor /= 1d0) then
               s% eps_nuc(k) = s% eps_nuc(k)*eps_nuc_factor
               s% d_epsnuc_dlnd(k) = s% d_epsnuc_dlnd(k)*eps_nuc_factor
               s% d_epsnuc_dlnT(k) = s% d_epsnuc_dlnT(k)*eps_nuc_factor
               s% d_epsnuc_dx(:,k) = s% d_epsnuc_dx(:,k)*eps_nuc_factor
               s% reaction_eps_nuc(:,:,k) = s% reaction_eps_nuc(:,:,k)*eps_nuc_factor
               s% eps_nuc_categories(:,:,k) = s% eps_nuc_categories(:,:,k)*eps_nuc_factor
            end if

            if (s% dxdt_nuc_factor /= 1d0) then
               s% dxdt_nuc(:,k) = s% dxdt_nuc(:,k)*s% dxdt_nuc_factor
               s% dxdt_dRho(:,k) = s% dxdt_dRho(:,k)*s% dxdt_nuc_factor
               s% dxdt_dT(:,k) = s% dxdt_dT(:,k)*s% dxdt_nuc_factor
               s% d_dxdt_dx(:,:,k) = s% d_dxdt_dx(:,:,k)*s% dxdt_nuc_factor
            end if
        
            if (is_bad_num(s% eps_nuc(k))) then
               write(*,*) 'k', k
               write(*,1) 's% eps_nuc(k)', s% eps_nuc(k)
               ierr = -1
               call show_stuff(s,k)
               write(*,*) '(is_bad_num(s% eps_nuc(k)))'
               write(*,*) 'failed in do1_net'
               return
            end if
         
            if (.false. .and. k == s% nz) then
               write(*,2) 's% T(k)', k, s% T(k)
               write(*,2) 's% lnT(k)/ln10', k, s% lnT(k)/ln10
               write(*,2) 's% rho(k)', k, s% rho(k)
               write(*,2) 's% lnd(k)/ln10', k, s% lnd(k)/ln10
               write(*,2) 's% eps_nuc(k)', k, s% eps_nuc(k)
               write(*,*)
               do j=1,num_categories
                  write(*,2) trim(category_name(j)), k, s% eps_nuc_categories(i_rate, j, k)
               end do
               write(*,*)
            end if

            if (.false. .and. k == 1469) then
               write(*,2) 'eos grada(k)', k, s% grada(k)
               do j=1,s% species
                  if (.true. .or. s% xa(j,k) > 1d-9) &
                     write(*,3) 'eos xin(net_iso(i' // &
                        trim(chem_isos% name(s% chem_id(j))) // '))= ', j,k,s% xa(j,k)
               end do
               write(*,*)
            end if
         
            ! DEBUGGING
            if (.false. .and. k == s% nz .and. s% model_number == 2559) then
               write(*,*)
               call show_stuff(s,k)
               write(*,2) 's% eps_nuc(k)', k, s% eps_nuc(k)
               write(*,2) 'sum xa', k, sum(s% xa(:,k))
               if (s% eps_nuc(k) > 1d18) stop 'do1_net'
               !return
            
            
            
               write(*,2) 's% d_epsnuc_dlnd(k)', k, s% d_epsnuc_dlnd(k)
               write(*,2) 's% d_epsnuc_dlnT(k)', k, s% d_epsnuc_dlnT(k)
               do j=1,species
                  write(*,1) 'd_epsnuc_dx ' // trim(chem_isos% name(s% chem_id(j))), &
                     s% d_epsnuc_dx(j,k)
               end do
               write(*,*)
               do j=1,species
                  write(*,1) 'd_dxdt_dx(1,:) ' // trim(chem_isos% name(s% chem_id(j))), &
                     s% d_dxdt_dx(1,j,k)
               end do
               write(*,*)
               write(*,*)
               write(*,*)
            
               stop 'debug: do1_net'
               return

            end if

            if (s% dbg_control > 0 .or. k == -s% nz) then
               write(*,*)
               call show_stuff(s,k)
               write(*,1) 's% eps_nuc(k)', s% eps_nuc(k)
               write(*,1) 's% d_epsnuc_dlnd(k)', s% d_epsnuc_dlnd(k)
               write(*,1) 's% d_epsnuc_dlnT(k)', s% d_epsnuc_dlnT(k)
               write(*,*)
               write(*,*) 'do1_net'
               ierr = -1
            end if
         
            if (.false.) call show_stuff(s,k)
         
            !call check_Z

         end if ! nse_fraction < 1d0

         if (s% nse_fraction(k) == 0d0) return
         
         if (s% nse_fraction(k) == 1d0) then ! just nse
            call set_net_info_to_zero(s, k)
            call get_nse_info( &
               s, k, species, dt, s% eps_nuc(k), s% eps_nuc_neu_total(k), &
               s% d_epsnuc_dlnd(k), s% d_epsnuc_dlnT(k), s% d_epsnuc_dx(:,k), &
               s% dxdt_nuc(:,k), s% dxdt_dRho(:,k), s% dxdt_dT(:,k), &
               s% d_dxdt_dx(:,:,k), ierr)
            if (.false. .and. dt > 0 .and. k == s% nz) then
               write(*,2) 'sum nse dt*dxdt_nuc', k, dt*sum(s% dxdt_nuc(:,k))
            end if
            return
         end if
         
         ! combine net and nse
         alfa = s% nse_fraction(k)
         beta = 1d0 - alfa
         
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         call save_net_info         
         
         call get_nse_info( &
            s, k, species, dt, nse_eps_nuc, nse_eps_nuc_neu_total, &
            nse_d_epsnuc_dlnd, nse_d_epsnuc_dlnT, nse_d_epsnuc_dx, &
            nse_dxdt_nuc, nse_dxdt_dRho, nse_dxdt_dT, nse_d_dxdt_dx, &
            ierr)
         if (ierr /= 0) then
            call dealloc
            return
         end if
            
         s% eps_nuc(k) = alfa*nse_eps_nuc + beta*net_eps_nuc
         s% eps_nuc_neu_total(k) = &
            alfa*nse_eps_nuc_neu_total + beta*net_eps_nuc_neu_total

         s% d_epsnuc_dlnd(k) = &
            alfa*nse_d_epsnuc_dlnd + beta*net_d_epsnuc_dlnd
         s% d_epsnuc_dlnT(k) = &
            alfa*nse_d_epsnuc_dlnT + beta*net_d_epsnuc_dlnT

         do j=1,species
            s% d_epsnuc_dx(j,k) = &
               alfa*nse_d_epsnuc_dx(j) + beta*net_d_epsnuc_dx(j)
            s% dxdt_nuc(j,k) = &
               alfa*nse_dxdt_nuc(j) + beta*net_dxdt_nuc(j)
            s% dxdt_dRho(j,k) = &
               alfa*nse_dxdt_dRho(j) + beta*net_dxdt_dRho(j)
            s% dxdt_dT(j,k) = &
               alfa*nse_dxdt_dT(j) + beta*net_dxdt_dT(j)
            do i=1,species
               s% d_dxdt_dx(i,j,k) = &
                  alfa*nse_d_dxdt_dx(i,j) + beta*net_d_dxdt_dx(i,j)
            end do
         end do
         
         call dealloc
         
         
         contains
         
         
         subroutine save_net_info
            integer :: j, i
            net_eps_nuc = s% eps_nuc(k)
            net_eps_nuc_neu_total = s% eps_nuc_neu_total(k)
            net_d_epsnuc_dlnd = s% d_epsnuc_dlnd(k)
            net_d_epsnuc_dlnT = s% d_epsnuc_dlnT(k)
            do j=1,species
               net_d_epsnuc_dx(j) = s% d_epsnuc_dx(j,k)
               net_dxdt_nuc(j) = s% dxdt_nuc(j,k)
               net_dxdt_dRho(j) = s% dxdt_dRho(j,k)
               net_dxdt_dT(j) = s% dxdt_dT(j,k)
               do i=1,species
                  net_d_dxdt_dx(i,j) = s% d_dxdt_dx(i,j,k)
               end do
            end do
         end subroutine save_net_info
            
         
         subroutine do_alloc(ierr)
            use alloc
            integer, intent(out) :: ierr
            ierr = 0
            call get_work_array(s, nse_dxdt_nuc, species, 0, 'do_net_nse', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, nse_d_epsnuc_dx, species, 0, 'do_net_nse', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, nse_dxdt_dRho, species, 0, 'do_net_nse', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, nse_dxdt_dT, species, 0, 'do_net_nse', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, nse_d_dxdt_dx1, species**2, 0, 'do_net_nse', ierr)
            if (ierr /= 0) return            
            nse_d_dxdt_dx(1:species,1:species) => nse_d_dxdt_dx1(1:species**2)         
            call get_work_array(s, net_dxdt_nuc, species, 0, 'do_net_nse', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, net_d_epsnuc_dx, species, 0, 'do_net_nse', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, net_dxdt_dRho, species, 0, 'do_net_nse', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, net_dxdt_dT, species, 0, 'do_net_nse', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, net_d_dxdt_dx1, species**2, 0, 'do_net_nse', ierr)
            if (ierr /= 0) return            
            net_d_dxdt_dx(1:species,1:species) => net_d_dxdt_dx1(1:species**2)         
         end subroutine do_alloc
         
         
         subroutine dealloc
            use alloc
            call return_work_array(s, nse_dxdt_nuc, 'do_net_nse')
            call return_work_array(s, nse_d_epsnuc_dx, 'do_net_nse')
            call return_work_array(s, nse_dxdt_dRho, 'do_net_nse')
            call return_work_array(s, nse_dxdt_dT, 'do_net_nse')
            call return_work_array(s, nse_d_dxdt_dx1, 'do_net_nse')
            call return_work_array(s, net_dxdt_nuc, 'do_net_nse')
            call return_work_array(s, net_d_epsnuc_dx, 'do_net_nse')
            call return_work_array(s, net_dxdt_dRho, 'do_net_nse')
            call return_work_array(s, net_dxdt_dT, 'do_net_nse')
            call return_work_array(s, net_d_dxdt_dx1, 'do_net_nse')
         end subroutine dealloc
         

         subroutine check_Z
            use chem_lib, only: basic_composition_info
            integer :: species
            real(dp) :: xsum, mass_correction, abar, zbar, z2bar, ye, prev_z, z, y, xh
            include 'formats'
            species = s% species
            call basic_composition_info( &
               species, s% chem_id, s% xa(1:species,k), &
               xh, y, abar, zbar, z2bar, ye, mass_correction, xsum)  
            z = 1d0 - (xh + y)
            prev_z = 1d0 - (s% X(k) + s% Y(k))
            if (z > prev_z - 1d-6) return
            call show_stuff(s,k)
            write(*,*)
            write(*,3) 'z dropped', s% model_number, k, prev_z-z, prev_z, z
            stop 'check_Z'
         end subroutine check_Z
      
      end subroutine do1_net
      
      
      subroutine show_stuff(s,k)
         use chem_def
         use rates_def
         use net_lib, only: get_reaction_id_table_ptr
         use num_lib, only: safe_log10, qsort
         type (star_info), pointer :: s         
         integer, intent(in) :: k

         integer, pointer :: reaction_id(:) ! maps net reaction number to reaction id
         integer :: i, j, ierr, species, num_reactions
         real(dp) :: log10_Rho, log10_T
         real(dp), pointer :: v(:) 
         integer, pointer :: index(:) 
         include 'formats'
         
         logical, parameter :: do_sort = .true.
         
         ierr = 0
         species = s% species
         num_reactions = s% num_reactions
         log10_T = s% lnT(k)/ln10
         log10_Rho = s% lnd(k)/ln10
         
         call get_reaction_id_table_ptr(s% net_handle, reaction_id, ierr) 
         if (ierr /= 0) return
         
         write(*,2) 'k', k
         write(*,*)
         write(*,*) 'net_name ', trim(s% net_name)
         write(*,*) 'species', species
         i = max(species, num_reactions)
         allocate(v(i), index(i))
         if (.true.) then
            write(*, *)
            if (do_sort) then
               do j=1,num_reactions
                  v(j) = abs(s% reaction_eps_nuc(i_rate, j, k))
               end do
               call qsort(index, num_reactions, v)
            else
               do j=1,num_reactions
                  index(j) = j
               end do
            end if
            write(*,*) 'reaction eps_nuc'
            do i=1,num_reactions
               j = index(num_reactions+1-i)
               if (abs(s% reaction_eps_nuc(i_rate, j, k)) > 1d0) &
               write(*,2) trim(reaction_Name(reaction_id(j))) // &
                  ' ' // category_name(reaction_categories(reaction_id(j))), &
                  k, s% reaction_eps_nuc(i_rate, j, k)
            end do
            if (.false.) then
               write(*, *)
               write(*,*) 'reaction d_lneps_nuc_dlnT'
               do j=1,num_reactions
                  if (abs(s% reaction_eps_nuc(i_rate, j, k)) > 1d0) &
                  write(*,3) trim(reaction_Name(reaction_id(j))) // &
                     ' ' // category_name(reaction_categories(reaction_id(j))), &
                     j, k, s% reaction_eps_nuc(i_rate_dT, j, k)*s% T(k)/ &
                        s% reaction_eps_nuc(i_rate, j, k)
               end do
               write(*, *)
               write(*,*) 'reaction d_lneps_nuc_dlnd'
               do j=1,num_reactions
                  if (abs(s% reaction_eps_nuc(i_rate, j, k)) > 1d0) &
                  write(*,3) trim(reaction_Name(reaction_id(j))) // &
                     ' ' // category_name(reaction_categories(reaction_id(j))), &
                     j, k, s% reaction_eps_nuc(i_rate_dRho, j, k)*s% rho(k)/ &
                        s% reaction_eps_nuc(i_rate, j, k)
               end do
               !return
            end if
         end if
         write(*,*)
         if (.true.) then
            write(*, *)
            if (do_sort) then
               do j=1,num_reactions
                  v(j) = abs(s% rate_raw(i_rate, j, k))
               end do
               call qsort(index, num_reactions, v)
            else
               do j=1,num_reactions
                  index(j) = j
               end do
            end if
         end if
         
         if (.false.) then
            write(*,*) 'reaction rate_raw'
            do i=1,num_reactions
               j = index(num_reactions+1-i)
               write(*,2) trim(reaction_Name(reaction_id(j))), k, s% rate_raw(i_rate, j, k)
            end do
         end if
         
         if (.false.) then
            write(*,*)
            write(*,*) 'screened rates'
            do j=1,num_reactions
               write(*,3) 'screened rate ' // trim(reaction_Name(reaction_id(j))), &
                  j, k, s% rate_screened(i_rate, j, k)
            end do
         end if
         
         if (.true.) then
            write(*,*)
            do j=1,species
               write(*,2) 'dxdt ' // trim(chem_isos% name(s% chem_id(j))), k, s% dxdt_nuc(j, k)
            end do
         end if
         write(*,*)
         
         if (.false.) then
            write(*,*)
            do j=1,species
               write(*,2) 'dt*dxdt ' // trim(chem_isos% name(s% chem_id(j))), k, &
                  s% dt * s% dxdt_nuc(j, k)
            end do
         end if
         
         
         if (do_sort) then
            do j=1,species
               v(j) = s% xa(j,k)
            end do
            call qsort(index, species, v)
         else
            do j=1,num_reactions
               index(j) = j
            end do
         end if
         write(*,*)
         do i=1,species
            j = index(species+1-i)
            if (.true. .or. s% xa(j,k) > 1d-9) &
               write(*,1) 'xin(net_iso(i' // trim(chem_isos% name(s% chem_id(j))) // '))= ', s% xa(j,k)
         end do
         write(*,*)
         write(*,1) 'T =', s% T(k)
         write(*,1) 'logT =', log10_T
         write(*,1) 'rho =', s% rho(k)
         write(*,1) 'logRho =', log10_Rho
         write(*,1) 'abar =', s% abar(k)
         write(*,1) 'zbar =', s% zbar(k)
         write(*,1) 'z2bar =', s% z2bar(k)
         write(*,1) 'ye =', s% ye(k)
         write(*,1) 'eta =', s% eta(k)
         write(*,*) 'screening_mode = ' // trim(s% screening_mode)
         write(*,1) 'theta_e =', s% theta_e(k)
         
         return
         stop 'do1_net'
         
         if (.false.) then
            write(*,*)
            do j=1,num_categories
               write(*,2) trim(category_name(j)), k, s% eps_nuc_categories(i_rate, j, k)
            end do
         end if
         if (.true.) then
            write(*, *)
            write(*,*) 'raw rates'
            do j=1,num_reactions
               write(*,2) 'raw rate ' // trim(reaction_Name(reaction_id(j))), &
                  k, s% rate_raw(i_rate, j, k)
            end do
         end if
         if (.false.) then
            write(*, *)
            write(*,*) 'raw rates dlnT'
            do j=1,num_reactions
               write(*,2) 'raw rate dlnT ' // trim(reaction_Name(reaction_id(j))), &
                  k, s% rate_raw(i_rate_dT, j, k)*s% T(k)
            end do
            write(*, *)
         end if
         
         
         
         !return
         
         
         if (.true.) then
            write(*, *)
            write(*,*) 'screened rates dlnT'
            do j=1,num_reactions
               write(*,2) 'screened rate dlnT ' // trim(reaction_Name(reaction_id(j))), &
                  k, s% rate_screened(i_rate_dT, j, k)*s% T(k)
            end do
         end if
         if (.true.) then
            write(*, *)
            write(*,*) 'screened rates dlnRho'
            do j=1,num_reactions
               write(*,2) 'screened rate dlnRho ' // trim(reaction_Name(reaction_id(j))), &
                  k, s% rate_screened(i_rate_dRho, j, k)*s% rho(k)
            end do
         end if
         if (.false.) then
            write(*,*)
            do j=1,species
               write(*,2) 'dxdt_dlnRho ' // trim(chem_isos% name(s% chem_id(j))), k, s% dxdt_dRho(j, k)*s% Rho(k)
            end do
         end if
         if (.false.) then
            write(*,*)
            do j=1,species
               write(*,2) 'dxdt_dlnT ' // trim(chem_isos% name(s% chem_id(j))), k, s% dxdt_dT(j, k)*s% T(k)
            end do
         end if
         write(*,*) 'X'
         write(*,*)
         write(*,2) 'sum(s% xa(1:species,k))', k, sum(s% xa(1:species,k))
         write(*,2) '1 - sum(s% xa(1:species,k))', k, 1 - sum(s% xa(1:species,k))
         !do j=1,species
         !   write(*,1) trim(chem_isos% name(s% chem_id(j))), s% xa(j,k)
         !end do
         write(*,*)
         write(*,2) 'nnuc = ', species
         write(*,*)
         do j=1,species
            write(*,'(a)') '      j' // trim(chem_isos% name(s% chem_id(j))) // ' = ' // &
               'get_nuclide_index_in_set("' // trim(chem_isos% name(s% chem_id(j))) // '", set)'
         end do
         write(*,*) 
         do j=1,species
            write(*,'(a)',advance='no') 'j' // trim(chem_isos% name(s% chem_id(j))) // ', '
         end do
         write(*,*)
         write(*,*)
         do j=1,species
            write(*,1) 'x(j' // trim(chem_isos% name(s% chem_id(j))) // ')= ', s% xa(j,k)
         end do
         write(*,*)
         write(*,*)
         do j=1,species
            write(*,1) 'xin(net_iso(i' // trim(chem_isos% name(s% chem_id(j))) // '))= ', s% xa(j,k)
         end do
         write(*,*)
         write(*,*)
         write(*,1) 'T =', s% T(k)
         write(*,1) 'logT =', log10_T
         write(*,1) 'rho =', s% rho(k)
         write(*,1) 'logRho =', log10_Rho
         write(*,1) 'abar =', s% abar(k)
         write(*,1) 'zbar =', s% zbar(k)
         write(*,1) 'z2bar =', s% z2bar(k)
         write(*,1) 'ye =', s% ye(k)
         write(*,1) 'eta =', s% eta(k)
         write(*,*) 'screening_mode = ' // trim(s% screening_mode)
         write(*,1) 'theta_e =', s% theta_e(k)
         write(*,*)
      end subroutine show_stuff


      subroutine do1_zero_net_vars(s,k)
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         s% eps_nuc(k) = 0
         s% eps_nuc_categories(:,:,k) = 0
         s% d_epsnuc_dlnd(k) = 0
         s% d_epsnuc_dlnT(k) = 0
         s% d_epsnuc_dx(:,k) = 0
         s% eps_nuc_neu_total(k) = 0
         s% reaction_eps_nuc(:,:,k) = 0
         s% dxdt_nuc(:,k) = 0
         s% dxdt_dRho(:,k) = 0
         s% dxdt_dT(:,k) = 0
         s% d_dxdt_dx(:,:,k) = 0
         s% rate_screened(:,:,k) = 0
         s% rate_raw(:,:,k) = 0
      end subroutine do1_zero_net_vars
      
      
      integer function get_screening_mode(s,ierr)
         use screen_lib, only: screening_option
         type (star_info), pointer :: s 
         integer, intent(out) :: ierr
         include 'formats'
         ierr = 0
         if (s% screening_mode_value >= 0) then
            get_screening_mode = s% screening_mode_value
            return
         end if
         get_screening_mode = screening_option(s% screening_mode, ierr)
         if (ierr /= 0) return
         s% screening_mode_value = get_screening_mode
         !write(*,2) 'get_screening_mode ' // &
         !   trim(s% screening_mode), get_screening_mode
      end function get_screening_mode

      
      subroutine do_micro_change_net(s, new_net_name, ierr)
         use net_def
         type (star_info), pointer :: s
         character (len=*), intent(in) :: new_net_name
         integer, intent(out) :: ierr
         ierr = 0
         s% net_name = new_net_name
         call set_net(s, new_net_name, ierr)
      end subroutine do_micro_change_net
      
      
      subroutine set_net(s, new_net_name, ierr)
         use net_lib, only: free_net_handle, alloc_net_handle
         use utils_lib, only: realloc_double
         use alloc, only: update_nreactions_allocs
         use rates_def
         type (star_info), pointer :: s
         character (len=*), intent(in) :: new_net_name
         integer, intent(out) :: ierr
         
         integer :: old_num_reactions, i, ir
         integer, parameter :: num_lowT_rates = 10
         integer, pointer :: net_reaction_ptr(:) 
         
         include 'formats'

         old_num_reactions = s% num_reactions

         if (s% net_handle /= 0) call free_net_handle(s% net_handle)
         
         s% net_handle = alloc_net_handle(ierr)
         if (ierr /= 0) return   
         
         call setup_new_net_info(s, ierr)
         if (ierr /= 0) return
         
         call update_nreactions_allocs(s, ierr)
         if (ierr /= 0) return

         contains
         
         subroutine show
            use chem_def, only: chem_isos
            integer :: i
            include 'formats'
            return
            
            
            do i=1,s% species
               write(*,2) chem_isos% name(s% chem_id(i)), s% chem_id(i)
            end do
            write(*,*)
         end subroutine show

      end subroutine set_net
      
      
      subroutine setup_new_net_info(s, ierr)
         use net_lib
         use rates_def, only: ir_fe58_ng_fe59
         use alloc, only: update_nvar_allocs, set_chem_names
         type (star_info), pointer :: s
         integer, intent(out) :: ierr 
         
         integer :: old_nvar_chem
         
         include 'formats'
              
         ierr = 0
         call net_tables(s, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_tables'
            return
         end if
         
         s% species = net_num_isos(s% net_handle, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_num_isos'
            return
         end if
         
         s% num_reactions = net_num_reactions(s% net_handle, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_num_reactions'
            return
         end if
         
         old_nvar_chem = s% nvar_chem
         s% nvar_chem = s% species
         call update_nvar_allocs(s, s% nvar_hydro, old_nvar_chem, ierr)
         if (ierr /= 0) return
         
         call get_chem_id_table_ptr(s% net_handle, s% chem_id, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in get_chem_id_table_ptr'
            return
         end if
         
         call get_net_iso_table_ptr(s% net_handle, s% net_iso, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in get_net_iso_table_ptr'
            return
         end if

         call set_chem_names(s)
         
         call s% set_rate_factors(s% id, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in s% set_rate_factors'
            return
         end if
         
      end subroutine setup_new_net_info
      
      
      subroutine net_tables(s, ierr)
         use net_lib ! setup net
         use rates_lib
         type (star_info), pointer :: s
         integer, intent(out) :: ierr         
         ierr = 0
         
         call net_start_def(s% net_handle, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_start_def'
            return
         end if
         
         if (len_trim(s% net_name) == 0) then
            write(*,*) 'missing net_name -- please set it and try again'
            ierr = -1
            return
         end if

         call read_net_file(s% net_name, s% net_handle, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in read_net_file ' // trim(s% net_name)
            return
         end if
         
         call net_finish_def(s% net_handle, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_finish_def'
            return
         end if
         
         if (associated(s% rate_factors)) deallocate(s% rate_factors)
         allocate(s% rate_factors(rates_reaction_id_max))
         
         call s% set_rate_factors(s% id, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in set_rate_factors'
            return
         end if
         
         if (associated(s% which_rates)) deallocate(s% which_rates)
         allocate(s% which_rates(rates_reaction_id_max))
         
         call s% set_which_rates(s% id, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in set_which_rates'
            return
         end if

         call net_set_which_rates(s% net_handle, s% which_rates, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_set_which_rates'
            return
         end if

         call net_set_logTcut(s% net_handle, s% net_logTcut_lo, s% net_logTcut_lim, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_set_logTcut'
            return
         end if
         
         call net_setup_tables( &
            s% net_handle, rate_tables_dir_for_star, rates_cache_suffix_for_star, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_setup_tables'
            return
         end if

      end subroutine net_tables
      
      
      subroutine default_set_which_rates(id, ierr)
         use rates_def, only: rates_NACRE_if_available
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         s% which_rates(:) = rates_NACRE_if_available
      end subroutine default_set_which_rates
      
      
      subroutine default_set_rate_factors(id, ierr)
         use rates_def, only: rates_NACRE_if_available
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         s% rate_factors(:) = 1
      end subroutine default_set_rate_factors


      end module net

