! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module mod_diffusion_support

      use const_def
      use chem_def
      use utils_lib, only: is_bad_num
      use star_private_def

      implicit none

      
      real(dp), parameter :: Xlim = 1d-14
      real(dp), parameter :: tiny_mass = 1d3 ! a kilogram
      real(dp), parameter :: tinyX = 1d-50
      real(dp), parameter :: smallX = 1d-20


      contains

      
      subroutine get_matrix_coeffs( &
            s, nz, nc, m, nzlo, nzhi, ih1, ihe4, pure_Coulomb, &
            dt, v_advection_max, tiny_C, diffusion_factor, &
            A, X, Z, rho_face, T_face, four_pi_r2_rho_face, &
            xm_face, cell_dm, dm_bar, dlnP_dr_face, dlnT_dr_face, dlnRho_dr_face, &
            r_face, r_mid, gamma_T_limit_coeffs_face, alfa_face, &
            rad_accel_face, log10_g_rad, g_rad, d_ln_g_rad_d_ln_chi, max_T_for_radaccel, &
            X_init, X_face, C, C_div_X, C_div_X_face, E_field_face, v_advection_face, &
            vlnP_face, vlnT_face, v_rad_face, &
            GT_face, D_self_face, AD_face, SIG_face, sigma_lnC, ierr)
         
         type (star_info), pointer :: s
         integer, intent(in) :: &
            nz, nc, m, nzlo, nzhi, ih1, ihe4
         logical, intent(in) :: pure_Coulomb
         real(dp), intent(in) :: &
            dt, v_advection_max, tiny_C, max_T_for_radaccel
         real(dp), dimension(:), intent(in) :: &
            diffusion_factor, A, rho_face, T_face, four_pi_r2_rho_face, &
            xm_face, cell_dm, dm_bar, dlnP_dr_face, dlnT_dr_face, dlnRho_dr_face, &
            r_face, r_mid, gamma_T_limit_coeffs_face, alfa_face
         real(dp), dimension(:,:), intent(in) :: &
            Z, X_init, rad_accel_face, log10_g_rad, g_rad, d_ln_g_rad_d_ln_chi
         real(dp), dimension(:,:), intent(inout) :: X

         real(dp), dimension(:), intent(out) :: E_field_face, AD_face
         real(dp), dimension(:,:), intent(out) :: &
            X_face, C, C_div_X, C_div_X_face, v_advection_face, &
            vlnP_face, vlnT_face, v_rad_face, GT_face, D_self_face
         real(dp), dimension(:,:,:), intent(out) :: &
            SIG_face, sigma_lnC
         integer, intent(out) :: ierr
         
         integer :: i, j, jj, k, op_err, im
         real(dp) :: dv_im, alfa, beta, cc, tmp, tinyX, dlamch, sfmin, &
            AD_dm_full_on, AD_dm_full_off, AD_boost_factor, sum_dm, &
            Vlimit_dm_full_on, Vlimit_dm_full_off, Vlimit, sigmax, &
            SIG_factor, GT_factor
         real(dp), dimension(m) :: C_face, Z_face
         
         include 'formats'
         
         ierr = 0
         sfmin = dlamch('S')  
            
         tinyX = 1d-50
         do k=nzlo,nzhi
            do j=1,nc
               X(j,k) = max(X(j,k),tinyX)
            end do
            tmp = sum(Z(1:nc,k)*X(1:nc,k)/A(1:nc))
            do j=1,nc
               C_div_X(j,k) = 1d0/(A(j)*tmp)
               C(j,k) = X(j,k)*C_div_X(j,k)
            end do
            C(m,k) = 1d0
            X(m,k) = A(m)/dot_product(A(1:nc),C(1:nc,k)) 
         end do
         
         Vlimit_dm_full_on = s% diffusion_Vlimit_dm_full_on*Msun
         Vlimit_dm_full_off = s% diffusion_Vlimit_dm_full_off*Msun
         Vlimit = s% diffusion_Vlimit
         
!$OMP PARALLEL DO PRIVATE(k, op_err, C_face, Z_face)

         do k = nzlo+1, nzhi
            
            call get1_CXZ_face( &
               k, nz, nc, m, nzlo, nzhi, C, X, Z, A, alfa_face, tiny_C, &
               four_pi_r2_rho_face(k)/dm_bar(k), &
               C_face, X_face, Z_face, C_div_X_face)
            
            op_err = 0
            call get1_coeffs_face( &
               k, nz, nc, m, nzlo, nzhi, ih1, ihe4, pure_Coulomb, &
               rho_face(k), T_face(k), gamma_T_limit_coeffs_face(k), &
               four_pi_r2_rho_face(k), dm_bar(k), v_advection_max, tiny_C, sfmin, &
               dlnP_dr_face(k), dlnT_dr_face(k), dlnRho_dr_face(k), &
               Vlimit_dm_full_on, Vlimit_dm_full_off, Vlimit, xm_face(k), r_mid, dt, &
               A, X_face(:,k), Z_face, C_face, C_div_X_face(:,k), &
               diffusion_factor, rad_accel_face(:,k), (T_face(k) <= max_T_for_radaccel), &
               v_advection_face(:,k), vlnP_face(:,k), vlnT_face(:,k), v_rad_face(:,k), &
               E_field_face(k), sigma_lnC(:,:,k), op_err)
            if (op_err /= 0) ierr = op_err
            
         end do

!$OMP END PARALLEL DO

         if (ierr /= 0) return
         sum_dm = cell_dm(nzlo)
         
         AD_dm_full_on = s% diffusion_AD_dm_full_on*Msun
         AD_dm_full_off = s% diffusion_AD_dm_full_off*Msun
         AD_boost_factor = s% diffusion_AD_boost_factor
         
         SIG_factor = s% diffusion_SIG_factor
         GT_factor = s% diffusion_GT_factor
         
         !write(*,1) 'GT_factor SIG_factor', GT_factor, SIG_factor

         do k = nzlo+1, nzhi         
            call get1_flow_coeffs( &
               k, nc, m, v_advection_face(:,k), v_advection_max, &
               SIG_factor, GT_factor, sigma_lnC(:,:,k), &
               four_pi_r2_rho_face(k), dm_bar(k), &
               C_div_X_face(:,k), GT_face(:,k), D_self_face(:,k), SIG_face(:,:,k))            
            if (sum_dm >= AD_dm_full_off) then
               AD_face(k) = 0d0
            else
               sigmax = 0d0
               do j=1,nc
                  if (SIG_face(j,j,k) > sigmax) sigmax = SIG_face(j,j,k)
               end do
               AD_face(k) = AD_boost_factor*sigmax
               if (sum_dm > AD_dm_full_on) &
                  AD_face(k) = AD_face(k) * &
                     (sum_dm - AD_dm_full_off)/&
                        (AD_dm_full_on - AD_dm_full_off)
               !write(*,2) 'boost factor AD_face', k, AD_face(k)/sigmax, AD_face(k)
            end if
            sum_dm = sum_dm + cell_dm(k)              
         end do
         
         do j=1,nc ! not used, but copy just for sake of plotting
            D_self_face(j,nzlo) = D_self_face(j,nzlo+1)
            v_advection_face(j,nzlo) = v_advection_face(j,nzlo+1)
            vlnP_face(j,nzlo) = vlnP_face(j,nzlo+1)
            vlnT_face(j,nzlo) = vlnT_face(j,nzlo+1)
            v_rad_face(j,nzlo) = v_rad_face(j,nzlo+1)
            GT_face(j,nzlo) = GT_face(j,nzlo+1)
            do i=1,nc
               SIG_face(i,j,nzlo) = SIG_face(i,j,nzlo+1)
            end do
         end do
         E_field_face(nzlo) = E_field_face(nzlo+1)
                              
      end subroutine get_matrix_coeffs
      
      
      subroutine get1_coeffs_face( &
            k, nz, nc, m, nzlo, nzhi, ih1, ihe4, pure_Coulomb, &
            rho_face, T_face, gamma_T_limit_coeff_face, &
            four_pi_r2_rho_face, dm_bar, &
            v_advection_max, tiny_C, sfmin, &
            dlnP_dr_face, dlnT_dr_face, dlnRho_dr_face, &
            Vlimit_dm_full_on, Vlimit_dm_full_off, Vlimit, xm_face, r_mid, dt, &
            A, X_face, Z_face, C_face, C_div_X_face, &
            diffusion_factor, rad_accel_face, rad, &
            v_advection_face, vlnP_face, vlnT_face, v_rad_face, &
            E_field_face, sigma_lnC, ierr)
            
         integer, intent(in) :: k, nz, nc, m, nzlo, nzhi, ih1, ihe4
         logical, intent(in) :: pure_Coulomb
         real(dp), intent(in) :: rho_face, T_face, gamma_T_limit_coeff_face, &
            xm_face, r_mid(:), Vlimit_dm_full_on, Vlimit_dm_full_off, Vlimit, dt
         real(dp), intent(in) :: four_pi_r2_rho_face, dm_bar
         real(dp), intent(in) :: v_advection_max, tiny_C, sfmin
         real(dp), intent(in) :: dlnP_dr_face, dlnT_dr_face, dlnRho_dr_face
         real(dp), intent(in), dimension(:) :: &
            A, X_face, Z_face, C_face, C_div_X_face, &
            rad_accel_face, diffusion_factor 
         logical, intent(in) :: rad
         real(dp), intent(out), dimension(:) :: &
            v_advection_face, vlnP_face, vlnT_face, v_rad_face ! (nc)
         real(dp), intent(out) :: E_field_face
         real(dp), intent(out) :: sigma_lnC(:,:) ! (nc,nc)
         integer, intent(out) :: ierr
         
         integer :: i, j
         real(dp), dimension(m) :: AP, AT, AR
         real(dp), dimension(m,m) :: kappa_st, Zdiff, Zdiff1, Zdiff2, AX
         
         include 'formats'
         
         ierr = 0
            
         call get1_burgers_coeffs( &
            k, nc, m, A, Z_face, X_face, C_face, &
            rho_face, T_face, pure_Coulomb, &
            kappa_st, Zdiff, Zdiff1, Zdiff2)
         
         call get1_gradient_coeffs( &
            k, m, sfmin, A, Z_face, X_face, C_face, &
            rad, rad_accel_face, kappa_st, Zdiff, Zdiff1, Zdiff2, &
            AP, AT, AR, AX, E_field_face, ierr)
         if (ierr /= 0) return
         
         call get1_diffusion_velocities( &
            k, nc, m, nzlo, nzhi, AP, AT, AR, AX, rho_face, T_face, &
            dlnP_dr_face, dlnT_dr_face, dlnRho_dr_face, X_face, &
            Vlimit_dm_full_on, Vlimit_dm_full_off, Vlimit, xm_face, r_mid, dt, &
            gamma_T_limit_coeff_face, v_advection_max, diffusion_factor, &
            v_advection_face, vlnP_face, vlnT_face, v_rad_face, sigma_lnC)        
         
      end subroutine get1_coeffs_face

            
      subroutine get1_CXZ_face( &
            k, nz, nc, m, nzlo, nzhi, C, X, Z, A, alfa_face, tiny_C, &
            d_dr_factor, C_face, X_face, Z_face, C_div_X_face)
         integer, intent(in) :: k, nc, m, nz, nzlo, nzhi         
         real(dp), dimension(:,:), intent(in) :: C, X, Z ! (m,nz)
         real(dp), intent(in) :: A(:) ! (m) atomic number
         real(dp), intent(in) :: alfa_face(:), d_dr_factor
         real(dp), intent(in) :: tiny_C
         real(dp), dimension(:), intent(out) :: C_face, Z_face ! (m)
         real(dp), dimension(:,:), intent(out) :: X_face, C_div_X_face ! (m,nz)
         integer :: j
         real(dp) :: tmp, alfa, beta

         alfa = alfa_face(k)
         beta = 1d0 - alfa
         do j = 1, m
            X_face(j,k) = alfa*X(j,k) + beta*X(j,k-1)
            Z_face(j) = alfa*Z(j,k) + beta*Z(j,k-1)
         end do
         tmp = sum(Z_face(1:nc)*X_face(1:nc,k)/A(1:nc))
         do j = 1, m
            C_div_X_face(j,k) = 1/(A(j)*tmp)
            C_face(j) = X_face(j,k)*C_div_X_face(j,k)
         end do

      end subroutine get1_CXZ_face
      
      
      subroutine get1_burgers_coeffs( &
            k, nc, m, A, Z, X, C, rho, T, pure_Coulomb, &
            kappa_st, Zdiff, Zdiff1, Zdiff2)
         
         use mod_paquette_coeffs, only: paquette_coefficients
         
         integer, intent(in) :: k, nc, m
         real(dp), intent(in) :: rho, T
         logical, intent(in) :: pure_Coulomb
         real(dp), intent(in), dimension(:) :: A, X, Z, C ! (m)
         real(dp), intent(out), dimension(:,:) :: &
            kappa_st, Zdiff, Zdiff1, Zdiff2 ! (m,m)

         integer :: i, j
         real(dp) :: ac, ni, cz, xij, ne, ao, lambdad, lambda
         real(dp), dimension(m) :: charge, na
         real(dp), dimension(m,m) :: cl, Ath, Ddiff, Kdiff
            
         do i = 1, nc
            charge(i) = max(1d0, Z(i)) ! assume some ionization
         end do
         charge(m) = Z(m)
         
         if (.not. pure_Coulomb) then ! use Paquette coeffs
            ! Get number densities (per cm^3)
            do i = 1, nc
               na(i) = rho*X(i)/(A(i)*amu)   
            end do         
            na(m) = 0.d0      
            do i = 1, nc
               na(m) = na(m) + charge(i)*na(i)
            end do
            ! Compute resistance coefficients from Paquette&al (1986)   
            call paquette_coefficients( &
               rho, T, m, A, charge, na, Ddiff, Kdiff, Zdiff, Zdiff1, Zdiff2, Ath)
            kappa_st(:,:) = Kdiff(:,:)/(1.41D-25*T**(-1.5D0)*na(m)**2)
               ! = kappa_st of eq 37, Thoul&al 1994 
            return
         end if
         
         ! calculate density of electrons (ne) from mass density (rho):
         ac=0.
         do i=1, m
   	      ac=ac+a(i)*c(i)
         end do	
         ne=rho/(mp*ac)	
         ! calculate interionic distance (ao): 
         ni=0.
         do i=1, nc
            ni=ni+c(i)*ne
         end do
         ao=(0.23873/ni)**(1./3.)	
         ! calculate debye length (lambdad):
         cz=0.
         do i=1, m
   	      cz=cz+c(i)*charge(i)**2
         end do
         lambdad=6.9010*sqrt(t/(ne*cz))
         ! calculate lambda to use in coulomb logarithm:
         lambda=max(lambdad, ao)
         ! calculate coulomb logarithms:
         do i=1, m
            do j=1, m
               xij=2.3939e3*t*lambda/abs(z(i)*z(j))
               cl(i,j)=0.81245*log(1.+0.18769*xij**1.2)
            end do
         end do

         ! set coeffs for pure Coulomb potential
         do i=1, m
            do j=1, m
               Zdiff(i,j) = 0.6d0
               Zdiff1(i,j) = 1.3d0
               Zdiff2(i,j) = 2d0
               kappa_st(i,j) = &
                  cl(i,j)*sqrt(a(i)*a(j)/(a(i)+a(j)))* &
                     c(i)*c(j)*charge(i)**2*charge(j)**2
            end do
         end do
         
      end subroutine get1_burgers_coeffs
      

      subroutine get1_gradient_coeffs( &
            k, m, sfmin, A, Z, X, C, &
            rad, rad_accel, kappa_st, Zdiff, Zdiff1, Zdiff2, &
            AP, AT, AR, AX, E_field, ierr)
         integer, intent(in) :: k, m
         real(dp), intent(in) :: sfmin
         real(dp), intent(in), dimension(:) :: A, X, Z, C, rad_accel ! (m)
         logical, intent(in) :: rad
         real(dp), dimension(:,:), intent(in) :: &
            kappa_st, Zdiff, Zdiff1, Zdiff2 ! (m,m)
         real(dp), dimension(:), intent(out) :: AP, AT, AR ! (m)
         real(dp), intent(out) :: AX(:,:) ! (m,m)
         real(dp), intent(out) :: E_field
         integer, intent(out) :: ierr
           
         integer :: i, j
         real(dp) :: charge(m)
         
         include 'formats'
         
         ierr = 0
            
         do i=1,m-1
            charge(i) = max(1d0, Z(i))
         end do
         charge(m) = Z(m)

         call do1_solve_thoul_hu( &
            2*m+2, m, sfmin, A, charge, X, C, rad_accel, rad, &
            kappa_st, Zdiff, Zdiff1, Zdiff2, &
            AP, AT, AR, AX, E_field, ierr)
            
         if (ierr /= 0) then
            return
            
            
            write(*,2) 'do1_solve_thoul_hu failed', k
            do i=1,m-1
               write(*,2) 'A X Z C', i, A(i), X(i), Z(i), C(i)
            end do
            stop 'get1_gradient_coeffs'
         end if
         
      end subroutine get1_gradient_coeffs
      
      
      subroutine get1_diffusion_velocities( &
            k, nc, m, nzlo, nzhi, AP, AT, AR, AX, rho, T, &
            dlnP_dr, dlnT_dr, dlnRho_dr, X_face, &
            Vlimit_dm_full_on, Vlimit_dm_full_off, Vlimit, xm_face, r_mid, dt, &
            limit_coeff, v_advection_max, diffusion_factor, &
            vgt, vlnP, vlnT, vrad, sigma_lnC)
         integer, intent(in) :: k, nc, m, nzlo, nzhi
         real(dp), intent(in), dimension(:) :: AP, AT, AR, r_mid
         real(dp), intent(in) :: AX(:,:) ! (m,m)
         real(dp), intent(in) :: rho, T, limit_coeff, v_advection_max, &
            Vlimit_dm_full_on, Vlimit_dm_full_off, Vlimit, xm_face, dt
         real(dp), intent(in) :: dlnP_dr, dlnT_dr, dlnRho_dr
         real(dp), intent(in) :: X_face(:)
         real(dp), intent(in) :: diffusion_factor(:)
         real(dp), intent(out), dimension(:) :: vgt, vlnP, vlnT, vrad
         real(dp), intent(out) :: sigma_lnC(:,:) ! (nc,nc)
         
         integer :: i, j, im
         real(dp) :: coef, coef_vrad, dv_im, dr, &
            vcross, vmax, vmax_limit, frac, alfa, beta
         real(dp) :: tau0  ! = 6d13*secyer, solar diffusion time (seconds)
         real(dp), parameter :: rho_unit = 1d2
         real(dp), parameter :: T_unit = 1d7
         
         include 'formats'

         if (limit_coeff <= 0) then
            vgt(:) = 0
            sigma_lnC(:,:) = 0
            return
         end if
         
         dr = r_mid(k-1) - r_mid(k)
         vcross = dr/dt
         if (xm_face >= Vlimit_dm_full_off .or. Vlimit <= 0d0) then
            vmax_limit = 1d99
            alfa = 0d0
            beta = 1d0
         else if (xm_face <= Vlimit_dm_full_on) then
            vmax_limit = vcross*Vlimit
            alfa = 1d0
            beta = 0d0
         else ! combine
            alfa = (xm_face - Vlimit_dm_full_off)/&
                        (Vlimit_dm_full_on - Vlimit_dm_full_off)
            beta = 1d0 - alfa ! fraction of normal v when it is > vmax
            vmax_limit = alfa*vcross*Vlimit
         end if
         
         tau0 = 6d13*secyer
         coef = limit_coeff*Rsun*(T/T_unit)**2.5d0/(rho/rho_unit)*(Rsun/tau0) 
         coef_vrad = (limit_coeff/T)*Rsun**2.*(T/T_unit)**2.5d0/(rho/rho_unit)/tau0
            ! converts to cgs units
         
         do i=1,nc
            do j=1,nc
               sigma_lnC(j,i) = -diffusion_factor(i)*coef*AX(j,i)
            end do
            vlnP(i) = AP(i)*dlnP_dr*diffusion_factor(i)*coef
            vlnT(i) = AT(i)*dlnT_dr*diffusion_factor(i)*coef
            vrad(i) = AR(i)*diffusion_factor(i)*coef_vrad
            vgt(i) = vlnP(i) + vlnT(i) + vrad(i)
            if (X_face(i) < 1d-15) then 
               vgt(i) = 0d0
            else if (X_face(i) < 1d-22) then
               vgt(i) = vgt(i)*X_face(i)/1d-22
            end if
         end do
         
         ! final fixup for vgt of most abundant so it gives baryon conservation.
         im = maxloc(X_face(1:nc),dim=1)
         dv_im = -dot_product(X_face(1:nc), vgt(1:nc))/X_face(im)
         vgt(im) = vgt(im) + dv_im
         
         vmax = maxval(abs(vgt(1:nc)))
         if (vmax > v_advection_max) then
            frac = v_advection_max/vmax
            do i=1,nc
               vgt(i) = vgt(i)*frac
            end do
            vmax = v_advection_max
            !write(*,3) 'vmax > v_advection_max', im, k, vmax, v_advection_max
            !stop 'get1_diffusion_velocities'
         end if

         if (alfa > 0d0 .and. vmax > vmax_limit) then
            frac = vmax_limit/vmax
            do i=1,nc
               vgt(i) = vgt(i)*frac
            end do
         end if
         
      end subroutine get1_diffusion_velocities
      
      
      subroutine get1_flow_coeffs( &
            k, nc, m, &
            v_advection_face, v_advection_max, SIG_factor, GT_factor, &
            sigma_lnC_face, four_pi_r2_rho_face, &
            dm_bar, C_div_X_face, GT_face, D_self_face, SIG_face)
         integer, intent(in) :: k, nc, m
         real(dp), intent(in) :: &
            v_advection_max, SIG_factor, GT_factor, v_advection_face(:) ! (nc)
         real(dp), intent(in) :: sigma_lnC_face(:,:) ! (nc,nc)
         real(dp), intent(in) :: four_pi_r2_rho_face, dm_bar
         real(dp), intent(in), dimension(:) :: C_div_X_face ! (m)
         real(dp), intent(out) :: GT_face(:) ! (nc)
         real(dp), intent(out) :: D_self_face(:) ! (nc)
         real(dp), intent(out) :: SIG_face(:,:) ! (nc,nc)
         
         integer :: i, j
         real(dp) :: c, boost
         
         include 'formats'

         c = SIG_factor*four_pi_r2_rho_face**2/dm_bar
         do i = 1, nc
            GT_face(i) = GT_factor*four_pi_r2_rho_face*v_advection_face(i)
            D_self_face(i) = sigma_lnC_face(i,i)  
            do j = 1, nc
               SIG_face(i,j) = c*sigma_lnC_face(i,j)/C_div_X_face(j)               
            end do
         end do
         
      end subroutine get1_flow_coeffs


!*************************************************************
! Original of this routine was written by Anne A. Thoul, at the Institute
! for Advanced Study, Princeton, NJ 08540.
! See Thoul et al., Ap.J. 421, p. 828 (1994)

! With modifications by Hali Hu for non Coulomb and rad levitation.
!*************************************************************
! This routine inverses the burgers equations.
!
! The system contains N equations with N unknowns. 
! The equations are: the M momentum equations, 
!                    the M energy equations, 
!                    two constraints: the current neutrality 
!                                     the zero fluid velocity.
! The unknowns are: the M diffusion velocities,
!                   the M heat fluxes,
!                   the electric field E
!                   the gravitational force g.
!
!**************************************************
      subroutine do1_solve_thoul_hu( &
            n, m, sfmin, a, z, x, c, rad_accel, rad, &
            kappa_st, Zdiff, Zdiff1, Zdiff2, &
            ap, at, ar, ax, e_field, ierr)

         ! the parameter m is the number of fluids considered (ions+electrons)
         ! the parameter n is the number of equations (2*m+2).
         !
         ! the vectors a,z and x contain the atomic mass numbers, 
         ! the charges (ionization), and the mass fractions, of the elements.
         ! note: since m is the electron fluid, its mass and charge must be
         !      a(m)=m_e/m_u
         !      z(m)=-1.
         !
         ! the array cl contains the values of the coulomb logarithms.
         ! the vector ap, at, and array ax contains the results for the diffusion 
         ! coefficients.

         integer, intent(in) :: m,n
         real(dp), intent(in) :: sfmin
         real(dp), intent(in), dimension(:) :: A, Z, X, C, rad_accel ! (m)
         logical, intent(in) :: rad
         real(dp), intent(in), dimension(:,:) :: &
            kappa_st, Zdiff, Zdiff1, Zdiff2 ! (m,m)
!           kappa_st from the resistance coefficient Kdiff with eq (37) Thoul&al.
!           Zdiff, Zdiff1, Zdiff2 = arrays of resistance coefficients,
         real(dp), intent(out), dimension(:) :: ap, at, ar ! (m)
         real(dp), intent(out) :: ax(:,:) ! (m,m)
         real(dp), intent(out) :: e_field
         integer, intent(out) :: ierr

         integer :: i, j, l, indx(n), nmax
         real(dp) :: aamax, cc, ac, temp, ko, d
         real(dp), dimension(m,m) :: xx, y, yy, k
         real(dp), dimension(n) :: alpha, nu, ga, beta
         real(dp), dimension(n,n) :: delta, gamma

         ! the vector c contains the concentrations
         ! cc is the total concentration: cc=sum(c_s)
         ! ac is proportional to the mass density: ac=sum(a_s c_s)
         ! the arrays xx,y,yy and k are various parameters which appear in 
         ! burgers equations.
         ! the vectors and arrays alpha, nu, gamma, delta, and ga represent
         ! the "right- and left-hand-sides" of burgers equations, and later 
         ! the diffusion coefficients.
      
         ! initialize:

         ierr = 0
         ko = 2d0  
         indx(1:n) = 0    

         ! calculate cc and ac:
      
         cc=sum(c(1:m))
         ac=dot_product(a(1:m),c(1:m))

         ! calculate the coefficients of the burgers equations

         do i=1,m
            do j=1,m
               xx(i,j)=a(j)/(a(i)+a(j))
               y(i,j)=a(i)/(a(i)+a(j))
		         yy(i,j) = 3D0*y(i,j) + Zdiff1(i,j)*xx(i,j)*a(j)/a(i)
               k(i,j) = kappa_st(i,j)
            end do
         end do

         ! write the burgers equations and the two constraints as
         ! alpha_s dp + nu_s dt + sum_t(not ihe or m) gamma_st dc_t 
         !                     = sum_t delta_st w_t

         do i=1,m
            alpha(i)=c(i)/cc
            nu(i)=0d0
            gamma(i,1:n)=0d0
            if (rad) then
               beta(i) = -(amu/boltzm)*alpha(i)*a(i)*rad_accel(i)  
            else
               beta(i) = 0d0
            end if
            do j=1,m
               if (j /= m) then ! HH: Include He gradient
                  gamma(i,j) = -c(j)/cc
                  if (j == i) gamma(i,j) = gamma(i,j) + 1d0
                  gamma(i,j) = gamma(i,j)*c(i)/cc
               end if
            end do
         end do
      
         do i=m+1,n-2
            alpha(i)=0d0
            nu(i)=2.5d0*c(i-m)/cc
            beta(i) = 0d0
            gamma(i,1:n)=0d0
         end do
      
         alpha(n-1)=0d0
         nu(n-1)=0d0
         beta(n-1)=0d0
         gamma(n-1,1:n)=0d0
      
         alpha(n)=0d0
         nu(n)=0d0
         beta(n)=0d0
         gamma(n,1:n)=0d0
      
         delta(1:n,1:n) = 0d0
      
         do i=1,m
         
            do j=1,m
               if (j == i) then
                  do l=1,m
                     if (l /= i) then
                        delta(i,j)=delta(i,j)-k(i,l)
                     end if
                  end do
               else
                  delta(i,j)=k(i,j)
               end if
            end do
         
            do j=m+1,n-2
               if (j-m == i) then
                  do l=1,m
                     if (l /= i) &
                        delta(i,j) = delta(i,j) + Zdiff(i,l)*xx(i,l)*k(i,l)
                  end do
               else
                  delta(i,j) = -Zdiff(i,j-m)*y(i,j-m)*k(i,j-m)
               end if
            end do
         
            delta(i,n-1)=c(i)*z(i)
         
            delta(i,n)=-c(i)*a(i)
            
         end do
      
         do i=m+1,n-2
         
            do j=1,m
               if (j == i-m) then
                  do l=1,m
                     if (l /= i-m) delta(i,j) = &
                        delta(i,j) + 2.5D0*Zdiff(i-m,l)*xx(i-m,l)*k(i-m,l)
                  end do
               else
                  delta(i,j) = -(2.5d0*Zdiff(i-m,j))*xx(i-m,j)*k(i-m,j)
               end if
            end do
         
            do j=m+1,n-2
               if (j-m == i-m) then
                  do l=1,m
                     if (l /= i-m) delta(i,j) = delta(i,j) - &
                           y(i-m,l)*k(i-m,l)*(0.8D0*Zdiff2(i-m,l)*xx(i-m,l)+yy(i-m,l))
                  end do
                  delta(i,j) = delta(i,j) - 0.4D0*Zdiff2(i-m,i-m)*k(i-m,i-m)
               else
                  delta(i,j) = k(i-m,j-m)*xx(i-m,j-m)*y(i-m,j-m) * &
                        (3D0 + Zdiff1(i-m,j-m) - 0.8D0*Zdiff2(i-m,j-m))
               end if
            end do
         
            delta(i,n-1:n)=0d0
            
         end do
      
         do j=1,m
            delta(n-1,j) = c(j)*z(j)
         end do
         delta(n-1,m+1:n) = 0d0
      
         do j=1,m
            delta(n,j) = c(j)*a(j)
         end do
         delta(n,m+1:n) = 0d0
         
         call dgetrf(n, n, delta, n, indx, ierr)
         if (ierr /= 0) return
      
         call dgetrs( 'n', n, 1, delta, n, indx, alpha, n, ierr )
         if (ierr /= 0) return
      
         call dgetrs( 'n', n, 1, delta, n, indx, nu, n, ierr )
         if (ierr /= 0) return
      
         if (rad) then
            call dgetrs( 'n', n, 1, delta, n, indx, beta, n, ierr )
            if (ierr /= 0) return
         end if
      
         do j=1,n
            do i=1,n
               ga(i)=gamma(i,j)
            end do
            call dgetrs( 'n', n, 1, delta, n, indx, ga, n, ierr )
            if (ierr /= 0) return
            do i=1,n
               gamma(i,j)=ga(i)
            end do
         end do
         
         do j=1,m
            ap(j)=alpha(j)*ko*ac*cc
            at(j)=nu(j)*ko*ac*cc
            ar(j)=beta(j)*ko*ac*cc
            do i=1,m
               ax(i,j)=gamma(i,j)*ko*ac*cc
            end do
         end do
         e_field = gamma(n-1,1)

      end subroutine do1_solve_thoul_hu                                                        


      end module mod_diffusion_support

